#include "pch.h"
#include "mbMainFrame.h"
#include "twArgs.h"
#include <bbcComputer.h>
#include <bbcKeyboardMatrix.h>
#include <bbcSystemVIA.h>
#include <bbcSound.h>
#include <bbcFdd.h>
#include <bbcUserVIA.h>
#include <bbcSystemVIA.h>
#include <bbc1770.h>
#include "mbMisc.h"
#include "mbQuickstart.h"
#include "mbRomsConfigDialog.h"
#include "mbSoundConfigDialog.h"
#include "mbKeyboardConfigDialog.h"
#include "mbJoysticksConfigDialog.h"
#ifdef BEEB_DEBUG_SAVE_CHALLENGER_RAM
#include <bbc1770OpusChallenger.h>
#endif
#include "mbDiscFormat.h"
#include <wx/progdlg.h>

//TODO the drive handling sucks. UiUnloadDrive similarly.

//Enable the File|Test item
//#define ENABLE_TEST

#define AUTO_PAINT() this->SetAutoPaint(true)

//Id offsets for each drive
enum {
	subid_drive=0,
	subid_drive_filename,
	subid_drive_load,
	subid_drive_save,
	subid_drive_unload,
	subid_drive_num,//MUST BE LAST
};

const mbMainFrame::DriveCommandMfn mbMainFrame::drive_subactions[subid_drive_num]={
	0,
	0,
	&mbMainFrame::OnDriveLoad,
	&mbMainFrame::OnDriveSave,
	&mbMainFrame::OnDriveUnload,
};

enum {
	id_status_bar=1,
		
	id_exit=1000,
	id_quickstart,
	id_hard_reset,
#ifdef ENABLE_TEST
	id_test,
#endif
	id_toggle_sound,
	id_toggle_limit_speed,
	id_toggle_status_bar,
	id_toggle_fast_forward_disc_access,
//	id_toggle_vertical_2x,
	id_toggle_full_screen,
	id_show_about_box,

	id_roms_dialog,
	id_sound_dialog,
	id_keymap_dialog,
	id_joysticks_dialog,
	
	id_keymaps_submenu,
	
	id_resolution_submenu,

	id_resolutions_begin,
	id_resolution_0=id_resolutions_begin,
	id_resolution_1,
	id_resolution_2,
	id_resolution_3,
	id_resolutions_end,
	
#ifdef bbcDEBUG_TRACE
	id_start_disassembly,
	id_stop_disassembly,
#endif

#ifdef BEEB_DEBUG_SAVE_CHALLENGER_RAM
	id_save_challenger_ram,
#endif
#ifdef BEEB_DEBUG_DISASSEMBLE_RAM
	id_disassemble_ram,
#endif
	
	id_screenshot_save_as,

#ifdef bbcDEBUG_PANELS
	id_toggle_dbgpanel_sysvia,
	id_dbgpanel_sysvia,
	id_toggle_dbgpanel_usrvia,
	id_dbgpanel_usrvia,
	id_toggle_dbgpanel_video,
	id_dbgpanel_video,
#endif
#ifdef bbcDEBUG_VIDEO
	id_toggle_showall,
	id_log_video_frames,
#ifdef bbcDEBUG_TRACE
	id_log_video_frames_disasm,
#endif
	id_toggle_show_t1_timeouts,
#endif
	//TODO should be begin/end
	id_first_keymap,
	id_last_keymap=id_first_keymap+1000,

	id_drives_begin,
	id_drives_end=id_drives_begin+bbcFdd::num_drives*subid_drive_num,

	id_sound_record_start,
	id_sound_record_stop,

	id_screen_type_submenu,
	id_screen_types_begin,
	id_screen_type_colour,
	id_screen_type_bnw,
	id_screen_type_amber,
	id_screen_type_green,
	id_screen_types_end,

	id_scanlines_begin,
	id_windowed_scanlines_submenu,
	id_windowed_scanlines_single,
	id_windowed_scanlines_double,
	id_windowed_scanlines_end,

	id_full_screen_scanlines_submenu,
	id_full_screen_scanlines_single,
	id_full_screen_scanlines_interlace,
	id_full_screen_scanlines_double,
	id_full_screen_scanlines_end,
	id_scanlines_end,
	
	id_file_menu,
	id_options_menu,
	id_special_menu,
#ifdef BEEB_DEBUG_ENABLE
	id_debug_menu,
#endif
	id_help_menu,
};

//////////////////////////////////////////////////////////////////////////
// just a pan
class NullPanel:
public wxWindow
{
public:
	NullPanel(wxWindow *parent,wxWindowID id);
private:
	DECLARE_EVENT_TABLE();
	void OnEraseBackground(wxEraseEvent &event);
	void OnRightUp(wxMouseEvent &event);
};

BEGIN_EVENT_TABLE(NullPanel,wxWindow)
	EVT_ERASE_BACKGROUND(NullPanel::OnEraseBackground)
	EVT_RIGHT_UP(NullPanel::OnRightUp)
END_EVENT_TABLE()

NullPanel::NullPanel(wxWindow *parent,wxWindowID id):
wxWindow(parent,id,wxDefaultPosition,wxDefaultSize,0)
{
}

void NullPanel::OnEraseBackground(wxEraseEvent &event) {
	return;
}

void NullPanel::OnRightUp(wxMouseEvent &event) {
	wxWindow *parent=this->GetParent();
	if(parent) {
		parent->AddPendingEvent(event);
	}
}

//////////////////////////////////////////////////////////////////////////
//
mbMainFrame::SbState::SbState():
caps(false),
shift(false)
{
	for(unsigned i=0;i<bbcFdd::num_drives;++i) {
		this->drives[i].what=-1;//unloaded
	}
}

//////////////////////////////////////////////////////////////////////////
//
mbMainFrame::DriveUi::DriveUi():
//loaded(false),
last_filter_index(0)
{
}

//////////////////////////////////////////////////////////////////////////
//
bool mbMainFrame::DoCommandLine(const std::vector<wxString> &argv) {
	std::vector<twArg *> all_args;
	unsigned i;

	all_args.push_back(new twArgPresent("selectini",&cl_select_ini_));
	all_args.push_back(new twArgString("inifile",&cl_ini_file_));
	all_args.push_back(new twArgPresent("autoboot",&cl_autoboot_));
	all_args.push_back(new twArgMulti<twArgString>("inioverride",
		&cl_ini_override_files_));
	all_args.push_back(new twArgState("inisave",&cl_ini_save_));
	all_args.push_back(new twArgState("verbose",&cl_verbose_));
	for(i=0;i<4;++i) {
		all_args.push_back(new twArgString(wxString::Format("mount%u",i),
			&cl_drive_mounts_[i]));
	}
	bool ok=twArg::ProcessAll(argv,all_args);
	for(i=0;i<all_args.size();++i) {
		delete all_args[i];
	}
	return ok;
}

//////////////////////////////////////////////////////////////////////////
//
mbMainFrame::mbMainFrame(const std::vector<wxString> &argv):
wxFrame(0,-1,"BBC Emulator",wxDefaultPosition,wxDefaultSize,wxDEFAULT_FRAME_STYLE),
auto_paint_(false),
ok_(false),
keymap_(0),
//cfg_(cfg),
hostgfx_(0),
hostinp_(0),
hostsnd_(0),
num_frames_(0),
next_update_keys_(0),
next_update_sys_(0),
next_update_sound_(0),
last_update_sys_time_(0),
debouncing_break_(false),
last_update_sound_hf_time_(0),
beeb_paused_(true),
keymaps_submenu_(0),
screenshot_countdown_(0),
beeb_screen_(0),
status_bar_(0),
beeb_fastforward_(false),
cl_select_ini_(false),
cl_autoboot_(false),
cl_ini_save_(true),
menu_bar_(0),
popup_menu_(0)
{
	int i;

#ifdef _DEBUG
	cl_verbose_=true;
#else
	cl_verbose_=false;
#endif

	if(!this->DoCommandLine(argv)) {
		return;
	}

	wxLog::GetActiveTarget()->SetVerbose(cl_verbose_);
	
	if(cl_select_ini_) {
		wxArrayString ini_names_arr;
		std::vector<wxFileName> ini_names;
		mbGetMatchingFileNames(wxFileName::DirName("."),"*.ini",&ini_names);
		if(ini_names.empty()) {
			wxLogError("No files matching \"*.ini\" for -selectini");
			return;
		}
		for(unsigned i=0;i<ini_names.size();++i) {
			ini_names_arr.Add(ini_names[i].GetName());
		}
		int index=wxGetSingleChoiceIndex("Select INI file to use","-selectini",
			ini_names_arr);
		if(index<0) {
			return;
		} else {
			cl_ini_file_=mbNormalizedFileName(ini_names[index].GetFullPath());
		}
	}

	if(cl_ini_file_.empty()) {
		cl_ini_file_="./beeb.ini";
	}
	cfg_.Load(cl_ini_file_);
	for(i=0;i<int(cl_ini_override_files_.size());++i) {
		cfg_.Load(cl_ini_override_files_[i]);
	}

	//By default, ini is readonly if overrides were specified. This stops override
	//settings leaking into the main file. However, this may be overridden with
	//-inisave.
	cfg_.SetFilename(cl_ini_file_,!cl_ini_override_files_.empty()&&!cl_ini_save_);

	//Child windows
	beeb_screen_=new NullPanel(this,-1);//,wxDefaultPosition,wxDefaultSize,wxNO_BORDER);
	beeb_screen_->SetBackgroundColour(*wxBLACK);
	beeb_screen_->Show(true);
	status_bar_=new mbStatusBar(this,id_status_bar);//,wxST_SIZEGRIP);
	wxBoxSizer *all_sz=new wxBoxSizer(wxVERTICAL);
	all_sz->Add(beeb_screen_,1,wxEXPAND);
	this->SetSizer(all_sz);
//	wxSlider *test=new wxSlider(status_bar_,-1,5,0,10);
//	test->Move(0,0);
//	test->SetSize(200,10);

	//host
	ok_=true;
	auto_paint_=true;
	hostgfx_=HostGfx_Start(this,beeb_screen_);
	if(!hostgfx_) {
		wxMessageBox("Failed to initialise graphics.\n\nThe emulator will now exit.",
			"Graphics failed",wxOK|wxICON_ERROR,this);
		ok_=false;
		return;
	}
	hostinp_=HostInp_Start(this);
	if(!hostinp_) {
		wxMessageBox("Failed to initialise keyboard.\n\nThe emulator will now exit.",
			"Keyboard failed",wxOK|wxICON_ERROR,this);
		ok_=false;
		return;
	}
	HostTmr_Start();
	HostTmr_GetHfFrequency(&hf_freq_);
	
	this->InitMenuBar();
	
	if(!this->ApplyConfig()) {
		ok_=false;
	} else {
		//Mount startup drives
		for(i=0;i<bbcFdd::num_drives;++i) {
			if(!cl_drive_mounts_[i].empty()) {
				this->UiLoadDisc(i,cl_drive_mounts_[i]);
			}
		}

		if(cl_autoboot_) {
#ifdef bbcQUICKSTART_BY_SHIFT
			bbcKeyboardMatrix::SetForceShiftPressed(true);
#else
			bbcKeyboardMatrix::SetBootKeylinkOneoff();
#endif
		}
	}
#ifndef _DEBUG
	//TODO starting in full screen doesn't work properly.
	//cfg_.video.full_screen=false;
#endif
}

bool mbMainFrame::Ok() const {
	return ok_;
}

void mbMainFrame::SetKeymap(const wxString &keymap_name) {
	for(mbKeymap::C::iterator keymap=cfg_.keyboard.keymaps.begin();
		keymap!=cfg_.keyboard.keymaps.end();++keymap)
	{
		if(keymap_name.CmpNoCase(keymap->name)==0) {
			break;
		}
	}
	if(keymap!=cfg_.keyboard.keymaps.end()) {
		keymap_=&*keymap;
	} else {
		keymap_=&cfg_.keyboard.keymaps.front();
		wxLogWarning("SetKeymap: \"%s\": not found\nSetKeymap: \"%s\": picked instead\n",
			keymap_name.c_str(),keymap_->name.c_str());
	}
	cfg_.keyboard.selected_keymap=keymap_->name;
}

//////////////////////////////////////////////////////////////////////////
// Returns true if closing is OK.
//
// If there's no changes to be saved, return true; otherwise, pop up a
// dialog box asking for confirmation.
bool mbMainFrame::CloseConfirmed() {
	for(int i=0;i<bbcFdd::num_drives;++i) {
		if(bbcFdd::DriveStatus(i)!=bbcFdd::DRIVE_SIDE2&&
			!this->UiUnloadDrive(i))
		{
			return false;
		}
	}
	return true;
}

//////////////////////////////////////////////////////////////////////////
//
void mbMainFrame::OnClose(wxCloseEvent &event) {
	if(event.CanVeto()) {
		if(!this->CloseConfirmed()) {
			event.Veto();
			return;
		}
	}

	//If running full screen, don't save window size.
	if(!hostgfx_||!HostGfx_IsFullScreen(hostgfx_)) {
		//If it's maximized, keep the size but set it to maximized.
		if(this->IsIconized()) {
			//If the window is iconized, don't save its size.
		} else if(this->IsMaximized()) {
			//it is maximized, don't save its window rect
			cfg_.misc.window_is_maximized=true;
		} else {
			//not maximized, save window rect.
			cfg_.misc.window_is_maximized=false;
			cfg_.misc.window_rect=this->GetRect();
		}
	}

	cfg_.Save();

	//the main frame can receive more idle events if any dialogs are
	//popped up whilst processing this next bit.
	//This stops anything being done with them.
	beeb_paused_=true;

	//Everything in popup_menu_ is shared with the menu bar, so must remove
	//them all before deleteing it so the items aren't double deleted. And
	//must copy data because it ischanged by Remove (duh!!)
//	popup_menu_->Remove(id_file_menu);
//	popup_menu_->Remove(id_options_menu);
//	popup_menu_->Remove(id_special_menu);
#ifdef BEEB_DEBUG_ENABLE
//	popup_menu_->Remove(id_debug_menu);
#endif
//	popup_menu_->Remove(id_help_menu);
//	delete popup_menu_;
//	popup_menu_=0;
//	this->SetMenuBar(0);
//	menu_bar_->Destroy();
//	menu_bar_=0;

	for(int i=0;i<bbcFdd::num_drives;++i) {
		bbcFdd::UnloadDiscImage(i);
	}
	HostSnd_Stop(hostsnd_);
	hostsnd_=0;
	HostInp_Stop(hostinp_);
	hostinp_=0;
	HostGfx_Stop(hostgfx_);
	hostgfx_=0;
	HostTmr_Stop();
	this->Destroy();
}

/*
void mbMainFrame::UpdateStatusBar() {
	static const wxString caps_lock_on="CAPS LOCK";
	static const wxString shift_lock_on="SHIFT LOCK";
	wxStatusBar *sb=this->GetStatusBar();
	
	if(bbc_system_via.CapslockLedLit()) {
		sb->SetStatusText(caps_lock_on,0);
	} else {
		sb->SetStatusText("",0);
	}
	if(bbc_system_via.ShiftlockLedLit()) {

	}
}
*/

void mbMainFrame::DoUpdateSys() {
	int this_time;
	HostTmr_ReadMilliseconds(&this_time);
	bbc1770::FdcAction action;
	int drive,side,track,sector;
	bbc1770::GetCurrentAction(&action,&drive,&side,&track,&sector);
	beeb_fastforward_=action!=bbc1770::ACTION_IDLE&&
		cfg_.misc.fast_forward_disc_access&&
		bbcKeyboardMatrix::NumKeysDown()==0;
//	if(this_time-last_update_sys_time_>100) 

#ifdef bbcDEBUG_PANELS
	mbDebugPanel *usr=static_cast<mbDebugPanel *>(this->FindWindow(id_dbgpanel_usrvia));
	mbDebugPanel *sys=static_cast<mbDebugPanel *>(this->FindWindow(id_dbgpanel_sysvia));
	mbDebugPanel *vid=static_cast<mbDebugPanel *>(this->FindWindow(id_dbgpanel_video));
	if(usr) {
		bbc_user_via.UpdateDebugPanel(usr->Panel());
		usr->UpdateFromPanel();
	}
	if(sys) {
		bbc_system_via.UpdateDebugPanel(sys->Panel());
		sys->UpdateFromPanel();
	}
	if(vid) {
		vid->UpdateFromPanel();
	}
#endif

	if(cfg_.misc.show_status_bar&&this_time-last_update_sys_time_>250) {
		//MHz
		static int last_time_cycles=0;
		int num_cycles=bbcComputer::cycles-last_time_cycles;
		float num_seconds=(this_time-last_update_sys_time_)/1000.f;
		float mhz=num_cycles/num_seconds/1000000.f;
		status_bar_->SetMhz(mhz);

		//Screen
		int screen_w,screen_h;
		beeb_screen_->GetClientSize(&screen_w,&screen_h);
		status_bar_->SetScreenSize(screen_w,screen_h);

		//FDC status
		int real_drive=bbcFdd::DriveFromFdcAddress(drive,side);
		status_bar_->SetFdcStatus(action,real_drive,track,sector);

		//
		last_time_cycles=bbcComputer::cycles;
		last_update_sys_time_=this_time;
/*
		static int last_time_cycles=0;
		int num_cycles=bbcComputer::cycles-last_time_cycles;
		float num_seconds=(this_time-last_update_sys_time_)/1000.f;
		float mhz=num_cycles/num_seconds/1000000.f;
		static const char *fdc_actions[]={
			"idle",
			"reading",
			"writing",
			"active",
		};
		int screen_w,screen_h;
		beeb_screen_->GetClientSize(&screen_w,&screen_h);
		int real_drive=bbcFdd::DriveFromFdcAddress(drive,side);
		wxString status;
		status.Printf("Speed: %.2fMHz Disc: %s %d T%d S%d Size: %dx%d",
			mhz,fdc_actions[action],real_drive,track,sector,screen_w,screen_h);
		status_bar_->SetStatusText(status,0);
		last_time_cycles=bbcComputer::cycles;
		last_update_sys_time_=this_time;
*/
	}
	next_update_sys_=bbcComputer::cycles+update_sys_cycles;
#ifdef __WXMSW__
	//This gives slightly smoother multitasking.
	//Windows will allegedly preemptively multitask GUI apps these days,
	//seems it does it very politely :)
	Sleep(0);
#endif
}

void mbMainFrame::Idle(wxIdleEvent &event) {
	if(!beeb_paused_) {
		this->DoBeebTick();
		this->DoUpdateSys();
//		this->UpdateStatusBar();
	}
	event.RequestMore();//TODO shouldn't request more if we're being shut down?
}

void mbMainFrame::DoUpdateKeys() {
	HostInp_Poll(hostinp_);
	if(keymap_) {
		BYTE key_states[256];
		HostInp_GetKeyboardState(hostinp_,key_states);
		for(unsigned i=0;i<256;++i) {
			const mbKeymap::HostKeys *pkl=&keymap_->keymap[i];
			if(!pkl->empty()) {
				bool key_state=false;
				mbKeymap::HostKeys::const_iterator pc_key,pkl_end=pkl->end();
				for(pc_key=pkl->begin();pc_key!=pkl_end;++pc_key) {
					if(key_states[*pc_key]) {
						key_state=true;
						break;
					}
				}
				bbcKeyboardMatrix::SetKeyState(bbcKey(i),key_state);
			}
		}
	}
	bbc_system_via.DoKeyboardIrq();
	if(bbcKeyboardMatrix::KeyState(bbcKEY_BREAK)&&!debouncing_break_) {	
		bbcComputer::ResetSoft();
	} else if(!bbcKeyboardMatrix::KeyState(bbcKEY_BREAK)) {
		debouncing_break_=false;
	}
}

void mbMainFrame::InitMenuBar() {
	unsigned i;

	wxMenu *file=new wxMenu;
#ifdef ENABLE_TEST
	file->Append(id_test,"&Test...");
#endif
	file->Append(id_quickstart,"&Quickstart...","Select game to boot from menu");
	file->Append(id_hard_reset,"&Hard reset...","Reset BBC as if switched off then on");
	file->AppendSeparator();
	for(i=0;i<bbcFdd::num_drives;++i) {
		wxString drive_num=wxString::Format("%1.1d",i);
		wxString str;
		wxMenu *drive_menu;
		int base_id=id_drives_begin+i*subid_drive_num;

		drive_menu=new wxMenu;
		drive_menu->Append(base_id+subid_drive_filename,"");
		drive_menu->Enable(base_id+subid_drive_filename,false);
		drive_menu->AppendSeparator();
		drive_menu->Append(base_id+subid_drive_load,"&Load...",
			"Insert disc into drive "+drive_num);
		drive_menu->Append(base_id+subid_drive_save,"&Save",
			"Save disc currently in "+drive_num);
		drive_menu->Append(base_id+subid_drive_unload,"&Unload",
			"Remove disc from "+drive_num);
		file->Append(base_id+subid_drive,"Drive &"+drive_num,drive_menu,
			"Controls for drive "+drive_num);
	}
	file->AppendSeparator();
	file->Append(id_exit,"E&xit","Exit the program");
	
	wxMenu *options=new wxMenu;
	options->AppendCheckItem(id_toggle_sound,"&Sound enabled",
		"Whether sound will be produced");
	options->AppendCheckItem(id_toggle_limit_speed,"&Limit speed",
		"Whether emulator is limited to max real BBC speed");
	options->AppendCheckItem(id_toggle_fast_forward_disc_access,
		"&Fast forward disc access",
		"Whether emulator should run flat out when acessing the disc");
//	options->AppendCheckItem(id_toggle_vertical_2x,"&Vertical doubling",
//		"Whether to double each scanline");
	options->AppendCheckItem(id_toggle_full_screen,"&Full screen");
	options->AppendCheckItem(id_toggle_status_bar,"&Show status bar",
		"Whether status bar is visible");
	keymaps_submenu_=new wxMenu;
	options->Append(id_keymaps_submenu,"&Keymap",keymaps_submenu_,
		"Select keyboard mapping");
	wxMenu *resolution_submenu=new wxMenu;
	for(i=0;i<4;++i) {
		wxSize size=this->ResolutionFromId(id_resolution_0+i);
		resolution_submenu->Append(id_resolution_0+i,
			wxString::Format("%dx%d",size.GetWidth(),size.GetHeight()));
	}
	options->Append(id_resolution_submenu,"&Resolution",resolution_submenu,
		"Select window size");
	wxMenu *screen_type_submenu=new wxMenu;
	screen_type_submenu->AppendCheckItem(id_screen_type_colour,"&Colour",
		"Colour TV/monitor");
	screen_type_submenu->AppendCheckItem(id_screen_type_bnw,"&Black && White",
		"Black && white TV");
	screen_type_submenu->AppendCheckItem(id_screen_type_green,"&Green",
		"Green screen monitor (green only)");
	screen_type_submenu->AppendCheckItem(id_screen_type_amber,"&Amber",
		"Amber monitor");
	options->Append(id_screen_type_submenu,"S&creen type",screen_type_submenu,
		"Select screen type");

	{
		wxMenu *wsc=new wxMenu;
		wsc->AppendCheckItem(id_windowed_scanlines_single,"&Single");
		wsc->AppendCheckItem(id_windowed_scanlines_double,"&Double");
		wxMenu *fssc=new wxMenu;
		fssc->AppendCheckItem(id_full_screen_scanlines_single,"&Single");
		fssc->AppendCheckItem(id_full_screen_scanlines_interlace,"&Interlace");
		fssc->AppendCheckItem(id_full_screen_scanlines_double,"&Double");
		options->Append(id_windowed_scanlines_submenu,"&Windowed scanlines",wsc);
		options->Append(id_full_screen_scanlines_submenu,"&Full screen scanlines",
			fssc);
	}
	options->AppendSeparator();
	options->Append(id_roms_dialog,"&ROMs...","Configure sideways ROMs and OS ROM");
	options->Append(id_sound_dialog,"S&ound...","Configure sound output");
	options->Append(id_keymap_dialog,"K&eymaps...","Define keyboard mappings");
#ifdef mbENABLE_JOYSTICKS
	options->Append(id_joysticks_dialog,"&Joysticks...","Configure joysticks");
#endif
//	options->AppendCheckItem(id_toggle_toolbar,"&Toolbar","Toggle toolbar display");

#ifdef BEEB_DEBUG_ENABLE
	wxMenu *debug=new wxMenu;
#ifdef bbcDEBUG_TRACE
	debug->Append(id_start_disassembly,"St&art disassembly",
		"Start saving a running disassembly");
	debug->Append(id_stop_disassembly,"St&op disassembly",
		"Stop the running disassembly and save results to disc");
#endif
#ifdef BEEB_DEBUG_SAVE_CHALLENGER_RAM
	debug->Append(id_save_challenger_ram,"Save &CHALLENGER RAM",
		"Save CHALLENGER RAM to a file (if CHALLENGER enabled)");
#endif
#ifdef BEEB_DEBUG_DISASSEMBLE_RAM
	debug->Append(id_disassemble_ram,"&Disassemble RAM",
		"Save a disassembly of all of RAM");
#endif
#ifdef bbcDEBUG_PANELS
	debug->AppendCheckItem(id_toggle_dbgpanel_sysvia,"S&ystem VIA debug panel");
	debug->AppendCheckItem(id_toggle_dbgpanel_usrvia,"&User VIA debug panel");
	debug->AppendCheckItem(id_toggle_dbgpanel_video,"&Video debug panel");
#endif
#ifdef bbcDEBUG_VIDEO
	debug->AppendCheckItem(id_toggle_showall,"Show all");
	debug->Append(id_log_video_frames,"Log a few video frames");
#ifdef bbcDEBUG_TRACE
	debug->Append(id_log_video_frames_disasm,"Log a few video frames w/ disasm");
#endif//bbcDEBUG_TRACE
	debug->AppendCheckItem(id_toggle_show_t1_timeouts,"Show sys (R)/usr (G) T1 timeouts");
#endif//bbcDEBUG_VIDEO
#endif//BEEB_DEBUG_ENABLE

	wxMenu *special=new wxMenu;
	special->Append(id_sound_record_start,"&Start recording sound");
	special->Append(id_sound_record_stop,"Stop &recording sound...");
	special->AppendSeparator();
	special->Append(id_screenshot_save_as,"&Save screenshot...",
		"Save screen contents to disc");

	wxMenu *help=new wxMenu;
	help->Append(id_show_about_box,"&About...","Show help box");

	menu_bar_=new wxMenuBar;
	popup_menu_=new wxMenu;

	menu_bar_->Append(file,"File");
	popup_menu_->Append(id_file_menu,"File",file);
	menu_bar_->Append(options,"Options");
	popup_menu_->Append(id_options_menu,"Options",options);
	menu_bar_->Append(special,"Special");
	popup_menu_->Append(id_special_menu,"Special",special);
#ifdef BEEB_DEBUG_ENABLE
	menu_bar_->Append(debug,"Debug");
	popup_menu_->Append(id_debug_menu,"Debug",debug);
#endif
	menu_bar_->Append(help,"Help");
	popup_menu_->Append(id_help_menu,"Help",help);

	this->SetMenuBar(menu_bar_);

//	this->RefreshWindow();
}

void mbMainFrame::ModalBackground() {
	if(hostsnd_) {
		HostSnd_PlayStop(hostsnd_);
	}
}

void mbMainFrame::GetConfig(mbConfig *cfg) const {
	wxASSERT(cfg);
	*cfg=cfg_;
}

void mbMainFrame::OnMenuOpen(wxMenuEvent &event) {
	wxLogDebug("OnMenuOpen: this->ModalBackground()\n");
	this->ModalBackground();
	//HostGfx_SetFullScreen(hostgfx_,false);
}

void mbMainFrame::OnMenuClose(wxMenuEvent &event) {
//	event.Skip();
	if(hostgfx_&&HostGfx_IsFullScreen(hostgfx_)) {
		//HostGfx_Cls(hostgfx_);
		wxLogDebug("OnMenuClose: clearing beeb screen.\n");
		beeb_screen_->Clear();
	}
}

void mbMainFrame::OnExit(wxCommandEvent &event) {
	this->Close();
}

void mbMainFrame::OnToggleSound(wxCommandEvent &event) {
	this->SetSoundEnabled(!cfg_.sound.enabled);
	this->RefreshWindow();
}

void mbMainFrame::OnToggleLimitSpeed(wxCommandEvent &event) {
	cfg_.misc.limit_speed=!cfg_.misc.limit_speed;
	this->RefreshWindow();
}

void mbMainFrame::OnShowAboutBox(wxCommandEvent &event) {
	this->ModalBackground();
	wxString msg;
	
	msg+=wxString::Format("This is version %s\n",mb_version);
	msg+="\n";
	msg+="Tom Seddon <bbc@tomseddon.plus.com>\n";
	msg+="\n";
	msg+=wxString::Format("Compiled at %s on %s\n",__TIME__,__DATE__);
	wxMessageDialog about(this,msg,"Author",wxOK|wxCENTRE|wxICON_INFORMATION);
	about.ShowModal();
}

void mbMainFrame::OnPaint(wxPaintEvent &event) {
	wxPaintDC dc(this);
	if(auto_paint_) {
		if(!hostgfx_) {
			dc.SetBrush(*wxBLACK_BRUSH);
			dc.Clear();
			dc.SetBrush(wxNullBrush);
		} else if(!HostGfx_IsFullScreen(hostgfx_)) {
			HostGfx_ShowWindowed(hostgfx_);
		}
		//In full screen mode, this is just ignored.
	}
}

void mbMainFrame::OnEraseBackground(wxEraseEvent &event) {
	return;
}

void mbMainFrame::OnSize(wxSizeEvent &event) {
	this->ModalBackground();
//	wxSize new_size=event.GetSize();
//	this->SetSize(new_size.GetWidth(),400);
	event.Skip();
}

void mbMainFrame::OnMove(wxMoveEvent &event) {
	this->ModalBackground();
	event.Skip();
}

void mbMainFrame::SetAutoPaint(bool new_auto_paint) {
	auto_paint_=new_auto_paint;
}

void mbMainFrame::DoUpdateSound() {
	HostTmr_HfValue now_hf_time;

	HostTmr_ReadHf(&now_hf_time);
	
	if(cfg_.misc.limit_speed&&!beeb_fastforward_) {
		HostTmr_HfValue sound_tick_length=hf_freq_/update_sound_freq;
		while(now_hf_time-last_update_sound_hf_time_<sound_tick_length) {
#ifdef __WXMSW__
//			Sleep(0);
#endif
			HostTmr_ReadHf(&now_hf_time);
		}
/*
		wxString tmp;
		tmp.Printf("%I64d/%I64d",__int64(now_hf_time-last_update_sound_hf_time_),
			__int64(hf_freq_));
		this->SetTitle(tmp);
*/
	}

	if(!cfg_.sound.enabled||!hostsnd_) {
		BASSERT(!cfg_.sound.enabled||(cfg_.sound.enabled&&hostsnd_));
		//doesn't matter whether the 8 or 16 bit version is used here!
		bbcSound::Render16BitMono(0,0,0);
	} else {
		HostSnd_PlayStart(hostsnd_);
		void *ps[2];
		unsigned sizes[2];
		//sound_buffer_->GetCurrentPosition(&snd_play,&snd_write);
		int flag=HostSnd_GetWriteArea(hostsnd_,&ps[0],&sizes[0],&ps[1],&sizes[1]);
		if(flag<0) {
			bbcSound::Render16BitMono(0,0,0);
		} else if(flag>0) {
			static void (*(sound_renderers[]))(void **,unsigned *,unsigned)={
				&bbcSound::Render8BitMono,
				&bbcSound::Render8BitStereo,
				&bbcSound::Render16BitMono,
				&bbcSound::Render16BitStereo,
			};
			unsigned num_buffers=sizes[1]>0?2:1;
			unsigned idx=(cfg_.sound.stereo?1:0)|(cfg_.sound.bits==16?2:0);
			(*sound_renderers[idx])(ps,sizes,num_buffers);

			HostSnd_CommitWriteArea(hostsnd_);
		}
	}
	last_update_sound_hf_time_=now_hf_time;
}

void mbMainFrame::SetSoundEnabled(bool new_sound_enabled) {
	if(new_sound_enabled) {
		if(!hostsnd_) {
			OutputDebugStringA("mbMainFrame::SetSoundEnabled(true): HostSnd_Start\n");
			hostsnd_=HostSnd_Start(this);
			if(hostsnd_) {
				OutputDebugStringA("mbMainFrame::SetSoundEnabled(true): HostSnd_* init stuff\n");
				int len_samples=cfg_.sound.hz/update_sound_freq*cfg_.sound.len_frames;
				wxLogDebug("Sound: %dHz %d bits %s %d samples\n",
					cfg_.sound.hz,cfg_.sound.bits,cfg_.sound.stereo?"stereo":"mono",
					len_samples);
				HostSnd_SetSoundParameters(hostsnd_,cfg_.sound.hz,cfg_.sound.bits,
					cfg_.sound.stereo?2:1,len_samples);
				bbcSound::SetFrequency(cfg_.sound.hz);
				for(unsigned i=0;i<4;++i) {
					bbcSound::SetChannelStereoLeftRight(i,
						cfg_.sound.is_stereo_left[i],cfg_.sound.is_stereo_right[i]);
				}
				HostSnd_PlayStart(hostsnd_);
//				Sleep(10);
//				HostSnd_SetVolume(hostsnd_,cfg_.sound.volume);
			}
		}
	} else {
		if(hostsnd_) {
			OutputDebugStringA("mbMainFrame::SetSoundEnabled(false): HostSnd_Stop\n");
			HostSnd_Stop(hostsnd_);
			hostsnd_=0;
		}
	}
	cfg_.sound.enabled=new_sound_enabled;
}

void mbMainFrame::RefreshWindow() {
	//firstly, set up display
	if(hostgfx_) {
		wxLogDebug("RefreshWindow: full screen=%s\n",
			cfg_.video.full_screen?"yes":"no");
		if(cfg_.video.full_screen) {
			HostGfx_SetScanlinesMode(hostgfx_,cfg_.video.full_screen_scanlines);
		} else {
			HostGfx_SetScanlinesMode(hostgfx_,cfg_.video.windowed_scanlines);
		}
		
		bbcVideo::SetPixelFixedBits(cfg_.video.full_screen?
			HostGfx_FULL_SCREEN_PALETTE_BASE:0);
		//	status_bar_->SetResizeable()
		static const long fs_style=wxFULLSCREEN_NOBORDER|wxFULLSCREEN_NOCAPTION|
			wxFULLSCREEN_NOMENUBAR;
		if(cfg_.video.full_screen) {
			this->ShowFullScreen(true,fs_style);
			HostGfx_SetFullScreen(hostgfx_,true,cfg_.video.full_screen_refresh_rate);
		} else {
			HostGfx_SetFullScreen(hostgfx_,false,cfg_.video.full_screen_refresh_rate);
			this->ShowFullScreen(false,fs_style);
		}
	}
	if(hostsnd_) {
		HostSnd_SetVolume(hostsnd_,cfg_.sound.volume);
	}
	
	//tweak ui
	int i;

	this->SetMenuItemCheck(id_toggle_sound,cfg_.sound.enabled);
	this->SetMenuItemCheck(id_toggle_limit_speed,cfg_.misc.limit_speed);
	this->SetMenuItemCheck(id_toggle_status_bar,cfg_.misc.show_status_bar);
	this->SetMenuItemCheck(id_toggle_fast_forward_disc_access,
		cfg_.misc.fast_forward_disc_access);
	//this->SetMenuItemCheck(id_toggle_vertical_2x,cfg_.video.vertical_2x);
	this->SetMenuItemCheck(id_toggle_full_screen,cfg_.video.full_screen);
	
#ifdef bbcDEBUG_PANELS
	this->SetMenuItemCheck(id_toggle_dbgpanel_sysvia,!!this->FindWindow(id_dbgpanel_sysvia));
	this->SetMenuItemCheck(id_toggle_dbgpanel_usrvia,!!this->FindWindow(id_dbgpanel_usrvia));
	this->SetMenuItemCheck(id_toggle_dbgpanel_video,!!this->FindWindow(id_dbgpanel_video));
#endif
#ifdef bbcDEBUG_VIDEO
	this->SetMenuItemCheck(id_toggle_showall,bbcVideo::show_all_);
	this->SetMenuItemCheck(id_toggle_show_t1_timeouts,bbcVideo::show_t1_timeouts_);
#endif

	HostGfx_BeebScreenType screen_type=HostGfx_COLOUR;
	if(hostgfx_) {
		screen_type=HostGfx_ScreenType(hostgfx_);
	}
	this->SetMenuItemCheck(id_screen_type_colour,screen_type==HostGfx_COLOUR);
	this->SetMenuItemCheck(id_screen_type_bnw,screen_type==HostGfx_BNW);
	this->SetMenuItemCheck(id_screen_type_amber,screen_type==HostGfx_AMBER);
	this->SetMenuItemCheck(id_screen_type_green,screen_type==HostGfx_GREEN);

	this->SetMenuItemCheck(id_windowed_scanlines_single,
		cfg_.video.windowed_scanlines==HostGfx_SINGLE);
	this->SetMenuItemCheck(id_windowed_scanlines_double,
		cfg_.video.windowed_scanlines==HostGfx_DOUBLE);
	this->SetMenuItemCheck(id_full_screen_scanlines_single,
		cfg_.video.full_screen_scanlines==HostGfx_SINGLE);
	this->SetMenuItemCheck(id_full_screen_scanlines_interlace,
		cfg_.video.full_screen_scanlines==HostGfx_INTERLACE);
	this->SetMenuItemCheck(id_full_screen_scanlines_double,
		cfg_.video.full_screen_scanlines==HostGfx_DOUBLE);

	bool is_fs=hostgfx_&&HostGfx_IsFullScreen(hostgfx_);
//	this->SetMenuItemEnabled(id_resolution_0,!is_fs);
//	this->SetMenuItemEnabled(id_resolution_1,!is_fs);
//	this->SetMenuItemEnabled(id_resolution_2,!is_fs);
//	this->SetMenuItemEnabled(id_resolution_3,!is_fs);
	if(popup_menu_) {
		popup_menu_->Enable(id_resolution_submenu,!is_fs);//TODO doesn't work, don't know why.
	}
	this->SetMenuItemEnabled(id_resolution_submenu,!is_fs);
	
	status_bar_->Show(false);
	wxASSERT(this->GetSizer());
	this->GetSizer()->Remove(status_bar_);
	if(cfg_.misc.show_status_bar) {
		status_bar_->Show(true);
		status_bar_->SetVolumeEnabled(cfg_.sound.enabled);
		status_bar_->SetVolume(cfg_.sound.volume);
		this->GetSizer()->Add(status_bar_,0,wxEXPAND);
	}
	this->GetSizer()->Layout();

	for(i=0;i<bbcFdd::num_drives;++i) {
		bbcFdd::DriveStatusType st=bbcFdd::DriveStatus(i);
		int id_base=id_drives_begin+i*subid_drive_num;

		if(st==bbcFdd::DRIVE_SIDE2) {
			this->SetMenuItemEnabled(id_base+subid_drive,false);
		} else {
			const DriveUi *ui=&drive_uis_[i];
			this->SetMenuItemEnabled(id_base+subid_drive,true);
			bool loaded=st!=bbcFdd::DRIVE_EMPTY;
			//this->SetMenuItemEnabled(id_base+subid_drive_load,!loaded);
			this->SetMenuItemEnabled(id_base+subid_drive_save,loaded&&!ui->no_save);
			this->SetMenuItemEnabled(id_base+subid_drive_unload,loaded);

			wxMenuItem *item=menu_bar_->FindItem(id_base+subid_drive_filename);
			wxASSERT(item);
			if(!loaded) {
				item->SetText("");
			} else {
				item->SetText(ui->disc_image.GetFullPath());
			}
		}
	}

	//Update keymaps

	wxMenuItemList &menu_items=keymaps_submenu_->GetMenuItems();
	if(menu_items.GetCount()>0) {
		//i'm a bit concerned about the deletion changing menu_items...
		std::vector<wxMenuItem *> items;
		for(wxMenuItemList::Node *node=menu_items.GetFirst();node;
			node=node->GetNext())
		{
			items.push_back(node->GetData());
		}
		for(i=0;i<items.size();++i) {
			keymaps_submenu_->Delete(items[i]);
		}
	}
	i=id_first_keymap;
	mbKeymap::C::const_iterator km;
	for(km=cfg_.keyboard.keymaps.begin();km!=cfg_.keyboard.keymaps.end();++km) {
		if(i>=id_last_keymap) {
			//You don't get any more than that!!
			break;
		}
		keymaps_submenu_->AppendCheckItem(i,km->name);
		keymaps_submenu_->Check(i,km->name==cfg_.keyboard.selected_keymap);
		++i;
	}

	if(this->GetMenuBar()) {
		wxASSERT(this->GetMenuBar()==menu_bar_);
		menu_bar_->Refresh();
	}

	//Clear any rubbish that may have accumulated.
	//This is done too often. It doesn't matter.
	beeb_screen_->Clear();
}

void mbMainFrame::SetMenuItemEnabled(int menu_item_id,bool enabled) {
//	wxMenuBar *menu_bar=this->GetMenuBar();
//	BASSERT(menu_bar);
//	if(menu_bar) {
	wxMenuItem *menu_item=menu_bar_->FindItem(menu_item_id);
	BASSERT(menu_item);
	menu_item->Enable(enabled);
//	}
}

void mbMainFrame::SetMenuItemCheck(int menu_item_id,bool checked) {
//	wxMenuBar *menu_bar=this->GetMenuBar();
//	BASSERT(menu_bar);
//	if(menu_bar) {
	wxMenuItem *menu_item=menu_bar_->FindItem(menu_item_id);
	BASSERT(menu_item);
	menu_item->Check(checked);
//	}
}

void mbMainFrame::OnDriveLoadInternal(wxCommandEvent &event,int drive) {
	wxString dialog_wildcards;
	wxString all_wildcards;
	std::vector<mbDiscFormat> fmts;

	//
	mbDiscFormatGetAll(&fmts);
	for(unsigned i=0;i<fmts.size();++i) {
		wxString wildcards;
		for(unsigned j=0;j<fmts[i].extensions.size();++j) {
			if(!wildcards.empty()) {
				wildcards+=";";
			}
			wildcards+="*."+fmts[i].extensions[j];
		}
		if(!all_wildcards.empty()) {
			all_wildcards+=";";
		}
		all_wildcards+=wildcards;
		dialog_wildcards+=fmts[i].description+"("+wildcards+")|"+wildcards+"|";
	}
	dialog_wildcards="Autodetect ("+all_wildcards+")|"+all_wildcards+"|"+dialog_wildcards;

	//
	DriveUi *ui=&drive_uis_[drive];
	wxFileDialog fd(this,"Select disc",ui->last_path,"",dialog_wildcards,
		wxOPEN|wxHIDE_READONLY);//the read only flag doesn't do anything :(
	fd.SetFilterIndex(ui->last_filter_index);
	int r=fd.ShowModal();
	if(r==wxID_CANCEL) {
		return;
	}

	//
	ui->last_filter_index=fd.GetFilterIndex();
	ui->last_path=fd.GetPath();
	this->UiLoadDisc(drive,fd.GetPath());
}

//////////////////////////////////////////////////////////////////////////
// UiUnloadDrive -- does user interface for unloading a drive, if that's
// necessary. Does not handle side 2 drives...
bool mbMainFrame::UiUnloadDrive(int drive) {
	int specified_drive=drive;
	BASSERT(drive>=0);

	//Early out if drive empty.
	bbcFdd::DriveStatusType stdrive=bbcFdd::DriveStatus(drive);
	if(stdrive==bbcFdd::DRIVE_EMPTY) {
		return true;
	}

	//'drive' should be the top side.
	BASSERT(stdrive!=bbcFdd::DRIVE_SIDE2);

	int other=bbcFdd::other_side_of[drive];
	bbcFdd::DriveStatusType stother=bbcFdd::DriveStatus(other);
	bool is_ds=stother==bbcFdd::DRIVE_SIDE2;

	if(stdrive==bbcFdd::DRIVE_CHANGED) {
		wxString msg;
		
		msg+=wxString::Format("Drive %d contains \"%s\".\n",
			drive,drive_uis_[drive].disc_image.GetFullPath().c_str());
		if(is_ds) {
			msg+=wxString::Format("(Unloading drive %d will also unload drive %d.)\n\n",
				drive,other);
		}
		msg+="This disc image has changed.\n";
		msg+="If you continue, these changes will be lost.\n\n";
		msg+="Do you want to continue?";
		int result=wxMessageBox(msg,wxString::Format("Unload drive %d",drive),
			wxYES_NO,this);
		return result==wxYES;
	}
	return true;
}

bool mbMainFrame::UiLoadDisc(int drive,wxFileName file_name) {
	wxString name=file_name.GetFullPath();

	//Determine the geometry of this disc image.
	mbDiscFormat fmt;
	if(!mbDiscFormatGetFromFileName(file_name,&fmt)) {
		wxLogError("Cannot determine disc format from filename \"%s\"\n",
			name.c_str());
		return false;
	}
//	const BeebDiscGeometry *geometry=DiscImageGeometry(file_name);
//	if(!geometry) {
//		wxLogError("Cannot determine disc geometry from filename \"%s\"\n",
//			name.c_str());
//		return false;
//	}
		
	//Load disc image
	std::vector<t65::byte> contents;
	OutputDebugStringA(name.c_str());
	OutputDebugStringA("\n");
	if(!file_name.IsOk()) {
		wxLogError("Invalid filename \"%s\"\n",name.c_str());
		return false;
	}
	bool loaded_ok=mbLoadFile(name,contents);
	if(!loaded_ok) {
		wxLogError("Failed to load \"%s\"\n",name.c_str());
		return false;
	}
	file_name.Normalize();

	bool ok=false;
	switch(fmt.sides) {
	case 1:
		{
			if(!this->UiUnloadDrive(drive)) {
				return false;
			}
			bbcFdd::UnloadDiscImage(drive);
			ok=bbcFdd::LoadSsDiscImage(drive,fmt.sectors,contents);
			wxLogDebug("mbMainFrame::UiLoadDisc: load drive %d from \"%s\"\n",drive,name.c_str());
			if(!ok) {
				wxLogError("Failed to load drive %d from \"%s\"\n",drive,name.c_str());
			}
		}
		break;
	case 2:
		{
			int other=bbcFdd::other_side_of[drive];
			bool is_ds=bbcFdd::DriveStatus(other)==bbcFdd::DRIVE_SIDE2;
			if(!this->UiUnloadDrive(drive)||(is_ds||!this->UiUnloadDrive(other))) {
				return false;
			}
			//'drive' is top
			if(other<drive) {
				std::swap(other,drive);
			}
			bbcFdd::UnloadDiscImage(drive);
			bbcFdd::UnloadDiscImage(other);
			wxLogDebug("mbMainFrame::UiLoadDisc: load drive %d from \"%s\"\n",drive,name.c_str());
			ok=bbcFdd::LoadDsDiscImage(drive,fmt.sectors,contents);
			if(!ok) {
				wxLogError("Failed to load drives %d+%d from \"%s\"\n",
					drive,other,name.c_str());
			}
		}
		break;
	}
	if(ok) {
		//Assign to that drive
		drive_uis_[drive].disc_image=file_name;
		drive_uis_[drive].no_save=false;
	}
	this->RefreshWindow();
	return ok;
}

void mbMainFrame::OnDriveLoad(wxCommandEvent &event,int drive) {
	this->ModalBackground();
//	wxASSERT(bbcFdd::DriveStatus(drive)==bbcFdd::DRIVE_EMPTY);
	this->OnDriveLoadInternal(event,drive);
	this->RefreshWindow();
}

void mbMainFrame::OnDriveSave(wxCommandEvent &event,int drive) {
	this->ModalBackground();
	bbcFdd::DriveStatusType status=bbcFdd::DriveStatus(drive);
	wxASSERT(status!=bbcFdd::DRIVE_EMPTY);
	wxASSERT(status!=bbcFdd::DRIVE_SIDE2);
	bool ok;
	std::vector<t65::byte> contents;
	if(bbcFdd::IsDs(drive)) {
		int other=bbcFdd::other_side_of[drive];
		ok=bbcFdd::GetDsDiscImage(drive,other,&contents);
	} else {
		ok=bbcFdd::GetSsDiscImage(drive,&contents);
	}
	//bbcFdd::GetDiscImage(drive,contents);
	wxString filename=drive_uis_[drive].disc_image.GetFullPath();
	if(!ok||!mbSaveFile(filename,contents)) {
		wxLogWarning("Failed to save file\n%s",filename.c_str());
	} else {
		bbcFdd::SetDriveChangedStatus(drive,false);
	}
}

void mbMainFrame::OnDriveXCommand(wxCommandEvent &event) {
	wxASSERT(event.GetId()>=id_drives_begin&&event.GetId()<id_drives_end);
	int id=event.GetId();
	id-=id_drives_begin;
	int subaction=id%subid_drive_num;
	int drive=id/subid_drive_num;
	if(drive_subactions[subaction]) {
		(this->*drive_subactions[subaction])(event,drive);
	}
}

void mbMainFrame::OnDriveUnload(wxCommandEvent &event,int drive) {
	bbcFdd::DriveStatusType status=bbcFdd::DriveStatus(drive);
	wxASSERT(status!=bbcFdd::DRIVE_EMPTY);
	if(status==bbcFdd::DRIVE_CHANGED) {
		static const wxString msg=
			"If this drive is unloaded, changes will be lost.\n"
			"\n"
			"Are you sure?";
		int r=wxMessageBox(msg,wxString::Format("Unload drive %d",drive),wxYES_NO);
		if(r!=wxYES) {
			return;
		}
	}
	bbcFdd::UnloadDiscImage(drive);
	//drive_uis_[drive].loaded=false;
	this->RefreshWindow();
	return;
}

void mbMainFrame::OnQuickstart(wxCommandEvent &event) {
	this->ModalBackground();
	mbQuickstartResult r;
	if(mbQuickstart(this,cfg_.quickstart,&r)) {
		if(this->UiLoadDisc(0,r.drive0)&&
			(!r.drive2.IsOk()||this->UiLoadDisc(2,r.drive2)))
		{
#ifdef bbcQUICKSTART_BY_SHIFT
			bbcKeyboardMatrix::SetForceShiftPressed(true);
#else
			bbcKeyboardMatrix::SetBootKeylinkOneoff();
#endif
			bbcComputer::ResetHard();//Power on reset so it does the keylinks thing.
		}
	}
}

#ifdef ENABLE_TEST
void mbMainFrame::OnTest(wxCommandEvent &event) {
	this->UiLoadDisc(0,wxFileName::FileName("z:\\beeb\\disctests\\arcadians.ssd"));
	this->UiLoadDisc(1,wxFileName::FileName("z:\\beeb\\disctests\\acorn.ssd"));
}
#endif
	
void mbMainFrame::OnKeymapDialog(wxCommandEvent &event) {
	beeb_paused_=true;
	this->ModalBackground();
	wxString old_selected_keymap=cfg_.keyboard.selected_keymap;
	mbKeyboardConfigDialog dlg(this,cfg_.keyboard);
	dlg.ShowModal();
	dlg.GetResult(&cfg_.keyboard);
	this->SetKeymap(old_selected_keymap);
	this->RefreshWindow();
	beeb_paused_=false;
}

void mbMainFrame::OnSoundDialog(wxCommandEvent &event) {
	beeb_paused_=true;
	this->ModalBackground();
	mbSoundConfigDialog snd_dlg(this,cfg_.sound);
	if(snd_dlg.ShowModal()==wxID_OK) {
		snd_dlg.GetResult(&cfg_.sound);
		bool old_enabled=cfg_.sound.enabled;
		this->SetSoundEnabled(false);
		this->SetSoundEnabled(old_enabled);
		this->RefreshWindow();
	}
	beeb_paused_=false;
}

void mbMainFrame::OnRomsDialog(wxCommandEvent &event) {
	beeb_paused_=true;
	this->ModalBackground();
	mbRomsConfigDialog roms_dlg(this,-1,cfg_.roms);
	int result=roms_dlg.ShowModal();
	if(result==wxID_OK||result==wxID_APPLY) {
		roms_dlg.GetResult(&cfg_.roms);
	}
	if(result==wxID_APPLY) {
		this->ApplyConfig();
	}
	beeb_paused_=false;
}

void mbMainFrame::OnKeymap(wxCommandEvent &event) {
	int id=event.GetId();
//	wxMenuBar *menu=this->GetMenuBar();
//	if(menu) {
	wxMenuItem *keymap=menu_bar_->FindItem(id);
	if(keymap) {
		this->SetKeymap(keymap->GetText());
		this->RefreshWindow();
	}
//	}
}

#ifdef bbcDEBUG_TRACE
void mbMainFrame::OnStartDisassembly(wxCommandEvent &event) {
	if(!bbcComputer::trace) {
		bbcComputer::trace=new bbcDebugTrace;
	}
}

static bool ProgressIncProgressDialog(int percent_done,void *context) {
	static_cast<wxProgressDialog *>(context)->Update(percent_done);
	return true;
}

void mbMainFrame::OnStopDisassembly(wxCommandEvent &event) {
	if(!bbcComputer::trace) {
		return;
	}
	FILE *h=0;
	for(;;) {
		wxString name=wxFileSelector("Save disassembly to file",0,0,0,
			wxFileSelectorDefaultWildcardStr,wxSAVE|wxOVERWRITE_PROMPT,this);
		if(name.empty()) {
			break;
		}
		h=fopen(name.c_str(),"wt");
		if(true) {//h) {
			//should have a 'try again' box here in case NAV goes mental as
			//usual.
			break;
		}
	}
	if(h) {
		wxProgressDialog progress("Please wait...","Saving disassembly.",100,this);
		if(!bbcComputer::trace->WriteTrace(h,&ProgressIncProgressDialog,&progress)) {
			wxLogMessage("Save was cancelled.");
		}
		fclose(h);
	}

	wxProgressDialog progress("Please wait...","Freeing memory...",-1,this);
	delete bbcComputer::trace;
	bbcComputer::trace=0;
}
#endif//BEEB_DEBUG_DISASSEMBLY

/*
static const char hex_digits[]="0123456789ABCDEF";
const HistoryIrq *irq=&ent->irq;
const HistoryInstr *ins=&ent->instr;

  switch(ent->type) {
  case HIST_IRQ:
		fprintf(h,"%-9d (@%-9d) IRQ: flags 0x%X, SysVIA: IER=&%02X IFR=&%02X ACR=&%02X, UsrVIA: IER=&%02X IFR=&%02X ACR=&%02X, 1770: Status=&%02X\n",
		ent->cycles,ent->next_stop,
		irq->irqflags,
		irq->sys.ier,irq->sys.ifr,irq->sys.acr,
		irq->usr.ier,irq->usr.ifr,irq->usr.acr,
		irq->fdc_status);
		break;
		case HIST_INSTR:
		{
		char opcode_buf[10],operand_buf[15];
		int len=bbcModelBSim::DisassemblerType::Disassemble(ins->pc,ins->bytes,
		opcode_buf,operand_buf);
		const char *nullbytes="  ";
		char bytes[3][3];
		for(int i=0;i<3;++i) {
		bytes[i][0]=hex_digits[ins->bytes[i]>>4];
		bytes[i][1]=hex_digits[ins->bytes[i]&0xf];
		bytes[i][2]=0;
		}
		
		  fprintf(h,"%-9d (@%-9d) %-4X: %-5s %-10s %s %s %s -> A:%02X X:%02X Y:%02X S:%02X P:%c%c%c%c%c%c%c%c\n",
		  ent->cycles,ent->next_stop,
		  ins->pc.w,opcode_buf,operand_buf,
		  len>0?bytes[0]:nullbytes,
		  len>1?bytes[1]:nullbytes,
		  len>2?bytes[2]:nullbytes,
		  ins->a,ins->x,ins->y,ins->s,
		  ins->p&t65::N_MASK?'N':'-',
		  ins->p&t65::V_MASK?'V':'-',
		  ins->p&t65::U_MASK?'-':'-',
		  ins->p&t65::B_MASK?'B':'-',
		  ins->p&t65::D_MASK?'D':'-',
		  ins->p&t65::I_MASK?'I':'-',
		  ins->p&t65::Z_MASK?'Z':'-',
		  ins->p&t65::C_MASK?'C':'-');
		  }
		  break;
		  default:
		  wxASSERT(false);
		  }
*/

////////////////////////////////////////////////////////////////////////
//
#ifdef BEEB_DEBUG_SAVE_CHALLENGER_RAM
void mbMainFrame::OnSaveChallengerRam(wxCommandEvent &event) {
	wxString name=wxFileSelector("Save Challenger RAM to file",0,0,0,
		wxFileSelectorDefaultWildcardStr,wxSAVE|wxOVERWRITE_PROMPT,this);
	if(!name.empty()) {
		if(!mbSaveFile(name,bbc1770OpusChallenger::ram_)) {
			wxLogError("Failed to save \"%s\"",name.c_str());
		}
	}
}
#endif

//////////////////////////////////////////////////////////////////////////
//
void mbMainFrame::OnResolution(wxCommandEvent &event) {
	this->SetBeebScreenSize(this->ResolutionFromId(event.GetId()));
}

//////////////////////////////////////////////////////////////////////////
//
void mbMainFrame::SetBeebScreenSize(const wxSize &screen_size) {
	if(hostgfx_&&HostGfx_IsFullScreen(hostgfx_)) {
		return;
	}
	wxSize deltas=this->GetSize()-beeb_screen_->GetSize();
	wxASSERT(deltas.GetWidth()>=0&&deltas.GetHeight()>=0);
	wxSize size=deltas+screen_size;
	wxLogDebug("mbMainFrame::OnResolution: beeb screen=%dx%d\n",
		screen_size.GetWidth(),screen_size.GetHeight());
	wxLogDebug("mbMainFrame::OnResolution: new window=%dx%d\n",
		size.GetWidth(),size.GetHeight());
	this->SetSize(size);
}

wxSize mbMainFrame::ResolutionFromId(int id) {
	wxASSERT(id>=id_resolution_0&&id<=id_resolution_3);
	id-=id_resolution_0;
	wxSize s(cfg_.video.width,cfg_.video.height);
	if(!(id&2)) {
		s.x/=2;
	}
	if(id&1) {
		s.y*=2;
	}
	return s;
}

//////////////////////////////////////////////////////////////////////////
//
void mbMainFrame::OnScreenshotSaveAs(wxCommandEvent &event) {
	screenshot_countdown_=0;
	wxString wildcards;
	wildcards+="Auto (*.*)|*.*";
	wxFileDialog fd(this,"Select file","","",wildcards,wxSAVE);
	int r=fd.ShowModal();
	if(r==wxID_CANCEL) {
		return;
	}
	screenshot_countdown_=2;
	screenshot_name_=fd.GetPath();
}

//////////////////////////////////////////////////////////////////////////
//
void mbMainFrame::DoScreenshot() {
	wxImage *image=new wxImage(bbcVideo::buffer_width,bbcVideo::buffer_height);
	const t65::byte *src=bbcVideo::buffer;
	for(int y=0;y<bbcVideo::buffer_height;++y) {
		for(int x=0;x<bbcVideo::buffer_width;++x) {
			unsigned char r=*src&1?255:0;
			unsigned char g=*src&2?255:0;
			unsigned char b=*src&4?255:0;
			++src;
			image->SetRGB(x,y,r,g,b);
		}
	}
	if(!image->SaveFile(screenshot_name_)) {
		wxLogError("Failed to save image\n\"%s\"\n",screenshot_name_.c_str());
	}
	delete image;
}

//////////////////////////////////////////////////////////////////////////
//
void mbMainFrame::OnToggleStatusBar(wxCommandEvent &event) {
	cfg_.misc.show_status_bar=!cfg_.misc.show_status_bar;
	this->RefreshWindow();
}

//////////////////////////////////////////////////////////////////////////
//
void mbMainFrame::OnToggleFastForwardDiscAccess(wxCommandEvent &event) {
	cfg_.misc.fast_forward_disc_access=!cfg_.misc.fast_forward_disc_access;
	this->RefreshWindow();
}

//////////////////////////////////////////////////////////////////////////
//
#ifdef bbcDEBUG_PANELS
static mbDebugPanel *ViaPanelInit(wxFrame *frame,long id,const char *t,bbcVIA *via) {
	mbDebugPanel *panel=static_cast<mbDebugPanel *>(frame->FindWindow(id));
	if(panel) {
		panel->Destroy();
		panel=0;
	} else {
		panel=new mbDebugPanel(frame,id,t);
		via->InitDebugPanel(panel->Panel());
	}
	return panel;
}

void mbMainFrame::OnToggleDbgpanelSysvia(wxCommandEvent &event) {
	ViaPanelInit(this,id_dbgpanel_sysvia,"System VIA",&bbc_system_via);
}

void mbMainFrame::OnToggleDbgpanelUsrvia(wxCommandEvent &event) {
	ViaPanelInit(this,id_dbgpanel_usrvia,"User VIA",&bbc_user_via);
}

void mbMainFrame::OnToggleDbgpanelVideo(wxCommandEvent &event) {
	mbDebugPanel *panel=
		static_cast<mbDebugPanel *>(this->FindWindow(id_dbgpanel_video));

	if(panel) {
		panel->Destroy();
		panel=0;
	} else {
		panel=new mbDebugPanel(this,id_dbgpanel_video,"Video System");
		panel->Panel()->SetSize(52,16);
	}
}
#endif

#ifdef bbcDEBUG_VIDEO
void mbMainFrame::OnToggleShowAll(wxCommandEvent &event) {
	bbcVideo::SetShowAll(!bbcVideo::show_all_);
	this->RefreshWindow();
}

void mbMainFrame::OnLogVideoFrames(wxCommandEvent &event) {
#ifdef bbcDEBUG_TRACE
	bool with_disassembly=event.GetId()==id_log_video_frames_disasm;
#else
	bool with_disassembly=false;
#endif
	wxString caption="Log frames";
	if(with_disassembly) {
		caption+=" w/ disasm";
	}
	wxString name=wxFileSelector(caption,0,0,0,
		"Text file (*.txt)|*.txt|",wxSAVE|wxOVERWRITE_PROMPT,this);
	if(!name.empty()) {
		bbcVideo::LogFrames(name.c_str(),5,with_disassembly);
	}
}

void mbMainFrame::OnToggleShowT1Timeouts(wxCommandEvent &event) {
	bbcVideo::show_t1_timeouts_=!bbcVideo::show_t1_timeouts_;
}
#endif

//////////////////////////////////////////////////////////////////////////
// Sound recording format:
//
// 32 bytes header:
//
//		+0: "BBCSoundWriteEntries\x0"
//		+22:spare
//		+32:first sound write entry
//
// Each sound write entry is 8 bytes. All values little endian.
//
//		+0:	4 bytes, 250Hz counter
//		+4:	1 byte, value written to sound chip
//		+5:	3 bytes, spare

void mbMainFrame::OnSoundRecordStart(wxCommandEvent &event) {
	bbcSound::StartRecording();
	this->RefreshWindow();
}

void mbMainFrame::OnSoundRecordStop(wxCommandEvent &event) {
	if(!bbcSound::IsRecording()) {
		return;
	}
	std::vector<bbcSound::SoundWriteEntry> ents;
	bbcSound::StopRecording();
	wxString name=wxFileSelector("Save VGM file",0,0,0,
		"Video Game Music file (*.vgm)|*.vgm|",wxSAVE|wxOVERWRITE_PROMPT,this);
	if(!name.empty()) {
		std::vector<t65::byte> binary;
		bbcSound::GetRecordedVgmData(&binary);
		if(!mbSaveFile(name,binary)) {
			wxLogError("Failed to save sound file \"%s\".\n",name.c_str());
		}
	}
}

//////////////////////////////////////////////////////////////////////////
// Hard reset -- power on type thing.
void mbMainFrame::OnHardReset(wxCommandEvent &event) {
	int result=wxMessageBox("This will totally reset the computer\n\nAre you sure?",
		"Hard reset",wxYES_NO|wxICON_QUESTION,this);
	if(result==wxYES) {
		bbcComputer::ResetHard();
	}
}

//////////////////////////////////////////////////////////////////////////
//
void mbMainFrame::OnSetScreenType(wxCommandEvent &event) {
	switch(event.GetId()) {
	case id_screen_type_colour:
		HostGfx_SetScreenType(hostgfx_,HostGfx_COLOUR);
		break;
	case id_screen_type_bnw:
		HostGfx_SetScreenType(hostgfx_,HostGfx_BNW);
		break;
	case id_screen_type_amber:
		HostGfx_SetScreenType(hostgfx_,HostGfx_AMBER);
		break;
	case id_screen_type_green:
		HostGfx_SetScreenType(hostgfx_,HostGfx_GREEN);
		break;
	}
	this->RefreshWindow();
}

//////////////////////////////////////////////////////////////////////////
//
void mbMainFrame::OnVolume(wxCommandEvent &event) {
	//printf("mbMainFrame::OnVolume: volume=%ld\n",event.m_extraLong);
	wxASSERT(event.m_extraLong>=0&&event.m_extraLong<=100);
	cfg_.sound.volume=event.m_extraLong;
	HostSnd_SetVolume(hostsnd_,cfg_.sound.volume);
}

//////////////////////////////////////////////////////////////////////////
//
void mbMainFrame::OnDropFile(wxCommandEvent &event) {
	this->UiLoadDisc(event.m_commandInt,event.m_commandString);
}

//////////////////////////////////////////////////////////////////////////
//
/*
void mbMainFrame::OnToggleVertical2x(wxCommandEvent &event) {
	cfg_.video.vertical_2x=!cfg_.video.vertical_2x;
	this->RefreshWindow();
}
*/
//////////////////////////////////////////////////////////////////////////
//
void mbMainFrame::OnActivate(wxActivateEvent &event) {
	//TODO for now this bit of code is a bit dumb.
	cfg_.video.full_screen=event.GetActive()&&cfg_.video.full_screen;
	this->RefreshWindow();
}

//////////////////////////////////////////////////////////////////////////
//
void mbMainFrame::OnToggleFullScreen(wxCommandEvent &event) {
	cfg_.video.full_screen=!cfg_.video.full_screen;
	this->RefreshWindow();
}

//////////////////////////////////////////////////////////////////////////
//
void mbMainFrame::OnScanlines(wxCommandEvent &event) {
	switch(event.GetId()) {
	case id_windowed_scanlines_single:
		cfg_.video.windowed_scanlines=HostGfx_SINGLE;
		break;
	case id_windowed_scanlines_double:
		cfg_.video.windowed_scanlines=HostGfx_DOUBLE;
		break;
	case id_full_screen_scanlines_single:
		cfg_.video.full_screen_scanlines=HostGfx_SINGLE;
		break;
	case id_full_screen_scanlines_interlace:
		cfg_.video.full_screen_scanlines=HostGfx_INTERLACE;
		break;
	case id_full_screen_scanlines_double:
		cfg_.video.full_screen_scanlines=HostGfx_DOUBLE;
		break;
	}
	this->RefreshWindow();
}

//////////////////////////////////////////////////////////////////////////
//
void mbMainFrame::OnJoysticksDialog(wxCommandEvent &event) {
	beeb_paused_=true;
	this->ModalBackground();
//	wxString old_selected_keymap=cfg_.keyboard.selected_keymap;
//	mbKeyboardConfigDialog dlg(this,cfg_.keyboard);
//	dlg.ShowModal();
//	dlg.GetResult(&cfg_.keyboard);
//	this->SetKeymap(old_selected_keymap);

	mbJoysticksConfigDialog dlg(this);
	dlg.ShowModal();

	this->RefreshWindow();
	beeb_paused_=false;
}

//////////////////////////////////////////////////////////////////////////
//
void mbMainFrame::OnRightUp(wxMouseEvent &event) {
//	if(hostgfx_&&HostGfx_IsFullScreen(hostgfx_)) {
	this->PopupMenu(popup_menu_,event.GetPosition());
//	}
}

//////////////////////////////////////////////////////////////////////////
//
BEGIN_EVENT_TABLE(mbMainFrame,wxFrame)
	EVT_IDLE(mbMainFrame::Idle)
	EVT_SIZE(mbMainFrame::OnSize)
	EVT_MOVE(mbMainFrame::OnMove)
	EVT_CLOSE(mbMainFrame::OnClose)
	EVT_ERASE_BACKGROUND(mbMainFrame::OnEraseBackground)
	EVT_PAINT(mbMainFrame::OnPaint)
	EVT_MENU_OPEN(mbMainFrame::OnMenuOpen)
	EVT_MENU_CLOSE(mbMainFrame::OnMenuClose)
	EVT_MENU(id_exit,mbMainFrame::OnExit)
	EVT_MENU(id_quickstart,mbMainFrame::OnQuickstart)
#ifdef ENABLE_TEST
	EVT_MENU(id_test,mbMainFrame::OnTest)
#endif
	EVT_MENU(id_toggle_sound,mbMainFrame::OnToggleSound)
	EVT_MENU(id_toggle_limit_speed,mbMainFrame::OnToggleLimitSpeed)
	EVT_MENU(id_toggle_status_bar,mbMainFrame::OnToggleStatusBar)
	EVT_MENU(id_toggle_fast_forward_disc_access,mbMainFrame::OnToggleFastForwardDiscAccess)
	EVT_MENU(id_show_about_box,mbMainFrame::OnShowAboutBox)
	EVT_MENU(id_roms_dialog,mbMainFrame::OnRomsDialog)
	EVT_MENU(id_sound_dialog,mbMainFrame::OnSoundDialog)
	EVT_MENU(id_keymap_dialog,mbMainFrame::OnKeymapDialog)

#ifdef bbcDEBUG_TRACE
	EVT_MENU(id_start_disassembly,mbMainFrame::OnStartDisassembly)
	EVT_MENU(id_stop_disassembly,mbMainFrame::OnStopDisassembly)
#endif

#ifdef BEEB_DEBUG_SAVE_CHALLENGER_RAM
	EVT_MENU(id_save_challenger_ram,mbMainFrame::OnSaveChallengerRam)
#endif
#ifdef BEEB_DEBUG_DISASSEMBLE_RAM
	EVT_MENU(id_disassemble_ram,mbMainFrame::OnDisassembleRam)
#endif
#ifdef bbcDEBUG_PANELS
	EVT_MENU(id_toggle_dbgpanel_sysvia,mbMainFrame::OnToggleDbgpanelSysvia)
	EVT_MENU(id_toggle_dbgpanel_usrvia,mbMainFrame::OnToggleDbgpanelUsrvia)
	EVT_MENU(id_toggle_dbgpanel_video,mbMainFrame::OnToggleDbgpanelVideo)
#endif
#ifdef bbcDEBUG_VIDEO
	EVT_MENU(id_toggle_showall,mbMainFrame::OnToggleShowAll)
	EVT_MENU(id_log_video_frames,mbMainFrame::OnLogVideoFrames)
#ifdef bbcDEBUG_TRACE
	EVT_MENU(id_log_video_frames_disasm,mbMainFrame::OnLogVideoFrames)
#endif
	EVT_MENU(id_toggle_show_t1_timeouts,mbMainFrame::OnToggleShowT1Timeouts)
#endif
	EVT_MENU(id_screenshot_save_as,mbMainFrame::OnScreenshotSaveAs)
	EVT_MENU_RANGE(id_first_keymap,id_last_keymap,mbMainFrame::OnKeymap)

	EVT_MENU_RANGE(id_resolution_0,id_resolution_3,mbMainFrame::OnResolution)

	EVT_MENU_RANGE(id_drives_begin,id_drives_end,mbMainFrame::OnDriveXCommand)

	EVT_MENU(id_sound_record_start,mbMainFrame::OnSoundRecordStart)
	EVT_MENU(id_sound_record_stop,mbMainFrame::OnSoundRecordStop)
	EVT_MENU(id_hard_reset,mbMainFrame::OnHardReset)
	EVT_MENU_RANGE(id_screen_types_begin,id_screen_types_end,mbMainFrame::OnSetScreenType)
	EVT_COMMAND(id_status_bar,mbEVT_VOLUME,mbMainFrame::OnVolume)
	EVT_COMMAND(id_status_bar,mbEVT_DROPFILE,mbMainFrame::OnDropFile)
//	EVT_MENU(id_toggle_vertical_2x,mbMainFrame::OnToggleVertical2x)
	EVT_MENU(id_toggle_full_screen,mbMainFrame::OnToggleFullScreen)
	EVT_ACTIVATE(mbMainFrame::OnActivate)
	EVT_MENU_RANGE(id_scanlines_begin,id_scanlines_end,mbMainFrame::OnScanlines)
	EVT_MENU(id_joysticks_dialog,mbMainFrame::OnJoysticksDialog)
	EVT_RIGHT_UP(mbMainFrame::OnRightUp)
END_EVENT_TABLE()


