#ifndef bbc65C12_H_
#define bbc65C12_H_

//Master 128 65C12
//
//Same as t65::Generic6502, but:
//
//1.	BRA/CLR/DEA/INA/PHX/PHY/PLX/PLY/TRB/TSB
//2.	Illegal opcodes are NOP
//
//and (64doc):
//
/*
Different CPU types

  The Rockwell data booklet 29651N52 (technical information about R65C00 
  microprocessors, dated October 1984), lists the following differences between
  NMOS R6502 microprocessor and CMOS R65C00 family:
  
1. Indexed addressing across page boundary. 
NMOS: Extra read of invalid address.
CMOS: Extra read of last instruction byte. TODO doubt this matters much... right?

2. Execution of invalid op codes.
NMOS: Some terminate only by reset. Results are undefined.
CMOS: All are NOPs (reserved for future use). DONE

3. Jump indirect, operand = XXFF.
NMOS: Page address does not increment.
CMOS: Page address increments and adds one additional cycle. DONE TODO extra read?

4. Read/modify/write instructions at effective address.
NMOS: One read and two write cycles.
CMOS: Two read and one write cycle. TODO sort this if it becomes necessary!

5. Decimal flag.
NMOS: Indeterminate after reset.
CMOS: Initialized to binary mode (D=0) after reset and interrupts. TODO what, after IRQ as well?!?!?!

6. Flags after decimal operation.
NMOS: Invalid N, V and Z flags.
CMOS: Valid flag adds one additional cycle. TODO do this.

7. Interrupt after fetch of BRK instruction.
NMOS: Interrupt vector is loaded, BRK vector is ignored.
CMOS: BRK is executed, then interrupt is executed. TODO can ignore this I think.
				
*/
#include <t65Generic6502.h>

//TODO this copied from bbcSY6502A.H, to be transferred into tom6502 one day...
#define START_INSTR(N) struct Instr##N {static const char *Mnemonic() {return #N "*";}
#define END_INSTR() };

template<class T>
struct bbc65C12:
public t65::Generic6502<T>
{
public:
	//////////////////////////////////////////////////////////////////////////
	//Illegal opcodes are NOP.
	typedef InstrNOP InstrILL;//I'm not sure that I like this.

	//////////////////////////////////////////////////////////////////////////
	//New instructions
	//BRA -- branch unconditional
	struct InstrBRA:
	public InstrBranch//<0,0> TODO what the chuff is going on here?!?!?!
	{
		static const char *Mnemonic() {
			return "BRA";
		}
	};

	//TRB -- RMW, M=(A^0xFF)&M, Z according to (A&M)
	//TODO (TSB is same) MasterAdvRef says "Z if (A&M)==0" -- is it only ever
	//set?
	START_INSTR(TRB)
		static FINLINE byte Execute(MachineType &state,byte val) {
			state.cpustate.p&=~Z_MASK;
			state.cpustate.p|=((state.cpustate.a&val)==0)<<1;//NB true/false used
			return (state.cpustate.a^0xff)&val;
		}
	END_INSTR()

	//TSB -- RMW, M=(A|M), Z according to (A&M)
	START_INSTR(TSB)
		static FINLINE byte Execute(MachineType &state,byte val) {
			state.cpustate.p&=~Z_MASK;
			state.cpustate.p|=((state.cpustate.a&val)==0)<<1;//NB true/false used
			return (state.cpustate.a|val);
		}
	END_INSTR()

	//CLR/STZ -- store 0 to memory
	START_INSTR(STZ)
		static FINLINE byte Execute(MachineType &state) {
			return 0;
		}
	END_INSTR()

	//PLX/PLY/PHX/PHY -- push and pop X and Y.
	START_INSTR(PLX)
		static FINLINE void Execute(MachineType &state) {
			state.cpustate.x=PopRegContentsInstr(state);
		}
	END_INSTR()

	START_INSTR(PLY)
		static FINLINE void Execute(MachineType &state) {
			state.cpustate.y=PopRegContentsInstr(state);
		}
	END_INSTR()

	START_INSTR(PHX)
		static FINLINE void Execute(MachineType &state) {
			PushRegContentsInstr(state,state.cpustate.x);
		}
	END_INSTR()

	START_INSTR(PHY)
		static FINLINE void Execute(MachineType &state) {
			PushRegContentsInstr(state,state.cpustate.y);
		}
	END_INSTR()

	//////////////////////////////////////////////////////////////////////////
	// New or otherwise altered addressing modes

	//Indirect zero page unindexed
	//
	//Copied from ModeINX
	struct ModeINZ {
		enum {operand_size=1,};
		static char *Disassemble(char *buf,const byte *bytes,Word) {
			*buf++='(';
			buf=AddHexByte(buf,bytes[0]);
			*buf++=')';
			*buf=0;
			return buf;
		}
		static FINLINE Word ReadModeEA(MachineType &state) {
			Word ptr,addr;
			
			ptr.w=state.ReadByte(state.cpustate.pc,catFetchInstr);
			++state.cpustate.pc.w;
			++state.cycles;
			state.ReadByte(ptr,catReadZP);
			//ptr.l+=state.cpustate.x;
			++state.cycles;
			addr.l=state.ReadByte(ptr,catReadZP);
			++state.cycles;
			++ptr.l;
			addr.h=state.ReadByte(ptr,catReadZP);
			++state.cycles;
			return addr;
		}
		
		static FINLINE Word WriteModeEA(MachineType &state) {
			Word ptr,addr;
			
			ptr.w=state.ReadByte(state.cpustate.pc,catFetchInstr);
			++state.cpustate.pc.w;
			++state.cycles;
			state.ReadByte(ptr,catReadZP);
			//ptr.l+=state.cpustate.x;
			++state.cycles;
			addr.l=state.ReadByte(ptr,catReadZP);
			++ptr.l;
			++state.cycles;
			addr.h=state.ReadByte(ptr,catReadZP);
			++state.cycles;
			return addr;
		}
		
		static FINLINE byte ReadOperand(MachineType &state,Word addr) {
			byte b=state.ReadByte(addr,catReadOperand);
			++state.cycles;
			return b;
		}
		
		static FINLINE void WriteOperand(MachineType &state,Word addr,byte val) {
			state.WriteByte(addr,val,catWriteOperand);
			++state.cycles;
		}
	};

	//Indirect -- handles page boundary crossings correctly
	//Unclear where the read during the fixup cycle comes from.
	//TODO so there isn't one, but the cycle timing is correct.
	struct ModeIND {
		enum {operand_size=2,};
		static char *Disassemble(char *buf,const byte *bytes,Word) {
			*buf++='(';
			buf=AddHexWord(buf,MakeWord(bytes[0],bytes[1]));
			*buf++=')';
			*buf=0;
			return buf;
		}
		static FINLINE Word ReadModeEA(MachineType &state) {
			Word ptr,addr;
			
			ptr.l=state.ReadByte(state.cpustate.pc,catFetchInstr);
			++state.cpustate.pc.w;//2
			++state.cycles;
			ptr.h=state.ReadByte(state.cpustate.pc,catFetchInstr);
			++state.cpustate.pc.w;//3
			++state.cycles;
			addr.l=state.ReadByte(ptr,catFetchAddress);
			++ptr.l;
			++state.cycles;//4
			if(ptr.l==0) {
				//carry.
				++ptr.h;
				++state.cycles;//5
			}
			addr.h=state.ReadByte(ptr,catFetchAddress);
			++state.cycles;//5 or 6
			return addr;
		}
	};
	
};

//TODO s/P::/t65TYPENAME P::/
template<class T>
struct bbcR65C12InstructionSet:
public t65::Generic6502InstructionSet<T>
{
	//BRA
	typedef P::Relative<P::InstrBRA,P::ModeREL> Opcode80;

	//STZ
	typedef P::Write<P::InstrSTZ,P::ModeZPG> Opcode64;
	typedef P::Write<P::InstrSTZ,P::ModeZPX> Opcode74;
	typedef P::Write<P::InstrSTZ,P::ModeABS> Opcode9C;
	typedef P::Write<P::InstrSTZ,P::ModeABX> Opcode9E;
	
	//DEA/INA
	typedef P::RMW<P::InstrDEC,P::ModeACC> Opcode3A;
	typedef P::RMW<P::InstrINC,P::ModeACC> Opcode1A;

	//PHX etc.
	typedef P::Implied<P::InstrPHX,P::ModeIMP> OpcodeDA;
	typedef P::Implied<P::InstrPHY,P::ModeIMP> Opcode5A;
	typedef P::Implied<P::InstrPLX,P::ModeIMP> OpcodeFA;
	typedef P::Implied<P::InstrPLY,P::ModeIMP> Opcode7A;

	//TRB and TSB
	typedef P::RMW<P::InstrTRB,P::ModeZPG> Opcode14;
	typedef P::RMW<P::InstrTRB,P::ModeABS> Opcode1C;
	typedef P::RMW<P::InstrTSB,P::ModeZPG> Opcode04;
	typedef P::RMW<P::InstrTSB,P::ModeABS> Opcode0C;

	//New zero page indirect instructions.
	typedef P::Read<P::InstrADC,P::ModeINZ> Opcode72;
	typedef P::Read<P::InstrAND,P::ModeINZ> Opcode32;
	typedef P::Read<P::InstrCMP,P::ModeINZ> OpcodeD2;
	typedef P::Read<P::InstrEOR,P::ModeINZ> Opcode52;
	typedef P::Read<P::InstrLDA,P::ModeINZ> OpcodeB2;
	typedef P::Read<P::InstrORA,P::ModeINZ> Opcode12;
	typedef P::Read<P::InstrSBC,P::ModeINZ> OpcodeF2;
	typedef P::Write<P::InstrSTA,P::ModeINZ> Opcode92;

	//JMP-INX
	typedef P::Special<P::InstrJMP,P::ModeINX> Opcode7C;
};

#endif
