/*

	ElectrEm (c) 2000 Thomas Harte - an Acorn Electron Emulator

	This is open software, distributed under the GPL 2, see 'Copying' for details

*/
#include "6502.h"
#include "tape.h"
#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include <malloc.h>
extern C_6502ULA electron;

#define DATA_START 0x8031

void C_Tape::GetNewChunk(void)
{
	broken_word id, ms;

	tape_counter = 0;
	tape.FindIdMajor(0x01);
	id.a = tape.GetId();
	tape_minor = id.b.l;

	switch(tape_minor)
	{
		case 0x00 :
			pause_display = true;
		break;

		case 0x10 : //high tone
		case 0x12 : //rest
			tape.ReadSome(&ms.b.l, 1);
			tape.ReadSome(&ms.b.h, 1);
			tape_target = ms.a / 16;
			pause_display = false;
		break;
	}
}

bool C_Tape::Open(char *name)
{
	tape.Open(name, "r", 5, 0);
	GetNewChunk();
	pause_display = false;
	data = NULL;

	return false;
}

void C_Tape::Close(void)
{
	tape.Close();
	pause_display = false;
	if(data)
	{
		free(data);
		data = NULL;
	}
}

void C_Tape::Rewind(void)
{
	tape.Restart();
	if(data)
	{
		free(data);
		data = NULL;
	}
	GetNewChunk();
}

void C_Tape::Keepalive(void)
{
	if(tape.HasFile())
	{
		if(motor_on)
		{
			tape_counter++;

			switch(tape_minor)
			{
				case 0x10 : //high tone
					electron.SetIRQ(HIGH_TONE_IRQ);
					tape_data = 0xff;

				case 0x12 :
					if(tape_counter == tape_target)
						GetNewChunk();
				break;

				case 0x00 : //tape data
					tape.ReadSome(&tape_data, 1);

					electron.TriggerIRQ(DATA_SEND_IRQ);
					electron.TriggerIRQ(DATA_RECEIVE_IRQ);

					if(tape.EOC())
						GetNewChunk();
				break;
			}
		}
	}
	else
		tape_data = 255;
}

unsigned char h, l;

void get_crc(unsigned char *start, int len)
{
	unsigned char x;

	h = l = 0;
	while(len--)
	{
		h ^= *start;

		x = 8;
		while(x--)
		{
			if(h&128)
			{
				h ^= 8;
				l ^= 16;

				h = (h << 1) | (l >> 7);
				l = (l << 1) | 1;
			}
			else
			{
				h = (h << 1) | (l >> 7);
				l = (l << 1);
			}
		}

		start++;
	}
}

void C_Tape::NewFileChunk(void)
{
	int stlen;

	tape_end = false;
	do
	{
		tape.FindId(0x0100);
		if(tape.OverRan())
			tape_end = true;
	}while(tape.GetLength() == 1);

	data_length = tape.GetLength();

	data = (unsigned char *)realloc(data, data_length);
	tape.ReadAll(data);
	fname = data+1;

	stlen = strlen((char *)fname);

	blocknum = data[stlen+10] | (data[stlen+11] << 8);
}

void C_Tape::NextFile(void)
{
	do
	{
		NewFileChunk();
	}while(blocknum && !tape_end);
}

bool my_strcmp(char *one, char *two)
{
	bool ret = false;

	while(*one)
	{
		if(tolower(*one) != tolower(*two))
			ret = true;

		one++;
		two++;
	}

	return ret;
}

bool C_Tape::FindFile(char *name)
{
	int c = 0;
	do
	{
		NextFile();
		if(tape_end)
			c++;
	}
	while(c < 2 && strlen((char *)name) && my_strcmp((char *)fname, (char *)name));

	return c < 2;
}

bool C_Tape::LoadFile(unsigned char *info)
{
	int stlen, length, c;
	broken_word writeaddr, readaddr;

	length = 0;
	stlen = strlen((char *)fname);

	if(info[6])
	{
		writeaddr.b.l = info[2] = data[stlen+2];
		writeaddr.b.h = info[3] = data[stlen+3];
	}
	else
	{
		writeaddr.b.l = info[2];
		writeaddr.b.h = info[3];
	}

	length = 0;

	while(1)
	{
		stlen = strlen((char *)fname);

		info[2] = data[stlen+2];
		info[3] = data[stlen+3];
		info[4] = data[stlen+4];
		info[5] = data[stlen+5];

		info[6] = data[stlen+6];
		info[7] = data[stlen+7];
		info[8] = data[stlen+8];
		info[9] = data[stlen+9];

		c = (data[stlen+14]&0x40) ? 0 : data[stlen+12] | (data[stlen+13] << 8);
		length += c;
		readaddr.a = stlen+21;

		while(c--)
		{
			*electron.GetNormalPtr(writeaddr.a) = data[readaddr.a];
			writeaddr.a++;
			readaddr.a++;
		}

		if(data[stlen+14]&0x80)
		{
			break;
		}

		NewFileChunk();
	}

	info[10] = length & 0xff;
	info[11] = (length >>  8)&0xff;
	info[12] = (length >> 16)&0xff;
	info[13] = (length >> 24)&0xff;

	info[14] =
	info[15] =
	info[16] =
	info[17] = 0;

	return true;
}

char *get_filename(unsigned __int16 addr)
{
	char *name;
	unsigned char *s_start;
	int c;

	name = (char *)malloc(256);
	c = 0;
	s_start = electron.GetNormalPtr(addr);

	while(*s_start != 0xd)
	{
		if(*s_start != '"')
		{
			name[c] = *s_start;
			c++;
		}
		s_start++;
	}
	name[c] = '\0';

	return name;
}

void C_Tape::HandleService(unsigned char &a, unsigned char &x, unsigned char &y, unsigned char type)
{
	char *name;
	unsigned char *ram_start = electron.GetNormalPtr(0);
	broken_word addr, blockaddr;
	unsigned char load_buffer[18];
	bool do_load;

	if(!tape.HasFile())
	{
		a = 0;
		return;
	}

	switch(type)
	{
		case 0 :	//OSFILE
//			fprintf(stderr, "OSFILE issued\n");
			if(a == 0xff)
			{
				blockaddr.a = (y << 8) | x;
				addr.b.l = ram_start[blockaddr.a];
				addr.b.h = ram_start[blockaddr.a + 1];

				name = get_filename(addr.a);

				if(FindFile((char *)name))
				{
					LoadFile(&ram_start[blockaddr.a]);

					a = 1;
				}
				else
				{
					a = 0;
				}

				free(name);
			}
		break;

		case 1 : //OSARGS
//			fprintf(stderr, "OSARGS issued\n");
		break;

		case 2 : //OSBGET
//			fprintf(stderr, "OSBGET issued\n");
		break;

		case 3 : //OSBPUT
//			fprintf(stderr, "OSBPUT issued\n");
		break;

		case 4 : //OSBGBPB
//			fprintf(stderr, "OSGBPB issued\n");
		break;

		case 5 : //OSFIND
//			fprintf(stderr, "OSFIND issued\n");
		break;

		case 6 : //OSFSC
//			fprintf(stderr, "OSFSC issued, a= %d\n", a);
			do_load = false;

			switch(a)
			{
				case 7 :
					x = y = 0;
				break;

				case 2 :
					name = strdup("");
					do_load = true;
				break;
				case 4 : //LOAD AND RUN
					name = get_filename((y << 8) | x);
					do_load = true;
				break;
			}

			if(do_load)
			{
				if(FindFile(name))
				{
					load_buffer[6] = 1;
					LoadFile(load_buffer);
					electron.DoPush((load_buffer[6] | (load_buffer[7] << 8))-1);
				}
				free(name);
			}
		break;

	}
}

