/*

	ElectrEm (c) 2000 Thomas Harte - an Acorn Electron Emulator

	This is open software, distributed under the GPL 2, see 'Copying' for details

*/
#include "gfx.h"
#include <string.h>
#include <math.h>

#ifdef TARGET_SVGALIB
#include <malloc.h>
#include <vga.h>
#include <vgamouse.h>
#endif

C_gfx_api *mouse;

#ifdef TARGET_WIN32
#include <malloc.h>

bool lbutton;
C_gfx_api *gfx_api;

LRESULT CALLBACK DefaultWndProc(HWND hwnd, UINT msg, WPARAM wparam, LPARAM lparam)
{
	PAINTSTRUCT ps;
	HDC hdc;

	switch(msg)
	{	
		case WM_LBUTTONDOWN:
			lbutton = true;
		return 0;

		case WM_LBUTTONUP :
			lbutton = false;
		return 0;

		case WM_CREATE: 
		return 0;

		case WM_PAINT: 
			gfx_api->Restore();

			hdc = BeginPaint(hwnd,&ps);	 
			EndPaint(hwnd,&ps);
		return 0;

		case WM_DESTROY: 
			PostQuitMessage(0);
		return 0;

		default:break;
	}

	return DefWindowProc(hwnd, msg, wparam, lparam);
}
#endif

#ifdef TARGET_ALLEGRO

#include <allegro.h>

#endif

//C_gfx_api class from here

C_gfx_api::C_gfx_api(void)
{
	#ifdef TARGET_WIN32
	lpdd = NULL;
	lpdd4 = NULL;
	front4 = NULL;
	className = NULL;
	pal = NULL;
	gui_active = false;
	#endif

	#ifdef TARGET_SVGALIB
	cursor = blogo = NULL;
	#endif

	#ifdef TARGET_ALLEGRO
	logo = curs = NULL;
	#endif

	#if TARGET_SVGALIB || TARGET_ALLEGRO
	backupscr = NULL;
	#endif

	chars = NULL;
}

void C_gfx_api::shutdown(void)
{
	int c;

	#ifdef TARGET_WIN32
	if(className)
	{
		delete className;
		className = NULL;
	}

	if(front4)
	{
		front4->Release();
		front4 = NULL;
	}

	if(lpdd)
	{
		lpdd->Release();
		lpdd = NULL;
	}

	if(lpdd4)
	{
		lpdd4->Release();
		lpdd4 = NULL;
	}

	#endif

	#ifdef TARGET_SVGALIB
	if(cursor)
	{
		free(cursor);
		cursor = NULL;
	}

	if(blogo)
	{
		free(blogo);
		blogo = NULL;
	}

	if(backupscr)
	{
		free(backupscr);
		backupscr = NULL;
	}
	#endif

	#ifdef TARGET_ALLEGRO
	if(curs)
	{
		destroy_bitmap(curs);
		curs = NULL;
	}

	if(logo)
	{
		destroy_bitmap(logo);
		logo = NULL;
	}

	if(backupscr)
	{
		destroy_bitmap(backupscr);
		backupscr = NULL;
	}
	#endif

	if(chars)
	{
		c = charend - charstart;
		while(c--)
		{
			if(chars[c].data)
				free(chars[c].data);
		}
		free(chars);
		chars = NULL;
	}
}

C_gfx_api::~C_gfx_api(void)
{
	shutdown();
}

#ifdef TARGET_WIN32
unsigned int rw, rh, tw, th;
int closedist;

HRESULT WINAPI displaycheck( LPDDSURFACEDESC2 sdesc, LPVOID lpContext)
{
	int difx, dify, dist;

	if(sdesc->dwHeight >= rh && sdesc->dwWidth >= rw && sdesc->ddpfPixelFormat.dwFlags&DDPF_PALETTEINDEXED8)
	{
		difx = (sdesc->dwWidth - rw);
		dify = (sdesc->dwHeight - rh);
		dist = difx*difx + dify*dify;

		if(closedist == -1 || dist < closedist)
		{
			tw = sdesc->dwWidth;
			th = sdesc->dwHeight;
			closedist = dist;
		}
	}

	return DDENUMRET_OK;
}

bool C_gfx_api::SetResolution(unsigned int w, unsigned int h)
{
	if(scw != w || sch != h)
	{
		if(pal)
		{
			pal->Release();
			pal = NULL;
		}

		if(front4)
		{
			front4->Release();
			front4 = NULL;
		}

		closedist = -1;
		rw = scw = w;
		rh = sch = h;
		lpdd4->EnumDisplayModes(0, NULL, NULL, displaycheck);

		gfx_api = this;

		if(FAILED(lpdd4->SetDisplayMode(tw, th, 8, 0, 0)))
		{
			return true;
		}

		tx = (tw >> 1) - (w >> 1);
		ty = (th >> 1) - (h >> 1);

		DDRAW_INIT_STRUCT(ddsd);

		ddsd.dwFlags = DDSD_CAPS;
		ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;

		if(FAILED(lpdd4->CreateSurface(&ddsd, &front4, NULL)))
		{
			return true;
		}

		if(FAILED(lpdd4->CreatePalette(DDPCAPS_8BIT, cols, &pal, NULL)))
		{
			return true;
		}

		if(FAILED(front4->SetPalette(pal)))
		{
			return true;
		}

		SelectStretchLevel(tw, th);
	}
	return false;
}

int C_gfx_api::Setup(unsigned int w, unsigned int h, char *appname)
{
	WNDCLASSEX winclass =
	{
		sizeof(WNDCLASSEX),
		CS_DBLCLKS | CS_OWNDC | CS_HREDRAW | CS_VREDRAW,
		DefaultWndProc,
		0, 0,
		winfo.hInstance,
		LoadIcon(NULL, IDI_APPLICATION),
		NULL,
		NULL,
		NULL,
		NULL,
		LoadIcon(NULL, IDI_APPLICATION)
	};
	int ret;

	className = new char[strlen(appname)+4];
	sprintf(className, "WC_%s", appname);
	winclass.lpszClassName = className;

	if (!RegisterClassEx(&winclass))
	{
		ret= 1;
		goto fail;
	}

	if (!(hWnd = CreateWindowEx(	NULL,
									className,
									appname,
									WS_POPUP | WS_VISIBLE,
									0,0,
									100,100,
									NULL,
									NULL,
									winfo.hInstance,
									NULL)))
	{
		ret= 1;
		goto fail;
	}
	winfo.hWnd = hWnd;

	if(FAILED(DirectDrawCreate(NULL, &lpdd, NULL)))
	{
		ret= 1;
		goto fail;
	}

	if(FAILED(lpdd->QueryInterface(IID_IDirectDraw4, (LPVOID *)&lpdd4)))
	{
		ret= 1;
		goto fail;
	}

	if(FAILED(lpdd4->SetCooperativeLevel(hWnd, DDSCL_EXCLUSIVE | DDSCL_FULLSCREEN | DDSCL_ALLOWREBOOT)))
	{
		ret= 1;
		goto fail;
	}

	if(SetResolution(w, h))
	{
		ret= 1;
		goto fail;
	}

	if(!(backupscr = (char *)malloc(640*512)))
	{
		ret = 3;
		goto fail;
	}

	ShowCursor(NULL);

	return 0;

	fail:
		shutdown();
		return ret;
}
#endif

#ifdef TARGET_ALLEGRO
struct known_modes
{
	int w, h, bpp;
};

known_modes some_modes[] =
{
	{800, 600, 8},
	{400, 300, 8},
	{-1, -1, -1}
};

known_modes other_modes[] =
{
	{800, 600, 8},
	{640, 400, 8},
	{320, 400, 8},
	{-1, -1, -1}
};

bool C_gfx_api::SetResolution(unsigned int w, unsigned int h)
{
	if(scw != w || sch != h)
	{
		known_modes *c = (slevel>1) ? other_modes : some_modes;

		while(c->w != -1 && c->w >= w && c->h >= h)
		{
			c++;
		}
		c--;

		scw = w;
		sch = h;

		tx = (c->w >> 1) - (w >> 1);
		ty = (c->h >> 1) - (h >> 1);

		set_color_depth(8);
		set_gfx_mode(GFX_AUTODETECT, c->w, c->h, 0, 0);
		set_palette(cols);

		SelectStretchLevel(c->w, c->h);
	}
	return false;
}

int C_gfx_api::Setup(unsigned int w, unsigned int h, char *appname)
{
	SetResolution(w, h);
	set_window_title(appname);
	install_mouse();
	backupscr = create_bitmap(640, 512);

	return 0;
}
#endif

#ifdef TARGET_SVGALIB
bool C_gfx_api::SetResolution(unsigned int w, unsigned int h)
{
	vga_modeinfo *inf;
	int c, dist, lowdist = -1, mode;

	if(scw != w || sch != h)
	{
		scw = w;
		sch = h;

		mode = -1;
		c = vga_lastmodenumber();
		while(c--)
		{
			inf = vga_getmodeinfo(c);

			if((inf->width >= w) && (inf->height >= h) && ((inf->bytesperpixel == 1) || (inf->colors == 256)))
			{
				dist = (w-inf->width)*(w-inf->width) + (h-inf->height)*(h-inf->height);
				if((lowdist == -1) || (dist < lowdist))
				{
					lowdist = dist;
					mode = c;
					mx = w >> 1;
					my = h >> 1;

					tx = (inf->width >> 1) - (w >> 1);
					ty = (inf->height >> 1) - (h >> 1);
				}
			}
		}

		if(vga_setmode(mode))
		{
			return true;
		}
		else
		{
			inf = vga_getmodeinfo(mode);

			SelectStretchLevel(inf->width, inf->height);
			if(inf->flags & CAPABLE_LINEAR)
				vga_setlinearaddressing();

			FinishPalette();
		}
	}

	return false;
}

int C_gfx_api::Setup(unsigned int w, unsigned int h, char *appname)
{
	vga_init();

	SetResolution(w, h);
	backupscr = (char *)malloc(640*512);

	return 0;
}
#endif

void C_gfx_api::FinishPalette(void)
{
	#ifdef TARGET_WIN32
	pal->SetEntries(0, 0, 256, cols);
	#endif

	#ifdef TARGET_ALLEGRO
	set_palette(cols);
	#endif

	#ifdef TARGET_SVGALIB
	int c;

	c = 256;
	while(c--)
		vga_setpalette(c, cols[c].red, cols[c].green, cols[c].blue);
	#endif
}

void C_gfx_api::SetPaletteEntry(int id, float r, float g, float b)
{
	#ifdef TARGET_WIN32
	cols[id].peRed = (int)(255*r);
	cols[id].peGreen = (int)(255*g);
	cols[id].peBlue = (int)(255*b);
	cols[id].peFlags = NULL;
	#endif

	#ifdef TARGET_ALLEGRO
	cols[id].r = (int)(63*r);
	cols[id].g = (int)(63*g);
	cols[id].b = (int)(63*b);
	#endif

	#ifdef TARGET_SVGALIB
	cols[id].red = (int)(63*r);
	cols[id].green = (int)(63*g);
	cols[id].blue = (int)(63*b);
	#endif
}

void C_gfx_api::SetStretchLevel(int newlev, bool clean)
{
	slevel = newlev;
	cleanscale = clean;
}

void C_gfx_api::SelectStretchLevel(int truew, int trueh)
{
	int wastedpels, v, a;
	//first plain size test

	rslevel = 2;

	wastedpels = truew*(trueh-sch) + sch*(truew-scw);
	if(((float)wastedpels / (float)(truew*trueh)) > 0.4f) //if more than 2/3rds of pixels are not used
		rslevel = 1;

	if( fabs( ((float)scw / (float)truew) - ((float)sch / (float) trueh) ) > 0.1)
		rslevel = 1;

	if(cleanscale)
	{
		v = scw;
		a = scw >> 1;
		while(v <= truew)
			v += a;
		v -= a;

		StretchArea.left = (truew >> 1) - (v >> 1);
		StretchArea.right = StretchArea.left + v;

		v = sch;
		a = sch >> 1;
		while(v <= trueh)
			v += a;
		v -= a;

		StretchArea.top = (trueh >> 1) - (v >> 1);
		StretchArea.bottom = StretchArea.top + v;
	}
	else
	{
		StretchArea.left = 0;
		StretchArea.right = truew;
		StretchArea.top = 0;
		StretchArea.bottom = trueh;
	}
}

void C_gfx_api::SetNormalResolution(unsigned int w, unsigned int h)
{
	oldw = w;
	oldh = h;
}

int C_gfx_api::GetStretchLevel(void)
{
	return slevel;
}

bool C_gfx_api::GetCleanScale(void)
{
	return cleanscale;
}

