/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// INITCARD.C
//
// (C) Mark Usher
// marku@magnet.at
// release 1.0  30.06.1999                                                       
*/

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Routines to initialise the ecolink card without loading the legacy driver
//
// The interrupt routines are also here.
*/

#include <go32.h>
#include <dpmi.h>
#include <string.h>
#include <sys/farptr.h>
#include <sys/movedata.h>

#include "initcard.h"
#include "6512code.h"

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Function prototypes
*/

int SetupEcolinkRAMSelector(void);
int download_microcode(void);
byte InstallInterrupt(void);
void SetLatch(unsigned short latch, bool state);
void my_timer_handler(void);
void end_timer_handler(void);
void card_handler();
void end_card_handler();
void default_event_handler(byte CBH, byte timeout);
void end_default_event_handler();
void EventChain();
void end_EventChain();
void (*EventHandlers[MAX_EVENT_HANDLERS])(byte CBH, byte Timeout);

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Global variables
*/

int  card_address_selector = 0;
byte DefaultEventAction    = STORED;
byte EventSchedulerStatus  = ENABLED;
byte counter_150           = 0x09;
byte Allow_reentry         = TRUE;
byte NoOfEventHandlers     = 0;

#include "ecolink.c"

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Main routine to control the cards initialisation
*/
int InitialiseCard( void)
{
    int i;
    byte value;
    int rc;

    rc = OK;

	#if debug
		printf("Initialising the Ecolink Card\n");
	#endif

	/* set up the pointer to the card's memory address */
	rc = SetupEcolinkRAMSelector();
    if (rc == ERROR)
       return rc;

	/* download the microcode to the Ecolink card */
    rc = download_microcode();
    if (rc == ERROR)
       return rc;

	/* Set the PC to acknowledge the interrupt the card generates */
    rc = InstallInterrupt();
    if (rc == ERROR)
       return rc;

	#if debug
	    printf("Starting the card.\n");
	#endif

	/* Not sure why this is getting initialised to 00 
	   Seems pointless as byte 0x40 is in the middle of the TX Chain list	
	*/
    _farpokeb(card_address_selector, LOC_0x0040,        0x00);

    /* clear the unused handles table */
    for (i = 1; i<=MAX_HANDLES; i++)
        _farpokeb(card_address_selector, UNUSED_LIST+i, i);

    /* initialise the index for the unused list*/
    _farpokeb(card_address_selector, UNUSED_LIST,      MAX_HANDLES);

    /* initialise the other indexes to 0x00*/
    for (i = OPEN_TX_LIST; i<=STORED_LIST; i += (MAX_HANDLES + 1))
        _farpokeb(card_address_selector, i, 0x00);

    /* This will reset all the values in all lists */
    #if debug
    for (i = OPEN_TX_LIST; i<=STORED_LIST+(MAX_HANDLES+1); i ++)
        _farpokeb(card_address_selector, i, 0x00);
    #endif

    /* other memory location initialisation */
    _farpokeb(card_address_selector, LOC_0x7027,       0xFF);
    _farpokeb(card_address_selector, EEPROM_COMMAND,   EEPROM_NOTHING);
    _farpokeb(card_address_selector, LOC_0x0001,       0x00);
    _farpokeb(card_address_selector, TX_PAUSE_VALUE,   0x14);
    _farpokeb(card_address_selector, TIMER_VALUE,      0x02);
    _farpokeb(card_address_selector, TXCB_OPENED_FLAG, 0x00);
    _farpokeb(card_address_selector, PC_STATUS,        0x00);       /* start microcode */

    _farpokeb(card_address_selector, MACHINE_TYPE,    MC_PC);
    _farpokeb(card_address_selector, LOC_0x7029,       0x00);
    _farpokeb(card_address_selector, MINOR_VERSION,  VMINOR);
    _farpokeb(card_address_selector, MAJOR_VERSION,  VMAJOR);

    return OK;
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Set a global variable with a pointer to the Ecolink cards memory area
*/
int SetupEcolinkRAMSelector(void)
{

    /* Hard coded at 0xD0000 for my card */
    static char selectorData[8] = {
           0xFF, 0x7F, 0x00, 0x00,
           0x0D, 0xF3, 0x40, 0x00 };

	/* set the address selector in a global variable */
	card_address_selector = __dpmi_allocate_ldt_descriptors(1);

    if (__dpmi_set_descriptor (card_address_selector, selectorData) < 0)
        return ERROR;

    return OK;
}


/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Download the 6512 machine code program to the Ecolink card
*/
int download_microcode(void)
{
    int i;
    int rc;
    unsigned char readback[microcode_length];

    rc=OK;
	#if debug 
		printf("Downloading microcode....\n");
	#endif

    /* set the latch states to load the code */
    SetLatch(_6512_Reset,    LATCH_ON);
    SetLatch(_6512_Cont_Acc, DISABLE);
    SetLatch(_6512_IRQ,      LATCH_OFF);
    SetLatch(NMI,            DISABLE);

    /* download the code */
    movedata(_my_ds(),
             (int)microcode,
             card_address_selector,
             0x3500,
             microcode_length);

    /* read back the code block and compare it to the original to check it download ok */
    movedata(card_address_selector,
             0x3500,
             _my_ds(),
             (int)readback,
             microcode_length);

    for (i=0; i < microcode_length; i++)
    {
        if (microcode[i] != readback[i])
        {
            printf("Error at offset 0x%X  microcode = 0x%X  Card memory = 0x%X\n",
                                                     i,microcode[i], readback[i]);
            rc = ERROR;
        }
    }

    #if debug
		if (rc==OK)
			printf("microcode download was succesful.\n");
	#endif

    /* set the top bit of PC_STATUS - 0x7026 to signal that the PC is finished with loading */
    _farpokeb(card_address_selector, PC_STATUS, INITIALISE_TESTING);

    /* set the latch states after the code has been loaded 
	   to signal completion of code download
	*/
    SetLatch(_6512_Reset, LATCH_OFF);
    SetLatch(_6512_Cont_Acc, ENABLE);

    return OK;
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Setup up the PC to acknowledge the interrupt generated by the card.
*/
byte InstallInterrupt(void)
{
    byte value;
    byte mask;
    byte attempt;

    /* test to see if the interrupt is in use */

    mask = 0x01 << CARD_INTERRUPT;
	
    asm ("cli");										/* clear interrupts */
    value = SPECIFIC_EOI | CARD_INTERRUPT;				/* command for Specific end of interrupt for interrupt no ... */
    outportb(MASTER_PIC, value);						/* OCW 2  send the command to the Master PIC controller */

    SetLatch(PC_INT, PC_INT_OFF);
    SetLatch(PC_INT, PC_INT_ON);

    attempt=0;
    do
    {
        outportb(MASTER_PIC, 0x0A);						/* OCW 3  Select IRR Interrupt Request Register */
        attempt++;
        value = inportb(MASTER_PIC) & mask;				/* Check if our interrupt is already in use: bit 0 = used 1 = not used */
    }
    while ((attempt<0xFF) && (value ==0));

	/* If the number of attempts has been exceeded then the interrupt is in use */
    if (attempt>=0xFF)
    {
       printf("Interrupt already in use. 0x%X\n",value);
       asm("sti");
       return ERROR;
    }

    SetLatch(PC_INT, PC_INT_OFF);
    value = SPECIFIC_EOI | CARD_INTERRUPT;				/* command for Specific end of interrupt for interrupt no ... */
    outportb(MASTER_PIC, value);						/* OCW 2  send the command to the Master PIC controller */

    /* enable the cards interrupt in the PIC controller */
	value = inportb(0x21);								/* Get the current mask */
    value = value & ~(0x1 << CARD_INTERRUPT);			/* Set the mask for our interrupt */
    outportb(0x21, value);								/* send the command */

    asm("sti");											/* enable interrupts */

    return OK;
}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Set the state of a latch
// These are located in the memory map from 7FF8 - 7FFF
*/
void SetLatch(unsigned short latch, bool state)
{
    byte dummy;

    if (state)                                              /* Write operation */
       _farpokeb(card_address_selector, latch, 0);          /* The value written is irrelevant
                                                               as it is the write operation that
                                                               sets the latches status */

    else                                                    /* Read operation */
       dummy = _farpeekb(card_address_selector, latch);     /* Read a value to set the latch
                                                               state */
}
void end_SetLatch() {/* dummy function to calculate length of previous function */}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// The PC timer interrupt is used to perform some functions in the cards memory
*/
void my_timer_handler()
{
    byte timer;
    byte RX_index;
    byte RX_CBH;
    byte nextCBH;
    int Address;
    int i,j;
    unsigned char *readback[0x11];

    struct RXCBHeader *RX_buffer;
    RX_buffer = readback;
        
    timer = _farpeekb(card_address_selector, TIMER_COUNTER);	    /* get the cards timer count */

    if (timer != 0 )											    /* if the ecolink timer counter is not 0 then decrease it */
       _farpokeb(card_address_selector, TIMER_COUNTER, --timer);

    /* Check to see if there are any RX Control blocks to process */
    RX_index = _farpeekb(card_address_selector, OPEN_RX_LIST);

    if (RX_index !=0)                            /* yes there are */
    {
        EnterCriticalSection(1);

        counter_150 --;

        if (counter_150 == 0)
        {
            counter_150 = 0x09;
            for (i=1; i <= RX_index; i++)										/* loop though the Rx buffer list */
            {
                RX_CBH    = _farpeekb(card_address_selector, i + OPEN_RX_LIST); /* Get the next CBH from the OPEN_RX_LIST */
                Address   = GetAddressOfBuffer(RX_CBH);                         /* Retrieve the address of the buffer */
                movedata(card_address_selector,									/* get the buffers header into RX_buffer */
                         Address,
                         _my_ds(),
                         (int)readback,
                         0x11);

																	/* if */
                if (((RX_buffer->E & 0x80) == 0) &&					/* unknown byte top bit not set */
                    ((RX_buffer->ControlByte & 0x80) == 0) &&       /* AND Control Byte top bit not set      ie = Receive handle */
                    ((RX_buffer->CtrlByteCopy & 0x80) == 0) &&		/* AND Control byte copy top bit not set  "       "      "   */
                    (RX_buffer->Timeout !=0))						/* AND Timeout NOT = 0 */
                {

                    RX_buffer->Timeout --;							/* decrease the timeout */
                    _farpokeb(card_address_selector, Address + 0x0D, RX_buffer->Timeout);  /* write the new timeout to the card */

                    /* if the timeout has now been reached */
                    if (RX_buffer->Timeout == 0x00)
                    {
                        _farpokeb(card_address_selector, Address + 0x07, RxCB_TIMED_OUT);	/* Set the status flag */
                        AddHandleToList(COMPLETED_LIST, RX_CBH);

                        if (RX_buffer->Event == ENABLED)
                            AddHandleToList(ENABLED_LIST, RX_CBH);

                        if (RX_buffer->Event == STORED)
                            AddHandleToList(STORED_LIST, RX_CBH);

                        /* Remove from the RX_List because it has timed out */
                        RemoveHandleFromList(OPEN_RX_LIST, RX_CBH, i);

                    }
                }
            }
        }

        LeaveCriticalSection(1);
    }

	EventChain();		/* Call the event handlers */

	/* display a grid of the contents of the lists */
	#if debug
		for (i = UNUSED_LIST; i<= STORED_LIST; i += (MAX_HANDLES +1 ))
		{
			for (j=0; j <= MAX_HANDLES; j++)
			{
			   RX_CBH = _farpeekb(card_address_selector, i+j);
			   RX_CBH = RX_CBH + 0x30;
			   ScreenPutChar((int)RX_CBH, (BLACK << 4) | WHITE , (j*2) + 14 , ((i-4) / 0x17)+1);
			}
		}
	#endif

	return;
}

void end_my_timer_handler() {/* dummy function to calculate length of previous function */}

/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// This function handles the interrupt that the card generates 
*/
void card_handler()
{
    byte CBH;
	byte test;
	byte ControlByte;
	byte StatusByte;
	byte EventByte;

	int Address;

	CBH = 0;
	test = 0;

    _farpokeb(card_address_selector, IN_INTERRUPT, 0x80); /* Set the top bit to signal that the routine is currently executing */

	do
	{
		do
		{

			CBH = _farpeekb(card_address_selector, COMPLETED_HANDLE);				/* get the completed handle */

			if (CBH !=0)
			{
				Address = GetAddressOfBuffer(CBH);									/* get the address of the buffer */

				ControlByte = _farpeekb(card_address_selector, Address);			/* read the control byte */
     			_farpokeb(card_address_selector, Address + 0x09, ControlByte);		/* set the control byte copy */

				StatusByte = _farpeekb(card_address_selector, Address + 0x08);		/* read the buffer's status byte */
				if ((StatusByte & 0x80) == 0)										/* It's a Rx handle */
				{
					EventByte = _farpeekb(card_address_selector, Address + 0x06);	/* get the event byte */
					if (EventByte == ENABLED)												/* add the handle to the correct list */
						AddHandleToList(ENABLED_LIST, CBH);
					else if (EventByte == STORED)
						AddHandleToList(STORED_LIST, CBH);
				}	
				else																/* else remove the handle from the lists */
				{
					 RemoveHandleFromList(  ENABLED_LIST, CBH, 0);
					 RemoveHandleFromList(   STORED_LIST, CBH, 0);
				}

				_farpokeb(card_address_selector, COMPLETED_HANDLE, 0x00);			/* Clear the handle */
				
			}

		}
		while (CBH !=0);

		test = _farpeekb(card_address_selector, LOC_0x00AE) & 0x80;					/* test the top bit */

    }
	while (test != 0);	/* loop until the top bit of 00AE is set */
	
	SetLatch(PC_INT, PC_INT_ON);
	
	/* End of interrupt to PIC */
	test = SPECIFIC_EOI | CARD_INTERRUPT;				/* command for Specific end of interrupt for interrupt no ... */
	outportb(MASTER_PIC,test);							/* OCW 2  send the command to the Master PIC controller */

	_farpokeb(card_address_selector, IN_INTERRUPT, 0x00);							/* Clear the IN_INTTERUPT flag */
	EventChain();		/* Call the event handlers */

}

void end_card_handler() {/* dummy function to calculate length of previous function */}


/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Function that runs through a list of functions calling them one after the other,
// The list - EventHandlers[i] is a global array that contains pointers to the functions to be called as event handlers
*/
void EventChain()
{

	byte CBH;
	byte timeout;
    int i;
    int Address;
	timeout = 0;

    if ((EventSchedulerStatus == ENABLED) && (Allow_reentry == TRUE))
    {
        Allow_reentry = FALSE;				/* set the status flag to show this function is executing */

        CBH = GetHandle(ENABLED_LIST);		/* Go through all handles stored in the enabled list */
        while (CBH !=0)
        {
            Address = GetAddressOfBuffer(CBH);		/* Get the timed out status of the handle */
            if ((_farpeekb(card_address_selector, Address + 0x07)) == RxCB_TIMED_OUT)
                 timeout=1;

            /* Here event handler routines are called */
            /* All handlers must accept CBH and Timeout as their parameters */
            for (i=0; i < NoOfEventHandlers; i++)
                 EventHandlers[i](CBH, timeout);		/* call the handler(CBH, timeout) */

            default_event_handler(CBH, timeout);		/* always call the default handler */

            CBH = GetHandle(ENABLED_LIST);				/* get the next handle in the list */
        }

        Allow_reentry = TRUE;							/* set function executing flag */
    }

}

void end_EventChain() {/* dummy function to calculate length of previous function */}


/*////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// This is the default event handler
*/
void default_event_handler(byte CBH, byte timeout)
{
     /*
        If events are enabled for a CB and no event handler picks it up the CB
        will be killed.
     */

     RemoveHandleFromList(COMPLETED_LIST, CBH, 0);
     AddHandleToList(        UNUSED_LIST, CBH);
     RemoveHandleFromList(  ENABLED_LIST, CBH, 0);
     RemoveHandleFromList(   STORED_LIST, CBH, 0);
}

void end_default_event_handler() {/* dummy function to calculate length of previous function */}


