/***********************************************************************/
/* BubblePad - Replacement class for the launchpad of Warp 3/4         */
/* (C) Chris Wohlgemuth                                                */
/*                                                                     */
/* E-Mail: chris.wohlgemuth@cityweb.de                                 */
/*     or  christopher.wohlgemuth@bch.siemens.de                       */
/*                                                                     */
/* Released under the GNU Public Licence                               */
/* See file COPYING for further information                            */
/*                                                                     */
/* 20.06.98 V1.00:  First public version                               */
/* 16.08.98 V1.00a: Corrected bug: fly over help window on some        */
/*                  systems remains visible. Workaround: When help     */
/*                  stucks on the desktop move pointer over another    */
/*                  button                                             */
/*                                                                     */
/* 18.08.98 V1.00b: Corrected positioning of drawers. Drawers below    */
/*                  or on the left of the launchpad were placed        */
/*                  with some distance to the launchpad                */
/*                  Timer routines rewritten.                          */
/*                                                                     */
/* 18.08.98 V1.01: Stucking window problem for drawers with autoclose  */
/*                  feature corrected                                  */
/*                                                                     */
/* 19.09.98 V1.02: Positioning of help windows for launchpad at the    */
/*                 top of the desktop corrected.                       */
/*                 Drawer button hide feature added.                   */
/*                 Individual settings for BubblePads added.           */
/*                 Bug with broken references causing drawers to       */
/*                 disapear fixed                                      */
/*                 Added saving of drawer state. Drawers are reopened  */
/*                 on next reboot. Doesn't work with normal open/close */
/*                 Moved button reordering from wpRefreshDrawer()      */
/*                 to wpSetDrawerHWND(). This eliminates the flickering*/
/*                 when opening drawers.                               */
/*                                                                     */
/***********************************************************************/

#define INCL_WINWORKPLACE
#define INCL_WPCLASS
#define INCL_GPILCIDS
#define INCL_GPIPRIMITIVES
#define INCL_GPIBITMAPS

#include "BUBBLEPD.hh"
#include <somcls.hh>
#include <wpobject.hh>
#include <pmwp.h>
#include <os2.h>


// for sprintf
#include <stdio.h>
#include <string.h>  //for strchr()

#define xVal  12      //x-distance of Bubble
#define yVal  12      //y-distance of Bubble
#define delta 0       //use this if you want to reduce the button size

//#define debug      //uncomment for debuging

static PFNWP  oldButtonProc2;  //place for original button-procedure
static HWND hwndBubbleWindow=NULLHANDLE;// The handle of the help window

M_WPObject  M_ob;  //We need a class object to get an object pointer from HOBJECT

#ifdef debug
/* Display a PM-Messagebox with an Errortext if debug is defined */
void showMessage(HWND hwnd,PSZ msgText)
{
	WinMessageBox(HWND_DESKTOP,hwnd ,msgText ,"Debug Launchpad",0,MB_OK);
}
#endif

/* This function returns the module handle of our class-dll */
HMODULE QueryModuleHandle(void)
{
	static HMODULE hmod=0;
	
	if(!hmod) {
		PSZ pathname=SOMClassMgrObject  //Query Pathname of class file
			->somLocateClassFile(somIdFromString("WPLnchCW"),1,2);
		DosQueryModuleHandle(pathname,&hmod);  //Query module handle
	}
	return hmod;
}

/*************************************************************/
/* This dialog procedure handles the Bubblehelp enable page  */
/*************************************************************/
MRESULT EXPENTRY BubblehelpDlgProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2) {	
		ULONG spinValue;

 	switch(msg) {
 	case WM_INITDLG :	
		//thisPtr
		WinSetWindowULong(hwnd,QWL_USER,LONGFROMMP(mp2));
		/* We have to initialize the dialog controls with the approbiate values */

		// Set the focus on the enable checkbox
		WinSetFocus(HWND_DESKTOP, WinWindowFromID(hwnd, CWDLG_ENABLEBUBBLE));
		// Set the linits for the delay spin button
		WinSendMsg(WinWindowFromID(hwnd, CWDLG_DELAYSPIN),
							 SPBM_SETLIMITS,(MPARAM)9999,(MPARAM)0);
		// Set the current value for delay
		WinSendMsg(WinWindowFromID(hwnd, CWDLG_DELAYSPIN),
							 SPBM_SETCURRENTVALUE,
							 (MPARAM)((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->delayValue,
							 (MPARAM)NULL);
		// Enable or disable the spin button according to the value of bubbleEnabled
		WinEnableWindow(WinWindowFromID(hwnd, CWDLG_DELAYSPIN),((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->bubbleEnabled);
		// Set the state of the enable checkbox
		WinSendMsg(WinWindowFromID(hwnd, CWDLG_ENABLEBUBBLE),
							 BM_SETCHECK,MPFROMSHORT(((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->bubbleEnabled),(MPARAM)NULL);
		return (MRESULT) TRUE;  //Initialisation done

 	case WM_DESTROY:
		/* The notebook closes and gets destroyed */
		// Query the value of the delay spin button
		WinSendMsg(WinWindowFromID(hwnd, CWDLG_DELAYSPIN),
							 SPBM_QUERYVALUE,(MPARAM)&spinValue,
							 MPFROM2SHORT(0,SPBQ_ALWAYSUPDATE));				
		((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->delayValue=spinValue;
			// Write the new  values to the instance permanent data
		((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->wpSaveDeferred();
		break;
	case WM_CONTROL:
		/* The window controls send WM_CONTROL messages */
		switch(SHORT1FROMMP(mp1)) {
		case CWDLG_ENABLEBUBBLE:
			// We only have one checkbox, so we do not really need this switch statement
			// but with it you can expand the dialog easily 
			if(((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->bubbleEnabled)// switch the bubbleEnabled state
				((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->bubbleEnabled=0;						
			else 
				((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->bubbleEnabled=1;				
			// Enable or disable the spin button according to the value of bubbleEnabled
			WinEnableWindow(WinWindowFromID(hwnd, CWDLG_DELAYSPIN),((WPLnchCW*)WinQueryWindowULong(hwnd,QWL_USER))->bubbleEnabled);			
			// Set the state of the enable checkbox	
			WinSendMsg(WinWindowFromID(hwnd, CWDLG_ENABLEBUBBLE),
								 BM_SETCHECK,MPFROMSHORT(((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->bubbleEnabled),(MPARAM)NULL);

			WinPostMsg(WinWindowFromID(((WPLnchCW*)WinQueryWindowULong(hwnd,QWL_USER))->wpQueryDrawerHWND(0),300),
								 BM_CLICK,FALSE,0);
			break;
		case CWDLG_DELAYSPIN:
			switch (SHORT2FROMMP(mp1)) {
			case SPBN_UPARROW:
			case SPBN_DOWNARROW:
			case SPBN_ENDSPIN:
				// Query the value of the delay spin button
				WinSendMsg(WinWindowFromID(hwnd, CWDLG_DELAYSPIN),
									 SPBM_QUERYVALUE,(MPARAM)&spinValue,
									 MPFROM2SHORT(0,SPBQ_ALWAYSUPDATE));				
				((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->delayValue=spinValue;			
				break;
			default:
				break;
			}
		} // end switch(SHORT1FROMMP(mp1))
		break;			
		//} // end switch(SHORT2FROMMP(mp1))
		return (MRESULT) TRUE;
	}
	// The WinDefDlgProc() handles the rest of the messages
	return WinDefDlgProc(hwnd, msg, mp1, mp2);
}

/*************************************************************/
/* This dialog procedure handles the colour and font page    */
/*************************************************************/			
MRESULT EXPENTRY ColorchangeDlgProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2) {
	HINI hini;
	HOBJECT  hObject;
	ULONG  attrFound;
	ULONG  len;
	static RGB b;
	static RGB f;
	static 	font[FONTNAMELENGTH];
	char msgText[250];
	
 	switch(msg) {
 	case WM_INITDLG :	
		WinSetWindowULong(hwnd,QWL_USER,LONGFROMMP(mp2));
	
		/* We have to initialize the dialog controls with the approbiate values */

		// Set the focus on the demo area
		WinSetFocus(HWND_DESKTOP, WinWindowFromID(hwnd, CWDLG_TEXTDEMOFIELD));
		// Set the background colour of the demo area
		WinSetPresParam(WinWindowFromID(hwnd, CWDLG_TEXTDEMOFIELD),
										PP_BACKGROUNDCOLOR,(ULONG)sizeof(RGB) ,&((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->background );
		// Set the foreground colour of the demo area
		WinSetPresParam(WinWindowFromID(hwnd, CWDLG_TEXTDEMOFIELD),
										PP_FOREGROUNDCOLOR,(ULONG)sizeof(RGB) ,&((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->foreground );
		// Set the font of the demo area
		WinSetPresParam(WinWindowFromID(hwnd, CWDLG_TEXTDEMOFIELD),
										PP_FONTNAMESIZE,(ULONG)sizeof(((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->fontName),
										&((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->fontName );
		b=((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->background;  // Save value for undo 
		f=((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->foreground;  //Save value for undo
		memcpy(&font,&((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->fontName,
					 sizeof(((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->fontName));
		return (MRESULT) TRUE;
	case WM_DESTROY:
		/* The notebook closes and gets destroyed */
		// Query the current background colour
		len=WinQueryPresParam(WinWindowFromID(hwnd, CWDLG_TEXTDEMOFIELD),
													PP_BACKGROUNDCOLOR,0,&attrFound,sizeof(((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->background),
													&((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->background,QPF_NOINHERIT);
		// Query the current font
		len=WinQueryPresParam(WinWindowFromID(hwnd, CWDLG_TEXTDEMOFIELD),
													PP_FONTNAMESIZE,0,&attrFound,sizeof(((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->fontName),
													&((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->fontName,QPF_NOINHERIT);
		// Query the current foreground colour
		len=WinQueryPresParam(WinWindowFromID(hwnd, CWDLG_TEXTDEMOFIELD),
													PP_FOREGROUNDCOLOR,0,&attrFound,sizeof(((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->foreground),
													&((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->foreground,QPF_NOINHERIT);
		// Save font to instance permanent data
		((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->wpSaveDeferred();
		break;
	case WM_COMMAND :
		switch (SHORT1FROMMP(mp1)) {
			// Process commands here //
		case CWDLG_COLORUNDO:
			/* The undo button was clicked */
			((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->background=b;  // Restore the previous background colour
			((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->foreground=f;  // Restore the previous background colour
			// Set the background colour of the demo area
			WinSetPresParam(WinWindowFromID(hwnd, CWDLG_TEXTDEMOFIELD),
											PP_BACKGROUNDCOLOR,(ULONG)sizeof(RGB) ,&((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->background );
			// Set the foreground colour of the demo area
			WinSetPresParam(WinWindowFromID(hwnd, CWDLG_TEXTDEMOFIELD),
											PP_FOREGROUNDCOLOR,(ULONG)sizeof(RGB) ,&((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->foreground );			
			// Set the font of the demo area
			WinSetPresParam(WinWindowFromID(hwnd, CWDLG_TEXTDEMOFIELD),
											PP_FONTNAMESIZE,(ULONG)sizeof(font) ,&font);
			break;
		case CWDLG_COLORSTANDARD:
			/* The default button was clicked */
			((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->background.bBlue=180;  // Set the default colours
			((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->background.bGreen=255; 
			((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->background.bRed=255;
			((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->foreground.bBlue=0;
			((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->foreground.bGreen=0;			
			((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->foreground.bRed=0;
			// Set the default font
			memcpy(&((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->fontName,"8.Helv",sizeof("8.Helv"));
			// Set the background colour of the demo area
			WinSetPresParam(WinWindowFromID(hwnd, CWDLG_TEXTDEMOFIELD),
											PP_BACKGROUNDCOLOR,(ULONG)sizeof(RGB) ,&((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->background );
			// Set the foreground colour of the demo area
			WinSetPresParam(WinWindowFromID(hwnd, CWDLG_TEXTDEMOFIELD),
											PP_FOREGROUNDCOLOR,(ULONG)sizeof(RGB) ,&((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->foreground );
			// Set the font of the demo area
			WinSetPresParam(WinWindowFromID(hwnd, CWDLG_TEXTDEMOFIELD),
											PP_FONTNAMESIZE,(ULONG)sizeof(((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->fontName),
											&((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->fontName );
			break;
		case CWDLG_COLORPALETTE:
			/* Colour... button was clicked */
			// Open the colorpalette 
			if((hObject=WinQueryObject("<WP_HIRESCLRPAL>"))!=NULL) {
				WinOpenObject(hObject,OPEN_DEFAULT,TRUE);
			}
			else {//Error, can't open the palette
				/*  Show an error msg.						   */
				WinLoadString(WinQueryAnchorBlock(hwnd),QueryModuleHandle(),ID_NOCOLORPALETTESTRING,
											sizeof(msgText), msgText);
				WinMessageBox(HWND_DESKTOP,
											hwnd,         
											msgText,          
											(PSZ) NULL,                      
											ID_COLORPALETTEMSGBOX,            /* Window ID */
											MB_OK |
											MB_MOVEABLE |
											MB_ICONEXCLAMATION |
											MB_DEFBUTTON1);                  /* Style     */
				
			}
			break;
		case CWDLG_FONTPALETTE:
			/* Font... button was clicked */
			// Open the fontpalette 
			if((hObject=WinQueryObject("<WP_FNTPAL>"))!=NULL) {
				WinOpenObject(hObject,OPEN_DEFAULT,TRUE);
			}
			else {//Error, can't open the palette
				/*  Show an error msg.						   */
				WinLoadString(WinQueryAnchorBlock(hwnd),QueryModuleHandle(),ID_NOFONTPALETTESTRING,
											sizeof(msgText), msgText);
				WinMessageBox(HWND_DESKTOP,
											hwnd,         
											msgText,          
											(PSZ) NULL,                      
											ID_COLORPALETTEMSGBOX,                   /* Window ID */
											MB_OK |
											MB_MOVEABLE |
											MB_ICONEXCLAMATION |
											MB_DEFBUTTON1);                  /* Style     */
			}
			break;
		}
		/* Don't call WinDefDlgProc here, or the dialog gets closed */
		return (MRESULT) TRUE;
	}
	// The WinDefDlgProc() handles the rest of the messages
	return WinDefDlgProc(hwnd, msg, mp1, mp2);
}


/*************************************************************/
/* This dialog procedure handles the third option page       */
/*************************************************************/
MRESULT EXPENTRY OptionDlgProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2) {
	
 	switch(msg) {
 	case WM_INITDLG :	
		/* We have to initialize the dialog controls with the approbiate values */
		WinSetWindowULong(hwnd,QWL_USER,LONGFROMMP(mp2));
		// Set the focus on the smallsize checkbox
		WinSetFocus(HWND_DESKTOP, WinWindowFromID(hwnd, CWDLG_SMALLSIZE));
		// Set limits of the distance spinbutton
		WinSendMsg(WinWindowFromID(hwnd, CWDLG_DISTANCESPIN),
							 SPBM_SETLIMITS,(MPARAM)18,(MPARAM)4);
		// Set distance spinbutton to the current value of icondistance
		WinSendMsg(WinWindowFromID(hwnd, CWDLG_DISTANCESPIN),
							 SPBM_SETCURRENTVALUE,
							 (MPARAM)((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->icondistance,
							 (MPARAM)NULL);
		// Enable or disable the spin button according to the value of smallSizeEnabled
		WinEnableWindow(WinWindowFromID(hwnd, CWDLG_DISTANCESPIN),((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->smallSizeEnabled);
		// Set the state of the smallSizeEnabled checkbox
		WinSendMsg(WinWindowFromID(hwnd, CWDLG_SMALLSIZE),
							 BM_SETCHECK,MPFROMSHORT(((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->smallSizeEnabled),(MPARAM)NULL);
		// Enable or disable the noDrawers checkbox according to the value of smallSizeEnabled
		WinEnableWindow(WinWindowFromID(hwnd, CWDLG_NODRAWERS),((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->smallSizeEnabled);
		// Set the state of the noDrawers checkbox
		WinSendMsg(WinWindowFromID(hwnd, CWDLG_NODRAWERS),
							 BM_SETCHECK,MPFROMSHORT(((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->noDrawers),(MPARAM)NULL);
		return (MRESULT) TRUE;
	case WM_DESTROY:
		/* The notebook closes and gets destroyed */
		// Save the value to the instance data
		((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->wpSaveDeferred();
		break;				
	case WM_CONTROL:
		switch (SHORT2FROMMP(mp1)) {
		case SPBN_ENDSPIN:
			/* The value of the spinbutton has changed */
			ULONG spinValue;
			// Query the new spinbutton value
			WinSendMsg(WinWindowFromID(hwnd, CWDLG_DISTANCESPIN),
								 SPBM_QUERYVALUE,(MPARAM)&spinValue,
								 MPFROM2SHORT(0,SPBQ_ALWAYSUPDATE));						 			
			((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->icondistance=spinValue;
			// Refresh the size of the launchpad
			((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->wpRefreshDrawer(0);//0: LaunchPad
			break;
		case BN_CLICKED:
			switch(SHORT1FROMMP(mp1)) {
			case CWDLG_SMALLSIZE:
				/* checkbox was clicked */
				if(((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->smallSizeEnabled)  // Change the value of smallSizeEnabled
					((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->smallSizeEnabled=0;
				else 
					((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->smallSizeEnabled=1;				
				// Set the checkbox state
				WinSendMsg(WinWindowFromID(hwnd, CWDLG_SMALLSIZE),
									 BM_SETCHECK,MPFROMSHORT(((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->smallSizeEnabled),(MPARAM)NULL);
				// Enable/disable the spinbutton
				WinEnableWindow(WinWindowFromID(hwnd, CWDLG_DISTANCESPIN),((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->smallSizeEnabled);
				// Enable/disable the noDrawers checkbox
				WinEnableWindow(WinWindowFromID(hwnd, CWDLG_NODRAWERS),((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->smallSizeEnabled);
				// Refresh the size of the launchpad
				((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->wpRefreshDrawer(0);//0: LaunchPad
				break;
			case CWDLG_NODRAWERS:
				if(((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->wpQueryHideDrawers())  // Change the value of smallSizeEnabled
					((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->wpSetHideDrawers(0);
				else 
					((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->wpSetHideDrawers(1);
				// Set the checkbox state
		 		WinSendMsg(WinWindowFromID(hwnd, CWDLG_NODRAWERS),
									 BM_SETCHECK,MPFROMSHORT(((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->wpQueryHideDrawers()),(MPARAM)NULL);
				// Refresh the size of the launchpad
				((WPLnchCW*) WinQueryWindowULong(hwnd,QWL_USER))->wpRefreshDrawer(0);//0: LaunchPad
				break;
			} // end switch(SHORT1FROMMP(mp1))
			break;
		}// end switch SHORT2FROMMP(mp1)
		return (MRESULT) TRUE;
	}
	// The WinDefDlgProc() handles the rest of the messages
	return WinDefDlgProc(hwnd, msg, mp1, mp2);
}

/*****************************************************************************/
/*  New launchpad button procedure with fly over help							 				   */
/*****************************************************************************/
static MRESULT EXPENTRY newButtonProc(HWND hwnd, ULONG msg,MPARAM mp1,MPARAM mp2 )
{
	static BOOL bBubbleOn=TRUE;
	POINTL ptl;
	HPS  hps;
	FONTMETRICS   fm;
	LONG  ulWinTextLen;
	POINTL aptlPoints[TXTBOX_COUNT];
	RECTL   rcl;
	LONG   deltaX,deltaY;
	RGB    rgb= {200,200,0};
	static USHORT id=0;//Initialisation new in V1.00a 
	RECTL  rclWork;
	ULONG bubbleEnabled;
	HWND hwndStore;
	
	switch (msg)
		{
		case WM_DESTROY:
			WinStopTimer(WinQueryAnchorBlock(hwnd),hwnd,1);//Stop timer if running
			if(hwndBubbleWindow) WinDestroyWindow(hwndBubbleWindow);/*  close the bubblewindow  */
			hwndBubbleWindow=0;
			/* Stop delay timer if running */
			WinStopTimer(WinQueryAnchorBlock(hwnd),hwnd,WinQueryWindowUShort(hwnd,QWS_ID));			
			break;
		case WM_NEWBUBBLE:
			ULONG bubbleEnabled;
			HWND hwndStore;
			bubbleEnabled=1;
			hwndStore=WinWindowFromID(WinQueryWindow(hwnd,QW_PARENT),ID_STOREWINDOW);
			if(hwndStore)bubbleEnabled=((WPLnchCW*) WinQueryWindowULong(hwndStore,QWL_USER) )->bubbleEnabled;
			/*  we have to build a new information window																   */
			if(hwndBubbleWindow){// if(){...} new in V1.00a 
				WinDestroyWindow(hwndBubbleWindow);/*  close the bubblewindow  */
				hwndBubbleWindow=0;
			}
			// Query the pointer position
			WinQueryPointerPos(HWND_DESKTOP,&ptl);
			WinMapWindowPoints(HWND_DESKTOP,hwnd,&ptl,1);
			WinQueryWindowRect(hwnd,&rclWork);				
			if(!hwndBubbleWindow 
				 && WinPtInRect(WinQueryAnchorBlock(hwnd),&rclWork,&ptl)
				 && bubbleEnabled) {
				static HWND hwndBubbleClient;
				ULONG style=FCF_BORDER;
				char winText[255];

				/* Get window text (It's the object title) for size calculating */
				WinQueryWindowText(hwnd,sizeof(winText),winText);
				ulWinTextLen=(LONG)strlen(winText); // Query text length

				/* Delete 'Returns' in object title */
				char *pBuchst;
				pBuchst=strchr(winText,10);
				if(pBuchst)	winText[pBuchst-winText]=' ';
				pBuchst=strchr(winText,13);
				if(pBuchst)	winText[pBuchst-winText]=' ';
				
				/* Create help window */
				hwndBubbleWindow=WinCreateStdWindow(HWND_DESKTOP,
																						0,
																						&style,
																						WC_STATIC,
																						"",
																						SS_TEXT|DT_CENTER|DT_VCENTER,
																						NULLHANDLE,
																						400,
																						&hwndBubbleClient);
		 				
				// Set the font for the help
				WinSetPresParam(hwndBubbleClient,PP_FONTNAMESIZE,sizeof(((WPLnchCW*) WinQueryWindowULong(hwndStore,QWL_USER) )->fontName),
												&((WPLnchCW*) WinQueryWindowULong(hwndStore,QWL_USER) )->fontName);
				/* Calculate text size in pixel */
				hps=WinBeginPaint(hwndBubbleClient,(HPS)NULL,(PRECTL)NULL);
				GpiQueryTextBox(hps,ulWinTextLen,winText,TXTBOX_COUNT,aptlPoints);
				WinEndPaint(hps);
				
				/* Set colors */
				WinSetPresParam(hwndBubbleClient,
												PP_BACKGROUNDCOLOR,sizeof(RGB) ,&((WPLnchCW*) WinQueryWindowULong(hwndStore,QWL_USER) )->background );
				WinSetPresParam(hwndBubbleClient,
												PP_FOREGROUNDCOLOR,sizeof(RGB) ,&((WPLnchCW*) WinQueryWindowULong(hwndStore,QWL_USER) )->foreground );				

				/* Calculate bubble positon and show bubble */
				WinQueryPointerPos(HWND_DESKTOP,&ptl);//Query pointer position in the desktop window
				WinQueryWindowRect(HWND_DESKTOP,&rcl);//Query desktop size
				aptlPoints[TXTBOX_BOTTOMRIGHT].x-aptlPoints[TXTBOX_BOTTOMLEFT].x+7+xVal+ptl.x 
					> rcl.xRight 
					? deltaX=-aptlPoints[TXTBOX_BOTTOMRIGHT].x-aptlPoints[TXTBOX_BOTTOMLEFT].x-xVal-xVal-7 
					: deltaX=0 ;
				/*
				aptlPoints[TXTBOX_TOPLEFT].x-aptlPoints[TXTBOX_BOTTOMLEFT].x+2+yVal+ptl.x 
					> rcl.yTop 
					? deltaY=-aptlPoints[TXTBOX_BOTTOMRIGHT].x-aptlPoints[TXTBOX_BOTTOMLEFT].x-yVal
					: deltaY=0 ;	*/			
				aptlPoints[TXTBOX_TOPLEFT].y-aptlPoints[TXTBOX_BOTTOMLEFT].y+2+yVal+ptl.y 
					> rcl.yTop 
					? deltaY=-aptlPoints[TXTBOX_TOPLEFT].y-aptlPoints[TXTBOX_BOTTOMLEFT].y-2*yVal-7
					: deltaY=0 ;				
				WinSetWindowPos(hwndBubbleWindow,
												HWND_DESKTOP,
												ptl.x+xVal+deltaX,ptl.y+yVal+deltaY,  //deltaY value not used yet
												aptlPoints[TXTBOX_BOTTOMRIGHT].x-aptlPoints[TXTBOX_BOTTOMLEFT].x+8,
												aptlPoints[TXTBOX_TOPLEFT].y-aptlPoints[TXTBOX_BOTTOMLEFT].y+2,
												SWP_SIZE|SWP_MOVE|SWP_SHOW);
				/* Set bubble text */
				WinSetWindowText(hwndBubbleClient,winText);
				WinStartTimer(WinQueryAnchorBlock(hwnd),hwnd,1,35); 
			} // end if(!hwndBubbleWindow)
			break;
		case WM_MOUSEMOVE:
			USHORT  tempID;
			ULONG delayValue;
			delayValue=250;
			hwndStore=WinWindowFromID(WinQueryWindow(hwnd,QW_PARENT),ID_STOREWINDOW);
			if(hwndStore)delayValue=((WPLnchCW*) WinQueryWindowULong(hwndStore,QWL_USER) )->delayValue;
			tempID=WinQueryWindowUShort(hwnd,QWS_ID);/*  get the id of the window under the pointer  */  			
			if(id!=tempID) {	// New Button?	
				WinStartTimer(WinQueryAnchorBlock(hwnd),hwnd,tempID,(ULONG)delayValue); // New timer for delay
				id=tempID;  // Save ID 
			}
			else {
				if(!hwndBubbleWindow)WinStartTimer(WinQueryAnchorBlock(hwnd),hwnd,tempID,(ULONG)delayValue); // New timer for delay	
			}			
			break;
		case WM_TIMER:			
			switch (SHORT1FROMMP(mp1))
				{
				case 1: //Intervall timer
					/* Test pointer position */
					WinQueryPointerPos(HWND_DESKTOP,&ptl);
					WinMapWindowPoints(HWND_DESKTOP,hwnd,&ptl,1);
					WinQueryWindowRect(hwnd,&rclWork);
					if(!WinPtInRect(WinQueryAnchorBlock(hwnd),&rclWork,&ptl))
						{	// Button has changed				 
							WinStopTimer(WinQueryAnchorBlock(hwnd),hwnd,1);  // stop the running timer
							if(hwndBubbleWindow) WinDestroyWindow(hwndBubbleWindow);/*  close the bubblewindow  */
							hwndBubbleWindow=0;
							id=0;
						}			 			
					break;
				default:// delay over
					//					if (SHORT1FROMMP(mp1)==id) {
					if (SHORT1FROMMP(mp1)==WinQueryWindowUShort(hwnd,QWS_ID)) {//our own timer. It has got the window id
						WinStopTimer(WinQueryAnchorBlock(hwnd),hwnd,SHORT1FROMMP(mp1));//Stop the delay timer
						/* Check the pointer position */
						WinQueryPointerPos(HWND_DESKTOP,&ptl);
						WinMapWindowPoints(HWND_DESKTOP,hwnd,&ptl,1);
						WinQueryWindowRect(hwnd,&rclWork);
						if(WinPtInRect(WinQueryAnchorBlock(hwnd),&rclWork,&ptl))
							WinPostMsg(hwnd,WM_NEWBUBBLE,NULL,NULL);//Request a help window
					}
					break;
				}
			break;			
		}
	// call the original button procedure to handle the rest of the messages
	return (*oldButtonProc2)(hwnd,msg,mp1,mp2);	
};

/* This class function returns the name of our class */
PSZ M_WPLnchCW::wpclsQueryTitle()
{
	return "Ext-LaunchPad";
}

/************************************************************/
/* We have to override wpRefreshDrawer() cause we must know */
/* if there're new objects on the launchpad. Every object   */
/* has got a button and we have to change the window        */
/* procedure of the button                                  */
/************************************************************/ 
void WPLnchCW::wpRefreshDrawer(ULONG ulDrawer)
{
	
	// Call parent so it can update its private structures and build the
	// buttons
	// We will change the button procedures later on to provide the 
	// bubble help 
	((somTD_WPLaunchPad_wpRefreshDrawer)
	 somParentNumResolve(__ClassObject->somGetPClsMtabs(),
											 1,
											 __ClassObject->
											 somGetMethodToken(somIdFromString
																				 ("wpRefreshDrawer")))
	 )(this,ulDrawer);		
	// This piece of code looks strange, because we're doing no additional processing. Only if we're doing that
	// wpSetDrawerHWND(ULONG ulDrawer,HWND hwnd) is called with the new hwnd. Might be a bug.
	// Docs say wpSetDrawer() is called every time a drawer is build. I realized it is only called when the drawer
	// closes. 
	return;
}

void WPLnchCW::wpSetDrawerHWND(ULONG ulDrawer,HWND hwndDrawer)
{
	SWP  swp;
	SWP  lpSWP;
	SWP  desktopSWP;
	
	HWND hwnd;
	HWND hwndNext;
	HWND hwndStore;
	
	HENUM henum;
	BOOL  succ;
	USHORT  id;
	PFNWP  tempProc;
	ULONG  num;
	int maxcy,maxcy1,maxcy3,maxcy2,maxcx,maxcx1,maxcx2,maxcx3;
	ULONG xpos,ypos;
	ULONG res;
	ULONG tempBit;	
	int titlebarSize;
	int actionButtonsText;

	int textDelta2;
	int textDelta;
	PSZ  objectTitle;
	HOBJECT *hobject;
	BOOL  isVertical;
	WPObject *wpObject;
	int hasActionButtons;

#ifdef debug
								char buf2[260];
								sprintf(buf2,"wpSetDrawerHWND() beginning    ulDrawer: %x hwndDrawer: %x ",ulDrawer,hwndDrawer);
								showMessage(HWND_DESKTOP,buf2);
#endif								
						
	actionButtonsText=0;	
	hasActionButtons=1;

	if(!hwndDrawer){//Drawer closes, no reordering of buttons necessary
		if(ulDrawer)
			{//We have a drawer. Mark it as closed
				ULONG tempBit;
				tempBit=1;
				tempBit=tempBit<<(ulDrawer-1);
				ulOpenDrawers&=~tempBit;
			}
	((somTD_WPLaunchPad_wpSetDrawerHWND)
	 somParentNumResolve(__ClassObject->somGetPClsMtabs(),
											 1,
											 __ClassObject->
											 somGetMethodToken(somIdFromString
																				 ("wpSetDrawerHWND")))
	 )(this,ulDrawer,hwndDrawer);				
		return;
	}	

	if(!WinWindowFromID(hwndDrawer,200)){//That's the first Button. If we have it, the launchpad is built.
		// No the pad isn't ready. Call parent and wait for next call
		((somTD_WPLaunchPad_wpSetDrawerHWND)
		 somParentNumResolve(__ClassObject->somGetPClsMtabs(),
												 1,
												 __ClassObject->
												 somGetMethodToken(somIdFromString
																					 ("wpSetDrawerHWND")))
		 )(this,ulDrawer,hwndDrawer);			
		return;
	}
	

	// The original Launchpad has built the buttons. We can now reorder them. The Launchpad isn't visible yet.
	// When calling wpSetDrawerHWND() of the parent the launchpad will be displayed

	if(ulDrawer)
		{//We have a drawer. Mark it as opened, so we can save the open state on shutdown and reopen it on IPL
			ULONG tempBit;
			tempBit=1;
			tempBit=tempBit << (ulDrawer-1);
			ulOpenDrawers|=tempBit;		
#ifdef debug
								char buf2[260];
								sprintf(buf2,"ulDrawer: %x ulOpenDrawers:  %x  tempBit: %x",ulDrawer,ulOpenDrawers,tempBit);								
								//	showMessage(HWND_DESKTOP,buf2);
#endif
												
								wpSaveDeferred();//Save open state
		}
			 
	//Query state of ActionButtons?
	res=wpQueryActionButtonStyle();	
	if(res==ACTION_BUTTONS_OFF)hasActionButtons=0;
	if(res==ACTION_BUTTONS_TEXT)actionButtonsText=1;

	//The Warpcenter is a Child of WPLaunchPad, so it inherits all the features.
	// We substitute WPLaunchPad with WPLnchCW, so the Warpcenter inherits the 
	// bubblehelp but it doesn't need it cause it has its own. So if it's the WarpCenter
	// do nothing.
	if(strcmp(somGetClassName(),"SmartCenter")) {// 0 if equal
		// It's not the Warpcenter
		
		// Query the window handle of the drawer
		//It may be the launchpad itself or a drawer
		//hwnd=wpQueryDrawerHWND(ulDrawer);
		hwnd=hwndDrawer;
		
		if(!WinWindowFromID(hwnd,ID_STOREWINDOW))//We use this window to store our object ptr
			WinCreateWindow(hwnd,WC_BUTTON,"StoreWindow",0,0,0,0,0,NULL,HWND_TOP,ID_STOREWINDOW,NULL,NULL);
		hwndStore=WinWindowFromID(hwnd,ID_STOREWINDOW);
		if(hwndStore)WinSetWindowULong(hwndStore,QWL_USER,(ULONG)this);
		isVertical=wpQueryDisplayVertical();//true if vertical 
		if(hwnd) {  // Just to catch errors
#ifdef debug
			//	showMessage(hwnd,"Main window");
#endif
			// Query the list of objects living on this drawer
			hobject=wpQueryObjectList(ulDrawer,&num);
			xpos=0;
			maxcx=0;
			maxcy=0;						
			maxcx1=-1;
			maxcy1=-1;

			henum=WinBeginEnumWindows(hwnd);// We will query all child windows
			while((hwndNext=WinGetNextWindow(henum))!=NULLHANDLE) {
				// Get windowsize and position
				WinQueryWindowPos(hwndNext,(PSWP)&swp);
				
				// Query window ID
				// The small drawer buttons have got ID 3XX
				// The object buttons have got ID 2XX
				// The action buttons (e.g. shutdown) have got ID 1XX
				id=WinQueryWindowUShort(hwndNext,QWS_ID);	
				if(id<400) {
					if(id>=300) //ID 3XX Drawer-Buttons
						{
							maxcy3=swp.cy;//Vert. size of Drawer-Buttons. We need it later
							maxcx3=swp.cx;//Horz. size of Drawer-Buttons
							if(noDrawers){
								maxcx3=-(int)(icondistance/2);
								maxcy3=-(int)(icondistance/2);
#ifdef debug
								char buf[260];
								sprintf(buf,"maxcx3: %d,     maxcy3:   %d,    icondistance: %d",maxcx3,maxcy3,icondistance);
								//					showMessage(HWND_DESKTOP,buf);
#endif									
							}
						}
					else{
						if(id>=200){   //ID 2XX object Buttons on the launchpad
							maxcy2=swp.cy;//Vert. sizeize of Prog-Buttons
							maxcx2=swp.cx;//Horz. size of Prog-Buttons
							maxcx+=swp.cx+icondistance;  // We need this for sizechanging of the launchpad
							maxcy+=swp.cy+icondistance;

							//		maxcx+=swp.cx+icondistance/2;  // We need this for sizechanging of the launchpad
							//		maxcy+=swp.cy+icondistance/2;// When using delta
						}
						else{
							if(id==100){ //ID 1XX System-Buttons auf der Klickstart
								xpos=swp.x;//x-pos of leftmost button
								ypos=swp.y;
								maxcx1=swp.cx;
								maxcy1=swp.cy;
								maxcx+=((swp.cx+1)*4);// There're 4 system buttons (normaly)
								maxcy+=((swp.cy+1)*4);							
							}
						}
					}				
				}//end id<400
				else {//id>400. One of these windows must be the Titlebar
					char classname[4];
					//Check if it's the titlebar
					if(WinQueryClassName(hwndNext,sizeof(classname),classname)==2) {
						if(*(classname+1)==0x39) {//WC_TITLEBAR
							// We can't use the system defined value for the titlebar size
							// because the user may use a custom font for the titlebar so
							// the size is unknown.
							WinQueryWindowPos(hwndNext,(PSWP)&swp);//Titlebardimensions
							// We have to take care of the titlebar size if we resize the launchpad
							// so save the value
							titlebarSize=swp.cy;//y-size=0 if titlebar isn't visible
						}
					}					
				}
			}
			succ=WinEndEnumWindows(henum);
			/****************************************
			 *                     
			 * Size Launchpad 
			 *
			 ****************************************/
#ifdef debug
			//	showMessage(hwnd,"Size Launchpad");
#endif
			WinQueryWindowPos(HWND_DESKTOP,(PSWP)&desktopSWP);//Get desktop size
			WinQueryWindowPos(hwnd,(PSWP)&swp);//Get launchpadsize or drawersize
			// Only resize launchpad if user enabled this feature
			if(smallSizeEnabled) {
				// Hide the launchpad (or drawer) so the user can't see the resizing
				//	WinShowWindow(hwnd,FALSE);				
				HWND lpHWND=wpQueryDrawerHWND(0);//Query HWND of launchpad
				
				//*************** Launchpad is horizontal				
				if(!isVertical) {
					if(!ulDrawer){//Launchpad
						WinSetWindowPos(hwnd,NULL,0,0,
														maxcx+(1+hasActionButtons)*icondistance-actionButtonsText*maxcx1*2,
														maxcy3+maxcy2+icondistance/2+2*icondistance+titlebarSize,
														SWP_SIZE|SWP_HIDE);
						/*WinSetWindowPos(hwnd,NULL,0,0,
							maxcx+(1+hasActionButtons)*icondistance-actionButtonsText*maxcx1*2+icondistance/2,
							maxcy3+maxcy2+icondistance/2+2*icondistance+titlebarSize-delta,
							SWP_SIZE|SWP_HIDE);
							*/
					}
					else {  //Drawer
						if(WinQueryWindowPos(lpHWND,(PSWP)&lpSWP)&&swp.y<lpSWP.y){//got launchpad size and drawer is below launchpad
							//Check if small drawer fits above launchpad
							if((icondistance+maxcy+titlebarSize)<(desktopSWP.cy-(lpSWP.y+lpSWP.cy+titlebarSize)))
								{// yes, place it above
									WinSetWindowPos(hwnd,NULL,swp.x+(swp.cx-maxcx2-icondistance)/2,
																	lpSWP.y+lpSWP.cy+titlebarSize,
																	maxcx2+icondistance,
																	icondistance+maxcy+titlebarSize,
																	SWP_SIZE|SWP_MOVE);
								}
							else
								{//No, no place for drawer
									WinSetWindowPos(hwnd,NULL,swp.x+(swp.cx-maxcx2-icondistance)/2,
																	lpSWP.y-(icondistance+maxcy+titlebarSize),
																	maxcx2+icondistance,
																	icondistance+maxcy+titlebarSize,
																	SWP_SIZE|SWP_MOVE);						
								}
						}
						else {//couldn't get launchpad size, using default positions
							WinSetWindowPos(hwnd,NULL,swp.x+(swp.cx-maxcx2-icondistance)/2,
															swp.y,
															maxcx2+icondistance,
															icondistance+maxcy+titlebarSize,
															SWP_SIZE|SWP_MOVE);
						}
					}
				}
				// ************** Launchpad is vertical **************
				else {
					if(!ulDrawer){//Launchpad
						WinSetWindowPos(hwnd,NULL,0,0,maxcx3+maxcx2+icondistance/2+2*icondistance,
														maxcy+titlebarSize+(1+hasActionButtons)*icondistance,SWP_SIZE);
						//WinSetWindowPos(hwnd,NULL,0,0,maxcx3+maxcx2+icondistance/2+2*icondistance-delta,
						//								maxcy+titlebarSize+(1+hasActionButtons)*icondistance+icondistance/2,SWP_SIZE);
					}
					else {  //Drawer needs some position tweaking
						if(WinQueryWindowPos(lpHWND,(PSWP)&lpSWP)&&swp.x<lpSWP.x){//got launchpad size and drawer is left of launchpad
							//Check if small drawer fits to the right of launchpad
							if((icondistance+maxcx)<(desktopSWP.cx-(lpSWP.x+lpSWP.cx)))
								{// yes, place it on the right side
									WinSetWindowPos(hwnd,NULL,lpSWP.x+lpSWP.cx,
																	swp.y+(swp.cy-maxcy2-icondistance)/2,
																	maxcx+icondistance,
																	icondistance+maxcy2+titlebarSize,
																	SWP_SIZE|SWP_MOVE);
								}
							else
								{//No, no place for drawer
									WinSetWindowPos(hwnd,NULL,lpSWP.x-(maxcx+icondistance),
																	swp.y+(swp.cy-maxcy2-icondistance)/2,
																	maxcx+icondistance,
																	icondistance+maxcy2+titlebarSize,
																	SWP_SIZE|SWP_MOVE);						
								}
						}
						else {//couldn't get launchpad size, using default positions
							WinSetWindowPos(hwnd,NULL,
															swp.x,
															swp.y+(swp.cy-maxcy2-icondistance)/2,
															icondistance+maxcx,
															maxcy2+icondistance+titlebarSize,SWP_SIZE|SWP_MOVE);
						}
					}	
				}
			}			
			/****************************************
			 *                     
			 * Size the buttons and change the window procedure
			 *
			 ****************************************/
#ifdef debug
			//	showMessage(hwnd,"Size buttons and change window procedures");
#endif
			henum=WinBeginEnumWindows(hwnd);// We will query all child windows
			while((hwndNext=WinGetNextWindow(henum))!=NULLHANDLE) {
				// Query the current position of the buttons
				WinQueryWindowPos(hwndNext,(PSWP)&swp);
				// Query the window ID
				id=WinQueryWindowUShort(hwndNext,QWS_ID);
				if(id>=300)//ID 3XX Drawer-Icons
					{
						if(smallSizeEnabled) {
							if(id<400) {
								if(!noDrawers) {
									//**************** Launchpad is horizontal *************	
									if(!isVertical) {
										// Set new position and size of the icons
										WinSetWindowPos(hwndNext,NULL,
																		4*(maxcx1+1)+(id-300)*(maxcx2+icondistance)
																		-actionButtonsText*maxcx1*2
																		+(1+hasActionButtons)*icondistance,
																		maxcy2+icondistance/2+icondistance,
																		swp.cx-delta,swp.cy-delta,SWP_MOVE|SWP_SIZE);
										/*		WinSetWindowPos(hwndNext,NULL,
													4*(maxcx1+1)+(id-300)*(maxcx2+icondistance/2)
													-actionButtonsText*maxcx1*2
													+(1+hasActionButtons)*icondistance,
													maxcy2+icondistance/2+icondistance,
													swp.cx-delta,swp.cy-delta,SWP_MOVE|SWP_SIZE);*/
									}
								//*************** Launchpad is vertical *****************
								else{
									//  Set new position and size of the icons
									WinSetWindowPos(hwndNext,NULL,maxcx2+icondistance/2+icondistance,
										4*(maxcy1+1)+(id-300)*(icondistance+maxcy2)
										+(1+hasActionButtons)*icondistance,
										swp.cx,swp.cy-delta,SWP_MOVE|SWP_SIZE);
									/*									WinSetWindowPos(hwndNext,NULL,maxcx2+icondistance/2+icondistance,
																			4*(maxcy1+1)+(id-300)*(icondistance/2+maxcy2)
																			+(1+hasActionButtons)*icondistance,
																			swp.cx,swp.cy-delta,SWP_MOVE|SWP_SIZE);*/
								}	// else
								} //if(!NODRAWERS)
							else {
								WinSetWindowPos(hwndNext,NULL,0,0,0,0,SWP_HIDE);
								
							} // else if(!NODRAWERS)
							}// endif(id<400)
						}// endif(smallSizeEnabled)
					}
			else //id>=300
				{
					if(id>=200)//ID 2XX Progicons auf der Klickstartleiste
						{
							// We subclass the window so we can track mouse movements
							// over the buttons and can build a bubble help if the mouse
							// resides over a button.
							// WinSubclassWindow() returns the old procedure and sets our procedure
							// as the default.
							// If the returned procedure is the same as our own we have already subclassed
							// the window otherwise we save the original procedure pointer.
							tempProc=WinSubclassWindow(hwndNext,newButtonProc);						
							if(tempProc!=(*newButtonProc)){
								oldButtonProc2=tempProc; 
							}
							// Now we need the class object to get a WPObject* from the HOBJECT
							
#ifdef debug
							//		showMessage(hwnd,"Inside enumeration loop...ProgIcons");
#endif
							wpObject=M_ob.wpclsQueryObject(hobject[id-200]);
							if(wpObject)
								objectTitle=wpObject->wpQueryTitle();
							else
								objectTitle="";
#ifdef debug
							//		showMessage(hwnd,"Inside enumeration loop...ProgIcons after wpQueryTitle()");
#endif
							
							// Set the object title as the window title. In our button proc we
							// take this window title and display it in the help window.
							// The original launchpad only sets the window title in this way if
							// you've switched on text on the launchpad.
							WinSetWindowText(hwndNext,objectTitle);
							
#ifdef debug
							//		showMessage(hwnd,"ID >=200, Program-Icons");
#endif
							if(smallSizeEnabled)
								WinSetWindowBits(hwndNext,QWL_STYLE,BS_NOPOINTERFOCUS,BS_NOPOINTERFOCUS);
							if(!ulDrawer)//Launchpad
								{
									if(smallSizeEnabled) {
										//************ Horizontal
										if(!isVertical) 
											// Set new position and size of the icons
											WinSetWindowPos(hwndNext,NULL,
																			4*(maxcx1+1)+(id-200)*(maxcx2+icondistance)
																			+(1+hasActionButtons)*icondistance
																			-actionButtonsText*maxcx1*2,
																			icondistance,swp.cx-delta,swp.cy-delta,
																			SWP_MOVE|SWP_SIZE);
										/* WinSetWindowPos(hwndNext,NULL,
											 4*(maxcx1+1)+(id-200)*(maxcx2+icondistance/2)
											 +(1+hasActionButtons)*icondistance
											 -actionButtonsText*maxcx1*2,
											 icondistance,swp.cx-delta,swp.cy-delta,
											 SWP_MOVE|SWP_SIZE);*/
												
												
												//************ Vertikal
												else  
													// Set new position and size of the icons
													WinSetWindowPos(hwndNext,NULL,icondistance,
																					4*(maxcy1+1)+(id-200)*(icondistance+maxcy2)
																					+(1+hasActionButtons)*icondistance,								
																					swp.cx-delta,swp.cy-delta,SWP_MOVE|SWP_SIZE);
										/*				WinSetWindowPos(hwndNext,NULL,icondistance,
															4*(maxcy1+1)+(id-200)*(icondistance/2+maxcy2)
															+(1+hasActionButtons)*icondistance,								
															swp.cx-delta,swp.cy-delta,SWP_MOVE|SWP_SIZE);*/
												}//endif(smallSizeEnabled)
								}// endif(!ulDrawer)
							else//Launchpad-drawer
								{
									if(smallSizeEnabled) {
										//************ Horizontal
										if(!isVertical)WinSetWindowPos(hwndNext,NULL,icondistance/2,
																									 icondistance+(id-200)*(maxcy2+icondistance),
																									 swp.cx-delta,swp.cy-delta,SWP_MOVE|SWP_SIZE);
										//************ Vertikal
										else	WinSetWindowPos(hwndNext,NULL,//swp.x,
																					icondistance+(id-200)*(maxcx2+icondistance),
																					icondistance/2,swp.cx-delta,
																					swp.cy-delta,SWP_MOVE|SWP_SIZE);	
									}
								}//#endelse	Launchpad-drawer						
						}
					else  //ID 1XX; shutdown, window list etc.
						{	
							tempProc=WinSubclassWindow(hwndNext,newButtonProc);						
							if(tempProc!=(*newButtonProc)){
								oldButtonProc2=tempProc;
							}
							if(smallSizeEnabled) {
								WinSetWindowBits(hwndNext,QWL_STYLE,BS_NOPOINTERFOCUS,BS_NOPOINTERFOCUS);
								//********** Launchpad is horizontal	
								if(!isVertical)
									if(actionButtonsText) {//ActionButtons with Text
										textDelta=id-100;
										textDelta2=textDelta;
										textDelta2/=2;
										textDelta&=1;
										WinSetWindowPos(hwndNext,NULL,
																		icondistance+textDelta2*(1+maxcx1),
																		//	icondistance+((icondistance/2)+maxcy1)*textDelta,
																		icondistance+((icondistance/2)+maxcy1-icondistance*noDrawers*2)*textDelta,
																		swp.cx,(maxcy2+maxcy3)/2,SWP_MOVE|SWP_SIZE);
									}
									else {
										WinSetWindowPos(hwndNext,NULL,
																		icondistance+(id-100)*(1+maxcx1),
																		icondistance,
																		swp.cx-delta,maxcy2+maxcy3+icondistance/2-delta,SWP_MOVE|SWP_SIZE);
									}
								//******** Verikal
								else WinSetWindowPos(hwndNext,NULL,icondistance,
																		 icondistance+(id-100)*(1+swp.cy),
																		 maxcx2+maxcx3+icondistance/2-delta,swp.cy-delta,SWP_MOVE|SWP_SIZE);
							}// endif(smallSizeEnabled)
						};
				};
			};
	succ=WinEndEnumWindows(henum);
	WinSetWindowPos(hwnd,NULL,0,0,0,0,SWP_SHOW);  // Show the launchpad
		};		
	}; // endif strcmp()
//Make Launchpad visible
	((somTD_WPLaunchPad_wpSetDrawerHWND)
	 somParentNumResolve(__ClassObject->somGetPClsMtabs(),
											 1,
											 __ClassObject->
											 somGetMethodToken(somIdFromString
																				 ("wpSetDrawerHWND")))
	 )(this,ulDrawer,hwndDrawer);			
	return;
}

/********************************************************/
/* New class function which inserts the bubblehelp page */
/* into the settings notebook                           */
/********************************************************/
ULONG WPLnchCW::AddBubblehelpPage(HWND hwndNotebook)
{
	PAGEINFO pageinfo;
	char pageName[50];
	
	//Clear the pageinfo structure
	memset((PCH)&pageinfo, 0, sizeof(PAGEINFO));
	//Fill the pageinfo structure
	pageinfo.cb = sizeof(PAGEINFO);
	pageinfo.hwndPage = NULLHANDLE;
	//The page has got a major tab and status text
	pageinfo.usPageStyleFlags = BKA_MAJOR|BKA_STATUSTEXTON;
	pageinfo.usPageInsertFlags = BKA_FIRST;
	//We want page numbers
	pageinfo.usSettingsFlags = SETTINGS_PAGE_NUMBERS;
	//The dialog procedure for this page
	pageinfo.pfnwp = BubblehelpDlgProc;
	//The resource is in the class file
	pageinfo.resid = QueryModuleHandle();
	pageinfo.dlgid = ID_BUBBLEENABLE;
	pageinfo.pszName = pageName;		
	//We need a pointer to our WPS-object in the dialog procedure
	//to call class functions
	pageinfo.pCreateParams = this;
	//The ID of the help panel for this page
	pageinfo.idDefaultHelpPanel = 001;
	//Tell the WPS the help library name
	pageinfo.pszHelpLibraryName = "bubblepd.hlp";
	//Load the name for the notebookpage
	WinLoadString(WinQueryAnchorBlock(hwndNotebook),QueryModuleHandle(),
								ID_BUBBLESTRING,sizeof(pageName),pageName);
	
	//Insert the page into the settings notebook
	return wpInsertSettingsPage(hwndNotebook,&pageinfo);
}

/********************************************************/
/* New class function which inserts the colour and font */
/* page into the settings notebook                      */
/********************************************************/
ULONG WPLnchCW::AddColorchangePage(HWND hwndNotebook)
{
	PAGEINFO pageinfo;
	
	//Clear the pageinfo structure
	memset((PCH)&pageinfo, 0, sizeof(PAGEINFO));
	//Fill the pageinfo structure
	pageinfo.cb = sizeof(PAGEINFO);
	pageinfo.hwndPage = NULLHANDLE;
	pageinfo.usPageStyleFlags = BKA_STATUSTEXTON;
	pageinfo.usPageInsertFlags = BKA_FIRST;
	//We want page numbers
	pageinfo.usSettingsFlags = SETTINGS_PAGE_NUMBERS;	
	//The dialog procedure for this page
	pageinfo.pfnwp = ColorchangeDlgProc;
	//The resource is in the class file
	pageinfo.resid = QueryModuleHandle();
	pageinfo.dlgid = ID_COLORCHANGE;
	//We need a pointer to our WPS-object in the dialog procedure
	//to call class functions
	pageinfo.pCreateParams = this;
	//The ID of the help panel for this page
	pageinfo.idDefaultHelpPanel = 202;
	//Tell the WPS the help library name
	pageinfo.pszHelpLibraryName = "bubblepd.hlp";
	//We have no major tab so we need no name
	
	//Insert the page into the settings notebook
	return wpInsertSettingsPage(hwndNotebook,&pageinfo);
}

/********************************************************/
/* New class function which inserts the size option     */
/* page into the settings notebook                      */
/********************************************************/
ULONG WPLnchCW::AddOptionPage(HWND hwndNotebook)
{
	PAGEINFO pageinfo;
	
	//Clear the pageinfo structure
	memset((PCH)&pageinfo, 0, sizeof(PAGEINFO));
	//Fill the pageinfo structure
	pageinfo.cb = sizeof(PAGEINFO);
	pageinfo.hwndPage = NULLHANDLE;
	pageinfo.usPageStyleFlags = BKA_STATUSTEXTON;
	pageinfo.usPageInsertFlags = BKA_FIRST;
	//We want page numbers
	pageinfo.usSettingsFlags = SETTINGS_PAGE_NUMBERS;
	//The dialog procedure for this page
	pageinfo.pfnwp = OptionDlgProc;
	//The resource is in the class file
	pageinfo.resid = QueryModuleHandle();
	pageinfo.dlgid = ID_OPTIONPAGE;
	//We need a pointer to our WPS-object in the dialog procedure
	//to call class functions
	pageinfo.pCreateParams = this;
	//The ID of the help panel for this page
	pageinfo.idDefaultHelpPanel = 200;
	//Tell the WPS the help library name
	pageinfo.pszHelpLibraryName = "bubblepd.hlp";
	//We have no major tab so we need no name

	//Insert the page into the settings notebook
	return wpInsertSettingsPage(hwndNotebook,&pageinfo);
}

BOOL WPLnchCW::wpAddLaunchPadPage2(HWND hwndNotebook)
{
	return(AddOptionPage(hwndNotebook)&&
				 ((somTD_WPLaunchPad_wpAddLaunchPadPage2)
					somParentNumResolve(__ClassObject->somGetPClsMtabs(),
															1,
															__ClassObject->
															somGetMethodToken(somIdFromString
																								("wpAddLaunchPadPage2")))
					)(this,hwndNotebook));		
}


BOOL WPLnchCW::wpAddSettingsPages(HWND hwndNotebook)
{
	ULONG   ulPageIdColor;
	ULONG   ulPageIdBubble;
	ULONG   ulPageIdUninstall;

	//The Warpcenter is a Child of WPLaunchPad, so it inherits all the features.
	// We substitute WPLaunchPad with WPLnchCW, so the Warpcenter inherits the 
	// Bubblehelp pages but it doesn't need them cause it has its own bubblehelp.
	
	// Check if it's the Warpcenter (classname is 'SmartCenter')
	// somGetClassName() returns our class name
	if(strcmp(somGetClassName(),"SmartCenter")) {// 0 if equal
		//Insert our settings pages
		ulPageIdColor=AddColorchangePage(hwndNotebook);
		ulPageIdBubble=AddBubblehelpPage(hwndNotebook);
		return(ulPageIdColor 
					 && ulPageIdBubble
					 //call parent
					 &&((somTD_WPObject_wpAddSettingsPages)
							somParentNumResolve(__ClassObject->somGetPClsMtabs(),
																	1,
																	__ClassObject->
																	somGetMethodToken(somIdFromString
																										("wpAddSettingsPages")))
							)(this,hwndNotebook));
		
	}
	else // If SmartCenter only call parent
	return	((somTD_WPObject_wpAddSettingsPages)
					 somParentNumResolve(__ClassObject->somGetPClsMtabs(),
															 1,
															 __ClassObject->
															 somGetMethodToken(somIdFromString
																								 ("wpAddSettingsPages")))
					 )(this,hwndNotebook);
					 
	// Durch obige Konstruktion wird die Elternmethode des direkten Vorgngers aufgerufen,
	// und nicht die ursprngliche von WPLaunchPad, falls WPLaunchPad replaced wurde.
	// Durch diesen Aufruf gehen eventuelle Erweiterungen durch Vererbung verloren:
	//            < return WPLaunchPad::wpMenuItemSelected(hwndFrame, ulMenuId); >
	// Dieser Aufruf macht Ergnzungen in wpobject.hh erforderlich!!! Wird durch ein VAC-Fixpack vielleicht spter
	//erledigt
}
BOOL WPLnchCW::wpSaveState()
{
#ifdef debug
								char buf[260];
								sprintf(buf,"wpSaveState()  ulOpenDrawers: %x",ulOpenDrawers);
								showMessage(HWND_DESKTOP,buf);
#endif
	wpSaveLong("WPLnchCW",ID_NODRAWERS,noDrawers);
	wpSaveLong("WPLnchCW",ID_SMALLSIZEENABLED,smallSizeEnabled);
	wpSaveLong("WPLnchCW",ID_DELAYVALUE,delayValue);
	wpSaveLong("WPLnchCW",ID_ICONDISTANCE,icondistance);
	wpSaveLong("WPLnchCW",ID_BUBBLEENABLED,bubbleEnabled);
	wpSaveLong("WPLnchCW",ID_OPENDRAWERBITS,ulOpenDrawers);
	wpSaveData("WPLnchCW",ID_BACKGROUND,(PBYTE)&background,sizeof(RGB));
	wpSaveData("WPLnchCW",ID_FOREGROUND,(PBYTE)&foreground,sizeof(RGB));
	wpSaveString("WPLnchCW",ID_FONT,(PBYTE)&fontName);

	return((somTD_WPObject_wpSaveState)
				 somParentNumResolve(__ClassObject->somGetPClsMtabs(),
														 1,
														 __ClassObject->
														 somGetMethodToken(somIdFromString
																							 ("wpSaveState")))					 
				 )(this);
}


BOOL WPLnchCW::wpRestoreState(ULONG ulReserved)
{
	BOOL  rc;
	ULONG dataSize=sizeof(USHORT);
	
	/* Read values from instance data                                         */
	/* If we start the BubblePad for the first time there're no values in the */
	/* instance data. In this case use defaults.                              */

	if(!wpRestoreLong("WPLnchCW",ID_NODRAWERS,&noDrawers))
		noDrawers=0;
	if(!wpRestoreLong("WPLnchCW",ID_SMALLSIZEENABLED,&smallSizeEnabled))
		smallSizeEnabled=1;
	if(!wpRestoreLong("WPLnchCW",ID_DELAYVALUE,&delayValue))
		delayValue=250;
	if(!wpRestoreLong("WPLnchCW",ID_ICONDISTANCE,&icondistance))
		icondistance=4;
	if(!wpRestoreLong("WPLnchCW",ID_BUBBLEENABLED,&bubbleEnabled))
		bubbleEnabled=1;
	if(!wpRestoreLong("WPLnchCW",ID_OPENDRAWERBITS,&ulOpenDrawers))
		ulOpenDrawers=(ULONG)0;
	dataSize=sizeof(RGB);
	if(!wpRestoreData("WPLnchCW",ID_BACKGROUND,(PBYTE)&background,&dataSize)) {
		background.bBlue=180;
		background.bGreen=255;
		background.bRed=255;
	}
	dataSize=sizeof(RGB);
	if(!wpRestoreData("WPLnchCW",ID_FOREGROUND,(PBYTE)&foreground,&dataSize)) {
		foreground.bRed=0;
		foreground.bGreen=0;
		foreground.bBlue=0;
	}
	memset(fontName,0,sizeof(fontName));	
	dataSize=sizeof(fontName);
	if(!wpRestoreString("WPLnchCW",ID_FONT,(PBYTE)&fontName,&dataSize))
		memcpy(&fontName,"8.Helv",sizeof("8.Helv"));
	
	return((somTD_WPObject_wpRestoreState)
				 somParentNumResolve(__ClassObject->somGetPClsMtabs(),
														 1,
														 __ClassObject->
														 somGetMethodToken(somIdFromString
																							 ("wpRestoreState")))					 
				 )(this,ulReserved);
	
	// Durch obige Konstruktion wird die direkte Elternmethode aufgerufen,
	// und nicht die ursprngliche von WPLaunchPad, falls WPLaunchPad replaced wurde.
	// Durch diesen Aufruf gehen eventuelle Erweiterungen durch Vererbung verloren:
	//    
}

HWND  WPLnchCW::wpOpen(HWND hwndCnr,ULONG ulView,ULONG ulParam)
{
	HWND rcHWND;
	HWND hwndDrawerButton;
	ULONG ulTemp;
	int a;
	
	rcHWND=((somTD_WPObject_wpOpen)
				 somParentNumResolve(__ClassObject->somGetPClsMtabs(),
														 1,
														 __ClassObject->
														 somGetMethodToken(somIdFromString
																							 ("wpOpen")))					 
				 )(this,hwndCnr, ulView, ulParam);		

	if(ulView==OPEN_SETTINGS)	return rcHWND;

	// The parent function opened the launchpad. We look which drawers were open on last shutdown and
	// reopen them now.
	ulTemp=ulOpenDrawers;
#ifdef debug
								char buf[260];
								sprintf(buf,"ulOpenDrawers:  %x",ulTemp);								
								showMessage(HWND_DESKTOP,buf);
#endif

		for(a=0;a<32;a++) {
		if(ulTemp&1) {
			hwndDrawerButton=WinWindowFromID(rcHWND,300+a);
			if(hwndDrawerButton)
				WinPostMsg(hwndDrawerButton,BM_CLICK,MPFROMSHORT(FALSE),MPFROMSHORT(0));
		}
		ulTemp=ulTemp>>1;
	}
	return rcHWND;
} 










