MODULE Weasel;

        (********************************************************)
        (*                                                      *)
        (*             Combined POP3/SMTP server                *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            12 April 1998                   *)
        (*  Last edited:        11 November 1999                *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (*     Inetd option not tested.                         *)
        (*     Otherwise this module seems to be complete.      *)
        (*                                                      *)
        (********************************************************)

IMPORT OS2, TextIO;

FROM SYSTEM IMPORT LOC;

FROM IOChan IMPORT
    (* type *)  ChanId;

FROM Sockets IMPORT
    (* const*)  NotASocket,
    (* type *)  Socket, SockAddr, AddressFamily, SocketType,
    (* proc *)  sock_init, socket, so_cancel, setsockopt,
                bind, listen, select, accept, soclose, psock_errno,
                getsockname;

FROM Internet IMPORT
    (* const*)  Zero8, INADDR_ANY;

FROM LowLevel IMPORT
    (* proc *)  EVAL;

FROM Timer IMPORT
    (* proc *)  Sleep;

FROM Semaphores IMPORT
    (* type *)  Semaphore,
    (* proc *)  CreateSemaphore, Wait, Signal;

FROM TaskControl IMPORT
    (* proc *)  CreateTask;

FROM SplitScreen IMPORT
    (* proc *)  NotDetached, WriteStringAt, WriteString, WriteLn;

FROM InetUtilities IMPORT
    (* proc *)  ConvertDecimal, OpenINIFile, INIGet,
                Swap2, AddToTransactionLog, WriteError, WriteCard;

FROM CtrlC IMPORT
    (* proc *)  SetBreakHandler;

FROM WSession IMPORT
    (* proc *)  SetVersion, SetTimeout, SetMaxUsers,
                NewSession, NumberOfUsers;

FROM ProgramArgs IMPORT
    (* proc *)  ArgChan, IsArgPresent;

FROM Names IMPORT
    (* type *)  ServiceType, CardArray;

(********************************************************************************)

TYPE
    SocketArray = ARRAY ServiceType OF Socket;
    ServiceNameArray = ARRAY ServiceType OF ARRAY [0..3] OF CHAR;

CONST
    version = "0.90";
    ServiceName = ServiceNameArray {"SMTP", "POP"};

CONST
    DefaultPort = CardArray {25, 110};
    DefaultMaxUsers = CardArray {10, 10};
    DefaultTimeout = CardArray {900, 900};               (* seconds   *)

VAR
    MainSocket: SocketArray;
    ServerPort: CardArray;
    ServerEnabled: CARDINAL;
    ShutdownRequest: Semaphore;
    ShutdownInProgress, RapidShutdown: BOOLEAN;
    CalledFromInetd: BOOLEAN;
    ScreenEnabled: BOOLEAN;

(********************************************************************************)

PROCEDURE ShutdownChecker;

    (* A separate task that waits for a shutdown request.  *)

    VAR StillRunning: BOOLEAN;  j: ServiceType;

    BEGIN
        StillRunning := TRUE;
        LOOP
            Wait (ShutdownRequest);
            IF StillRunning THEN
                FOR j := MIN(ServiceType) TO MAX(ServiceType) DO
                    IF MainSocket[j] <> NotASocket THEN
                        so_cancel (MainSocket[j]);
                    END (*IF*);
                END (*FOR*);
                StillRunning := FALSE;
            END (*IF*);
            IF RapidShutdown THEN
                EXIT (*LOOP*);
            END (*IF*);
        END (*LOOP*);
    END ShutdownChecker;

(********************************************************************************)

PROCEDURE ["C"] ControlCHandler(): BOOLEAN;

    (* Intercepts a Ctrl/C from the keyboard. *)

    BEGIN
        RapidShutdown := ShutdownInProgress;
        Signal (ShutdownRequest);
        ShutdownInProgress := TRUE;
        RETURN TRUE;
    END ControlCHandler;

(********************************************************************************)

PROCEDURE LoadINIData;

    (* Loads setup parameters from "weasel.ini". *)

    VAR hini: OS2.HINI;

    PROCEDURE GetItem (name: ARRAY OF CHAR;
                            VAR (*OUT*) variable: ARRAY OF LOC): BOOLEAN;

        BEGIN
            RETURN INIGet (hini, "$SYS", name, variable);
        END GetItem;

    (********************************************************************)

    VAR TimeoutLimit, MaxUsers: CardArray;

    BEGIN
        hini := OpenINIFile ("weasel.ini");
        IF hini <> OS2.NULLHANDLE THEN
            IF NOT GetItem ("Enable", ServerEnabled) THEN
                ServerEnabled := 2;
            END (*IF*);
            IF NOT GetItem ("ServerPort", ServerPort) THEN
                ServerPort := DefaultPort;
            END (*IF*);
            IF NOT GetItem ("MaxUsers", MaxUsers) THEN
                MaxUsers := DefaultMaxUsers;
            END (*IF*);
            IF NOT GetItem ("TimeOut", TimeoutLimit) THEN
                TimeoutLimit := DefaultTimeout;
            END (*IF*);
            OS2.PrfCloseProfile (hini);
        END (*IF*);

        SetMaxUsers (MaxUsers);
        SetTimeout (TimeoutLimit);

    END LoadINIData;

(********************************************************************************)

PROCEDURE GetParameter (VAR (*OUT*) result: CARDINAL): BOOLEAN;

    (* Picks up an optional program argument from the command line.  If an      *)
    (* argument is present, returns TRUE.                                       *)

    TYPE CharSet = SET OF CHAR;
    CONST Digits = CharSet {'0'..'9'};

    VAR j: CARDINAL;
        args: ChanId;
        ParameterString: ARRAY [0..79] OF CHAR;

    BEGIN
        args := ArgChan();
        IF IsArgPresent() THEN
            TextIO.ReadString (args, ParameterString);
            j := 0;
            WHILE ParameterString[j] = ' ' DO
                INC (j);
            END (*WHILE*);
            result := 0;
            WHILE ParameterString[j] IN Digits DO
                result := 10*result + ORD(ParameterString[j]) - ORD('0');
                INC (j);
            END (*WHILE*);
            RETURN TRUE;
        ELSE
            RETURN FALSE;
        END (*IF*);
    END GetParameter;

(************************************************************************)

PROCEDURE RunTheServer;

    (*  OPERATING AS A SERVER                                                       *)
    (*     1. (Compulsory) Call "bind" to bind the socket with a local address.     *)
    (*        You can usually afford to specify INADDR_ANY as the machine           *)
    (*        address, but you'd normally bind to a specific port number.           *)
    (*     2. Call "listen" to indicate your willingness to accept connections.     *)
    (*     3. Call "accept", getting a new socket (say ns) from the client.         *)
    (*     4. Use procedures "send" and "recv" to transfer data, using socket ns.   *)
    (*        (Meanwhile, your original socket remains available to accept          *)
    (*        more connections, so you can continue with more "accept" operations   *)
    (*        in parallel with these data operations.  If so, you should of course  *)
    (*        be prepared to run multiple threads.)                                 *)
    (*     5. Use "soclose(ns)" to terminate the session with that particular       *)
    (*        client.                                                               *)
    (*     6. Use "soclose" on your original socket to clean up at the end.         *)

    VAR ns: Socket;  myaddr, client: SockAddr;
        temp: CARDINAL;
        SocketsToTest: SocketArray;
        j: ServiceType;
        Enabled: ARRAY ServiceType OF BOOLEAN;
        StartupSuccessful: BOOLEAN;

    BEGIN
        IF sock_init() <> 0 THEN
            IF ScreenEnabled THEN
                WriteString ("No network.");
            END (*IF*);
            RETURN;
        END (*IF*);

        Enabled[SMTP] := ODD(ServerEnabled);
        Enabled[POP] := ServerEnabled > 1;
        CalledFromInetd := GetParameter (ns);
        SetVersion (version);

        IF CalledFromInetd THEN

            IF ScreenEnabled THEN
                WriteString ("Weasel started from inetd, socket ");
                WriteCard (ns);  WriteLn;
            END (*IF*);
            AddToTransactionLog ("Server started.");
            temp := SIZE (myaddr);
            IF getsockname (ns, myaddr, temp) THEN
                IF ScreenEnabled THEN
                    WriteString ("Can't identify inetd session type");
                    WriteLn;
                END (*IF*);
                AddToTransactionLog ("Can't identify inetd session type");
            ELSE
                j := POP;
                IF myaddr.in_addr.port = DefaultPort[SMTP] THEN
                    j := SMTP;
                END (*IF*);
                NewSession (j, ns, client);
            END (*IF*);

        ELSE

            IF ScreenEnabled THEN
                WriteStringAt (0, 0, "Weasel v");
                WriteStringAt (0, 8, version);
                WriteStringAt (0, 20, "Copyright (C) 1998-99 Peter Moylan");

                EVAL (SetBreakHandler (ControlCHandler));
            END (*IF*);

            CreateTask (ShutdownChecker, 1, "ctrl/c hook");
            StartupSuccessful := FALSE;

            FOR j := MIN(ServiceType) TO MAX(ServiceType) DO
                MainSocket[j] := socket (AF_INET, SOCK_STREAM, AF_UNSPEC);

                (* Allow reuse of the port we're binding to. *)

                temp := 1;
                setsockopt (MainSocket[j], 0FFFFH, 4, temp, SIZE(CARDINAL));

                IF ScreenEnabled THEN
                    WriteString (ServiceName[j]);
                    IF Enabled[j] THEN
                        WriteString (" listening on port ");
                        WriteCard (ServerPort[j]);
                        WriteString (", socket ");
                        WriteCard (MainSocket[j]);
                    ELSE
                        WriteString (" disabled.");
                    END (*IF*);
                    WriteLn;
                END (*IF*);

                IF Enabled[j] THEN

                    (* Now have the socket, bind to our machine. *)

                    WITH myaddr DO
                        family := AF_INET;
                        WITH in_addr DO
                            port := Swap2 (ServerPort[j]);
                            (* Bind to all interfaces. *)
                            addr := INADDR_ANY;
                            zero := Zero8;
                        END (*WITH*);
                    END (*WITH*);

                    IF bind (MainSocket[j], myaddr, SIZE(myaddr)) THEN
                        IF ScreenEnabled THEN
                            WriteError;
                            WriteString ("Cannot bind to server port.");
                            WriteLn;
                        END (*IF*);

                    ELSE

                        (* Go into listening mode. *)

                        IF listen (MainSocket[j], 1) THEN
                            IF ScreenEnabled THEN
                                WriteError;
                            END (*IF*);
                        ELSE
                            StartupSuccessful := TRUE;
                        END (*IF*);

                    END (*IF bind*);

                END (*IF Enabled*);

            END (*FOR*);

            IF StartupSuccessful THEN

                AddToTransactionLog ("Server started.");

                (* Here's the main service loop. *)

                SocketsToTest := MainSocket;
                WHILE select (SocketsToTest, 2, 0, 0, MAX(CARDINAL)) > 0 DO
                    FOR j := MIN(ServiceType) TO MAX(ServiceType) DO
                        IF SocketsToTest[j] <> NotASocket THEN
                            temp := SIZE(client);
                            ns := accept (MainSocket[j], client, temp);
                            IF ns <> NotASocket THEN
                                NewSession (j, ns, client);
                            END (*IF*);
                        END (*IF*);
                    END (*FOR*);
                    SocketsToTest := MainSocket;
                END (*WHILE*);

                (* Close both sockets. *)

                FOR j := MIN(ServiceType) TO MAX(ServiceType) DO
                    IF soclose(MainSocket[j]) THEN
                        psock_errno ("");
                    END (*IF*);
                END (*FOR*);

            END (*IF*);

        END (*IF (not) CalledFromInetd *);

        (* End of operation, shut down the server. *)

        IF NOT RapidShutdown THEN
            IF CalledFromInetd THEN
                Sleep (3000);
            ELSIF NumberOfUsers() > 0 THEN
                AddToTransactionLog ("Waiting for existing users to finish");
            END (*IF*);
            WHILE (NumberOfUsers() > 0) AND NOT RapidShutdown DO
                Sleep (1000);
            END (*WHILE*);
        END (*IF*);

        AddToTransactionLog ("Weasel closing down");

    END RunTheServer;

(********************************************************************************)
(*                                 MAIN PROGRAM                                 *)
(********************************************************************************)

BEGIN
    ScreenEnabled := NotDetached();
    LoadINIData;
    ShutdownInProgress := FALSE;  RapidShutdown := FALSE;
    CreateSemaphore (ShutdownRequest, 0);
    RunTheServer;
END Weasel.

