MODULE SMTPd;

        (********************************************************)
        (*                                                      *)
        (*               Simple SMTP server                     *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            27 April 1998                   *)
        (*  Last edited:        27 April 1998                   *)
        (*  Status:             Just started                    *)
        (*                                                      *)
        (********************************************************)

IMPORT OS2;

FROM SYSTEM IMPORT LOC;

FROM IOChan IMPORT
    (* type *)  ChanId;

FROM Keyboard IMPORT
    (* proc *)  NotDetached, InKey, StuffKeyboardBuffer;

FROM Sockets IMPORT
    (* const*)  NotASocket,
    (* type *)  Socket, SockAddr, AddressFamily, SocketType,
    (* proc *)  sock_init, socket, so_cancel, setsockopt, gethostid,
                bind, listen, accept, soclose, psock_errno;

FROM Internet IMPORT
    (* const*)  Zero8, INADDR_ANY;

FROM Timer IMPORT
    (* proc *)  Sleep;

FROM TaskControl IMPORT
    (* proc *)  CreateTask;

FROM GlassTTY IMPORT
    (* proc *)  WriteString, WriteCard, WriteLn;

FROM InetUtilities IMPORT
    (* proc *)  ConvertDecimal, OpenINIFile, INIGet, IPToString,
                Swap2, Swap4, WaitForSocket, SetTransactionLoggingOption,
                AddToTransactionLog,
                WriteError;

FROM CtrlC IMPORT
    (* type *)  BreakHandler,
    (* proc *)  SetBreakHandler;

FROM SMTPSession IMPORT
    (* proc *)  SetVersion, SetTimeout, SetMaxUsers, SetTransactionLogging,
                NewSession, NumberOfUsers;

(*
FROM SYSTEM IMPORT CARD8, CARD16, CAST;

IMPORT TextIO;

FROM FtpTransfers IMPORT
    (* proc *)  SetLogLevel, SetFreeSpaceThreshold;

FROM Sockets IMPORT
    (* proc *)  socket, bind, sock_errno, psock_errno, soclose,
                accept, connect, listen, gethostid,
                sock_init, so_cancel, setsockopt;

FROM InetUtilities IMPORT
    (* proc *)  Swap2, Swap4, DumpRecord, WriteError,
                ConvertDecimal,
                AddToTransactionLog;

FROM GlassTTY IMPORT
    (* proc *)  WriteString, WriteLn, WriteChar, WriteInt, WriteCard,
                SetCursor;

FROM Keyboard IMPORT
    (* proc *)  NotDetached, InKey, StuffKeyboardBuffer;

FROM ProgramArgs IMPORT
    (* proc *)  ArgChan, IsArgPresent;

FROM Timer IMPORT
    (* proc *)  Sleep;

FROM TaskControl IMPORT
    (* proc *)  CreateTask;
*)

(********************************************************************************)

CONST
    version = "0.01";

CONST
    DefaultPort = 25;
    DefaultMaxUsers = 10;
    DefaultTimeout = 900;               (* seconds   *)
    DefaultLogLevel = 1;

VAR
    MainSocket: Socket;
    ServerPort: CARDINAL;
    RapidShutdown: BOOLEAN;
    CalledFromInetd: BOOLEAN;
    ScreenEnabled: BOOLEAN;

(********************************************************************************)

PROCEDURE ShutdownChecker;

    (* A separate task that looks for a Q command from the keyboard.  *)

    VAR ch: CHAR;  StillRunning: BOOLEAN;

    BEGIN
        StillRunning := TRUE;
        LOOP
            ch := InKey();
            IF CAP(ch) = 'Q' THEN
                RapidShutdown := TRUE;  ch := 'G';
            END (*IF*);
            IF CAP(ch) = 'G' THEN
                IF StillRunning THEN
                    so_cancel (MainSocket);
                    StillRunning := FALSE;
                END (*IF*);
                IF RapidShutdown THEN
                    EXIT (*LOOP*);
                END (*IF*);
            END(*IF*);
        END (*LOOP*);
    END ShutdownChecker;

(********************************************************************************)

PROCEDURE ["C"] ControlCHandler(): BOOLEAN;

    (* Intercepts a Ctrl/C from the keyboard. *)

    BEGIN
        RapidShutdown := TRUE;
        StuffKeyboardBuffer ('Q');
        RETURN TRUE;
    END ControlCHandler;

(********************************************************************************)

PROCEDURE GetParameters;

    (* Picks up program arguments from the command line. *)

    TYPE CharNumber = [0..79];

    VAR j: CARDINAL;
        args: ChanId;
        Options: ARRAY CharNumber OF CHAR;

    (****************************************************************************)

    PROCEDURE SkipBlanks;

        BEGIN
            LOOP
                IF Options[j] <> ' ' THEN EXIT(*LOOP*) END(*IF*);
                IF j = MAX(CharNumber) THEN
                    Options[j] := CHR(0);  EXIT (*LOOP*);
                ELSE
                    INC (j);
                END (*IF*);
            END (*LOOP*);
        END SkipBlanks;

    (****************************************************************************)

    PROCEDURE GetNumber(): CARDINAL;

        BEGIN
            SkipBlanks;
            RETURN ConvertDecimal (Options, j);
        END GetNumber;

    (****************************************************************************)

    VAR level: CARDINAL;

    BEGIN
(*
        args := ArgChan();
        IF IsArgPresent() THEN
            TextIO.ReadString (args, Options);
            j := 0;  SkipBlanks;
            LOOP
                CASE CAP(Options[j]) OF
                    CHR(0):   EXIT (*LOOP*);
                  | 'D':      INC (j);  level := GetNumber();
                                        IF NOT ScreenEnabled THEN
                                            level := level MOD 2;
                                        END (*IF*);
                                        SetTransactionLogging (level);
                  | 'F':      INC (j);  SetFreeSpaceThreshold (1024*GetNumber());
                  | 'G':      INC (j);  SetGuestLimit (GetNumber());
                  | 'L':      INC (j);  SetLogLevel (GetNumber());
                  | 'M':      INC (j);  SetMaxUsers (GetNumber());
                  | 'P':      INC (j);  ServerPort := GetNumber();
                  | 'T':      INC (j);  SetTimeout (GetNumber());
                  | '0'..'9': MainSocket := GetNumber();
                              CalledFromInetd := TRUE;
                ELSE
                    IF ScreenEnabled THEN
                        WriteString ("Unknown option ");
                        WriteChar (Options[j]);  WriteLn;
                    END (*IF*);
                    INC(j);
                END (*CASE*);
                SkipBlanks;
            END (*LOOP*);
        END (*IF*);
*)
    END GetParameters;

(************************************************************************)

PROCEDURE LoadINIData;

    (* Loads setup parameters from "weasel.ini". *)

    VAR hini: OS2.HINI;

    PROCEDURE GetItem (name: ARRAY OF CHAR;
                            VAR (*OUT*) variable: ARRAY OF LOC): BOOLEAN;

        BEGIN
            RETURN INIGet (hini, "$POP", name, variable);
        END GetItem;

    (********************************************************************)

    VAR MaxUsers, UserLogging, TimeoutLimit, TransLevel: CARDINAL;
        dummy: BOOLEAN;

    BEGIN
        ServerPort := DefaultPort;
        MaxUsers := DefaultMaxUsers;
        TimeoutLimit := DefaultTimeout;
        UserLogging := DefaultLogLevel;
        TransLevel := 2;

        hini := OpenINIFile ("weasel.ini");
        IF hini <> OS2.NULLHANDLE THEN
            dummy := GetItem ("ServerPort", ServerPort);
            dummy := GetItem ("MaxUsers", MaxUsers);
            dummy := GetItem ("TimeOut", TimeoutLimit);
            dummy := GetItem ("TransLevel", TransLevel);
            OS2.PrfCloseProfile (hini);
        END (*IF*);

        SetMaxUsers (MaxUsers);
        SetTimeout (TimeoutLimit);
        IF NOT ScreenEnabled THEN
            TransLevel := TransLevel MOD 2;
        END (*IF*);
        SetTransactionLogging (TransLevel);
    END LoadINIData;

(********************************************************************************)

PROCEDURE WriteHostID (ID: ARRAY OF LOC);

    VAR result: ARRAY [0..16] OF CHAR;

    BEGIN
        IPToString (ID, result);
        WriteString (result);
    END WriteHostID;

(********************************************************************************)

PROCEDURE RunTheServer;

    (*  OPERATING AS A SERVER                                                       *)
    (*     1. (Compulsory) Call "bind" to bind the socket with a local address.     *)
    (*        You can usually afford to specify INADDR_ANY as the machine           *)
    (*        address, but you'd normally bind to a specific port number.           *)
    (*     2. Call "listen" to indicate your willingness to accept connections.     *)
    (*     3. Call "accept", getting a new socket (say ns) from the client.         *)
    (*     4. Use procedures "send" and "recv" to transfer data, using socket ns.   *)
    (*        (Meanwhile, your original socket remains available to accept          *)
    (*        more connections, so you can continue with more "accept" operations   *)
    (*        in parallel with these data operations.  If so, you should of course  *)
    (*        be prepared to run multiple threads.)                                 *)
    (*     5. Use "soclose(ns)" to terminate the session with that particular       *)
    (*        client.                                                               *)
    (*     6. Use "soclose" on your original socket to clean up at the end.         *)

    VAR ns: Socket;  myaddr, client: SockAddr;
        temp: CARDINAL;
        OldBreakHandler: BreakHandler;

    BEGIN
        RapidShutdown := FALSE;
        IF sock_init() <> 0 THEN
            IF ScreenEnabled THEN
                WriteString ("No network.");
            END (*IF*);
            RETURN;
        END (*IF*);
        CalledFromInetd := FALSE;
        GetParameters;
        SetVersion (version);

        IF CalledFromInetd THEN

            IF ScreenEnabled THEN
                WriteString ("SMTPd started from inetd, socket ");
                WriteCard (MainSocket);  WriteLn;
            END (*IF*);
            AddToTransactionLog ("Server started.");
            NewSession (MainSocket, client);

        ELSE

            MainSocket := socket (AF_INET, SOCK_STREAM, AF_UNSPEC);

            (* Allow reuse of the port we're binding to. *)

            temp := 1;
            setsockopt (MainSocket, 0FFFFH, 4, temp, SIZE(CARDINAL));

            IF ScreenEnabled THEN
                WriteString ("SMTPd v");  WriteString (version);  WriteLn;
                WriteString ("Copyright (C) 1998 Peter Moylan.");
                WriteLn;  WriteHostID (Swap4(gethostid()));  WriteString (" port ");
                WriteCard (ServerPort);
                WriteString (", socket ");
                WriteCard (MainSocket);  WriteString ("     ");  WriteLn;
                OldBreakHandler := SetBreakHandler (ControlCHandler);
            END (*IF*);
            AddToTransactionLog ("Server started.");
            CreateTask (ShutdownChecker, 1, "smtp kbd");

            (* Now have the socket, bind to our machine. *)

            WITH myaddr DO
                family := AF_INET;
                WITH in_addr DO
                    port := Swap2 (ServerPort);
                    (* Bind to all interfaces. *)
                    addr := INADDR_ANY;
                    zero := Zero8;
                END (*WITH*);
            END (*WITH*);

            IF bind (MainSocket, myaddr, SIZE(myaddr)) THEN
                IF ScreenEnabled THEN
                    WriteError;
                    WriteString ("Cannot bind to server port, exiting.");
                    WriteLn;
                END (*IF*);

            ELSE

                (* Go into listening mode. *)

                IF listen (MainSocket, 1) THEN
                    IF ScreenEnabled THEN
                        WriteError;
                    END (*IF*);
                END (*IF*);

                WHILE WaitForSocket (MainSocket, MAX(CARDINAL)) > 0 DO
                    temp := SIZE(client);
                    ns := accept (MainSocket, client, temp);
                    IF ns <> NotASocket THEN
                        NewSession (ns, client);
                    END (*IF*);
                END (*WHILE*);

            END (*IF*);

            IF soclose(MainSocket) THEN
                psock_errno ("");
            END (*IF*);

        END (*IF*);

        (* End of operation, shut down the server. *)

        IF NOT RapidShutdown THEN
            IF CalledFromInetd THEN
                Sleep (3000);
            ELSIF NumberOfUsers() > 0 THEN
                AddToTransactionLog ("Waiting for existing users to finish");
            END (*IF*);
            WHILE (NumberOfUsers() > 0) AND NOT RapidShutdown DO
                Sleep (1000);
            END (*WHILE*);
        END (*IF*);

        AddToTransactionLog ("SMTPd closing down");

    END RunTheServer;

(********************************************************************************)
(*                                 MAIN PROGRAM                                 *)
(********************************************************************************)

BEGIN
    ServerPort := 25;
    ScreenEnabled := NotDetached();
    SetTransactionLoggingOption (2);
    LoadINIData;
    RunTheServer;
END SMTPd.

