IMPLEMENTATION MODULE RelayMail;

        (********************************************************)
        (*                                                      *)
        (* Part of the SMTP server - relays mail to other hosts *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            12 May 1998                     *)
        (*  Last edited:        8 October 1999                  *)
        (*  Status:             OK                              *)
        (*                                                      *)
        (*  Improvements to consider:                           *)
        (*    - send the rejection letter as a multipart mail   *)
        (*    - limit relay list size to 100 recipients         *)
        (*                                                      *)
        (********************************************************)

FROM SYSTEM IMPORT CAST, ADR, ADDRESS, LOC, CARD8, BOOL8;

IMPORT IOChan, IOConsts, ChanConsts, RndFile, TextIO, Strings, OS2, FileSys;

FROM Sockets IMPORT
    (* const*)  AF_INET, SOCK_STREAM, AF_UNSPEC, NotASocket,
    (* type *)  Socket, SockAddr,
    (* proc *)  socket, connect;

FROM Internet IMPORT
    (* const*)  Zero8;

FROM SBuffers IMPORT
    (* type *)  SBuffer,
    (* proc *)  CreateSBuffer, CloseSBuffer, SendLine, SendChar,
                SendString, SendEOL, PositiveResponse, GetLastLine;

FROM Hosts IMPORT
    (* proc *)  OurHostName, RecomputeOurHostName;

FROM Names IMPORT
    (* type *)  UserName, HostName, PathString, FilenameString;

FROM MXCheck IMPORT
    (* proc *)  DoMXLookup;

FROM TransLog IMPORT
    (* type *)  TransactionLogID,
    (* proc *)  CreateLogID, LogTransaction;

FROM FDFiles IMPORT
    (* type *)  DirectoryEntry,
    (* proc *)  OpenNewFile, CloseFile, FWriteString, FWriteLn,
                FWriteChar, FirstDirEntry, NextDirEntry, DirSearchDone;

FROM InetUtilities IMPORT
    (* proc *)  Swap2, OpenINIFile, INIGet, INIGetString, INIPut,
                IPToString, StartTransactionLogging;

FROM MyClock IMPORT
    (* proc *)  CurrentDateAndTime;

FROM SplitScreen IMPORT
    (* proc *)  NotDetached, ClearScreen, SetBoundary, WriteStringAt;

FROM TimeConv IMPORT
    (* proc *)  time;

FROM Semaphores IMPORT
    (* type *)  Semaphore,
    (* proc *)  CreateSemaphore, Wait, Signal;

FROM Timer IMPORT
    (* proc *)  TimedWait, Sleep;

FROM TaskControl IMPORT
    (* type *)  Lock,
    (* proc *)  CreateTask, CreateTask1, CreateLock, Obtain, Release;

FROM LowLevel IMPORT
    (* proc *)  EVAL;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

(************************************************************************)

CONST
    Nul = CHR(0);  CR = CHR(13);  LF = CHR(10);
    MaxRetries = 26;

TYPE Interval = ARRAY [0..MaxRetries-1] OF CARDINAL;

CONST
    (* Time between retries, in minutes. *)

    RetryInterval = Interval {5,   10,  30,  60,  60,  60, 150, 300,
                             300, 300, 300, 300, 300, 300, 300, 300,
                             300, 300, 300, 300, 300, 300, 300, 300,
                             300, 300};

TYPE
    (* A list of recipients for a single mail item. *)

    RelayListPointer = POINTER TO RelayListEntry;
    RelayList = POINTER TO
                       RECORD
                           count: CARDINAL;
                           head: RelayListPointer;
                       END (*RECORD*);

    OldRelayList = RelayListPointer;
    RelayListEntry = RECORD
                         next: RelayListPointer;
                         forwardpath: PathString;
                         failuremessage: ARRAY [0..79] OF CHAR;
                     END (*RECORD*);

    (* A list of mail items waiting to be sent.                         *)
    (*   next             next item on the list                         *)
    (*   sendtime         time that the item is due to be sent          *)
    (*   RetryNumber      number of times this has been tried already   *)
    (*   ID               ID for transaction log                        *)
    (*   sender           the MAIL FROM parameter                       *)
    (*   file             name of the message file                      *)
    (*   offset           starting position within the file for the     *)
    (*                     data to be sent to the recipient.  (There    *)
    (*                     could be bookkeeping data before that,       *)
    (*                     which is not to be sent.)                    *)
    (*   sendto           list of recipients.                           *)
    (*   NotifyOnFailure  TRUE if a failure to send this item should    *)
    (*                     cause a notification to be mailed back       *)
    (*                     to the sender.                               *)

    OutJobPtr = POINTER TO OutJob;
    OutJob = RECORD
                 next: OutJobPtr;
                 sendtime: CARDINAL;
                 RetryNumber: CARDINAL;
                 ID: TransactionLogID;
                 sender: PathString;
                 file: FilenameString;
                 offset: RndFile.FilePos;
                 sendto: RelayList;
                 NotifyOnFailure: BOOLEAN;
             END (*RECORD*);

    (* While relay mail is waiting to be sent it is stored as a file    *)
    (* which starts with the following details.                         *)
    (*   4 bytes   format version, value 'V000'                         *)
    (*   4 bytes   send time                                            *)
    (*   1 byte    retry number                                         *)
    (*   1 byte    Boolean notify-on-failure flag                       *)
    (*   variable  sender (character string)                            *)
    (*   variable  recipient list, bounded by () and comma-separated    *)
    (*                                                                  *)
    (* The message content starts immediately after this.               *)

    PreambleType = RECORD
                       version: ARRAY [0..3] OF CHAR;
                       sendtime: CARDINAL;
                       RetryNumber: CARD8;
                       NotifyOnFailure: BOOL8;
                   END (*RECORD*);

(************************************************************************)

VAR
    (* A flag saying whether we may write to the screen. *)

    ScreenEnabled: BOOLEAN;

    (* A flag to say that we're offline and can't send any mail out. *)

    WeAreOffline: BOOLEAN;

    (* A count of items that are held up because we're offline. *)

    OfflineData: RECORD
                     access: Lock;
                     PendingCount: CARDINAL;
                 END (*RECORD*);

    (* A semaphore to say that we should check again to see whether     *)
    (* we are now online.                                               *)

    CheckIfOnline: Semaphore;

    (* A semaphore to tell the mailer task that it should check the     *)
    (* list of mail waiting to be sent.                                 *)

    SomethingToSend: Semaphore;

    (* A semaphore to wake up the "retry" task. *)

    Retry: Semaphore;

    (* List of mail waiting to be sent (without delay).   *)

    OutboundMail: RECORD
                      access: Lock;
                      head, tail: OutJobPtr;
                  END (*RECORD*);

    (* List of jobs waiting to be reattempted (with a delay).   *)

    RetryList: RECORD
                   access: Lock;
                   head: OutJobPtr;
               END (*RECORD*);

    (* Name of the directory used for mail to be forwarded. *)

    ForwardDirName: FilenameString;

    (* String used in creating a unique file name. *)

    NextName: ARRAY [0..7] OF CHAR;
    NextNameLock: Lock;

    (* Name of the local host. *)

    LocalHost: HostName;

    (* Relay host for our outbound mail. *)

    ForwardRelayHost: HostName;

    (* Number of extra threads that are now running. *)

    TaskCount: CARDINAL;

    (* A flag and semaphore used in shutdown processing. *)

    ShutdownRequest: BOOLEAN;
    TaskDone: Semaphore;

(************************************************************************)
(*                    CREATING A UNIQUE FILENAME                        *)
(************************************************************************)

PROCEDURE MakeUniqueName (VAR (*OUT*) name: ARRAY OF CHAR);

    (* Generates a unique 8-character string.  The argument must of     *)
    (* course be big enough to take at least 8 characters.              *)

    (********************************************************************)

    PROCEDURE Increment (N: CARDINAL);

        (* Increments NextName[N], with carry as appropriate. *)

        BEGIN
            IF NextName[N] = '9' THEN
                NextName[N] := 'A';
            ELSIF NextName[N] = 'Z' THEN
                NextName[N] := '0';
                IF N > 0 THEN
                    Increment (N-1);
                END (*IF*);
            ELSE
                INC (NextName[N]);
            END (*IF*);
        END Increment;

    (********************************************************************)

    BEGIN
        Obtain (NextNameLock);
        Strings.Assign (NextName, name);
        Increment (7);
        Release (NextNameLock);
    END MakeUniqueName;

(************************************************************************)

PROCEDURE MakeNewFilename (BaseName, tail: ARRAY OF CHAR;
                       VAR (*OUT*) NewName: ARRAY OF CHAR);

    (* Creates a file name of the form BaseNamexxxtail, where xxx is    *)
    (* chosen such that a file of that name does not already exist.     *)
    (* Note that BaseName and tail can include punctuation.             *)

    VAR UName: FilenameString;

    BEGIN
        REPEAT
            MakeUniqueName (UName);
            Strings.Assign (BaseName, NewName);
            Strings.Append (UName, NewName);
            Strings.Append (tail, NewName);
        UNTIL NOT FileSys.Exists(NewName);
    END MakeNewFilename;

(************************************************************************)
(*              ADDING A JOB TO THE QUEUE OF OUTBOUND MAIL              *)
(************************************************************************)

PROCEDURE AppendBody (srcname: ARRAY OF CHAR;  offset: RndFile.FilePos;
                      dstcid: IOChan.ChanId);

    (* Copies from a source file, starting at position "offset" and     *)
    (* continuing to the end of the file, to the destination.  The      *)
    (* caller should have already opened the destination file, but      *)
    (* not the source file.                                             *)

    VAR srccid: IOChan.ChanId;  NumberRead: CARDINAL;
        status: ChanConsts.OpenResults;
        buffer: ARRAY [0..2047] OF CHAR;

    BEGIN
        (* Open the source file and position it at the right place. *)

        RndFile.OpenOld (srccid, srcname,
                ChanConsts.read+ChanConsts.raw, status);
        RndFile.SetPos (srccid, offset);

        (* Copy across the message body. *)

        LOOP
            IOChan.RawRead (srccid, ADR(buffer), SIZE(buffer), NumberRead);
            IF NumberRead = 0 THEN EXIT(*LOOP*) END(*IF*);
            IOChan.RawWrite (dstcid, ADR(buffer), NumberRead);
        END (*LOOP*);

        RndFile.Close (srccid);

    END AppendBody;

(************************************************************************)

PROCEDURE AddToOutQueue (p: OutJobPtr);

    (* Adds a new job to the queue OutboundMail. *)

    BEGIN
        Signal (CheckIfOnline);
        p^.next := NIL;
        Obtain (OutboundMail.access);
        IF OutboundMail.head = NIL THEN
            OutboundMail.head := p;
        ELSE
            OutboundMail.tail^.next := p;
        END (*IF*);
        OutboundMail.tail := p;
        Release (OutboundMail.access);
        Signal (SomethingToSend);
    END AddToOutQueue;

(************************************************************************)

PROCEDURE AddToRetryList (p: OutJobPtr);

    (* Adds a new job to the list of jobs that have to be retried.      *)
    (* This list differs from the OutboundMail queue in that its        *)
    (* entries are ordered by time.                                     *)

    VAR previous, current: OutJobPtr;

    BEGIN
        p^.next := NIL;
        Obtain (RetryList.access);
        previous := NIL;  current := RetryList.head;
        WHILE (current <> NIL) AND (p^.sendtime > current^.sendtime) DO
            previous := current;  current := current^.next;
        END (*WHILE*);

        (* The new item has to be inserted between previous and current. *)

        p^.next := current;
        IF previous = NIL THEN
            RetryList.head := p;
        ELSE
            previous^.next := p;
        END (*IF*);

        Release (RetryList.access);
        Signal (Retry);

    END AddToRetryList;

(************************************************************************)
(*            STORING A MESSAGE FILE IN THE FORWARD DIRECTORY           *)
(************************************************************************)

PROCEDURE WriteNameList (fid: IOChan.ChanId;  list: RelayList);

    (* Writes a comma-separated list of names, enclosed in parentheses. *)

    VAR p: RelayListPointer;

    BEGIN
        p := list^.head;
        FWriteChar (fid, '(');
        WHILE p <> NIL DO
            IOChan.RawWrite (fid, ADR(p^.forwardpath),
                                        LENGTH(p^.forwardpath));
            p := p^.next;
            IF p <> NIL THEN
                FWriteChar (fid, ',');
            END (*IF*);
        END (*WHILE*);
        FWriteChar (fid, ')');
    END WriteNameList;

(************************************************************************)

PROCEDURE StartNewFile (p: OutJobPtr): IOChan.ChanId;

    (* Creates a new file, opens it, and writes the preamble details    *)
    (* for job p^.  On return p^.offset has been set, the file is still *)
    (* open, and the function result is the file handle.                *)

    VAR cid: IOChan.ChanId;
        preamble: PreambleType;

    BEGIN
        MakeNewFilename (ForwardDirName, ".FWD", p^.file);
        cid := OpenNewFile (p^.file);

        (* Write the preamble to the new file. *)

        WITH preamble DO
            version := "V000";
            sendtime := p^.sendtime;
            RetryNumber := p^.RetryNumber;
            NotifyOnFailure := p^.NotifyOnFailure;
        END (*WITH*);
        IOChan.RawWrite (cid, ADR(preamble), SIZE(preamble));
        IOChan.RawWrite (cid, ADR(p^.sender), LENGTH(p^.sender));
        WriteNameList (cid, p^.sendto);

        (* Set the offset. *)

        p^.offset := RndFile.CurrentPos(cid);

        RETURN cid;

    END StartNewFile;

(************************************************************************)

PROCEDURE StoreMessageFile (p: OutJobPtr);

    (* On entry p^ should have all fields filled in.  This procedure    *)
    (* creates a new file (whose name is returned in p^.file) with a    *)
    (* preamble reflecting the details in p^ and with a body being the  *)
    (* body of the original p^.file.  The preamble details are given in *)
    (* the comments near the beginning of this module.                  *)
    (* Important assumption: p^.sendto is a non-empty list.             *)

    VAR oldfilename: FilenameString;
        oldoffset: RndFile.FilePos;
        newfid: IOChan.ChanId;

    BEGIN
        oldfilename := p^.file;
        oldoffset := p^.offset;
        newfid := StartNewFile(p);
        AppendBody (oldfilename, oldoffset, newfid);
        RndFile.Close (newfid);
    END StoreMessageFile;

(************************************************************************)
(*                     THE MAIN CLIENT PROCEDURES                       *)
(************************************************************************)

PROCEDURE DeleteRelayList (VAR (*INOUT*) RL: RelayList);

    (* Discards a list of relay mail recipients, and sets RL to NIL. *)

    VAR this, next: RelayListPointer;

    BEGIN
        IF RL <> NIL THEN
            this := RL^.head;
            WHILE this <> NIL DO
                next := this^.next;
                DISPOSE (this);
                this := next;
            END (*WHILE*);
            DISPOSE (RL);
        END (*IF*);
    END DeleteRelayList;

(************************************************************************)

PROCEDURE Empty (RL: RelayList): BOOLEAN;

    (* Tests whether the list has any entries. *)

    BEGIN
        RETURN (RL = NIL) OR (RL^.count = 0);
    END Empty;

(************************************************************************)

PROCEDURE AddToRelayList (VAR (*INOUT*) RL: RelayList;
                             VAR (*IN*) to: ARRAY OF CHAR);

    (* Adds one non-local recipient to the list of recipients. *)

    VAR RLP: RelayListPointer;

    BEGIN
        IF RL = NIL THEN
            NEW (RL);
            WITH RL^ DO
                count := 0;  head := NIL;
            END (*WITH*);
        END (*IF*);
        NEW (RLP);
        WITH RLP^ DO
            next := RL^.head;
            Strings.Assign (to, forwardpath);
            failuremessage := "";
        END (*WITH*);
        RL^.head := RLP;
        INC (RL^.count);
    END AddToRelayList;

(************************************************************************)

PROCEDURE WriteRelayList (cid: IOChan.ChanId;  RL: RelayList);

    (* Writes a list of e-mail addresses to a file. *)

    VAR p: RelayListPointer;

    BEGIN
        IF RL <> NIL THEN
            p := RL^.head;
            WHILE p <> NIL DO
                FWriteString (cid, p^.forwardpath);
                FWriteString (cid, " ");
                p := p^.next;
            END (*WHILE*);
        END (*IF*);
    END WriteRelayList;

(************************************************************************)

PROCEDURE SendRelayMail (VAR (*IN*) filename, from: ARRAY OF CHAR;
                         VAR (*INOUT*) RL: RelayList;
                         offset: RndFile.FilePos;  ID: TransactionLogID);

    (* Takes a copy of the file, and queues it to be sent to the        *)
    (* recipients on RL; then deletes RL.  We do not delete the         *)
    (* original file, but we will ultimately delete the copy we have    *)
    (* taken.                                                           *)
    (* (Actually we don't immediately delete RL; we take over the list  *)
    (* internally, but the caller sees RL=NIL.  The list data are kept  *)
    (* as long as needed, but are ultimately discarded.)                *)

    VAR p: OutJobPtr;

    BEGIN
        IF RL <> NIL THEN
            NEW (p);
            Strings.Assign (filename, p^.file);
            Strings.Assign (from, p^.sender);
            p^.sendto := RL;
            p^.ID := ID;
            p^.NotifyOnFailure := TRUE;
            p^.offset := offset;
            p^.sendtime := time();
            p^.RetryNumber := 0;
            StoreMessageFile (p);
            LogTransaction (ID, p^.file);
            AddToOutQueue (p);
            RL := NIL;
        END (*IF*);
    END SendRelayMail;

(********************************************************************************)
(*                          PARSING A PATH STRING                               *)
(********************************************************************************)

PROCEDURE UserAndDomain (source: ARRAY OF CHAR;
                            VAR (*OUT*) user: UserName;
                            VAR (*OUT*) domain: HostName);

    (* Extracts user and domain from a path string. *)

    TYPE CharSet = SET OF CHAR;

    VAR srcpos: CARDINAL;  Stoppers: CharSet;

    (****************************************************************************)

    PROCEDURE CopyString (VAR (*OUT*) dest: ARRAY OF CHAR);

        (* Copies up to (but not including) a character in Stoppers. *)

        VAR dstpos: CARDINAL;

        BEGIN
            dstpos := 0;
            WHILE (srcpos <= HIGH(source)) AND NOT (source[srcpos] IN Stoppers) DO
                IF dstpos <= HIGH(dest) THEN
                    dest[dstpos] := source[srcpos];
                    INC (dstpos);
                END (*IF*);
                INC (srcpos);
            END (*WHILE*);
            IF dstpos <= HIGH(dest) THEN
                dest[dstpos] := Nul;
            END (*IF*);
        END CopyString;

    (****************************************************************************)

    BEGIN
        srcpos := 0;
        WHILE (srcpos <= HIGH(source)) AND (source[srcpos] = ' ') DO
            INC (srcpos);
        END (*WHILE*);
        IF (srcpos <= HIGH(source)) AND (source[srcpos] = '"') THEN
            INC (srcpos);
            REPEAT
                INC (srcpos);
            UNTIL (srcpos > HIGH(source)) OR (source[srcpos-1] = '"');
        END (*IF*);
        WHILE (srcpos <= HIGH(source)) AND (source[srcpos] = ' ') DO
            INC (srcpos);
        END (*WHILE*);
        IF (srcpos <= HIGH(source)) AND (source[srcpos] = '<') THEN
            INC (srcpos);
        END (*IF*);
        Stoppers := CharSet {Nul, ' ', '@', '>'};
        CopyString (user);
        IF (srcpos <= HIGH(source)) AND (source[srcpos] = '@') THEN
            INC (srcpos);
            EXCL (Stoppers, '@');
            CopyString (domain);
        ELSE
            domain := "";
        END (*IF*);
    END UserAndDomain;

(************************************************************************)
(*              SENDING A "SORRY, IT DIDN'T WORK" MESSAGE               *)
(************************************************************************)

PROCEDURE SendRejectionLetter (job: OutJobPtr;  ID: TransactionLogID;
                                                        final: BOOLEAN);

    (* Sends an e-mail to the sender to say that the message described  *)
    (* by "job" could not be delivered.  If "final" is TRUE, we tell    *)
    (* the sender that there will be no further attempts.               *)

    (* The job descriptor and message file are not altered by this      *)
    (* procedure - we create our own copies of what we need.            *)

    VAR p: OutJobPtr;
        cid: IOChan.ChanId;  RLP: RelayListPointer;
        TimeBuffer: ARRAY [0..31] OF CHAR;

    BEGIN
        (* Create a new job descriptor. *)

        RLP := job^.sendto^.head;
        NEW (p);
        NEW (p^.sendto);
        WITH p^.sendto^ DO
            NEW (head);
            WITH head^ DO
                next := NIL;
                forwardpath := job^.sender;
            END (*WITH*);
            count := 1;
        END (*WITH*);
        p^.NotifyOnFailure := FALSE;  p^.sender := "<>";
        p^.ID := ID;
        p^.sendtime := time();
        p^.RetryNumber := 0;

        (* Create a new message file. *)

        cid := StartNewFile (p);

        (* Fill in the header details. *)

        CurrentDateAndTime (TimeBuffer);
        FWriteString (cid, "Date: ");
        FWriteString (cid, TimeBuffer);
        FWriteLn (cid);
        FWriteString (cid, "From: Mailer daemon <postmaster@");
        FWriteString (cid, LocalHost);
        FWriteString (cid, ">");
        FWriteLn (cid);
        FWriteString (cid, "To: ");
        FWriteString (cid, p^.sendto^.head^.forwardpath);
        FWriteLn (cid);
        FWriteString (cid, "Subject: Delivery Notification: Delivery has ");
        IF final THEN
            FWriteString (cid, "failed");
        ELSE
            FWriteString (cid, "been delayed");
        END (*IF*);
        FWriteLn (cid);
        FWriteLn (cid);

        (* Fill in the message body. *)

        FWriteString (cid, "Your mail was not delivered");
        FWriteString (cid, " to the following recipients.");
        FWriteLn (cid);
        REPEAT
            FWriteString (cid, RLP^.forwardpath);
            FWriteString (cid, ": ");
            FWriteString (cid, RLP^.failuremessage);
            FWriteLn (cid);
            RLP := RLP^.next;
        UNTIL RLP = NIL;
        FWriteLn (cid);
        IF final THEN
            FWriteString (cid, "There will be no further attempts to deliver this mail.");
        ELSE
            FWriteString (cid, "You do not need to re-send this mail.  The mail system");
            FWriteLn (cid);
            FWriteString (cid, "will keep trying to deliver it for several days.");
        END (*IF*);
        FWriteLn (cid);  FWriteLn (cid);

        (* Append a copy of the message that couldn't be delivered. *)

        IF job^.file[0] <> Nul THEN
            AppendBody (job^.file, job^.offset, cid);
        END (*IF*);
        CloseFile (cid);

        (* Put our message on the output queue.  *)

        AddToOutQueue (p);

    END SendRejectionLetter;

(************************************************************************)
(*            THE PROCEDURES THAT DELIVER THE OUTGOING MAIL             *)
(************************************************************************)

PROCEDURE ConnectToHost (IPaddress: CARDINAL;
                      VAR (*INOUT*) FailureReason: ARRAY OF CHAR): Socket;

    (* Tries to open a connection to the specified host.  Returns the   *)
    (* value NotASocket if we don't succeed; in this case, the string   *)
    (* FailureReason might be updated.                                  *)
    (* IPaddress is in network byte order.                              *)

    VAR s: Socket;  peer: SockAddr;

    BEGIN
        IF IPaddress <> 0 THEN
            s := socket (AF_INET, SOCK_STREAM, AF_UNSPEC);

            IF s = NotASocket THEN
                Strings.Assign ("Can't allocate socket", FailureReason);
            ELSE

                (* Socket open, connect to the client. *)

                WITH peer DO
                    family := AF_INET;
                    WITH in_addr DO
                        port := Swap2(25);
                        addr := IPaddress;
                        zero := Zero8;
                    END (*WITH*);
                END (*WITH*);

                IF connect (s, peer, SIZE(peer)) THEN

                    Strings.Assign ("Failed to connect", FailureReason);
                    s := NotASocket;

                END (*IF*);
            END (*IF*);

        ELSE

            Strings.Assign ("500 Unknown host", FailureReason);
            s := NotASocket;

        END (*IF*);

        RETURN s;

    END ConnectToHost;

(************************************************************************)

PROCEDURE SendCommand (SB: SBuffer;  command: ARRAY OF CHAR;
                         VAR (*OUT*) ConnectionLost: BOOLEAN): BOOLEAN;

    (* Sends a command, returns TRUE if the command was sent OK and     *)
    (* a positive response was returned.                                *)

    BEGIN
        ConnectionLost := NOT SendLine (SB, command);
        RETURN (NOT ConnectionLost) AND PositiveResponse(SB, ConnectionLost);
    END SendCommand;

(************************************************************************)

PROCEDURE SendFile (SB: SBuffer;  name: FilenameString;  offset: RndFile.FilePos;
                         VAR (*OUT*) ConnectionLost: BOOLEAN): BOOLEAN;

    (* Sends the file, returns TRUE if it was successfully transmitted  *)
    (* and a positive response was returned.                            *)

    VAR success, MoreToGo, AtEOL: BOOLEAN;
        cid: IOChan.ChanId;
        buffer: ARRAY [0..2047] OF CHAR;
        result: IOConsts.ReadResults;
        res: ChanConsts.OpenResults;

    BEGIN
        AtEOL := TRUE;
        RndFile.OpenOld (cid, name,
                        RndFile.read + RndFile.text + RndFile.old, res);
        success := res = ChanConsts.opened;
        IF success THEN
            RndFile.SetPos (cid, offset);
        END (*IF*);
        MoreToGo := TRUE;
        WHILE success AND MoreToGo DO
            TextIO.ReadString (cid, buffer);

            (* Result is set to the value allRight, endOfLine, or endOfInput. *)

            result := IOChan.ReadResult (cid);
            IF result = IOConsts.endOfInput THEN

                MoreToGo := FALSE;

            ELSIF result = IOConsts.endOfLine THEN

                MoreToGo := TRUE;
                success := SendEOL (SB);
                TextIO.SkipLine (cid);
                AtEOL := TRUE;

            ELSE

                IF AtEOL AND (buffer[0] = '.') THEN

                    (* Special case: byte-stuffing to handle the case   *)
                    (* of a line that starts with a period.             *)

                    success := SendChar (SB, '.');
                END (*IF*);
                success := success AND SendString (SB, buffer);
                AtEOL := FALSE;

            END (*IF*);

        END (*WHILE*);
        RndFile.Close (cid);
        IF NOT AtEOL THEN
            success := success AND SendEOL(SB);
        END (*IF*);

        success := success AND SendChar (SB, '.') AND SendEOL (SB);
        ConnectionLost := NOT success;
        RETURN success AND PositiveResponse (SB, ConnectionLost);

    END SendFile;

(************************************************************************)
(*                       DELIVERING ONE ITEM                            *)
(************************************************************************)

PROCEDURE DeliverDeLetter (p: RelayListPointer;
                           VAR (*IN*) from: PathString;
                           VAR (*IN*) file: FilenameString;
                           offset: RndFile.FilePos;
                           IPaddress: CARDINAL): BOOLEAN;

    (* Sends one mail item to a single recipient via the server at the  *)
    (* specified IP address.  The failuremessage field of p^ is         *)
    (* updated.  The file is not deleted.                               *)
    (* Precondition: p <> NIL.                                          *)

    VAR success, ConnectionLost: BOOLEAN;
        s: Socket;
        Buffer: ARRAY [0..511] OF CHAR;
        SB: SBuffer;

    BEGIN

        (* Connect to a host and send the HELO command. *)

        s := ConnectToHost (IPaddress, p^.failuremessage);
        SB := CreateSBuffer (s);
        success := (s <> NotASocket) AND PositiveResponse(SB, ConnectionLost);
        IF success THEN
            Buffer := "HELO ";
            Strings.Append (LocalHost, Buffer);
            success := SendCommand (SB, Buffer, ConnectionLost);
        END (*IF*);

        IF success THEN

            (* We are now logged in. *)

            Buffer := "MAIL FROM: ";
            Strings.Append (from, Buffer);
            success := SendCommand (SB, Buffer, ConnectionLost);
            IF success THEN
                Buffer := "RCPT TO: ";
                Strings.Append (p^.forwardpath, Buffer);
                success := SendCommand (SB, Buffer, ConnectionLost)
                             AND SendCommand (SB, "DATA", ConnectionLost)
                             AND SendFile (SB, file, offset, ConnectionLost);
            END (*IF*);

            IF success THEN
                p^.failuremessage := "No error";
            ELSIF ConnectionLost THEN
                p^.failuremessage := "Connection lost";
            ELSE
                GetLastLine (SB, p^.failuremessage);
            END (*IF*);

            (* We should try to log out even if the above failed. *)

            EVAL (SendCommand (SB, "QUIT", ConnectionLost));

        END (*IF*);

        CloseSBuffer (SB);
        RETURN success;

    END DeliverDeLetter;

(************************************************************************)

PROCEDURE SendToOneRecipient (p: RelayListPointer;
                           VAR (*IN*) from: PathString;
                           VAR (*IN*) file: FilenameString;
                           offset: RndFile.FilePos): BOOLEAN;

    (* Sends one mail item to a single recipient.  The failuremessage   *)
    (* field of p^ is updated.  The file is not deleted.                *)
    (* Precondition: p <> NIL.                                          *)

    CONST Max = 15;

    VAR user: UserName;  domain: HostName;
        success: BOOLEAN;  j: CARDINAL;
        address: ARRAY [0..Max] OF CARDINAL;

    BEGIN
        IF ForwardRelayHost[0] = Nul THEN
            UserAndDomain (p^.forwardpath, user, domain);
        ELSE
            domain := ForwardRelayHost;
        END (*IF*);
        success := FALSE;
        CASE DoMXLookup (domain, address) OF
          |  0: j := 0;
                REPEAT
                    success := DeliverDeLetter (p, from, file, offset, address[j]);
                    INC (j);
                UNTIL success OR (j > Max) OR (address[j] = 0);
          |  1: Strings.Assign ("500 unknown host or domain", p^.failuremessage);
          |  2: Strings.Assign ("400 domain name lookup failure", p^.failuremessage);
          |  3: Strings.Assign ("400 nameserver not responding", p^.failuremessage);
        END (*CASE*);
        IF NOT success THEN
            Signal(CheckIfOnline);
        END (*IF*);
        RETURN success;
    END SendToOneRecipient;

(************************************************************************)

PROCEDURE MailOneMessage (VAR (*INOUT*) p: OutJobPtr);

    (* Sends one mail item, possibly to multiple recipients.  On        *)
    (* return, p^.sendto lists the recipients for whom the operation    *)
    (* failed and for whom no retry should be attempted.  (Special case:*)
    (* after a shutdown request, p^.sendto might also contain intended  *)
    (* recipients that we didn't get time to deal with.)  Retries are   *)
    (* handled inside this procedure by generating a new job descriptor.*)

    (* The message file is not deleted, regardless of success or        *)
    (* failure; we leave that decision up to the caller.                *)
    (* Precondition: p <> NIL.                                          *)

    VAR ToRetry: RelayList;
        previous, current: RelayListPointer;
        q: OutJobPtr;  success, FileExists: BOOLEAN;
        LogMessage: ARRAY [0..511] OF CHAR;

    BEGIN
        FileExists := FileSys.Exists (p^.file);
        IF NOT FileExists THEN
            p^.file := "";
        END (*IF*);
        NEW (ToRetry);
        WITH ToRetry^ DO
            count := 0;  head := NIL;
        END (*WITH*);
        previous := NIL;  current := p^.sendto^.head;
        WHILE (current <> NIL) AND NOT ShutdownRequest DO

            IF FileExists THEN
                success := SendToOneRecipient (current, p^.sender,
                                               p^.file, p^.offset);
            ELSE
                success := FALSE;
                current^.failuremessage := "500 Message file has been lost";
            END (*IF*);
            Strings.Assign ("Mail to ", LogMessage);
            Strings.Append (current^.forwardpath, LogMessage);

            IF success OR ((p^.RetryNumber < MaxRetries)
                           AND (current^.failuremessage[0] <> '5')) THEN

                (* Remove current from the list. *)

                IF previous = NIL THEN
                    p^.sendto^.head := current^.next;
                ELSE
                    previous^.next := current^.next;
                END (*IF*);
                DEC (p^.sendto^.count);

                IF success THEN

                    (* Successful delivery. *)

                    Strings.Append (" - sent", LogMessage);
                    DISPOSE (current);

                ELSE

                    (* Possibly recoverable delivery failure, move *)
                    (* current to the ToRetry list.                *)

                    Strings.Append (" - deferred", LogMessage);
                    current^.next := ToRetry^.head;  ToRetry^.head := current;
                    INC (ToRetry^.count);

                END (*IF*);

            ELSE

                (* Nonrecoverable delivery failure, *)
                (* leave current on the list.       *)

                Strings.Append (" - failed", LogMessage);
                previous := current;

            END (*IF*);
            LogTransaction (p^.ID, LogMessage);
            IF NOT success THEN
                LogTransaction (p^.ID, current^.failuremessage);
            END (*IF*);

            (* Move to next recipient. *)

            IF previous = NIL THEN
                current := p^.sendto^.head;
            ELSE
                current := previous^.next;
            END (*IF*);

        END (*WHILE*);

        (* At this stage the intended recipients fall into three groups: *)
        (*   - failed deliveries that are worth a reattempt are on       *)
        (*     the ToRetry list;                                         *)
        (*   - other failed deliveries are still on p^.sendto;           *)
        (*   - successful deliveries are on neither list.                *)

        IF p^.sendto^.count = 0 THEN
            DISPOSE (p^.sendto);
        END (*IF*);

        IF ToRetry^.count = 0 THEN
            DISPOSE (ToRetry);
        ELSE

            (* Make a new copy of the job details, and a copy of the    *)
            (* message file, because we want to preserve the state of   *)
            (* p for the caller.                                        *)

            NEW (q);  q^ := p^;  q^.sendto := ToRetry;
            INC (q^.sendtime, 60*RetryInterval[q^.RetryNumber]);
            INC (q^.RetryNumber);
            StoreMessageFile (q);
            IF q^.NotifyOnFailure AND (q^.RetryNumber = 2) THEN
                SendRejectionLetter (q, q^.ID, FALSE);
            END (*IF*);
            AddToRetryList (q);

        END (*IF*);

    END MailOneMessage;

(************************************************************************)
(*                 THE TASK THAT HANDLES OUTGOING MAIL                  *)
(************************************************************************)

PROCEDURE MailerTask (tasknum: ADDRESS);

    (* Takes the mail on the OutQueue, and sends it.  We could be       *)
    (* running multiple copies of this task.                            *)

    VAR p: OutJobPtr;  LogID: TransactionLogID;
        LogFilePrefix: ARRAY [0..6] OF CHAR;
        filename: FilenameString;

    BEGIN
        LogFilePrefix := "Daemon ";
        LogFilePrefix[6] := CHR(ORD('0') + CAST(CARDINAL,tasknum) MOD 10);
        LogID := CreateLogID (LogFilePrefix);
        LOOP
            Wait (SomethingToSend);
            IF ShutdownRequest THEN
                EXIT (*LOOP*);
            END (*IF*);

            IF WeAreOffline THEN

                WITH OfflineData DO
                    Obtain(access);
                    INC (PendingCount);
                    Release(access);
                    p := NIL;
                END (*WITH*);

            ELSE
                (* Take the first element of OutboundMail, if any. *)

                Obtain (OutboundMail.access);
                p := OutboundMail.head;
                IF p <> NIL THEN
                    OutboundMail.head := p^.next;
                    p^.next := NIL;
                    IF OutboundMail.head = NIL THEN
                        OutboundMail.tail := NIL;
                    END (*IF*);
                END (*IF*);
                Release (OutboundMail.access);
            END (*IF*);

            (* This next section of code is skipped if there is nothing *)
            (* to send.                                                 *)

            IF p <> NIL THEN
                p^.ID := LogID;
                filename := p^.file;
                MailOneMessage (p);

                (* On return from MailOneMessage, p^.sendto lists only  *)
                (* the failures.  If the failure was because of a       *)
                (* shutdown request then we should preserve the message *)
                (* for a reattempt on the next startup.  Otherwise this *)
                (* is a hard failure, one for which we give up trying.  *)
                (* (Soft failures, for which a retry is appropriate,    *)
                (* have already been handled by MailOneMessage.)        *)

                IF (p^.sendto <> NIL) AND NOT ShutdownRequest THEN
                    IF p^.NotifyOnFailure THEN
                        SendRejectionLetter (p, LogID, TRUE);
                    END (*IF*);
                    DeleteRelayList (p^.sendto);
                END (*IF*);

                (* Delete the job record and message file, except in    *)
                (* the special case of shutdown.  Even then we can      *)
                (* delete the job record - but not the message file -   *)
                (* because the job will be re-created the next time     *)
                (* this program is run.                                 *)

                IF p^.sendto = NIL THEN
                    IF filename[0] <> Nul THEN
                        OS2.DosDelete (filename);
                    END (*IF*);
                ELSE
                    DeleteRelayList (p^.sendto);
                END (*IF*);
                DISPOSE (p);

            END (*IF*);

        END (*LOOP*);

        (* For neatness, and to make it easier to check the program for *)
        (* correctness, discard any pending jobs that we didn't get to  *)
        (* because of shutdown.                                         *)

        Obtain (OutboundMail.access);
        p := OutboundMail.head;
        WHILE p <> NIL DO
            OutboundMail.head := p^.next;
            p^.next := NIL;
            DeleteRelayList (p^.sendto);
            DISPOSE (p);
            p := OutboundMail.head;
        END (*WHILE*);
        Release (OutboundMail.access);

        Signal (TaskDone);

    END MailerTask;

(************************************************************************)
(*            THE TASK THAT HANDLES RETRANSMISSION ATTEMPTS             *)
(************************************************************************)

PROCEDURE RetryTask;

    (* Runs forever, taking mail off the retry list and putting each    *)
    (* item on the output queue as it becomes time to send it.          *)
    (* We give up after about five days.                                *)

    (* This task does not create or destroy data, it merely moves       *)
    (* jobs from the retry list to the output queue.                    *)

    CONST DefaultCheckInterval = 30*1000;    (* thirty seconds *)

    VAR TimedOut: BOOLEAN;  CheckInterval, Now: CARDINAL;
        p: OutJobPtr;

    BEGIN
        CheckInterval := DefaultCheckInterval;
        LOOP
            TimedWait (Retry, CheckInterval, TimedOut);
            IF ShutdownRequest THEN
                EXIT (*LOOP*);
            END (*IF*);
            Obtain (RetryList.access);
            LOOP
                p := RetryList.head;
                Now := time();
                IF (p = NIL) OR (p^.sendtime > Now) THEN
                    EXIT (*LOOP*);
                END (*IF*);

                (* It's time to resend this item. *)

                RetryList.head := p^.next;
                AddToOutQueue (p);

            END (*LOOP*);
            Release (RetryList.access);

            IF p = NIL THEN
                CheckInterval := DefaultCheckInterval;
            ELSE
                CheckInterval := 1000*(p^.sendtime - Now);
            END (*IF*);

        END (*LOOP*);
        Signal (TaskDone);

    END RetryTask;

(************************************************************************)
(*                      THE ONLINE/OFFLINE CHECKER                      *)
(************************************************************************)

PROCEDURE OnlineChecker;

    (* Runs forever, periodically checking whether a file called        *)
    (* 'ONLINE' exists.  If it does, we enable the sending of outbound  *)
    (* mail.                                                            *)

    CONST InitialDelay = 10*1000;              (* ten seconds  *)
          (*DefaultCheckInterval = 5*60*1000;    (* five minutes *)*)
          DefaultCheckInterval = 15*1000;    (* 15 seconds while testing *)

    VAR TimedOut, WeWereOffline: BOOLEAN;  CheckInterval: CARDINAL;
        OurIPAddress: CARDINAL;
        txtbuf: ARRAY [0..16] OF CHAR;

    BEGIN
        CheckInterval := InitialDelay;
        LOOP
            WeWereOffline := WeAreOffline;
            TimedWait (CheckIfOnline, CheckInterval, TimedOut);
            IF ShutdownRequest THEN
                EXIT (*LOOP*);
            END (*IF*);

            WeAreOffline := NOT FileSys.Exists ('ONLINE');
            IF WeAreOffline <> WeWereOffline THEN
                IF WeAreOffline THEN

                    (* We've just gone off-line. *)

                    IF ScreenEnabled THEN
                        WriteStringAt (0, 60, "Offline          ");
                    END (*IF*);
                ELSE

                    (* We've just come on-line. *)

                    RecomputeOurHostName (OurIPAddress, LocalHost);
                    IF ScreenEnabled THEN
                        IPToString (OurIPAddress, txtbuf);
                        WriteStringAt (0, 60, txtbuf);
                    END (*IF*);
                    WITH OfflineData DO
                        Obtain (access);
                        WHILE PendingCount > 0 DO
                            Signal (SomethingToSend);
                            DEC (PendingCount);
                        END (*WHILE*);
                        Release (access);
                    END (*WITH*);
                END (*IF*);
            END (*IF*);

            CheckInterval := DefaultCheckInterval;

        END (*LOOP*);
        Signal (TaskDone);

    END OnlineChecker;

(************************************************************************)
(*                    FILE INPUT/OUTPUT UTILITIES                       *)
(************************************************************************)

PROCEDURE FReadChar (cid: IOChan.ChanId;  VAR (*OUT*) ch: CHAR): BOOLEAN;

    (* Reads a single character from a file. *)

    VAR NumberRead: CARDINAL;

    BEGIN
        IOChan.RawRead (cid, ADR(ch), 1, NumberRead);
        IF NumberRead = 1 THEN
            RETURN TRUE;
        ELSE
            ch := Nul;  RETURN FALSE;
        END (*IF*);
    END FReadChar;

(************************************************************************)

TYPE CharSet = SET OF CHAR;
CONST Digits = CharSet {'0'..'9'};

(************************************************************************)

PROCEDURE SkipBlanks (cid: IOChan.ChanId;
                                   VAR (*INOUT*) NextChar: CHAR): BOOLEAN;

    (* Sets NextChar to the next nonblank character in the file. *)

    VAR success: BOOLEAN;

    BEGIN
        success := TRUE;
        WHILE success AND (NextChar = ' ') DO
            success := FReadChar (cid, NextChar);
        END (*WHILE*);
        RETURN success;
    END SkipBlanks;

(************************************************************************)

PROCEDURE SkipBlanksAndEOL (cid: IOChan.ChanId;
                                   VAR (*INOUT*) NextChar: CHAR): BOOLEAN;

    (* Like SkipBlanks, but also skips CR and LF. *)

    VAR success: BOOLEAN;

    BEGIN
        success := TRUE;
        WHILE success AND (NextChar IN CharSet{' ', CR, LF}) DO
            success := FReadChar (cid, NextChar);
        END (*WHILE*);
        RETURN success;
    END SkipBlanksAndEOL;

(************************************************************************)

PROCEDURE HexToBin (ch: CHAR): CARDINAL;

    (* Converts one hexadecimal character. *)

    BEGIN
        IF ch IN CharSet {'0'..'9'} THEN
            RETURN ORD(ch) - ORD('0');
        ELSE
            RETURN ORD(CAP(ch)) - ORD('A') + 10;
        END (*IF*);
    END HexToBin;

(************************************************************************)

PROCEDURE FReadHexByte (cid: IOChan.ChanId;  VAR (*OUT*) value: CARDINAL;
                                   VAR (*INOUT*) NextChar: CHAR): BOOLEAN;

    (* Reads a two-digit hexadecimal number from the file. *)

    VAR success: BOOLEAN;

    BEGIN
        value := 16*HexToBin (NextChar);
        success := FReadChar (cid, NextChar);
        IF success THEN
            INC (value, HexToBin(NextChar));
            success := FReadChar (cid, NextChar);
        END (*IF*);
        RETURN success;
    END FReadHexByte;

(************************************************************************)

PROCEDURE FReadCard (cid: IOChan.ChanId;  VAR (*OUT*) value: CARDINAL;
                                   VAR (*INOUT*) NextChar: CHAR): BOOLEAN;

    (* Reads a decimal number from the file. *)

    VAR success: BOOLEAN;

    BEGIN
        success := TRUE;  value := 0;
        WHILE success AND (NextChar IN Digits) DO
            value := 10*value + (ORD(NextChar) - ORD('0'));
            success := FReadChar (cid, NextChar);
        END (*WHILE*);
        RETURN success;
    END FReadCard;

(************************************************************************)

PROCEDURE FReadString (cid: IOChan.ChanId;
                                VAR (*OUT*) string: ARRAY OF CHAR;
                                Stoppers: CharSet;
                                VAR (*INOUT*) NextChar: CHAR): BOOLEAN;

    (* Reads a string from a file, stopping when NextChar is one of the *)
    (* characters in Stoppers, or when we reach the end of the file.    *)

    VAR j: CARDINAL;  success: BOOLEAN;

    BEGIN
        success := TRUE;  j := 0;
        WHILE success AND NOT (NextChar IN Stoppers) DO
            string[j] := NextChar;  INC(j);
            success := FReadChar (cid, NextChar);
        END (*WHILE*);
        string[j] := Nul;
        RETURN success;
    END FReadString;

(************************************************************************)
(*                  MOVING UNSENT MAIL TO AND FROM DISK                 *)
(************************************************************************)

PROCEDURE LoadRelayList (cid: IOChan.ChanId;  VAR (*OUT*) RL: RelayList;
                                   VAR (*INOUT*) NextChar: CHAR): BOOLEAN;

    (* Reads a relay list from a file. *)

    VAR previous, current: RelayListPointer;  success: BOOLEAN;

    BEGIN
        NEW(RL);
        WITH RL^ DO
            count := 0;  head := NIL;
        END (*WITH*);
        previous := NIL;  success := TRUE;
        IF NextChar = '(' THEN
            success := FReadChar (cid, NextChar);
            WHILE success AND (NextChar <> ')') DO
                NEW (current);
                success := FReadString (cid, current^.forwardpath,
                                        CharSet{' ', ',', ')'}, NextChar)
                            AND SkipBlanks (cid, NextChar);
                IF success AND (NextChar = ',') THEN
                    success := FReadChar (cid, NextChar);
                END (*IF*);
                current^.next := NIL;
                IF previous = NIL THEN RL^.head := current
                ELSE previous^.next := current
                END (*IF*);
                previous := current;
                INC (RL^.count);
            END (*WHILE*);
        END (*IF*);

        (* Consume the final ')'. *)

        IF success THEN
            EVAL (FReadChar (cid, NextChar));
        END (*IF*);

        IF RL^.count = 0 THEN
            DISPOSE (RL);
        END (*IF*);

        RETURN success;

    END LoadRelayList;

(************************************************************************)

PROCEDURE LoadOffset (cid: IOChan.ChanId;
                      VAR (*OUT*) offset: RndFile.FilePos;
                      VAR (*INOUT*) NextChar: CHAR): BOOLEAN;

    (* Reads a file position in hexadecimal. *)

    CONST HexDigits = CharSet {'0'..'9', 'A'..'F'};

    VAR j: [1..RndFile.FilePosSize];  value: CARDINAL;

    BEGIN
        FOR j := 1 TO RndFile.FilePosSize DO
            IF NextChar IN HexDigits THEN
                EVAL (FReadHexByte (cid, value, NextChar));
            ELSE
                value := 0;
            END (*IF*);
            offset[j] := CAST(LOC, value);
        END (*FOR*);
        RETURN TRUE;
    END LoadOffset;

(************************************************************************)

PROCEDURE LoadJob (cid: IOChan.ChanId;
                            VAR (*INOUT*) NextChar: CHAR): OutJobPtr;

    (* Loads a previously saved mail job from a disk file.  Returns     *)
    (* NIL if there's nothing left to load.                             *)

    VAR p: OutJobPtr;  boolcode: CARDINAL;

    BEGIN
        p := NIL;  boolcode := 0;
        EVAL (SkipBlanksAndEOL (cid, NextChar));
        IF NextChar IN Digits THEN
            NEW (p);
            IF FReadCard (cid, p^.sendtime, NextChar)
                        AND SkipBlanks (cid, NextChar)
                        AND FReadCard (cid, p^.RetryNumber, NextChar)
                        AND SkipBlanks (cid, NextChar)
                        AND FReadString (cid, p^.file, CharSet{' '}, NextChar)
                        AND SkipBlanks (cid, NextChar)
                        AND FReadCard (cid, boolcode, NextChar)
                        AND SkipBlanks (cid, NextChar)
                        AND LoadRelayList (cid, p^.sendto, NextChar)
                        AND SkipBlanks (cid, NextChar)
                        AND FReadString (cid, p^.sender, CharSet{' ', CR}, NextChar)
                        AND SkipBlanks (cid, NextChar)
                        AND LoadOffset (cid, p^.offset, NextChar)
            THEN
                p^.ID := NIL;
                p^.NotifyOnFailure := CAST (BOOLEAN, boolcode);
            ELSE
                DISPOSE (p);
            END (*IF*);
        END (*IF*);
        RETURN p;
    END LoadJob;

(************************************************************************)

PROCEDURE OldLoadUnsentMail;

    (* Loads the queue of outbound mail from a disk file.  This queue   *)
    (* results from a shutdown and subsequent restart, so we give at    *)
    (* least one extra try for mail in the "retry" category.            *)

    VAR cid: IOChan.ChanId;  result: RndFile.OpenResults;
        p: OutJobPtr;  NextChar: CHAR;  Now: CARDINAL;
        SaveFileName: FilenameString;

    BEGIN
        SaveFileName := ForwardDirName;
        Strings.Append ("Delayed.MAI", SaveFileName);
        RndFile.OpenOld (cid, SaveFileName,
                          ChanConsts.read+ChanConsts.raw, result);
        IF result = ChanConsts.opened THEN
            NextChar := ' ';  Now := time();
            LOOP
                p := LoadJob (cid, NextChar);
                IF p = NIL THEN EXIT(*LOOP*)
                ELSIF p^.sendtime <= Now THEN
                    p^.sendtime := Now;
                    AddToOutQueue (p);
                ELSE
                    AddToRetryList (p);
                END (*IF*);
            END (*LOOP*);
            RndFile.Close (cid);
        END (*IF*);
        OS2.DosDelete (SaveFileName);
    END OldLoadUnsentMail;

(************************************************************************)
(*                    LOAD NEW-FORMAT UNSENT MAIL                       *)
(************************************************************************)

PROCEDURE ReadNameList (cid: IOChan.ChanId;
                        VAR (*INOUT*) NextChar: CHAR): RelayList;

    (* Reads a comma-separated list of names enclosed in parentheses. *)

    VAR p, last: RelayListPointer;  result: RelayList;

    BEGIN
        NEW(result);
        WITH result^ DO
            count := 0;  head := NIL;
        END (*WITH*);
        last := NIL;
        IF NextChar = '(' THEN
            LOOP
                NEW(p);
                p^.next := NIL;  p^.failuremessage := "";
                EVAL( FReadChar(cid, NextChar)
                          AND FReadString (cid, p^.forwardpath,
                                           CharSet{',', ')'}, NextChar));
                IF last = NIL THEN result^.head := p
                ELSE last^.next := p
                END (*IF*);
                last := p;
                INC (result^.count);
                IF NextChar <> ',' THEN EXIT(*LOOP*) END(*IF*);
            END (*LOOP*);
        END (*IF*);
        IF result^.count = 0 THEN
            DISPOSE (result);
        END (*IF*);
        RETURN result;
    END ReadNameList;

(************************************************************************)

PROCEDURE NewLoadUnsentMail;

    (* Loads the queue of outbound mail from a disk file.  This queue   *)
    (* results from a shutdown and subsequent restart, so we give at    *)
    (* least one extra try for mail in the "retry" category.            *)

    VAR cid: IOChan.ChanId;  result: RndFile.OpenResults;
        p: OutJobPtr;  NextChar: CHAR;  Now, CharsRead: CARDINAL;
        mask, filename: FilenameString;
        preamble: PreambleType;
        D: DirectoryEntry;
        found: BOOLEAN;

    BEGIN
        mask := ForwardDirName;
        Strings.Append ("*.FWD", mask);
        found := FirstDirEntry (mask, FALSE, D);
        WHILE found DO
            filename := ForwardDirName;
            Strings.Append (D.name, filename);
            RndFile.OpenOld (cid, filename,
                              ChanConsts.read+ChanConsts.raw, result);
            IF result = ChanConsts.opened THEN
                IOChan.RawRead (cid, ADR(preamble), SIZE(preamble), CharsRead);
                IF (CharsRead = SIZE(preamble)) AND Strings.Equal(preamble.version, "V000") THEN
                    NEW (p);
                    p^.file := filename;
                    WITH preamble DO
                        p^.sendtime := sendtime;
                        p^.RetryNumber := RetryNumber;
                        p^.NotifyOnFailure := NotifyOnFailure;
                    END (*WITH*);
                    EVAL( FReadChar(cid, NextChar)
                          AND FReadString (cid, p^.sender, CharSet{'('}, NextChar));
                    p^.sendto := ReadNameList (cid, NextChar);
                    p^.offset := RndFile.CurrentPos (cid);

                    Now := time();
                    IF p^.sendto = NIL THEN
                        DISPOSE(p);
                    ELSIF p^.sendtime <= Now THEN
                        p^.sendtime := Now;
                        AddToOutQueue (p);
                    ELSE
                        AddToRetryList (p);
                    END (*IF*);

                END (*IF*);
                RndFile.Close(cid);
            END (*IF*);

            found := NextDirEntry(D);
        END (*WHILE*);
        DirSearchDone (D);

    END NewLoadUnsentMail;

(************************************************************************)
(*          INITIALISING THE LIST OF UNSENT MAIL AT STARTUP             *)
(************************************************************************)

(************************************************************************)
(* In this transition period there will be a mixture of old-format and  *)
(* new-format mail in the "forward" directory.  What we need to do is:  *)
(*  (a) load the two output queues with the old-format mail.            *)
(*  (b) for each file in the directory, check whether we already have   *)
(*      it on one of the two lists.  If not, try to load it as          *)
(*      a new-format item.                                              *)
(*                                                                      *)
(* OLD FORMAT                                                           *)
(*   The file Delayed.MAI lists the items.  For each unsent item        *)
(*   there is a line in Delayed.MAI giving send time; retry number;     *)
(*   message file name; a character '0' or '1' where '1' means          *)
(*   "notify on failure"; recipient list; sender; and offset.           *)
(*                                                                      *)
(* NEW FORMAT                                                           *)
(*   See description near the beginning of this module.                 *)
(*                                                                      *)
(************************************************************************)

PROCEDURE LoadUnsentMail;

    BEGIN
        OldLoadUnsentMail;
        NewLoadUnsentMail;
    END LoadUnsentMail;

(************************************************************************)
(*                           INITIALISATION                             *)
(************************************************************************)

PROCEDURE LoadINIData;

    (* Loads the values of ForwardDirName, ForwardRelayHost, and the initial    *)
    (* value of NextName.  Also starts the transaction logging.                 *)

    VAR hini: OS2.HINI;  TransLevel: CARDINAL;

    BEGIN
        TransLevel := 2;
        hini := OpenINIFile ("weasel.ini");
        IF hini <> OS2.NULLHANDLE THEN
            IF NOT INIGetString (hini, "$SYS", "MailRoot",
                                       ForwardDirName) THEN
                ForwardDirName := "/MPTN/ETC/MAIL/";
            END (*IF*);
            IF NOT INIGetString (hini, "$SYS", "ForwardRelay",
                                       ForwardRelayHost) THEN
                ForwardRelayHost := "";
            END (*IF*);
            IF NOT INIGet (hini, "$SYS", "VName", NextName) THEN
                NextName := "00000000";
            END (*IF*);
            IF NOT INIGet (hini, "$SYS", "TransLevel", TransLevel) THEN
                TransLevel := 2;
            END (*IF*);
            OS2.PrfCloseProfile (hini);
        END (*IF*);
        WHILE ForwardRelayHost[0] = ' ' DO
            Strings.Delete (ForwardRelayHost, 0, 1);
        END (*WHILE*);
        IF TransLevel > 3 THEN
            TransLevel := 3;
        END (*IF*);
        IF NOT NotDetached() THEN
            TransLevel := TransLevel MOD 2;
        END (*IF*);
        StartTransactionLogging ("WEASEL.LOG", TransLevel);
    END LoadINIData;

(************************************************************************)

VAR hini: OS2.HINI;

BEGIN
    ScreenEnabled := NotDetached();
    ShutdownRequest := FALSE;  TaskCount := 0;
    OurHostName (LocalHost);
    LoadINIData;
    Strings.Append ("Forward/", ForwardDirName);
    CreateLock (NextNameLock);
    IF ScreenEnabled THEN
        ClearScreen;  SetBoundary (2);
        WriteStringAt (0, 60, "Offline          ");
    END (*IF*);
    WeAreOffline := TRUE;
    WITH OfflineData DO
        CreateLock (access);
        PendingCount := 0;
    END (*WITH*);
    WITH OutboundMail DO
        CreateLock (access);
        head := NIL;
        tail := NIL;
    END (*WITH*);
    CreateSemaphore (CheckIfOnline, 0);
    CreateSemaphore (SomethingToSend, 0);
    CreateSemaphore (TaskDone, 0);
    WITH RetryList DO
        head := NIL;
        CreateLock (access);
    END (*WITH*);
    CreateSemaphore (Retry, 0);
    CreateTask (RetryTask, 3, "retry send");  INC (TaskCount);
    CreateTask1 (MailerTask, 4, "send mail A", CAST(ADDRESS,TaskCount));
    INC (TaskCount);
    CreateTask1 (MailerTask, 4, "send mail B", CAST(ADDRESS,TaskCount));
    INC (TaskCount);
    CreateTask1 (MailerTask, 4, "send mail C", CAST(ADDRESS,TaskCount));
    INC (TaskCount);
    CreateTask1 (MailerTask, 4, "send mail D", CAST(ADDRESS,TaskCount));
    INC (TaskCount);
    LoadUnsentMail;
    CreateTask (OnlineChecker, 2, "online check");  INC (TaskCount);
FINALLY
    ShutdownRequest := TRUE;  Signal(Retry);  Signal(CheckIfOnline);
    REPEAT
        Signal(SomethingToSend);
        Wait (TaskDone);  DEC(TaskCount);
    UNTIL TaskCount = 0;
    hini := OpenINIFile ("weasel.ini");
    IF hini <> OS2.NULLHANDLE THEN
        Obtain (NextNameLock);
        INIPut (hini, "$SYS", "VName", NextName);
        Release (NextNameLock);
        OS2.PrfCloseProfile (hini);
    END (*IF*);
END RelayMail.

