IMPLEMENTATION MODULE POPCommands;

        (********************************************************)
        (*                                                      *)
        (*       Command interpreter for POP3 server            *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            21 April 1998                   *)
        (*  Last edited:        2 September 1999                *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (********************************************************)

(********************************************************************************)
(*                        COMPLIANCE WITH THE STANDARD                          *)
(********************************************************************************)
(*                                                                              *)
(* I'm working from the POP3 standard RFC1939                                   *)
(* All commmands from that standard are now implemented:                        *)
(*                                                                              *)
(*    APOP, DELE, LIST, NOOP, PASS, QUIT, RETR, RSET, STAT, TOP, UIDL, USER     *)
(*                                                                              *)
(********************************************************************************)

FROM SYSTEM IMPORT CAST, LOC, CARD8;

IMPORT Strings, OS2, IOChan;

FROM Storage IMPORT
    (* proc *)  ALLOCATE;

FROM LowLevel IMPORT
    (* proc *)  EVAL;

FROM TaskControl IMPORT
    (* type *)  Lock,
    (* proc *)  CreateLock, Obtain, Release;

FROM Semaphores IMPORT
    (* type *)  Semaphore;

FROM Conversions IMPORT
    (* proc *)  StringToCardinal;

FROM Sockets IMPORT
    (* type *)  Socket,
    (* proc *)  send;

FROM Names IMPORT
    (* type *)  FilenameString, UserName, UserNameIndex;

FROM Hosts IMPORT
    (* proc *)  AppendOurHostName;

FROM MyClock IMPORT
    (* proc *)  AppendTimeString;

FROM MD5 IMPORT
    (* type *)  DigestType;

FROM InetUtilities IMPORT
    (* proc *)  StartTransactionLogging, AddToTransactionLog,
                AddEOL, ConvertCard, OpenINIFile, INIGet, CurrentTimeToString;

FROM POPData IMPORT
    (* type *)  Mailbox,
    (* proc *)  OpenMailbox, DiscardMailbox, PasswordOK, NumberAndSize,
                SizeOfMessage, SendMessage, MarkForDeletion, UndeleteAll,
                CommitChanges, MaxMessageNumber, GetUID, APOPCheck;

FROM FDFiles IMPORT
    (* proc *)  OpenAtEnd, CloseFile, FWriteString, FWriteLJCard, FWriteLn;

FROM TransLog IMPORT
    (* type *)  TransactionLogID,
    (* proc *)  LogTransaction;

(********************************************************************************)

CONST Nul = CHR(0);

TYPE
    FourChar = ARRAY [0..3] OF CHAR;

    ClientState = (Idle, LoggedIn, MustExit);

    (* The session record.  The fields are:                             *)
    (*     ID          a session identifier for transaction logging     *)
    (*     socket      The command socket                               *)
    (*     state       To track whether the user is currently logged in.*)
    (*     mailbox     Information about the user's mailbox.            *)
    (*     watchdog    A semaphore on which we have to signal on slow   *)
    (*                  transfers so as to avoid timeout.               *)
    (*     retrcount   Number of messages the client has retrieved      *)
    (*     retrchars   Number of characters retrieved                   *)
    (*     delecount   Number of messages the client has deleted        *)
    (*     delechars   Number of characters deleted                     *)
    (*     tdelcount   Tentative values for delecount and delechars,    *)
    (*     tdelchars     - deletions not yet committed                  *)

    Session = POINTER TO
                  RECORD
                      ID: TransactionLogID;
                      TimeStamp: FilenameString;
                      socket: Socket;
                      state: ClientState;
                      mailbox: Mailbox;
                      watchdog: Semaphore;
                      retrcount, retrchars,
                        delecount, delechars,
                        tdelcount, tdelchars: CARDINAL;
                      username: UserName;
                  END (*RECORD*);

(********************************************************************************)

VAR
    (* LogPOPusers is TRUE iff we are keeping a user log. *)

    LogPOPusers: BOOLEAN;

    (* Exclusive access lock for the user log. *)

    LogFileLock: Lock;

(********************************************************************************)
(*                             USER LOGGING                                     *)
(********************************************************************************)

PROCEDURE WriteLogData (S: Session);

    (* Writes the summary for this session to the user log. *)

    VAR cid: IOChan.ChanId;  datetime: ARRAY [0..31] OF CHAR;

    BEGIN
        Obtain (LogFileLock);
        cid := OpenAtEnd ("POP.LOG");
        CurrentTimeToString (datetime);
        IF S^.retrcount > 0 THEN
            FWriteString (cid, datetime);  FWriteString (cid, " ");
            FWriteString (cid, S^.username);  FWriteString (cid, " retrieved ");
            FWriteLJCard (cid, S^.retrcount);  FWriteString (cid, " files (");
            FWriteLJCard (cid, S^.retrchars);  FWriteString (cid, " bytes)");
            FWriteLn (cid);
        END (*IF*);
        IF S^.delecount > 0 THEN
            FWriteString (cid, datetime);  FWriteString (cid, " ");
            FWriteString (cid, S^.username);  FWriteString (cid, " deleted ");
            FWriteLJCard (cid, S^.delecount);  FWriteString (cid, " files (");
            FWriteLJCard (cid, S^.delechars);  FWriteString (cid, " bytes)");
            FWriteLn (cid);
        END (*IF*);

        CloseFile (cid);
        Release (LogFileLock);
    END WriteLogData;

(********************************************************************************)
(*                         STARTING A NEW SESSION                               *)
(********************************************************************************)

PROCEDURE CreateTimeStamp (ID: TransactionLogID;  VAR (*OUT*) result: FilenameString);

    (* Sets result to a string of the form <nnn.clock@hostname>, where nnn is   *)
    (* derived from the ID, clock is derived from the current date and time,    *)
    (* and hostname is our local host name.                                     *)

    VAR pos: CARDINAL;

    BEGIN
        result[0] := '<';  pos := 1;
        ConvertCard (CAST(CARDINAL, ID), result, pos);
        result[pos] := '.';  INC(pos);  result[pos] := Nul;
        AppendTimeString (result);
        Strings.Append ('@', result);
        AppendOurHostName (result);
        Strings.Append ('>', result);
    END CreateTimeStamp;

(********************************************************************************)

PROCEDURE AppendTimeStamp (S: Session;  VAR (*INOUT*) buffer: ARRAY OF CHAR);

    (* Appends the TimeStamp string to a string buffer. *)

    BEGIN
        Strings.Append (S^.TimeStamp, buffer);
    END AppendTimeStamp;

(********************************************************************************)

PROCEDURE OpenSession (CommandSocket: Socket;  KeepAlive: Semaphore;
                                    LogID: TransactionLogID): Session;

    (* Creates a new session state record.  During lengthy operations           *)
    (* we have to do a Signal(KeepAlive) every so often in order to stop the    *)
    (* session from timing out.                                                 *)

    VAR result: Session;

    BEGIN
        NEW (result);
        WITH result^ DO
            ID := LogID;
            mailbox := NIL;
            socket := CommandSocket;
            state := Idle;
            CreateTimeStamp (ID, TimeStamp);
            watchdog := KeepAlive;
            retrcount := 0;  retrchars := 0;
            delecount := 0;  delechars := 0;
            tdelcount := 0;  tdelchars := 0;
            username := "";
        END (*WITH*);
        RETURN result;
    END OpenSession;

(********************************************************************************)

PROCEDURE CloseSession (S: Session);

    (* Destroys the session state record. *)

    BEGIN
        IF LogPOPusers AND ((S^.retrcount > 0) OR (S^.delecount > 0)) THEN
            WriteLogData (S);
        END (*IF*);
        DiscardMailbox (S^.mailbox);
        DISPOSE (S);
    END CloseSession;

(********************************************************************************)
(*                       SENDING REPLY BACK TO CLIENT                           *)
(********************************************************************************)

PROCEDURE Reply2 (session: Session;  message1, message2: ARRAY OF CHAR);

    (* Sends all of message1, followed by message2, followed by end-of-line.    *)
    (* If the operation fails, session^.state is set to MustExit.               *)

    VAR buffer: ARRAY [0..511] OF CHAR;  length: CARDINAL;

    BEGIN
        Strings.Assign (message1, buffer);
        Strings.Append (message2, buffer);
        length := AddEOL (buffer);
        IF send (session^.socket, buffer, length, 0) = MAX(CARDINAL) THEN
            session^.state := MustExit;
        END (*IF*);
    END Reply2;

(********************************************************************************)

PROCEDURE Reply (session: Session;  message: ARRAY OF CHAR);

    (* Like Reply2, except that there is no message2. *)

    VAR buffer: ARRAY [0..511] OF CHAR;  length: CARDINAL;

    BEGIN
        Strings.Assign (message, buffer);
        length := AddEOL (buffer);
        IF send (session^.socket, buffer, length, 0) = MAX(CARDINAL) THEN
            session^.state := MustExit;
        END (*IF*);
    END Reply;

(********************************************************************************)
(*                     HANDLERS FOR SOME ERROR CONDITIONS                       *)
(********************************************************************************)

PROCEDURE NoSuchCommand (session: Session;  VAR (*IN*) Command: ARRAY OF CHAR);

    (* Command is not a recognised command. *)

    BEGIN
        Reply2 (session, "-ERR Unknown command ", Command);
    END NoSuchCommand;

(********************************************************************************)

PROCEDURE NotLoggedIn (session: Session;  VAR (*IN*) Command: ARRAY OF CHAR);

    (* Command is illegal because user is not yet logged in. *)

    BEGIN
        Reply2 (session, "-ERR Not logged in ", Command);
    END NotLoggedIn;

(********************************************************************************)
(*                     HANDLERS FOR THE INDIVIDUAL COMMANDS                     *)
(********************************************************************************)

PROCEDURE APOP (session: Session;  VAR (*IN*) args: ARRAY OF CHAR);

    VAR name: UserName;  j: CARDINAL;

    BEGIN
        (* Extract the username. *)

        j := 0;
        WHILE (args[j] <> Nul) AND (args[j] <> ' ') DO
            IF j <= MAX(UserNameIndex) THEN
                name[j] := args[j];
            END (*IF*);
            INC (j);
        END (*WHILE*);
        IF j <= MAX(UserNameIndex) THEN
            name[j] := Nul;
        END (*IF*);
        WHILE args[j] = ' ' DO
            INC (j);
        END (*WHILE*);
        Strings.Delete (args, 0, j);

        (* Open the mailbox, check that username is valid. *)

        IF OpenMailbox (session^.mailbox, name) THEN

            (* Check the digest string in args. *)

            session^.username := name;
            CASE APOPCheck (session^.mailbox, session^.ID, args, session^.TimeStamp) OF
               |  0:  session^.state := LoggedIn;
                      Reply (session, "+OK logged in");
               |  1:  Reply (session, "-ERR authorisation failure");
               |  2:  session^.state := LoggedIn;
                      Reply (session, "+OK mailbox is locked");
               |  3:  Reply (session, "-ERR configuration error");
            END (*CASE*);
        ELSE
            Reply (session, "-ERR no such user");
        END (*IF*);

    END APOP;

(********************************************************************************)

PROCEDURE DELE (session: Session;  VAR (*IN*) number: ARRAY OF CHAR);

    VAR N, ItemSize: CARDINAL;

    BEGIN
        N := StringToCardinal (number);
        IF MarkForDeletion (session^.mailbox, N, ItemSize) THEN
            INC (session^.tdelcount);  INC (session^.tdelchars, ItemSize);
            Reply (session, "+OK message deleted");
        ELSE
            Reply (session, "-ERR no such message");
        END (*IF*);
    END DELE;

(********************************************************************************)

PROCEDURE LIST (session: Session;  VAR (*IN*) number: ARRAY OF CHAR);

    CONST buffersize = 128;

    VAR N, size, pos: CARDINAL;
        buffer: ARRAY [0..buffersize-1] OF CHAR;

    (****************************************************************************)

    PROCEDURE SendNumberAndSize;

        (* Appends the values of N and size to buffer, then sends buffer. *)

        BEGIN
            pos := LENGTH (buffer);
            ConvertCard (N, buffer, pos);
            buffer[pos] := ' ';  INC(pos);
            ConvertCard (size, buffer, pos);
            IF pos < buffersize THEN
                buffer[pos] := Nul;
            END (*IF*);
            Reply (session, buffer);
        END SendNumberAndSize;

    (****************************************************************************)

    BEGIN
        IF number[0] = Nul THEN

            (* List all. *)

            Reply (session, "+OK");
            FOR N := 1 TO MaxMessageNumber(session^.mailbox) DO
                buffer := "";
                IF SizeOfMessage (session^.mailbox, N, size) THEN
                    SendNumberAndSize;
                END (*IF*);
            END (*FOR*);
            Reply (session, ".");

        ELSE

            (* List for message "number". *)

            N := StringToCardinal (number);
            IF SizeOfMessage (session^.mailbox, N, size) THEN
                Strings.Assign ("+OK ", buffer);
                SendNumberAndSize;
            ELSE
                Reply (session, "-ERR No such message");
            END (*IF*);

        END (*IF*);

    END LIST;

(********************************************************************************)

PROCEDURE NOOP (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    BEGIN
        dummy[0] := dummy[0];     (* to avoid a compiler warning *)
        Reply (session, "+OK");
    END NOOP;

(********************************************************************************)

PROCEDURE PASS (session: Session;  VAR (*IN*) password: ARRAY OF CHAR);

    BEGIN
        CASE PasswordOK (session^.mailbox, password) OF
           |  0:  session^.state := LoggedIn;
                  Reply (session, "+OK logged in");
           |  1:  Reply (session, "-ERR authorisation failure");
           |  2:  session^.state := LoggedIn;
                  Reply (session, "+OK mailbox is locked");
           |  3:  Reply (session, "-ERR configuration error");
        END (*CASE*);
    END PASS;

(********************************************************************************)

PROCEDURE QUIT (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    BEGIN
        dummy[0] := dummy[0];                   (* to avoid a compiler warning *)
        WITH session^ DO
            IF state = LoggedIn THEN
                CommitChanges (mailbox);
                delecount := tdelcount;  delechars := tdelchars;
            END (*IF*);
        END (*WITH*);
        Reply (session, "+OK");
        session^.state := MustExit;
    END QUIT;

(********************************************************************************)

PROCEDURE RETR (session: Session;  VAR (*IN*) number: ARRAY OF CHAR);

    VAR N, size: CARDINAL;

    BEGIN
        N := StringToCardinal (number);
        IF SizeOfMessage (session^.mailbox, N, size) THEN
            Reply (session, "+OK");
            IF SendMessage (session^.socket, session^.watchdog,
                            session^.mailbox, N, MAX(CARDINAL), session^.ID) THEN
                INC (session^.retrcount);  INC (session^.retrchars, size);
            ELSE
                session^.state := MustExit;
            END (*IF*);
        ELSE
            Reply (session, "-ERR no such message");
        END (*IF*);
    END RETR;

(********************************************************************************)

PROCEDURE RSET (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    BEGIN
        dummy[0] := dummy[0];                   (* to avoid a compiler warning *)
        UndeleteAll (session^.mailbox);
        session^.tdelcount := 0;  session^.tdelchars := 0;
        Reply (session, "+OK");
    END RSET;

(********************************************************************************)

PROCEDURE STAT (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    CONST buffersize = 128;

    VAR N, size, pos: CARDINAL;
        buffer: ARRAY [0..buffersize-1] OF CHAR;

    BEGIN
        dummy[0] := dummy[0];                   (* to avoid a compiler warning *)
        NumberAndSize (session^.mailbox, N, size);
        Strings.Assign ("+OK ", buffer);
        pos := 4;
        ConvertCard (N, buffer, pos);
        buffer[pos] := ' ';  INC(pos);
        ConvertCard (size, buffer, pos);
        IF pos < buffersize THEN
            buffer[pos] := Nul;
        END (*IF*);
        Reply (session, buffer);
    END STAT;

(********************************************************************************)

PROCEDURE TOP (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    VAR N, size, lines: CARDINAL;

    BEGIN
        N := StringToCardinal (Params);
        size := 1;
        WHILE Params[size] <> ' ' DO
            INC (size);
        END (*WHILE*);
        Strings.Delete (Params, 0, size+1);
        lines := StringToCardinal (Params);
        IF SizeOfMessage (session^.mailbox, N, size) THEN
            Reply (session, "+OK");
            IF NOT SendMessage (session^.socket, session^.watchdog,
                            session^.mailbox, N, lines, session^.ID) THEN
                session^.state := MustExit;
            END (*IF*);
        ELSE
            Reply (session, "-ERR no such message");
        END (*IF*);
    END TOP;

(********************************************************************************)

PROCEDURE UIDL (session: Session;  VAR (*IN*) number: ARRAY OF CHAR);

    CONST buffersize = 64;

    VAR N, pos: CARDINAL;
        buffer: ARRAY [0..buffersize-1] OF CHAR;
        UID: DigestType;

    (****************************************************************************)

    PROCEDURE PutHexDigit (N: CARD8);

        (* Puts a single hexadecimal digit at buffer[pos], updates pos. *)

        BEGIN
            IF N < 10 THEN
                buffer[pos] := CHR(N + ORD('0'));
            ELSE
                buffer[pos] := CHR(N - 10 + ORD('A'));
            END (*IF*);
            INC (pos);
        END PutHexDigit;

    (****************************************************************************)

    PROCEDURE ConvertUID (array: ARRAY OF LOC);

        (* Converts a 16-byte code to a 32-char hexadecimal string, stores      *)
        (* the result at buffer[pos], updates pos.                              *)

        VAR j: [0..15];  val: CARD8;

        BEGIN
            FOR j := 0 TO 15 DO
                val := CAST(CARD8, array[j]);
                PutHexDigit (val DIV 16);
                PutHexDigit (val MOD 16);
            END (*FOR*);
        END ConvertUID;

    (****************************************************************************)

    PROCEDURE SendNumberAndUID;

        (* Appends the values of N and size to buffer, then sends buffer. *)

        BEGIN
            pos := LENGTH (buffer);
            ConvertCard (N, buffer, pos);
            buffer[pos] := ' ';  INC(pos);
            ConvertUID (UID);
            IF pos < buffersize THEN
                buffer[pos] := Nul;
            END (*IF*);
            Reply (session, buffer);
        END SendNumberAndUID;

    (****************************************************************************)

    BEGIN
        IF number[0] = Nul THEN

            (* UIDL for everything in the mailbox, excluding deleted messages. *)

            Reply (session, "+OK");
            FOR N := 1 TO MaxMessageNumber(session^.mailbox) DO
                buffer := "";
                IF GetUID (session^.mailbox, N, UID) THEN
                    SendNumberAndUID;
                END (*IF*);
            END (*FOR*);
            Reply (session, ".");

        ELSE

            (* List for message "number". *)

            N := StringToCardinal (number);
            IF GetUID (session^.mailbox, N, UID) THEN
                Strings.Assign ("+OK ", buffer);
                SendNumberAndUID;
            ELSE
                Reply (session, "-ERR No such message");
            END (*IF*);

        END (*IF*);

    END UIDL;

(********************************************************************************)

PROCEDURE USER (session: Session;  VAR (*IN*) username: ARRAY OF CHAR);

    BEGIN
        IF OpenMailbox (session^.mailbox, username) THEN
            Strings.Assign (username, session^.username);
            Reply (session, "+OK send password");
        ELSE
            Reply (session, "-ERR no such user");
        END (*IF*);
    END USER;

(********************************************************************************)
(*                      THE MAIN COMMAND DISPATCHER                             *)
(********************************************************************************)

TYPE
    KeywordNumber = [0..11];
    HandlerProc = PROCEDURE (Session, VAR (*IN*) ARRAY OF CHAR);
    HandlerArray = ARRAY KeywordNumber OF HandlerProc;
    KeywordArray = ARRAY KeywordNumber OF FourChar;

CONST
    KeywordList = KeywordArray {'APOP', 'DELE', 'LIST', 'NOOP', 'PASS', 'QUIT',
                                'RETR', 'RSET', 'STAT', 'TOP ', 'UIDL', 'USER'};

CONST
    HandlerList = HandlerArray {APOP, DELE, LIST, NOOP, PASS, QUIT,
                                RETR, RSET, STAT, TOP, UIDL, USER};

(********************************************************************************)

PROCEDURE HandleCommand (S: Session;  Command: ARRAY OF CHAR;
                                                     VAR (*OUT*) Quit: BOOLEAN);

    (* Executes one user command.  Returns with Quit=TRUE if the command is one *)
    (* that closes the session, or if the connection is lost.                   *)

    VAR k: [0..3];

    (****************************************************************************)

    PROCEDURE Compare4 (n: KeywordNumber): INTEGER;

        (* Compares the first four characters of Command with KeywordList[n].   *)
        (* Returns >0 if Command[0..3] > KeywordList[n], and so on.             *)

        VAR ch1, ch2: CHAR;

        BEGIN
            k := 0;
            LOOP
                ch1 := Command[k];  ch2 := KeywordList[n][k];
                IF ch1 > ch2 THEN RETURN +1
                ELSIF ch1 < ch2 THEN RETURN -1
                ELSIF k = 3 THEN RETURN 0
                END (*IF*);
                INC (k);
            END (*LOOP*);
        END Compare4;

    (****************************************************************************)

    VAR m: CARDINAL;  Match, QuitReceived: BOOLEAN;
        first, middle, last: CARDINAL;  test: INTEGER;
        Handler: HandlerProc;

    BEGIN
        (* Special case: add a space to a three-character command, to make the  *)
        (* search algorithm simpler.                                            *)

        IF Command[3] = CHR(0) THEN
            Command[3] := ' ';
            Command[4] := CHR(0);
        END (*IF*);

        (* Watch out for lower case. *)

        FOR k := 0 TO 3 DO
            Command[k] := CAP(Command[k]);
        END (*FOR*);

        (* Go through the keyword list to find a match with the command.  *)
        (* In this version I'm using a binary search.                     *)

        first := 0;  last := MAX(KeywordNumber);  Match := FALSE;
        LOOP
            middle := (first + last) DIV 2;
            test := Compare4 (middle);
            IF test < 0 THEN
                IF middle = 0 THEN
                    EXIT (*LOOP*);
                ELSE
                    last := middle - 1;
                END (*IF*);
            ELSIF test = 0 THEN
                Match := TRUE;  EXIT (*LOOP*);
            ELSIF test > 0 THEN
                first := middle + 1;
            END (*IF*);
            IF first > last THEN EXIT (*LOOP*) END (*IF*);
        END (*LOOP*);

        IF Match THEN
            Handler := HandlerList[middle];
        ELSE
            Handler := NoSuchCommand;
        END (*IF*);

        (* Echo command to transaction log. *)

        IF Handler = PASS THEN
            LogTransaction (S^.ID, "PASS ******");
        ELSE
            LogTransaction (S^.ID, Command);
        END (*IF*);

        (* If the user is not yet logged in, only APOP, USER, PASS, and QUIT are legal. *)

        QuitReceived := Handler = QUIT;
        IF NOT QuitReceived AND (S^.state <> LoggedIn) AND (Handler <> APOP)
                            AND (Handler <> USER) AND (Handler <> PASS) THEN
            Handler := NotLoggedIn;
        END (*IF*);

        (* Strip out the command characters, leaving only the parameters. *)

        IF Handler = NoSuchCommand THEN m := 0 ELSE m := 4 END(*IF*);
        WHILE (m < HIGH(Command)) AND (Command[m] = " ") DO INC(m) END (*WHILE*);
        Strings.Delete (Command, 0, m);

        (* Call the handler. *);

        Handler (S, Command);
        IF (S^.state = MustExit) AND NOT QuitReceived THEN
            LogTransaction (S^.ID, "Connection lost");
        END (*IF*);
        Quit := S^.state = MustExit;

    END HandleCommand;

(********************************************************************************)

VAR hini: OS2.HINI;

BEGIN
    CreateLock (LogFileLock);
    LogPOPusers := FALSE;
    hini := OpenINIFile ("weasel.ini");
    IF hini <> OS2.NULLHANDLE THEN
        EVAL (INIGet (hini, "$SYS", "LogPOPusers", LogPOPusers));
        OS2.PrfCloseProfile (hini);
    END (*IF*);
END POPCommands.

