IMPLEMENTATION MODULE AliasLists;

        (********************************************************)
        (*                                                      *)
        (*                Editor for alias lists                *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            7 July 1998                     *)
        (*  Last edited:        5 November 1998                 *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (********************************************************)

IMPORT OS2, Strings;

FROM LowLevel IMPORT
    (* proc *)  EVAL, Copy;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

FROM Keyboard IMPORT
    (* proc *)  StuffKeyboardBuffer, PutBack;

FROM Windows IMPORT
    (* type *)  Window, Colour, FrameType, DividerType,
    (* proc *)  OpenWindowHidden, CloseWindow, SetCursor, WriteChar,
                WriteString, GetScreenSize, GetKey, ColourSwap,
                EditString, EditAborted;

FROM MultiScreen IMPORT
    (* type *)  VirtualScreen,
    (* proc *)  MapToVirtualScreen, VirtualScreenOf, EnableScreenSwitching;

FROM SetupINI IMPORT
    (* proc *)  OurINIHandle;

FROM ListBoxes IMPORT
    (* type *)  ListBox,
    (* proc *)  CreateListBox, HighlightOn, HighlightOff, CursorMovements,
                LBAppend, LBCurrent, LBDeleteCurrent, LBInsertAfter,
                ReplaceCurrent, CursorBackward, WindowOf, LBSort,
                DestroyListBox, DisableScreenOutput;

(************************************************************************)

CONST Nul = CHR(0);  Esc = CHR(1BH);

TYPE
    NameStringIndex = [0..127];
    NameString = ARRAY NameStringIndex OF CHAR;
    CharArrayPointer = POINTER TO ARRAY [0..MAX(CARDINAL) DIV 4] OF CHAR;

VAR
    (* The number of display rows on the screen. *)

    ScreenRows: CARDINAL;

(************************************************************************)
(*                   MOVING DATA TO/FROM THE INI FILE                   *)
(************************************************************************)

PROCEDURE LoadNames (bufptr: CharArrayPointer;  BufferSize: CARDINAL;
                                                LB: ListBox);

    (* Loads a sequence of strings from bufptr^ to LB. *)

    VAR j, k: CARDINAL;
        name: NameString;

    BEGIN
        j := 1;

        (* Each time around this loop we extract one name. *)

        WHILE (j < BufferSize) AND (bufptr^[j] <> Nul) DO
            k := 0;
            REPEAT
                name[k] := bufptr^[j];
                INC (k);  INC (j);
            UNTIL (j >= BufferSize) OR (bufptr^[j-1] = Nul)
                                    OR (k > MAX(NameStringIndex));
            IF name[0] <> "$" THEN
                LBAppend (LB, name);
            END (*IF*);
        END (*WHILE*);
    END LoadNames;

(************************************************************************)

PROCEDURE LoadTheList (LB: ListBox;  VAR (*IN*) name: ARRAY OF CHAR;
                                            VAR (*OUT*) Public: BOOLEAN);

    (* Fills the listbox with the contents of alias list "name". *)

    VAR hini: OS2.HINI;
        bufptr: CharArrayPointer;
        BufferSize: CARDINAL;

    BEGIN
        (* Get the list of users from the INI file. *)

        Public := FALSE;
        hini := OurINIHandle();
        IF (hini <> OS2.NULLHANDLE)
                  AND OS2.PrfQueryProfileSize (hini, "$ALIAS", name, BufferSize)
                  AND (BufferSize > 0) THEN
            ALLOCATE (bufptr, BufferSize);
            OS2.PrfQueryProfileData (hini, "$ALIAS", name, bufptr, BufferSize);
            Public := bufptr^[0] = CHR(1);
            LoadNames (bufptr, BufferSize, LB);
            DEALLOCATE (bufptr, BufferSize);
            LBSort (LB);
        END (*IF*);

    END LoadTheList;

(************************************************************************)

PROCEDURE StoreTheList (LB: ListBox;  VAR (*IN*) name: ARRAY OF CHAR;
                                                       Public: BOOLEAN);

    (* Writes the list of users back to the INI file.  The contents of  *)
    (* LB are destroyed as a side-effect, but this is acceptable; we're *)
    (* going to destroy the list after finishing this job anyway.       *)

    CONST AllocationChunk = 8192;

    VAR hini: OS2.HINI;
        bufptr, auxptr: CharArrayPointer;
        BufferSize, UsedSize, j, k, length: CARDINAL;
        NextName: NameString;

    BEGIN
        DisableScreenOutput (LB);

        (* Make sure we're at the beginning of the listbox. *)

        WHILE CursorBackward(LB) DO
        END (*WHILE*);

        (* Start with a buffer of arbitrary size; we'll expand it later *)
        (* as needed.                                                   *)

        BufferSize := AllocationChunk;
        ALLOCATE (bufptr, AllocationChunk);

        (* The first byte is the public/private indicator. *)

        IF Public THEN
            bufptr^[0] := CHR(1);
        ELSE
            bufptr^[0] := CHR(0);
        END (*IF*);
        UsedSize := 1;  j := 1;

        LOOP
            LBCurrent (LB, NextName);
            IF NextName[0] = Nul THEN
                EXIT (*LOOP*);
            END (*IF*);
            LBDeleteCurrent (LB);
            length := Strings.Length (NextName);

            IF UsedSize + length + 2 > BufferSize THEN
                (* Expand the result buffer. *)
                ALLOCATE (auxptr, BufferSize+AllocationChunk);
                Copy (bufptr, auxptr, UsedSize);
                DEALLOCATE (bufptr, BufferSize);
                INC (BufferSize, AllocationChunk);
                bufptr := auxptr;
            END (*IF*);

            (* Store the current name in the buffer. *)

            FOR k := 0 TO length-1 DO
                bufptr^[j] := NextName[k];  INC (j);
            END (*FOR*);
            bufptr^[j] := Nul;  INC(j);
            INC (UsedSize, length+1);

        END (*LOOP*);

        (* Add a Nul to terminate the list. *)

        bufptr^[j] := Nul;  INC(j);

        (* We've assembled the data, now write it out. *)

        hini := OurINIHandle();
        IF (hini <> OS2.NULLHANDLE) THEN
            OS2.PrfWriteProfileData (hini, "$ALIAS", name, bufptr, j);
        END (*IF*);

        DEALLOCATE (bufptr, BufferSize);

    END StoreTheList;

(************************************************************************)
(*                      THE MAIN LIST EDITOR                            *)
(************************************************************************)

PROCEDURE EditTheList (LB: ListBox;  BoxWidth: CARDINAL): BOOLEAN;

    (* Allows delete/edit/add in this listbox.  The function result is  *)
    (* TRUE if the user moved off the top of the list, and FALSE if     *)
    (* the operation was terminated with the Esc key.                   *)

    VAR w, BottomBar: Window;  ch: CHAR;
        ListEntry: NameString;

    BEGIN
        w := WindowOf(LB);
        OpenWindowHidden (BottomBar, yellow, red, ScreenRows-1, ScreenRows-1, 0, 79, noframe, nodivider);
        MapToVirtualScreen (BottomBar, VirtualScreenOf(w));
        WriteString (BottomBar, " A add  E edit  Del delete  Esc finished");

        LOOP
            IF CursorMovements (LB) THEN
                CloseWindow (BottomBar);  RETURN TRUE;
            END (*IF*);
            ch := GetKey(w);
            IF ch = Nul THEN
                ch := GetKey(w);
                IF ch = 'S' THEN                      (* Del = delete *)
                    LBCurrent (LB, ListEntry);
                    LBDeleteCurrent (LB);
                END (*IF*);
            ELSIF CAP(ch) = 'A' THEN                       (* A = add *)
                ListEntry := "";
                LBInsertAfter (LB, ListEntry);
                EditString (WindowOf(LB), ListEntry,
                         MAX(NameStringIndex)+1, BoxWidth);
                IF EditAborted() OR (ListEntry[0] = Nul) THEN
                    LBDeleteCurrent (LB);
                ELSE
                    ReplaceCurrent (LB, ListEntry);
                END (*IF*);
            ELSIF (CAP(ch) = 'E') OR (ch = CHR(13)) THEN   (* E = edit *)
                LBCurrent (LB, ListEntry);
                IF ListEntry[0] <> Nul THEN
                    EditString (WindowOf(LB), ListEntry,
                            MAX(NameStringIndex)+1, BoxWidth);
                    IF EditAborted() OR (ListEntry[0] = Nul) THEN
                        LBDeleteCurrent (LB);
                    ELSE
                        ReplaceCurrent (LB, ListEntry);
                    END (*IF*);
                END (*IF*);
            ELSIF ch = Esc THEN                            (* Esc = done *)
                CloseWindow (BottomBar);  RETURN FALSE;
            END (*IF*);
        END (*LOOP*);

    END EditTheList;

(************************************************************************)

PROCEDURE EditAliasData (Screen: VirtualScreen;  VAR (*INOUT*) name: ARRAY OF CHAR);

    (* Editor for one alias. *)

    CONST BoxTop = 2;  BoxLeft = 1;  BoxWidth = 76;

    VAR ListWindow, TopBar, BottomBar: Window;  LB: ListBox;
        ch: CHAR;  Public: BOOLEAN;
        BoxHeight: CARDINAL;

    (********************************************************************)

    PROCEDURE ShowPublic;

        (* Writes "Private" or "Public" in the top bar. *)

        BEGIN
            SetCursor (TopBar, 0, 1);
            IF Public THEN
                WriteString (TopBar, " Public");
            ELSE
                WriteString (TopBar, "Private");
            END (*IF*);
        END ShowPublic;

    (********************************************************************)

    BEGIN
        EnableScreenSwitching (FALSE);
        BoxHeight := ScreenRows - BoxTop - 4;

        OpenWindowHidden (TopBar, yellow, red, 0, 0, 0, 79, noframe, nodivider);
        MapToVirtualScreen (TopBar, Screen);
        WriteString (TopBar, "         alias name: ");
        WriteString (TopBar, name);

        OpenWindowHidden (BottomBar, yellow, red, ScreenRows-1, ScreenRows-1, 0, 79, noframe, nodivider);
        MapToVirtualScreen (BottomBar, Screen);
        WriteChar (BottomBar, ' ');
        WriteChar (BottomBar, CHR(27));
        WriteString (BottomBar, " Private    ");
        WriteChar (BottomBar, CHR(26));
        WriteString (BottomBar, " Public    ");
        WriteChar (BottomBar, CHR(25));
        WriteString (BottomBar, " Edit list    Esc finished");

        OpenWindowHidden (ListWindow, black, white, BoxTop, BoxTop+BoxHeight+1,
                      BoxLeft, BoxLeft+BoxWidth+2, noframe, nodivider);
        MapToVirtualScreen (ListWindow, Screen);
        LB := CreateListBox (ListWindow, 1, 1, BoxHeight, BoxWidth);
        LoadTheList (LB, name, Public);

        LOOP
            HighlightOff (LB);
            LOOP
                ShowPublic;
                ColourSwap (TopBar, 0, 1, 8);
                ch := GetKey (TopBar);
                ColourSwap (TopBar, 0, 1, 8);
                IF ch = Esc THEN
                    PutBack(ch);  EXIT (*LOOP*);
                ELSIF ch = CHR(13) THEN
                    Public := NOT Public;
                ELSIF ch = Nul THEN
                    ch := GetKey (TopBar);
                    IF ch = "P" THEN                     (* cursor down *)
                        EXIT (*LOOP*);
                    ELSIF ch = "Q" THEN                  (* page down *)
                        PutBack(ch);  PutBack(Nul);
                        EXIT (*LOOP*);
                    ELSIF ch = "M" THEN                  (* cursor right *)
                        Public := TRUE;
                    ELSIF ch = "K" THEN                  (* cursor left *)
                        Public := FALSE;
                    END (*IF*);
                END (*IF*);
            END (*LOOP*);
            HighlightOn (LB);
            IF NOT EditTheList (LB, BoxWidth) THEN
                EXIT (*LOOP*);
            END (*IF*);
        END (*LOOP*);

        StoreTheList (LB, name, Public);
        DestroyListBox (LB);  CloseWindow (ListWindow);
        CloseWindow (BottomBar);  CloseWindow(TopBar);
        EnableScreenSwitching (TRUE);

    END EditAliasData;

(************************************************************************)

PROCEDURE RemoveAlias (name: ARRAY OF CHAR);

    (* Deletes this alias from the system. *)

    VAR hini: OS2.HINI;

    BEGIN
        hini := OurINIHandle();
        IF hini <> OS2.NULLHANDLE THEN
           OS2.PrfWriteProfileData (hini, "$ALIAS", name, NIL, 0);
        END (*IF*);
    END RemoveAlias;

(********************************************************************************)

PROCEDURE RenameAlias (oldname, newname: ARRAY OF CHAR);

    (* Changes the name of an alias. *)

    VAR w: Window;  LB: ListBox;  Public: BOOLEAN;

    BEGIN
        (* For the sake of code re-use, we use a screen window and a listbox,   *)
        (* even though these will never appear on the screen.  This is not      *)
        (* very efficient in time, but the overhead is not enough to matter.    *)

        OpenWindowHidden (w, black, white, 0, 1, 0, 1, noframe, nodivider);
        LB := CreateListBox (w, 1, 1, 1, 1);
        Public := FALSE;
        IF oldname[0] <> Nul THEN
            LoadTheList (LB, oldname, Public);
            RemoveAlias (oldname);
        END (*IF*);
        IF newname[0] <> Nul THEN
            StoreTheList (LB, newname, Public);
        END (*IF*);
        DestroyListBox (LB);
        CloseWindow (w);
    END RenameAlias;

(********************************************************************************)
(*                              INITIALISATION                                  *)
(********************************************************************************)

VAR dummy: CARDINAL;

BEGIN
    GetScreenSize (ScreenRows, dummy);
END AliasLists.

