IMPLEMENTATION MODULE WSession;

        (********************************************************)
        (*                                                      *)
        (*    Session handler for the Weasel POP/SMTP server    *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            28 April 1998                   *)
        (*  Last edited:        1 June 1999                     *)
        (*  Status:             Complete, I think               *)
        (*                                                      *)
        (********************************************************)

IMPORT Strings, OS2;

FROM SYSTEM IMPORT
    (* type *)  ADDRESS;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

FROM LowLevel IMPORT
    (* proc *)  EVAL;

FROM Conversions IMPORT
    (* proc *)  CardinalToString;

FROM Sockets IMPORT
    (* const*)  AF_INET,
    (* type *)  Socket, SockAddr,
    (* proc *)  send, recv, getsockname, soclose, so_cancel;

FROM NetDB IMPORT
    (* type *)  HostEntPtr,
    (* proc *)  gethostbyaddr, tcp_h_errno;

FROM InetUtilities IMPORT
    (* proc *)  LockScreen, UnlockScreen, IPToString, AddEOL, Synch;

FROM Semaphores IMPORT
    (* type *)  Semaphore,
    (* proc *)  CreateSemaphore, DestroySemaphore, Signal;

FROM Timer IMPORT
    (* proc *)  TimedWait, Sleep;

FROM TaskControl IMPORT
    (* type *)  Lock,
    (* proc *)  CreateLock, Obtain, Release, CreateTask1, TaskExit;

FROM Hosts IMPORT
    (* proc *)  CheckHost;

FROM Names IMPORT
    (* proc *)  CardArray, ServiceType;

IMPORT POPCommands, SMTPCommands;

FROM SMTPData IMPORT
    (* proc *)  StoreVersion;

FROM TransLog IMPORT
    (* type *)  TransactionLogID,
    (* proc *)  CreateLogID, DiscardLogID, LogTransaction;

(************************************************************************)

TYPE
    (* Session state record. *)

    Session = RECORD
                  LogID: TransactionLogID;
                  CASE service: ServiceType OF
                     | SMTP:  SS:  SMTPCommands.Session;
                     | POP:   SP:  POPCommands.Session;
                  END (*CASE*);
              END (*RECORD*);

    (* Data used in creating a new instance of the session handler task. *)

    NewSessionPointer = POINTER TO
                           RECORD
                               service: ServiceType;
                               socket: Socket;
                               IPAddress: CARDINAL;  (* network byte order *)
                           END (*RECORD*);

    (* Data needed by the timeout checker task. *)

    KeepAlivePointer = POINTER TO
                           RECORD
                               SocketOpen, dying: BOOLEAN;
                               sem: Semaphore;
                               service: ServiceType;
                               socket: Socket;
                               TimedOut: BOOLEAN;
                           END (*RECORD*);

    (* Error message type. *)

    StringArray = ARRAY ServiceType OF ARRAY [0..40] OF CHAR;

CONST
    AccessDenied = StringArray {"421 Access denied",
                                "-ERR Access denied"};
    TooManyUsers = StringArray {"421 User limit exceeded, try again later",
                                "-ERR User limit exceeded, try again later"};
    InitialMessage = StringArray {"220 ", "+OK "};

VAR
    (* Maximum allowed number of simultaneous users. *)

    MaxUsers: CardArray;

    (* Count of active users, and a lock to protect it. *)

    UserCount: CardArray;
    UserCountLock: Lock;

    (* Version number. *)

    version: ARRAY [0..15] OF CHAR;

    (* Timeout delay, in milliseconds. *)

    MaxTime: CardArray;

(************************************************************************)

PROCEDURE SetVersion (v: ARRAY OF CHAR);

    (* Stores the version number. *)

    BEGIN
        Strings.Assign (v, version);
        StoreVersion (v);
    END SetVersion;

(************************************************************************)

PROCEDURE SetMaxUsers (limit: CardArray);

    (* Sets an upper limit on the number of simultaneous users. *)

    BEGIN
        MaxUsers := limit;
    END SetMaxUsers;

(************************************************************************)

PROCEDURE SetTimeout (seconds: CardArray);

    (* Specifies how long a session can be idle before it is forcibly   *)
    (* closed.                                                          *)

    VAR j: ServiceType;

    BEGIN
        FOR j := MIN(ServiceType) TO MAX(ServiceType) DO
            IF seconds[j] > MAX(CARDINAL) DIV 1000 THEN
                MaxTime[j] := MAX(CARDINAL);
            ELSE
                MaxTime[j] := 1000*seconds[j];
            END (*IF*);
        END (*FOR*);
    END SetTimeout;

(************************************************************************)

PROCEDURE UpdateCount (service: ServiceType;  increment: INTEGER): CARDINAL;

    (* Updates the count of the number of users, and returns the new    *)
    (* count.  Special case: if this would take us beyond the MaxUsers  *)
    (* limit, then the count is not updated and the returned value      *)
    (* is zero.                                                         *)

    VAR value: CARDINAL;

    BEGIN
        Obtain (UserCountLock);
        IF increment > 0 THEN INC (UserCount[service], increment);
        ELSIF increment < 0 THEN DEC (UserCount[service], -increment)
        END (*IF*);
        value := UserCount[service];
        IF value > MaxUsers[service] THEN
            DEC (UserCount[service], increment);  value := 0;
        END (*IF*);
        Release (UserCountLock);
        RETURN value;
    END UpdateCount;

(************************************************************************)

PROCEDURE NumberOfUsers(): CARDINAL;

    (* Returns the number of users who are currently logged on. *)

    BEGIN
        RETURN UpdateCount (SMTP, 0) + UpdateCount (POP,0);
    END NumberOfUsers;

(************************************************************************)

PROCEDURE TimeoutChecker (arg: ADDRESS);

    (* A new instance of this task is created for each client session.  *)
    (* It kills the corresponding SessionHandler task if more than      *)
    (* MaxTime milliseconds have passed since the last Signal() on the  *)
    (* session's KeepAlive semaphore.                                   *)

    VAR p: KeepAlivePointer;

    BEGIN
        p := arg;
        REPEAT
            TimedWait (p^.sem, MaxTime[p^.service], p^.TimedOut);
        UNTIL p^.TimedOut OR p^.dying;
        IF p^.SocketOpen THEN
            so_cancel (p^.socket);
        END (*IF*);

        (* Wait for the socket to be closed. *)

        WHILE p^.SocketOpen DO
            Sleep (500);
        END (*WHILE*);
        DestroySemaphore (p^.sem);
        DISPOSE (p);

    END TimeoutChecker;

(************************************************************************)

PROCEDURE OpenSession (VAR (*INOUT*) session: Session;  S: Socket;
                                KeepAlive: Semaphore;  MayRelay: BOOLEAN);

    (* Initialise the data structures for a new session. *)

    BEGIN
        WITH session DO
            IF service = SMTP THEN
                SS := SMTPCommands.OpenSession (S, KeepAlive, LogID, MayRelay);
            ELSE
                SP := POPCommands.OpenSession (S, KeepAlive, LogID);
            END (*IF*);
        END (*WITH*);
    END OpenSession;

(************************************************************************)

PROCEDURE CloseSession (session: Session);

    (* End-of-session tidying up. *)

    BEGIN
        WITH session DO
            IF service = SMTP THEN
                SMTPCommands.CloseSession (SS);
            ELSE
                POPCommands.CloseSession (SP);
            END (*IF*);
        END (*WITH*);
    END CloseSession;

(************************************************************************)

PROCEDURE HandleCommand (session: Session;  VAR (*IN*) command: ARRAY OF CHAR;
                                               VAR (*OUT*) Quit: BOOLEAN);

    (* Handles a single command from the client.  Returns with Quit     *)
    (* set to TRUE if this command should end the session.              *)

    BEGIN
        WITH session DO
            IF service = SMTP THEN
                SMTPCommands.HandleCommand (session.SS, command, Quit);
            ELSE
                POPCommands.HandleCommand (session.SP, command, Quit);
            END (*IF*);
        END (*WITH*);
    END HandleCommand;

(************************************************************************)

PROCEDURE SessionHandler (arg: ADDRESS);

    (* The task that handles a client session, i.e. this is where all the real  *)
    (* work is done.  There might be several instances of this task running,    *)
    (* one for each session that is still open.                                 *)

    CONST CR = CHR(13);  LF = CHR(10);

    VAR S: Socket;
        CmdBuffer: ARRAY [0..511] OF CHAR;

        (* Temporary buffer for BuildCommand. *)

        TmpBuffer: ARRAY [0..127] OF CHAR;
        tmppos, tmplength: CARDINAL;  ch: CHAR;

    (********************************************************************)

    PROCEDURE BuildCommand(): BOOLEAN;

        VAR length: CARDINAL;
            IACreceived, IgnoreNext: BOOLEAN;

        BEGIN
            length := 0;
            IACreceived := FALSE;  IgnoreNext := FALSE;
            LOOP
                IF tmppos >= tmplength THEN
                    tmplength := recv (S, TmpBuffer, SIZE(TmpBuffer), 0);
                    IF (tmplength = MAX(CARDINAL)) OR (tmplength = 0) THEN
                        RETURN FALSE;
                    END (*IF*);
                    tmppos := 0;
                END (*IF*);
                ch := TmpBuffer[tmppos];  INC(tmppos);

                (* This next section skips over Telnet control codes (which we  *)
                (* don't really want to know about).  A Telnet control code is  *)
                (* two or three bytes long, where the first byte is CHR(255).   *)

                IF IgnoreNext THEN
                    IgnoreNext := FALSE;
                ELSIF IACreceived THEN
                    IACreceived := FALSE;
                    IF ORD(ch) = 255 THEN
                        IF length < SIZE(CmdBuffer) THEN
                            CmdBuffer[length] := ch;  INC(length);
                        END (*IF*);
                    ELSIF ORD(ch) > 250 THEN
                        IgnoreNext := TRUE;
                    END (*IF*);
                ELSIF ORD(ch) = 255 THEN
                    IACreceived := TRUE;

                (* Command should end with CR LF, but for simplicity we'll      *)
                (* ignore the CR.                                               *)

                ELSIF ch = CR THEN  (* Do nothing *)
                ELSIF ch = LF THEN
                    IF length < SIZE(CmdBuffer) THEN
                        CmdBuffer[length] := CHR(0);
                    END (*IF*);
                    RETURN TRUE;
                ELSIF length < SIZE(CmdBuffer) THEN
                    CmdBuffer[length] := ch;  INC(length);
                END (*IF*);

            END (*LOOP*);

        END BuildCommand;

    (********************************************************************)

    VAR NSP: NewSessionPointer;
        sess: Session;
        size, UserNumber: CARDINAL;
        KA: KeepAlivePointer;
        KeepAliveSemaphore: Semaphore;
        Quit: BOOLEAN;
        ClientIPAddress: CARDINAL;    (* network byte order *)
        LogFilePrefix: ARRAY [0..6] OF CHAR;
        IPBuffer: ARRAY [0..16] OF CHAR;
        LogMessage: ARRAY [0..127] OF CHAR;
        HostName: ARRAY [0..511] OF CHAR;
        ServerName: SockAddr;
        NameDetails: HostEntPtr;
        IsBanned, MayRelay: BOOLEAN;

    BEGIN                   (* Body of SessionHandler *)

        OS2.DosError (OS2.FERR_DISABLEHARDERR);              (* disable hard error popups *)

        (* Copy the NewSessionPointer^ information. *)

        NSP := arg;
        sess.service := NSP^.service;
        S := NSP^.socket;
        ClientIPAddress := NSP^.IPAddress;
        DISPOSE (NSP);

        (* Create the log file ID for this session. *)

        CardinalToString (S, LogFilePrefix, 7);
        IF sess.service = SMTP THEN
            LogFilePrefix[0] := 'S';
        ELSE
            LogFilePrefix[0] := 'P';
        END (*IF*);
        sess.LogID := CreateLogID (LogFilePrefix);

        (* Log the new session commencement. *)

        Strings.Assign ("New client ", LogMessage);
        IPToString (ClientIPAddress, IPBuffer);
        Strings.Append (IPBuffer, LogMessage);
        LogTransaction (sess.LogID, LogMessage);

        (* Work out our host name. *)

        size := SIZE(SockAddr);
        getsockname (S, ServerName, size);
        NameDetails := gethostbyaddr (ServerName.in_addr.addr, SIZE(CARDINAL), AF_INET);
        IF tcp_h_errno() = 0 THEN
            Strings.Assign (NameDetails^.h_name^, HostName);
        ELSE
            HostName := "";
        END (*IF*);

        (* Check whether the client is on one of our special lists. *)

        CheckHost (ClientIPAddress, IsBanned, MayRelay);

        (* Client on blacklist? *)

        IF IsBanned THEN
            LogTransaction (sess.LogID, "Blacklisted client rejected");
            Strings.Assign (AccessDenied[sess.service], CmdBuffer);
            size := AddEOL (CmdBuffer);
            EVAL (send (S, CmdBuffer, size, 0));
            EVAL (soclose (S));
            DiscardLogID (sess.LogID);
            TaskExit;
        END (*IF*);

        (* Check for too many users. *)

        UserNumber := UpdateCount (sess.service, +1);
        IF UserNumber = 0 THEN
            LogTransaction (sess.LogID, "Too many users");
            Strings.Assign (TooManyUsers[sess.service], CmdBuffer);
            size := AddEOL (CmdBuffer);
            EVAL (send (S, CmdBuffer, size, 0));
            EVAL (soclose (S));
            DiscardLogID (sess.LogID);
            TaskExit;
        END (*IF*);

        (* Create the session information structure. *)

        CreateSemaphore (KeepAliveSemaphore, 0);
        OpenSession (sess, S, KeepAliveSemaphore, MayRelay);

        (* Create an instance of the TimeoutChecker task. *)

        NEW (KA);
        WITH KA^ DO
            SocketOpen := TRUE;  socket := S;  dying := FALSE;
            service := sess.service;
            sem := KeepAliveSemaphore;
            TimedOut := FALSE;
        END (*WITH*);
        CreateTask1 (TimeoutChecker, 3, "weasel timeout", KA);

        (* Send the "welcome" message. *)

        Strings.Assign (InitialMessage[sess.service], CmdBuffer);
        Strings.Append (HostName, CmdBuffer);
        Strings.Append (" Weasel ", CmdBuffer);
        Strings.Append (version, CmdBuffer);
        Strings.Append (" ready", CmdBuffer);
        IF sess.service = POP THEN
            Strings.Append (' ', CmdBuffer);
            POPCommands.AppendTimeStamp (sess.SP, CmdBuffer);
        END (*IF*);
        size := AddEOL (CmdBuffer);
        Quit := send (S, CmdBuffer, size, 0) = MAX(CARDINAL);


        (* Here's the main command processing loop.  We leave it when the client  *)
        (* issues a QUIT command, or when socket communications are lost, or      *)
        (* when we get a timeout on the KeepAlive semaphore.                      *)

        tmppos := 0;  tmplength := 0;
        LOOP
            IF Quit THEN EXIT(*LOOP*) END(*IF*);
            IF BuildCommand() THEN
                Signal (KeepAliveSemaphore);
                HandleCommand (sess, CmdBuffer, Quit);
                Synch (S);
            ELSE
                EXIT (*LOOP*);
            END (*IF*);
        END (*LOOP*);

        (* Work out whether the session was terminated by a QUIT, or a timeout, *)
        (* or a communications failure.                                         *)

        IF KA^.TimedOut THEN
            LogTransaction (sess.LogID, "Timed out");
        ELSIF NOT Quit THEN
            LogTransaction (sess.LogID, "Session aborted by client");
        END (*IF*);

        CloseSession (sess);
        KA^.dying := TRUE;  Signal (KA^.sem);
        soclose (S);
        KA^.SocketOpen := FALSE;
        EVAL (UpdateCount (sess.service, -1));
        DiscardLogID (sess.LogID);
        TaskExit;

    (*
    EXCEPT
        LogTransaction (sess.LogID, "SessionHandler detected exception.");
        CloseSession (sess);
        UserNumber := UpdateCount (sess.service, -1);
        soclose(S);
        TaskExit;
    *)

    END SessionHandler;

(********************************************************************************)

PROCEDURE NewSession (serv: ServiceType;  S: Socket;  addr: SockAddr);

    (* Starts and runs a client session.  The session runs in a separate        *)
    (* thread; this procedure returns after starting the session, it does not   *)
    (* wait until the session is over.                                          *)

    VAR NSP: NewSessionPointer;

    BEGIN
        NEW (NSP);
        WITH NSP^ DO
            service := serv;
            socket := S;  IPAddress := addr.in_addr.addr;
        END (*WITH*);
        CreateTask1 (SessionHandler, 3, "mail session", NSP);
    END NewSession;

(********************************************************************************)
(*                            MODULE INITIALISATION                             *)
(********************************************************************************)

BEGIN
    CreateLock (UserCountLock);
    Obtain (UserCountLock);
    UserCount := CardArray {0, 0};
    Release (UserCountLock);
END WSession.

