MODULE Scavenge;

        (********************************************************)
        (*                                                      *)
        (*   Searches the Weasel "forward" directory for lost   *)
        (*        mail, rebuilds the file Delayed.MAI           *)
        (*                                                      *)
        (*  New version: the file Delayed.MAI is no longer      *)
        (*  used by Weasel (the forward mail format was changed *)
        (*  in Weasel version 0.831), but this program          *)
        (*  will pick up the old-format files, including those  *)
        (*  on an existing Delayed.MAI, and convert them to the *)
        (*  new format.                                         *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            9 December 1998                 *)
        (*  Last edited:        2 June 1999                     *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (********************************************************)

FROM SYSTEM IMPORT CAST, ADR, LOC, CARD8, BOOL8;

IMPORT IOChan, ChanConsts, RndFile, FileSys, Strings, OS2;

FROM STextIO IMPORT
    (* proc *)  WriteChar, WriteString, WriteLn;

FROM Names IMPORT
    (* type *)  PathString,
                FilenameIndex, FilenameString;

FROM FDFiles IMPORT
    (* type *)  DirectoryEntry,
    (* proc *)  FirstDirEntry, NextDirEntry, DirSearchDone, FWriteChar;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

(************************************************************************)

CONST
    Nul = CHR(0);  CR = CHR(13);  LF = CHR(10);  Tab = CHR(9);

TYPE
    (* A list of recipients for a single mail item. *)

    AddressList = POINTER TO AddressListEntry;
    AddressListEntry = RECORD
                         next: AddressList;
                         forwardpath: PathString;
                     END (*RECORD*);

    (* A list of mail items waiting to be sent.                         *)
    (*   next             next item on the list                         *)
    (*   sendtime         time that the item is due to be sent          *)
    (*   RetryNumber      number of times this has been tried already   *)
    (*   sender           the MAIL FROM parameter                       *)
    (*   file             name of the message file                      *)
    (*   offset           starting position within the file for the     *)
    (*                     data to be sent to the recipient.  (There    *)
    (*                     could be bookkeeping data before that,       *)
    (*                     which is not to be sent.)                    *)
    (*   sendto           list of recipients.                           *)
    (*   NotifyOnFailure  TRUE if a failure to send this item should    *)
    (*                     cause a notification to be mailed back       *)
    (*                     to the sender.                               *)

    OutJobPtr = POINTER TO OutJob;
    OutJob = RECORD
                 next: OutJobPtr;
                 sendtime: CARDINAL;
                 RetryNumber: CARDINAL;
                 sender: PathString;
                 file: FilenameString;
                 offset: RndFile.FilePos;
                 sendto: AddressList;
                 NotifyOnFailure: BOOLEAN;
             END (*RECORD*);

    (* While relay mail is waiting to be sent it is stored as a file    *)
    (* which starts with the following details.                         *)
    (*   4 bytes   format version, value 'V000'                         *)
    (*   4 bytes   send time                                            *)
    (*   1 byte    retry number                                         *)
    (*   1 byte    Boolean notify-on-failure flag                       *)
    (*   variable  sender (character string)                            *)
    (*   variable  recipient list, bounded by () and comma-separated    *)
    (*                                                                  *)
    (* The message content starts immediately after this.               *)

    PreambleType = RECORD
                       version: ARRAY [0..3] OF CHAR;
                       sendtime: CARDINAL;
                       RetryNumber: CARD8;
                       NotifyOnFailure: BOOL8;
                   END (*RECORD*);

(************************************************************************)

VAR
    (* List of jobs waiting to be reattempted.   *)

    JobList: OutJobPtr;

    (* Name of the directory used for mail to be forwarded. *)

    ForwardDirName: FilenameString;

    (* String used in creating a unique file name. *)

    NextName: ARRAY [0..7] OF CHAR;

(************************************************************************)
(*                      NUMERIC OUTPUT TO SCREEN                        *)
(************************************************************************)

PROCEDURE WriteCard (N: CARDINAL);

    (* Numeric to decimal, left justified. *)

    BEGIN
        IF N > 9 THEN WriteCard (N DIV 10) END(*IF*);
        WriteChar (CHR(ORD('0') + N MOD 10));
    END WriteCard;

(************************************************************************)

PROCEDURE EVAL (f:ARRAY OF LOC);

    (* Function evaluation, throw away the result. *)

    BEGIN
    END EVAL;

(************************************************************************)
(*                    CREATING A UNIQUE FILENAME                        *)
(************************************************************************)

PROCEDURE MakeUniqueName (VAR (*OUT*) name: ARRAY OF CHAR);

    (* Generates a unique 8-character string.  The argument must of     *)
    (* course be big enough to take at least 8 characters.              *)

    (********************************************************************)

    PROCEDURE Increment (N: CARDINAL);

        (* Increments NextName[N], with carry as appropriate. *)

        BEGIN
            IF NextName[N] = '9' THEN
                NextName[N] := 'A';
            ELSIF NextName[N] = 'Z' THEN
                NextName[N] := '0';
                IF N > 0 THEN
                    Increment (N-1);
                END (*IF*);
            ELSE
                INC (NextName[N]);
            END (*IF*);
        END Increment;

    (********************************************************************)

    BEGIN
        Strings.Assign (NextName, name);
        Increment (7);
    END MakeUniqueName;

(************************************************************************)

PROCEDURE MakeNewFilename (BaseName, tail: ARRAY OF CHAR;
                       VAR (*OUT*) NewName: ARRAY OF CHAR);

    (* Creates a file name of the form BaseNamexxxtail, where xxx is    *)
    (* chosen such that a file of that name does not already exist.     *)
    (* Note that BaseName and tail can include punctuation.             *)

    VAR UName: FilenameString;

    BEGIN
        REPEAT
            MakeUniqueName (UName);
            Strings.Assign (BaseName, NewName);
            Strings.Append (UName, NewName);
            Strings.Append (tail, NewName);
        UNTIL NOT FileSys.Exists(NewName);
    END MakeNewFilename;

(************************************************************************)
(*              ADDING A JOB TO THE QUEUE OF OUTBOUND MAIL              *)
(************************************************************************)

PROCEDURE AddToJobList (p: OutJobPtr);

    (* Adds a new job to the list of jobs that have to be retried.      *)
    (* The entries in this list are ordered by time.                    *)

    VAR previous, current: OutJobPtr;

    BEGIN
        p^.next := NIL;
        previous := NIL;  current := JobList;
        WHILE (current <> NIL) AND (p^.sendtime > current^.sendtime) DO
            previous := current;  current := current^.next;
        END (*WHILE*);

        (* The new item has to be inserted between previous and current. *)

        p^.next := current;
        IF previous = NIL THEN
            JobList := p;
        ELSE
            previous^.next := p;
        END (*IF*);

    END AddToJobList;

(************************************************************************)

PROCEDURE NameMatch (VAR (*IN*) name1, name2: FilenameString): BOOLEAN;

    (* Returns TRUE iff the two names are equal, modulo case differences. *)

    VAR j: CARDINAL;

    BEGIN
        j := 0;
        LOOP
            IF j > MAX(FilenameIndex) THEN RETURN TRUE
            ELSIF name1[j] = Nul THEN RETURN name2[j] = Nul
            ELSIF CAP(name1[j]) <> CAP(name2[j]) THEN RETURN FALSE
            ELSE INC(j);
            END (*IF*);
        END (*LOOP*);
    END NameMatch;

(************************************************************************)

PROCEDURE NameOnJoblist (name: FilenameString): BOOLEAN;

    (* Returns TRUE iff this file is already on the master list. *)

    VAR current: OutJobPtr;

    BEGIN
        current := JobList;
        LOOP
            IF current = NIL THEN RETURN FALSE
            ELSIF NameMatch (current^.file, name) THEN RETURN TRUE
            ELSE current := current^.next
            END (*IF*);
        END (*LOOP*);
    END NameOnJoblist;

(************************************************************************)
(*                  DEALING WITH LISTS OF RECIPIENTS                    *)
(************************************************************************)

PROCEDURE DumpAddressList (RL: AddressList);

    (* Writes the addresses to the screen. *)

    BEGIN
        WHILE RL <> NIL DO
            WriteString (RL^.forwardpath);  WriteLn;
            RL := RL^.next;
        END (*WHILE*);
    END DumpAddressList;

(************************************************************************)

PROCEDURE DiscardAddressList (VAR (*INOUT*) RL: AddressList);

    (* Disposes of the data structure. *)

    VAR next: AddressList;

    BEGIN
        WHILE RL <> NIL DO
            next := RL^.next;
            DISPOSE (RL);  RL := next;
        END (*WHILE*);
    END DiscardAddressList;

(************************************************************************)
(*                    FILE INPUT/OUTPUT UTILITIES                       *)
(************************************************************************)

PROCEDURE FReadChar (cid: IOChan.ChanId;  VAR (*OUT*) ch: CHAR): BOOLEAN;

    (* Reads a single character from a file. *)

    VAR NumberRead: CARDINAL;

    BEGIN
        IOChan.RawRead (cid, ADR(ch), 1, NumberRead);
        IF NumberRead = 1 THEN
            RETURN TRUE;
        ELSE
            ch := Nul;  RETURN FALSE;
        END (*IF*);
    END FReadChar;

(************************************************************************)

TYPE CharSet = SET OF CHAR;
CONST Digits = CharSet {'0'..'9'};

(************************************************************************)

PROCEDURE SkipBlanks (cid: IOChan.ChanId;
                                   VAR (*INOUT*) NextChar: CHAR): BOOLEAN;

    (* Sets NextChar to the next nonblank character in the file. *)

    VAR success: BOOLEAN;

    BEGIN
        success := TRUE;
        WHILE success AND ((NextChar = ' ') OR (NextChar = Tab)) DO
            success := FReadChar (cid, NextChar);
        END (*WHILE*);
        RETURN success;
    END SkipBlanks;

(************************************************************************)

PROCEDURE SkipBlanksAndEOL (cid: IOChan.ChanId;
                                   VAR (*INOUT*) NextChar: CHAR): BOOLEAN;

    (* Like SkipBlanks, but also skips CR and LF. *)

    VAR success: BOOLEAN;

    BEGIN
        success := TRUE;
        WHILE success AND (NextChar IN CharSet{' ', Tab, CR, LF}) DO
            success := FReadChar (cid, NextChar);
        END (*WHILE*);
        RETURN success;
    END SkipBlanksAndEOL;

(************************************************************************)

PROCEDURE SkipToNewLine (cid: IOChan.ChanId;
                          VAR (*INOUT*) NextChar: CHAR): BOOLEAN;

    (* Scans until NextChar has moved beyond the next end-of-line. *)
    (* Returns FALSE if file input fails before we reach the EOL.  *)

    VAR success: BOOLEAN;

    BEGIN
        success := TRUE;
        WHILE success AND NOT (NextChar IN CharSet{CR,LF}) DO
            success := FReadChar (cid, NextChar);
        END (*WHILE*);
        IF success AND (NextChar = CR) THEN
            success := FReadChar (cid, NextChar);
        END (*IF*);
        IF success AND (NextChar = LF) THEN
            success := FReadChar (cid, NextChar);
        END (*IF*);
        RETURN success;
    END SkipToNewLine;

(************************************************************************)

PROCEDURE HexToBin (ch: CHAR): CARDINAL;

    (* Converts one hexadecimal character. *)

    BEGIN
        IF ch IN CharSet {'0'..'9'} THEN
            RETURN ORD(ch) - ORD('0');
        ELSE
            RETURN ORD(CAP(ch)) - ORD('A') + 10;
        END (*IF*);
    END HexToBin;

(************************************************************************)

PROCEDURE FReadHexByte (cid: IOChan.ChanId;  VAR (*OUT*) value: CARDINAL;
                                   VAR (*INOUT*) NextChar: CHAR): BOOLEAN;

    (* Reads a two-digit hexadecimal number from the file. *)

    VAR success: BOOLEAN;

    BEGIN
        value := 16*HexToBin (NextChar);
        success := FReadChar (cid, NextChar);
        IF success THEN
            INC (value, HexToBin(NextChar));
            success := FReadChar (cid, NextChar);
        END (*IF*);
        RETURN success;
    END FReadHexByte;

(************************************************************************)

PROCEDURE FReadCard (cid: IOChan.ChanId;  VAR (*OUT*) value: CARDINAL;
                                   VAR (*INOUT*) NextChar: CHAR): BOOLEAN;

    (* Reads a decimal number from the file. *)

    VAR success: BOOLEAN;

    BEGIN
        success := TRUE;  value := 0;
        WHILE success AND (NextChar IN Digits) DO
            value := 10*value + (ORD(NextChar) - ORD('0'));
            success := FReadChar (cid, NextChar);
        END (*WHILE*);
        RETURN success;
    END FReadCard;

(************************************************************************)

PROCEDURE FReadString (cid: IOChan.ChanId;
                                VAR (*OUT*) string: ARRAY OF CHAR;
                                Stoppers: CharSet;
                                VAR (*INOUT*) NextChar: CHAR): BOOLEAN;

    (* Reads a string from a file, stopping when NextChar is one of the *)
    (* characters in Stoppers, or when we reach the end of the file.    *)

    VAR j: CARDINAL;  success: BOOLEAN;

    BEGIN
        success := TRUE;  j := 0;
        WHILE success AND NOT (NextChar IN Stoppers) DO
            string[j] := NextChar;  INC(j);
            success := FReadChar (cid, NextChar);
        END (*WHILE*);
        string[j] := Nul;
        RETURN success;
    END FReadString;

(************************************************************************)
(*                PARSING A LIST OF ADDRESSES IN A FILE                 *)
(************************************************************************)

PROCEDURE ParseOneAddress (cid: IOChan.ChanId;
                             VAR (*INOUT*) NextChar: CHAR): AddressList;

    VAR result: AddressList;  success, AngleBracket: BOOLEAN;
        temp: PathString;

    BEGIN
        NEW(result);  result^.next := NIL;

        (* Skip leading blanks, possibly going on to a continuation line. *)

        success := SkipBlanksAndEOL (cid, NextChar);

        (* Skip quoted string if present. *)

        IF success AND (NextChar = '"') THEN
            REPEAT
                success := success AND FReadChar (cid, NextChar);
            UNTIL NOT success OR (NextChar = '"');
            success := success AND FReadChar (cid, NextChar);
        END (*IF*);
        success := success AND SkipBlanks (cid, NextChar);

        (* Now the string we expect is either the address, or something *)
        (* of the form Comment<address>.                                *)

        AngleBracket := NextChar = '<';
        result^.forwardpath := '<';
        IF AngleBracket THEN
            success := success AND FReadChar (cid, NextChar);
        END (*IF*);

        success := success AND FReadString (cid, temp,
                            CharSet{CR, LF, ',', '<', '>'}, NextChar);
        IF NextChar = '<' THEN
            AngleBracket := TRUE;
            success := success AND FReadChar (cid, NextChar)
                         AND FReadString (cid, temp,
                                CharSet {CR, LF, ',', '<', '>'}, NextChar);
        END (*IF*);
        Strings.Append (temp, result^.forwardpath);
        Strings.Append ('>', result^.forwardpath);
        IF NextChar = '>' THEN
            success := success AND AngleBracket AND FReadChar (cid, NextChar);
        ELSE
            success := success AND NOT AngleBracket;
        END (*IF*);

        IF success THEN
            IF NextChar <> ',' THEN
                EVAL(SkipToNewLine (cid, NextChar));
            END (*IF*);
        ELSE
            DISPOSE (result);
        END (*IF*);
        RETURN result;

    END ParseOneAddress;

(************************************************************************)

PROCEDURE ParseAddressList (cid: IOChan.ChanId;
                             VAR (*INOUT*) NextChar: CHAR): AddressList;

    VAR tail, result, p: AddressList;  success: BOOLEAN;

    BEGIN
        tail := NIL;  result := NIL;
        LOOP
            p := ParseOneAddress (cid, NextChar);
            IF p = NIL THEN EXIT(*LOOP*) END(*IF*);
            IF tail = NIL THEN result := p
            ELSE tail^.next := p;
            END (*IF*);
            p^.next := NIL;  tail := p;
            success := SkipBlanks (cid, NextChar)
                       AND (NextChar = ',') AND FReadChar (cid, NextChar);
            IF NOT success THEN
                EXIT (*LOOP*);
            END (*IF*);
        END (*LOOP*);
        RETURN result;
    END ParseAddressList;

(************************************************************************)
(*                      SCANNING FOR LOST MESSAGES                      *)
(************************************************************************)

PROCEDURE HeaderMatch (cid: IOChan.ChanId;  Pattern: ARRAY OF CHAR;
                                   VAR (*INOUT*) NextChar: CHAR): BOOLEAN;

    (* Tries to match a character string at the current position in     *)
    (* the file.  We don't attempt to backtrack on partial success,     *)
    (* because the caller isn't that fussy.                             *)

    VAR j: CARDINAL;   success: BOOLEAN;

    BEGIN
        j := 0;  success := TRUE;
        LOOP
            IF NOT success THEN RETURN FALSE
            ELSIF j > HIGH(Pattern) THEN RETURN TRUE
            ELSIF Pattern[j] = Nul THEN RETURN TRUE
            ELSIF Pattern[j] <> NextChar THEN RETURN FALSE
            ELSE
                INC(j);  success := FReadChar (cid, NextChar);
            END (*IF*);
        END (*LOOP*);
    END HeaderMatch;

(************************************************************************)

PROCEDURE BuildNewJob (name: FilenameString): BOOLEAN;

    (* Extracts the "from" and "to" information from the header of this *)
    (* file, and adds a new job to the master list.                     *)

    VAR cid: IOChan.ChanId;  result: RndFile.OpenResults;
        NextChar: CHAR;  success: BOOLEAN;
        Job: OutJobPtr;  AL: AddressList;

    BEGIN
        NEW (Job);
        WITH Job^ DO
            next := NIL;
            sendtime := 0;
            RetryNumber := 0;
            sender := "<>";
            file := name;
            sendto := NIL;
            NotifyOnFailure := TRUE;
        END (*WITH*);
        WriteString ("Attempting to recover ");
        WriteString (name);  WriteLn;
        RndFile.OpenOld (cid, name,
                          ChanConsts.read+ChanConsts.raw, result);
        IF result = ChanConsts.opened THEN
            Job^.offset := RndFile.StartPos (cid);
            NextChar := ' ';
            success := FReadChar (cid, NextChar);
            LOOP
                IF NOT success OR (NextChar = CR) THEN EXIT(*LOOP*)
                ELSIF HeaderMatch (cid, "From:", NextChar) THEN
                    WriteString ("From: ");
                    AL := ParseAddressList (cid, NextChar);
                    DumpAddressList (AL);
                    Job^.sender := AL^.forwardpath;
                    DiscardAddressList(AL);
                ELSIF HeaderMatch (cid, "To:", NextChar) THEN
                    WriteString ("To: ");
                    Job^.sendto := ParseAddressList (cid, NextChar);
                    DumpAddressList (Job^.sendto);
                ELSE success := SkipToNewLine (cid, NextChar)
                END (*IF*);
            END (*LOOP*);
            RndFile.Close (cid);
        END (*IF*);
        IF Job^.sendto <> NIL THEN
            AddToJobList (Job);
            RETURN TRUE;
        ELSE
            WriteString ("No recipients found.");  WriteLn;
            RETURN FALSE;
        END (*IF*);
    END BuildNewJob;

(************************************************************************)

PROCEDURE FindLostMessages;

    (* This is the important part of the program.  We check the         *)
    (* forward directory, and see whether there are any files that are  *)
    (* not on our list of mail to be sent.  For each such file, we try  *)
    (* to get sender and recipient information, and if possible we add  *)
    (* it to the list of mail to be sent.                               *)

    VAR SearchString, FileName: FilenameString;  MoreToGo: BOOLEAN;
        D: DirectoryEntry;  NumberRecovered: CARDINAL;

    BEGIN
        NumberRecovered := 0;
        SearchString := ForwardDirName;
        Strings.Append ("*.MSG", SearchString);
        MoreToGo := FirstDirEntry (SearchString, FALSE, D);
        WHILE MoreToGo DO

            Strings.Assign (ForwardDirName, FileName);
            Strings.Append (D.name, FileName);
            IF NOT NameOnJoblist (FileName) THEN
                IF BuildNewJob (FileName) THEN
                    INC(NumberRecovered);
                END (*IF*);
            END (*IF*);
            MoreToGo := NextDirEntry (D);

        END (*WHILE*);
        DirSearchDone (D);
        WriteCard (NumberRecovered);
        WriteString (" messages recovered.");  WriteLn;
    END FindLostMessages;

(************************************************************************)
(*                  MOVING UNSENT MAIL TO AND FROM DISK                 *)
(************************************************************************)

PROCEDURE AppendBody (srcname: ARRAY OF CHAR;  offset: RndFile.FilePos;
                      dstcid: IOChan.ChanId);

    (* Copies from a source file, starting at position "offset" and     *)
    (* continuing to the end of the file, to the destination.  The      *)
    (* caller should have already opened the destination file, but      *)
    (* not the source file.                                             *)

    VAR srccid: IOChan.ChanId;  NumberRead: CARDINAL;
        status: ChanConsts.OpenResults;
        buffer: ARRAY [0..2047] OF CHAR;

    BEGIN
        (* Open the source file and position it at the right place. *)

        RndFile.OpenOld (srccid, srcname,
                ChanConsts.read+ChanConsts.raw, status);
        RndFile.SetPos (srccid, offset);

        (* Copy across the message body. *)

        LOOP
            IOChan.RawRead (srccid, ADR(buffer), SIZE(buffer), NumberRead);
            IF NumberRead = 0 THEN EXIT(*LOOP*) END(*IF*);
            IOChan.RawWrite (dstcid, ADR(buffer), NumberRead);
        END (*LOOP*);

        RndFile.Close (srccid);

    END AppendBody;

(************************************************************************)

PROCEDURE LoadAddressList (cid: IOChan.ChanId;  VAR (*OUT*) RL: AddressList;
                                   VAR (*INOUT*) NextChar: CHAR): BOOLEAN;

    (* Reads a relay list from a file. *)

    VAR previous, current: AddressList;  success: BOOLEAN;

    BEGIN
        RL := NIL;  previous := NIL;  success := TRUE;
        IF NextChar = '(' THEN
            success := FReadChar (cid, NextChar);
            WHILE success AND (NextChar <> ')') DO
                NEW (current);
                success := FReadString (cid, current^.forwardpath,
                                        CharSet{' ', ',', ')'}, NextChar)
                            AND SkipBlanks (cid, NextChar);
                IF success AND (NextChar = ',') THEN
                    success := FReadChar (cid, NextChar);
                END (*IF*);
                current^.next := NIL;
                IF previous = NIL THEN RL := current
                ELSE previous^.next := current
                END (*IF*);
                previous := current;
            END (*WHILE*);
        END (*IF*);

        (* Consume the final ')'. *)

        IF success THEN
            EVAL (FReadChar (cid, NextChar));
        END (*IF*);

        RETURN success;

    END LoadAddressList;

(************************************************************************)

PROCEDURE LoadOffset (cid: IOChan.ChanId;
                      VAR (*OUT*) offset: RndFile.FilePos;
                      VAR (*INOUT*) NextChar: CHAR): BOOLEAN;

    (* Reads a file position in hexadecimal. *)

    CONST HexDigits = CharSet {'0'..'9', 'A'..'F'};

    VAR j: [1..RndFile.FilePosSize];  value: CARDINAL;

    BEGIN
        FOR j := 1 TO RndFile.FilePosSize DO
            IF NextChar IN HexDigits THEN
                EVAL (FReadHexByte (cid, value, NextChar));
            ELSE
                value := 0;
            END (*IF*);
            offset[j] := CAST(LOC, value);
        END (*FOR*);
        RETURN TRUE;
    END LoadOffset;

(************************************************************************)

PROCEDURE LoadJob (cid: IOChan.ChanId;
                            VAR (*INOUT*) NextChar: CHAR): OutJobPtr;

    (* Loads a previously saved mail job from a disk file.  Returns     *)
    (* NIL if there's nothing left to load.                             *)

    VAR p: OutJobPtr;  boolcode: CARDINAL;

    BEGIN
        p := NIL;  boolcode := 0;
        EVAL (SkipBlanksAndEOL (cid, NextChar));
        IF NextChar IN Digits THEN
            NEW (p);
            IF FReadCard (cid, p^.sendtime, NextChar)
                        AND SkipBlanks (cid, NextChar)
                        AND FReadCard (cid, p^.RetryNumber, NextChar)
                        AND SkipBlanks (cid, NextChar)
                        AND FReadString (cid, p^.file, CharSet{' '}, NextChar)
                        AND SkipBlanks (cid, NextChar)
                        AND FReadCard (cid, boolcode, NextChar)
                        AND SkipBlanks (cid, NextChar)
                        AND LoadAddressList (cid, p^.sendto, NextChar)
                        AND SkipBlanks (cid, NextChar)
                        AND FReadString (cid, p^.sender, CharSet{' ', CR}, NextChar)
                        AND SkipBlanks (cid, NextChar)
                        AND LoadOffset (cid, p^.offset, NextChar)
            THEN
                p^.NotifyOnFailure := CAST (BOOLEAN, boolcode);
            ELSE
                DISPOSE (p);
            END (*IF*);
        END (*IF*);
        RETURN p;
    END LoadJob;

(************************************************************************)
(*            STORING A MESSAGE FILE IN THE FORWARD DIRECTORY           *)
(************************************************************************)

PROCEDURE WriteNameList (fid: IOChan.ChanId;  list: AddressList);

    (* Writes a comma-separated list of names, enclosed in parentheses. *)

    BEGIN
        FWriteChar (fid, '(');
        WHILE list <> NIL DO
            IOChan.RawWrite (fid, ADR(list^.forwardpath),
                                        LENGTH(list^.forwardpath));
            list := list^.next;
            IF list <> NIL THEN
                FWriteChar (fid, ',');
            END (*IF*);
        END (*WHILE*);
        FWriteChar (fid, ')');
    END WriteNameList;

(************************************************************************)

PROCEDURE StoreMessageFile (VAR (*INOUT*) p: OutJobPtr);

    (* On entry p^ should have all fields filled in.  This procedure    *)
    (* creates a new file (whose name is returned in p^.file) with a    *)
    (* preamble reflecting the details in p^ and with a body being the  *)
    (* body of the original p^.file.  The preamble details are given in *)
    (* the comments near the beginning of this module.                  *)
    (* Important assumption: p^.sendto is a non-empty list.             *)

    VAR oldfilename: FilenameString;
        newfid: IOChan.ChanId;  status: ChanConsts.OpenResults;
        preamble: PreambleType;
        oldoffset: RndFile.FilePos;

    BEGIN
        oldfilename := p^.file;

        (* Create a new file name. *)

        MakeNewFilename (ForwardDirName, ".FWD", p^.file);
        RndFile.OpenClean (newfid, p^.file,
                ChanConsts.write+ChanConsts.raw, status);

        (* Write the preamble to the new file. *)

        WITH preamble DO
            version := "V000";
            sendtime := p^.sendtime;
            RetryNumber := p^.RetryNumber;
            NotifyOnFailure := p^.NotifyOnFailure;
        END (*WITH*);
        IOChan.RawWrite (newfid, ADR(preamble), SIZE(preamble));
        IOChan.RawWrite (newfid, ADR(p^.sender), LENGTH(p^.sender));
        WriteNameList (newfid, p^.sendto);

        (* Account for "offset" of both old and new files. *)

        oldoffset := p^.offset;
        p^.offset := RndFile.CurrentPos (newfid);

        (* Copy across the message body. *)

        AppendBody (oldfilename, oldoffset, newfid);

        RndFile.Close (newfid);

    END StoreMessageFile;

(************************************************************************)

PROCEDURE SaveInNewFormat;

    (* For each job on the JobList, writes out a new-format file and    *)
    (* deletes the old file.                                            *)

    VAR p, q: OutJobPtr;
        OriginalFileName: FilenameString;

    BEGIN
        p := JobList;
        WHILE p <> NIL DO
            OriginalFileName := p^.file;
            StoreMessageFile (p);
            DiscardAddressList (p^.sendto);
            OS2.DosDelete (OriginalFileName);
            q := p;  p := p^.next;
            DISPOSE(q);
        END (*WHILE*);
    END SaveInNewFormat;

(************************************************************************)

PROCEDURE LoadUnsentMail;

    (* Loads the queue of outbound mail from a disk file.  This queue   *)
    (* results from a shutdown and subsequent restart, so we give at    *)
    (* least one extra try for mail in the "retry" category.            *)

    VAR cid: IOChan.ChanId;  result: RndFile.OpenResults;
        p: OutJobPtr;  NextChar: CHAR;
        SaveFileName: FilenameString;

    BEGIN
        SaveFileName := ForwardDirName;
        Strings.Append ("Delayed.MAI", SaveFileName);
        RndFile.OpenOld (cid, SaveFileName,
                          ChanConsts.read+ChanConsts.raw, result);
        IF result = ChanConsts.opened THEN
            NextChar := ' ';
            LOOP
                p := LoadJob (cid, NextChar);
                IF p = NIL THEN EXIT(*LOOP*)
                ELSE
                    AddToJobList (p);
                END (*IF*);
            END (*LOOP*);
            RndFile.Close (cid);
        END (*IF*);
        OS2.DosDelete (SaveFileName);
    END LoadUnsentMail;

(************************************************************************)
(*                           INITIALISATION                             *)
(************************************************************************)

PROCEDURE INIGetString (hini: OS2.HINI;  name1, name2: ARRAY OF CHAR;
                                    VAR (*OUT*) variable: ARRAY OF CHAR): BOOLEAN;

    (* Like INIGet, but we accept any size data that will fit in the variable,  *)
    (* and we add a Nul terminator in the case of a size mismatch.              *)

    VAR size: CARDINAL;

    BEGIN
        OS2.PrfQueryProfileSize (hini, name1, name2, size);
        IF size <= HIGH(variable)+1 THEN
            OS2.PrfQueryProfileData (hini, name1, name2, ADR(variable), size);
            IF size <= HIGH(variable) THEN
                variable[size] := Nul;
            END (*IF*);
            RETURN TRUE;
        ELSE
            RETURN FALSE;
        END (*IF*);
    END INIGetString;

(********************************************************************************)

PROCEDURE LoadINIData;

    (* Loads the value of ForwardDirName.                 *)

    VAR hab: OS2.HAB;  hini: OS2.HINI;

    BEGIN
        hab := OS2.WinInitialize (0);
        hini := OS2.PrfOpenProfile (hab, "weasel.ini");
        IF hini <> OS2.NULLHANDLE THEN
            IF NOT INIGetString (hini, "$SYS", "MailRoot",
                                       ForwardDirName) THEN
                ForwardDirName := "/MPTN/ETC/MAIL/";
            END (*IF*);
            OS2.PrfCloseProfile (hini);
        END (*IF*);
        Strings.Append ("Forward/", ForwardDirName);
        IF hab <> OS2.NULLHANDLE THEN
            OS2.WinTerminate (hab);
        END (*IF*);
    END LoadINIData;

(************************************************************************)

PROCEDURE RebuildDelayed;

    BEGIN
        WriteString ("Scavenge version 1999-06-02");  WriteLn;
        LoadINIData;
        JobList := NIL;
        LoadUnsentMail;
        FindLostMessages;
        SaveInNewFormat;
    END RebuildDelayed;

(************************************************************************)

BEGIN
    NextName := "00000000";
    RebuildDelayed;
END Scavenge.

