IMPLEMENTATION MODULE SBuffers;

        (********************************************************)
        (*                                                      *)
        (*       Buffers for line-oriented socket input         *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            24 May 1998                     *)
        (*  Last edited:        8 October 1999                  *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (********************************************************)

FROM Sockets IMPORT
    (* const*)  NotASocket,
    (* type *)  Socket,
    (* proc *)  send, recv, soclose;

FROM LowLevel IMPORT
    (* proc *)  EVAL;

IMPORT InetUtilities, Strings;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

(************************************************************************)

CONST Nul = CHR(0); CR = CHR(13); LF = CHR(10);

TYPE
    ResponseBufferSubscript = [0..127];
    LineBufferSubscript = [0..1024];

    (* ResponseBuffer is a buffer to hold incoming data.  RBpos is the  *)
    (* character position we're up to in ResponseBuffer,                *)
    (* and RBlength is the number of characters in ResponseBuffer.      *)
    (* LineBuffer is a copy of the incoming line.                       *)

    SBuffer = POINTER TO RECORD
                             socket: Socket;
                             ResponseBuffer:
                                    ARRAY ResponseBufferSubscript OF CHAR;
                             RBpos, RBlength: CARDINAL;
                             LineBuffer:
                                    ARRAY LineBufferSubscript OF CHAR;
                         END (*RECORD*);

(************************************************************************)
(*                CREATING AND DESTROYING AN SBuffer                    *)
(************************************************************************)

PROCEDURE CreateSBuffer (s: Socket): SBuffer;

    (* Creates a new SBuffer.  We assume that the connection on socket  *)
    (* s has already been established by the caller.                    *)

    VAR result: SBuffer;

    BEGIN
        NEW (result);
        WITH result^ DO
            socket := s;
            RBpos := 0;  RBlength := 0;
        END (*WITH*);
        RETURN result;
    END CreateSBuffer;

(************************************************************************)

PROCEDURE CloseSBuffer (VAR (*INOUT*) SB: SBuffer);

    (* Releases the buffer space, closes the socket. *)

    BEGIN
        IF SB <> NIL THEN
            IF SB^.socket <> NotASocket THEN
                EVAL (soclose(SB^.socket));
            END (*IF*);
            DISPOSE (SB);
        END (*IF*);
    END CloseSBuffer;

(************************************************************************)
(*                          SOCKET OUTPUT                               *)
(************************************************************************)

PROCEDURE SendLine (SB: SBuffer;  line: ARRAY OF CHAR): BOOLEAN;

    (* Sends the string, appending a CRLF. *)

    VAR buffer: ARRAY LineBufferSubscript OF CHAR;
        length: CARDINAL;

    BEGIN
        Strings.Assign (line, buffer);
        length := InetUtilities.AddEOL (buffer);
        RETURN (SB <> NIL)
              AND (send(SB^.socket, buffer, length, 0) <> MAX(CARDINAL));
    END SendLine;

(************************************************************************)

PROCEDURE SendString (SB: SBuffer;  line: ARRAY OF CHAR): BOOLEAN;

    (* Sends the string, without appending a CRLF. *)

    BEGIN
        RETURN (SB <> NIL)
           AND (send(SB^.socket, line, LENGTH(line), 0) <> MAX(CARDINAL));
    END SendString;

(************************************************************************)

PROCEDURE SendChar (SB: SBuffer;  ch: CHAR): BOOLEAN;

    (* Sends a single character. *)

    VAR buffer: ARRAY [0..0] OF CHAR;

    BEGIN
        buffer[0] := ch;
        RETURN (SB <> NIL)
               AND (send (SB^.socket, buffer, 1, 0) <> MAX(CARDINAL));
    END SendChar;

(************************************************************************)

PROCEDURE SendEOL (SB: SBuffer): BOOLEAN;

    (* Sends a CRLF. *)

    VAR CRLF: ARRAY [0..1] OF CHAR;

    BEGIN
        CRLF[0] := CR;  CRLF[1] := LF;
        RETURN (SB <> NIL)
                AND (send (SB^.socket, CRLF, 2, 0) <> MAX(CARDINAL));
    END SendEOL;

(************************************************************************)
(*                          SOCKET INPUT                                *)
(************************************************************************)

PROCEDURE Getch (SB: SBuffer): CHAR;

    (* Result is Nul if connection fails. *)
    (* Assumption: SB <> NIL. *)

    CONST timeout = 75*1000;      (* milliseconds *)

    VAR result: CHAR;  EmptyResponseCount: CARDINAL;

    BEGIN
        WITH SB^ DO
            IF RBpos >= RBlength THEN
                EmptyResponseCount := 0;
                REPEAT
                    IF InetUtilities.WaitForSocket (socket, timeout) > 0 THEN
                        RBlength := recv (socket, ResponseBuffer,
                                      MAX(ResponseBufferSubscript) + 1, 0);
                    ELSE
                        RBlength := MAX(CARDINAL);
                    END (*IF*);
                    IF RBlength = 0 THEN
                        INC (EmptyResponseCount);
                        IF EmptyResponseCount > 20 THEN
                            RBlength := MAX(CARDINAL);
                        END (*IF*);
                    END (*IF*);
                UNTIL RBlength <> 0;
                IF RBlength = MAX(CARDINAL) THEN
                    RBpos := 0;  RBlength := 0;
                    RETURN Nul;
                END (*IF*);
                RBpos := 0;
            END (*IF*);
            result := ResponseBuffer[RBpos];  INC(RBpos);
        END (*WITH*);
        RETURN result;
    END Getch;

(************************************************************************)

PROCEDURE GetLine (SB: SBuffer): BOOLEAN;

    (* Receives a single line of text from the server, storing it in    *)
    (* SB^.LineBuffer.  A function return of FALSE meant that the       *)
    (* connection failed.                                               *)
    (* Assumption: SB <> NIL. *)

    VAR j: CARDINAL;  ch: CHAR;

    BEGIN
        j := 0;
        LOOP
            ch := Getch(SB);
            IF ch = Nul THEN RETURN FALSE;
            ELSIF ch = LF THEN EXIT (*LOOP*);
            ELSIF ch <> CR THEN
                IF j <= MAX(LineBufferSubscript) THEN
                    SB^.LineBuffer[j] := ch;  INC(j);
                END (*IF*);
            END (*IF*);
        END (*LOOP*);
        IF j <= MAX(LineBufferSubscript) THEN
            SB^.LineBuffer[j] := CHR(0);
        END (*IF*);
        RETURN TRUE;
    END GetLine;

(************************************************************************)

PROCEDURE GetResponse (SB: SBuffer;  VAR (*OUT*) MoreToCome: BOOLEAN): BOOLEAN;

    (* Returns one line of the server response to a command.            *)
    (* MoreToCome is set if this is part of a multi-line response, and  *)
    (* it's not the last line.   The function result is FALSE if we've  *)
    (* lost the connection.                                             *)
    (* Assumption: SB <> NIL. *)

    TYPE CharSet = SET OF CHAR;

    CONST Digits = CharSet {'0'..'9'};

    VAR status: BOOLEAN;

    BEGIN
        status := GetLine (SB);
        MoreToCome := NOT(SB^.LineBuffer[0] IN Digits)
                         OR
                         ((SB^.LineBuffer[1] IN Digits)
                          AND (SB^.LineBuffer[2] IN Digits)
                          AND (SB^.LineBuffer[3] = '-'));
        RETURN status;
    END GetResponse;

(************************************************************************)

PROCEDURE ResponseCode (SB: SBuffer): CARDINAL;

    (* Receives a (possibly multi-line) response from the server, and   *)
    (* returns the first digit of the numeric code.  The values are:    *)
    (*      0  Connection lost                                          *)
    (*      1  OK, another reply still to come                          *)
    (*      2  OK, command done                                         *)
    (*      3  OK, another command expected                             *)
    (*      4  Transient failure, try again later                       *)
    (*      5  Definite failure                                         *)
    (*      6  Reply code is not numeric                                *)
    (*      7  Connection lost or SB=NIL                                *)

    TYPE CharSet = SET OF CHAR;
    CONST Digits = CharSet {'0'..'9'};

    VAR active, MoreToCome: BOOLEAN;

    BEGIN
        IF SB = NIL THEN
            RETURN 7;
        END (*IF*);
        REPEAT
            active := GetResponse (SB, MoreToCome);
        UNTIL NOT (MoreToCome AND active);
        IF active THEN
            IF SB^.LineBuffer[0] IN Digits THEN
                RETURN ORD(SB^.LineBuffer[0]) - ORD('0');
            ELSE
                RETURN 6;
            END (*IF*);
        ELSE
            RETURN 7;
        END (*IF*);
    END ResponseCode;

(************************************************************************)

PROCEDURE PositiveResponse (SB: SBuffer;
                            VAR (*OUT*) LostConnection: BOOLEAN): BOOLEAN;

    (* Returns TRUE if a positive response was returned.  *)

    TYPE ReplySet = SET OF [0..7];

    VAR code: [0..7];

    BEGIN
        code := ResponseCode(SB);
        LostConnection := code IN ReplySet{0, 7};
        RETURN code IN ReplySet{1..3};
    END PositiveResponse;

(************************************************************************)

PROCEDURE GetLastLine (SB: SBuffer;  VAR (*OUT*) line: ARRAY OF CHAR);

    (* Returns a copy of the last line received. *)

    BEGIN
        IF SB = NIL THEN
            line[0] := Nul;
        ELSE
            Strings.Assign (SB^.LineBuffer, line);
        END (*IF*);
    END GetLastLine;

(************************************************************************)

END SBuffers.

