IMPLEMENTATION MODULE POPData;

        (********************************************************)
        (*                                                      *)
        (*        Main data operations on a POP mailbox         *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            22 April 1998                   *)
        (*  Last edited:        31 August 1999                  *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (********************************************************)

FROM SYSTEM IMPORT CAST, LOC, CARD8;

IMPORT Strings, IOChan, IOConsts, ChanConsts, SeqFile,
       FileSys, TextIO, OS2;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

FROM Sockets IMPORT
    (* type *)  Socket,
    (* proc *)  send;

FROM Semaphores IMPORT
    (* type *)  Semaphore,
    (* proc *)  Signal;

FROM InetUtilities IMPORT
    (* proc *)  OpenINIFile, INIGetString, ToLower;

FROM FDFiles IMPORT
    (* type *)  FileAttr, FileAttribute, DirectoryEntry,
    (* proc *)  FirstDirEntry, NextDirEntry, DirSearchDone;

FROM MD5 IMPORT
    (* type *)  MD5_CTX, DigestType,
    (* proc *)  MD5Init, MD5Update, MD5Final;

FROM TransLog IMPORT
    (* type *)  TransactionLogID,
    (* proc *)  LogTransaction;

FROM SMTPData IMPORT
    (* proc *)  GetMailRoot;

FROM Names IMPORT
    (* type *)  FilenameString;

(************************************************************************)

CONST
    Nul = CHR(0);
    LockFileName = "LOCK.!!!";

TYPE
    PassString = ARRAY [0..31] OF CHAR;

    (* We have one of these records for each message in a mailbox.      *)
    (* The fields are:                                                  *)
    (*      next         next message in this mailbox                   *)
    (*      size         size in characters of this message             *)
    (*      name         full filename of this message                  *)
    (*      ToBeDeleted  TRUE iff this message is marked for deletion   *)

    MessageListPointer = POINTER TO
                  RECORD
                      next: MessageListPointer;
                      size: CARDINAL;
                      name: FilenameString;
                      ToBeDeleted: BOOLEAN;
                  END (*RECORD*);

    (* A mailbox is the collection of waiting messages for one user.    *)
    (* The fields are:                                                  *)
    (*   name             the user's login name                         *)
    (*   pass             the user's password                           *)
    (*   directory        file filename of the user's mailbox directory *)
    (*   NumberOfMessages the number of messages in the mailbox, not    *)
    (*                       including the ones marked for deletion     *)
    (*   TotalBytes       the sum of file sizes for all messages in the *)
    (*                       mailbox, except those marked for deletion  *)
    (*   MessageList      linked list of message descriptors            *)
    (*   HaveLock         TRUE iff we have obtained exclusive access to *)
    (*                       this mailbox.                              *)

    Mailbox = POINTER TO
                  RECORD
                      name, pass: PassString;
                      directory: FilenameString;
                      NumberOfMessages: CARDINAL;
                      TotalBytes: CARDINAL;
                      MessageList: MessageListPointer;
                      HaveLock: BOOLEAN;
                  END (*RECORD*);

(************************************************************************)

VAR
    (* The directory which contains all mailboxes.  Each mailbox is a   *)
    (* subdirectory of this directory.                                  *)

    MailRoot: FilenameString;

(************************************************************************)

PROCEDURE MessageDescriptor (M: Mailbox;  N: CARDINAL): MessageListPointer;

    (* Returns a pointer to the Nth message in mailbox M; the result is *)
    (* NIL if there is no Nth message.                                  *)

    VAR p: MessageListPointer;

    BEGIN
        IF M = NIL THEN
            p := NIL;
        ELSE
            p := M^.MessageList;
        END (*IF*);
        WHILE (p <> NIL) AND (N > 1) DO
            p := p^.next;  DEC(N);
        END (*WHILE*);
        RETURN p;
    END MessageDescriptor;

(************************************************************************)

PROCEDURE DiscardMessageList (VAR (*INOUT*) ML: MessageListPointer);

    (* Deallocates the storage occupied by a message list, without      *)
    (* committing any changes.                                          *)

    VAR next: MessageListPointer;

    BEGIN
        WHILE ML <> NIL DO
            next := ML^.next;
            DISPOSE (ML);
            ML := next;
        END (*WHILE*);
    END DiscardMessageList;

(************************************************************************)

PROCEDURE BuildMessageList (M: Mailbox);

    (* Assumption: the M^.directory is already set up, and the message  *)
    (* list of M is empty.  This procedure fills the message list, and  *)
    (* updates the counts for number of messages and total bytes.       *)

    VAR SearchString: FilenameString;  D: DirectoryEntry;
        MoreToGo: BOOLEAN;
        previous, current: MessageListPointer;

    BEGIN
        SearchString := M^.directory;
        Strings.Append ("*.MSG", SearchString);
        previous := NIL;
        MoreToGo := FirstDirEntry (SearchString, FALSE, D);
        WHILE MoreToGo DO

            NEW (current);
            WITH current^ DO
                next := NIL;
                size := D.size;
                Strings.Assign (M^.directory, name);
                Strings.Append (D.name, name);
                ToBeDeleted := FALSE;
            END (*WITH*);
            INC (M^.NumberOfMessages);
            INC (M^.TotalBytes, current^.size);
            IF previous = NIL THEN
                M^.MessageList := current;
            ELSE
                previous^.next := current;
            END (*IF*);
            previous := current;
            MoreToGo := NextDirEntry (D);

        END (*WHILE*);
        DirSearchDone (D);
    END BuildMessageList;

(************************************************************************)

PROCEDURE LoadUserData (VAR (*INOUT*) M: Mailbox;
                               username: ARRAY OF CHAR): BOOLEAN;

    (* Checks that the username is valid, and if so initialises the     *)
    (* user data part of M.                                             *)

    VAR hini: OS2.HINI;  success: BOOLEAN;

    BEGIN
        success := FALSE;

        hini := OpenINIFile ("weasel.ini");
        IF hini <> OS2.NULLHANDLE THEN
            WITH M^ DO
                Strings.Assign (username, name);
                IF INIGetString (hini, name, "Password", pass) THEN
                    directory := MailRoot;
                    Strings.Append (name, directory);
                    Strings.Append ('/', directory);
                    NumberOfMessages := 0;
                    TotalBytes := 0;
                    MessageList := NIL;
                    HaveLock := FALSE;
                    success := TRUE;
                END (*IF*);
            END (*WITH*);
            OS2.PrfCloseProfile (hini);
        END (*IF*);

        RETURN success;

    END LoadUserData;

(************************************************************************)

PROCEDURE OpenMailbox (VAR (*INOUT*) M: Mailbox;
                               username: ARRAY OF CHAR): BOOLEAN;

    (* Creates a new Mailbox structure, throwing away any existing      *)
    (* user data in M.  Returns TRUE iff successful.                    *)

    BEGIN
        ToLower (username);
        IF M <> NIL THEN
            DiscardMailbox (M);
        END (*IF*);
        NEW (M);
        IF LoadUserData (M, username) THEN
            RETURN TRUE;
        ELSE
            DISPOSE(M);
            RETURN FALSE;
        END (*IF*);
    END OpenMailbox;

(************************************************************************)

PROCEDURE DiscardMailbox (VAR (*INOUT*) M: Mailbox);

    (* Throws away all data belonging to this Mailbox, and sets the     *)
    (* argument to NILMailbox.                                          *)

    VAR filename: FilenameString;
        dummy: BOOLEAN;

    BEGIN
        IF M <> NIL THEN
            IF M^.HaveLock THEN
                filename := M^.directory;
                Strings.Append (LockFileName, filename);
                FileSys.Remove (filename, dummy);
            END (*IF*);
            DiscardMessageList (M^.MessageList);
            DISPOSE (M);
        END (*IF*);
    END DiscardMailbox;

(************************************************************************)

PROCEDURE LockMailbox (M: Mailbox): CARDINAL;

    (* Attempts to lock the mailbox.  The possible results are          *)
    (*       0     OK, you have exclusive access to the mailbox         *)
    (*       1     not used by this procedure                           *)
    (*       2     can't access mailbox, it's already locked            *)
    (*       3     user directory does not exist                        *)

    VAR filename: FilenameString;  cid: SeqFile.ChanId;
        res: SeqFile.OpenResults;

    BEGIN
        filename := M^.directory;
        Strings.Append (LockFileName, filename);
        IF FileSys.Exists (filename) THEN
            RETURN 2;
        ELSE
            SeqFile.OpenWrite (cid, filename, SeqFile.write, res);
            IF res = ChanConsts.opened THEN
                SeqFile.Close (cid);
                M^.HaveLock := TRUE;
                BuildMessageList (M);
                RETURN 0;
            ELSE
                RETURN 3;
            END (*IF*);
        END (*IF*);
    END LockMailbox;

(************************************************************************)

PROCEDURE PasswordOK (M: Mailbox;  password: ARRAY OF CHAR): CARDINAL;

    (* Locks the mailbox if the password is correct.  The possible      *)
    (* results are                                                      *)
    (*       0     OK, you have exclusive access to the mailbox         *)
    (*       1     password not acceptable                              *)
    (*       2     can't access mailbox, it's already locked            *)
    (*       3     password is OK but user directory does not exist     *)

    BEGIN
        IF (M = NIL) OR NOT Strings.Equal (password, M^.pass) THEN
            RETURN 1;
        ELSE
            RETURN LockMailbox (M);
        END (*IF*);
    END PasswordOK;

(************************************************************************)

PROCEDURE APOPCheck (M: Mailbox;  LogID: TransactionLogID;
                      digeststring, TimeStamp: ARRAY OF CHAR): CARDINAL;

    (* Locks the mailbox if the MD5 digest string is correct.  The      *)
    (* possible results are                                             *)
    (*       0     OK, you have exclusive access to the mailbox         *)
    (*       1     digest string not acceptable                         *)
    (*       2     can't access mailbox, it's already locked            *)
    (*       3     password is OK but user directory does not exist     *)

    (********************************************************************)

    (*
    PROCEDURE HexChar (value: CARDINAL): CHAR;

        (* Converts number to one-digit hex. *)

        BEGIN
            IF value < 10 THEN RETURN CHR(ORD('0') + value)
            ELSE RETURN CHR(ORD('A') - 10 + value)
            END (*IF*);
        END HexChar;

    (********************************************************************)

    PROCEDURE AppendHexByte (VAR (*INOUT*) message: ARRAY OF CHAR;
                             VAR (*INOUT*) j: CARDINAL;  value: CARD8);

        (* Converts value to hex, stores it at message[j], updates j. *)

        BEGIN
            message[j] := HexChar(value DIV 16);  INC(j);
            message[j] := HexChar(value MOD 16);  INC(j);
        END AppendHexByte;

    (********************************************************************)

    PROCEDURE AppendDigest (VAR (*INOUT*) message: ARRAY OF CHAR;
                                                digest: ARRAY OF LOC);

        (* Converts digest to text and appends it to message. *)

        VAR j, k: CARDINAL;

        BEGIN
            j := LENGTH(message);
            FOR k := 0 TO 15 DO
                AppendHexByte (message, j, CAST(CARD8,digest[k]));
            END (*FOR*);
            message[j] := CHR(0);
        END AppendDigest;

    (********************************************************************)

    PROCEDURE DumpDigest (LogID: TransactionLogID;  text: ARRAY OF CHAR;
                          digest: DigestType);

        (* Puts out a log message for testing. *)

        VAR message: ARRAY [0..79] OF CHAR;

        BEGIN
            message := "Digest ";
            Strings.Append (text, message);
            Strings.Append (": ", message);
            AppendDigest (message, digest);
            LogTransaction (LogID, message);
        END DumpDigest;
    *)

    (********************************************************************)

    PROCEDURE CodeOf (hexchar: CHAR): CARDINAL;

        (* Converts a one-digit hexadecimal number. *)

        TYPE CharSet = SET OF CHAR;
        CONST Digits = CharSet {'0'..'9'};

        BEGIN
            IF hexchar IN Digits THEN
                RETURN ORD(hexchar) - ORD('0');
            ELSE
                RETURN ORD(CAP(hexchar)) - ORD('A') + 10;
            END (*IF*);
        END CodeOf;

    (********************************************************************)

    PROCEDURE ConvertDigest (VAR (*OUT*) result: ARRAY OF LOC);

        (* Converts the hexadecimal string digeststring to a *)
        (* 16-byte array result.                             *)

        VAR j: [0..15];

        BEGIN
            FOR j := 0 TO 15 DO
                result[j] := CAST (LOC, 16*CodeOf(digeststring[2*j])
                                        + CodeOf(digeststring[2*j+1]));
            END (*FOR*);
        END ConvertDigest;

    (********************************************************************)

    VAR ctx: MD5_CTX;
        supplied, computed: DigestType;

    BEGIN
        IF (M = NIL) OR (LENGTH(digeststring) <> 32) THEN
            LogTransaction (LogID, "Digest length error");
            RETURN 1;
        END (*IF*);
        ConvertDigest (supplied);
        ctx := MD5Init();
        MD5Update (ctx, TimeStamp, LENGTH(TimeStamp));
        MD5Update (ctx, M^.pass, LENGTH(M^.pass));
        MD5Final (ctx, computed);
        (*
        DumpDigest (LogID, "supplied", supplied);
        DumpDigest (LogID, "expected", computed);
        *)
        IF (supplied[0] <> computed[0]) OR (supplied[1] <> computed[1])
                            OR (supplied[2] <> computed[2])
                            OR (supplied[3] <> computed[3]) THEN
            RETURN 1;
        ELSE
            RETURN LockMailbox (M);
        END (*IF*);
    END APOPCheck;

(************************************************************************)

PROCEDURE NumberAndSize (M: Mailbox;  VAR (*OUT*) N, size: CARDINAL);

    (* Sets N to the number of messages in the mailbox, and size to     *)
    (* the total number of bytes in the messages.                       *)

    BEGIN
        IF M = NIL THEN
            N := 0;  size := 0;
        ELSE
            N := M^.NumberOfMessages;
            size := M^.TotalBytes;
        END (*IF*);
    END NumberAndSize;

(************************************************************************)

PROCEDURE MaxMessageNumber (M: Mailbox): CARDINAL;

    (* Returns the message number of the last undeleted message in the  *)
    (* mailbox.                                                         *)

    VAR result, N: CARDINAL;  current: MessageListPointer;

    BEGIN
        result := 0;  N := 1;
        current := M^.MessageList;
        WHILE current <> NIL DO
            IF NOT current^.ToBeDeleted THEN
                result := N;
            END (*IF*);
            INC (N);
            current := current^.next;
        END (*WHILE*);
        RETURN result;
    END MaxMessageNumber;

(************************************************************************)

PROCEDURE SizeOfMessage (M: Mailbox;  MessageNumber: CARDINAL;
                                   VAR (*OUT*) size: CARDINAL): BOOLEAN;

    (* If message MessageNumber exists, sets size to its size and       *)
    (* returns TRUE.  Otherwise result is FALSE and size is undefined.  *)

    VAR p: MessageListPointer;

    BEGIN
        p := MessageDescriptor (M, MessageNumber);
        IF (p = NIL) OR p^.ToBeDeleted THEN
            RETURN FALSE;
        ELSE
            size := p^.size;
            RETURN TRUE;
        END (*IF*);
    END SizeOfMessage;

(************************************************************************)

PROCEDURE GetUID (M: Mailbox;  MessageNumber: CARDINAL;
                                   VAR (*OUT*) UID: DigestType): BOOLEAN;

    (* If message MessageNumber exists, sets UID to a persistent and    *)
    (* unique identifier for this message, and returns TRUE.  Otherwise *)
    (* result is FALSE and UID is undefined.                            *)

    VAR p: MessageListPointer;  ctx: MD5_CTX;

    BEGIN
        p := MessageDescriptor (M, MessageNumber);
        IF (p = NIL) OR p^.ToBeDeleted THEN
            RETURN FALSE;
        ELSE
            ctx := MD5Init();
            MD5Update (ctx, p^.name, LENGTH(p^.name));
            MD5Final (ctx, UID);
            RETURN TRUE;
        END (*IF*);
    END GetUID;

(************************************************************************)

PROCEDURE SendFile (S: Socket;  sem: Semaphore;  filename: ARRAY OF CHAR;
                      MaxLines: CARDINAL;  id: TransactionLogID): BOOLEAN;

    (* Sends the contents of a file to socket S.                        *)
    (* MaxLines refers to the number of non-header lines to be sent.    *)

    VAR success, MoreToGo, PastHeader, AtEOL: BOOLEAN;
        cid: IOChan.ChanId;
        buffer: ARRAY [0..2047] OF CHAR;
        result: IOConsts.ReadResults;
        res: ChanConsts.OpenResults;
        lines: CARDINAL;

    BEGIN
        LogTransaction (id, filename);
        lines := 0;  PastHeader := FALSE;  AtEOL := TRUE;
        SeqFile.OpenRead (cid, filename,
                        SeqFile.read + SeqFile.text + SeqFile.old, res);
        success := res = ChanConsts.opened;
        MoreToGo := TRUE;
        WHILE success AND MoreToGo DO
            Signal (sem);
            TextIO.ReadString (cid, buffer);

            (* Result is set to the value allRight, endOfLine, or endOfInput. *)

            result := IOChan.ReadResult (cid);
            IF result = IOConsts.endOfInput THEN

                MoreToGo := FALSE;

            ELSIF result = IOConsts.endOfLine THEN

                IF PastHeader THEN
                    INC (lines);
                    MoreToGo := lines < MaxLines;
                ELSIF AtEOL THEN
                    PastHeader := TRUE;
                    MoreToGo := MaxLines > 0;
                END (*IF*);
                buffer[0] := CHR(13);  buffer[1] := CHR(10);
                success := send (S, buffer, 2, 0) <> MAX(CARDINAL);
                TextIO.SkipLine (cid);
                AtEOL := TRUE;

            ELSE

                IF buffer[0] = '.' THEN

                    (* Special case: byte-stuffing to handle the case   *)
                    (* of a line that starts with a period.             *)

                    success := send (S, buffer, 1, 0) <> MAX(CARDINAL);
                END (*IF*);
                success := success AND
                          (send (S, buffer, LENGTH(buffer), 0) <> MAX(CARDINAL));
                AtEOL := FALSE;

            END (*IF*);

        END (*WHILE*);
        SeqFile.Close (cid);

        IF success AND NOT AtEOL THEN
            buffer[0] := CHR(13);  buffer[1] := CHR(10);
            success := send (S, buffer, 2, 0) <> MAX(CARDINAL);
        END (*IF*);
        IF success THEN
            buffer[0] := '.';  buffer[1] := CHR(13);  buffer[2] := CHR(10);
            success := send (S, buffer, 3, 0) <> MAX(CARDINAL);
        END (*IF*);

        RETURN success;

    END SendFile;

(************************************************************************)

PROCEDURE SendMessage (S: Socket;  sem: Semaphore;  M: Mailbox;
                   N, MaxLines: CARDINAL;  id: TransactionLogID): BOOLEAN;

    (* Sends message N in mailbox N to socket S.  The caller must       *)
    (* already have confirmed that this message exists.                 *)
    (* MaxLines refers to the number of non-header lines to be sent.    *)
    (* We must Signal(sem) every so often to ensure that the operation  *)
    (* does not time out.                                               *)
    (* A FALSE result means a communications failure.                   *)

    VAR success: BOOLEAN;  p: MessageListPointer;

    BEGIN
        p := MessageDescriptor (M, N);
        IF (p = NIL) OR p^.ToBeDeleted THEN
            success := FALSE;
        ELSE
            success := SendFile (S, sem, p^.name, MaxLines, id);
        END (*IF*);
        RETURN success;
    END SendMessage;

(************************************************************************)

PROCEDURE MarkForDeletion (M: Mailbox;  N: CARDINAL;
                              VAR (*OUT*) MessageSize: CARDINAL): BOOLEAN;

    (* Marks message number N for deletion.  (The actual deletion       *)
    (* won't happen until a clean logout from the client.)  A return    *)
    (* value of FALSE means "no such message".                          *)

    VAR p: MessageListPointer;

    BEGIN
        p := MessageDescriptor (M, N);
        IF p = NIL THEN
            MessageSize := 0;
            RETURN FALSE;
        ELSE
            MessageSize := p^.size;
            p^.ToBeDeleted := TRUE;
            DEC (M^.NumberOfMessages);
            DEC (M^.TotalBytes, MessageSize);
            RETURN TRUE;
        END (*IF*);
    END MarkForDeletion;

(************************************************************************)

PROCEDURE UndeleteAll (M: Mailbox);

    (* Any messages in M that are marked for deletion are unmarked.     *)

    VAR current: MessageListPointer;

    BEGIN
        IF M <> NIL THEN
            current := M^.MessageList;
            WHILE current <> NIL DO
                IF current^.ToBeDeleted THEN
                    current^.ToBeDeleted := FALSE;
                    INC (M^.NumberOfMessages);
                    INC (M^.TotalBytes, current^.size);
                END (*IF*);
                current := current^.next;
            END (*WHILE*);
        END (*IF*);
    END UndeleteAll;

(************************************************************************)

PROCEDURE CommitChanges (M: Mailbox);

    (* Deletes all files that have been marked for deletion. *)

    VAR previous, current: MessageListPointer;
        dummy: BOOLEAN;

    BEGIN
        IF M <> NIL THEN
            previous := NIL;  current := M^.MessageList;
            WHILE current <> NIL DO
                IF current^.ToBeDeleted THEN
                    FileSys.Remove (current^.name, dummy);
                    IF previous = NIL THEN
                        M^.MessageList := current^.next;
                    ELSE
                        previous^.next := current^.next;
                    END (*IF*);
                    DISPOSE (current);
                ELSE
                    previous := current;
                END (*IF*);
                IF previous = NIL THEN
                    current := M^.MessageList;
                ELSE
                    current := previous^.next;
                END (*IF*);
            END (*WHILE*);
        END (*IF*);
    END CommitChanges;

(************************************************************************)
(*                           INITIALISATION                             *)
(************************************************************************)

PROCEDURE ClearLocks;

    (* Removes any locks that were left set by a premature shutdown. *)

    VAR searchname, filename: FilenameString;  OK: BOOLEAN;
        D: DirectoryEntry;

    BEGIN
        searchname := MailRoot;
        Strings.Append ("*", searchname);
        OK := FirstDirEntry (searchname, TRUE, D);
        WHILE OK DO
            IF directory IN D.attr THEN
                filename := MailRoot;
                Strings.Append (D.name, filename);
                Strings.Append ('/', filename);
                Strings.Append (LockFileName, filename);
                FileSys.Remove (filename, OK);
            END (*IF*);
            OK := NextDirEntry (D);
        END (*WHILE*);
        DirSearchDone (D);
    END ClearLocks;

(************************************************************************)

BEGIN
    GetMailRoot (MailRoot);
    ClearLocks;
END POPData.

