/* Colors.c */

#include "Sys.h"
#include "Curses.h"

#include <ctype.h>
#include <string.h>
#include <signal.h>
#include <setjmp.h>

#include "Util.h"
#include "Cmds.h"
#include "Progress.h"
#include "Hostwin.h"
#include "Prefs.h"
#include "Colors.h"
#include "RCmd.h"
#include "Bookmark.h"
#include "Main.h"

#ifdef USE_CURSES

#include "WGets.h"

/* This is the full-screen window that pops up when you run the
 * preferences editor.
 */
WINDOW *gColorWin = NULL;

extern void WAddCenteredStr(WINDOW *w, int y, char *str);
extern void WAttr(WINDOW *w, int attr, int on);

#endif  /* USE_CURSES */

jmp_buf gColorWinJmp;
extern int gWinInit;

extern int gListWinFG;
extern int gListWinBG;
extern int gPromptWinFG;
extern int gPromptWinBG;
extern int gInputWinFG;
extern int gInputWinBG;
extern int gStatusWinFG;
extern int gStatusWinBG;
extern int gStatusWinAttr;

extern int gListWin;
extern int gPromptWin;
extern int gInputWin;
extern int gBarWin;

PrefOpt gColorOpts[] = {

  {"ListWinForeGround",   "LIST   Win Foreground:",
  kToggleMsg,
  PREFTOGGLE(gListWinFG, COLOR_BLACK, COLOR_WHITE) },

  {"ListWinBackGround",   "LIST   Win Background:",
  kToggleMsg,
  PREFTOGGLE(gListWinBG, COLOR_BLACK, COLOR_WHITE) },

  {"PromptWinForeGround", "PROMPT Win Foreground:",
  kToggleMsg,
  PREFTOGGLE(gPromptWinFG, COLOR_BLACK, COLOR_WHITE) },

  {"PromptWinBackGround", "PROMPT Win Background:",
  kToggleMsg,
  PREFTOGGLE(gPromptWinBG, COLOR_BLACK, COLOR_WHITE) },

  {"InputWinForeGround",  "INPUT  Win Foreground:",
  kToggleMsg,
  PREFTOGGLE(gInputWinFG, COLOR_BLACK, COLOR_WHITE) },

  {"InputWinBackGround",  "INPUT  Win Background:",
  kToggleMsg,
  PREFTOGGLE(gInputWinBG, COLOR_BLACK, COLOR_WHITE) },

  {"StatusWinForeGround", "STATUS Win Foreground:",
  kToggleMsg,
  PREFTOGGLE(gStatusWinFG, COLOR_BLACK, COLOR_WHITE) },

  {"StatusWinBackGround", "STATUS Win Background:",
  kToggleMsg,
  PREFTOGGLE(gStatusWinBG, COLOR_BLACK, COLOR_WHITE) },

  {"StatusWinAttr",       "STATUS Win Attribute:",
  kToggleMsg,
  PREFTOGGLE(gStatusWinAttr, 0, 1) },



};

int gNumEditableColorOpts = ((int)(sizeof(gColorOpts) / sizeof(PrefOpt)));

void GetColorSetting(char *dst, size_t siz, int item)
{
  char *cp;
  string str;
  size_t len;

  *dst = '\0';

  switch (item) {
    case kListForeGroundWinItem:
      if ( gListWinFG == 0 )
        cp = "Black";
      else if ( gListWinFG == 1 )
        cp = "Blue";
      else if ( gListWinFG == 2 )
        cp = "Green";
      else if ( gListWinFG == 3 )
        cp = "Cyan";
      else if ( gListWinFG == 4 )
        cp = "Red";
      else if ( gListWinFG == 5 )
        cp = "Magenta";
      else if ( gListWinFG == 6 )
        cp = "Yellow";
      else if ( gListWinFG == 7 )
        cp = "White";
      else cp = "Unknown";
      (void) Strncpy(dst, cp, siz);
      break;
    
    case kListBackGroundWinItem:
      if ( gListWinBG == 0 )
        cp = "Black";
      else if ( gListWinBG == 1 )
        cp = "Blue";
      else if ( gListWinBG == 2 )
        cp = "Green";
      else if ( gListWinBG == 3 )
        cp = "Cyan";
      else if ( gListWinBG == 4 )
        cp = "Red";
      else if ( gListWinBG == 5 )
        cp = "Magenta";
      else if ( gListWinBG == 6 )
        cp = "Yellow";
      else if ( gListWinBG == 7 )
        cp = "White";
      else cp = "Unknown";
      (void) Strncpy(dst, cp, siz);
      break;

    case kPromptForeGroundWinItem:
      if ( gPromptWinFG == 0 )
        cp = "Black";
      else if ( gPromptWinFG == 1 )
        cp = "Blue";
      else if ( gPromptWinFG == 2 )
        cp = "Green";
      else if ( gPromptWinFG == 3 )
        cp = "Cyan";
      else if ( gPromptWinFG == 4 )
        cp = "Red";
      else if ( gPromptWinFG == 5 )
        cp = "Magenta";
      else if ( gPromptWinFG == 6 )
        cp = "Yellow";
      else if ( gPromptWinFG == 7 )
        cp = "White";
      else cp = "Unknown";
      (void) Strncpy(dst, cp, siz);
      break;
    
    case kPromptBackGroundWinItem:
      if ( gPromptWinBG == 0 )
        cp = "Black";
      else if ( gPromptWinBG == 1 )
        cp = "Blue";
      else if ( gPromptWinBG == 2 )
        cp = "Green";
      else if ( gPromptWinBG == 3 )
        cp = "Cyan";
      else if ( gPromptWinBG == 4 )
        cp = "Red";
      else if ( gPromptWinBG == 5 )
        cp = "Magenta";
      else if ( gPromptWinBG == 6 )
        cp = "Yellow";
      else if ( gPromptWinBG == 7 )
        cp = "White";
      else cp = "Unknown";
      (void) Strncpy(dst, cp, siz);
      break;

    case kInputForeGroundWinItem:
      if ( gInputWinFG == 0 )
        cp = "Black";
      else if ( gInputWinFG == 1 )
        cp = "Blue";
      else if ( gInputWinFG == 2 )
        cp = "Green";
      else if ( gInputWinFG == 3 )
        cp = "Cyan";
      else if ( gInputWinFG == 4 )
        cp = "Red";
      else if ( gInputWinFG == 5 )
        cp = "Magenta";
      else if ( gInputWinFG == 6 )
        cp = "Yellow";
      else if ( gInputWinFG == 7 )
        cp = "White";
      else cp = "Unknown";
      (void) Strncpy(dst, cp, siz);
      break;
    
    case kInputBackGroundWinItem:
      if ( gInputWinBG == 0 )
        cp = "Black";
      else if ( gInputWinBG == 1 )
        cp = "Blue";
      else if ( gInputWinBG == 2 )
        cp = "Green";
      else if ( gInputWinBG == 3 )
        cp = "Cyan";
      else if ( gInputWinBG == 4 )
        cp = "Red";
      else if ( gInputWinBG == 5 )
        cp = "Magenta";
      else if ( gInputWinBG == 6 )
        cp = "Yellow";
      else if ( gInputWinBG == 7 )
        cp = "White";
      else cp = "Unknown";
      (void) Strncpy(dst, cp, siz);
      break;

    case kStatusForeGroundWinItem:
      if ( gStatusWinFG == 0 )
        cp = "Black";
      else if ( gStatusWinFG == 1 )
        cp = "Blue";
      else if ( gStatusWinFG == 2 )
        cp = "Green";
      else if ( gStatusWinFG == 3 )
        cp = "Cyan";
      else if ( gStatusWinFG == 4 )
        cp = "Red";
      else if ( gStatusWinFG == 5 )
        cp = "Magenta";
      else if ( gStatusWinFG == 6 )
        cp = "Yellow";
      else if ( gStatusWinFG == 7 )
        cp = "White";
      else cp = "Unknown";
      (void) Strncpy(dst, cp, siz);
      break;
    
    case kStatusBackGroundWinItem:
      if ( gStatusWinBG == 0 )
        cp = "Black";
      else if ( gStatusWinBG == 1 )
        cp = "Blue";
      else if ( gStatusWinBG == 2 )
        cp = "Green";
      else if ( gStatusWinBG == 3 )
        cp = "Cyan";
      else if ( gStatusWinBG == 4 )
        cp = "Red";
      else if ( gStatusWinBG == 5 )
        cp = "Magenta";
      else if ( gStatusWinBG == 6 )
        cp = "Yellow";
      else if ( gStatusWinBG == 7 )
        cp = "White";
      else cp = "Unknown";
      (void) Strncpy(dst, cp, siz);
      break;

    case kStatusAttrItem:
      if ( gStatusWinAttr == 0 )
        cp = "Normal";
      else if ( gStatusWinAttr == 1 )
        cp = "Bold";
      else cp = "Unknown";
      (void) Strncpy(dst, cp, siz);
      break;



  }
}       /* GetColorSetting */


#ifdef USE_CURSES

/* Draws the screen when we're using the host editor's main screen.
 * You can can specify whether to draw each character whether it needs
 * it or not if you like.
 */
void UpdateColorWindow(int uptAll)
{
  if (uptAll) {
    touchwin(gColorWin);
  }
  wnoutrefresh(gColorWin);
  doupdate();
}       /* UpdatePrefsWindow */


/* This displays a message in the preferences window. */
void ColorWinWinMsg(char *msg)
{
  int maxx, maxy;

  getmaxyx(gColorWin, maxy, maxx);
  mvwaddstr(gColorWin, maxy - 2, 0, msg);
  wclrtoeol(gColorWin);
  wmove(gColorWin, maxy - 1, 0);
  wrefresh(gColorWin);
}       /* PrefsWinWinMsg */




/* Prompts for a line of input. */
void ColorWinGetStr(char *dst, int canBeEmpty, int canEcho)
{
  string str;
  WGetsParams wgp;
  int maxx, maxy;
  
  getmaxyx(gColorWin, maxy, maxx);
  WAttr(gColorWin, kBold, 1);
  mvwaddstr(gColorWin, maxy - 1, 0, "> ");
  WAttr(gColorWin, kBold, 0);
  wclrtoeol(gColorWin);
  wrefresh(gColorWin);
  curs_set(1);

  wgp.w = gColorWin;
  wgp.sy = maxy - 1;
  wgp.sx = 2;
  wgp.fieldLen = maxx - 3;
  wgp.dst = str;
  wgp.dstSize = sizeof(str);
  wgp.useCurrentContents = 0;
  wgp.echoMode = canEcho ? wg_RegularEcho : wg_BulletEcho;
  wgp.history = wg_NoHistory;
  (void) wg_Gets(&wgp);
  cbreak();                                               /* wg_Gets turns off cbreak and delay. */

  TraceMsg("[%s]\n", str);

  /* See if the user just hit return.  We may not want to overwrite
   * the dst here, which would make it an empty string.
   */
  if ((wgp.changed) || (canBeEmpty == kOkayIfEmpty))
    strcpy(dst, str);

  wmove(gColorWin, maxy - 1, 0);
  wclrtoeol(gColorWin);
  wrefresh(gColorWin);
  curs_set(0);
}       /* ColorWinGetStr */


/* Prompts for an integer of input. */
void ColorWinGetNum(int *dst)
{
  string str;
  WGetsParams wgp;
  int maxx, maxy;

  getmaxyx(gColorWin, maxy, maxx);
  WAttr(gColorWin, kBold, 1);
  mvwaddstr(gColorWin, maxy - 1, 0, "> ");
  WAttr(gColorWin, kBold, 0);
  wclrtoeol(gColorWin);
  wrefresh(gColorWin);
  curs_set(1);

  wgp.w = gColorWin;
  wgp.sy = maxy - 1;
  wgp.sx = 2;
  wgp.fieldLen = maxx - 3;
  wgp.dst = str;
  wgp.dstSize = sizeof(str);
  wgp.useCurrentContents = 0;
  wgp.echoMode = wg_RegularEcho;
  wgp.history = wg_NoHistory;
  (void) wg_Gets(&wgp);
  cbreak();                                               /* wg_Gets turns off cbreak and delay. */

  TraceMsg("[%s]\n", str);
  AtoIMaybe(dst, str);
  wmove(gColorWin, maxy - 1, 0);
  wclrtoeol(gColorWin);
  wrefresh(gColorWin);
  curs_set(0);
}       /* ColorWinGetNum */




/* This is the meat of the preferences window.  We can selectively update
 * portions of the window by using a bitmask with bits set for items
 * we want to update.
 */
void ColorWinDraw(int flags, int hilite)
{
  int i;
  string value;
  char spec[40];
  int maxx, maxy;

  getmaxyx(gColorWin, maxy, maxx);
  /* Draw the keys the user can type in reverse text. */
  WAttr(gColorWin, kReverse, 1);
  for (i = kFirstColorWinItem; i <= kLastColorWinItem; i++) {
    if (TESTBIT(flags, i))
      mvwaddch(gColorWin, 2 + i, 2, 'A' + i);
  }

  /* The "quit" item is a special item that is offset a line, and
   * always has the "X" key assigned to it.
   */
  i = kQuitColorWinItem;
  if (TESTBIT(flags, i))
          mvwaddch(gColorWin, 3 + i, 2, 'X');
  WAttr(gColorWin, kReverse, 0);

  /* We can use this to hilite a whole line, to indicate to the
   * user that a certain item is being edited.
   */
  if (hilite)
    WAttr(gColorWin, kReverse, 1);

  sprintf(spec, " %%-35s%%-%ds", maxx - 35 - 6);

  for (i = kFirstColorWinItem; i <= kLastColorWinItem; i++) {
    if (TESTBIT(flags, i)) {
      GetColorSetting(value, sizeof(value), i);
      mvwprintw(gColorWin, i + 2 , 3, spec, gColorOpts[i].label, value);
      wclrtoeol(gColorWin);
    }
  }

  if (TESTBIT(flags, kQuitColorWinItem)) {
    mvwprintw(gColorWin, kQuitColorWinItem + 3, 3, " %-35s","(Done editing)");
    wclrtoeol(gColorWin);
  }

  if (hilite)
    WAttr(gColorWin, kReverse, 0);

  wmove(gColorWin, maxy - 1, 0);
  wrefresh(gColorWin);
}       /* ColorWinDraw */



/* The user can hit space to change values.  For these toggle
 * functions we do an update each time so the user can see the change
 * immediately.
 */
void ColorWinToggle(int *val, int bitNum, int min, int max)
{
  int c;

  while (1) {
    c = wgetch(gColorWin);
    TraceMsg("[%c, 0x%x]\n", c, c);
    if ((c == 'q') || (c == 10) || (c == 13)
#ifdef KEY_ENTER
      || (c == KEY_ENTER)
#endif
    )
      break;
    else if (isspace(c)) {
      TogglePref(val, min, max);
      ColorWinDraw(BIT(bitNum), kHilite);
    }
  }
}       /* ColorWinToggle */


/*ARGSUSED*/
void SigIntColorWin(int sigNum)
{
  alarm(0);
  longjmp(gColorWinJmp, 1);
} /* SigIntColorWin */


#endif  /* USE_CURSES */


/* Runs the preferences editor. */
int ColorWindow(void)
{
#ifdef USE_CURSES
  int c, field, i;

  if (gWinInit) {
    gColorWin = newwin(LINES, COLS, 0, 0);
    if (gColorWin == NULL)
      return (kCmdErr);

    wbkgd(gColorWin, COLOR_PAIR(1));

    if (setjmp(gColorWinJmp) == 0) {
      /* Gracefully cleanup the screen if the user ^C's. */
      SIGNAL(SIGINT, SigIntColorWin);

      curs_set(0);
      cbreak();
  
      /* Set the clear flag for the first update. */
      wclear(gColorWin);

      WAttr(gColorWin, kBold, 1);
      WAddCenteredStr(gColorWin, 0, "NcFTP Color Configuration");
      WAttr(gColorWin, kBold, 0);

      ColorWinDraw(kAllWindowItems, kNoHilite);
      while (1) {
        ColorWinWinMsg("Select an item to edit by typing its corresponding letter.");
        c = wgetch(gColorWin);
        TraceMsg("[%c, 0x%x]\n", c, c);
        if (islower(c))
          c = toupper(c);
        if (!isupper(c))
          continue;
        if (c == 'X')
          break;
        field = c - 'A';
        i = field;
        if (i > kLastColorWinItem)
          continue;

        /* Hilite the current item to edit. */
        ColorWinDraw(BIT(i), kHilite);

        /* Print the instructions. */
        ColorWinWinMsg(gColorOpts[i].msg);

        switch (gColorOpts[i].type) {
          case kPrefInt:
            ColorWinGetNum((int *) gColorOpts[i].storage);
            break;
          case kPrefToggle:
            ColorWinToggle((int *) gColorOpts[i].storage,
            i, gColorOpts[i].min, gColorOpts[i].max);
            break;
          case kPrefStr:
            ColorWinGetStr((char *) gColorOpts[i].storage,
              gColorOpts[i].min,       /* Used for Empty */
              gColorOpts[i].max        /* Used for Echo */
              );
            break;
        }

        /* Update and unhilite it. */
        ColorWinDraw(BIT(field), kNoHilite);
      }
    }

    delwin(gColorWin);
    gColorWin = NULL;
    UpdateScreen(1);
    flushinp();
    nocbreak();
    curs_set(1);

    init_pair(1,gListWinFG,gListWinBG);
    wbkgd(gListWin, COLOR_PAIR(1));

    init_pair(2,gStatusWinFG,gStatusWinBG);
    wbkgd(gBarWin, COLOR_PAIR(2));

    init_pair(3, gPromptWinFG,gPromptWinBG);
    wbkgd(gPromptWin,COLOR_PAIR(3));

    init_pair(4, gInputWinFG,gInputWinBG);
    wbkgd(gInputWin,COLOR_PAIR(4));

    if ( gStatusWinAttr )
      WAttr(gBarWin, kBold, 1);
    else
      WAttr(gBarWin, kNormal, 1);

  }
#endif  /* USE_CURSES */
  return (kNoErr);
}   /* ColorWindow */


int ColorCmd(int argc, char **argv)
{
#ifdef USE_CURSES
  int err;

  if (!gWinInit) {
    EPrintF("%s\n%s\n",
    "The color editor only works in visual mode.",
    "However, you can use the 'set' command to set a single option at a time."
    );
    return (kCmdErr);
  }
  err = ColorWindow();
  Beep(0);    /* User should be aware that it took a while, so no beep. */
  return (err);
#else
  EPrintF("%s\n%s\n",
  "You can't do this because the program doesn't have the curses library.",
  "However, you can use the 'set' command to set a single option at a time."
  );
  return (kCmdErr);
#endif  /* USE_CURSES */
}       /* ColorCmd */


void WriteColor(void)
{
  longstring path;
  FILE *fp;
  int i;

  if (gOurDirectoryPath[0] == '\0')
    return;         /* Don't create in root directory. */

  OurDirectoryPath(path, sizeof(path), kColorName);
  fp = fopen(path, "w");
  if (fp == NULL) {
    Error(kDoPerror, "Can't open %s for writing.\n", path);
    return;
  }
  for (i = kFirstColorWinItem; i <= kLastColorWinItem; i++) {
    switch (gColorOpts[i].type) {
      case kPrefInt:
      case kPrefToggle:
        fprintf(fp, "%s %d\n", gColorOpts[i].name,* (int *) gColorOpts[i].storage);
        break;
      case kPrefStr:
        fprintf(fp, "%s %s\n", gColorOpts[i].name,(char *) gColorOpts[i].storage);
        break;
    }
  }

  fclose(fp);
}  /* WriteColor */

static
int ColorSearchProc(char *key, const PrefOpt *b)
{
  return (ISTRCMP(key, (*b).name));
}       /* ColorSearchProc */

void ReadColor(void)
{
  longstring path;
  FILE *fp;
  string option;
  longstring val;
  longstring line;
  int o;
  PrefOpt *pop;
  int i, x;

  OurDirectoryPath(path, sizeof(path), kColorName);
  fp = fopen(path, "r");
  if (fp == NULL) {
    /* It's okay if we don't have one. */
    return;
  }

  while (FGets(line, (int) sizeof(line) - 1, fp) != NULL) {
    if (sscanf(line, "%s", option) < 1)
      continue;
    o = strlen(option) + 1;
    STRNCPY(val, line + o);

    pop = NULL;
    for ( i = 0; i < kLastColorWinItem; i++ ) {
      for ( x = 0; x < gNumEditableColorOpts; x++ ) {
        if ( ( strcmp( gColorOpts[x].name, option ) ) == 0 )
          pop = (PrefOpt *) &gColorOpts[x];
      }
    }
/*
    pop = (PrefOpt *) BSEARCH(option, gColorOpts, SZ(gNumEditableColorOpts),
                  sizeof(PrefOpt), ColorSearchProc);
*/
    if (pop == NULL) {
      Error(kDontPerror, "Unrecognized color option \"%s\".\n",option);
      continue;
    }
    switch (pop->type) {
      case kPrefInt:
      case kPrefToggle:
        * (int *) pop->storage = atoi(val);
        break;
      case kPrefStr:
        (void) Strncpy((char *) pop->storage, val, pop->siz);
        break;
    }
  }

  fclose(fp);
} /* ReadColor */