IMPLEMENTATION MODULE Users;

        (********************************************************)
        (*                                                      *)
        (*         Editor for the user access rights            *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            2 December 1997                 *)
        (*  Last edited:        28 January 1999                 *)
        (*  Status:             OK                              *)
        (*                                                      *)
        (********************************************************)

FROM SYSTEM IMPORT ADDRESS, ADR, CAST;

IMPORT OS2, Strings;

FROM PermissionEditor IMPORT
    (* type *)  DirEntryPtr, CharArrayPointer,
    (* proc *)  LoadTreeData, SizeOfDirectoryData, StoreDirectoryData,
                KillList, CreateDefaultRoot, EditDirectoryData,
                DirectorySummary;

FROM SetupINI IMPORT
    (* proc *)  OurINIHandle;

FROM ListBoxes IMPORT
    (* type *)  ListBox,
    (* proc *)  CreateListBox, DestroyListBox;

FROM InetUtilities IMPORT
    (* proc *)  INIGet, INIGetString, ToLower;

FROM ScreenEditor IMPORT
    (* type *)  Structure,
    (* proc *)  CreateField, MenuField, CardinalField, Combine,
                ScreenEdit, DeleteStructure;

FROM FieldEditor IMPORT
    (* type *)  FieldType,
    (* proc *)  DefineFieldType, DiscardFieldType;

FROM Menus IMPORT
    (* type *)  Menu, ItemText,
    (* proc *)  CreateMenu, MenuColours;

FROM Windows IMPORT
    (* type *)  Window, Colour, FrameType, DividerType,
    (* proc *)  OpenWindowHidden, CloseWindow, WriteChar, WriteString, WriteLn,
                SetCursor, GetKey, EditString, GetScreenSize, ColourSwap,
                ReadCharWithoutEcho;

FROM MultiScreen IMPORT
    (* type *)  VirtualScreen,
    (* proc *)  MapToVirtualScreen, EnableScreenSwitching;

FROM Keyboard IMPORT
    (* proc *)  PutBack;

FROM LowLevel IMPORT
    (* proc *)  EVAL;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

(********************************************************************************)

CONST Nul = CHR(0);

TYPE
    FileNameSubscript = [0..255];
    FileNameString = ARRAY FileNameSubscript OF CHAR;
    PassString = ARRAY [0..31] OF CHAR;
    PassStringPtr = POINTER TO PassString;
    UserCategory = (NoSuchUser, NoPasswordNeeded, GuestUser, NormalUser, Manager);
    CategoryMapType = ARRAY UserCategory OF CHAR;

CONST CategoryMap = CategoryMapType {'?', 'N', 'G', 'U', 'M'};

    (* The fields in a UserPermission record have the following meanings.       *)
    (*      category     What sort of user this is                              *)
    (*      Password     The user's password                                    *)
    (*      UserLimit    Maximum number of instances of this user               *)
    (*      SpeedLimit   Rate limit (bytes per second)                          *)
    (*      TreeRoot     The private structure that stores information about    *)
    (*                    the directories to which the user has access.         *)

TYPE
    User = POINTER TO UserPermission;

    UserPermission = RECORD
                         category: UserCategory;
                         Password: PassString;
                         UserLimit: CARDINAL;
                         SpeedLimit: CARDINAL;
                         TreeRoot: DirEntryPtr;
                     END (*RECORD*);

VAR
    (* Type for field editor. *)

    ShortString: FieldType;

    (* Number of display rows on screen.  *)

    ScreenRows: CARDINAL;

(********************************************************************************)
(*                    CHECKING WHETHER A USER EXISTS                            *)
(********************************************************************************)

PROCEDURE UserInINIFile (VAR (*OUT*) hini: OS2.HINI;
                                username: ARRAY OF CHAR): BOOLEAN;

    (* If username is in INI file, initialises hini and returns TRUE. *)

    VAR size: CARDINAL;  result: BOOLEAN;

    BEGIN
        hini := OurINIHandle();
        IF hini = OS2.NULLHANDLE THEN
            result := FALSE;
        ELSE
            result := OS2.PrfQueryProfileSize (hini, username, NIL, size)
                                   AND (size <> 0);
        END (*IF*);
        RETURN result;
    END UserInINIFile;

(********************************************************************************)

PROCEDURE FindUser (UserName: ARRAY OF CHAR): BOOLEAN;

    (* Returns TRUE iff the user data exists.        *)

    VAR hini: OS2.HINI;

    BEGIN
        RETURN UserInINIFile (hini, UserName);
    END FindUser;

(********************************************************************************)
(*                                  PARSER                                      *)
(********************************************************************************)

PROCEDURE ReadUserData (name: ARRAY OF CHAR): User;

    (* Fetches the password, etc., for the user whose username is specified     *)
    (* as the argument.  Returns a NIL result if no user data is found.         *)

    VAR hini: OS2.HINI;
        bufptr: CharArrayPointer;
        BufferSize: CARDINAL;
        result: User;

    BEGIN       (* Body of ReadUserData *)

        IF UserInINIFile (hini, name) THEN

            NEW (result);

            IF NOT INIGet (hini, name, "Category", result^.category) THEN
                result^.category := NormalUser;
            END (*IF*);
            IF NOT INIGetString (hini, name, "Password", result^.Password) THEN
                result^.Password := "@";
            END (*IF*);
            IF NOT INIGet (hini, name, "UserLimit", result^.UserLimit) THEN
                result^.UserLimit := MAX(CARDINAL);
            END (*IF*);
            IF NOT INIGet (hini, name, "SpeedLimit", result^.SpeedLimit) THEN
                result^.SpeedLimit := MAX(CARDINAL);
            ELSIF result^.SpeedLimit = 0 THEN
                result^.SpeedLimit := 1;
            END (*IF*);
            IF OS2.PrfQueryProfileSize (hini, name, "Volume", BufferSize)
                                                    AND (BufferSize <> 0) THEN
                ALLOCATE (bufptr, BufferSize);
                OS2.PrfQueryProfileData (hini, name, "Volume", bufptr, BufferSize);
            ELSE
                bufptr := NIL;
            END (*IF*);

            result^.TreeRoot := LoadTreeData (bufptr, BufferSize);
            IF bufptr <> NIL THEN
                DEALLOCATE (bufptr, BufferSize);
            END (*IF*);

        ELSE
            result := NIL;
        END (*IF*);

        RETURN result;

    END ReadUserData;

(********************************************************************************)
(*                      WRITING USER DATA TO THE INI FILE                       *)
(********************************************************************************)

PROCEDURE StoreUserData (U: User;  username: ARRAY OF CHAR);

    (* Writes permission data to the INI file. *)

    VAR hini: OS2.HINI;
        size: CARDINAL;
        bufptr: CharArrayPointer;

    BEGIN
        hini := OurINIHandle();
        IF hini = OS2.NULLHANDLE THEN
            RETURN;
        END (*IF*);
        OS2.PrfWriteProfileData (hini, username, "Category",
                                ADR(U^.category), SIZE(UserCategory));
        OS2.PrfWriteProfileData (hini, username, "Password",
                                ADR(U^.Password), LENGTH(U^.Password));
        OS2.PrfWriteProfileData (hini, username, "UserLimit",
                                ADR(U^.UserLimit), SIZE(CARDINAL));
        OS2.PrfWriteProfileData (hini, username, "SpeedLimit",
                                ADR(U^.SpeedLimit), SIZE(CARDINAL));

        (* Store all volume information in the bufptr^ array. *)

        size := SizeOfDirectoryData (U^.TreeRoot);
        ALLOCATE (bufptr, size);
        StoreDirectoryData (U^.TreeRoot, bufptr^);

        (* Copy the result from bufptr^ to the INI file. *)

        IF size > 0 THEN
            OS2.PrfWriteProfileData (hini, username, "Volume", bufptr, size);
            DEALLOCATE (bufptr, size);
        END (*IF*);

    END StoreUserData;

(********************************************************************************)
(*                        DISCARDING A USER RECORD                              *)
(********************************************************************************)

PROCEDURE DestroyUserData (VAR (*INOUT*) U: User);

    (* Discards the data structure.  *)

    BEGIN
        KillList (U^.TreeRoot);
    END DestroyUserData;

(************************************************************************)

PROCEDURE RemoveUser (name: ARRAY OF CHAR);

    (* Deletes this user's INI file entry. *)

    VAR hini: OS2.HINI;

    BEGIN
        hini := OurINIHandle();
        IF hini <> OS2.NULLHANDLE THEN
           OS2.PrfWriteProfileData (hini, name, NIL, NIL, 0);
        END (*IF*);
    END RemoveUser;

(************************************************************************)
(*                        ShortString EDITOR                            *)
(************************************************************************)

PROCEDURE WriteShortString (w: Window;  p: ADDRESS;  width: CARDINAL);

    VAR address: PassStringPtr;

    BEGIN
        address := p;
        WriteString (w, address^);
    END WriteShortString;

(************************************************************************)

PROCEDURE EditShortString (w: Window;  VAR (*INOUT*) p: ADDRESS;
                                         stringsize, width: CARDINAL);

    VAR address: PassStringPtr;

    BEGIN
        address := p;
        EditString (w, address^, stringsize, width);
    END EditShortString;

(********************************************************************************)
(*                          MASTER EDITING PROCEDURE                            *)
(********************************************************************************)

PROCEDURE EditUser (Screen: VirtualScreen;  VAR (*INOUT*) name: ARRAY OF CHAR;
                                                                clone: BOOLEAN);

    (* Top-level editor for the user permission data. *)

    VAR UserData, ErrorWindow: Window;
        ch: CHAR;

    (****************************************************************************)

    PROCEDURE WaitForKey;

        (* Called when we are waiting for the user to type something after     *)
        (* we've displayed an "invalid user name" error message.               *)

        VAR ch: CHAR;

        BEGIN
            SetCursor (UserData, 1, 20);
            ColourSwap (UserData, 1, 20, 32);
            ReadCharWithoutEcho (UserData, ch);  PutBack(ch);
            ColourSwap (UserData, 1, 20, 32);
        END WaitForKey;

    (****************************************************************************)

    VAR TopBar, Background, DirStatus, bottombar: Window;
        UserName: PassString;
        R: Structure;
        abort: BOOLEAN;
        U: User;
        HomeLabel: FileNameString;
        M: Menu;
        CatLabel: ARRAY [0..4] OF ItemText;
        CategoryNumber: CARDINAL;

    BEGIN
        EnableScreenSwitching (FALSE);
        Strings.Assign (name, UserName);

        OpenWindowHidden (TopBar, yellow, red, 0, 0, 0, 79, noframe, nodivider);
        MapToVirtualScreen (TopBar, Screen);
        WriteString (TopBar, "Username: ");
        WriteString (TopBar, UserName);

        OpenWindowHidden (Background, yellow, black, 1, ScreenRows-2, 0, 79, noframe, nodivider);
        MapToVirtualScreen (Background, Screen);
        SetCursor (Background, 9, 20);
        WriteString (Background, "Cursor down to set up the directories");

        OpenWindowHidden (bottombar, yellow, red, ScreenRows-1, ScreenRows-1, 0, 79, noframe, nodivider);
        MapToVirtualScreen (bottombar, Screen);
        WriteString (bottombar, " Esc exit  ");
        WriteChar (bottombar, CHR(24));  WriteChar (bottombar, CHR(25));
        WriteString (bottombar, " select");

        U := ReadUserData (name);
        IF clone THEN
            name[0] := Nul;  UserName[0] := Nul;
        END (*IF*);
        IF U = NIL THEN
            NEW (U);
            U^.category := NormalUser;
            U^.UserLimit := 10;
            U^.SpeedLimit := MAX(CARDINAL);
            Strings.Assign ("", U^.Password);
            U^.TreeRoot := NIL;
        END (*IF*);
        IF U^.TreeRoot = CAST(DirEntryPtr, NIL) THEN
            U^.TreeRoot := CreateDefaultRoot();
        END (*IF*);

        OpenWindowHidden (UserData, white, blue, 2, 8, 1, 70, noframe, nodivider);
        MapToVirtualScreen (UserData, Screen);

        (* Create the menu of user categories. *)

        CatLabel[1] := "NoPassword";
        CatLabel[2] := "Guest";
        CatLabel[3] := "User";
        CatLabel[4] := "Manager";
        CreateMenu (M, 4, CatLabel, 4);
        MenuColours (M, white, blue, blue, cyan, yellow);
        CategoryNumber := ORD (U^.category);

        (* Create the username/category/password editing structure. *)

        SetCursor (UserData, 1, 2);  WriteString (UserData, "User name");
        SetCursor (UserData, 2, 2);  WriteString (UserData, "Category");
        SetCursor (UserData, 3, 2);  WriteString (UserData, "Password");
        SetCursor (UserData, 4, 2);  WriteString (UserData, "User limit");
        SetCursor (UserData, 5, 2);  WriteString (UserData, "Speed limit");
        SetCursor (UserData, 5, 33);  WriteString (UserData, "bytes/second");

        R := CreateField (ADR(UserName), SIZE(UserName), ShortString, 1, 20, 32);
        Combine (R, MenuField (CategoryNumber, 2, 20, 1, 48, M));
        Combine (R, CreateField (ADR(U^.Password), SIZE(PassString), ShortString, 3, 20, 32));
        Combine (R, CardinalField (U^.UserLimit, 4, 20, 12));
        Combine (R, CardinalField (U^.SpeedLimit, 5, 20, 12));

        (* Create the directory overview window.  *)

        OpenWindowHidden (DirStatus, white, black, 11, ScreenRows-3,
                                            1, 78, noframe, nodivider);
        MapToVirtualScreen (DirStatus, Screen);

        (* Here's the main editing loop. *)

        SetCursor (UserData, 1, 1);
        LOOP
            LOOP
                DirectorySummary (DirStatus, U^.TreeRoot, ScreenRows-13);
                ScreenEdit (UserData, R, abort);
                ToLower (UserName);
                U^.category := VAL(UserCategory, CategoryNumber);
                IF abort THEN EXIT(*LOOP*) END(*IF*);

                (* Check the character that took us off the edge.  If it's *)
                (* "cursor down", proceed to editing directories.  If it's *)
                (* "cursor up", leave this procedure.                      *)

                ch := GetKey (UserData);
                IF ch = CHR(0) THEN
                    ch := GetKey (UserData);
                    IF ch = 'H' THEN
                        EXIT (*LOOP*);
                    ELSIF ch = 'P' THEN
                        EditDirectoryData (U^.TreeRoot, Screen);
                        SetCursor (UserData, 5, 1);
                    END (*IF*);
                END (*IF*);

            END (*LOOP*);

            (* At this point, "name" holds the original username, and "UserName"    *)
            (* holds the new name.  We need to check for the following errors:      *)
            (*    1.  UserName is empty.                                            *)
            (*    2.  UserName duplicates an existing name.                         *)

            IF UserName[0] = CHR(0) THEN
                OpenWindowHidden (ErrorWindow, white, black, 13, 15, 20, 59, noframe, nodivider);
                MapToVirtualScreen (ErrorWindow, Screen);
                WriteLn (ErrorWindow);
                WriteString (ErrorWindow, "  You have not assigned a user name");
                WaitForKey;
                CloseWindow (ErrorWindow);
            ELSE
                Strings.Assign (name, HomeLabel);
                ToLower (HomeLabel);
                IF NOT Strings.Equal (HomeLabel, UserName) AND FindUser (UserName) THEN
                    OpenWindowHidden (ErrorWindow, white, black, 13, 16, 16, 63, noframe, nodivider);
                    MapToVirtualScreen (ErrorWindow, Screen);
                    WriteLn (ErrorWindow);
                    WriteString (ErrorWindow, " You've duplicated the name of an existing user");
                    WriteLn (ErrorWindow);
                    WriteString (ErrorWindow, "       Please assign a different name");
                    WaitForKey;
                    CloseWindow (ErrorWindow);
                ELSE
                    EXIT (*LOOP*);
                END (*IF*);
            END (*IF*);

        END (*LOOP*);

        DeleteStructure (R);
        CloseWindow (UserData);
        CloseWindow (TopBar);
        CloseWindow (DirStatus);

        (* Destroy the old user permission file, and create a new one. *)

        IF name[0] <> Nul THEN
            RemoveUser (name);
        END (*IF*);
        Strings.Assign (UserName, name);
        StoreUserData (U, name);
        DestroyUserData (U);

        CloseWindow (bottombar);
        CloseWindow (Background);
        EnableScreenSwitching (TRUE);

    END EditUser;

(********************************************************************************)
(*                              INITIALISATION                                  *)
(********************************************************************************)

VAR dummy: CARDINAL;

BEGIN
    GetScreenSize (ScreenRows, dummy);
    ShortString := DefineFieldType (WriteShortString, EditShortString);
FINALLY
    DiscardFieldType (ShortString);
END Users.

