MODULE StorePRM;

        (********************************************************)
        (*                                                      *)
        (*  Program to store data from ftpd.ini to PRM files    *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            6 March 1998                    *)
        (*  Last edited:        10 October 1998                 *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (********************************************************)

FROM SYSTEM IMPORT
    (* type *)  LOC,
    (* proc *)  ADR;

IMPORT OS2, IOChan, ChanConsts, Strings, STextIO, TextIO, SeqFile, FileSys;

FROM ProgramArgs IMPORT
    (* proc *)  ArgChan, IsArgPresent;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

(************************************************************************)

CONST Nul = CHR(0);

TYPE
    NameString = ARRAY [0..31] OF CHAR;
    FileNameString = ARRAY [0..255] OF CHAR;
    CharSet = SET OF CHAR;
    UserCategory = (NoSuchUser, NoPasswordNeeded, GuestUser, NormalUser, Manager);

(************************************************************************)

VAR
    (* Anchor block handle for this application.  *)

    hab: OS2.HAB;

(********************************************************************************)
(*                          OUTPUT TO PRM FILE                                  *)
(********************************************************************************)

PROCEDURE WriteRaw (cid: IOChan.ChanId;  data: ARRAY OF LOC;  amount: CARDINAL);

    (* Writes a string to a file. *)

    BEGIN
        IOChan.RawWrite (cid, ADR(data), amount);
    END WriteRaw;

(************************************************************************)

PROCEDURE FWriteChar (cid: IOChan.ChanId;  character: CHAR);

    (* Writes a single character to a file. *)

    BEGIN
        IOChan.RawWrite (cid, ADR(character), 1);
    END FWriteChar;

(************************************************************************)

PROCEDURE FWriteQuotedString (cid: IOChan.ChanId;  string: ARRAY OF CHAR);

    (* Writes a string to a file, surrounded by quotation marks. *)

    BEGIN
        FWriteChar (cid, '"');
        IOChan.RawWrite (cid, ADR(string), LENGTH(string));
        FWriteChar (cid, '"');
    END FWriteQuotedString;

(************************************************************************)

PROCEDURE FWriteCard (cid: IOChan.ChanId;  number: CARDINAL);

    (* Writes a cardinal number to a file.  *)

    BEGIN
        IF number > 9 THEN
            FWriteCard (cid, number DIV 10);
        END (*IF*);
        FWriteChar (cid, CHR(ORD('0') + number MOD 10));
    END FWriteCard;

(************************************************************************)

PROCEDURE FWriteLn (cid: IOChan.ChanId);

    (* Writes end-of-line to the file. *)

    TYPE TwoChar = ARRAY [0..1] OF CHAR;
    CONST CRLF = TwoChar {CHR(13), CHR(10)};

    BEGIN
        WriteRaw (cid, CRLF, 2);
    END FWriteLn;

(************************************************************************)

PROCEDURE OpenPRMFile (VAR (*OUT*) cid: IOChan.ChanId;
                       username: ARRAY OF CHAR): BOOLEAN;

    VAR filename, BAKname: FileNameString;
        result: ChanConsts.OpenResults;  dummy: BOOLEAN;

    BEGIN
        Strings.Assign (username, filename);
        Strings.Append (".PRM", filename);
        SeqFile.OpenWrite (cid, filename, ChanConsts.write+ChanConsts.raw, result);
        IF result = ChanConsts.fileExists THEN
            Strings.Assign (username, BAKname);
            Strings.Append (".BAK", BAKname);
            FileSys.Remove (BAKname, dummy);
            FileSys.Rename (filename, BAKname, dummy);
            SeqFile.OpenWrite (cid, filename, ChanConsts.write+ChanConsts.raw, result);
        END (*IF*);
        RETURN result = ChanConsts.opened;
    END OpenPRMFile;

(************************************************************************)
(*                PRETTYPRINTING OF A USER'S DIRECTORY DATA             *)
(************************************************************************)

PROCEDURE WriteOneChar (cid: IOChan.ChanId;  ch: CHAR);

    (* Writes a single character to the file. *)

    BEGIN
        IOChan.RawWrite (cid, ADR(ch), 1);
    END WriteOneChar;

(************************************************************************)

PROCEDURE NewLine (cid: IOChan.ChanId;  indent: CARDINAL);

    (* Writes a CRLF to the file, followed by indent space characters. *)

    VAR j: CARDINAL;

    BEGIN
        FWriteLn (cid);
        FOR j := 1 TO indent DO
            WriteOneChar (cid, ' ');
        END (*FOR*);
    END NewLine;

(************************************************************************)

PROCEDURE CopyString (cid: IOChan.ChanId;  LookForQuotes: BOOLEAN;
                      Stoppers: CharSet;  VAR (*IN*) Buffer: ARRAY OF CHAR;
                      VAR (*INOUT*) pos: CARDINAL;  BufferSize: CARDINAL);

    (* Writes from Buffer to file, starting at Buffer[pos], and         *)
    (* stopping when we run off the end of the buffer or when           *)
    (* Buffer[pos] contains a character in Stoppers.  If LookForQuotes  *)
    (* is TRUE then we don't look for stopper characters when we're     *)
    (* inside a quoted string.                                          *)

    CONST QuoteChars = CharSet {'"', "'"};

    VAR ch, delimiter: CHAR;

    BEGIN
        WHILE (pos < BufferSize) AND NOT (Buffer[pos] IN Stoppers) DO
            ch := Buffer[pos];  INC (pos);
            WriteOneChar (cid, ch);
            IF LookForQuotes AND (ch IN QuoteChars) THEN
                delimiter := ch;
                CopyString (cid, FALSE, CharSet{delimiter},
                            Buffer, pos, BufferSize);
                IF (pos < BufferSize) AND (Buffer[pos] = delimiter) THEN
                    WriteOneChar (cid, delimiter);
                    INC (pos);
                END (*IF*);
            END (*IF*);
        END (*WHILE*);
    END CopyString;

(************************************************************************)

PROCEDURE WriteDirectoryData (cid: IOChan.ChanId;  indent: CARDINAL;
                      VAR (*IN*) Buffer: ARRAY OF CHAR;
                      VAR (*INOUT*) pos: CARDINAL;  BufferSize: CARDINAL);
                                                                  FORWARD;

(************************************************************************)

PROCEDURE WriteDirectoryList (cid: IOChan.ChanId;  indent: CARDINAL;
                      VAR (*IN*) Buffer: ARRAY OF CHAR;
                      VAR (*INOUT*) pos: CARDINAL;  BufferSize: CARDINAL);

    BEGIN
        WHILE (pos < BufferSize) AND (Buffer[pos] <> ')') DO
            WriteDirectoryData (cid, indent, Buffer, pos, BufferSize);
            IF (pos < BufferSize) AND (Buffer[pos] = ',') THEN
                WriteOneChar (cid, ',');  INC(pos);
                NewLine (cid, indent);
            END (*IF*);
        END (*WHILE*);
    END WriteDirectoryList;

(************************************************************************)

PROCEDURE WriteDirectoryData (cid: IOChan.ChanId;  indent: CARDINAL;
                      VAR (*IN*) Buffer: ARRAY OF CHAR;
                      VAR (*INOUT*) pos: CARDINAL;  BufferSize: CARDINAL);

    (* Writes the data for one directory to the file, starting at       *)
    (* Buffer[pos].                                                     *)

    BEGIN
        CopyString (cid, TRUE, CharSet{',', ')', '('}, Buffer, pos, BufferSize);
        IF (pos < BufferSize) AND (Buffer[pos] = '(') THEN
            NewLine (cid, indent+3);
            WriteOneChar (cid, '(');  INC(pos);
            WriteDirectoryList (cid, indent+4, Buffer, pos, BufferSize);
            IF (pos < BufferSize) AND (Buffer[pos] = ')') THEN
                INC(pos);
            END (*IF*);
            NewLine (cid, indent+3);
            WriteOneChar (cid, ')');
        END (*IF*);
    END WriteDirectoryData;

(************************************************************************)
(*                        PERFORMING THE CONVERSION                     *)
(************************************************************************)

PROCEDURE WriteVolumeData (cid: IOChan.ChanId;
                 VAR (*IN*) Buffer: ARRAY OF CHAR;  BufferSize: CARDINAL);

    (* This is a formatted dump of what's in Buffer.  *)

    VAR pos: CARDINAL;

    BEGIN
        pos := 0;
        WriteDirectoryData (cid, 0, Buffer, pos, BufferSize);
        NewLine (cid, 0);
    END WriteVolumeData;

(************************************************************************)

PROCEDURE ConvertOneUser (username: ARRAY OF CHAR);

    (* Converts one INI file entry to a PRM file. *)

    TYPE BufferIndex = [0..65535];
        CategoryMap = ARRAY UserCategory OF CHAR;

    CONST CategoryCode = CategoryMap {'?', 'N', 'G', 'U', 'M'};

    VAR hini: OS2.HINI;  cid: IOChan.ChanId;
        category: UserCategory;
        password: NameString;
        BufferSize, UserLimit: CARDINAL;
        bufptr: POINTER TO ARRAY BufferIndex OF CHAR;
        ch: CHAR;

    BEGIN
        STextIO.WriteString ("Converting ");
        STextIO.WriteString (username);
        STextIO.WriteLn;
        IF OpenPRMFile (cid, username) THEN
            hini := OS2.PrfOpenProfile (hab, "ftpd.ini");

            BufferSize := SIZE(UserCategory);
            category := NormalUser;
            UserLimit := MAX(CARDINAL);
            OS2.PrfQueryProfileData (hini, username, "Category",
                                            ADR(category), BufferSize);
            ch := CategoryCode[category];
            FWriteChar (cid, ch);
            FWriteLn (cid);

            BufferSize := SIZE(NameString);
            OS2.PrfQueryProfileData (hini, username, "Password", ADR(password), BufferSize);
            FWriteQuotedString (cid, password);
            FWriteLn (cid);

            BufferSize := SIZE(CARDINAL);
            OS2.PrfQueryProfileData (hini, username, "UserLimit", ADR(UserLimit), BufferSize);
            FWriteCard (cid, UserLimit);
            FWriteLn (cid);

            IF OS2.PrfQueryProfileSize (hini, username, "Volume", BufferSize)
                      AND (BufferSize > 0) THEN
                ALLOCATE (bufptr, BufferSize);
                OS2.PrfQueryProfileData (hini, username, "Volume", bufptr, BufferSize);
                WriteVolumeData (cid, bufptr^, BufferSize);
                DEALLOCATE (bufptr, BufferSize);
            END (*IF*);

            OS2.PrfCloseProfile (hini);
            SeqFile.Close (cid);
        ELSE
            STextIO.WriteString ("Can't create ");
            STextIO.WriteString (username);
            STextIO.WriteString (".PRM");
            STextIO.WriteLn;
        END (*IF*);
    END ConvertOneUser;

(********************************************************************************)

PROCEDURE WildcardMatch (string, mask: ARRAY OF CHAR): BOOLEAN;

    (* Checks for string=mask, except that mask is allowed to contain '*' and   *)
    (* '?' wildcard characters.  In this version, '*' is allowed only at the end.*)

    VAR k: CARDINAL;

    BEGIN
        k := 0;
        LOOP
            IF (k > HIGH(mask)) OR (mask[k] = Nul) THEN
                RETURN (k > HIGH(string)) OR (string[k] = Nul);
            ELSIF mask[k] = '*' THEN
                RETURN TRUE;
            ELSIF (k > HIGH(string)) OR (string[k] = Nul) THEN
                RETURN FALSE;
            ELSIF mask[k] = '?' THEN
                INC (k);
            ELSIF CAP(string[k]) <> CAP(mask[k]) THEN
                RETURN FALSE;
            ELSE
                INC (k);
            END (*IF*);
        END (*LOOP*);
    END WildcardMatch;

(********************************************************************************)

PROCEDURE GetParameter (VAR (*OUT*) result: ARRAY OF CHAR);

    (* Picks up program argument from the command line. *)

    VAR args: IOChan.ChanId;  j: CARDINAL;

    BEGIN
        args := ArgChan();
        IF IsArgPresent() THEN
            TextIO.ReadString (args, result);
            j := LENGTH (result);
        ELSE
            j := 0;
        END (*IF*);

        (* Strip trailing spaces. *)

        WHILE (j > 0) AND (result[j-1] = ' ') DO
            DEC (j);
        END (*WHILE*);
        result[j] := CHR(0);
    END GetParameter;

(************************************************************************)

PROCEDURE PerformTheConversions;

    (* Reads command-line argument, converts all the users that match. *)

    TYPE BufferIndex = [0..65535];

    VAR mask: ARRAY [0..127] OF CHAR;
        hini: OS2.HINI;
        BufferSize: CARDINAL;
        bufptr: POINTER TO ARRAY BufferIndex OF CHAR;
        index: BufferIndex;
        Name: ARRAY [0..31] OF CHAR;
        j: [0..31];

    BEGIN
        GetParameter (mask);
        hini := OS2.PrfOpenProfile (hab, "ftpd.ini");
        IF (hini = OS2.NULLHANDLE)
                  OR NOT OS2.PrfQueryProfileSize (hini, NIL, NIL, BufferSize)
                  OR (BufferSize = 0) THEN
            STextIO.WriteString ("Nothing to convert");
            STextIO.WriteLn;
            RETURN;
        END (*IF*);
        ALLOCATE (bufptr, BufferSize);
        OS2.PrfQueryProfileData (hini, NIL, NIL, bufptr, BufferSize);
        OS2.PrfCloseProfile (hini);

        index := 0;
        LOOP
            IF index >= BufferSize THEN
                EXIT (*LOOP*);
            END (*IF*);
            j := 0;
            LOOP
                Name[j] := bufptr^[index];
                INC(index);  INC(j);
                IF (Name[j-1] = Nul) OR (index >= BufferSize) THEN
                    EXIT (*LOOP*);
                END (*IF*);
            END (*LOOP*);
            IF Name[0] = Nul THEN
                EXIT (*LOOP*);
            END (*IF*);
            IF WildcardMatch (Name, mask) AND NOT WildcardMatch (Name, '$SYS') THEN
                ConvertOneUser (Name);
            END (*IF*);
        END (*LOOP*);

        DEALLOCATE (bufptr, BufferSize);

    END PerformTheConversions;

(********************************************************************************)
(*                               MAIN PROGRAM                                   *)
(********************************************************************************)

BEGIN
    hab := OS2.WinInitialize (0);
    PerformTheConversions;
FINALLY
    IF hab <> OS2.NULLHANDLE THEN
        OS2.WinTerminate (hab);
    END (*IF*);
END StorePRM.

