MODULE LoadPRM;

        (********************************************************)
        (*                                                      *)
        (*  Program to load data from PRM files into ftpd.ini   *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            5 March 1998                    *)
        (*  Last edited:        9 March 2000                    *)
        (*  Status:             Working                         *)
        (*                                                      *)
        (********************************************************)

(************************************************************************)
(*                                                                      *)
(*     Syntax for the directory permissions in the PRM file:            *)
(*                                                                      *)
(*     <result>     ->  <dirlist>                                       *)
(*     <dirlist>    ->  <diritem> { , <diritem> }*                      *)
(*     <diritem>    ->  <dirname> <dirrule>                             *)
(*     <dirname>    ->  <namestring>  |  <namestring> = <namestring>    *)
(*     <dirrule>   -> { <permission> }* { ( <dirlist> ) }               *)
(*     <dirlist>   ->  <diritem> { , <diritem> }*                       *)
(*     <dirlist>   -> { <diritem> }+                                    *)
(*     <permission> ->  V+ | V- | R+ | R- | W- | W+ | D- | D+ | N- | N+ *)
(*     <namestring> ->  <string1> | " <string2> " | ' <string3> '       *)
(*     <string1>   -> any string not including space char               *)
(*     <string2>   -> any string not including double quote             *)
(*     <string3>   -> any string not including single quote             *)
(*                                                                      *)
(*  Notation: {} means optional, {}* means zero or more, {}+ means      *)
(*            one or more.                                              *)
(*                                                                      *)
(************************************************************************)

FROM SYSTEM IMPORT
    (* type *)  LOC,
    (* proc *)  ADR;

IMPORT IOChan, ChanConsts, IOConsts, TextIO, STextIO, Strings, OS2,
       RndFile, FIO;

FROM ProgramArgs IMPORT
    (* proc *)  ArgChan, IsArgPresent;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

(************************************************************************)

CONST Nul = CHR(0);

TYPE
    CharSet = SET OF CHAR;
    PassString = ARRAY [0..31] OF CHAR;
    FileNameString = ARRAY [0..255] OF CHAR;

    (* The following declarations relate to the data structure that we keep,    *)
    (* for a logged-in user, to show which directories and files are accessible.*)
    (* For each directory, we have a linked list of subdirectories.             *)

    Permission = (Visible, AllowRead, AllowWrite, AllowDelete, AllowRename);
    PermissionSet = SET OF Permission;

    DirEntryPtr = POINTER TO DirEntry;

    DirEntry = RECORD
                   flags: PermissionSet;
                   parent, FirstChild, next: DirEntryPtr;
                   name: FileNameString;
                   link: POINTER TO FileNameString;
               END (*RECORD*);

    UserCategory = (NoSuchUser, NoPasswordNeeded, GuestUser, NormalUser, Manager);

    (* The fields in a UserPermission record have the following meanings.       *)
    (*      username     The user's login name                                  *)
    (*      Password     The user's password                                    *)
    (*      category     The user's category                                    *)
    (*      TreeRoot     Root of the user's permission tree.                    *)

    User = POINTER TO UserPermission;

    UserPermission = RECORD
                         username, Password: PassString;
                         category: UserCategory;
                         UserLimitSpecified: BOOLEAN;
                         UserLimit: CARDINAL;
                         TreeRoot: DirEntryPtr;
                     END (*RECORD*);

(************************************************************************)

VAR
    (* Anchor block handle for this application.  *)

    hab: OS2.HAB;

(********************************************************************************)
(*                          TREE OUTPUT, FOR TESTING                            *)
(********************************************************************************)

(*
PROCEDURE Indent (level: CARDINAL);

    (* Writes some spaces. *)

    VAR j: CARDINAL;

    BEGIN
        FOR j := 1 TO 3*level DO
            STextIO.WriteChar (" ");
        END (*FOR*);
    END Indent;

(********************************************************************************)

PROCEDURE DumpList (p: DirEntryPtr;  level: CARDINAL);

    (* Writes out a visual representation of a linked list of trees.  The       *)
    (* level parameter controls the indentation.                                *)

    BEGIN
        REPEAT
            WITH p^ DO
                Indent (level);  WriteString (name);
                IF link <> NIL THEN
                    WriteString (" = ");  WriteString (link^);
                END (*IF*);
                WriteString ("  ");
                IF Visible IN flags THEN STextIO.WriteChar ('V') ELSE STextIO.WriteChar (' ')  END(*IF*);
                IF AllowRead IN flags THEN STextIO.WriteChar ('R') ELSE STextIO.WriteChar (' ')  END(*IF*);
                IF AllowWrite IN flags THEN STextIO.WriteChar ('W') ELSE STextIO.WriteChar (' ')  END(*IF*);
                IF AllowDelete IN flags THEN STextIO.WriteChar ('D') ELSE STextIO.WriteChar (' ')  END(*IF*);
                IF AllowRenamee IN flags THEN STextIO.WriteChar ('N') ELSE STextIO.WriteChar (' ')  END(*IF*);
                STextIO.WriteLn;
                IF FirstChild <> NIL THEN
                    DumpList (FirstChild, level+1);
                END (*IF*);
            END (*WITH*);
            p := p^.next;
        UNTIL p = NIL;
    END DumpList;

(********************************************************************************)
(*                   DUMP TO SCREEN OF ALL USER DATA                            *)
(********************************************************************************)

PROCEDURE DumpUserData (U: User);

    (* Writes out the permissions of this user.  *)

    BEGIN
        WriteString ("Password is ");  WriteString (U^.Password);  STextIO.WriteLn;
        DumpList (U^.TreeRoot, 0);
    END DumpUserData;
*)

(********************************************************************************)
(*                                PARSER                                        *)
(********************************************************************************)

PROCEDURE ToLower (VAR (*INOUT*) string: ARRAY OF CHAR);

    (* Converts all letters in string to lower case. *)

    TYPE CharSet = SET OF CHAR;

    CONST shift = ORD('a') - ORD('A');

    VAR j: CARDINAL;

    BEGIN
        FOR j := 0 TO HIGH(string) DO
            IF string[j] IN CharSet {'A'..'Z'} THEN
                INC (string[j], shift);
            END (*IF*);
        END (*FOR*);
    END ToLower;

(********************************************************************************)

PROCEDURE ReadUserData (filename: ARRAY OF CHAR): User;

    (* Fetches the password, etc., from the specified PRM file.  Returns with   *)
    (* category = NoSuchUser and result = NIL if the user's data could not be   *)
    (* found.                                                                   *)

    CONST Space = " ";
          UsingINIdata = FALSE;

    CONST Digits = CharSet {'0'..'9'};

    VAR NextChar: CHAR;  InComment: BOOLEAN;  cid: IOChan.ChanId;

    (****************************************************************************)

    PROCEDURE Scan;

        (* Puts the next input character into variable NextChar. *)

        CONST CR = CHR(13);  LF = CHR(10);  CtrlZ = CHR(26);

        VAR charsRead: CARDINAL;  status: IOConsts.ReadResults;

        BEGIN
            IOChan.RawRead (cid, ADR(NextChar), 1, charsRead);
            status := IOChan.ReadResult (cid);

            IF status = IOConsts.endOfInput THEN
                NextChar := Nul;
            ELSIF status <> IOConsts.allRight THEN
                STextIO.WriteString ("Failure to read next character.");
                STextIO.WriteLn;
            ELSIF NextChar = CtrlZ THEN
                NextChar := Nul;
            ELSIF (NextChar = CR) OR (NextChar = LF) THEN
                NextChar := Space;  InComment := FALSE;
            ELSIF NextChar = '%' THEN
                InComment := TRUE;
            END (*IF*);
            IF InComment THEN NextChar := Space
            END (*IF*);
        END Scan;

    (****************************************************************************)

    PROCEDURE SkipBlanks;

        BEGIN
            WHILE NextChar = Space DO Scan END(*WHILE*);
        END SkipBlanks;

    (****************************************************************************)

    PROCEDURE LoadString (VAR (*OUT*) result: ARRAY OF CHAR;  Stoppers: CharSet);

        (* Reads a string up to but not including a character in Stoppers.      *)

        VAR j: CARDINAL;

        BEGIN
            j := 0;
            WHILE (j <= HIGH(result)) AND NOT (NextChar IN Stoppers) DO
                result[j] := NextChar;  INC(j);
                Scan;
            END (*WHILE*);
            IF j <= HIGH(result) THEN result[j] := Nul END(*IF*);
        END LoadString;

    (****************************************************************************)

    PROCEDURE NameString (VAR (*OUT*) result: ARRAY OF CHAR;  Stoppers: CharSet);

        (*     <namestring> ->  <string1> | " <string2> " | ' <string3> '       *)
        (*     <string1>   -> any string not including space char               *)
        (*     <string2>   -> any string not including double quote             *)
        (*     <string3>   -> any string not including single quote             *)
        (* The strings in <string1> also may not contain characters in Stoppers.*)

        CONST SingleQuote = "'";  DoubleQuote = '"';

        VAR Delimiter: CHAR;

        BEGIN
            INCL (Stoppers, Nul);
            IF (NextChar = SingleQuote) OR (NextChar = DoubleQuote) THEN
                Delimiter := NextChar;
                Stoppers := CharSet {Nul, Delimiter};
                Scan;
                LoadString (result, Stoppers);
                IF NextChar = Delimiter THEN
                    Scan;
                ELSE
                    STextIO.WriteString ("Mismatched quotation marks.");
                    STextIO.WriteLn;
                END (*IF*);
            ELSE
                INCL (Stoppers, Space);
                LoadString (result, Stoppers);
            END (*IF*);
        END NameString;

    (****************************************************************************)

    PROCEDURE DirItem (mother: DirEntryPtr): DirEntryPtr;  FORWARD;

    (****************************************************************************)

    PROCEDURE DirList (mother: DirEntryPtr): DirEntryPtr;

        (*     <dirlist>   ->  <diritem> { , <diritem> }*       *)
        (* Result returned: a linked list of directory nodes.   *)

        CONST Comma = ",";

        VAR result, lastnode: DirEntryPtr;

        BEGIN
            result := DirItem (mother);  lastnode := result;
            SkipBlanks;
            WHILE (NextChar = Comma) OR (NextChar = ';') DO
                Scan;  SkipBlanks;
                lastnode^.next := DirItem (mother);
                lastnode := lastnode^.next;
                SkipBlanks;
            END (*WHILE*);
            RETURN result;
        END DirList;

    (****************************************************************************)

    PROCEDURE DirRule (pnode: DirEntryPtr);

        (* Fills in the permissions and subdirectory info for pnode.            *)
        (*     <dirrule>   -> { <permission> }* { ( <dirlist> ) }               *)
        (*     <permission> ->  V+ | V- | R+ | R- | W- | W+ | D- | D+ | N- | N+ *)

        VAR option: Permission;

        BEGIN
            (* Default flags are inherited from parent. *)

            IF pnode^.parent = NIL THEN
                pnode^.flags := PermissionSet {Visible, AllowRead}
            ELSE pnode^.flags := pnode^.parent^.flags
            END (*IF*);

            (* Look for optional permission codes. *)

            WHILE CAP(NextChar) IN CharSet {'V', 'R', 'W', 'D', 'N'} DO
                CASE CAP(NextChar) OF
                  | 'V':  option := Visible;
                  | 'R':  option := AllowRead;
                  | 'W':  option := AllowWrite;
                  | 'D':  option := AllowDelete;
                  | 'N':  option := AllowRename;
                END (*CASE*);
                Scan;
                IF NextChar = '-' THEN
                    Scan;
                    EXCL (pnode^.flags, option);
                ELSIF NextChar = '+' THEN
                    Scan;
                    INCL (pnode^.flags, option);
                ELSE
                    STextIO.WriteString ("Option modifier must be + or -");
                    STextIO.WriteLn;
                END (*IF*);

                SkipBlanks;
            END (*WHILE*);

            (* Look for optional list of subdirectories. *)

            IF NextChar = '(' THEN
                Scan;  SkipBlanks;
                IF NextChar <> ')' THEN
                    pnode^.FirstChild := DirList(pnode);
                END (*IF*);
                IF NextChar = ')' THEN
                    Scan;
                ELSE
                    STextIO.WriteString ("Missing closing parenthesis.");
                    STextIO.WriteLn;
                END (*IF*);
            END (*IF*);

        END DirRule;

    (****************************************************************************)

    PROCEDURE DirName (mother: DirEntryPtr): DirEntryPtr;

        (*     <dirname>    ->  <namestring>  |  <namestring> = <namestring>    *)
        (* The alternative with the '=' sign means that we have to create       *)
        (* a symbolic link.                                                     *)

        VAR result: DirEntryPtr;  k: CARDINAL;

        BEGIN
            NEW (result);
            WITH result^ DO
                flags := PermissionSet {Visible, AllowRead};
                parent := mother;  FirstChild := NIL;  next := NIL;
                NameString (name, CharSet{'=', ',', '('} );
                link := NIL;
                SkipBlanks;
                IF NextChar = '=' THEN
                    Scan;  SkipBlanks;
                    NEW (link);
                    NameString (link^, CharSet{});

                    (* Transition arrangement: get rid of the trailing '/'.  *)

                    k := LENGTH (link^);
                    IF k > 0 THEN
                        DEC (k);
                        IF (link^[k] = '/') OR (link^[k] = '\') THEN
                            link^[k] := Nul;
                        END (*IF*);
                    END (*IF*);
                END (*IF*);
            END (*WITH*);
            RETURN result;
        END DirName;

    (****************************************************************************)

    PROCEDURE DirItem (mother: DirEntryPtr): DirEntryPtr;

        (*     <diritem>   -> <dirname> <dirrule>                            *)

        VAR result: DirEntryPtr;

        BEGIN
            result := DirName(mother);
            SkipBlanks;
            DirRule (result);
            RETURN result;
        END DirItem;

    (************************************************************************)

    PROCEDURE GetCard(): CARDINAL;

        (* Reads a cardinal number from the input stream. *)

        VAR result: CARDINAL;

        BEGIN
            result := 0;
            WHILE NextChar IN Digits DO
                result := 10*result;
                INC (result, ORD(NextChar) - ORD('0'));
                Scan;
            END (*WHILE*);
            RETURN result;
        EXCEPT
            WHILE NextChar IN Digits DO
                Scan;
            END (*WHILE*);
            RETURN MAX(CARDINAL);
        END GetCard;

    (************************************************************************)
    (*                     BODY OF READUSERDATA                             *)
    (*                                                                      *)
    (*     <result>  ->  <userclass> <password> <userlimit>                 *)
    (*                                           { <volumeinfo> }*          *)
    (*     <userclass> -> G | U | N | M                                     *)
    (*     <password>  -> <namestring>                                      *)
    (*                                                                      *)
    (* Note: for simplicity, a password must always be specified, even if   *)
    (* it is not going to be used.                                          *)
    (*                                                                      *)
    (************************************************************************)

    VAR result: User;  status: ChanConsts.OpenResults;
        root: DirEntryPtr;
        position: CARDINAL;  found: BOOLEAN;

    BEGIN       (* Body of ReadUserData *)

        InComment := FALSE;

        RndFile.OpenOld (cid, filename, ChanConsts.read+ChanConsts.old+ChanConsts.raw, status);
        IF status <> ChanConsts.opened THEN
            RETURN NIL;
        END (*IF*);

        NEW (result);  result^.TreeRoot := NIL;
        Strings.Assign (filename, result^.username);
        Strings.FindPrev (".", result^.username, LENGTH(result^.username),
                                                                found, position);
        IF found THEN
            result^.username[position] := Nul;
        END (*IF*);
        ToLower (result^.username);

        result^.category := NoSuchUser;

        Scan;  SkipBlanks;

        (* Decode the user class code. *)

        IF NextChar = 'G' THEN
            Scan;  result^.category := GuestUser;
        ELSIF NextChar = 'U' THEN
            Scan;  result^.category := NormalUser;
        ELSIF NextChar = 'N' THEN
            Scan;  result^.category := NoPasswordNeeded;
        ELSIF NextChar = 'M' THEN
            Scan;  result^.category := Manager;
        ELSE
            STextIO.WriteString ("Unknown user category ");  STextIO.WriteChar (NextChar);
            STextIO.WriteLn;  Scan;
        END (*IF*);

        (* Read the password. *);

        SkipBlanks;  NameString (result^.Password, CharSet{});

        (* Read the user limit, if present. *)

        SkipBlanks;
        result^.UserLimitSpecified := NextChar IN Digits;
        result^.UserLimit := GetCard();

        (* Give the user a root directory. *)

        NEW (root);
        result^.TreeRoot := root;
        WITH root^ DO
            flags := PermissionSet {Visible, AllowRead};
            name := "";
            parent := NIL;  FirstChild := NIL;  next := NIL;
            NEW (link);  link^ := "";
        END (*WITH*);

        (* Load the volume information. *)

        SkipBlanks;
        root^.FirstChild := DirList (root);

        (* Remove redundant top-level pseudo-directories. *)

        WHILE (root <> NIL) AND (root^.link <> NIL) AND (root^.link^[0] = Nul)
                                  AND (root^.FirstChild <> NIL)
                                  AND (root^.FirstChild^.next = NIL) DO
            DISPOSE (root^.link);
            root := root^.FirstChild;
            DISPOSE (result^.TreeRoot);
            result^.TreeRoot := root;
            root^.parent := NIL;
        END (*WHILE*);

        (* Close the data file. *)

        RndFile.Close (cid);
        RETURN result;

    END ReadUserData;

(********************************************************************************)
(*                        DISCARDING A USER RECORD                              *)
(********************************************************************************)

PROCEDURE KillList (VAR (*INOUT*) p: DirEntryPtr);

    (* Discards a directory tree.  *)

    VAR q: DirEntryPtr;

    BEGIN
        WHILE p <> NIL DO
            KillList (p^.FirstChild);
            q := p^.next;
            IF p^.link <> NIL THEN
                DISPOSE (p^.link);
            END (*IF*);
            DISPOSE (p);
            p := q;
        END (*IF*);
    END KillList;

(********************************************************************************)

PROCEDURE DestroyUserData (VAR (*INOUT*) U: User);

    (* Discards the data structure.  *)

    BEGIN
        KillList (U^.TreeRoot);
        DISPOSE (U);
    END DestroyUserData;

(************************************************************************)
(*         ENCODING THE VOLUME INFORMATION AS AN ASCII STRING           *)
(************************************************************************)

PROCEDURE AppendString (string: ARRAY OF CHAR;
                                       VAR (*INOUT*) result: ARRAY OF CHAR;
                                       VAR (*INOUT*) index: CARDINAL);

    (* Puts string into the result array, starting at result[index].    *)
    (* On return index has been updated to the next unused array slot.  *)

    VAR j, length: CARDINAL;

    BEGIN
        length := LENGTH(string);
        IF length > 0 THEN
            FOR j := 0 TO length-1 DO
                result[index] := string[j];  INC(index);
            END (*FOR*);
        END (*IF*);
    END AppendString;

(************************************************************************)

PROCEDURE AppendQuotedString (string: ARRAY OF CHAR;
                                       VAR (*INOUT*) result: ARRAY OF CHAR;
                                       VAR (*INOUT*) index: CARDINAL);

    (* Puts quotation marks around string, and puts it into the result  *)
    (* array, starting at result[index].  On return index has been      *)
    (* updated to the next unused array slot.                           *)

    BEGIN
        result[index] := '"';  INC(index);
        AppendString (string, result, index);
        result[index] := '"';  INC(index);
    END AppendQuotedString;

(************************************************************************)

PROCEDURE StoreTree (D: DirEntryPtr;  DefaultFlags: PermissionSet;
                                      VAR (*INOUT*) result: ARRAY OF CHAR;
                                      VAR (*INOUT*) index: CARDINAL);
                                                                 FORWARD;

(************************************************************************)

PROCEDURE SizeOfTreeData (D: DirEntryPtr;
                          DefaultFlags: PermissionSet): CARDINAL;
                                                                 FORWARD;

(************************************************************************)

PROCEDURE StorePermAndSubdir (D: DirEntryPtr;  DefaultFlags: PermissionSet;
                              VAR (*INOUT*) result: ARRAY OF CHAR;
                              VAR (*INOUT*) index: CARDINAL);

    (* Puts the flags for D, then information for all subdirectories    *)
    (* of D, into the result array, starting at result[index].          *)
    (* On return index has been updated to the next unused array slot.  *)

    TYPE CodeArray = ARRAY Permission OF CHAR;

    CONST PermissionCode = CodeArray {'V','R','W','D','N'};

    VAR child: DirEntryPtr;  perm: Permission;  difference: PermissionSet;

    BEGIN
        difference := DefaultFlags - D^.flags;
        FOR perm := MIN(Permission) TO MAX(Permission) DO
            IF perm IN difference THEN
                result[index] := PermissionCode[perm];  INC (index);
                result[index] := '-';  INC (index);
            END (*IF*);
        END (*FOR*);
        difference := D^.flags - DefaultFlags;
        FOR perm := MIN(Permission) TO MAX(Permission) DO
            IF perm IN difference THEN
                result[index] := PermissionCode[perm];  INC (index);
                result[index] := '+';  INC (index);
            END (*IF*);
        END (*FOR*);

        child := D^.FirstChild;
        IF child <> NIL THEN
            result[index] := '(';  INC(index);
            REPEAT
                StoreTree (child, D^.flags, result, index);
                child := child^.next;
                IF child <> NIL THEN
                    result[index] := ',';  INC(index);
                END (*IF*);
            UNTIL child = NIL;
            result[index] := ')';  INC(index);
        END (*IF*);

    END StorePermAndSubdir;

(************************************************************************)

PROCEDURE SizeOfPermAndSubdir (D: DirEntryPtr;
                                   DefaultFlags: PermissionSet): CARDINAL;

    (* Works out how much space to allocate for the flags for D, plus   *)
    (* information for all subdirectories of D.                         *)

    VAR size: CARDINAL;  child: DirEntryPtr;
        perm: Permission;  difference: PermissionSet;

    BEGIN
        size := 0;
        difference := DefaultFlags / D^.flags;
        FOR perm := MIN(Permission) TO MAX(Permission) DO
            IF perm IN difference THEN
                INC (size, 2);
            END (*IF*);
        END (*FOR*);

        child := D^.FirstChild;
        IF child <> NIL THEN
            INC (size);
            REPEAT
                INC (size, 1 + SizeOfTreeData (child, D^.flags));
                child := child^.next;
            UNTIL child = NIL;
        END (*IF*);

        RETURN size;

    END SizeOfPermAndSubdir;

(************************************************************************)

PROCEDURE StoreTree (D: DirEntryPtr;  DefaultFlags: PermissionSet;
                                      VAR (*INOUT*) result: ARRAY OF CHAR;
                                      VAR (*INOUT*) index: CARDINAL);

    (* Puts the data for the tree whose root is D into the result       *)
    (* array, starting at result[index].  On return index has been      *)
    (* updated to the next unused array slot.                           *)

    BEGIN
        AppendQuotedString (D^.name, result, index);
        IF D^.link <> NIL THEN
            result[index] := '=';  INC(index);
            AppendQuotedString (D^.link^, result, index);
        END (*IF*);
        StorePermAndSubdir (D, DefaultFlags, result, index);
    END StoreTree;

(************************************************************************)

PROCEDURE SizeOfTreeData (D: DirEntryPtr;
                          DefaultFlags: PermissionSet): CARDINAL;

    (* Works out how much space to allocate to hold all the data for    *)
    (* the tree whose root is D.                                        *)

    VAR result: CARDINAL;

    BEGIN
        result := LENGTH(D^.name) + 2 + SizeOfPermAndSubdir (D, DefaultFlags);
        IF D^.link <> NIL THEN
            INC (result, LENGTH(D^.link^) + 3);
        END (*IF*);
        RETURN result;
    END SizeOfTreeData;

(************************************************************************)
(*                      MAIN CONVERSION PROCEDURES                      *)
(************************************************************************)

PROCEDURE WriteUserData (U: User): BOOLEAN;

    TYPE Big = [0..65535];

    VAR hini: OS2.HINI;  size, index: CARDINAL;
        bufptr: POINTER TO ARRAY Big OF CHAR;

    BEGIN
        hini := OS2.PrfOpenProfile (hab, "ftpd.ini");
        IF hini = OS2.NULLHANDLE THEN
            RETURN FALSE;
        END (*IF*);
        OS2.PrfWriteProfileData (hini, U^.username, "Category",
                                ADR(U^.category), SIZE(UserCategory));
        OS2.PrfWriteProfileData (hini, U^.username, "Password",
                                ADR(U^.Password), SIZE(PassString));
        IF U^.UserLimitSpecified THEN
            OS2.PrfWriteProfileData (hini, U^.username, "UserLimit",
                                    ADR(U^.UserLimit), SIZE(CARDINAL));
        END (*IF*);

        size := SizeOfTreeData (U^.TreeRoot,
                                PermissionSet {Visible, AllowRead});
        IF size = 0 THEN
            bufptr := NIL;
        ELSE
            ALLOCATE (bufptr, size);

            (* Store all volume information in the bufptr^ array. *)

            index := 0;
            StoreTree (U^.TreeRoot, PermissionSet {Visible, AllowRead},
                                                 bufptr^, index);
        END (*IF*);

        (* Copy the result from bufptr^ to the INI file. *)

        OS2.PrfWriteProfileData (hini, U^.username, "Volume", bufptr, size);
        OS2.PrfCloseProfile (hini);
        IF size > 0 THEN
            DEALLOCATE (bufptr, size);
        END (*IF*);
        RETURN TRUE;

    END WriteUserData;

(************************************************************************)

PROCEDURE ConvertOneUser (filename: ARRAY OF CHAR);

    (* Converts one PRM file to an INI file entry. *)

    VAR U: User;

    BEGIN
        U := ReadUserData (filename);
        IF U <> NIL THEN
            IF WriteUserData (U) THEN
                STextIO.WriteString ("Converted ");
            ELSE
                STextIO.WriteString ("Failed to convert ");
            END (*IF*);
            STextIO.WriteString (filename);
            STextIO.WriteLn;
            DestroyUserData (U);
        END (*IF*);
    END ConvertOneUser;

(********************************************************************************)

PROCEDURE GetParameter (VAR (*OUT*) result: ARRAY OF CHAR);

    (* Picks up program argument from the command line. *)

    VAR args: IOChan.ChanId;  j: CARDINAL;

    BEGIN
        args := ArgChan();
        IF IsArgPresent() THEN
            TextIO.ReadString (args, result);
            j := LENGTH (result);
        ELSE
            j := 0;
        END (*IF*);

        (* Strip trailing spaces. *)

        WHILE (j > 0) AND (result[j-1] = ' ') DO
            DEC (j);
        END (*WHILE*);
        result[j] := CHR(0);

    END GetParameter;

(************************************************************************)

PROCEDURE PerformTheConversions;

    (* Reads command-line argument, converts all the PRM files that match. *)

    VAR string: ARRAY [0..127] OF CHAR;
        D: FIO.DirEntry;
        pos: CARDINAL;  found: BOOLEAN;

    BEGIN
        GetParameter (string);
        Strings.FindNext ('.', string, 0, found, pos);
        IF NOT found THEN
            Strings.Append (".PRM", string);
        END (*IF*);

        IF FIO.ReadFirstEntry (string, FIO.FileAttr{}, D) THEN
            REPEAT
                ConvertOneUser (D.name);
            UNTIL NOT FIO.ReadNextEntry (D);
        ELSE
            STextIO.WriteString ("Nothing to convert");
            STextIO.WriteLn;
        END (*IF*);
        FIO.ReadClose (D);

    END PerformTheConversions;

(********************************************************************************)
(*                               MAIN PROGRAM                                   *)
(********************************************************************************)

BEGIN
    hab := OS2.WinInitialize (0);
    PerformTheConversions;
FINALLY
    IF hab <> OS2.NULLHANDLE THEN
        OS2.WinTerminate (hab);
    END (*IF*);
END LoadPRM.

