IMPLEMENTATION MODULE FtpCommands;

        (********************************************************)
        (*                                                      *)
        (*       Command interpreter for ftp server             *)
        (*                                                      *)
        (*  Programmer:         P. Moylan                       *)
        (*  Started:            23 August 1997                  *)
        (*  Last edited:        9 March 2000                    *)
        (*  Status:             Most commands now implemented   *)
        (*                                                      *)
        (********************************************************)

(********************************************************************************)
(*                        COMPLIANCE WITH THE STANDARD                          *)
(********************************************************************************)
(*                                                                              *)
(* The type/mode/structure options I've actually implemented are:               *)
(*                                                                              *)
(*       TYPE - ASCII Non-print (default), IMAGE, LOCAL 8                       *)
(*       MODE - Stream (default)                                                *)
(*       STRUCTURE - File (default), Record                                     *)
(*                                                                              *)
(* This meets the minimum requirements specified in RFC959 and RFC1123.         *)
(*                                                                              *)
(* I have now implemented at least the minimum implementation of RFC 959,       *)
(* as amended by RFC 1123, except that the following commands in the minimum    *)
(* set are either unimplemented or untested:                                    *)
(*                                                                              *)
(*      STAT                                                                    *)
(*                                                                              *)
(* The commmands I have implemented and tested are:                             *)
(*                                                                              *)
(*      ABOR, ACCT, APPE, CDUP, CWD, DELE, HELP, LIST, PWD, MDTM, MKD,          *)
(*      MODE (S only), NLST, NOOP, PASS, PASV, PORT, QUIT, REIN, RETR,          *)
(*      RNFR, RNTO, SIZE, SMNT, STOR, STOU, STRU (F or R), SYST,                *)
(*      TYPE (I or A or L 8), USER                                              *)
(*                                                                              *)
(* The commmands I have implemented but not yet adequately tested are:          *)
(*                                                                              *)
(*      ALLO, REST                                                              *)
(*                                                                              *)
(* The commmands I have not yet implemented are:                                *)
(*                                                                              *)
(*      STAT                                                                    *)
(*                                                                              *)
(* Note also: the commands SIZE and MDTM are used by a lot of clients, even     *)
(* though they're not in the standard, so I've implemented them.  I've also     *)
(* implemented the obsolete commands XCUP, XCWD, XMKD, XPWD, and XRMD,          *)
(* which are used by some pre-1985 clients.                                     *)
(*                                                                              *)
(********************************************************************************)

FROM SYSTEM IMPORT CARD8, CARD16, CARD32, CAST, ADDRESS, ADR;

IMPORT Strings, IOChan, ChanConsts, RndFile, OS2;

FROM Sockets IMPORT
    (* type *)  Socket, SockAddr,
    (* proc *)  send;

FROM Internet IMPORT
    (* type *)  InternetSocketAddress;

FROM InetUtilities IMPORT
    (* proc *)  Synch, ToLower, ConvertCard, AddEOL, StartTransactionLogging;

FROM TransLog IMPORT
    (* type *)  TransactionLogID,
    (* proc *)  LogTransaction;

FROM LoggedOnUsers IMPORT
    (* type *)  ClientDataPointer,
    (* proc *)  StartSession, EndSession, ListAllUsers,
                NewUser, RemoveUser, AbortDataOperations, KillUser;

FROM FtpTransfers IMPORT
    (* type *)  ClientFileInfo,
    (* proc *)  SetLogLevel, PasswordOK, SendMessageFile, SendDirectoryMessage,
                SetPort, SetTransferType, SetTransferMode, EnterPassiveMode,
                SetFileStructure, SetDirectory, SetFileName,
                ListingIsPossible, SendDirectory, FileAvailable, SendFile,
                CanWrite, FetchFile, AppendFile, CanDelete, DeleteFile,
                RenameFile, SetRestartPoint, CanRemoveDirectory,
                DeleteDirectory, MakeDirectory, GetFileDate, GetSize,
                CurrentDirectory, GetCurrentPermissions, AllocateFileSize,
                CreateUniqueFileName, CloseDataPort;

FROM FDUsers IMPORT
    (* type *)  UserCategory;

FROM Keyboard IMPORT
    (* proc *)  StuffKeyboardBuffer;

FROM Semaphores IMPORT
    (* type *)  Semaphore;

FROM Storage IMPORT
    (* proc *)  ALLOCATE, DEALLOCATE;

(********************************************************************************)

CONST
    Nul = CHR(0);

TYPE
    FourChar = ARRAY [0..3] OF CHAR;

    ClientState = (Idle, WaitingForPassword, LoggedIn, MustExit);

    (* The session record.  The fields are:                             *)
    (*     logID       Identifier for transaction logging               *)
    (*     ClientData  Structure set up on opening a session to keep    *)
    (*                 track of the current state of this session.      *)
    (*     user        User handle that is given to us by StartSession  *)
    (*                 and that can be used in FtpTransfer calls.       *)
    (*     socket      The command socket                               *)
    (*     usernumber  A number to use in welcome messages.             *)
    (*     state       To track whether the user is currently logged in.*)
    (*     IsManager   TRUE iff this user is a manager.                 *)
    (*     MSGenabled  TRUE iff this user is willing to accept MSG files.*)

    Session = POINTER TO
                  RECORD
                      logID: TransactionLogID;
                      ClientData: ClientDataPointer;
                      user: ClientFileInfo;
                      socket: Socket;
                      usernumber: CARDINAL;
                      state: ClientState;
                      IsManager, MSGenabled: BOOLEAN;
                  END (*RECORD*);

(********************************************************************************)

VAR
    (* Flag to say whether transactions are to be logged. *)

    TransactionLoggingEnabled: BOOLEAN;

(********************************************************************************)
(*                           TRANSACTION LOGGING                                *)
(********************************************************************************)

PROCEDURE SetTransactionLogLevel (level: CARDINAL);

    (* Option to control transaction logging: 0 for none, 1 for disk,   *)
    (* 2 for screen, 3 for both.                                        *)

    BEGIN
        TransactionLoggingEnabled := level > 0;
        StartTransactionLogging ("FTPTRANS.LOG", level);
    END SetTransactionLogLevel;

(********************************************************************************)
(*                         STARTING A NEW SESSION                               *)
(********************************************************************************)

PROCEDURE OpenSession (CommandSocket: Socket;  UserNumber, IPaddr: CARDINAL;
                         LogID: TransactionLogID;  KeepAlive: Semaphore): Session;

    (* Creates a new session state record.  During lengthy operations           *)
    (* we have to do a Signal(KeepAlive) every so often in order to stop the    *)
    (* session from timing out.                                                 *)

    VAR result: Session;

    BEGIN
        NEW (result);
        WITH result^ DO
            logID := LogID;
            ClientData := StartSession (CommandSocket, UserNumber,
                                        IPaddr, KeepAlive, user);
            socket := CommandSocket;
            state := Idle;
            usernumber := UserNumber;
            IsManager := FALSE;
            MSGenabled := TRUE;
        END (*WITH*);
        RETURN result;
    END OpenSession;

(********************************************************************************)

PROCEDURE CloseSession (S: Session);

    (* Destroys the session state record. *)

    BEGIN
        IF S <> NIL THEN
            IF S^.state <> Idle THEN
                RemoveUser (S^.ClientData);
            END (*IF*);
            EndSession (S^.ClientData);
            DISPOSE (S);
        END (*IF*);
    END CloseSession;

(********************************************************************************)

PROCEDURE KillDataChannel (S: Session);

    (* Aborts the data transfer, if any, now in progress for this session. *)

    BEGIN
        AbortDataOperations (S^.ClientData);
    END KillDataChannel;

(********************************************************************************)
(*                       SENDING REPLY BACK TO CLIENT                           *)
(********************************************************************************)

PROCEDURE Reply2 (session: Session;  message1, message2: ARRAY OF CHAR);

    (* Sends all of message1, followed by message2, followed by end-of-line.    *)
    (* If the operation fails, session^.state is set to MustExit.               *)

    VAR buffer: ARRAY [0..511] OF CHAR;  length: CARDINAL;

    BEGIN
        Strings.Assign (message1, buffer);
        Strings.Append (message2, buffer);
        length := AddEOL (buffer);
        IF send (session^.socket, buffer, length, 0) = MAX(CARDINAL) THEN
            session^.state := MustExit;
        END (*IF*);
        (*Synch (session^.socket);*)
    END Reply2;

(********************************************************************************)

PROCEDURE Reply3 (session: Session;  message1, message2, message3: ARRAY OF CHAR);

    (* Like Reply2, but with an extra string message3 before the end-of-line.   *)

    VAR buffer: ARRAY [0..511] OF CHAR;  length: CARDINAL;

    BEGIN
        Strings.Assign (message1, buffer);
        Strings.Append (message2, buffer);
        Strings.Append (message3, buffer);
        length := AddEOL (buffer);
        IF send (session^.socket, buffer, length, 0) = MAX(CARDINAL) THEN
            session^.state := MustExit;
        END (*IF*);
        (*Synch (session^.socket);*)
    END Reply3;

(********************************************************************************)

PROCEDURE Reply (session: Session;  message: ARRAY OF CHAR);

    (* Like Reply2, except that there is no message2. *)

    VAR buffer: ARRAY [0..511] OF CHAR;  length: CARDINAL;

    BEGIN
        Strings.Assign (message, buffer);
        length := AddEOL (buffer);
        IF send (session^.socket, buffer, length, 0) = MAX(CARDINAL) THEN
            session^.state := MustExit;
        END (*IF*);
        (*Synch (session^.socket);*)
    END Reply;

(********************************************************************************)
(*                     HANDLERS FOR SOME ERROR CONDITIONS                       *)
(********************************************************************************)

PROCEDURE NotImplemented (session: Session;  VAR (*IN*) Command: ARRAY OF CHAR);

    (* Default handler for anything I haven't yet implemented. *)

    BEGIN
        Reply2 (session, "502 Not implemented ", Command);
    END NotImplemented;

(********************************************************************************)

PROCEDURE NoSuchCommand (session: Session;  VAR (*IN*) Command: ARRAY OF CHAR);

    (* Command is not a recognised command. *)

    BEGIN
        Reply2 (session, "500 Unknown command ", Command);
    END NoSuchCommand;

(********************************************************************************)

PROCEDURE NotLoggedIn (session: Session;  VAR (*IN*) Command: ARRAY OF CHAR);

    (* Command is illegal because user is not yet logged in. *)

    BEGIN
        Reply2 (session, "530 Not logged in ", Command);
    END NotLoggedIn;

(************************************************************************)
(*                    EXECUTING AN EXTERNAL PROGRAM                     *)
(************************************************************************)

PROCEDURE ExecProg (VAR (*IN*) ProgName, Params: ARRAY OF CHAR): CARDINAL;

    (* This procedure executes the specified program on behalf of the   *)
    (* client.  The client thread remains blocked until the command     *)
    (* completes.  The program should return a result code of 0 in      *)
    (* order for the client to be given an "OK" response.               *)

    CONST ONLength = 256;

    VAR rc: CARDINAL;
        FailureObjectName: ARRAY [0..ONLength-1] OF CHAR;
        StartData: OS2.STARTDATA;
        idSession: CARDINAL;  pid: OS2.PID;

    BEGIN
        IF ProgName[0] = Nul THEN
            RETURN MAX(CARDINAL);
        END (*IF*);

        WITH StartData DO
            Length     :=  SIZE(OS2.STARTDATA);
            Related    :=  OS2.SSF_RELATED_INDEPENDENT;
            FgBg       :=  OS2.SSF_FGBG_FORE;
            TraceOpt   :=  OS2.SSF_TRACEOPT_NONE;
            PgmTitle   :=  NIL;
            PgmName    :=  ADR(ProgName);
            IF Params[0] = Nul THEN
                PgmInputs := NIL;
            ELSE
                PgmInputs := ADR(Params);
            END (*IF*);
            TermQ      :=  NIL;
            Environment:=  NIL;
            InheritOpt :=  OS2.SSF_INHERTOPT_PARENT;
            SessionType:=  OS2.SSF_TYPE_DEFAULT;
            IconFile   :=  NIL;
            PgmHandle  :=  0;
            PgmControl :=  OS2.SSF_CONTROL_VISIBLE;
            InitXPos   :=  30;
            InitYPos   :=  40;
            InitXSize  :=  200;
            InitYSize  :=  140;
            Reserved   :=  0;
            ObjectBuffer  :=  ADR(FailureObjectName);
            ObjectBuffLen :=  ONLength;
        END (*WITH*);

        rc := OS2.DosStartSession (StartData, idSession, pid);

        (*
        WriteString ("Back from DosStartSession, result code is ");
        WriteCard (rc);  WriteLn;
        WriteString ("FailureObjectName is ");  WriteString (FailureObjectName);
        WriteLn;
        *)

        (* Starting in background (code 457) is not an error. *)

        IF rc = 457 THEN rc := 0 END(*IF*);
        RETURN rc;

    END ExecProg;

(********************************************************************************)
(*                     HANDLERS FOR THE INDIVIDUAL COMMANDS                     *)
(********************************************************************************)

PROCEDURE ABOR (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    (* The ABOR command does nothing, in effect, because we don't have          *)
    (* asynchronous processing of control and data streams in this version.     *)

    BEGIN
        dummy[0] := dummy[0];    (* To avoid a compiler warning. *)
        Reply (session, "226 Command aborted");
    END ABOR;

(********************************************************************************)

PROCEDURE ACCT (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    BEGIN
        dummy[0] := dummy[0];    (* To avoid a compiler warning. *)
        Reply (session, "202 Superfluous at this site");
    END ACCT;

(********************************************************************************)

PROCEDURE ALLO (session: Session;  VAR (*IN*) size: ARRAY OF CHAR);

    BEGIN
        AllocateFileSize (session^.user, size);
        Reply (session, "200 ALLO OK");
    END ALLO;

(********************************************************************************)

PROCEDURE APPE (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    BEGIN
        SetFileName (session^.user, Params, FALSE);
        IF CanWrite (session^.user) THEN
            Reply (session, "150 OK, opening data connection.");
            IF AppendFile (session^.user) THEN
                Reply (session, "226 Transfer complete, closing data connection.");
            ELSE
                Reply (session, "451 Transfer failed.");
            END (*IF*);
        ELSE
            Reply (session, "553 Permission denied.");
        END (*IF*);
    END APPE;

(********************************************************************************)

PROCEDURE CDUP (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    BEGIN
        dummy[0] := dummy[0];    (* To avoid a compiler warning. *)
        IF SetDirectory (session^.user, session^.MSGenabled, "..") THEN
            Reply (session, "250 CWD OK.");
        ELSE
            Reply (session, "550 Cannot change directory.");
        END (*IF*);
    END CDUP;

(********************************************************************************)

PROCEDURE CWD (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    BEGIN
        IF SetDirectory (session^.user, session^.MSGenabled, Params) THEN
            Reply (session, "250 CWD OK");
        ELSE
            Reply (session, "550 Cannot change directory.");
        END (*IF*);
    END CWD;

(********************************************************************************)

PROCEDURE DELE (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    BEGIN
        SetFileName (session^.user, Params, FALSE);
        IF CanDelete (session^.user) THEN
            IF DeleteFile (session^.user) THEN
                Reply (session, "250 File deleted.");
            ELSE
                Reply (session, "550 Deletion failed.");
            END (*IF*);
        ELSE
            Reply (session, "550 Permission denied.");
        END (*IF*);
    END DELE;

(********************************************************************************)

PROCEDURE HELP (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    BEGIN
        dummy[0] := dummy[0];    (* To avoid a compiler warning. *)

        Reply (session, "214-The following commands are recognized.");
        Reply (session, "   ABOR    ACCT    ALLO    APPE    CDUP    CWD     DELE    HELP");
        Reply (session, "   LIST    MDTM    MKD     MODE    NLST    NOOP    PASS    PASV");
        Reply (session, "   PORT    PWD     QUIT    REIN    REST    RETR    RMD     RNFR");
        Reply (session, "   RNTO    SITE    SIZE    SMNT    STAT    STOR    STOU    STRU");
        Reply (session, "   SYST    TYPE    USER    XCUP    XCWD    XMKD    XPWD    XRMD");
        Reply (session, "214 Direct comments to peter@ee.newcastle.edu.au.");

    END HELP;

(********************************************************************************)

PROCEDURE LIST (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    BEGIN
        SetFileName (session^.user, Params, TRUE);
        IF ListingIsPossible (session^.user) THEN
            Reply (session, "150 OK, opening data connection.");
            IF SendDirectory (session^.user, TRUE) THEN
                CloseDataPort (session^.user);
                Reply (session, "226 Transfer complete.");
            ELSE
                CloseDataPort (session^.user);
                Reply (session, "451 Transfer failed.");
            END (*IF*);
        ELSE
            Reply (session, "550 Directory not available.");
        END (*IF*);
    END LIST;

(********************************************************************************)

PROCEDURE MDTM (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    VAR result: ARRAY [0..13] OF CHAR;

    BEGIN
        SetFileName (session^.user, Params, FALSE);
        GetFileDate (session^.user, result);
        IF result[0] = Nul THEN
            Reply (session, "550 Can't find file");
        ELSE
            Reply2 (session, "213 ", result);
        END (*IF*);
    END MDTM;

(********************************************************************************)

PROCEDURE MKD (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    BEGIN
        SetFileName (session^.user, Params, FALSE);
        IF MakeDirectory (session^.user) THEN
            Reply3 (session, '257 "', Params, '" created');
        ELSE
            Reply (session, "550 Permission denied.");
        END (*IF*);
    END MKD;

(********************************************************************************)

PROCEDURE MODE (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    (* Parameter can be S, B, or C.  I've implemented only S.     *)

    VAR option: CHAR;

    BEGIN
        option := CAP(Params[0]);
        IF (option = 'S') THEN
            SetTransferMode (session^.user, option);
            Reply (session, "200 OK");
        ELSE
            Reply (session, "504 Not supported");
        END (*IF*);
    END MODE;

(********************************************************************************)

PROCEDURE NLST (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    BEGIN
        SetFileName (session^.user, Params, TRUE);
        IF ListingIsPossible (session^.user) THEN
            Reply (session, "150 OK, opening data connection.");
            IF SendDirectory (session^.user, FALSE) THEN
                CloseDataPort (session^.user);
                Reply (session, "226 Transfer complete, closing data connection.");
            ELSE
                CloseDataPort (session^.user);
                Reply (session, "451 Transfer failed.");
            END (*IF*);
        ELSE
            Reply (session, "550 File not available.");
        END (*IF*);
    END NLST;

(********************************************************************************)

PROCEDURE NOOP (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    BEGIN
        dummy[0] := dummy[0];    (* To avoid a compiler warning. *)
        Reply (session, "200 NOOP successfully executed!");
    END NOOP;

(********************************************************************************)

PROCEDURE PASS (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    BEGIN
        IF session^.state = WaitingForPassword THEN
            IF Params[0] = '-' THEN
                session^.MSGenabled := FALSE;
                Strings.Delete (Params, 0, 1);
            END (*IF*);
            IF PasswordOK (session^.user, Params) THEN
                IF session^.MSGenabled THEN
                    SendMessageFile (session^.socket, "Welcome.MSG", "230",
                                           session^.usernumber, session^.user);
                    SendDirectoryMessage (session^.user, "230");
                END (*IF*);
                Reply (session, "230 Login OK.");
                session^.state := LoggedIn;
            ELSE
                Reply (session, "530 Password not accepted.");
            END (*IF*);
        ELSE
            Reply (session, "503 Bad sequence of commands.");
        END (*IF*);
    END PASS;

(********************************************************************************)

PROCEDURE PASV (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    VAR myaddr: SockAddr;  PortString: ARRAY [0..23] OF CHAR;

    (****************************************************************************)

    PROCEDURE Decode;

        (* Converts the information in myaddr to a text form in PortString. *)

        TYPE Raw = ARRAY [0..SIZE(InternetSocketAddress)-1] OF CARD8;

        VAR j, place: CARDINAL;  Bytes: Raw;

        (************************************************************************)

        PROCEDURE Convert (value: CARDINAL);

            BEGIN
                IF value > 9 THEN Convert (value DIV 10) END(*IF*);
                PortString[place] := CHR(ORD('0') + value MOD 10);
                INC (place);
            END Convert;

        (************************************************************************)

        BEGIN     (* body of Decode *)
            Bytes := CAST (Raw, myaddr.in_addr);
            place := 0;
            FOR j := 2 TO 5 DO
                Convert (Bytes[j]);
                PortString[place] := ',';  INC(place);
            END (*FOR*);
            Convert (Bytes[0]);
            PortString[place] := ',';  INC(place);
            Convert (Bytes[1]);
            PortString[place] := Nul;
        END Decode;

    (****************************************************************************)

    BEGIN     (* body of PASV *)
        dummy[0] := dummy[0];    (* To avoid a compiler warning. *)
        IF EnterPassiveMode (session^.user, myaddr) THEN
            Decode;
            Reply3 (session, "227 Entering passive mode (", PortString, ")");
        ELSE
            Reply (session, "502 PASV did not succeed");
        END (*IF*);
    END PASV;

(********************************************************************************)

PROCEDURE PORT (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    TYPE CharSet = SET OF CHAR;
    CONST Digits = CharSet{'0'..'9'};

    VAR Arg: RECORD
                 CASE :BOOLEAN OF
                     FALSE: byte: ARRAY [0..5] OF CARD8;
                     |TRUE: addr: CARD32;  port: CARD16;
                 END (*CASE*);
             END (*RECORD*);

    VAR j, pos: CARDINAL;  val: CARD8;  Error: BOOLEAN;

    BEGIN
        pos := 0;  Error := FALSE;
        FOR j := 0 TO 5 DO
            val := 0;
            WHILE Params[pos] IN Digits DO
                val := 10*val + (ORD(Params[pos]) - ORD('0'));  INC(pos);
            END (*WHILE*);
            Arg.byte[j] := val;
            IF (pos > HIGH(Params)) OR (Params[pos] = Nul) THEN
                Error := Error OR (j<>5)
            ELSIF Params[pos] = ',' THEN INC(pos)
            ELSE Error := TRUE;
            END (*IF*);
        END (*FOR*);
        IF Error THEN
            Reply (session, "501 Syntax error.");
        ELSE
            SetPort (session^.user, Arg.addr, Arg.port);
            Reply (session, "200 Data port set");
        END (*IF*);
    END PORT;

(********************************************************************************)

PROCEDURE PWD (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    VAR DirString: ARRAY [0..511] OF CHAR;

    BEGIN
        dummy[0] := dummy[0];    (* To avoid a compiler warning. *)
        CurrentDirectory (session^.user, DirString);
        Reply3 (session, '257 "', DirString, '" is current directory.');
    END PWD;

(********************************************************************************)

PROCEDURE QUIT (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    BEGIN
        dummy[0] := dummy[0];    (* To avoid a compiler warning. *)
        Reply (session, "221 Goodbye.");
        session^.state := MustExit;
    END QUIT;

(********************************************************************************)

PROCEDURE REIN (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    BEGIN
        dummy[0] := dummy[0];    (* To avoid a compiler warning. *)
        WITH session^ DO
            RemoveUser (ClientData);
            state := Idle;
        END (*WITH*);
        Reply (session, "220 Ready for new user");
    END REIN;

(********************************************************************************)

PROCEDURE REST (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    BEGIN
        SetRestartPoint (session^.user, Params);
        Reply (session, "350 Restart marker set.");
    END REST;

(********************************************************************************)

PROCEDURE RETR (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    BEGIN
        SetFileName (session^.user, Params, FALSE);
        IF FileAvailable (session^.user) THEN
            Reply (session, "150 OK, opening data connection.");
            IF SendFile (session^.user) THEN
                Reply (session, "226 Transfer complete, closing data connection.");
            ELSE
                Reply (session, "426 Transfer failed.");
            END (*IF*);
        ELSE
            Reply (session, "550 File not available.");
        END (*IF*);
    END RETR;

(********************************************************************************)

PROCEDURE RMD (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    BEGIN
        SetFileName (session^.user, Params, FALSE);
        IF CanRemoveDirectory (session^.user) THEN
            IF DeleteDirectory (session^.user) THEN
                Reply (session, "250 Directory deleted.");
            ELSE
                Reply (session, "550 Deletion failed.");
            END (*IF*);
        ELSE
            Reply (session, "550 Permission denied.");
        END (*IF*);
    END RMD;

(********************************************************************************)

PROCEDURE RNFR (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    BEGIN
        SetFileName (session^.user, Params, FALSE);
        IF FileAvailable (session^.user) THEN
            Reply (session, "350 Send new name.");
        ELSE
            Reply (session, "550 File not found.");
        END (*IF*);
    END RNFR;

(********************************************************************************)

PROCEDURE RNTO (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    (* Remark: it is possible for the target name to imply a move from one      *)
    (* directory to another.  We count on procedure RenameFile to check that    *)
    (* the user has the appropriate permissions.                                *)

    BEGIN
        IF RenameFile (session^.user, Params) THEN
            Reply (session, "250 Rename succeeded.");
        ELSE
            Reply (session, "553 Permission denied.");
        END (*IF*);
    END RNTO;

(********************************************************************************)

PROCEDURE SITE (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    (* Only supported commands are PERM and the SITE MNGR group. *)

    TYPE KeywordNumber = [0..7];
         KeywordArray = ARRAY KeywordNumber OF FourChar;

    CONST KeywordList = KeywordArray {'    ', 'EXEC', 'EXIT', 'GXIT', 'KILL',
                                      'LIST', 'MNGR', 'PERM'};

    (****************************************************************************)

    PROCEDURE StripLeadingSpaces;

        (* Removes the leading spaces from Params. *)

        VAR m: CARDINAL;

        BEGIN
            m := 0;
            WHILE (m < HIGH(Params)) AND (Params[m] = " ") DO INC(m) END (*WHILE*);
            IF m > 0 THEN
                Strings.Delete (Params, 0, m);
            END (*IF*);
        END StripLeadingSpaces;

    (****************************************************************************)

    PROCEDURE Compare4 (n: KeywordNumber): INTEGER;

        (* Compares the first four characters of Params with KeywordList[n].   *)
        (* Returns >0 if Params[0..3] > KeywordList[n], and so on.             *)

        VAR ch1, ch2: CHAR;  k: [0..3];

        BEGIN
            k := 0;
            LOOP
                ch1 := Params[k];  ch2 := KeywordList[n][k];
                IF ch1 > ch2 THEN RETURN +1
                ELSIF ch1 < ch2 THEN RETURN -1
                ELSIF k = 3 THEN RETURN 0
                END (*IF*);
                INC (k);
            END (*LOOP*);
        END Compare4;

    (****************************************************************************)

    PROCEDURE FindKeyWord(): CARDINAL;

        VAR first, middle, last: CARDINAL;  test: INTEGER;

        BEGIN
            (* Go through the keyword list to find a match with the command.  *)
            (* In this version I'm using a binary search.                     *)

            first := 0;  last := MAX(KeywordNumber);
            LOOP
                middle := (first + last) DIV 2;
                test := Compare4 (middle);
                IF test > 0 THEN
                    first := middle + 1;
                ELSIF (test < 0) AND (middle > 0) THEN
                    last := middle - 1;
                ELSE
                    RETURN middle;
                END (*IF*);
                IF first > last THEN RETURN 0 END (*IF*);
            END (*LOOP*);

        END FindKeyWord;

    (****************************************************************************)

    VAR permstring: ARRAY [0..3] OF CHAR;
        Args: ARRAY [0..511] OF CHAR;
        pos: CARDINAL;  found: BOOLEAN;

    BEGIN
        Strings.Capitalize (Params);
        CASE FindKeyWord() OF

          |  6:  (* SITE MNGR *)
                 IF session^.IsManager THEN
                     Strings.Delete (Params, 0, 4);
                     StripLeadingSpaces;
                     CASE FindKeyWord() OF

                       | 1:  (* SITE MNGR EXEC *)

                             Strings.Delete (Params, 0, 4);
                             StripLeadingSpaces;
                             Strings.Assign (Params, Args);
                             Strings.FindNext (' ', Params, 0, found, pos);
                             IF found THEN
                                 Params[pos] := Nul;
                                 Strings.Delete (Args, 0, pos+1);
                             ELSE
                                 Args := "";
                             END (*IF*);
                             IF ExecProg (Params, Args) = 0 THEN
                                 Reply (session, "200 Program started.");
                             ELSE
                                 Reply (session, "200 Command failed.");
                             END (*IF*);

                       | 2:  (* SITE MNGR EXIT *)

                             Reply (session, "200 Closing FtpServer.");
                             StuffKeyboardBuffer ('Q');

                       | 3:  (* SITE MNGR GXIT *)

                             Reply (session, "200 FtpServer gradual shutdown.");
                             StuffKeyboardBuffer ('G');

                       | 4:  (* SITE MNGR KILL *)

                             Strings.Delete (Params, 0, 4);
                             StripLeadingSpaces;
                             KillUser (Params);
                             Reply (session, "200 Kill in progress.");

                       | 5:  (* SITE MNGR LIST *)

                             Reply (session, "200-List of current users");
                             ListAllUsers (session^.socket);
                             Reply (session, "200 End of list.");
                     ELSE
                         NotImplemented (session, Params);
                     END (*CASE*);
                 ELSE
                     Reply (session, "550 Permission denied.");
                 END (*IF*);

          |  7:  (* SITE PERM *)

                 GetCurrentPermissions (session^.user, permstring);
                 Reply2 (session, "200 Current permissions: ", permstring);

        ELSE
                 NotImplemented (session, Params);
        END (*CASE*);

    END SITE;

(********************************************************************************)

PROCEDURE Size (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    (* Note: non-standard command. *)

    VAR size, pos: CARDINAL;  buffer: ARRAY [0..15] OF CHAR;

    BEGIN
        SetFileName (session^.user, Params, FALSE);
        size := GetSize (session^.user);
        IF size = MAX(CARDINAL) THEN
            Reply (session, "550 Can't find file");
        ELSE
            buffer := "213 ";  pos := 4;
            ConvertCard (size, buffer, pos);
            IF pos < SIZE(buffer) THEN
                buffer[pos] := Nul;
            END (*IF*);
            Reply (session, buffer);
        END (*IF*);
    END Size;

(********************************************************************************)

PROCEDURE SMNT (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    BEGIN
        dummy[0] := dummy[0];    (* To avoid a compiler warning. *)
        Reply (session, "202 Superfluous at this site");
    END SMNT;

(********************************************************************************)

PROCEDURE STAT (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    BEGIN
        NotImplemented (session, Params);
    END STAT;

(********************************************************************************)

PROCEDURE STOR (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    BEGIN
        SetFileName (session^.user, Params, FALSE);
        IF CanWrite (session^.user) THEN
            Reply (session, "150 OK, opening data connection.");
            IF FetchFile (session^.user) THEN
                Reply (session, "226 Transfer complete, closing data connection.");
            ELSE
                Reply (session, "451 Transfer failed.");
            END (*IF*);
        ELSE
            Reply (session, "553 Permission denied.");
        END (*IF*);
    END STOR;

(********************************************************************************)

PROCEDURE STOU (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    VAR Name: ARRAY [0..255] OF CHAR;

    BEGIN
        dummy[0] := dummy[0];    (* To avoid a compiler warning. *)
        IF CreateUniqueFileName (session^.user, Name) AND CanWrite (session^.user) THEN
            Reply2 (session, "150 Transfer started ", Name);
            IF FetchFile (session^.user) THEN
                Reply (session, "226 Transfer complete, closing data connection.");
            ELSE
                Reply (session, "451 Transfer failed.");
            END (*IF*);
        ELSE
            Reply (session, "553 Permission denied.");
        END (*IF*);
    END STOU;

(********************************************************************************)

PROCEDURE STRU (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    (* Parameter can be F, R, or P.  I've implemented only F and R.     *)

    VAR option: CHAR;

    BEGIN
        option := CAP(Params[0]);
        IF (option = 'F') OR (option = 'R') THEN
            SetFileStructure (session^.user, option);
            Reply (session, "200 OK");
        ELSE
            Reply (session, "504 Not supported");
        END (*IF*);
    END STRU;

(********************************************************************************)

PROCEDURE SYST (session: Session;  VAR (*IN*) dummy: ARRAY OF CHAR);

    (* The reply here should really be "215 OS/2", but that seems to make       *)
    (* WebExplorer confused, so we have to fake being a Unix system.            *)

    BEGIN
        dummy[0] := dummy[0];    (* To avoid a compiler warning. *)
        Reply (session, "215 UNIX type:OS/2");
    END SYST;

(********************************************************************************)

PROCEDURE Type (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    (* Options supported are 'A', 'I', and 'L 8'.  *)

    VAR option: CHAR;

    BEGIN
        option := CAP(Params[0]);
        IF (option = 'L') AND (Params[2] = '8') THEN
            option := 'I';
        END (*IF*);
        IF (option = 'A') OR (option = 'I') THEN
            SetTransferType (session^.user, option);
            Reply (session, "200 OK");
        ELSE
            Reply (session, "504 Not supported");
        END (*IF*);
    END Type;

(********************************************************************************)

PROCEDURE USER (session: Session;  VAR (*IN*) Params: ARRAY OF CHAR);

    BEGIN
        WITH session^ DO
            IF state <> Idle THEN
                RemoveUser (ClientData);
                state := Idle;
            END (*IF*);
            IsManager := FALSE;
            ToLower (Params);
            CASE NewUser(ClientData, Params) OF
              | NoPasswordNeeded:
                          SendMessageFile (session^.socket, "Welcome.MSG", "230",
                                           session^.usernumber, session^.user);
                          SendDirectoryMessage (session^.user, "230");
                          Reply (session, "230 User logged in, proceed.");
                          session^.state := LoggedIn;
              | GuestUser:
                          Reply (session, "331 Send e-mail address as password.");
                          session^.state := WaitingForPassword;
              | NoSuchUser, NormalUser:
                          Reply (session, "331 Password required.");
                          session^.state := WaitingForPassword;
              | Manager:
                          Reply (session, "331 Password required.");
                          session^.state := WaitingForPassword;
                          session^.IsManager := TRUE;
              | OverflowUser:
                          Reply (session, "421 Too many users, try again later.");
                          session^.state := MustExit;
            END (*CASE*);
        END (*WITH*);
    END USER;

(********************************************************************************)
(*                      THE MAIN COMMAND DISPATCHER                             *)
(********************************************************************************)

TYPE
    KeywordNumber = [0..39];
    HandlerProc = PROCEDURE (Session, VAR (*IN*) ARRAY OF CHAR);
    HandlerArray = ARRAY KeywordNumber OF HandlerProc;
    KeywordArray = ARRAY KeywordNumber OF FourChar;

CONST
    KeywordList = KeywordArray {'ABOR', 'ACCT', 'ALLO', 'APPE', 'CDUP', 'CWD ',
                               'DELE', 'HELP', 'LIST', 'MDTM', 'MKD ', 'MODE',
                               'NLST', 'NOOP', 'PASS', 'PASV', 'PORT', 'PWD ',
                               'QUIT', 'REIN', 'REST', 'RETR', 'RMD ', 'RNFR',
                               'RNTO', 'SITE', 'SIZE', 'SMNT', 'STAT', 'STOR',
                               'STOU', 'STRU', 'SYST', 'TYPE', 'USER',
                               'XCUP', 'XCWD', 'XMKD', 'XPWD', 'XRMD'};

CONST
    HandlerList = HandlerArray {ABOR, ACCT, ALLO, APPE, CDUP, CWD,
                               DELE, HELP, LIST, MDTM, MKD, MODE,
                               NLST, NOOP, PASS, PASV, PORT, PWD,
                               QUIT, REIN, REST, RETR, RMD, RNFR,
                               RNTO, SITE, Size, SMNT, STAT, STOR,
                               STOU, STRU, SYST, Type, USER,
                               CDUP, CWD, MKD, PWD, RMD};

(********************************************************************************)

PROCEDURE HandleCommand (S: Session;  Command: ARRAY OF CHAR;
                                                     VAR (*OUT*) Quit: BOOLEAN);

    (* Executes one user command.  Returns with Quit=TRUE if the command is one *)
    (* that closes the session, or if the connection is lost.                   *)

    VAR k: [0..3];

    (****************************************************************************)

    PROCEDURE Compare4 (n: KeywordNumber): INTEGER;

        (* Compares the first four characters of Command with KeywordList[n].   *)
        (* Returns >0 if Command[0..3] > KeywordList[n], and so on.             *)

        VAR ch1, ch2: CHAR;

        BEGIN
            k := 0;
            LOOP
                ch1 := Command[k];  ch2 := KeywordList[n][k];
                IF ch1 > ch2 THEN RETURN +1
                ELSIF ch1 < ch2 THEN RETURN -1
                ELSIF k = 3 THEN RETURN 0
                END (*IF*);
                INC (k);
            END (*LOOP*);
        END Compare4;

    (****************************************************************************)

    VAR m: CARDINAL;  Match, QuitReceived: BOOLEAN;
        first, middle, last: CARDINAL;  test: INTEGER;
        Handler: HandlerProc;

    BEGIN
        (* Special case: add a space to a three-character command, to make the  *)
        (* search algorithm simpler.                                            *)

        IF Command[3] = Nul THEN
            Command[3] := ' ';  Command[4] := Nul;
        END (*IF*);

        (* Watch out for lower case. *)

        FOR k := 0 TO 3 DO
            Command[k] := CAP(Command[k]);
        END (*FOR*);

        (* Go through the keyword list to find a match with the command.  *)
        (* In this version I'm using a binary search.                     *)

        first := 0;  last := MAX(KeywordNumber);  Match := FALSE;
        LOOP
            middle := (first + last) DIV 2;
            test := Compare4 (middle);
            IF test < 0 THEN
                IF middle = 0 THEN
                    EXIT (*LOOP*);
                ELSE
                    last := middle - 1;
                END (*IF*);
            ELSIF test = 0 THEN
                Match := TRUE;  EXIT (*LOOP*);
            ELSIF test > 0 THEN
                first := middle + 1;
            END (*IF*);
            IF first > last THEN EXIT (*LOOP*) END (*IF*);
        END (*LOOP*);

        IF Match THEN
            Handler := HandlerList[middle];
        ELSE
            Handler := NoSuchCommand;
        END (*IF*);

        (* Echo command to transaction log. *)

        IF Handler = PASS THEN
            LogTransaction (S^.logID, "PASS ******");
        ELSE
            LogTransaction (S^.logID, Command);
        END (*IF*);

        (* If the user is not yet logged in, only HELP, USER, PASS, and QUIT are legal. *)

        QuitReceived := Handler = QUIT;
        IF NOT QuitReceived AND (S^.state <> LoggedIn) AND (Handler <> HELP)
                            AND (Handler <> USER) AND (Handler <> PASS) THEN
            Handler := NotLoggedIn;
        END (*IF*);

        (* Strip out the command characters, leaving only the parameters. *)

        IF Handler = NoSuchCommand THEN m := 0 ELSE m := 4 END(*IF*);
        WHILE (m < HIGH(Command)) AND (Command[m] = " ") DO INC(m) END (*WHILE*);
        Strings.Delete (Command, 0, m);

        (* Call the handler. *);

        Handler (S, Command);
        IF (S^.state = MustExit) AND NOT QuitReceived THEN
            LogTransaction (S^.logID, "Connection lost");
        END (*IF*);
        Quit := S^.state = MustExit;

    END HandleCommand;

(********************************************************************************)

BEGIN
    TransactionLoggingEnabled := FALSE;
END FtpCommands.

