/**********************************************************************
 *
 *  FILSPLIT v1.0 - Yet another file splitter/unsplitter
 *
 *  Module   : FILSPLIT.C
 *  Info     : Main module
 *  Author   : Bernd Giesen
 *  Compiler : Watcom C/C++ v10.6
 *             Gnu C/C++ v2.7.2.3 (Linux)
 *
 *  Copyright (c) 1999 Bernd Giesen. All rights reserved.
 *
 **********************************************************************
 *
 *  History:
 *
 *  271199 - Created
 *  051299 - adapted to run under Linux
 *  111299 - GetOrgFileTime()/SetOrgFileTime() added
 *  121299 - Used fix MAX_PATH_LEN value instead of platform dependent
 *           _MAX_PATH value. This will guarantee a constant splitfile
 *           header size for different platforms!
 *  131299 - Fix in ParseCommandline(): Don't parse outfile name for
 *           invalid slashes under Linux, since they may be part of
 *           the pathname
 *
 **********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/stat.h>

#include "filsplit.h"
#include "buildcnt.h"

#if !defined __LINUX__
 #include <sys/utime.h>
#else
 #include <sys/types.h>
 #include <utime.h>
#endif

#if defined __WATCOMC__
  #include <conio.h>
#endif

#if defined __LINUX__
struct termios kt;
struct termios bk;
#endif

struct utimbuf SourceFileTime;

char SourceFile[MAX_PATH_LEN];
char ListFileName[MAX_PATH_LEN];
char OutFileName[MAX_PATH_LEN] = "";
char StartExtension[] = ".fsp";

long InFileSize;
long InFileRest;
long MaxOutFileSize = 0L;
long CurOutFileSize = 0L;
long NumOfSplitFiles = 0L;

bool Unsplit = false;
bool QuietMode = false;
bool ProcessFileList = false;
bool AssumeYes = false;
bool MustReadName; /* true, if filename to unsplit is discovered from splitfile */

int ExtIndex = 0;

enum ErrCode
{
    EC_NO_ERROR = 0,
    EC_OPEN_FILE,
    EC_READ_FILE,
    EC_WRITE_FILE,
    EC_USER_ABORT,
    EC_INVALID_FILELIST,
    EC_FILE_SIZE,
    EC_SIZE_TOO_LOW,
    EC_TOO_MANY_FILES,
    EC_MALLOC,
    EC_MISSING_FILE
};

const char *ErrMsg[]  = {
                          "\r\nDone.",
                          "\r\nError opening file '%s'",
                          "\r\nError reading file '%s'",
                          "\r\nError writing file '%s'",
                          "\r\n\r\nProcess aborted by user!",
                          "\r\nInvalid file list '%s'",
                          "\r\nError getting file size of '%s'",
                          "\r\n'%s' ignored: File size is less or equal to current split size!",
                          "\r\nOnly one source file allowed!\r\n\r\n",
                          "\r\nMemory allocation error!\r\n",
                          "\r\nMissing required input file '%s'. Unsplitting incomplete!"
                        };

void ErrMessage( const char *pMsg, const char *pParam );

/**********************************************************************/

void Syntax( void )
{

  printf("\r\nFileSplit %s for %s (Build %d)", VERSION, OS_VER, BUILDCOUNT_NUM);
  printf("\r\nCopyright (c) 1999 by Bernd Giesen. All rights reserved.\r\n");
  printf("\r\nSyntax: xFilSplt [options] infile [-o outfile]\r\n");
  printf("\r\nwhere x = d (DOS version), o (OS/2 version), n (Windows 9x/NT version),");
  printf("\r\n          l (Linux version)\r\n");
  printf("\r\nOptions:  -u          unsplit infile");
  printf("\r\n          -1440       split to 1,44 MB files (default)");
  printf("\r\n          -720        split to 720k files");
  printf("\r\n          -sxxx       split to files of xxx bytes size");
  printf("\r\n          -o outfile  split to destination file(s) named outfile");
  printf("\r\n                      (Please note: Unlike for unsplitting, passed");
  printf("\r\n                       outfile extensions for splitting are ignored!)");
  printf("\r\n          -q          quiet mode (no screen output), implies -y switch!!!");
  printf("\r\n          -@ listfile split all input files specified by a filelist");
  printf("\r\n          -y          answer all questions with yes");
  printf("\r\n          -? -h       this help screen\r\n");
  printf("\r\nIf no split size was selected, a size of 1,44 MB is assumed by default.\r\n\r\n");

}

/**********************************************************************/

#if defined __LINUX__
char* strlwr( char *pStrn )
{
  char *p = pStrn;

  while ( *p )
   tolower(*p++);

  return pStrn;

}
#endif

/**********************************************************************/

bool SetOrgFileTime( const char *pFileName )
{
  if ( 0 == utime( pFileName, &SourceFileTime ) )
    return true;

  return false;

}

/**********************************************************************/

bool GetOrgFileTime( const char *pFileName )
{
  struct stat Buffer;

  if ( -1 != stat( pFileName, &Buffer ) )
  {
    SourceFileTime.modtime = Buffer.st_mtime;
    SourceFileTime.actime = Buffer.st_atime;
    return true;
  }

  return false;

}

/**********************************************************************/

void Idle()
{

#if defined __DOS__

  union REGS ri, ro;

#if defined __386__
  ri.w.ax = 0x1680;
  int386(0x2F, &ri, &ro);
#else
  ri.x.ax = 0x1680;
  int86(0x2F, &ri, &ro);
#endif

#endif

#if defined __OS2__
  DosSleep(1);
#elif defined __NT__
  // Sleep(..) seems not to give any timeslices under Windows 98 (FE),
  // at least not on my machine. Why not, where's the bug?
  Sleep(10);
#else /* __LINUX__ */
  /* seems not to be required */
#endif

}

/**********************************************************************/

unsigned long CalcDiskSize( int Base )
{
  switch( Base )
  {
    case 720:   return 728832L;
    case 1440:  return 1457664L;
  }

  return 0L;
}

/**********************************************************************/

long FileSize( FILE *fp )
{
  long int save_pos, size_of_file;

  save_pos = ftell( fp );
  fseek( fp, 0L, SEEK_END );
  size_of_file = ftell( fp );
  fseek( fp, save_pos, SEEK_SET );

  return( size_of_file );
}

/**********************************************************************/

int FileExist( const char *pFileName )
{
  FILE *fp;

  if ( NULL == ( fp = fopen( pFileName, "r" ) ) )
     return ( false );
  else
    fclose( fp );

  return ( true );
}

/**********************************************************************/

bool Overwrite( const char *pFileName )
{
  int ch;

  if ( !AssumeYes )
  {
    printf( "\r\nFile '%s' exists. Overwrite(Y/N)? ", pFileName );

    while( true )
    {

#if !defined __LINUX__
      if ( kbhit() )
      {
        ch = toupper( getch() );
#else

    /***
         what the hell is a simple(!) Linux alternative
         for kbhit() and getch()?
     ***/
    ch =  toupper( fgetc( stdin ) );
#endif

    if ( ( 'Y' == ch ) || ( 'N' == ch ) )
          break;

        printf( "\a" );
#if !defined __LINUX__
      }

      Idle();
#endif
    } /* while */

    if  ( 'Y' != ch )
    {
      ErrMessage( ErrMsg[EC_USER_ABORT], NULL );
      return ( false );
    }
  }

  return ( true );
}

/**********************************************************************/

void ErrMessage( const char *pMsg, const char *pParam )
{
  if ( !QuietMode )
  {
    if ( NULL != pParam )
      fprintf( stderr, pMsg, pParam );
    else
      fprintf( stderr, pMsg );
    fflush( stderr );
  }
}

/**********************************************************************/

void Message( const char *pMsg )
{
  if ( !QuietMode )
    printf( pMsg );
}


/**********************************************************************/

bool ParseCommandLine( int Count, char **ppStrn )
{

 int i = 1, j;
 char *pArg;
 bool InfileProcessed = false;

 if ( Count < 2 )
  return ( false );

 while( i < Count )
 {

  if ( ( *( pArg = ppStrn[i] ) != '-' ) && ( *pArg !='/' ) )
  {
   if ( !InfileProcessed ) /* only accept one sourcefile name */
   {
     char *pExt, *pCompBuffer;
     strcpy( SourceFile, pArg );
     pCompBuffer = ( char * ) malloc((strlen( SourceFile ) + 1 ) * sizeof( char ) );
     if ( NULL == pCompBuffer)
     {
         ErrMessage( ErrMsg[EC_MALLOC], NULL );
         return ( false );
     }
     strcpy( pCompBuffer, SourceFile );
     pExt = strstr( strlwr( pCompBuffer ), StartExtension );
     if ( NULL != pExt )
        Unsplit = true;
     free( pCompBuffer );
     ++i;
     InfileProcessed = true;

     continue;
   }
   else
   {
     ErrMessage( ErrMsg[EC_TOO_MANY_FILES], NULL );
     return ( false );
   }
  }

  pArg = ++ppStrn[i];  /* overread '-' or '/' */

  switch( toupper( *pArg ) )
  {

   case 'U':
     Unsplit = true;
    break;

   case 'Q':
     QuietMode = true;
     AssumeYes = true;
    break;

   case '@':
     ProcessFileList = true;
     /* read next argument */
     if ( ( *(pArg+1) == '\0')  &&
           ( *( pArg = ppStrn[++i] ) != '-' ) && ( *pArg !='/' ) )
       strcpy(ListFileName, pArg);
     else
      return ( false );
    break;

   case 'H':
   case '?':
     return( false );
    break;

   case 'O':
     /* read next argument */
     if ( ( *(pArg+1) == '\0')  &&
#if defined __LINUX__
           ( *( pArg = ppStrn[++i] ) != '-' ) )
#else
           ( *( pArg = ppStrn[++i] ) != '-' ) && ( *pArg !='/' ) )
#endif
       strcpy(OutFileName, pArg);
     else
      return ( false );
    break;

   case 'S':
     for ( j=0; j < strlen( pArg+1 ); j++ )
     {
       if ( !isdigit( *( pArg+1+j ) ) )
        return ( false );
     }
     sscanf( pArg+1,"%lu", &MaxOutFileSize );
    break;

   case '1':
     if ( 0 != strcmp( pArg, "1440" ) )
       return ( false );
     MaxOutFileSize = CalcDiskSize( 1440 );
    break;

   case '7':
     if ( 0 != strcmp( pArg, "720" ) )
      return ( false );
     MaxOutFileSize = CalcDiskSize( 720 );
    break;

   case 'Y':
     AssumeYes = true;
    break;

   default:
    return ( false );
  }

  ++i;

 }

 if ( 0L == MaxOutFileSize )
   MaxOutFileSize = CalcDiskSize( 1440 ); /* use size of 1,44 MB by default */

 return ( true );
}

/**********************************************************************/

bool BuildFileName( char *pFileName, bool CheckFile )
{
  char *p1, *p2;
  char Index[5];

  if ( 0 == strcmp( pFileName, "" ) )
    strcpy( pFileName, SourceFile );

  p1 = strrchr(pFileName, '.');           /* search the most right period */
  if ( NULL != p1 )
  {
    p2 = strrchr(pFileName, '\\');     /* search the most right backslash */
    if ( NULL != p2 )
    {
     if ( p1 > p2 )   /* period is period of a filename, not of a directory */
      *p1 = '\0';     /* cut extension */
    }
    else
      *p1 = '\0';     /* cut extension */
  }

  if ( 0 == ExtIndex )
    strcat( pFileName, StartExtension );
  else
  {
    sprintf(Index, ".%03d", ExtIndex );
    strcat( pFileName, Index );
  }

   if ( CheckFile )
    if ( FileExist( pFileName ) )
     if ( !Overwrite( pFileName ) )
       return ( false );

  ++ExtIndex;

  return ( true );
}

/**********************************************************************/

bool OpenNewInFile( FILE **fp )
{
  if ( !BuildFileName( SourceFile, false ) )
    return ( false );

  if ( NULL == ( *fp = fopen( SourceFile, "rb" ) ) )
  {
    ErrMessage( ErrMsg[EC_OPEN_FILE], SourceFile );
    return ( false );
  }

  InFileSize = FileSize( *fp );

  Message( "\r\n" );
  Message( SourceFile );

  return ( true );
}

/**********************************************************************/

bool OpenNewOutFile( FILE **fp )
{

  if ( ProcessFileList )
    strcpy( OutFileName, "" );

  if ( !BuildFileName( OutFileName, true ) )
    return ( false );

  if ( NULL == ( *fp = fopen( OutFileName, "wb" ) ) )
  {
    ErrMessage( ErrMsg[EC_OPEN_FILE], OutFileName );
    return ( false );
  }

  Message( "\r\n" );
  Message( OutFileName );

  if ( InFileRest - MaxOutFileSize >= 0L)
  {
    CurOutFileSize = MaxOutFileSize;
    InFileRest -= MaxOutFileSize;
  }
  else
    CurOutFileSize = InFileRest;

  return ( true );
}

/**********************************************************************/

/* store original filesize and filename
   at the beginning of the first splitted file
 */

bool WriteFileHeader( FILE *fp )
{
  char FileName[MAX_PATH_LEN], *p;
  long Sum;

  Message(" - Writing header information...");

  p = strrchr(SourceFile, '\\');
  if ( NULL == p)
    p = strrchr(SourceFile, ':');

  if ( NULL == p)
   p = SourceFile;
  else
   p += 1;

  memset( FileName, 0, sizeof( FileName ) );
  strcpy( FileName, p );

  if ( ( sizeof( InFileSize ) > fwrite( &InFileSize, 1, sizeof( InFileSize ), fp ) ) ||
       ( sizeof( NumOfSplitFiles ) > fwrite( &NumOfSplitFiles, 1, sizeof( NumOfSplitFiles ), fp ) ) ||
       ( sizeof( SourceFileTime ) > fwrite( &SourceFileTime, 1, sizeof( SourceFileTime), fp ) ) ||
       ( sizeof( FileName) > fwrite( FileName, sizeof( char ), sizeof( FileName ), fp ) )
     )
     return ( false );

  Sum =  sizeof( NumOfSplitFiles ) +
         sizeof( InFileSize ) +
         sizeof( char )*sizeof( FileName ) +
         sizeof( SourceFileTime );

  CurOutFileSize = MaxOutFileSize - Sum;
  InFileRest += Sum;

  return ( true );

}

/**********************************************************************/

/* read original filesize and filename
   at the beginning of the first splitted file
 */

bool ReadFileHeader( FILE *fp )
{
  char TempFile[MAX_PATH_LEN];
  MustReadName = ( 0 == strcmp( OutFileName, "") );

  if ( ( sizeof( MaxOutFileSize ) > fread( &MaxOutFileSize, 1, sizeof( MaxOutFileSize ), fp ) ) ||
       ( sizeof( NumOfSplitFiles ) > fread( &NumOfSplitFiles, 1, sizeof( NumOfSplitFiles ), fp ) ) ||
       ( sizeof( SourceFileTime ) > fread( &SourceFileTime, 1, sizeof( SourceFileTime), fp ) ) ||
       ( sizeof( TempFile ) > fread( TempFile, sizeof( char ), sizeof( TempFile ), fp ) )
     )
    return ( false );

  if ( 0 == strcmp( OutFileName, "") ) /* if outfilename not specified by commandline option */
    strcpy( OutFileName, TempFile );

  Message(" - Reading header information...");

  if ( FileExist( OutFileName ) )
   if ( !Overwrite( OutFileName ) )
     return ( false );

  if ( MustReadName )
  {
    Message( "\r\nUnSplitting to \"" );
    Message( OutFileName );
    Message( "\":\r\n" );
  }

  return ( true );
}

/**********************************************************************/

bool UnSplitFile( void )
{
  FILE *fpIn, *fpOut;
  long i;
  int ch;

  if ( 0 != strcmp( OutFileName, "") )
  {
    Message( "\r\nUnSplitting to \"" );
    Message( OutFileName );
    Message( "\":\r\n\r\n" );
  }

  if ( FileExist( OutFileName ) )
   if ( !Overwrite( OutFileName ) )
     return ( false );

  if ( !OpenNewInFile( &fpIn ) )
  {
    fclose( fpIn );
    return ( false );
  }

  if ( !ReadFileHeader( fpIn ) || ( MaxOutFileSize < 1L) )
  {
    fclose( fpIn );
    return ( false );
  }

  if ( NULL == ( fpOut = fopen( OutFileName, "wb" ) ) )
  {
    fclose( fpIn );
    ErrMessage( ErrMsg[EC_OPEN_FILE], OutFileName );
    return ( false );
  }

  for ( i = 0 ; i < NumOfSplitFiles; i++ )
  {
    while( EOF != ( ch = fgetc( fpIn ) ) )
    {
      if ( EOF == fputc( ch, fpOut ) )
      {
        ErrMessage( ErrMsg[EC_WRITE_FILE], SourceFile );
        fclose( fpOut );
        fclose( fpIn );
        return ( false );
      }
    }

    fclose( fpIn );

    if (i < NumOfSplitFiles - 1 )
      if ( !OpenNewInFile( &fpIn ) )
      {
        fclose( fpOut );
        return ( false );
      }

  }


  fclose( fpOut );

  /* if we unsplit back to the original filename,
     set the original date/time stamp, too */
  if ( MustReadName )
    SetOrgFileTime( OutFileName );

  if ( i < NumOfSplitFiles )
  {
    ErrMessage( ErrMsg[EC_MISSING_FILE], SourceFile );
    return ( false );
  }

  return ( true );
}

/**********************************************************************/

bool SplitFile( void )
{
  FILE *fpIn, *fpOut;
  long i, j;
  int ch;

  CurOutFileSize = MaxOutFileSize;

  if ( ( NULL == ( fpIn = fopen( SourceFile, "rb" ) ) ) ||
       !GetOrgFileTime( SourceFile )
     )
  {
    ErrMessage( ErrMsg[EC_OPEN_FILE], SourceFile );
    return ( false );
  }

  InFileRest = InFileSize = FileSize( fpIn );
  if ( -1L == ( InFileSize ) )
  {
    fclose ( fpIn );
    ErrMessage( ErrMsg[EC_FILE_SIZE], SourceFile );
    return ( false );
  }

  if ( InFileSize <= MaxOutFileSize )
  {
    fclose ( fpIn );
    ErrMessage( ErrMsg[EC_SIZE_TOO_LOW], SourceFile );
    return ( false );
  }


  NumOfSplitFiles = InFileSize / MaxOutFileSize;
  if ( InFileSize % MaxOutFileSize )
    ++NumOfSplitFiles;

  Message( "\r\nSplitting \"" );
  Message( SourceFile );
  Message( "\" to:" );

  if ( !OpenNewOutFile( &fpOut ) )
  {
    fclose ( fpIn );
    return ( false );
  }

  if ( !WriteFileHeader( fpOut ) )
  {
    fclose ( fpIn );
    return ( false );
  }

  for ( i = 0; i < InFileSize; i++ )
  {
    for ( j = 0; j < CurOutFileSize; j++ )
    {
      if ( EOF == ( ch = fgetc( fpIn ) ) )
      {
        fclose( fpOut );
        fclose( fpIn );
        return ( false );
      }

      if ( EOF == fputc( ch, fpOut ) )
      {
        fclose( fpOut );
        fclose( fpIn );
        return ( false );
      }
    } // j

    if ( j == InFileRest )  // Rest of input file successfully done
      break;

    fclose( fpOut );

    if ( !OpenNewOutFile( &fpOut ) )
    {
        fclose( fpIn );
        return ( false );
    }

  } // i

  fclose( fpOut );
  fclose( fpIn );

  return ( true );
}

/**********************************************************************/

bool SplitFileList( void )
{
  FILE *fpList;
  int Cnt = 0;
  bool Result = true;

  if ( NULL == ( fpList = fopen( ListFileName, "rt" ) ) )
  {
      ErrMessage( ErrMsg[EC_OPEN_FILE], ListFileName );
      return ( false );
  }

  while ( NULL != fgets( SourceFile, sizeof( SourceFile ), fpList ) )
  {
      char *p = strchr( SourceFile, '\n' );
      if ( NULL != p )
        *p = '\0';

      if ( 0 == strcmp( SourceFile, "" ) ) /* overread empty lines */
       continue;

      ExtIndex = 0;

      /* Don't check for a result of Splitfile() here, process all
         files of the list even if Splitfile() failes on some files
         for some reason */
      SplitFile();

      Message("\r\n");

      ++Cnt;
  }

  fclose ( fpList );

  return ( Result );
}

/**********************************************************************/

#if defined __LINUX__
void tty_atexit()
{
  tcsetattr( STDIN_FILENO, TCSAFLUSH, &bk );
}
#endif

/**********************************************************************/

int main( int argc, char *argv[] )
{

  setbuf( stdout, NULL ); /* no output stream buffering */
  memset( SourceFile, 0, sizeof( SourceFile ) );

#if defined __LINUX__
  tcgetattr( STDIN_FILENO, &kt );
  bk = kt;
  cfmakeraw( &kt );
  atexit( tty_atexit );
  tcsetattr( STDIN_FILENO, TCSAFLUSH, &kt );
#endif

  if ( !ParseCommandLine( argc, argv ) )
  {
    Syntax();
    return EXIT_FAILURE;
  }

  Message( "\r\nProcessing...\r\n" );
  if ( Unsplit )
  {
     if ( !UnSplitFile() )
     {
       Message("\r\n\r\n");
       return EXIT_FAILURE;
     }
  }
  else
  {
     bool Success = ( ProcessFileList ) ? SplitFileList() : SplitFile();
     if ( !Success )
     {
       Message("\r\n\r\n");
       return EXIT_FAILURE;
     }
  }

  Message(ErrMsg[EC_NO_ERROR]);
  Message("\r\n\r\n");

  return EXIT_SUCCESS;
}

/**********************************************************************/
/**********************************************************************/

