/********************************************************************/
/*                                                                  */
/* CDCOPY: Version 1.5                                              */
/* Author: Michel SUCH. Email: msuch@free.fr                        */
/*                                                                  */
/* Purpose: Automatic copy of audio CDs and CDROMs.                 */
/*                                                                  */
/* This program requires several components to be installed         */
/*                                                                  */
/* 1: CDRECORD/2 at minimal level 1.8a19                            */
/* 2: LEECH 1.2.                                                    */
/* 3: EMXRT libraries 0.9d with fixpack 1.                          */
/* All components may be obtained on HOBBES: http://hobbes.nmsu.edu */
/*                                                                  */
/* Note: All components except EMXRT must be installed in the same  */
/* directory.                                                       */
/*                                                                  */
/********************************************************************/

   parse arg parms
   say "CDCOPY Version 1.6, 2000-02-28"
   say "Author: Michel SUCH, Email: msuch@free.fr"
   say ""
   call getparms
   call init_proc
   select
      when cdtype = 'audio' then do /* audio CD, use LEECH */
         "@"audioreader indrive "t:all" leechparms
         if rc <> 0 then do
            say "Error reading audio CD in drive" indrive "rc="rc
            call exit_proc rc
         end
      end
      when cdtype = 'data' then do
         "@"cdromreader "-a -f -l -L -R -v" get_label() "-o cdrom.raw" indrive"/"
         if rc <> 0 then do
            say "Error reading cdrom in drive" indrive "rc="rc
            call exit_proc rc
         end
      end
      otherwise do
         "@"cdromreader "-a -f -l -L -R -v" get_label() "-o cdrom.raw" indrive"/"
         if rc <> 0 then do
            say "Error reading cdrom in drive" indrive "rc="rc
            call exit_proc rc
         end
         "@"audioreader indrive "t:all" leechparms
         if rc <> 0 then do
            say "Error reading audio CD in drive" indrive "rc="rc
            call exit_proc rc
         end
      end
   end

      call write_cd

   if ncopies = 1 then
      say "CD recorded successfuly."
   else
      say "CDs recorded successfuly."
   call exit_proc rc
exit


getparms:

   parse value("") with ncopies quiet dummy preemp
   do i = 1 to words(parms)
      parm = word(parms, i)
      keyword = translate(parm)
      select
         when keyword = '-Q' then do
            if quiet = '' then quiet = 1
            else call badparm "duplicate parameter"
         end
         when substr(keyword, 1, 4) = '-NC:' then do
            if ncopies = '' then do
               ncopies = substr(keyword, 5)
               if datatype(ncopies) <> 'NUM' then
                  call badparm "Invalid number of copies"
            end
            else call badparm "duplicate parameter"
         end
         when keyword = '-DUMMY' | keyword = '-TEST' then do
            if dummy = '' then dummy = '-dummy'
            else call badparm "Duplicate parameter"
         end
         when keyword = '-PREEMP' then do
            if preemp = '' then preemp = '-preemp'
            else call badparm "Duplicate parameter"
         end
         when keyword = '-H' then call help
         otherwise call badparm "Invalid parameter"
      end
   end
   if ncopies = '' then ncopies = 1
   if quiet = '' then quiet = 0
return /* getparms */


badparm:

   parse arg msg
   say parm":" msg
   say "Type: CDCOPY -h for help"
   exit 60 /* badparm */


help:

   say "syntax: CDCOPY [options]"
   say "   where options can be:"
   say "   -nc:x - Number of copies (default 1)"
   say "   -dummy/-test - do everything with laser turned off"
   say "   -preemp - Audio tracks are mastered with 50/15 s preemphasis"
   say "   -q - run in quiet mode: do not ask to insert the disks"
   say "   -h - Display this help screen"
exit /* help */


init_proc:

/*------------------- Init Rexxutil-----------------------------------*/
   If RXFUNCQUERY('SysLoadFuncs') Then Do
      Call rxfuncadd 'SysLoadFuncs', 'REXXUTIL', 'SYSLOADFUNCS'
      Call sysloadfuncs
   End

   parse source . . progname .
   mydir = filespec('d', progname) || filespec('p', progname)

   /* Check if we have all appropriate programs */
   audioreader = find_prog("LEECH.EXE")
   if audioreader = '' then do
      say "Cannot find AUDIO CD grabber LEECH.EXE")
      call exit_proc 20
   end

   cdromreader = find_prog("MKISOFS.EXE")
   if cdromreader = '' then do
      say "Cannot find CDROM grabber MKISOFS.EXE"
      call exit_proc 20
   end

   writer = find_prog("CDRECORD.EXE")
   if writer = '' then do
      say "Cannot find CD burner CDRECORD.EXE"
      call exit_proc 20
   end

   /* Setup a local queue to avoid conflicts with other process */
   cdrqueue = rxqueue('create')
   oldq = rxqueue('set', cdrqueue)

   call readcfg
   if savedir = '' then savedir = mydir || "SAVEDISK"
   org = directory()
   dir = directory(savedir)
   if translate(dir) <> translate(savedir) then do
      ret = sysmkdir(savedir)
      if ret <> 0 then do
         say "Cannot create directory" savedir
         call exit_proc 40
      end
      dir = directory(savedir)
   end
   call check_emx
   if outdev = '' then call get_cdr
   if indrive = '' then call get_cdrom
   call get_cdtype
   call cleanup
return /* init_proc */


find_prog:
procedure

   /* First, search in the program's directory */
   parse source . . me .
   dir = filespec('d', me) || filespec('p', me)
   call sysfiletree dir || arg(1), lst.
   if lst.0 > 0 then  pgm = word(lst.1, 5)
   else do /* search in the path environment variable */
      pgm = syssearchpath( 'path', arg(1))
   end
   drop lst.
return pgm /* get_progs */


readcfg:

   parse value('') with savedir indrive outdev maxspeed leechparms
   config = mydir'cdcopy.cfg'
   i = 0
   do while lines(config) > 0
      i = i + 1
      line = translate(linein(config))
      parse var line cmd';'comment
      cmd = strip(cmd)
      if cmd = '' then iterate
      parse var cmd keyword'='value
      value = strip(value)
      if value = '' then call badcfg
      keyword = strip(keyword)
      select
         when keyword = 'CD-R' then do
            if outdev = '' then outdev = value
            else call badcfg "Duplicate CD-R ID"
         end
         when keyword = 'SPEED' then do
            if maxspeed = '' then maxspeed = value
            else call badcfg "Duplicate speed definition"
            if wordpos(maxspeed, '1 2 4 6 8') = 0 then
               call badparm "Speed value can be 1, 2, 4, 6 or 8"
         end
         when keyword = 'CD-ROM' then do
            if indrive = '' then indrive = value
            else call badcfg "Duplicate CD-ROM unit letter"
         end
         when keyword = 'DIRECTORY' then do
            if savedir = '' then savedir = value
            else call badcfg "duplicate save directory definition"
         end
         when keyword = 'LEECHPARMS' then do
            if leechparms = '' then leechparms = value
            else call badcfg "duplicate LEECH parameters definition"
         end
         otherwise call badcfg
      end
   end
   call stream config, 'c', 'close'
   if maxspeed = '' then maxspeed = 8
return /* readcfg */


badcfg:

   call stream config, 'c', 'close'
   parse arg msg
   say "Error in file" config "line" i
   if msg <> '' then say msg
   call exit_proc 50
exit 50 /* badcfg */


check_emx:

   /* Check if EMXRT is installed */
   "@call emxrev | rxqueue" cdrqueue "2>nul"
   if rc <> 0 then do
      say "The EMXRT libraries are not installed on your system"
      say "or are not at the right level."
         call require_emx
      end

   /* check if levels are good */
   emxref = "EMX EMXIO EMXLIBC EMXLIBCM EMXLIBCS EMXWRAP"
   emxfound = ''
   do queued()
      parse upper pull emx
      dll = word(emx, 1)
      level = word(emx, words(emx))
      select
         when dll = 'EMX' then do
            if level < 60 then call require_emx
            emxfound = emxfound' 'dll
         end
         when dll = 'EMXIO' then do
            if level < 60 then call require_emx
            emxfound = emxfound' 'dll
         end
         when dll = 'EMXLIBC' then do
            if level < 60 then call require_emx
            emxfound = emxfound' 'dll
         end
         when dll = 'EMXLIBCM' then do
            if level < 61 then call require_emx
            emxfound = emxfound' 'dll
         end
         when dll = 'EMXLIBCS' then do
            if level < 61 then call require_emx
            emxfound = emxfound' 'dll
         end
         when dll = 'EMXWRAP' then do
            if level < 60 then call require_emx
            emxfound = emxfound' 'dll
         end
         otherwise nop
      end
   end
   do i = 1 to words(emxref)
      emxlib = word(emxref, i)
      if wordpos(emxlib, emxfound) = 0 then do
         say "EMXRT problem: Library" emxlib "not found."
         call require_emx
      end
   end
   "@SET EMXOPT=-h120"
return /* check_emx */


require_emx:

   say "This program requires EMXRT Version 09D"
   say "With fixpack 1"
   say "You can get EMXRT 09D and EMXFIX01 packages"
   say "on the WEB from HOBBES: http://hobbes.nmsu.edu"
   call exit_proc 70
return /* require_emx */


cleanup:

   call sysfiletree savedir'\track_*.wav', lf.
   do i = 1 to lf.0
      call sysfiledelete word(lf.i, 5)
   end
   call sysfiledelete "cdrom.raw"
return /* cleanup */


exit_proc:

   parse arg rcode
   call cleanup
   dir = directory(org)
   if symbol("OLDQ") <> "LIT" then do
      oldq = rxqueue('set', oldq)
      cdrqueue = rxqueue('delete', cdrqueue)
   end
   if rcode <> 0 then call beep 440, 200
exit rcode /* exit_proc */


write_cd:

   "@eject" indrive
   do n = 1 to ncopies
      if quiet = 0 | n > 1 then do
         call beep 220, 100
         say "Insert a blank CD in the CD-R unit for copy" n
         say "press enter to continue,"
         say "any other key to cancel."
         if c2d(sysgetkey('noecho')) <> 13 then do
            say "Program aborted by user"
            call exit_proc 1
         end
      end

      /* try to determine the max speed for the writer */
      do forever
         "@"writer" dev="outdev" speed="maxspeed dummy "-v -eject" parmend
         if rc = 0 then leave
         if maxspeed = 1 then leave
         if maxspeed > 2 then maxspeed = maxspeed - 2
         else maxspeed = 1
      end
      if rc <> 0 then do
         say "Error writing CD rc="rc
         call exit_proc rc
      end
   end
return /* write_cd */


get_cdr:

   /* Find out the device for CD-r if any */
   /* Also find out the letter for CD-ROM */
   "@"writer" -scanbus | rxqueue" cdrqueue "2>nul"
   if rc <> 0 then do
      say "Error: cannot detect the SCSI id of your CD-R unit."
      call exit_proc rc
   end
   parse value('') with scsibus outdev cdr
   do queued()
      parse upper pull line
      select
         when (pos('CD-R ', line) > 0) | (pos('CD-R/', line) > 0) then do
            if cdr = '' then cdr = substr( word(line, 1), 3)
         end
         otherwise nop
      end
   end
   if cdr <> '' then outdev = cdr
   else do
      say "No CD-R unit found"
      call exit_proc 10
   end
return


get_cdrom:

   /* Determine the unit letter for the CD-ROM */
   /* and the type of CD that is in. */

   indrive = ''
   list = sysdrivemap('c')
   do i = 1 to words(list)
      dk = word(list, i)
      info = sysdriveinfo(dk)
      free = word(info, 2)
      select
         when info = '' then do
            indrive = dk
            leave
         end
         when free = 0 then do
            indrive = dk
            leave
         end
         otherwise nop
      end
   end
   if indrive = '' then do
      say "Cannot determine your CD-ROM drive letter"
      call exit_proc 30
   end
return /* get_cdrom */


get_cdtype:

   cdtype = ''
   if quiet = 0 then do
      call beep 220, 100
      say "Insert the source CD in the CD-ROM unit,"
      say "press enter to continue,"
      say"any other key to cancel."
      if c2d(sysgetkey('noecho')) <> 13 then do
         say "Program aborted by user"
         call exit_proc 1
      end
   end

   "@"audioreader indrive "disc | rxqueue" cdrqueue "2>nul"

   ndata = 0
   naudio = 0
   nlast = 0
   do queued()
      parse upper pull line
      if word( line, 2) = "PANIC" then do
         call beep 440, 100
         say "There is no CD in drive" indrive
         rd = rc
         call exit_proc rd
      end

      keyword = word(line, 1)
      select
         when keyword = 'LAST' then do
            nlast= word( line, words(line))
            if nlast = 0 then do
               Say "Unsupported CD type or empty CD"
               call exit_proc 70
            end
         end
         when keyword = 'AUDIO' then naudio = word(line, words(line))
         when keyword = 'DATA' then ndata = word(line, words(line))
         otherwise nop
      end
   end
   select
      when naudio = 0 then do
         cdtype = 'data'
         parmend = '-pad -data cdrom.raw'
      end
      when ndata = 0 then do
         cdtype = 'audio'
         parmend = '-dao defpregap=0 -audio track_*.wav'
      end
      otherwise do
         cdtype = 'mixed'
         parmend = '-pad -data cdrom.raw -audio track_*.wav'
      end
   end
return /* get_cdtype */


get_label:
/* Added by Rafal Szymczak to preserve the volume name of the */
/* source CD. Feel free to modify this routine (i.e. casing,  */
/* indentations) to fit your programming style.               */

return '-V "'delword(sysdriveinfo(indrive), 1, 3)'"'
