/****************************************************************************
*
*  CDDBMMCD.CMD - update OS/2 PM CD-Player via CDDB
*
*  Copyright (C) 2001 by Marcel Mller
*
*  This program is free software; you can redistribute it and/or
*  modify it under the terms of the GNU General Public License
*  as published by the Free Software Foundation; either version 2
*  of the License, or (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*  
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
****************************************************************************/

CALL RxFuncAdd 'SysLoadFuncs', 'RexxUtil', 'SysLoadFuncs'
CALL SysLoadFuncs


/* SECTION 1 - main ********************************************************/

/* settings */
fuzzy = 41 /* however, this is most likely to help for CD-R, because lead-in
              and lead out are fortune in this case */

CALL GetTOC

IF LocalCDDB(,fuzzy) \= 0 THEN EXIT(1)

CALL StoreMMCD
SAY 'Information stored.'
EXIT


/* SECTION 2 - high level functions ****************************************/

/* global vars common to different subfunctions

   TOCn   number of tracks
   TOC.i  start of track i [frames], i runs from 1 to TOCn + 1
          TOC.TOCn+1 is the total length
   TOCl   total length of CD [frames] without leadout

   TIT.0  CD title
   TIT.i  track titles, i runs from 1 to TOCn
*/

GetTOC: PROCEDURE EXPOSE TOCn TOC. TOCl
/* get TOC of CD

   parameters:
      ARG(1) mci device name, default= 'cdaudio'

   return:
      0      everything OK
      \= 0   error
             currently all errors are fatal, so this procedure will not return in case of errors

   return in global vars:
      TOCn, TOC.i, TOCl as defined above
*/
   IF ARG(1,'e') THEN dev = ARG(1)
    ELSE dev = 'cdaudio'
   /* Multimedia-REXX-Untersttzung laden und initialisieren */
   CALL RXFUNCADD 'mciRxInit','MCIAPI','mciRxInit'
   CALL mciRxInit
   /* open CDAUDIO */
   CALL MCIcmd 'open 'dev' alias c wait'
   /* get TOC */
   CALL MCIcmd 'set c time format mmtime wait'
   TOCn = MCIcmd('status c number of tracks wait')
   DO i = 1 TO TOCn
      TOC.i = MCIcmd('status c position track 'i' wait') / 40
      END
   TOC.i = TOC.TOCn + MCIcmd('status c length track 'TOCn' wait') / 40
   TOCl = MCIcmd('status c length wait') / 40
   /* release device */
   CALL MCIcmd 'close c'
   CALL mciRxExit
   RETURN 0

LocalCDDB: PROCEDURE EXPOSE TOCn TOC. TOCl TIT.
/* request to local CDDB database

   parameters:
      ARG(1) path to CDDB root including trailing '\', default: current dir
      ARG(2) fuzzy threshold [frames], default: 0

   parameters in global vars:
      TOCn, TOC.i, TOCl as defined above

   return:
      0      OK, found
      1      not found

   return in global vars:
      TIT.i as defined above
*/
   IF ARG(2,'e') THEN fuzzy = ARG(2)
    ELSE fuzzy = 0
   /* calc CD ID */
   s = 0
   DO i = 1 TO TOCn
      s = s + Qsum(TRUNC(TOC.i / 75))
      END
   l = TRUNC(TOC.i / 75)-TRUNC(TOC.1 / 75)
   cdid = D2X(Mod255(s), 2)D2X(l, 4)D2X(TOCn, 2)
   SAY 'CD-ID: 'cdid

   /* search for cddb record */
   CALL SysFileTree ARG(1)'*', 'dirs', 'DO'
   DO i = 1 TO dirs.0
      IF ReadCDDBFile(dirs.i'\', cdid) = 0 THEN RETURN 0 /* got it */
      END
   SAY 'no cddb record for id 'cdid' found.'
   IF (fuzzy = 0) | (TOCn = 1) THEN RETURN 1
   /* try fuzzy search */
   lendelta = TRUNC(fuzzy / 75) + 1
   qsdelta = 10
   cdids = ""
   DO qs = s - qsdelta TO s + qsdelta
      DO len = l - lendelta TO l + lendelta
         cdids = cdids||D2X(Mod255(qs), 2)D2X(len, 4)D2X(TOCn, 2)" "
         END
      END
   SAY "Fuzzy search ("fuzzy"): "cdids
   DO UNTIL cdids = ""
      PARSE VAR cdids cdid cdids
      DO i = 1 TO dirs.0
         IF ReadCDDBFile(dirs.i'\', cdid, fuzzy) = 0 THEN RETURN 0 /* got it */
         END
      END
   SAY '... no success.'
   RETURN 1

ReadCDDBFile: PROCEDURE EXPOSE TOCn TOC. TOCl TIT.
/* read CDDB file and check if it matches TOC

   parameters:
      ARG(1) path including trailing '\'
      ARG(2) cdid (filename)
      ARG(3) fuzzy threshold [frames], default: 0

   parameters in global vars:
      TOCn, TOC.i, TOCl as defined above

   return:
      0      information read in TIT.i
      1      file not found (or so)
      2      file format error
      3      file did not match

   return in global vars:
      TIT.i as defined above
*/
   IF ARG(3,'e') THEN fuzzy = ARG(3)
    ELSE fuzzy = 0
   file = ARG(1)ARG(2)
   IF STREAM(file, 'c', 'open read') \= 'READY:' THEN DO
      /* try win version of cddb */
      c1 = LEFT(ARG(2),1)
      c2 = SUBSTR(ARG(2), 2, 1)
      file = c1||TRANSLATE(c2,"02468ACE","13579BDF")"to"c1||TRANSLATE(c2,"13579BDF","02468ACE")
      IF STREAM(file, 'c', 'open read') \= 'READY:' THEN RETURN 1
      END
   DO i = 0 TO TOCn
      TIT.i = "" /* DROP ist NOT sufficient (perl would be nice) */
      END
   state = 1
   DO line = 0 WHILE STREAM(file, 'S') = 'READY'
      /* read line */
      l = LINEIN(file)
      SELECT
       WHEN LEFT(l, 9) = "#FILENAME=" THEN DO
         IF state = 2 THEN LEAVE line /* faster finish */
         state = TRANSLATE(SUBSTR(l, 11) = ARG(2)
         END
       WHEN state = 0 THEN NOP
       WHEN state = 1 THEN DO
         IF LEFT(l, 6) \= '# xmcd' THEN DO
            SAY 'cddb file 'file' has an unknown format.'
            CALL STREAM file, 'c', 'close'
            RETURN 2
            END
         state = 2
         END
       WHEN LEFT(l, 22) = "# Track frame offsets:" THEN DO
         /* read and check track offsets */
         IF fuzzy \= 0 THEN DO
            PARSE VALUE LINEIN(file) WITH '#' offset
            diff = STRIP(TRANSLATE(offset,,'	')) - TOC.1
            DO i = 2 TO TOCn
               PARSE VALUE LINEIN(file) WITH '#' offset
               offset = STRIP(TRANSLATE(offset,,'	'))
               IF ABS(TOC.i + diff - offset) >= fuzzy THEN DO
                  CALL STREAM file, 'c', 'close'
                  SAY file "did not match: Track "i", "offset", "TOC.i", "diff
                  RETURN 3
                  END
               END
            END
         ELSE DO i = 1 TO TOCn
            PARSE VALUE LINEIN(file) WITH '#' offset
            offset = STRIP(TRANSLATE(offset,,'	'))
            IF offset \= TOC.i THEN DO
               CALL STREAM file, 'c', 'close'
               SAY file "did not match: Track "i", "offset", "TOC.i
               RETURN 3
               END
            END
         /* well, the number of tracks should match */
         END
       WHEN LEFT(l, 1) = '#' THEN NOP /* any other comment */
       OTHERWISE
         PARSE VAR l item'='val
         SELECT
          WHEN item = 'DTITLE' THEN TIT.0 = TIT.0" "STRIP(val)
          WHEN LEFT(item,6) = 'TTITLE' THEN DO
            i = SUBSTR(item,7) + 1
            TIT.i = TIT.i" "STRIP(val)
            END
          OTHERWISE /* this should be superflous - well, not in my case W4,FP15 */
          END
       END
      END
   IF state = 0 THEN RETURN 1 /* win cddb: no entry found */
   DO i = 0 TO TOCn
      TIT.i = SUBSTR(TIT.i, 2)
      END
   CALL STREAM file, 'c', 'close'
   SAY 'CDDB-file: 'file' - 'TIT.0
   RETURN 0

StoreMMCD: PROCEDURE EXPOSE TOCn TOC. TOCl TIT.
/* store CD information in CDP.INI

   parameters:
      ARG(1) cdp.ini location, default: auto detect

   parameters in global vars:
      TOCn, TOC.i, TOCl, TIT.i as defined above
*/
   /* get CDP.INI location */
   IF ARG(1,'e') THEN ini = ARG(1)
    ELSE ini = STRIP(VALUE('MMBASE',,'OS2ENVIRONMENT'),'T',';')'\CDP.INI'
   /* calc MMCD key */
   mmcdkey = TRANSLATE('1245',F2MSF(TOCl),'12345')TRANSLATE('1245.78',F2MSF(TOC.TOCn-TOC.1),'12345678')
   SAY 'MMCD-Key: 'mmcdkey
   /* store data */
   CALL SysIni ini, mmcdkey, 'IMMCDDiscTitle', TIT.0
   DO i = 1 TO TOCn
      CALL SysIni ini, mmcdkey, i, TIT.i
      END
   RETURN


/* SECTION 3 - helper functions ********************************************/

MCIcmd: PROCEDURE /* execute MCI command and return result */
   /*SAY ">"ARG(1)">"*/
   rc = mciRxSendString(ARG(1), 'mcir', 0, 0)
   IF rc \= 0 THEN DO
      CALL mciRxGetErrorString rc, 'errstr'
      SAY "MCI Error "rc" - "errstr
      SAY "Command: "ARG(1)
      SAY "Result: "mcir
      CALL mciRxSendString 'close c', 'mcir', 0, 0
      CALL mciRxExit
      EXIT(255)
      END
   /*SAY "<"mcir"<"*/
   RETURN mcir

Qsum: PROCEDURE /* Quersumme */
   r = 0
   DO i = 1 TO LENGTH(ARG(1))
      r = r + SUBSTR(ARG(1), i, 1)
      END
   RETURN r

Mod255: PROCEDURE /* modulus 255 */
   /* not the world's most efficient modulus function, but without roundoff errors */
   r = ARG(1)
   DO WHILE r > 255
      r = r - 255
      END
   RETURN r


F2MSF: PROCEDURE /* frames -> mm:ss:ff */
   s = TRUNC(ARG(1)/75)
   f = ARG(1) - s*75
   m = TRUNC(s/60)
   s = s - m*60
   RETURN RIGHT(m,2,'0')':'RIGHT(s,2,'0')':'RIGHT(f,2,'0')

