/*
 * lower.c -- Defines a simple REXX function which will translate a string
 *            into lowercase.
 */

#define INCL_REXXSAA

#include <os2.h>
#include <rexxsaa.h>

#include <ctype.h>

#include "lower.h"

/*
 * Lower -- A REXX external function that converts its single argument
 *          into lowercase.
 */

RexxFunctionHandler Lower;

ULONG Lower( PUCHAR funcname, ULONG numargs, PRXSTRING args,
             PSZ queuename, PRXSTRING result )
  {
    PVOID mem = NULL;
    ULONG i;

    /* Get rid of compiler warnings... */

    funcname  = funcname;
    queuename = queuename;

    /* Need exactly one argument... */

    if( numargs != 1 ) return( 40 );

    /* Allocate new buffer if necessary... don't use malloc or any
       other library routine -- use DosAllocMem only */

    if( args[0].strlength > result->strlength ){
        if( DosAllocMem( &mem, args[0].strlength,
                         PAG_COMMIT | PAG_READ | PAG_WRITE ) != 0 ){
            return( 40 );
        }

        result->strptr = mem;
    }

    /* Copy and convert... */

    for( i = 0; i < args[0].strlength; ++i ){
        result->strptr[i] = tolower( args[0].strptr[i] );
    }

    result->strlength = args[0].strlength;

    return( 0 );
  }

/*
 * Function registration... before starting any REXX macros, we have
 * to register our external functions with the interpreter.  We just
 * keep a table of functions and register each one in sequence.
 */

typedef struct {
    PSZ name;
    PFN function;
} rxfunc_entry, *rxfunc_entry_p;

static rxfunc_entry REXXFuncs[] =
  {
    { "LOWER", (PFN) Lower },
    { NULL,    NULL  }
  };

/*
 * RegisterREXXFuncs -- Register each external REXX function in the
 *                      table.  We use RexxRegisterFunctionExe to do
 *                      the registration.
 *
 *  NOTE: RexxRegisterFunctionExe takes two parms, one is the name of
 *        the function to register and the second is the address of the
 *        function.  The function does not need to be exported.  After
 *        registration the function is only available to the process
 *        that registered it.
 *
 *        In contrast, RexxRegisterFunctionDll requires that the function
 *        be exported from a DLL and is then made available to all REXX
 *        programs in any process.
 *
 *        When the REXX interpreter looks for a function, it first looks
 *        in the list of functions registered using RexxRegisterFunctionExe
 *        before looking at those registered using RexxRegisterFunctionDll.
 */

BOOL RegisterREXXFuncs( void )
  {
    rxfunc_entry_p ptr;

    for( ptr = REXXFuncs; ptr->name != NULL; ++ptr ){
        RexxRegisterFunctionExe( ptr->name, ptr->function );
    }

    return( TRUE );
  }

/*
 * DeregisterREXXFuncs -- Deregister each external REXX function.  Strictly
 *                        speaking, this is not necessary since the
 *                        registrations are freed once the process exits,
 *                        but it's always good practice to clean up nicely.
 *
 *  NOTE: Never deregister external REXX functions that were registered
 *        using RexxRegisterFunctionDll, as this swipes them out from
 *        under the feet of ANY and ALL REXX programs, even those
 *        currently running!  Deregistering functions that were
 *        registered with RexxRegisterFunctionEXE is OK to do since they
 *        were only registered as part of your process space.
 */

void DeregisterREXXFuncs( void )
  {
    rxfunc_entry_p ptr;

    for( ptr = REXXFuncs; ptr->name != NULL; ++ptr ){
        RexxDeregisterFunction( ptr->name );
    }
  }
