/***********************************************************************/
/* ora.c - REXX/SQL for Oracle                                         */
/***********************************************************************/
/*
 * REXX/SQL. A REXX interface to SQL databases.
 * Copyright Impact Systems Pty Ltd, 1994-1997.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to 
 * address below.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                    Email:       M.Hessling@qut.edu.au
 * PO Box 203                       Phone:              +617 3802 0800
 * Bellara                          http://www.lightlink.com/hessling/
 * QLD 4507                         **** Author of THE & Rexx/SQL ****
 * Australia                        ****** Maintainer PDCurses *******
 *
 * Author:	Chris O'Sullivan  Ph (Australia) 015 123414
 *
 *    Purpose:	This module provides an Oracle SQL interface for REXX. It
 *		allows REXX programs to connect to Oracle servers, issue SQL
 *		queries, DDL, DCL & DML statements. Multiple concurrent
 *		connections are supported. Additionally, multiple statements
 *		may be active concurrently against any or all of these
 *		connections. This interface is called "REXX/SQL".
 *
 *      Notes:	1) The REXX/SQL interface is built using the Oracle Call
 *		Interface (OCI libraries). As such it does require that the
 *		object code be linked with the OCI libraries. Consequently,
 *		it does not require the Oracle Pre-compilers (i.e. Pro*C) to
 *		compile it. Also, it does not require the Pro* libraries
 *		(e.g. SQLLIB) at link time.
 *
 *		2) I have attempted to write this module so that the Oracle
 *              interface could be replaced by another (say GUPTA) without
 *              affecting REXX programs which have been written for this inter-
 *              face (providing these programs use ANSI SQL). Any programs
 *		using DDL or DCL will not be portable without some clever
 *		programming!
 *
 *		3) Very little has been added to support the performance
 *		features of Oracle. The interface does allow a statement to be
 *		parsed once and executed multiple times. Binding is another
 *		matter. Since arguments are passed by value, it was a tradeoff
 *		as to whether to dynamically allocate a buffer for each bind
 *		value and allow the statement to be reexecuted without rebinding
 *		(or rebinding of selected variables). The alternative was to
 *		assume that the extra overhead of the malloc() and strcpy()
 *		would be less efficient in many cases. At this point in time,
 *		the values passed to the function are bound directly. A rebind
 *		is thus necessary each time SQLEXECUTE() or SQLOPEN() is
 *		called.
 *
 *              5) As far as performance features there is currently no support
 *		for the following:
 *		 + Array fetches
 *		 + Array binding
 *		 + Bundled calls.
 *
 *		5) PL/SQL blocks are supported, however since all arguments
 *		to REXX functions are passed by value I have not thought of
 *		a suitable method of supporting bind values in PL/SQL blocks
 *		where the bind value is used for output. Input bind values
 *		are supported.
 *
 */


#define INCL_RXSHV	/* Shared variable support */
#define INCL_RXFUNC	/* External functions support */

#include "dbdefine.h"
#include "rexxsql.h"
#include "dbheader.h"
#include "dbtypes.h"

/* Connection structure. Each connection requires one of these! 
 * NB! The field; name, MUST be the first entry in the structure.
 */
typedef struct 
{
 char      name[MAX_IDENTIFIER+1];	/* Connection name */
 CURSOR    lda;			/* Logon Data Area for connection */
 ub1       hda[SIZEOF_HDA];		/* Host Data Area for connection */
 void      *dflt_stmt;		/* Pointer to implicit statement */
 int       num_statements;		/* Number of explicit statements */
 SQLVAL    *db_opt;	/* SqlGetInfo linked list */
} DBCON;

/* Statement structure. Each statement requires one of these!
 * NB! The field; name, MUST be the first entry in the structure.
 */
typedef struct 
{
 char      name[MAX_IDENTIFIER+1];	/* Statement name */
 DBCON     *db;			/* Pointer to connection structure */
 SQLWA     sqlwa;			/* SQL Work Area */
 ULONG     Argc;
 RXSTRING  *Argv;
} STMT;

/* 
 * Basic REXXSQL environment structure - allocated when 1st connection made! 
 */
typedef struct 
{
 int       num_connections;		/* Number of active connections */
 DBCON     *current_connection;	/* Pointer to current connection */
 BUCKET    *db_tbl[TBL_CONNECTIONS];	/* Connection hash table */
 BUCKET    *stmt_tbl[TBL_STATEMENTS];/* Statement hash table */
} DBENV;

char *column_attribute[] =
{
 "NAME", "TYPE", "SIZE", "PRECISION", "SCALE", "NULLABLE", (char *)NULL
};

/*
 * ReadOnly variables
 *
 * ---------------------------------------------------------------------
 * Current version/platform
 */
static RXSTRING RexxSqlVersion;
static char RexxSqlVersionString[100];

/*
 * ---------------------------------------------------------------------
 * Database Name and Version
 */
static RXSTRING RXSDBMSName;
static char DBMSName[100];

static RXSTRING RXSDBMSVer;
static char DBMSVer[50+1];

/* ---------------------------------------------------------------------
 * Error value returned for errors returned by Oracle as opposed to
 * those generated by the interface.
 */
#define DB_ERROR	(-1L)

/* Debugging flags */
int run_flags = 0;          /* can also be set via SQLVariable() */

/* Global Function Name - set on entry to every REXX/SQL function */
char FName[40];

extern ULONG RowLimit;
extern ULONG LongLimit;
extern ULONG SaveSQL;
extern ULONG AutoCommit;
extern ULONG IgnoreTruncate;
extern ULONG StandardPlacemarkers;
extern ULONG SupportsTransactions;
extern ULONG SupportsPlacemarkers;
extern ULONG SupportsDMLRowcount;
extern ULONG SupportsSqlGetData;
extern RXSTRING RXSNullStringOut;
extern char NullStringOut[MAX_IDENTIFIER+1];
extern RXSTRING RXSNullStringIn;
extern char NullStringIn[MAX_IDENTIFIER+1];

SQLVAL *opt_tbl = NULL;

#ifdef HAVE_PROTO
static void *SetDescribeColumns(DBCON *);
static void *SetDatatypes(DBCON *);
static void *SetVersion(void *);
static void *SetNullStringIn(void *);
static void *SetNullStringOut(void *);
static void *SetDBMSName(DBCON *);
static void *SetDBMSVer(DBCON *);
static void *SetSupportsSqlGetData(DBCON *);
static void *SetSupportsTransactions(DBCON *);
static int ValidateDataType(char *, int, DBCON *);
#else
static void *SetDescribeColumns();
static void *SetDatatypes();
static void *SetVersion();
static void *SetNullStringIn();
static void *SetNullStringOut();
static void *SetDBMSName();
static void *SetDBMSVer();
static void *SetSupportsSqlGetData();
static void *SetSupportsTransactions();
static int ValidateDataType();
#endif

/* ---------------------------------------------------------------------
 * The following table defines those values that can be used with the
 * SQLGetInfo() and SQLSetInfo() functions.
 */
/* name            settable  type         dyn/conn reqd                     maxlen  value func */
SQLVAL sqlvariable_vars[] =
 {
 {NULL,NULL,"DEBUG",                1, TYPE_INT,     OPT_STATIC ,0, (void*)&run_flags,0},
 {NULL,NULL,"ROWLIMIT",             1, TYPE_INT,     OPT_STATIC ,0, (void*)&RowLimit,0},
 {NULL,NULL,"LONGLIMIT",            1, TYPE_INT,     OPT_STATIC ,0, (void*)&LongLimit,0},
 {NULL,NULL,"SAVESQL",              1, TYPE_BOOL,    OPT_STATIC ,0, (void*)&SaveSQL,0},
 {NULL,NULL,"AUTOCOMMIT",           1, TYPE_BOOL,    OPT_STATIC ,0, (void*)&AutoCommit,0},
 {NULL,NULL,"IGNORETRUNCATE",       1, TYPE_BOOL,    OPT_STATIC ,0, (void*)&IgnoreTruncate,0},
 {NULL,NULL,"NULLSTRINGIN",         1, TYPE_STRING,  OPT_STATIC ,0, NULL,   SQLVAL_NULLSTRINGIN},
 {NULL,NULL,"NULLSTRINGOUT",        1, TYPE_STRING,  OPT_STATIC ,0, NULL,   SQLVAL_NULLSTRINGOUT},
 {NULL,NULL,"STANDARDPLACEMARKERS", 1, TYPE_BOOL,    OPT_STATIC ,0, (void*)&StandardPlacemarkers,0},
 {NULL,NULL,"SUPPORTSDMLROWCOUNT",  0, TYPE_BOOL,    OPT_STATIC ,0, (void*)&SupportsDMLRowcount,0},
 {NULL,NULL,"SUPPORTSPLACEMARKERS", 0, TYPE_BOOL,    OPT_STATIC ,0, (void*)&SupportsPlacemarkers,0},
 {NULL,NULL,"VERSION",              0, TYPE_STRING,  OPT_STATIC ,0, NULL,   SQLVAL_VERSION},
 {NULL,NULL,"",                     0, 0,            0          ,0, NULL,0},
};

SQLVAL sqlgetinfo_vars[] =
 {
 {NULL,NULL,"SUPPORTSTRANSACTIONS", 0, TYPE_BOOL,    OPT_STATIC ,0, NULL,   SQLVAL_SUPPORTSTRANSACTIONS},
 {NULL,NULL,"SUPPORTSSQLGETDATA",   0, TYPE_BOOL,    OPT_STATIC ,0, NULL,   SQLVAL_SUPPORTSSQLGETDATA},
 {NULL,NULL,"DESCRIBECOLUMNS",      0, TYPE_ARRAY,   OPT_DYNAMIC,0, NULL,   SQLVAL_DESCRIBECOLUMNS},
 {NULL,NULL,"DATATYPES",            0, TYPE_DATATYPE,OPT_DYNAMIC,0, NULL,   SQLVAL_DATATYPES},
 {NULL,NULL,"DBMSNAME",             0, TYPE_STRING,  OPT_STATIC ,0, NULL,   SQLVAL_DBMSNAME},
 {NULL,NULL,"DBMSVERSION",          0, TYPE_STRING,  OPT_STATIC ,0, NULL,   SQLVAL_DBMSVERSION},
 {NULL,NULL,"",                     0, 0,            0          ,0, NULL,0},
};

/* ---------------------------------------------------------------------
 * Pointer to REXXSQL environment - initially unallocated!
 * Only one object of this type is required and this object is created
 * the first time a connection is made. This object is released whenever
 * the last connection is released or upon exit.
 */
static DBENV *DbEnv = (DBENV*)NULL;

/* ---------------------------------------------------------------------
 * These global variables hold the result of the last function call.
 */
RXSQL_SQLCODE_TYPE  SQLCA_SqlCode  = -1L;	/* Force a clear on startup */
RXSQL_ROWCOUNT_TYPE SQLCA_RowCount = -1L;	/* Force a clear on startup */
long SQLCA_IntCode                 = -1L;	/* Force a clear on startup */
RXSQL_SQLSTATE_TYPE SQLCA_SqlState[SQL_SQLSTATE_SIZE+1]; /* UNUSED */

/*-----------------------------------------------------------------------------
 * Fetch the DB error and put into REXX "SQLCA." compound variable.
 *----------------------------------------------------------------------------*/
static void SetDBError

#ifdef HAVE_PROTO
    (CURSOR *lda, SQLWA *swa)
#else
    (lda, swa)
    CURSOR    *lda;
    SQLWA     *swa;
#endif

{
 CURSOR  *cursor=NULL;
 long    sqlcode=0L;
 char    *txt=NULL;
 char    msg[MAX_ERROR_TEXT];

 InternalFunctionPrologue("SetDbError");

 cursor = (swa) ? CTX(swa) : lda;

 /* Get the Oracle error code */
 sqlcode = V4ERRORCODE(cursor);

 /* Get the Oracle error message */
 (void)oerhms(lda, V4ERRORCODE(cursor), (text*)msg, sizeof(msg));
    
 /* Remove trailing newline character */
 msg[strlen(msg)-1] = '\0';

 /* Get the statement text */
 txt = (swa == (SQLWA*)NULL) ? "" : (swa->sql_stmt) ? swa->sql_stmt : "";

 /* Set SQLCA. variable */
 SetSQLCA(sqlcode, "ZZZZZ", msg, txt);

 return;
}

/*-----------------------------------------------------------------------------
 * Allocate a DBENV object. Only one database environment is allocated.
 * This structure is allocated when the first connection is made and deallocated
 * when the last connection is released.
 * Note that statement names are global and hence are defined in this structure
 * and not in the connection structure as would be the case if they were unique
 * to a connection!
 *----------------------------------------------------------------------------*/
static DBENV *AllocDbEnvironment

#ifdef HAVE_PROTO
    (void)
#else
    ()
#endif

{
 DBENV  *dbenv=NULL;
 int    i=0;

 InternalFunctionPrologue("AllocateDbEnvironment");

 if ((dbenv = (DBENV*)malloc(sizeof(DBENV))) == (DBENV*)NULL)
     return (DBENV*)NULL;
 dbenv->num_connections = 0;
 dbenv->current_connection = (DBCON*)NULL;
 for (i = 0; i < TBL_CONNECTIONS; i++)
     dbenv->db_tbl[i] = (BUCKET*)NULL;
 for (i = 0; i < TBL_STATEMENTS; i++)
     dbenv->stmt_tbl[i] = (BUCKET*)NULL;
 return dbenv;
}

/*-----------------------------------------------------------------------------
 * Allocate a DBCON object. One of these is required for each database
 * connection. This structure is allocated when the connection is made and
 * deallocated when the connection is released.
 *----------------------------------------------------------------------------*/
static DBCON *AllocConnection

#ifdef HAVE_PROTO
    (char *name)
#else
    (name)
    char *name;
#endif

{
 DBCON  *db=NULL;
 int    i=0;

 InternalFunctionPrologue("AllocateConnection");

 if ((db = (DBCON*)NewObject(sizeof(DBCON))) == (DBCON*)NULL)
     return (DBCON*)NULL;
 (void)strncpy(db->name, name, MAX_IDENTIFIER);
 db->dflt_stmt = (void*)NULL;
 db->db_opt = (SQLVAL*)NULL;
 db->num_statements = 0;
 return db;
}

/*-----------------------------------------------------------------------------
 * Open a database connection. This requires allocating a connection object
 * and making the connection to Oracle.
 *----------------------------------------------------------------------------*/
static int OpenConnection

#ifdef HAVE_PROTO
    (PSZ name, PSZ uid, size_t uidlen, DBCON **new_db)
#else
    (name, uid, uidlen, new_db)
    PSZ     name;
    PSZ     uid;
    size_t  uidlen;
    DBCON   **new_db;
#endif

{
 DBCON   *db=NULL;
 CURSOR  *lda=NULL;
 int     rc=0,i=0;
 void    *result=NULL;
 SQLVAL  *curr=NULL;

 InternalFunctionPrologue("OpenConnection");

 if ((db = AllocConnection(name)) == (DBCON*)NULL)
     return(SetIntError(10, "out of memory", __FILE__,__LINE__));
 lda = &(db->lda);

 /*
  * Do the database-specific connection
  */
 rc = orlon(lda, HDA(db), (ub1*)uid, uidlen, (ub1*)0, -1, 0);
 if (rc) 
   {
    SetDBError(lda, (SQLWA*)NULL);
    FreeObject(db);
    return DB_ERROR;
   }

 /*
  * Setup any connection-specific variable information
  */
 for (i=0;strlen(sqlgetinfo_vars[i].name)!=0;i++)
   {
    if (sqlgetinfo_vars[i].value)
       result = sqlgetinfo_vars[i].value;
    else
       result = GetSqlVariableValue(sqlgetinfo_vars[i].valtype,(void*)db);
    if ((curr = InstallSQLVariable(
                &rc,
                sqlgetinfo_vars[i].name,
                db->db_opt,
                sqlgetinfo_vars[i].user_update, 
                sqlgetinfo_vars[i].dtype, 
                0, 
                sqlgetinfo_vars[i].option,
                result)) == NULL)
      {
       return(rc);
      }
    if (db->db_opt == NULL)
       db->db_opt = curr;
   }
 *new_db = db;
 return 0;
}

/*-----------------------------------------------------------------------------
 * Allocate a STMT object. One of these is required for each statement
 * including the default statement for a connection. An instance of this object
 * is allocated when (i) a statement is prepared & (ii) the first time
 * SQLCOMMAND() is called for a connection (ie. the default statement is used).
 *----------------------------------------------------------------------------*/
static STMT *AllocStatement

#ifdef HAVE_PROTO
    (char *name, DBCON *db)
#else
    (name, db)
    char  *name;
    DBCON *db;
#endif

{
 STMT    *stmt=NULL;
 SQLWA   *swa=NULL;
 int	    i=0;

 InternalFunctionPrologue("AllocateStatement");

 if ((stmt = (STMT*)NewObject(sizeof(STMT))) == (STMT*)NULL)
     return (STMT*)NULL;
 stmt->db = db;
 (void)strncpy(stmt->name, name, MAX_IDENTIFIER);

 /* Initialise SQL Work Area */
 swa = SWA(stmt);
 swa->bind_cnt = 0;
 swa->expr_cnt = 0;
 swa->sql_stmt = (char*)NULL;
 swa->sql_stmt_sz = 0;

 stmt->Argc = 0L;
 stmt->Argv = NULL;

 /* 
  * Set pointer for each column descriptor to NULL (unused) 
  */
 for (i = 0; i < MAX_COLS; i++)
    swa->fa[i] = (FLDDSCR*)NULL;

 /* 
  * Create bind value arrays (empty)
  */
 for (i = 0; i < MAX_BINDVARS; i++)
    swa->pm[i] = (PMDSCR*)NULL;

 return stmt;
}

/*-----------------------------------------------------------------------------
 * Open a statement. This allocates a statement & then opens an Oracle cursor
 * (also called a context area).
 *----------------------------------------------------------------------------*/
static int OpenStatement

#ifdef HAVE_PROTO
    (char *name, DBCON *db, STMT **new_stmt)
#else
    (name, db, new_stmt)
    char  *name;
    DBCON *db;
    STMT  **new_stmt;
#endif

{
 int	    rc=0;
 STMT    *stmt=NULL;
 SQLWA   *swa=NULL;

 InternalFunctionPrologue("OpenStatement");
 
 if ((stmt = AllocStatement(name, db)) == (STMT*)NULL)
     return(SetIntError(10, "out of memory", __FILE__,__LINE__));

 /* Open the Oracle cursor */
 swa = SWA(stmt);
 if ((rc = oopen(CTX(swa), LDA(stmt), (ub1*)-1, -1, -1, (ub1*)-1, -1)))
   {
    SetDBError(LDA(stmt), (SQLWA*)NULL);
    FreeObject(stmt);
   }

 *new_stmt = stmt;
 return (rc==0) ? 0 : DB_ERROR;
}

/*-----------------------------------------------------------------------------
 * Disposes a SQL statement. This closes the Oracle cursor and frees the
 * statement structure and all resources associated with it.
 *----------------------------------------------------------------------------*/
static int ReleaseStatement

#ifdef HAVE_PROTO
    (STMT *stmt)
#else
    (stmt)
    STMT *stmt;
#endif

{
 SQLWA    *swa = SWA(stmt);
 FLDDSCR  *fd=NULL;
 PMDSCR   *pmd=NULL;
 int	     rc=0, i=0;

 InternalFunctionPrologue("ReleaseStatement");

 /* 
  * Close Oracle cursor 
  */
 if ((rc = oclose(CTX(swa))))
    SetDBError(LDA(stmt), swa);

 /* 
  * Free up select column descriptors 
  */
 for (i = 0; i < MAX_COLS; i++) 
   {
    fd = swa->fa[i];
    if (fd == NULL)
       break;

    if (fd->rbuf)
      free(fd->rbuf);
    if (fd->cbuf)
      free(fd->cbuf);
    free(fd);
    fd = NULL;
   } 

 /* 
  * Free up parameter marker descriptors 
  */
 for (i = 0; i < MAX_BINDVARS; i++)
   {
    pmd = swa->pm[i];
    if (pmd == NULL)
       break;
    if (pmd->buf)
       free(pmd->buf);
    free(pmd);
    pmd = NULL;
   }
    
 /* 
  * Free sql statement buffer (if any) 
  */
 if (swa->sql_stmt)
   free(swa->sql_stmt);

 /* 
  * Free the input bind values (if any)
  */
 stmt->Argv = FreeArgv(&stmt->Argc, stmt->Argv);
 FreeObject(stmt);

 return (rc==0) ? 0 : DB_ERROR;
}

/*-----------------------------------------------------------------------------
 * Release an Oracle connection. This closes any open statements associated
 * with the connection, closes the connection and frees all resources
 * associated with it.
 *----------------------------------------------------------------------------*/
static int ReleaseConnection

#ifdef HAVE_PROTO
    (DBCON *db)
#else
    (db)
    DBCON *db;
#endif

{
 int    i=0, rc=0, last_error=0;
 STMT   *stmt=NULL, *t=NULL;
 CURSOR *lda = &(db->lda);

 InternalFunctionPrologue("ReleaseConnection");

 /* 
  * Remove the Connection structure from the hash structures etc. 
  */
 (void)RemoveObject(db);

 /* 
  * Decrement the count of active connections. 
  */
 DbEnv->num_connections--;

 /* 
  * Dispose all active statements for this connection. 
  */
 for (i = 0; db->num_statements && i < TBL_STATEMENTS; i++) 
   {
   stmt = (STMT*)FirstObject(i, DbEnv->stmt_tbl);
   while (stmt && db->num_statements) 
     {
      t = (STMT*)NextObject(stmt);
      if (stmt->db == db) 
        {
         RemoveObject(stmt);
         last_error = (rc = ReleaseStatement(stmt)) ? rc : last_error;
         db->num_statements--;
        }
      stmt = t;
     }
   }

 /* 
  * Dispose all informational data for this connection.
  */
  db->db_opt = ll_free(db->db_opt);

 /* 
  * Dispose the default statement (if any). 
  */
 if (db->dflt_stmt)
     last_error = (rc = ReleaseStatement((STMT*)(db->dflt_stmt)))
                      ? rc : last_error;

 /* 
  * Disconnect from Oracle 
  */
 if ((rc = ologof(lda)))
   {
    SetDBError(lda, (SQLWA*)NULL);
    last_error = rc;
   }

 /* 
  * Free the connection structure 
  */
 FreeObject(db);

 return (last_error);
}

/*-----------------------------------------------------------------------------
 * Release the Oracle environment. This releases all active connections
 * (if any).
 *----------------------------------------------------------------------------*/
int ReleaseDbEnvironment

#ifdef HAVE_PROTO
    (void)
#else
    ()
#endif

{
 int    i=0, rc=0, last_error=0;
 DBCON  *db=NULL, *t=NULL;

 InternalFunctionPrologue("ReleaseDbEnvironment");

 /* Ensure there is an environment to release! */
 if (DbEnv == (DBENV*)NULL)
     return 0;

 /* Release all connections. */
 for (i = 0; DbEnv->num_connections && i < TBL_CONNECTIONS; i++) 
   {
    db = (DBCON*)FirstObject(i, DbEnv->db_tbl);
    while (db && DbEnv->num_connections) 
      {
       t = (DBCON*)NextObject(db);
       last_error = (rc = ReleaseConnection(db)) ? rc : last_error;
       db = t;
      }
   }

 /* Free the DB environment */
 free(DbEnv);
 DbEnv = (DBENV*)NULL;

 return (last_error);
}

/*-----------------------------------------------------------------------------
 * Convert "standard" parameter markers; ? with Oracle's :1 -> :n
 * placemarkers.
 *----------------------------------------------------------------------------*/
static char *ConvertPlacemarkers

#ifdef HAVE_PROTO
    (char *stmt, int len, int *new_len)
#else
    (stmt, len, new_len)
    char    *stmt;
    int     len;
    int     *new_len;
#endif

{
 static char *converted_stmt=NULL;
 static int  converted_len=0;
 int num_placemarkers=0;
 int placemarker=1;
 register int i=0,j=0,k=0;
 char buf[5];

 InternalFunctionPrologue("ConvertPlacemarkers");

 for (i=0;i<len;i++)
   {
    if (stmt[i] == '?')
       num_placemarkers++;
   }

 if ((1+len+(num_placemarkers*4)) > converted_len)
   {
    /* New statement size is > than previous */
    if (converted_stmt != (char*)NULL)
      {
       /* Free old statement */
       free(converted_stmt);
       converted_stmt = (char*)NULL;
       converted_len = 0;
      }

    /* Allocate a buffer for the new statement */
    if ((converted_stmt = (char*)malloc(1+len+(num_placemarkers*4))) == (char*)NULL)
       return(NULL);
    converted_len = 1+len+(num_placemarkers*4);
   }

 for (i=0,j=0;i<len;i++)
   {
    if (stmt[i] == '?')
      {
       converted_stmt[j++] = ':';
       sprintf(buf,"%d",placemarker++);
       for (k=0;k<(int)strlen(buf);k++)
          converted_stmt[j++] = buf[k];
      }
    else
      converted_stmt[j++] = stmt[i];
   }
 converted_stmt[j] = '\0';
 *new_len = j;
 return(converted_stmt);
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS:  SQLCONNECT( [connection-name, ] [username], [password], [database name], [host])
 *
 * ARGUMENTS: 
 *            0 - connection_name (optional)
 *            1 - username (optional)
 *            2 - password (optional)
 *            3 - database name (ignored under Oracle)
 *            4 - host name (optional)
 *
 * RETURNS :  0-success, <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLCONNECT

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
 char   uid[256];
 int    uidlen=0;
 char   dbname[MAX_IDENTIFIER+1];
 int    i=0, rc=0;
 char   tmp[128];
 DBCON  *db=NULL;

 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc > 5) return 1;

 /* Allocate a DB environment if none exists! */
 if (DbEnv == (DBENV*)NULL) 
   {
    if ((DbEnv = AllocDbEnvironment()) == (DBENV*)NULL)
       return ReturnError(retstr, 10, "out of memory.");
   }
 switch(argc)
   {
    case 0:
    case 1:
         RXSTRCPY(uid,uidlen,"/",1);
         break;
    case 2:
         if (RXSTRLEN(argv[1]))
            RXSTRCPY(uid,uidlen,RXSTRPTR(argv[1]),RXSTRLEN(argv[1]));
         else
            RXSTRCPY(uid,uidlen,"/",1);
         break;
    case 4:
    case 5:
         if (RXSTRLEN(argv[3]))
            return ReturnError(retstr, 18, "extraneous parameter <database name>.");
         /* meant to fall through */
    case 3:
         RXSTRCPY(uid,uidlen,RXSTRPTR(argv[1]),RXSTRLEN(argv[1]));
         RXSTRCAT(uid,uidlen,"/",1);
         RXSTRCAT(uid,uidlen,RXSTRPTR(argv[2]),RXSTRLEN(argv[2]));
         if (argc == 5
         &&  RXSTRLEN(argv[4]))
           {
            if (*(RXSTRPTR(argv[4])) != '@')
               RXSTRCAT(uid,uidlen,"@",1);
            RXSTRCAT(uid,uidlen,RXSTRPTR(argv[4]),RXSTRLEN(argv[4]));
           }
         break;
   }
#if REXXSQL_12_CODE
 /*
  * If a name is given then it is arg#1 and the connect string
  * is arg#2. If no name is given then connect-string is arg#1 unless
  * both name & connect-string are omitted.
  */
 if (argc == 0)
   {
    uid = "/";
    uidlen = 1;
   }
 else
   {
    if (argc == 1)
      {
       uid = RXSTRPTR(argv[0]);
       uidlen = RXSTRLEN(argv[0]);
      }
    else
      {
       uid = RXSTRPTR(argv[1]);
       uidlen = RXSTRLEN(argv[1]);
      }
   }
#endif

 /*
  * If a name is given it is the first argument!/
  * Get the name of the connection (default if not specified).
  */
 if (argc > 1 && RXSTRLEN(argv[0])) 
   {
    if ((rc = MkIdentifier(argv[0], dbname, sizeof(dbname))))
       return ReturnInt(retstr, rc);
   }
 else 
   {
    (void)strcpy(dbname, DEFAULT_CONNECTION);
   }

 /* Make sure there is no existing connection with the same name */
 if (FindObject(dbname, DbEnv->db_tbl, TBL_CONNECTIONS)) 
   {
    (void)sprintf(tmp, "connection already open with name \"%s\".", dbname);
    return ReturnError(retstr, 20, tmp);
   }

 /* Open a new connection for the given connect string. */
 if ((rc = OpenConnection(dbname, uid, uidlen, &db)))
     return ReturnInt(retstr, (long)rc);

 DbEnv->num_connections++;
 DbEnv->current_connection = db;
 (void)InsertObject(db, DbEnv->db_tbl, TBL_CONNECTIONS);

 return ReturnInt(retstr, 0L);
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS:  SQLDISCONNECT( [connection-name ] )
 *
 * RETURNS :  0-success, <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLDISCONNECT

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
 int		rc = 0;
 char        dbname[MAX_IDENTIFIER+1];
 DBCON       *db=NULL;
 char        tmp[128];


 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc > 1) return 1;

 if (DbEnv == (DBENV*)NULL)
     return ReturnInt(retstr, 0L);

 if (argc && RXSTRLEN(argv[0])) 
   {
    /* A connection has been named */
    if ((rc = MkIdentifier(argv[0], dbname, sizeof(dbname))))
       return ReturnInt(retstr, rc);

    /* Make sure the named connection exists. */
    if ((db = FindObject(dbname, DbEnv->db_tbl, TBL_CONNECTIONS))
                                           == (DBCON*)NULL) 
      {
       (void)sprintf(tmp,"connection \"%s\" is not open.",
                       dbname);
       return ReturnError(retstr, 21, tmp);
      }
   }
 else 
   {
    if (DbEnv->current_connection)
       db = DbEnv->current_connection;
    else
       return ReturnError(retstr, 25, "no connection is current");
   }
 /*
  * If terminating the current connection then make it so there is
  * no current connection!
  */
 if (db == DbEnv->current_connection)
     DbEnv->current_connection = (DBCON*)NULL;

 /* Do the disconnection */
 rc = ReleaseConnection(db);

 /* Free the environment if zero connections remaining */
 if (DbEnv->num_connections == 0) 
   {
    free(DbEnv);
    DbEnv = (DBENV*)NULL;
   }

 return ReturnInt(retstr, (long)rc);
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS:  SQLDEFAULT( [connection-name ] )
 *
 * RETURNS :  When called with 0 args : 0-success, <0-error.
 *         :  When called with 1 arg  : Name of current connection else "".
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLDEFAULT

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
 char        dbname[MAX_IDENTIFIER+1];
 DBCON       *db=NULL;
 char        tmp[128];
 int rc=0;

 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc > 1) return 1;

 if (argc && RXSTRLEN(argv[0])) 
   {
    /* Get the normalised name of the connection. */
    if ((rc = MkIdentifier(argv[0], dbname, sizeof(dbname))))
       return ReturnInt(retstr, rc);

    /* Make sure we have an environment! */
    if (DbEnv == (DBENV*)NULL)
       return ReturnError(retstr, 22, "no connections open.");

    /* Make sure the Connection Name is a valid one! */
    if ((db = FindObject(dbname, DbEnv->db_tbl, TBL_CONNECTIONS))
                                           == (DBCON*)NULL) 
      {
       (void)sprintf(tmp,"connection \"%s\" is not open.", dbname);
       return ReturnError(retstr, 21, tmp);
      }

    /* Make connection the default one! */
    DbEnv->current_connection = db;

    return ReturnInt(retstr, 0L);
   }
 else 
   {
    if (DbEnv && DbEnv->current_connection) 
      {
       return ReturnString(retstr,
                         DbEnv->current_connection->name,
                         strlen(DbEnv->current_connection->name));
      }
    else 
      {
       return ReturnString(retstr, "", 0);
      }
   }
}



/*-----------------------------------------------------------------------------
 * SYNOPSIS:  Called by SQLCOMMIT( ) &  SQLROLLBACK( )
 *
 * RETURNS :  0-success, <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLTRANSACT

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr,int commit)
#else
    (name, argc, argv, stck, retstr, commit)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
    int commit;
#endif
{ 
 int		rc = 0;
 DBCON       *db=NULL;
 CURSOR      *lda=NULL;


 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc > 1) return 1;

 if (DbEnv == (DBENV*)NULL)
     return ReturnError(retstr, 22, "no connections open.");

 if (DbEnv->current_connection)
    db = DbEnv->current_connection;
 else
    return ReturnError(retstr, 25, "no connection is current");

 lda = &(db->lda);

 if ((rc = (commit) ? ocom(lda) : orol(lda)))
    SetDBError(lda, (SQLWA*)NULL);

 return ReturnInt(retstr, (rc==0)?0:DB_ERROR);
}

/*-----------------------------------------------------------------------------
 * Binds all arguments for standard placemarkers (by number)
 *----------------------------------------------------------------------------*/
static int BindByStandard

#ifdef HAVE_PROTO
    (STMT *stmt,ULONG argc,RXSTRING argv[])
#else
    (stmt, argc, argv)
    STMT          *stmt;
    ULONG         argc;
    RXSTRING      argv[];
#endif
{
 int     i=0,file_bind=0,bind_type=0;
 int     rc=0, bindlen=0;
 void    *bindaddr=NULL;
 short   *indp=NULL;
 char    tmp[128];
 SQLWA   *swa= SWA(stmt);
 PMDSCR  *pmd=(PMDSCR*)NULL;
 FILE    *fp=NULL;
 struct stat stat_buf;
 RXSTRING *Varg=NULL, *Darg=NULL;
 int     colno=0;

 static short  not_null = 0;
 static short  is_null = (-1);

 InternalFunctionPrologue("BindByStandard");

 /*
  * For StandardPlacemarkers, each second parameter is ignored, as
  * Oracle knows the datatype of the bind value. Except for FILE.
  */
 if (argc % 2)
    return(SetIntError(62, "bind values must be paired", __FILE__,__LINE__));

 /*
  * Bind all variables (if any) to placeholders in SQL statement. It is
  * assumed that the variables are static for the life of the SQL statement
  * i.e. their addresses remain fixed.
  */
 for (i=0,colno=0;i<(int)argc && i<MAX_BINDVARS;colno++)
   {
    Darg = &argv[i++];
    Varg = &argv[i++];
    /* 
     * Get the parameter marker descriptor
     */
    if (swa->pm[colno] == (PMDSCR*)NULL)
      {
       if ((pmd = (swa->pm[colno] = (PMDSCR*)malloc(sizeof(PMDSCR)))) == (PMDSCR*)NULL)
          return(SetIntError(10, "out of memory", __FILE__,__LINE__));
       pmd->buf = (ub1*)NULL;
       pmd->buflen = 0;
      }
    else
       pmd = swa->pm[colno];
    /*
     * Check bind value type. If "FILE:", then get bind value from
     * the filename.
     */
    file_bind = 0;
    if (memcmp(RXSTRPTR(*Darg),"FILE:",5) == 0)
      {
       ub1 *ptr = (ub1 *)RXSTRPTR(*Darg);
       bind_type = ValidateDataType((char*)(ptr+5),RXSTRLEN(*Darg)-5,stmt->db);
       if (bind_type == (RXSQL_INVALID_DATATYPE))
         {
          sprintf(tmp,"invalid datatype <%s> specified.",(ptr+5));
          return(SetIntError(64, tmp, __FILE__,__LINE__));
         }
       file_bind = 1;
      }
    else
      {
       bind_type = ValidateDataType(RXSTRPTR(*Darg),RXSTRLEN(*Darg),stmt->db);
       if (bind_type == (RXSQL_INVALID_DATATYPE))
         {
          sprintf(tmp,"invalid datatype <%s> specified.",RXSTRPTR(*Darg));
          return(SetIntError(64, tmp, __FILE__,__LINE__));
         }
      }
    if (file_bind)
      {
       if ((fp = fopen(RXSTRPTR(*Varg),"rb")) == NULL)
         {
          sprintf(tmp,"<open> on file <%s> failed: %s.",RXSTRPTR(*Varg),strerror(errno));
          return(SetIntError(86, tmp, __FILE__,__LINE__));
         }
       stat(RXSTRPTR(*Varg),&stat_buf);
       bindlen = (int)stat_buf.st_size;
       /*
        * In an attempt to save malloc() & free() calls we keep the old
        * buffer providing it is not grossly oversized. 
        */
       if ((size_t)pmd->buflen < (size_t)bindlen
       || (size_t)pmd->buflen > (size_t)(bindlen + 256))
         {
          /* Free the old buffer (if any) */
          if (pmd->buf)
            {
             free(pmd->buf);
             pmd->buflen = 0;
             pmd->buf = NULL;
            }
          /* Allocate a new buffer */
          if ((pmd->buf = (ub1*)malloc(bindlen+1)) == (ub1*)NULL)
             return(SetIntError(10, "out of memory", __FILE__,__LINE__));
          pmd->buflen = bindlen;
         }
       if (fseek(fp,0L,SEEK_SET) != 0)
         {
          sprintf(tmp,"<read> on file <%s> failed: %s.",RXSTRPTR(*Varg),strerror(errno));
          return(SetIntError(86, tmp, __FILE__,__LINE__));
         }
       if ((int)fread(pmd->buf,sizeof(ub1),bindlen,fp) != bindlen)
         {
          sprintf(tmp,"<seek> on file <%s> failed: %s.",RXSTRPTR(*Varg),strerror(errno));
          return(SetIntError(86, tmp, __FILE__,__LINE__));
         }
       if (fclose(fp))
         {
          sprintf(tmp,"<close> on file <%s> failed: %s.",RXSTRPTR(*Varg),strerror(errno));
          return(SetIntError(86, tmp, __FILE__,__LINE__));
         }
       if ((memcmp((char*)pmd->buf,NullStringIn,bindlen)) == 0)
          /* indicate a NULL value */
          indp = (short*)&is_null;
       else
          indp = (short*)&not_null;
      }
    else
      {
       /*
        * In an attempt to save malloc() & free() calls we keep the old
        * buffer providing it is not grossly oversized. 
        */
       if ((size_t)pmd->buflen < RXSTRLEN(*Varg)
       || (size_t)pmd->buflen > RXSTRLEN(*Varg) + 256)
         {
          /* Free the old buffer (if any) */
          if (pmd->buf)
            {
             free(pmd->buf);
             pmd->buflen = 0;
             pmd->buf = NULL;
            }
          /* Allocate a new buffer */
          if ((pmd->buf = (ub1*)malloc(RXSTRLEN(*Varg)+1)) == (ub1*)NULL)
             return(SetIntError(10, "out of memory", __FILE__,__LINE__));
          pmd->buflen = RXSTRLEN(*Varg);
         }
       memcpy((char*)pmd->buf,RXSTRPTR(*Varg),RXSTRLEN(*Varg));
       if (memcmp(RXSTRPTR(*Varg),NullStringIn,RXSTRLEN(*Varg)) == 0)
          /* indicate a NULL value */
          indp = (short*)&is_null;
       else
          indp = (short*)&not_null;
      }
    switch(bind_type)
      {
       case ORA_RAW:     bind_type = EXT_LONGRAW; break;
       case ORA_LONGRAW: bind_type = EXT_LONGRAW; break;
       default:          bind_type = EXT_LONG; break;
      }
    if ((rc = obndrn(CTX(swa), (int)colno+1, pmd->buf, pmd->buflen, bind_type,
                                            -1, indp, (ub1*)-1, -1, -1)))
      {
       SetDBError(LDA(stmt), swa);
       return DB_ERROR;
      }
   }

 /*
  * If previous bind count is non-zero (i.e. a previous bind for 
  * this statement then check all values are re-bound. 
  */
 if (swa->bind_cnt && colno < swa->bind_cnt)
   {
    (void)sprintf(tmp, "%d bind values passed. %d expected", colno, swa->bind_cnt);
    return(SetIntError(61, tmp, __FILE__,__LINE__));
   }

 /* 
  * save the new bind count 
  */
 swa->bind_cnt = colno;

 return (0);
}

/*-----------------------------------------------------------------------------
 * Binds all arguments (by number) to the parsed (prepared) SQL statement.
 *----------------------------------------------------------------------------*/
static int BindByNumber

#ifdef HAVE_PROTO
    (STMT *stmt,ULONG argc,RXSTRING argv[])
#else
    (stmt, argc, argv)
    STMT          *stmt;
    ULONG         argc;
    RXSTRING      argv[];
#endif
{
 int     i=0;
 int     rc=0, bindlen=0;
 void    *bindaddr=NULL;
 short   *indp=NULL;
 char    tmp[128];
 SQLWA   *swa= SWA(stmt);

 static short  not_null = 0;
 static short  is_null = (-1);

 InternalFunctionPrologue("BindByNumber");

 /*
  * Bind all variables (if any) to placeholders in SQL statement. It is
  * assumed that the variables are static for the life of the SQL statement
  * i.e. their addresses remain fixed.
  */
 for (i=0;i!=(int)argc && i<MAX_BINDVARS;i++)
   {
    if (memcmp(RXSTRPTR(argv[i]),NullStringIn,RXSTRLEN(argv[i])) == 0)
      {
       /* indicate a NULL value */
       bindaddr = (void*)NULL;
       bindlen = 0;
       indp = (short*)&is_null;
      }
    else 
      {
       bindaddr = (void*)RXSTRPTR(argv[i]);
       bindlen = (int)RXSTRLEN(argv[i]);
       indp = (short*)&not_null;
      }
    if ((rc = obndrn(CTX(swa), (int)i+1, bindaddr, bindlen, EXT_VARCHAR2,
                                            -1, indp, (ub1*)-1, -1, -1)))
      {
       SetDBError(LDA(stmt), swa);
       return DB_ERROR;
      }
   }

 /*
  * If previous bind count is non-zero (i.e. a previous bind for 
  * this statement then check all values are re-bound. 
  */
 if (swa->bind_cnt && i < swa->bind_cnt)
   {
    (void)sprintf(tmp, "%d bind values passed. %d expected", i, swa->bind_cnt);
    return(SetIntError(61, tmp, __FILE__,__LINE__));
   }

 /* save the new bind count */
 swa->bind_cnt = i;

 return (0);
}

/*-----------------------------------------------------------------------------
 * Binds all arguments (by name) to the parsed (prepared) SQL statement.
 *----------------------------------------------------------------------------*/
static int BindByName
#ifdef HAVE_PROTO
    (STMT *stmt,ULONG argc,RXSTRING argv[])
#else
    (stmt, argc, argv)
    STMT          *stmt;
    ULONG         argc;
    RXSTRING	  argv[];
#endif
{
 SQLWA         *swa= SWA(stmt);
 RXSTRING      *Varg=NULL, *Narg=NULL;
 int           i=0, cnt=0;
 int           rc=0, bindlen=0;
 void          *bindaddr=NULL;
 short         *indp=NULL;
 static short  not_null = 0;
 static short  is_null = (-1);
 char          tmp[128];

 InternalFunctionPrologue("BindByName");

 /*
  * Bind all variables (if any) to placeholders in SQL statement. It is
  * assumed that the variables are static for the life of the SQL statement
  * i.e. their addresses remain fixed.
  */
 if (argc % 2)
    return(SetIntError(62, "bind values must be paired", __FILE__,__LINE__));

 for (i=0,cnt=0;i<(int)argc && cnt<=MAX_BINDVARS;)
   {
    cnt++;
    Narg = &argv[i++];
    Varg = &argv[i++];

    /*
     * Check that each substitution variable name begins with a ':' and
     * is at least 2 characters in length.
     */
    if (RXSTRLEN(*Narg) < 2 || *(RXSTRPTR(*Narg)) != ':') 
      {
       (void)sprintf(tmp,"invalid substitution variable name at bind pair %d.",cnt);
       return(SetIntError(63, tmp, __FILE__,__LINE__));
      }

    if (memcmp(RXSTRPTR(*Varg),NullStringIn,RXSTRLEN(*Varg)) == 0)
      {
       /* indicate a NULL value */
       bindaddr = (void*)NULL;
       bindlen = 0;
       indp = (short*)&is_null;
      }
    else 
      {
       bindaddr = (void*)RXSTRPTR(*Varg);
       bindlen = (int)RXSTRLEN(*Varg);
       indp = (short*)&not_null;
      }

    if ((rc = obndrv(CTX(swa), (ub1*)RXSTRPTR(*Narg), (int)RXSTRLEN(*Narg),
                        bindaddr, bindlen, EXT_VARCHAR2, -1, indp,
                        (ub1*)-1, -1, -1)))
      {
       SetDBError(LDA(stmt), swa);
       return DB_ERROR;
      }
   }

 /*
  * If previous bind count is non-zero (i.e. a previous bind for 
  * this statement then check all values are re-bound. 
  */
 if (swa->bind_cnt && cnt < swa->bind_cnt) 
   {
    (void)sprintf(tmp, "%d bind values passed. %d expected", cnt, swa->bind_cnt);
    return(SetIntError(61, tmp, __FILE__,__LINE__));
   }

 /* save the new bind count */
 swa->bind_cnt = cnt;

 return (0);
}

/*-----------------------------------------------------------------------------
 * This function binds all values (supplied as a list of arguments) to
 * the specified cursor. Two types of binding are supported:
 * (i)  By Name - This is the DEFAULT. Each bind value is a pair of consecutive
 *      arguments the first being the name of the substitution variable
 *      (placeholder) complete with a leading colon (":") and the second is
 *	the value to be bound.
 * (ii) By Number - First arg must be the token '#' anything else defaults
 *      By Name. Args are bound to placeholders in the order supplied, one
 *      argument per placeholder.
 *
 *   Parameters            Bind Type      Using Array?   Number Params
 *  ------------------------------------------------------------------
 *  '#','a','b','c',...     name             no              ?
 *  '.','a.','b.'           name             yes             3
 *  ':a','a',':b','b',...   number           no              ?
 *  '.','a.'                number           yes             2
 *  'a','b',...             standard         no              ?
 *  'a.','b.'               standard         yes             2
 *----------------------------------------------------------------------------*/
static int BindValues

#ifdef HAVE_PROTO
    (STMT *stmt,ULONG argc,RXSTRING argv[])
#else
    (stmt, argc, argv)
    STMT	  *stmt;
    ULONG         argc;
    RXSTRING	  argv[];
#endif
{
#define BIND_BY_NAME     1
#define BIND_BY_NUMBER   2
#define BIND_BY_STANDARD 3
 int bind_by_array=0,bind_type=0;
 int arglen1=0,arglen2=0;
 int rc=0;

 InternalFunctionPrologue("BindValues");

 if (StandardPlacemarkers)
   {
    bind_type = BIND_BY_STANDARD;
    if ((stmt->Argv = (RXSTRING*)malloc(2*MAX_BINDVARS*sizeof(RXSTRING))) == NULL)
       return(SetIntError(10, "out of memory", __FILE__,__LINE__));
    memset(stmt->Argv,0,2*MAX_BINDVARS*sizeof(RXSTRING));
    if (argc == 2)
      {
       arglen1 = RXSTRLEN(argv[0]);
       arglen2 = RXSTRLEN(argv[1]);
       if (arglen1 && *(RXSTRPTR(argv[0])+arglen1-1) == '.'
       &&  arglen2 && *(RXSTRPTR(argv[1])+arglen2-1) == '.')
         {
          bind_by_array = 1;
          if ((rc = ArrayToArgv(&(argv[0]),&(argv[1]),&stmt->Argc,stmt->Argv)) != 0)
            {
             stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
             return(rc);
            }
         }
       else
         {
          if (ArgvToArgv(argc,argv,&stmt->Argc,stmt->Argv))
            {
             stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
             return(SetIntError(10, "out of memory", __FILE__,__LINE__));
            }
         }
      }
    else
      {
       if (ArgvToArgv(argc,argv,&stmt->Argc,stmt->Argv))
         {
          stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
          return(SetIntError(10, "out of memory", __FILE__,__LINE__));
         }
      }
   }
 else
   {
    if ((stmt->Argv = (RXSTRING*)malloc(2*MAX_BINDVARS*sizeof(RXSTRING))) == NULL)
       return(SetIntError(10, "out of memory", __FILE__,__LINE__));
    memset(stmt->Argv,0,2*MAX_BINDVARS*sizeof(RXSTRING));
    if (RXSTRLEN(argv[0]) == 1 && *(RXSTRPTR(argv[0])) == '#')
      {
       bind_type = BIND_BY_NUMBER;
       if (ArgvToArgv(--argc,&argv[1],&stmt->Argc,stmt->Argv))
         {
          stmt->Argv = FreeArgv(&stmt->Argc, stmt->Argv);
          return(SetIntError(10, "out of memory", __FILE__,__LINE__));
         }
      }
    else
      {
       if (RXSTRLEN(argv[0]) == 1 && *(RXSTRPTR(argv[0])) == '.')
         {
          /*
           * Binding with arrays (by name or number)
           */
          bind_by_array = 1;
          if (argc == 2)
            {
             bind_type = BIND_BY_NUMBER;
             arglen1 = RXSTRLEN(argv[1]);
             if (arglen1 && *(RXSTRPTR(argv[1])+arglen1-1) == '.')
               {
                if ((rc = ArrayToArgv(&(argv[1]),NULL,&stmt->Argc,stmt->Argv)) != 0)
                  {
                   stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
                   return(rc);
                  }
               }
             else
               {
                if (ArgvToArgv(argc,argv,&stmt->Argc,stmt->Argv))
                  {
                   stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
                   return(SetIntError(10, "out of memory", __FILE__,__LINE__));
                  }
               }
            }
          else
            {
             if (argc == 3)
               {
                bind_type = BIND_BY_NAME;
                arglen1 = RXSTRLEN(argv[1]);
                arglen2 = RXSTRLEN(argv[2]);
                if (arglen1 && *(RXSTRPTR(argv[1])+arglen1-1) == '.'
                &&  arglen2 && *(RXSTRPTR(argv[2])+arglen2-1) == '.')
                  {
                   bind_by_array = 1;
                   if ((rc = ArrayToArgv(&(argv[1]),&(argv[2]),&stmt->Argc,stmt->Argv)) != 0)
                     {
                      stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
                      return(rc);
                     }
                  }
                else
                  {
                   if (ArgvToArgv(argc,argv,&stmt->Argc,stmt->Argv))
                     {
                      stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
                      return(SetIntError(10, "out of memory", __FILE__,__LINE__));
                     }
                  }
               }
             else
               {
                stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
                return(-1);
               }
            }
         }
       else
         {
          bind_type = BIND_BY_NAME;
          if (ArgvToArgv(argc,argv,&stmt->Argc,stmt->Argv))
            {
             stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
             return(SetIntError(10, "out of memory", __FILE__,__LINE__));
            }
         }
      }
   }

 /* 
  * Call the appropriate bind function...
  */
 switch(bind_type)
   {
    case BIND_BY_NAME:
         rc = BindByName(stmt, stmt->Argc, stmt->Argv);
         break;
    case BIND_BY_NUMBER:
         rc = BindByNumber(stmt, stmt->Argc, stmt->Argv);
         break;
    case BIND_BY_STANDARD:
         rc = BindByStandard(stmt, stmt->Argc, stmt->Argv);
         break;
   }

 return (rc);
}

/*-----------------------------------------------------------------------------
 * Get a column expression descriptor.  Assumes the index is valid!
 *----------------------------------------------------------------------------*/
static FLDDSCR *GetFldDscr

#ifdef HAVE_PROTO
    (SQLWA *swa,int i)
#else
    (swa, i)
    SQLWA   *swa;
    int	    i;
#endif
{
 FLDDSCR *fd=NULL;

 InternalFunctionPrologue("GetFldDscr");

 fd = swa->fa[i];
 if (fd == (FLDDSCR*)NULL)
   {
    fd = swa->fa[i] = (FLDDSCR*)malloc(sizeof(FLDDSCR));
    if (fd == (FLDDSCR*)NULL) 
      {
       return (FLDDSCR*)NULL;
      }
    fd->rbuf = (ub1*)NULL;
    fd->cbuf = (ub1*)malloc(MAX_NAMELEN+1);
    if (fd->cbuf == (ub1*)NULL)
      {
       free(fd);
       fd = swa->fa[i] = NULL;
       return (FLDDSCR*)NULL;
      }

    fd->rbufl = 0;
    fd->cbufl = 0;
   }
 return fd;
}

/*-----------------------------------------------------------------------------
 * Describes columns (expressions) and defines the output buffers for each
 * column for the select statement in nominated cursor.
 * Returns the number of expressions in the select statement.
 *----------------------------------------------------------------------------*/
static int DefineExpressions

#ifdef HAVE_PROTO
    (STMT *stmt)
#else
    (stmt)
    STMT     *stmt;
#endif
{
 SQLWA    *swa= SWA(stmt);
 FLDDSCR  *fd=NULL;
 int      i=0;
 sb4      dsize=0, dbsize=0;
 size_t   varlen=0;
 int      ext_type=EXT_LONG;

 InternalFunctionPrologue("DefineExpressions");

 /* Describe & define buffer for each expression in the SELECT statement */
 for (i = 0; i < MAX_COLS ; i++)
   {
    /* Get a new field descriptor */
    if ((fd = GetFldDscr(swa, i)) == (FLDDSCR*)NULL)
       return(SetIntError(10, "out of memory", __FILE__,__LINE__));

    fd->cbufl = MAX_NAMELEN - 1;

    if (odescr(CTX(swa), i+1, &dbsize, &fd->dbtype, (sb1*)fd->cbuf,
               &fd->cbufl, &dsize, &fd->prec, &fd->scale, &fd->nullok))
      {
       if (V4ERRORCODE(CTX(swa))
       &&  V4ERRORCODE(CTX(swa)) != 1007)
         {
          SetDBError(LDA(stmt), swa);
          return (DB_ERROR);
         }
       break;
      }

    fd->cbuf[fd->cbufl] = '\0';     /* Terminate the Column Name */

    ext_type = EXT_LONG;
    switch (fd->dbtype) 
      {
       case ORA_DATE:   
            varlen = ORA_DATE_LEN;  
            break;
       case ORA_LONGRAW:
            varlen = LongLimit;
            ext_type = EXT_LONGRAW;
            break;
       case ORA_RAW:
            varlen = dsize;         
            ext_type = EXT_LONGRAW;
            break;
       case ORA_LONG:   
            varlen = LongLimit;
            break;
       default:         
            varlen = dsize;         
            break;
      }

    /*
     * In an attempt to save malloc() & free() calls we keep the old
     * buffer providing it is not grossly oversized. 
     */
    if ((size_t)fd->rbufl < varlen || (size_t)fd->rbufl > varlen + 256)
      {
       /* Free the old buffer (if any) */
       if (fd->rbufl) 
         {
          free(fd->rbuf);
          fd->rbufl = 0;
          fd->rbuf = NULL;
         }
       /* Allocate a new buffer */
       if ((fd->rbuf = (ub1*)malloc(varlen)) == (ub1*)NULL)
          return(SetIntError(10, "out of memory", __FILE__,__LINE__));
      }
    fd->rbufl = varlen;

    if (odefin(CTX(swa), i+1, fd->rbuf, fd->rbufl, ext_type, 0,
               &fd->indp, (text*)NULL, 0, 0, (ub2*)&fd->retl,(ub2*)&fd->rcode))
      {
       SetDBError(LDA(stmt), swa);
       return (DB_ERROR);
      }
   }

 /*
  ** Check if too many SELECT expressions.
  ** ORACLE should trap it if MAX_COLS is set correctly!
  */
 if (i >= MAX_COLS) 
     return(SetIntError(71, "Too many columns specified in SELECT", __FILE__,__LINE__));

 return (swa->expr_cnt = i);
}

/*-----------------------------------------------------------------------------
 * Fetches the next row from the nominated cursor and returns the values
 * of the expressions for the fetched row into the compound variable with
 * name constructed as follows:
 *
 * For single fetches. eg. SQLFETCH('s1')
 *                     or  SQLFETCH('s1','') :
 *  <statement-name>.<column-name>
 *
 * For bulk fetches. eg. SQLCOMMAND(stmt1)
 *                   or  SQLFETCH('s1',0)
 *                   or  SQLFETCH('s1',1)
 *                   or  SQLFETCH('s1',20) :
 *  <statement-name>.<column-name>.<row-number>
 *
 * Note that the row-number always starts at 1!
 *
 * Returns:
 *	success:  0
 *	failure: ORACLE return code (V2 codes).
 *----------------------------------------------------------------------------*/
static int FetchRow

#ifdef HAVE_PROTO
    (STMT *stmt,char *stem,ULONG rowcount)
#else
    (stmt, stem, rowcount)
    STMT    *stmt;
    char    *stem;
    ULONG   rowcount;
#endif
{
 FLDDSCR *fd=NULL;
 SQLWA   *swa=NULL;
 int	    rc=0, i=0;
 size_t  varlen=0;
 char    varname[MAX_IDENTIFIER+1+10];
 char    tmp[128];

 InternalFunctionPrologue("FetchRow");

 swa = SWA(stmt);

 rc = ofetch(CTX(swa));
 if (rc == NO_DATA_FOUND)
    return (rc);
 if (rc)
   {
    SetDBError(LDA(stmt), swa);
    return DB_ERROR;
   }

 /* Get each expr value in turn */
 for (i = 0; rc == 0 && i < swa->expr_cnt; i++)
   {
    fd = swa->fa[i];
        
    /* Add each column value to the stem's values */
    (void)sprintf(varname, rowcount ? "%s.%s.%lu" : "%s.%s",
                      stem, fd->cbuf, rowcount);
    varlen = strlen(varname);

    if (fd->indp < 0)      /* returned value is NULL */
      {
       rc = SetRexxVariable(varname, varlen, RXSNullStringOut.strptr, RXSNullStringOut.strlength);
      }
    else 
      {
       if (fd->rcode)    /* truncation or conversion error */
         {
          if (IgnoreTruncate)
            rc = SetRexxVariable(varname, varlen, (char*)fd->rbuf, (size_t)fd->rbufl);
          else
            {
             sprintf(tmp,"Conversion/truncation occurred on column %s: Expecting %d, got %d",
                     fd->cbuf,fd->cbufl,fd->retl);
             return(SetIntError(15, tmp, __FILE__,__LINE__));
            }
         }
       else 
          rc = SetRexxVariable(varname, varlen, (char*)fd->rbuf, (size_t)fd->retl);
      }
   }
 if (rc)
    return(SetIntError(16, "unable to set REXX variable", __FILE__,__LINE__));
 return rc;
}

/*-----------------------------------------------------------------------------
 *
 *----------------------------------------------------------------------------*/
static int SetRowCountVar

#ifdef HAVE_PROTO
    (SQLWA *swa,char *stem_name,long rowcount)
#else
    (swa, stem_name, rowcount)
    SQLWA *swa;
    char *stem_name;
    long rowcount;
#endif
{
 int     i=0,rc=0;
 char    varname[MAX_IDENTIFIER*2+4], buf[11];

 InternalFunctionPrologue("SetRowCountVar");

 for (i = 0; i < swa->expr_cnt; i++) 
   {
    sprintf(varname, "%s.%s.0", stem_name,swa->fa[i]->cbuf);
    sprintf(buf, "%lu", rowcount);
    if (rc = SetRexxVariable(varname,strlen(varname),
                         buf,strlen(buf)))
       return(rc);
   }
 return 0;
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS: SQLCOMMAND(stem-name, sql-statement-text [, <bind-args>])
 *
 * RETURNS :  0-success < 0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLCOMMAND

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
 int         rc = 0, i=0;
 ULONG       rowcount=0L;
 int		expr_cnt=0;
 DBCON       *db=NULL;
 STMT        *stmt=NULL;
 SQLWA	*swa=NULL;
 char        stem_name[MAX_IDENTIFIER+1];
 int         stmt_len=0;
 char        *stmt_ptr=NULL;

 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc == 0) return 1;

 /* Get pointer to current connection */
 if (DbEnv && DbEnv->current_connection)
    db = DbEnv->current_connection;
 else
    return ReturnError(retstr, 25, "no connection is current");

 if (argc == 1 || RXSTRLEN(argv[0]) == 0) /* No stem name specified! */
    (void)strcpy(stem_name, DEFAULT_STEM);
 else 
   {
    if ((rc = MkIdentifier(argv[0], stem_name, sizeof(stem_name))))
       return ReturnInt(retstr, rc);
   }

 /* If no default statement then create it! */
 if ((stmt = (STMT*)(db->dflt_stmt)) == (STMT*)NULL) 
   {
    /* Open a statement for the default statement. */
    if ((rc = OpenStatement(DEFAULT_STATEMENT, db, &stmt)))
       return ReturnInt(retstr, (long)rc);

    db->dflt_stmt = (void*)stmt;
   }

 swa = SWA(stmt);

 /*
  * If only 1 arg then it is the SQL-statement-text. If more than 1 args
  * then arg#1 is stem-name, arg#2 is text and rest are bind values.
  * 'i' is index (base zero) to sql-statement-text arg.
  */
 i = (argc == 1) ? 0 : 1;

 /* set flag indicating if SELECT statement supplied */
 swa->select = SelectStatement(RXSTRPTR(argv[i]),RXSTRLEN(argv[i]));

 /*
  * If the request is to use "standard" placemarkers; ?, then we
  * need to convert these into Oracle's placemarkers, :1 -> :n
  */
 if (StandardPlacemarkers)
   {
    if ((stmt_ptr = ConvertPlacemarkers(RXSTRPTR(argv[i]),(int)RXSTRLEN(argv[i]),&stmt_len)) == NULL)
       return ReturnError(retstr, 10, "out of memory");
   }
 else
   {
    stmt_ptr = RXSTRPTR(argv[i]);
    stmt_len = (int)RXSTRLEN(argv[i]);
   }

 /* Save the SQL statement. MUST do this as oexec fails if you don't */
 if (SaveSqlStatement(&(swa->sql_stmt), &(swa->sql_stmt_sz), stmt_ptr,stmt_len))
    return ReturnError(retstr, 10, "out of memory");

 /* Parse the statement */
#if defined(NO_OPARSE)
 if ((rc = osql3(CTX(swa), stmt_ptr,stmt_len)))
   {
    SetDBError(LDA(stmt), swa);
    return ReturnInt(retstr, DB_ERROR);
   }
#else
 if ((rc = oparse(CTX(swa), (ub1*)swa->sql_stmt, stmt_len, 1, 1L)))
   {
    SetDBError(LDA(stmt), swa);
    return ReturnInt(retstr, DB_ERROR);
   }
#endif
 /* Bind variables in WHERE/INSERT/UPDATE clause (if any). */
 if (argc > 2) 
   {
    if ((rc = BindValues(stmt, argc-2, &argv[2])))
      {
       stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
       return ReturnInt(retstr, (long)rc);
      }
   }

 /* Describe the expression list & define output buffers (if any). */
 if (swa->select)
   {
    if ((expr_cnt = DefineExpressions(stmt)) < 0)
      {
       stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
       return ReturnInt(retstr, (long)expr_cnt);
      }
   }

 /* Execute the SQL statement */
 if ((rc = oexec(CTX(swa))))
   {
    SetDBError(LDA(stmt), swa);
    stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
    return ReturnInt(retstr, DB_ERROR);
   }

 stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);

 /* Test if select statement. If not then done else fetch rows. */
 if (expr_cnt == 0) 
   { 
    rowcount = SQLROWCNT(CTX(swa));
    /* Inform ORACLE that operation is complete */
    if ((rc = ocan(CTX(swa))))   /* this should never fail */
      {
       SetDBError(LDA(stmt), swa);
       return ReturnInt(retstr, DB_ERROR);
      }
    /*
     * If the statement is NOT a SELECT, and we have AUTOCOMMIT
     * ON, then commit the statement.
     */
    if (AutoCommit)
      {
       if ((rc = ocom(LDA(stmt))))
          SetDBError(LDA(stmt), (SQLWA*)NULL);
      }
   }
 else /* Then must be a SELECT statement */
   {
    /* Fetch each row in turn */
    for (rowcount = 1; RowLimit == 0 || rowcount <= RowLimit; rowcount++)
      {
       if ((rc = FetchRow(stmt, stem_name, rowcount)))
          break;
      }
    rowcount--;

    if (rc && rc != NO_DATA_FOUND) 
      {
       return ReturnInt(retstr, (long)rc);
      }

    if ((rc = SetRowCountVar(swa, stem_name, rowcount)))
       return ReturnInt(retstr, (long)rc);

    /* Inform ORACLE that operation is complete */
    if ((rc = ocan(CTX(swa)))) /* this should never fail */
      {
       SetDBError(LDA(stmt), swa);
       return ReturnInt(retstr, DB_ERROR);
      }
   }
 SetRowCount(rowcount);

 /* 
  * Release the statement and associated storage. 
  */
 if ((rc = ReleaseStatement(stmt)))
    return ReturnInt(retstr, (long)rc);
 db->dflt_stmt = (STMT*)NULL;

 return ReturnInt(retstr, 0L);
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS:  SQLPREPARE(statement-name, sql-statement-text)
 *
 * RETURNS :  0-success, <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLPREPARE

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
 int         rc=0;
 DBCON       *db=NULL;
 STMT        *stmt=NULL;
 SQLWA	*swa=NULL;
 char        stmt_name[MAX_IDENTIFIER+1];
 int         stmt_len=0;
 char        *stmt_ptr=NULL;

 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc != 2) return 1;

 /* Get pointer to current connection */
 if (DbEnv && DbEnv->current_connection)
     db = DbEnv->current_connection;
 else
     return ReturnError(retstr, 25, "no connection is current");

 if (RXSTRLEN(argv[0]) == 0)		/* No statement name specified! */
     return ReturnError(retstr, 23, "statement name omitted or null");
 else if ((rc = MkIdentifier(argv[0], stmt_name, sizeof(stmt_name))))
     return ReturnInt(retstr, rc);

 /*
  * Find the named statement or create if necessary. We have to be a
  * bit careful here because the statement may exist but point to a
  * different database connection!
  */
 stmt = FindObject(stmt_name, DbEnv->stmt_tbl, TBL_STATEMENTS);

 if (stmt == (STMT*)NULL || stmt->db != db) {

     if (stmt) {

         /*
          * Statement is not for the same db, therefore we must dispose
          * & re-alloc it!
          */
         RemoveObject(stmt);
         ((DBCON*)stmt->db)->num_statements--;
         if ((rc = ReleaseStatement(stmt)))
             return ReturnInt(retstr, (long)rc);
     }

     /* Open a statement for this statement. */
     if ((rc = OpenStatement(stmt_name, db, &stmt)))
         return ReturnInt(retstr, (long)rc);

     /* Insert this statement into the connection hash table. */
     (void)InsertObject(stmt, DbEnv->stmt_tbl, TBL_STATEMENTS);

     db->num_statements++;
 }

 swa = SWA(stmt);
 /* set flag indicating if SELECT statement supplied */
 swa->select = SelectStatement(RXSTRPTR(argv[1]),RXSTRLEN(argv[1]));

 /*
  * If the request is to use "standard" placemarkers; ?, then we
  * need to convert these into Oracle's placemarkers, :1 -> :n
  */
 if (StandardPlacemarkers)
   {
    if ((stmt_ptr = ConvertPlacemarkers(RXSTRPTR(argv[1]),(int)RXSTRLEN(argv[1]),&stmt_len)) == NULL)
       return ReturnError(retstr, 10, "out of memory");
   }
 else
   {
    stmt_ptr = RXSTRPTR(argv[1]);
    stmt_len = (int)RXSTRLEN(argv[1]);
   }

 /* Save the SQL statement. MUST do this as oexec fails if you don't */
 if (SaveSqlStatement(&(swa->sql_stmt), &(swa->sql_stmt_sz), stmt_ptr,stmt_len))
     return ReturnError(retstr, 10, "out of memory");

 if (StandardPlacemarkers)
    free(stmt_ptr);

 swa->bind_cnt = 0;
 swa->expr_cnt = 0;

 /* Parse the statement */
#if defined(NO_OPARSE)
 if ((rc = osql3(CTX(swa), stmt_ptr, stmt_len)))
   {
    SetDBError(LDA(stmt), swa);
   }
#else
 if ((rc = oparse(CTX(swa), (ub1*)swa->sql_stmt, stmt_len, 1, 1L)))
   {
    SetDBError(LDA(stmt), swa);
   }
#endif

 return ReturnInt(retstr, rc ? DB_ERROR : 0);
}

/*-----------------------------------------------------------------------------
 * Get a pointer to the nominated statement. Returns NULL on error.
 *----------------------------------------------------------------------------*/
static STMT *GetStatement

#ifdef HAVE_PROTO
    (RXSTRING	var,PSZ		buf)
#else
    (var, buf)
    RXSTRING	var; 
    PSZ		buf;
#endif
{
 STMT        *stmt=NULL;
 char        tmp[128];

 InternalFunctionPrologue("GetStatement");

 if (DbEnv == (DBENV*)NULL) 
   {
    SetIntError(22, "no connections open.", __FILE__,__LINE__);
    return (STMT*)NULL;
   }

 if (RXSTRLEN(var) == 0)
   {
    /* 
     * No statement name specified! 
     */
    SetIntError(23, "statement name omitted or null", __FILE__,__LINE__);
    return (STMT*)NULL;
   }

 /* 
  * Get the normalised form of the name 
  */
 if (MkIdentifier(var, buf, MAX_DB_CURSOR_NAME+1))
    return (STMT*)NULL;

 /* 
  * Find the named statement or create if necessary 
  */
 if ((stmt = FindObject(buf, DbEnv->stmt_tbl, TBL_STATEMENTS)) == (STMT*)NULL)
   {
    /*
     * Statement not an existing one!
     */
    (void)sprintf(tmp,"statement \"%s\" does not exist", buf);
    SetIntError(24, tmp,__FILE__,__LINE__);
    return (STMT*)NULL;
   }
 return stmt;
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS:  SQLDISPOSE(statement-name)
 *
 * RETURNS :  0-success, <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLDISPOSE

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
 int         rc=0;
 STMT        *stmt=NULL;
 DBCON       *db=NULL;
 char        stmt_name[MAX_IDENTIFIER+1];

 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc != 1) return 1;

 if ((stmt = GetStatement(argv[0], stmt_name)) == (STMT*)NULL)
     return ReturnInt(retstr, SQLCA_IntCode);
 
 /* Get pointer to statement's connection structure */
 db = stmt->db;

 /* Dispose the statement */
 RemoveObject(stmt);
 rc = ReleaseStatement(stmt);

 db->num_statements--;

 return ReturnInt(retstr, (long)rc);
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS:  SQLEXEC (called by SQLEXECUTE() and SQLOPEN()
 *
 * RETURNS :  0-success,
 *           >0-number of rows affected for SQLEXECUTE(),
 *           <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLEXEC

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr,int open)
#else
    (name, argc, argv, stck, retstr, open)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
    int open;
#endif
{
 int         rc=0;
 int		expr_cnt=0;
 ULONG	rowcount=0L;
 STMT        *stmt=NULL;
 SQLWA	*swa=NULL;
 char        stmt_name[MAX_IDENTIFIER+1];
 char        tmp[128];
 DBCON       *db=NULL;

 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc == 0) return 1;

 if ((stmt = GetStatement(argv[0], stmt_name)) == (STMT*)NULL)
    return ReturnInt(retstr, SQLCA_IntCode);

 swa = SWA(stmt);

 /*
  * Make sure we have a current connection
  */
 if (DbEnv == (DBENV*)NULL)
    return ReturnError(retstr, 22, "no connections open.");

 /* 
  * Get pointer to current connection 
  */
 if (DbEnv->current_connection)
    db = DbEnv->current_connection;
 else
    return ReturnError(retstr, 25, "no connection is current");

 /*
  * Rest of args (if any) are bind values. Therefore bind the substitution
  * variables in WHERE/INSERT/UPDATE clause (if any).
  */
 argc--;
 if (argc) 
   {
    if ((rc = BindValues(stmt, argc, &argv[1])))
      {
       stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
       return ReturnInt(retstr, (long)rc);
      }
   }

 /*
  * This function can be called as SQLOPEN() or SQLEXECUTE(). These
  * operations are similar except for the describe. Use the function
  * name to determine what operation we are performing.
  */
 if (open) 
   {
    /* 
     * Describe the expression list & define output buffers (if any). 
     */
    if ((expr_cnt = DefineExpressions(stmt)) < 0) 
      {
       stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
       return ReturnInt(retstr, (long)expr_cnt);
      }

    /* 
     * Since this is a call to SQLOPEN() the statement must be a query! 
     */
    if (expr_cnt == 0) 
      { 
       if (!SEL(swa))
         {
          (void)sprintf(tmp,"statement \"%s\" is not a query.", stmt_name);
          stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
          return ReturnError(retstr, 13, tmp);
         }
       else
         {
          SetDBError(LDA(stmt), swa);
          stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
          return ReturnInt(retstr, DB_ERROR);
         }
      }
   }

 /* 
  * Execute the SQL statement. This is required for Queries & DML! 
  */
 if ((rc = oexec(CTX(swa))))
   {
    SetDBError(LDA(stmt), swa);
    stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);
    return ReturnInt(retstr, DB_ERROR);
   }

 stmt->Argv = FreeArgv(&stmt->Argc,stmt->Argv);

 /*
  * Return the ROWCOUNT.  For a query it will be zero at this stage.  For
  * a DML statement it will be the number of rows affected by the INSERT/
  * UPDATE/DELETE statement.
  */
 rowcount = SQLROWCNT(CTX(swa));
 SetRowCount(rowcount);

 /*
  * If the statement is NOT called from SQLOPEN(), and we have AUTOCOMMIT
  * ON, then commit the statement.
  */
 if (!open 
 && AutoCommit)
   {
    if ((rc = ocom(LDA(stmt))))
       SetDBError(LDA(stmt), (SQLWA*)NULL);
   }
 return ReturnInt(retstr, (long)0L);
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS: SQLCLOSE(statement-name)
 *
 * RETURNS :  0-success, <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLCLOSE

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
 int         rc=0;
 STMT        *stmt=NULL;
 SQLWA	*swa=NULL;
 char        stmt_name[MAX_IDENTIFIER+1];

 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc != 1) return 1;

 if ((stmt = GetStatement(argv[0], stmt_name)) == (STMT*)NULL)
     return ReturnInt(retstr, SQLCA_IntCode);
 
 swa = SWA(stmt);

 /* Inform ORACLE that operation is complete. This should never fail! */
 if ((rc = ocan(CTX(swa))))
     SetDBError(LDA(stmt), swa);

 return ReturnInt(retstr, rc ? DB_ERROR : 0);
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS:  SQLFETCH(statement-name [, number-of-rows])
 *
 * RETURNS :  0-end-of-data,
 *           >0- single fetch: row number of last row fetched
 *           >0- group fetch : number of rows fetched if < number-of-rows then
 *                             end--of-data is indicated.
 *           <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLFETCH

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
 long        rc=0;
 int         single_fetch=0;
 ULONG       num_rows=0L;
 ULONG       rowcount=0L;
 STMT        *stmt=NULL;
 SQLWA	*swa=NULL;
 char        stmt_name[MAX_IDENTIFIER+1];

 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc == 0 || argc > 2) return 1;

 if ((stmt = GetStatement(argv[0], stmt_name)) == (STMT*)NULL)
     return ReturnInt(retstr, SQLCA_IntCode);
 
 swa = SWA(stmt);

 /* Determine # of rows to fetch */
 if (argc > 1 && RXSTRLEN(argv[1])) 
   {
    if (StrToInt(&argv[1], &num_rows)) 
      {
       return ReturnError(retstr, 14,
                            "<num-rows> is not a valid integer.");
      }
    single_fetch = FALSE;
   }
 else
     single_fetch = TRUE;

 if (single_fetch) 
   {
    /* Fetch a single row */
    if ((rc = FetchRow(stmt, stmt_name, 0L)))
       rc = (rc == NO_DATA_FOUND) ? 0 : rc;
    else
       rc = SQLROWCNT(CTX(swa));	/* Nasty! */
   }
 else 
   {
    /* Fetch each row in turn */
    for (rowcount = 1; num_rows == 0 || rowcount <= num_rows; rowcount++)
       if ((rc = FetchRow(stmt, stmt_name, rowcount)))
          break;

    rowcount--;

    if (rc && rc != NO_DATA_FOUND) 
      {
       return ReturnInt(retstr, (long)rc);
      }

    if ((rc = SetRowCountVar(swa, stmt_name, rowcount)))
       return ReturnInt(retstr, (long)rc);

    rc = rowcount;
   }

 SetRowCount(SQLROWCNT(CTX(swa)));
 return ReturnInt(retstr, rc);
}

/*-----------------------------------------------------------------------------
 * SYNOPSIS:  SQLGETDATA(statement-name, column-name, start-byte, length [, file-name])
 *
 * RETURNS :  0-end-of-data,
 *           >0- bytes obtained - <= length
 *           <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLGETDATA

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
#define LONG_BLOCK_SIZE 0x8000      /* 32k */
 FLDDSCR *fd=NULL;
 long    rc=0,ret_len=0;
 ULONG   num_bytes=0L;
 ULONG   start_byte=0L;
 STMT    *stmt=NULL;
 SQLWA   *swa=NULL;
 char    stmt_name[MAX_IDENTIFIER+1];
 int     long_col=0,ext_type=0,i=0;
 size_t  varlen=0;
 char    varname[MAX_IDENTIFIER+1+10];
 FILE    *fp=NULL;
 char    tmp[128];                      

 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc < 4 || argc > 5) return 1;

 if ((stmt = GetStatement(argv[0], stmt_name)) == (STMT*)NULL)
     return ReturnInt(retstr, SQLCA_IntCode);
 
 swa = SWA(stmt);

 /* 
  * Check that the column name specified is really a "LONG" and
  * get column number.
  */
 if (RXSTRLEN(argv[1]))
   {
    for (i=0;i<swa->expr_cnt;i++)
      {
       fd = swa->fa[i];
       if (memcmp(RXSTRPTR(argv[1]),fd->cbuf,RXSTRLEN(argv[1])) == 0)
         {
          /*
           * Found column name, is it a "LONG" ?
           */
          if (!DBLONGCOLUMN(fd->dbtype))
            {
             sprintf(tmp,"Column <%s> does not have a 'LONG' datatype.",fd->cbuf);
             return ReturnError(retstr, 85,tmp);
            }
          long_col = i+1;
          break;
         }
      }
    if (long_col == 0)
      {
       sprintf(tmp,"Column <%s> not present in statement.",fd->cbuf);
       return ReturnError(retstr, 85,tmp);
      }
   }
 else
   {
    return ReturnError(retstr, 84,"<column-name> parameter MUST be supplied.");
   }

 /* 
  * Determine starting byte
  */
 if (RXSTRLEN(argv[2]))
   {
    if (StrToInt(&argv[2], &start_byte))
      {
       return ReturnError(retstr, 14,"<start-byte> is not a valid integer.");
      }
   }
 else
     start_byte = 0L;

 /* 
  * Determine length
  */
 if (RXSTRLEN(argv[3]))
   {
    if (StrToInt(&argv[3], &num_bytes))
      {
       return ReturnError(retstr, 14,"<number-bytes> is not a valid integer.");
      }
   }
 else
     num_bytes = 0L;

 /* 
  * Check for filename if start and length = 0
  */
 if (start_byte == 0L
 &&  num_bytes == 0L
 &&  RXSTRLEN(argv[4])==0)
   {
    return ReturnError(retstr, 84,"<file-name> parameter MUST be supplied.");
   }

 ext_type = (fd->dbtype==ORA_RAW||fd->dbtype==ORA_LONGRAW)?EXT_LONGRAW:EXT_LONG;
 if (start_byte == 0L
 &&  num_bytes == 0L)
   {
    /*
     * Process a file
     */
    ub1 *buf=NULL;

    if ((fp = fopen(RXSTRPTR(argv[4]),"wb")) == NULL)
      {
       sprintf(tmp,"<open> on file <%s> failed: %s.",RXSTRPTR(argv[4]),strerror(errno));
       return ReturnError(retstr, 86,tmp);
      }
    if ((buf = (ub1*)malloc(LONG_BLOCK_SIZE+1)) == (ub1*)NULL)
       return ReturnError(retstr, 10, "out of memory");
    ret_len = 0;
    for (i=0;;i++)
      {
       if (oflng(CTX(swa),long_col,buf,LONG_BLOCK_SIZE,ext_type,(ub4*)&ret_len,(i*LONG_BLOCK_SIZE)))
         {
          if (buf) free(buf);
          SetDBError(LDA(stmt), swa);
          return ReturnInt(retstr, DB_ERROR);
         }
       if (ret_len == 0)
          break;
       if ((long)fwrite((void*)buf,sizeof(ub1),ret_len,fp) != ret_len)
         {
          if (buf) free(buf);
          sprintf(tmp,"<write> on file <%s> failed: %s.",RXSTRPTR(argv[4]),strerror(errno));
          return ReturnError(retstr, 86, tmp);
         }
      }
    if (buf) free(buf);
    if (fclose(fp))
      {
       sprintf(tmp,"<close> on file <%s> failed: %s.",RXSTRPTR(argv[4]),strerror(errno));
       return ReturnError(retstr, 86, tmp);
      }
   }
 else 
   {
    /* 
     * Get the chunk of data specified
     * Size of the chunk MUST be < 64K
     */
    if (num_bytes >= 0x10000)
       return ReturnError(retstr, 87,"<number-bytes> parameter must be < <65536>.");
    /*
     * In an attempt to save malloc() & free() calls we keep the old
     * buffer providing it is not grossly oversized. 
     */
    if ((size_t)fd->rbufl < num_bytes 
    || (size_t)fd->rbufl > num_bytes + 256)
      {
       /* Free the old buffer (if any) */
       if (fd->rbufl) 
         {
          free(fd->rbuf);
          fd->rbufl = 0;
          fd->rbuf = NULL;
         }
       /* Allocate a new buffer */
       if ((fd->rbuf = (ub1*)malloc(num_bytes)) == (ub1*)NULL)
          return ReturnError(retstr, 10, "out of memory");
      }
    fd->rbufl = num_bytes;
    if (oflng(CTX(swa),long_col,fd->rbuf,fd->rbufl,ext_type,(ub4*)&ret_len,(start_byte-1)))
      {
       SetDBError(LDA(stmt), swa);
       return ReturnInt(retstr, DB_ERROR);
      }
    (void)sprintf(varname, "%s.%s", stmt_name, fd->cbuf);
    varlen = strlen(varname);
    rc = SetRexxVariable(varname, varlen, (char*)fd->rbuf, (size_t)ret_len);
   }

 return ReturnInt(retstr, ret_len);
}


/*-----------------------------------------------------------------------------
 * Fetch the description for the column expression. Used by SQLDESCRIBE().
 *----------------------------------------------------------------------------*/
static int GetColumn

#ifdef HAVE_PROTO
    (SQLWA *swa,int i,char *stem_name)
#else
    (swa, i, stem_name)
    SQLWA   *swa;
    int	    i;
    char    *stem_name;
#endif
{
 FLDDSCR *fd=NULL;
 int     idx=0, rc=0;
 char    column_size[7], column_prec[7], column_scale[7], column_nullok[7];
 char    *column_type=NULL;
 char    name[MAX_IDENTIFIER+32];
 char    *value[NUM_DESCRIBE_COLUMNS];
 int     value_len[NUM_DESCRIBE_COLUMNS];

 InternalFunctionPrologue("GetColumn");

 if (i >= swa->expr_cnt)
     return 1;

 fd = swa->fa[i];

 for (idx=0;datatype_conversion[idx].ext_datatype;idx++)
   {
    if (datatype_conversion[idx].int_datatype == fd->dbtype)
      {
       column_type = datatype_conversion[idx].ext_datatype;
      }
   }
 if (column_type == NULL)
    column_type = "UNKNOWN";
#if 0
 switch (fd->dbtype) 
   {
    case ORA_CHAR:     column_type = "CHAR";     break;
    case ORA_VARCHAR2: column_type = "VARCHAR2"; break;
    case ORA_RAW:      column_type = "RAW";      break;
    case ORA_NUMBER:   column_type = "NUMBER";   break;
    case ORA_DATE:     column_type = "DATE";     break;
    case ORA_LONG:     column_type = "LONG";     break;
    case ORA_LONGRAW:  column_type = "LONG RAW"; break;
    case ORA_ROWID:    column_type = "ROWID";    break;
    case ORA_MLSLABEL: column_type = "MLSLABEL"; break;
    default:           column_type = "UNKNOWN";  break;
   }
#endif

 /* Set up the array */
 value[0] = (char*)fd->cbuf;
 value_len[0] = fd->cbufl;

 value[1] = column_type;
 value_len[1] = strlen(column_type);

 (void)sprintf(column_size, "%ld", fd->rbufl);
 value[2] = column_size;
 value_len[2] = strlen(value[2]);

 (void)sprintf(column_prec, "%d", fd->prec);
 value[3] = column_prec;
 value_len[3] = strlen(value[3]);

 (void)sprintf(column_scale, "%d", fd->scale);
 value[4] = column_scale;
 value_len[4] = strlen(value[4]);

 (void)sprintf(column_nullok, "%d", fd->nullok);
 value[5] = column_nullok;
 value_len[5] = strlen(value[5]);

 /* Output into Rexx variable */
 i++;
 for (idx = 0; idx < NUM_DESCRIBE_COLUMNS; idx++) 
   {
    sprintf(name, "%s.COLUMN.%s.%d", stem_name,column_attribute[idx], i);
    if (rc = SetRexxVariable(name,strlen(name),value[idx],value_len[idx]))
       break;
   }

 return rc;
}



/*-----------------------------------------------------------------------------
 * SYNOPSIS:  SQLDESCRIBE(statement-name [, stem-name])
 *
 * RETURNS : >0-number of columns in the prepared SELECT statement.
 *            0-prepared statement is not a SELECT statement.
 *           <0-error.
 *----------------------------------------------------------------------------*/
ULONG RXSQL_APIENTRY SQLDESCRIBE

#ifdef HAVE_PROTO
    (RXSQL_PUCHAR		name,ULONG	argc,RXSTRING	argv[],RXSQL_PCSZ		stck,RXSTRING	*retstr)
#else
    (name, argc, argv, stck, retstr)
    RXSQL_PUCHAR		name;
    ULONG	argc;
    RXSTRING	argv[];
    RXSQL_PCSZ		stck;
    RXSTRING	*retstr;
#endif
{
 int         i=0, len1=0, len2=0, rc=0;
 STMT        *stmt=NULL;
 SQLWA	*swa=NULL;
 char        **p=NULL, buf1[MAX_IDENTIFIER+32], buf2[16];
 char        stmt_name[MAX_IDENTIFIER+1];
 char        stem_name[MAX_IDENTIFIER+1];


 FunctionPrologue(name, argc, argv);

 if (SQLCA_SqlCode) ClearDBError();
 if (SQLCA_IntCode) ClearIntError();

 if (argc < 1 || argc > 2) return 1;

 if ((stmt = GetStatement(argv[0], stmt_name)) == (STMT*)NULL)
     return ReturnInt(retstr, SQLCA_IntCode);
 
 swa = SWA(stmt);

 /* Get the name of the stem into which to put output */
 if (argc < 2 || RXSTRLEN(argv[1]) == 0) /* No stem name specified! */
    (void)strcpy(stem_name, stmt_name);
 else 
    if ((rc = MkIdentifier(argv[1], stem_name, sizeof(stem_name))))
       return ReturnInt(retstr, (long)rc);

 /* Describe the expression list & define output buffers (if any). */
 if ((rc = DefineExpressions(stmt)) < 0)
     return ReturnInt(retstr, (long)rc);

 /* Describe the parsed statement into REXX variables */
 for (rc = 0, i = 0; rc == 0; i++) 
   {
    rc = GetColumn(swa, i, stem_name);
   }

 if (rc >= 0) 
   {
    (void)sprintf(buf2,"%d", --i);
    len2 = strlen(buf2);
    for (p = column_attribute; *p && rc >= 0; p++) 
      {
       (void)sprintf(buf1, "%s.COLUMN.%s.0", stem_name, *p);
       len1 = strlen(buf1);
       rc = SetRexxVariable(buf1, len1, buf2, len2);
      }
    rc = rc < 0 ? rc : i;
   }

 return ReturnInt(retstr, (long)rc);
}

int DBInitialise
#ifdef HAVE_PROTO
    (void)
#else
    ()
#endif
{
 return(0);
}

/*-----------------------------------------------------------------------------
 * Set up the string containing database name
 *----------------------------------------------------------------------------*/
static void *SetDBMSName

#ifdef HAVE_PROTO
    (DBCON *db)
#else
    (db)
    DBCON *db;
#endif
{
 InternalFunctionPrologue("SetDBMSName");
 strcpy(DBMSName,"ORACLE");
 MAKERXSTRING(RXSDBMSName,DBMSName,(ULONG)strlen(DBMSName));
 return((void*)&RXSDBMSName);
}

/*-----------------------------------------------------------------------------
 * Set up the DBMSVer variable
 *----------------------------------------------------------------------------*/
static void *SetDBMSVer

#ifdef HAVE_PROTO
    (DBCON *db)
#else
    (db)
    DBCON *db;
#endif

{
 strcpy(DBMSVer,"unknown");
 MAKERXSTRING(RXSDBMSVer,DBMSVer,(ULONG)strlen(DBMSVer));
 return((void*)&RXSDBMSVer);
}

/*-----------------------------------------------------------------------------
 * Set up the Datatypes supported
 *----------------------------------------------------------------------------*/
static void *SetDatatypes

#ifdef HAVE_PROTO
    (DBCON *db)
#else
    (db)
    DBCON *db;
#endif
{
 int i=0;
 RXSTRING_DT *dt;

 InternalFunctionPrologue("SetDatatypes");

 dt = (RXSTRING_DT *)malloc(sizeof(RXSTRING_DT)*(MAX_DATATYPES+1));
 for (i=0;i<MAX_DATATYPES,(datatype_conversion[i].ext_datatype!=NULL);i++)
   {
    dt[i].strptr = (char*)malloc(strlen(datatype_conversion[i].ext_datatype)+1);
    strcpy(dt[i].strptr, datatype_conversion[i].ext_datatype);
    dt[i].strlength = strlen(datatype_conversion[i].ext_datatype);
    dt[i].internal = datatype_conversion[i].int_datatype;
   }
 dt[i].strptr = NULL;
 dt[i].strlength = 0;
 dt[i].internal = 0;
 return((void*)dt);
}

/*-----------------------------------------------------------------------------
 * Set up the Describe Columns supported
 *----------------------------------------------------------------------------*/
static void *SetDescribeColumns

#ifdef HAVE_PROTO
    (DBCON *db)
#else
    (db)
    DBCON *db;
#endif
{
 int i=0;
 RXSTRING *dc=NULL;

 InternalFunctionPrologue("SetDescribeColumns");

 dc = (RXSTRING *)malloc(sizeof(RXSTRING)*(NUM_DESCRIBE_COLUMNS+1));
 for (i=0;i<NUM_DESCRIBE_COLUMNS;i++)
   {
    dc[i].strptr = (char*)malloc(strlen(column_attribute[i])+1);
    strcpy(dc[i].strptr, column_attribute[i]);
    dc[i].strlength = strlen(column_attribute[i]);
   }
 dc[i].strptr = NULL;
 dc[i].strlength = 0;
 return((void*)dc);
}

/*-----------------------------------------------------------------------------
 * Set up the SupportsTransactions variable
 *----------------------------------------------------------------------------*/
static void *SetSupportsTransactions

#ifdef HAVE_PROTO
    (DBCON *db)
#else
    (idb)
    DBCON *idb;
#endif
{
 InternalFunctionPrologue("SetSupportsTransactions");
 return((void*)&SupportsTransactions);
}

/*-----------------------------------------------------------------------------
 * Set up the SupportsSqlGetData variable
 *----------------------------------------------------------------------------*/
static void *SetSupportsSqlGetData

#ifdef HAVE_PROTO
    (DBCON *db)
#else
    (db)
    DBCON *db;
#endif
{
 InternalFunctionPrologue("SetSupportsSqlGetData");
 return((void*)&SupportsSqlGetData);
}

#include "common.h"
