/***********************************************************************
 * util.c - Rexx/SQL utility functions
 ***********************************************************************
 *
 * Rexx/SQL. A Rexx interface to SQL databases.
 * Copyright Mark Hessling, 1996-1997.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to 
 * address below.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                    Email:       M.Hessling@qut.edu.au
 * PO Box 203                       Phone:              +617 3802 0800
 * Bellara                          http://www.lightlink.com/hessling/
 * QLD 4507                         **** Author of THE & Rexx/SQL ****
 * Australia                        ****** Maintainer PDCurses *******
 *
 */

#include "dbdefine.h"
#if defined(UNIX)
# include "dbheader.h"
#endif
#include "rexxsql.h"

extern int run_flags;
extern char FName[40];
extern SQLVAL *opt_tbl;
extern char *column_attribute[];

extern RXSQL_SQLCODE_TYPE  SQLCA_SqlCode;
extern RXSQL_ROWCOUNT_TYPE SQLCA_RowCount;
extern long SQLCA_IntCode;
#if defined(USES_COMPUTE_COUNT)
extern RXSQL_COMPUTECOUNT_TYPE  SQLCA_ComputeCount;
#endif

#if !defined(DYNAMIC_LIBRARY) && (defined(USE_WINREXX) || defined(USE_QUERCUS))
RexxExitHandler RxExitHandlerForSayTraceRedirection;
#endif

/*
 * Sigh.. REXX/6000 uses RexxRegisterFunction() rather than
 * RexxRegisterFunctionExe() when installing external function
 * packages from an executable.
 * It gets worse :-(
 * Because of the way that AIX support dynamic libraries,
 * REXX/6000 uses a completely different mechanism for
 * registering functions from a dynamic library...
 */
#if defined(USE_REXX6000)
#  define RexxRegisterFunctionExe RexxRegisterFunction
#else

/*-----------------------------------------------------------------------------
 * Table entry for a REXX/SQL function.
 *----------------------------------------------------------------------------*/
typedef	struct {
	RXSQL_PCSZ	external_name;
	RXSQL_PRXFUNC	EntryPoint;
	RXSQL_PCSZ	internal_name;
} RexxFunction;

/*-----------------------------------------------------------------------------
 * Table of REXX/SQL Functions. Used to install/de-install functions.
 *----------------------------------------------------------------------------*/
static RexxFunction RexxSqlFunctions[] = {
  { (RXSQL_PCSZ)NAME_SQLCONNECT_EXT,    (RXSQL_PRXFUNC)SQLCONNECT   ,(RXSQL_PCSZ)NAME_SQLCONNECT_INT         },
  { (RXSQL_PCSZ)NAME_SQLDISCONNECT_EXT, (RXSQL_PRXFUNC)SQLDISCONNECT,(RXSQL_PCSZ)NAME_SQLDISCONNECT_INT      },
  { (RXSQL_PCSZ)NAME_SQLDEFAULT_EXT,    (RXSQL_PRXFUNC)SQLDEFAULT   ,(RXSQL_PCSZ)NAME_SQLDEFAULT_INT         },
  { (RXSQL_PCSZ)NAME_SQLCOMMIT_EXT,     (RXSQL_PRXFUNC)SQLCOMMIT    ,(RXSQL_PCSZ)NAME_SQLCOMMIT_INT          },
  { (RXSQL_PCSZ)NAME_SQLROLLBACK_EXT,   (RXSQL_PRXFUNC)SQLROLLBACK  ,(RXSQL_PCSZ)NAME_SQLROLLBACK_INT        },
  { (RXSQL_PCSZ)NAME_SQLCOMMAND_EXT,    (RXSQL_PRXFUNC)SQLCOMMAND   ,(RXSQL_PCSZ)NAME_SQLCOMMAND_INT         },
  { (RXSQL_PCSZ)NAME_SQLPREPARE_EXT,    (RXSQL_PRXFUNC)SQLPREPARE   ,(RXSQL_PCSZ)NAME_SQLPREPARE_INT         },
  { (RXSQL_PCSZ)NAME_SQLDISPOSE_EXT,    (RXSQL_PRXFUNC)SQLDISPOSE   ,(RXSQL_PCSZ)NAME_SQLDISPOSE_INT         },
  { (RXSQL_PCSZ)NAME_SQLEXECUTE_EXT,    (RXSQL_PRXFUNC)SQLEXECUTE   ,(RXSQL_PCSZ)NAME_SQLEXECUTE_INT         },
  { (RXSQL_PCSZ)NAME_SQLOPEN_EXT,       (RXSQL_PRXFUNC)SQLOPEN      ,(RXSQL_PCSZ)NAME_SQLOPEN_INT            },
  { (RXSQL_PCSZ)NAME_SQLCLOSE_EXT,      (RXSQL_PRXFUNC)SQLCLOSE     ,(RXSQL_PCSZ)NAME_SQLCLOSE_INT           },
  { (RXSQL_PCSZ)NAME_SQLFETCH_EXT,      (RXSQL_PRXFUNC)SQLFETCH     ,(RXSQL_PCSZ)NAME_SQLFETCH_INT           },
  { (RXSQL_PCSZ)NAME_SQLGETDATA_EXT,    (RXSQL_PRXFUNC)SQLGETDATA   ,(RXSQL_PCSZ)NAME_SQLGETDATA_INT         },
  { (RXSQL_PCSZ)NAME_SQLVARIABLE_EXT,   (RXSQL_PRXFUNC)SQLVARIABLE  ,(RXSQL_PCSZ)NAME_SQLVARIABLE_INT        },
  { (RXSQL_PCSZ)NAME_SQLGETINFO_EXT,    (RXSQL_PRXFUNC)SQLGETINFO   ,(RXSQL_PCSZ)NAME_SQLGETINFO_INT         },
  { (RXSQL_PCSZ)NAME_SQLDESCRIBE_EXT,   (RXSQL_PRXFUNC)SQLDESCRIBE  ,(RXSQL_PCSZ)NAME_SQLDESCRIBE_INT        },
  { (RXSQL_PCSZ)NAME_SQLDROPFUNCS_EXT,  (RXSQL_PRXFUNC)SQLDROPFUNCS ,(RXSQL_PCSZ)NAME_SQLDROPFUNCS_INT       },
#if !defined(DYNAMIC_LIBRARY)
  { (RXSQL_PCSZ)NAME_SQLLOADFUNCS_EXT,  (RXSQL_PRXFUNC)SQLLOADFUNCS ,(RXSQL_PCSZ)NAME_SQLLOADFUNCS_INT       },
#endif
  { NULL,                               NULL                        ,NULL                                    }
};
#endif
/*
 * This variable is used to determine if the dynamic library version
 * of Rexx/SQL has been initialised.  It is used to run InitRexxSQL()
 * after the DLL has been loaded, but SQLLoadFuncs() is not called
 * again.
 */
ULONG RexxSQLInitialised = 0;

/*
 * Settable variables
 * 
 * ---------------------------------------------------------------------
 * Limits number of rows returned by SQLCOMMAND(). If set to zero (0) then
 * all rows in query are returned (unless memory is exhausted in which case
 * an error is returned)!
 */
ULONG RowLimit =  0;	/* Default is unlimited! */

/* ---------------------------------------------------------------------
 * Limits size of a "LONG" column to the specified size.
 * This limit is only imposed by a normal row FETCH, and is ignored by
 * the size specified in SqlGetData() 
 */
ULONG LongLimit = DEFAULT_LONG_LIMIT;	/* Default is 32k */

/* ---------------------------------------------------------------------
 * If TRUE, the SQL statement is saved on each parse (SqlCommand()
 * & SqlPrepare()).  If TRUE then on errors, the last SQL statement can be
 * retrieved in a REXX program by calling SQLVARIABLE().
 * eg.      error_stmt = SqlVariable('sqltext');
 */
ULONG SaveSQL = TRUE;	/* Default is to save each SQL statement! */

/* ---------------------------------------------------------------------
 * If TRUE, the default AUTOCOMMIT behaviour for statements is ON.
 */
ULONG AutoCommit = FALSE;	/* Default is to have AUTOCOMMIT OFF! */

/* ---------------------------------------------------------------------
 * If TRUE, any truncation errors will be ignored
 */
ULONG IgnoreTruncate = FALSE;	/* Default is to fail on truncate errors */

/* ---------------------------------------------------------------------
 * If TRUE, REXX/SQL will allow the user to supply ? as placemarkers
 * instead of :1 or :name placemarkers. Only applicable to Oracle.
 */
ULONG StandardPlacemarkers = FALSE;	/* Default is to NOT substitute ? */

/* ---------------------------------------------------------------------
 * If TRUE, the database supports transactions.
 */
ULONG SupportsTransactions = SUPPORTSTRANSACTIONS;	/* Default database dependent */

/* ---------------------------------------------------------------------
 * If TRUE, the database supports placemarkers.
 */
ULONG SupportsPlacemarkers = SUPPORTSPLACEMARKERS;	/* Default database dependent */

/* ---------------------------------------------------------------------
 * If TRUE, the database sets SQLCA.ROWCOUNT for DML statements
 */
ULONG SupportsDMLRowcount = SUPPORTSDMLROWCOUNT;	/* Default database dependent */

/* ---------------------------------------------------------------------
 * If TRUE, the database supports the SqlGetData() function
 */
ULONG SupportsSqlGetData = SUPPORTSSQLGETDATA;	/* Default database dependent */

/* ---------------------------------------------------------------------
 * String representing a NULL value returned from a SELECT statement.
 * Default is the empty string; ""
 */
RXSTRING RXSNullStringOut;
char NullStringOut[MAX_IDENTIFIER+1]; /* set in InstallSQLVariables() */

/* ---------------------------------------------------------------------
 * String representing a NULL value when passed as a bind value.
 * Default is ""
 */
RXSTRING RXSNullStringIn;
char NullStringIn[MAX_IDENTIFIER+1]; /* set in InstallSQLVariables() */

#if !defined(HAVE_STRERROR)
/*
 * This function and the following description borrowed from Regina 0.08a
 *
 * Sigh! This must probably be done this way, although it's incredibly 
 * backwards. Some versions of gcc comes with a complete set of ANSI C
 * include files, which contains the definition of strerror(). However,
 * that function does not exist in the default libraries of SunOS. 
 * To circumvent that problem, strerror() is #define'd to get_sys_errlist() 
 * in config.h, and here follows the definition of that function. 
 * Originally, strerror() was #defined to sys_errlist[x], but that does
 * not work if string.h contains a declaration of the (non-existing) 
 * function strerror(). 
 *
 * So, this is a mismatch between the include files and the library, and
 * it should not create problems for Regina. However, the _user_ will not
 * encounter any problems until he compiles Regina, so we'll have to 
 * clean up after a buggy installation of the C compiler!
 */
char *rxsql_get_sys_errlist( int num )
{
   extern char *sys_errlist[] ;
   return sys_errlist[num] ;
}
#endif


/*-----------------------------------------------------------------------------
 * Compares buffers for equality ignoring case
 *----------------------------------------------------------------------------*/
int memcmpi

#ifdef HAVE_PROTO
    (char *buf1, char *buf2, int len)
#else
    (buf1,buf2,len)
    char    *buf1;
    char    *buf2;
    int     len;
#endif
{
 register short i=0;
 char c1=0,c2=0;
 for(i=0;i<len;i++)
   {
    if (isupper(*buf1))
       c1 = tolower(*buf1);
    else
       c1 = *buf1;
    if (isupper(*buf2))
       c2 = tolower(*buf2);
    else
       c2 = *buf2;
    if (c1 != c2)
       return(c1-c2);
    ++buf1;
    ++buf2;
   }
 return(0);
}
/*-----------------------------------------------------------------------------
 * Uppercases the supplied string.
 *----------------------------------------------------------------------------*/
char *make_upper

#ifdef HAVE_PROTO
    (char *str)
#else
    (str)
    char    *str;
#endif

{
 char *save_str=str;
 while(*str)
   {
    if (islower(*str))
       *str = toupper(*str);
    ++str;
   }
 return(save_str);
}

/*-----------------------------------------------------------------------------
 * Allocate memory for a char * based on an RXSTRING
 *----------------------------------------------------------------------------*/
char *AllocString

#ifdef HAVE_PROTO
    (char *buf, size_t bufsize)
#else
    (buf, bufsize)
    char    *buf;
    size_t  bufsize;
#endif
{
 char *tempstr=NULL;

 tempstr = (char *)malloc(sizeof(char)*(bufsize+1));
 return tempstr;
}


/*-----------------------------------------------------------------------------
 * Copy a non terminated character array to the nominated buffer (truncate
 * if necessary) and null terminate.
 *----------------------------------------------------------------------------*/
char *MkAsciz

#ifdef HAVE_PROTO
    (char *buf, size_t bufsize, char *str, size_t len)
#else
    (buf, bufsize, str, len)
    char    *buf;
    size_t  bufsize;
    char    *str;
    size_t  len;
#endif

{
 bufsize--;	/* Make room for terminating byte */
 if (len > bufsize)
    len = bufsize;
 (void)memcpy(buf, str, len);
 buf[len] = '\0';
 return buf;
}
/*-----------------------------------------------------------------------------
 * Parse the supplied statement and return 1 if the first word of the
 * statement is "select" (any case) or 0 otherwise.
 *----------------------------------------------------------------------------*/
int SelectStatement

#ifdef HAVE_PROTO
    (char *stmt,long len)
#else
    (stmt,len)
    char    *stmt;
    long    len;
#endif

{
 char buf[7];
 char *p=stmt;
 int i=0,j=0;


 /* Remove leading whitespace */
 while (len && isspace(*p)) 
   {
    p++;
    len--;
   }
 if (len<6L)
    return(0);
 buf[6] = '\0';

 for(i=0;i<6;i++)
   {
    buf[i] = (islower(*p)) ? toupper(*p) : *p;
    p++;
   }
 if (strcmp(buf,"SELECT") == 0)
    return(1);
 return(0);
}

/*-----------------------------------------------------------------------------
 * Store the error message for INTERNAL errors in SQLCA.SQLTEXT variable.
 *----------------------------------------------------------------------------*/
int SetIntError

#ifdef HAVE_PROTO
    (int errcode, char *errmsg, char *fn, long lineno)
#else
    (errcode,errmsg,fn,lineno)
    int   errcode;
    char *errmsg;
    char *fn;
    long  lineno;
#endif

{
 char  msg[MAX_ERROR_TEXT];

 InternalFunctionPrologue("SetIntError");

 SQLCA_IntCode = -errcode;

 /* 
  * Set SQLCA.INTERRM variable
  */
 if (run_flags == 10)
    (void)sprintf(msg, "REXX/SQL-%-02d: %s [%s:%d]", errcode, errmsg, fn, lineno);
 else
    (void)sprintf(msg, "REXX/SQL-%-02d: %s", errcode, errmsg);
 (void)SetRexxVariable(SQLCA_INTERRM, strlen(SQLCA_INTERRM),msg, strlen(msg));

 /* 
  * Set SQLCA.INTCODE variable 
  */
 (void)sprintf(msg, "%d", SQLCA_IntCode);
 (void)SetRexxVariable(SQLCA_INTCODE, strlen(SQLCA_INTCODE),msg, strlen(msg));

 return(SQLCA_IntCode);
}

/*-----------------------------------------------------------------------------
 * Set elements of the compound variable "SQLCA.".  SQLCA. holds the results of
 * any call to REXXSQL. It should never be set by the REXX program!
 *----------------------------------------------------------------------------*/
void SetSQLCA

#ifdef HAVE_PROTO
    (RXSQL_SQLCODE_TYPE sqlcode, RXSQL_SQLSTATE_TYPE *sqlstate, RXSQL_SQLERRM_TYPE *errm, char *txt)
#else
    (sqlcode, sqlstate, errm, txt)
    RXSQL_SQLCODE_TYPE   sqlcode;
    RXSQL_SQLSTATE_TYPE  *sqlstate;
    RXSQL_SQLERRM_TYPE   *errm;
    char   *txt;
#endif

{
 char    buf[50];

 InternalFunctionPrologue("SetSQLCA");

 SQLCA_SqlCode = sqlcode;

 (void)sprintf(buf, "%ld", sqlcode);
 (void)SetRexxVariable(SQLCA_SQLCODE,  strlen(SQLCA_SQLCODE),buf, strlen(buf));
 (void)SetRexxVariable(SQLCA_SQLERRM,  strlen(SQLCA_SQLERRM),errm, strlen(errm));
 (void)SetRexxVariable(SQLCA_SQLTEXT,  strlen(SQLCA_SQLTEXT),txt, strlen(txt));
 (void)SetRexxVariable(SQLCA_SQLSTATE, strlen(SQLCA_SQLSTATE),sqlstate, strlen(sqlstate));

 if (!(sqlcode==0 && strcmp(sqlstate,"")==0 && strcmp(errm,"")==0))
   {
    (void)SetRexxVariable(SQLCA_INTCODE, strlen(SQLCA_INTCODE),"-1", 2);
    (void)sprintf(buf, "REXX/SQL-1: Database Error");
    (void)SetRexxVariable(SQLCA_INTERRM, strlen(SQLCA_INTERRM),buf, strlen(buf));
   }
 return;
}

/*-----------------------------------------------------------------------------
 * Set RowCount and put into REXX "SQLCA.ROWCOUNT" compound variable.
 *----------------------------------------------------------------------------*/
void SetRowCount

#ifdef HAVE_PROTO
    (RXSQL_ROWCOUNT_TYPE rowcount)
#else
    (rowcount)
    RXSQL_ROWCOUNT_TYPE rowcount;
#endif

{
 char    buf[50];

 InternalFunctionPrologue("SetRowCount");

 if (SQLCA_RowCount != rowcount) 
   {
    SQLCA_RowCount = rowcount;
    (void)sprintf(buf, "%lu", rowcount);
    (void)SetRexxVariable(SQLCA_ROWCOUNT,  strlen(SQLCA_ROWCOUNT),buf, strlen(buf));
   }
 return;
}

#if defined(USES_COMPUTE_COUNT)
/*-----------------------------------------------------------------------------
 * Set ComputeCount and put into REXX "SQLCA.COMPUTECOUNT" compound variable.
 *----------------------------------------------------------------------------*/
void SetComputeCount

#ifdef HAVE_PROTO
    (RXSQL_COMPUTECOUNT_TYPE computecount)
#else
    (computecount)
    RXSQL_COMPUTECOUNT_TYPE computecount;
#endif

{
 char    buf[50];

 InternalFunctionPrologue("SetComputeCount");

 if (SQLCA_ComputeCount != computecount) 
   {
    SQLCA_ComputeCount = computecount;
    (void)sprintf(buf, "%lu", computecount);
    (void)SetRexxVariable(SQLCA_COMPUTECOUNT,  strlen(SQLCA_COMPUTECOUNT),buf, strlen(buf));
   }
 return;
}
#endif

/*-----------------------------------------------------------------------------
 * Clear the error code, error message etc.
 *----------------------------------------------------------------------------*/
void ClearDBError

#ifdef HAVE_PROTO
    (void)
#else
    ()
#endif

{
 SetRowCount(0L);
#if defined(USES_COMPUTE_COUNT)
 SetComputeCount(0L);
#endif
 SetSQLCA(0L, "", "", "");
 return;
}

/*-----------------------------------------------------------------------------
 * Clear the internal error code, error message etc.
 *----------------------------------------------------------------------------*/
void ClearIntError

#ifdef HAVE_PROTO
    (void)
#else
    ()
#endif

{
 SQLCA_IntCode = 0L;

 /* 
  * Set SQLCA.INTERRM variable 
  */
 (void)SetRexxVariable(SQLCA_INTERRM, strlen(SQLCA_INTERRM),"", 0);

 /* 
  * Set SQLCA.INTCODE variable 
  */
 (void)SetRexxVariable(SQLCA_INTCODE, strlen(SQLCA_INTCODE),"0", 1);

 return;
}

/*-----------------------------------------------------------------------------
 * Create a REXX variable of the specified name and bind the value to it.
 *----------------------------------------------------------------------------*/
int SetRexxVariable

#ifdef HAVE_PROTO
    (char *name, size_t namelen, char *value, size_t valuelen)
#else
    (name, namelen, value, valuelen)
    char    *name;
    size_t  namelen;
    char    *value;
    size_t  valuelen;
#endif

{
 ULONG	rc=0L;
 SHVBLOCK	shv;

 if (run_flags & MODE_DEBUG) 
   {
    char buf1[50], buf2[50];

    (void)fprintf(stderr, "*DEBUG* Setting variable \"%s\" to \"%s\".\n",
                      MkAsciz(buf1, sizeof(buf1), name, namelen),
                      MkAsciz(buf2, sizeof(buf2), value, valuelen));
   }
 shv.shvnext = (SHVBLOCK*)NULL;
#if defined(USE_REGINA)
 shv.shvcode = RXSHV_SYSET;
#else
 shv.shvcode = RXSHV_SET;
#endif
 MAKERXSTRING(shv.shvname, name, (ULONG)namelen);
 MAKERXSTRING(shv.shvvalue, value, (ULONG)valuelen);
 shv.shvnamelen = shv.shvname.strlength;
 shv.shvvaluelen = shv.shvvalue.strlength;
 assert(shv.shvname.strptr);
 assert(shv.shvvalue.strptr);
 rc = RexxVariablePool(&shv);
 if (rc == RXSHV_OK
 ||  rc == RXSHV_NEWV)
    return(0);
 else
    return(-16);
}

/*-----------------------------------------------------------------------------
 * Converts a RXSTRING to integer. Return 0 if OK and -1 if error.
 * Assumes a string of decimal digits and does not check for overflow!
 *----------------------------------------------------------------------------*/
int StrToInt

#ifdef HAVE_PROTO
    (RXSTRING *ptr, ULONG *result) 
#else
    (ptr, result) 
    RXSTRING *ptr;
    ULONG    *result; 
#endif

{
 int    i=0;
 char   *p=NULL;
 ULONG  sum=0L;

 p = ptr->strptr;
 for (i = ptr->strlength; i; i--)
   {
    if (isdigit(*p))
       sum = sum * 10 + (*p++ - '0');
    else
       return -1;
   }
 *result = sum;
 return 0;
}


/*-----------------------------------------------------------------------------
 * Converts a RXSTRING to a boolean. Input can be ON, Yes, 1 or OFF No, 0
 *----------------------------------------------------------------------------*/
int StrToBool

#ifdef HAVE_PROTO
    (RXSTRING *ptr, ULONG *result) 
#else
    (ptr, result) 
    RXSTRING *ptr;
    ULONG    *result; 
#endif

{
 char   *p=ptr->strptr;
 int    len=ptr->strlength;

 if (memcmp(p,"YES",len) == 0
 ||  memcmp(p,"yes",len) == 0
 ||  memcmp(p,"Y",len) == 0
 ||  memcmp(p,"y",len) == 0
 ||  memcmp(p,"ON",len) == 0
 ||  memcmp(p,"on",len) == 0
 ||  memcmp(p,"1",len) == 0)
   {
    *result = 1;
    return(0);
   }
 if (memcmp(p,"NO",len) == 0
 ||  memcmp(p,"no",len) == 0
 ||  memcmp(p,"N",len) == 0
 ||  memcmp(p,"n",len) == 0
 ||  memcmp(p,"OFF",len) == 0
 ||  memcmp(p,"off",len) == 0
 ||  memcmp(p,"0",len) == 0)
   {
    *result = 0;
    return(0);
   }
 return (-1);
}



/*-----------------------------------------------------------------------------
 * This is called when in VERBOSE mode. It prints function name & arg values.
 *----------------------------------------------------------------------------*/
void FunctionPrologue

#ifdef HAVE_PROTO
    (RXSQL_PSZ name, ULONG argc, RXSTRING *argv)
#else
    (name, argc, argv)
    RXSQL_PSZ		name;
    ULONG	argc;
    RXSTRING	*argv;
#endif

{
 ULONG	i=0L;
 char	buf[61];

#if defined(DYNAMIC_LIBRARY)
 if (RexxSQLInitialised != MAGIC_NUMBER)
    InitRexxSQL(DLLNAME,0);
#endif
 if (run_flags & MODE_VERBOSE) 
   {
    (void)fprintf(stderr, "++\n");
    (void)fprintf(stderr, "++ Call %s%s\n", name, argc ? "" : "()");
    for (i = 0; i < argc; i++) 
      {
       (void)fprintf(stderr, "++ %3ld: \"%s\"\n", i+1,
                          MkAsciz(buf, sizeof(buf), RXSTRPTR(argv[i]),
                                  (size_t)RXSTRLEN(argv[i])));
      }
    if (argc) (void)fprintf(stderr, "++\n");
   }

 /* Set SQLCA.FUNCTION variable */
 if (strcmp(name,FName) != 0)
   {
    (void)SetRexxVariable(SQLCA_FUNCTION, strlen(SQLCA_FUNCTION),
                          name, strlen(name));
    strcpy(FName,name);
   }
}


/*-----------------------------------------------------------------------------
 * This is called when run_flags = 10
 *----------------------------------------------------------------------------*/
void InternalFunctionPrologue

#ifdef HAVE_PROTO
    (char *name)
#else
    (name)
    char *name;
#endif

{
 if (run_flags == 10)
    (void)fprintf(stderr, ">>>> Call %s\n", name);
 return;
}

/*-----------------------------------------------------------------------------
 * Copy a string "s" to the RXSTRING structure pointed to by "target".
 * The string is not null terminated (unless the null is included as the
 * last byte of the string and is counted in "len".
 *----------------------------------------------------------------------------*/
ULONG PutString

#ifdef HAVE_PROTO
    (RXSTRING *target, char *s, size_t len)
#else
    (target, s, len)
    RXSTRING	*target;
    char	*s;
    size_t      len;
#endif

{
 char        *t=NULL;

 if (len > RXAUTOBUFLEN)
   {
#if defined(WIN32) && (defined(USE_OREXX) || defined(USE_WINREXX) || defined(USE_QUERCUS))
    if ((t = (char*)GlobalLock(GlobalAlloc(GMEM_FIXED,len+1))) == (char*)NULL)
#elif defined(USE_OS2REXX)
    if ((BOOL)DosAllocMem( (void**)&t, len+1, fPERM|PAG_COMMIT))
#else
    if ((t = (char*)malloc(len+1)) == (char*)NULL)
#endif
    if ((t = (char*)malloc(len+1)) == (char*)NULL)
       return 1;
    target->strptr = (RXSQL_RXSTRPTR)t;
   }

 memcpy(target->strptr, s, len);
 target->strlength = len;
 return 0;
}



/*-----------------------------------------------------------------------------
 * Return an integer number as the return value of the function. Also
 * handles verbose mode!
 *----------------------------------------------------------------------------*/
ULONG ReturnInt

#ifdef HAVE_PROTO
    (RXSTRING *retptr, long retval)
#else
    (retptr, retval)
    RXSTRING	*retptr;
    long	retval;
#endif

{
 char        buf[32];

 (void)sprintf(buf, "%ld", retval);

 if (run_flags & MODE_VERBOSE)
    (void)fprintf(stderr, "++ Exit %s with value \"%ld\"\n", FName, retval);
 
 return PutString(retptr, buf, strlen(buf));
}



/*-----------------------------------------------------------------------------
 * Return a string as the return value of the function. Also
 * handles verbose mode!
 *----------------------------------------------------------------------------*/
ULONG ReturnString

#ifdef HAVE_PROTO
    (RXSTRING *retptr, char *val, size_t len)
#else
    (retptr, val, len)
    RXSTRING	*retptr;
    char	*val;
    size_t      len;
#endif

{
 if (run_flags & MODE_VERBOSE) 
   {
    char  buf[50];

    (void)fprintf(stderr, "++ Exit %s with value \"%s\"\n",
                      FName, MkAsciz(buf, sizeof(buf), val, len));
   }

 return PutString(retptr, val, len);
}



/*-----------------------------------------------------------------------------
 * Return an error to REXX interpreter.
 *----------------------------------------------------------------------------*/
ULONG ReturnError

#ifdef HAVE_PROTO
    (RXSTRING *retptr, int errcode, char *errmsg)
#else
    (retptr, errcode, errmsg)
    RXSTRING	*retptr;
    int         errcode;
    char	*errmsg;
#endif

{
 SetIntError(errcode, errmsg, __FILE__, __LINE__);
 return ReturnInt(retptr, -errcode);
}




/*-----------------------------------------------------------------------------
 * Makes an identifier (for a Connection, Stem or Statement) from a given
 * string. Leading and trailing whitespace is removed and the string is
 * capitalised. The variable must be the following regular expression:
 *   [A-Za-z!?][A-Za-z0-9!?_\.]*
 * That is the variable :-
 *   (i) The variable name must start with an alpha character or '!' or '?'.
 *  (ii) The rest of the variable name must be one of the above characters or
 *       a decimal digit or underscore or period.
 *----------------------------------------------------------------------------*/
int MkIdentifier

#ifdef HAVE_PROTO
    (RXSTRING var, PSZ buf, size_t buflen)
#else
    (var, buf, buflen)
    RXSTRING  var;
    PSZ       buf;
    size_t    buflen;
#endif

{
 char      *p = RXSTRPTR(var);
 size_t    len = RXSTRLEN(var);
 size_t    cnt = 0;

 if (len == 0)
    return(SetIntError(51, "zero length identifier", __FILE__, __LINE__));

 /* 
  * Make room for the terminating byte. 
  */
 buflen--;

 /* 
  * Remove leading whitespace 
  */
 while (len && isspace(*p)) 
   {
    p++;
    len--;
   }

 /* 
  * Special check for 1st character 
  */
 if (len && (isalpha(*p) || *p == '!' || *p == '?')) 
   {
    *buf++ = (islower(*p)) ? toupper(*p) : *p;
    p++;
    len--;
   }

 /* 
  * Copy identifier to destination 
  */
 while (len && (isalnum(*p) || *p == '_' || *p == '!' || *p == '?' || *p== '.'))
   {
    *buf = (islower(*p)) ? toupper(*p) : *p;
    p++;
    len--;
    if (++cnt <= buflen) 
       buf++;
   }

 /* 
  * Remove trailing whitespace 
  */
 while (len && isspace(*p)) 
   {
    p++;
    len--;
   }

 /* 
  * Check for garbage 
  */
 if (len)
    return(SetIntError(52, "garbage in identifier name",__FILE__, __LINE__));

 *buf = '\0';
 return 0;
}

/*-----------------------------------------------------------------------------
 * Save SQL statement in cursor statement buffer.  Dynamically extends the
 * statement buffer when required.
 *----------------------------------------------------------------------------*/
int SaveSqlStatement

#ifdef HAVE_PROTO
    (char **sql_stmt, int *sql_stmt_sz, char *statement, int length)
#else
    (sql_stmt, sql_stmt_sz, statement, length)
    char    **sql_stmt;
    int     *sql_stmt_sz;
    char    *statement;
    int     length;
#endif

{
 int len=length+1;
 char *ptr=NULL;

 if (len > *sql_stmt_sz)
   {
    /* New statement size is > than previous */
    if (*sql_stmt != (char*)NULL)
      {
       /* Free old statement */
       free(*sql_stmt); 
       *sql_stmt = (char*)NULL;
       *sql_stmt_sz = 0;
      }

    /* Allocate a buffer for the new statement */
    if ((*sql_stmt = (char*)malloc(len)) == (char*)NULL)
       return(SetIntError(10, "out of memory", __FILE__, __LINE__));
    *sql_stmt_sz = len;
   }

 /* Save the statement */
 (void)memcpy(*sql_stmt, statement, length);
 ptr=*sql_stmt;
 ptr[length] = '\0';
 return 0;
}

/*-----------------------------------------------------------------------------
 * Install a REXX/SQL variable structure.
 *----------------------------------------------------------------------------*/
SQLVAL *InstallSQLVariable

#ifdef HAVE_PROTO
    (int *rc,char *name, SQLVAL *first, int upd, int dtype, int max, int option, void *val)
#else
    (rc,name, first, upd, dtype, max, option, val)
    int    *rc;
    char   *name;
    SQLVAL *first;
    int    upd;
    int    dtype;
    int    max;
    int    option;
    void   *val;
#endif

{
 SQLVAL  *opt=NULL;
 SQLVAL  *new=NULL;

 InternalFunctionPrologue("InstallSQLVariable");

 if ((opt = (SQLVAL*)ll_add(first,first)) == (SQLVAL*)NULL)
   {
    *rc = SetIntError(10, "out of memory", __FILE__, __LINE__);
    return(NULL);
   }
 (void)strcpy(opt->name, name);
 opt->user_update = (char)upd;
 opt->dtype = (char)dtype;
 opt->maxlen = max;
 opt->value = val;
 opt->option = option;
 *rc = 0;
 return(opt);
}

/*-----------------------------------------------------------------------------
 * This function gets a REXX variable and returns it in an RXSTRING
 *----------------------------------------------------------------------------*/
static RXSTRING *GetRexxVariable

#ifdef HAVE_PROTO
    (char *name, RXSTRING *value, int suffix)
#else
    (name, value, suffix)
    char *name;
    RXSTRING *value;
    int suffix;
#endif
{
 static SHVBLOCK shv;
 char variable_name[50];
 long rc=0;

 InternalFunctionPrologue("GetRexxVariable");

 shv.shvnext=NULL;                                   /* only one block */
#if defined(USE_REGINA)
 shv.shvcode=RXSHV_SYFET; /* for Regina (no direct set) use symbolic set */
#else
 shv.shvcode=RXSHV_FETCH;             /* ... for others use direct set */
#endif
/*---------------------------------------------------------------------*/
/* This calls the RexxVariablePool() function for each value. This is  */
/* not the most efficient way of doing this.                           */
/*---------------------------------------------------------------------*/
 sprintf(variable_name,"%s.%-d",name,suffix);
 (void)make_upper(variable_name);/* make variable name uppercase */
/*---------------------------------------------------------------------*/
/* Now (attempt to) get the REXX variable                              */
/* Set shv.shvvalue to NULL to force interpreter to allocate memory.   */
/*---------------------------------------------------------------------*/
 assert(variable_name);
 MAKERXSTRING(shv.shvname, variable_name, strlen(variable_name));
 assert(shv.shvname.strptr);
 shv.shvvalue.strptr = NULL;
 shv.shvvalue.strlength = 0;
/*---------------------------------------------------------------------*/
/* One or both of these is needed, too <sigh>                          */
/*---------------------------------------------------------------------*/
 shv.shvnamelen=strlen(variable_name);
 shv.shvvaluelen=0;
#if defined(USE_OS2REXX) || defined(USE_REXX6000)
 rc=(short)RexxVariablePool(&shv);              /* Set the REXX variable */
#else
 rc = RexxVariablePool(&shv);                 /* Set the REXX variable */
/* rc = RXSHV_OK;*/
#endif
 if (rc == RXSHV_OK)
   {
    assert(value);
#ifdef USE_REXX6000
    value->strptr = (PUCHAR)malloc((sizeof(char)*shv.shvvalue.strlength)+1);
#else
    value->strptr = (char *)malloc((sizeof(char)*shv.shvvalue.strlength)+1);
#endif
    if (value->strptr != NULL)
      {
       value->strlength = shv.shvvalue.strlength;
       memcpy(value->strptr,shv.shvvalue.strptr,value->strlength);
       *(value->strptr+value->strlength) = '\0';
      }
#if defined(WIN32) && (defined(USE_OREXX) || defined(USE_WINREXX) || defined(USE_QUERCUS))
    GlobalFree( shv.shvvalue.strptr );
#elif defined(USE_OS2REXX)
    DosFreeMem( shv.shvvalue.strptr );
#else
    free( shv.shvvalue.strptr );
#endif
   }
 else
   value = NULL;
 return(value);
}

/*-----------------------------------------------------------------------------
 * This function converts a REXX array into an array of argv's
 *----------------------------------------------------------------------------*/
int ArrayToArgv

#ifdef HAVE_PROTO
    (RXSTRING *names, RXSTRING *values, ULONG *Argc, RXSTRING *Argv)
#else
    (names, values, Argc, Argv)
    RXSTRING *names;
    RXSTRING *values;
    ULONG *Argc;
    RXSTRING *Argv;
#endif

{
 int rc=0,i=0,num_args=1;
 RXSTRING counter;
 char *new_name=NULL;
 char *new_value=NULL;
 int num_names=0,num_values=0;

 InternalFunctionPrologue("ArrayToArgv");

 if ((new_name = (char *)malloc(RXSTRLEN(*names)+1)) == NULL)
    return(SetIntError(10, "out of memory", __FILE__, __LINE__));
 memcpy(new_name,RXSTRPTR(*names),RXSTRLEN(*names));
/*
 if ((counter.strptr = (char *)malloc(50+1)) == NULL)
    return(SetIntError(10, "out of memory", __FILE__, __LINE__));
 counter.strlength = 50;
*/
 new_name[RXSTRLEN(*names)-1] = '\0';
 if ((GetRexxVariable(new_name,&counter,0)) == NULL)
   {
    free(new_name);
    free(counter.strptr);
    return(SetIntError(10, "rexx variable not found", __FILE__, __LINE__));
   }
 num_names = atoi(RXSTRPTR(counter));

 if (values != NULL)
   {
    num_args = 2;
    if ((new_value = (char *)malloc(RXSTRLEN(*values)+1)) == NULL)
      {
       free(new_name);
       free(counter.strptr);
       return(SetIntError(10, "out of memory", __FILE__, __LINE__));
      }
    memcpy(new_value,RXSTRPTR(*values),RXSTRLEN(*values));
    new_value[RXSTRLEN(*values)-1] = '\0';
    if ((GetRexxVariable(new_value,&counter,0)) == NULL)
      {
       free(new_name);
       free(counter.strptr);
       return(SetIntError(10, "out of memory", __FILE__, __LINE__));
      }
    num_values = atoi(RXSTRPTR(counter));
    if (num_names != num_values)
      {
       free(new_name);
       free(counter.strptr);
       return(SetIntError(88, "array index counts not equal", __FILE__, __LINE__));
      }
   }
 free(counter.strptr);

 for (i=1;i<=num_names;i++)
   {
    if ((GetRexxVariable(new_name,Argv,i)) == NULL)
      {

       free(new_name);
       return(SetIntError(10, "out of memory", __FILE__, __LINE__));
      }
    Argv++;
    if (values != NULL)
      {
       if ((GetRexxVariable(new_value,Argv,i)) == NULL)
         {
          free(new_name);
          return(SetIntError(10, "out of memory", __FILE__, __LINE__));
         }
       Argv++;
      }
   }
 free(new_name);
 free(new_value);
 *Argc = num_names*num_args;
 return 0;
}

/*-----------------------------------------------------------------------------
 * This function copies an array of arvs to another
 *----------------------------------------------------------------------------*/
int ArgvToArgv

#ifdef HAVE_PROTO
    (ULONG argc, RXSTRING *argv, ULONG *Argc, RXSTRING *Argv)
#else
    (argc, argv, Argc, Argv)
    ULONG argc;
    RXSTRING *argv;
    ULONG *Argc;
    RXSTRING *Argv;
#endif

{
 ULONG i=0L;

 InternalFunctionPrologue("ArgvToArgv");

 for (i=0;i<argc;i++)
   {
    if ((Argv[i].strptr = (char *)malloc(RXSTRLEN(argv[i])+1)) == NULL)
       return(SetIntError(10, "out of memory", __FILE__, __LINE__));
    memcpy(RXSTRPTR(Argv[i]),RXSTRPTR(argv[i]),RXSTRLEN(argv[i]));
    Argv[i].strptr[RXSTRLEN(argv[i])] = '\0';
    Argv[i].strlength = RXSTRLEN(argv[i]);
   }
 *Argc = argc;
 return 0;
}

/*-----------------------------------------------------------------------------
 * This function frees an array of argvs
 *----------------------------------------------------------------------------*/
RXSTRING *FreeArgv

#ifdef HAVE_PROTO
    (ULONG *Argc, RXSTRING *Argv)
#else
    (Argc,Argv)
    ULONG *Argc;
    RXSTRING *Argv;
#endif

{
 int i=0;

 InternalFunctionPrologue("FreeArgv");

 if (*Argc == 0
 || Argv == NULL)
    return(NULL);

 for (i=0;i<MAX_BINDVARS*2;i++)
   {
    if (RXSTRPTR(Argv[i]) != NULL)
       free(Argv[i].strptr);
    Argv[i].strptr = NULL;
    Argv[i].strlength = 0;
   }
 *Argc = 0;
 free(Argv);
 return(NULL);
}


/*-----------------------------------------------------------------------------
 * This function is called to initiate REXX/SQL interface.
 *----------------------------------------------------------------------------*/
int InitRexxSQL

#ifdef HAVE_PROTO
    (RXSQL_PSZ progname,int regfunc)
#else
    (progname,regfunc)
    RXSQL_PSZ progname;
    int regfunc;
#endif

{
 RexxFunction  *func=NULL;
 ULONG rc=0L;

 InternalFunctionPrologue("InitRexxSQL");

 /* 
  * Install REXX/SQL variable descriptors 
  */
 if ((rc = InstallSQLVariables()))
     return(rc);
 
 /*
  * Call any database-specific initialisation function here
  */
 if ((rc = DBInitialise()))
    return(rc);

 RexxSQLInitialised = MAGIC_NUMBER;
 /* 
  * Register all REXX/SQL functions 
  *
  */
 if (regfunc)
   {
    for (func = RexxSqlFunctions; func->internal_name; func++)
      {
       assert(func->external_name);
       assert(func->internal_name);
       assert(func->EntryPoint);
#if defined(DYNAMIC_LIBRARY)
# if !defined(USE_REXX6000)
       rc = RexxRegisterFunctionDll(func->external_name,DLLNAME,func->internal_name);
# endif
#else
# if defined(WIN32) && (defined(USE_WINREXX) || defined(USE_QUERCUS))
       RexxDeregisterFunction(func->external_name);
# endif
       rc = RexxRegisterFunctionExe(func->external_name,	func->EntryPoint);
#endif
       if (rc != RXFUNC_OK
       &&  rc != RXFUNC_DEFINED
       &&  rc != RXFUNC_NOTREG)
          return 1;
      }
#if !defined(DYNAMIC_LIBRARY) && (defined(USE_WINREXX) || defined(USE_QUERCUS))
    if ( RexxRegisterExitExe(DLLNAME,
                             (FARPROC)RxExitHandlerForSayTraceRedirection,
                             NULL) != RXEXIT_OK )
       return 1;
#endif
   }

 return 0;
}


/*-----------------------------------------------------------------------------
 * This function is called to terminate all activity with REXX/SQL.
 *----------------------------------------------------------------------------*/
int TerminateRexxSQL

#ifdef HAVE_PROTO
    (RXSQL_PSZ progname)
#else
    (progname)
    RXSQL_PSZ progname;
#endif

{
 int rc=0;
 RexxFunction  *func=NULL;

 FunctionPrologue(progname, 0L, NULL);

 /* Release the REXX/SQL environment. */
 if ((rc = ReleaseDbEnvironment()))
    return rc;

 /* 
  * De-register all REXX/SQL functions only 
  * if DEBUG value = 99
  * DO NOT DO THIS FOR DYNAMIC LIBRARY
  * AS IT WILL DEREGISTER FOR ALL PROCESSES
  * NOT JUST THE CURRENT ONE.               
  */
 if (run_flags == 99)
   {
    for (func = RexxSqlFunctions; func->internal_name; func++)
      {
       assert(func->external_name);
       (void)RexxDeregisterFunction(func->external_name);
      }
   }
#if !defined(DYNAMIC_LIBRARY) && (defined(USE_WINREXX) || defined(USE_QUERCUS))
   RexxDeregisterExit( DLLNAME, NULL );
#endif

 opt_tbl = ll_free(opt_tbl);
 return 0;
}

/*-----------------------------------------------------------------------------
 * Add a SQLVAL structure to a linked list.
 *----------------------------------------------------------------------------*/
SQLVAL *ll_add

#ifdef HAVE_PROTO
    (SQLVAL *first,SQLVAL *curr)
#else
    (first,curr)
    SQLVAL *first;
    SQLVAL *curr;
#endif
{
 SQLVAL *next=NULL;
 if ((next=(SQLVAL *)malloc(sizeof(SQLVAL))) != (SQLVAL *)NULL)
   {
    if (curr == NULL)
      {
       first = next;
       next->next = NULL;
      }
    else
      {
       if (curr->next != NULL)
          curr->next->prev = next;
       next->next = curr->next;
       curr->next = next;
      }
    next->prev = curr;
   }
 next->value = NULL;
 return(next);
}

/*-----------------------------------------------------------------------------
 * Free a SQLVAL linked list.
 *----------------------------------------------------------------------------*/
SQLVAL *ll_free

#ifdef HAVE_PROTO
    (SQLVAL *first)
#else
    (first)
    SQLVAL *first;
#endif
{
 SQLVAL *curr=first;
 SQLVAL *new_curr=NULL;
 RXSTRING *rxs=NULL;
 RXSTRING_DT *rxs_dt=NULL;
 int i=0;

 while (curr != NULL)
   {
    if ((curr->option)&OPT_DYNAMIC)
      {
       switch(curr->dtype)
         {
          case TYPE_INT:
          case TYPE_BOOL:
               if (curr->value)
                  free(curr->value);
               break;
          case TYPE_STRING:
               rxs =(RXSTRING*)(curr->value);
               if (rxs)
                 {
                  if (rxs->strptr) free(rxs->strptr);
                  free(rxs);
                 }
               break;
          case TYPE_ARRAY:
               rxs =(RXSTRING*)(curr->value);
               if (rxs)
                 {
                  for (i=0;;i++)
                    {
                     if (rxs[i].strlength == 0)
                        break;
                     if (rxs[i].strptr) free(rxs[i].strptr);
                    }
                  free(rxs);
                 }
               break;
          case TYPE_DATATYPE:
               rxs_dt =(RXSTRING_DT*)(curr->value);
               if (rxs_dt)
                 {
                  for (i=0;;i++)
                     {
                      if (rxs_dt[i].strlength == 0)
                         break;
                      if (rxs_dt[i].strptr) free(rxs_dt[i].strptr);
                     }
                 }
               free(rxs_dt);
               break;
         }
      }
    new_curr = curr->next;
    free(curr);
    curr = new_curr;
   }
 return((SQLVAL *)NULL);
}

/*-----------------------------------------------------------------------------
 * Find an entry in a SQLVAL linked list.
 *----------------------------------------------------------------------------*/
SQLVAL *ll_find

#ifdef HAVE_PROTO
    (SQLVAL *first, char *name)
#else
    (first,name)
    SQLVAL *first;
    char *name;
#endif
{
 SQLVAL *curr=NULL;
 curr = first;
 while (curr != NULL)
   {
    if (strcmp(curr->name,name) == 0)
      {
       return(curr);
      }
    curr = curr->next;
   }
 return(NULL);
}

/*-----------------------------------------------------------------------------
 * This function is used to initialise the external functions from
 * within a dynamic library when the SysAddFunc() REXX function
 * is called.
 *----------------------------------------------------------------------------*/
#if defined(USE_REXX6000)
USHORT InitFunc(RXFUNCBLOCK **FuncBlock)
{
 static RXFUNCBLOCK funcarray[] =
  {
  { NAME_SQLCONNECT_EXT,    SQLCONNECT   ,NULL },
  { NAME_SQLDISCONNECT_EXT, SQLDISCONNECT,NULL },
  { NAME_SQLDEFAULT_EXT,    SQLDEFAULT   ,NULL },
  { NAME_SQLCOMMIT_EXT,     SQLCOMMIT    ,NULL },
  { NAME_SQLROLLBACK_EXT,   SQLROLLBACK  ,NULL },
  { NAME_SQLCOMMAND_EXT,    SQLCOMMAND   ,NULL },
  { NAME_SQLPREPARE_EXT,    SQLPREPARE   ,NULL },
  { NAME_SQLDISPOSE_EXT,    SQLDISPOSE   ,NULL },
  { NAME_SQLEXECUTE_EXT,    SQLEXECUTE   ,NULL },
  { NAME_SQLOPEN_EXT,       SQLOPEN      ,NULL },
  { NAME_SQLCLOSE_EXT,      SQLCLOSE     ,NULL },
  { NAME_SQLFETCH_EXT,      SQLFETCH     ,NULL },
  { NAME_SQLGETDATA_EXT     SQLGETDATA   ,NULL },
  { NAME_SQLVARIABLE_EXT,   SQLVARIABLE  ,NULL },
  { NAME_SQLGETINFO_EXT,    SQLGETINFO   ,NULL },
  { NAME_SQLDESCRIBE_EXT,   SQLDESCRIBE  ,NULL },
  { NAME_SQLDROPFUNCS_EXT,  SQLDROPFUNCS ,NULL },
  { NULL,                   NULL,         NULL }
  };
 *FuncBlock = funcarray;
 return (USHORT)0;
}
#endif

#if !defined(DYNAMIC_LIBRARY) && (defined(USE_WINREXX) || defined(USE_QUERCUS))
# if defined(USE_WINREXX)
int RXSQL_APIENTRY RxExitHandlerForSayTraceRedirection
   ( int ExitNumber, int Subfunction, PEXIT ParmBlock )
# else
LONG RxExitHandlerForSayTraceRedirection
   ( LONG ExitNumber, LONG Subfunction, PEXIT ParmBlock )
# endif
{
   long i = 0;
   int rc = 0;

   switch( Subfunction )
   {
      case RXSIOSAY:
      {
         RXSIOSAY_PARM *say_parm = (RXSIOSAY_PARM *)ParmBlock;
         if ( say_parm->rxsio_string.strptr != NULL )
         {
            for( i = 0; i < (long)say_parm->rxsio_string.strlength; i++ )
               fputc( ( char )say_parm->rxsio_string.strptr[i], stdout );
         }
         fputc( '\n', stdout );
         rc = RXEXIT_HANDLED;
         break;
      }
      case RXSIOTRC:
      {
         RXSIOTRC_PARM *trc_parm = (RXSIOTRC_PARM *)ParmBlock;
         if ( trc_parm->rxsio_string.strptr != NULL )
         {
            for( i = 0; i < (long)trc_parm->rxsio_string.strlength; i++ )
               fputc( ( char )trc_parm->rxsio_string.strptr[i], stdout );
         }
         fputc( '\n', stdout );
         rc = RXEXIT_HANDLED;
         break;
      }
      case RXSIOTRD:
      {
         RXSIOTRD_PARM *trd_parm = (RXSIOTRD_PARM *)ParmBlock;
         int i = 0, ch = 0;
         do
         {
            if ( i < 256 )
               trd_parm->rxsiotrd_retc.strptr[i++] = ch = getc( stdin ) ;
         } while( ch != '\012' && (ch != EOF ) ) ;
         trd_parm->rxsiotrd_retc.strlength = i - 1;
         rc = RXEXIT_HANDLED;
         break;
      }
      case RXSIODTR:
      {
         RXSIODTR_PARM *dtr_parm = (RXSIODTR_PARM *)ParmBlock;
         int i = 0, ch = 0;
         do
         {
            if ( i < 256 )
               dtr_parm->rxsiodtr_retc.strptr[i++] = ch = getc( stdin ) ;
         } while( ch != '\012' && (ch != EOF ) ) ;
         dtr_parm->rxsiodtr_retc.strlength = i - 1;
         rc = RXEXIT_HANDLED;
         break;
      }
         break;
      default:
         rc = RXEXIT_NOT_HANDLED;
         break;
   }
  return rc;
}
#endif

/*
 * Because WinRexx and Personal Rexx load and unload the Rexx/SQL DLL
 * each time an external function is called, we need to ensure that
 * the DLL stays loaded in memory, so that memory we have initialised
 * stays set.
 * Why they do this is beyond me :-(
 */
#if defined(DYNAMIC_LIBRARY) && (defined(USE_WINREXX) || defined(USE_QUERCUS))
BOOL WINAPI DllMain( HINSTANCE hDLL, DWORD dwReason, LPVOID pReserved)
{
   switch( dwReason)
   {
      case DLL_PROCESS_ATTACH:
         LoadLibrary( DLLNAME );
         break;
      case DLL_PROCESS_DETACH:
         FreeLibrary( GetModuleHandle( DLLNAME ) );
         break;
      case DLL_THREAD_ATTACH:
         break;
      case DLL_THREAD_DETACH:
         break;
   }
   return(TRUE);
}
#endif
