/* RXRNX.C - Copyright (C) Bruce Eric Hogman 1995.  All Rights Reserved.

   Entry points:
      RXSESSIONFOCUS    REXX seize focus
      RXSESSIONRUN      REXX create session
      RXSESSIONHIDE     REXX hide/show   1/0 session window
      RXSESSIONMAX      REXX max/restore 1/0 session window
      RXSESSIONMIN      REXX min/restore 1/0 session window
      RXSESSIONTITLE    REXX set window list title for window session
      RXSESSIONVISIBLE  REXX set visible/invisible 1/0

      rxrnxgo           C function for RXSESSIONRUN
      rxhide            C
      rxmax             C
      rxmin             C
      rxtitle           C
      rxvisible         C

      rxswhnd           C returns HSWITCH, SWCNTRL

   arguments passed to rxrnxgo:
   PSZ pgmname,PSZ title,PSZ parmstring,USHORT sesstype, PSZ sessctl

   Return codes:
   values are those returned from invoked programs.
   Value of ULONG ~0 is returned upon any error in call parameters.

 --------------------------------------------------------------------
   V2.0 changes:
   1.  Added code to switch focus to child session upon its creation.
   2.  Added code to switch focus to self after child termination.
 --------------------------------------------------------------------
   V2.1 changes:
   1.  Added rxmax to maximize session, changed behavior of rxmin to
       restore.
 --------------------------------------------------------------------

 STARTDATA variable values of importance:
 =======================================

 sesstype (USHORT)
 --------------------
    The type of session that should be created for this program.

    The values of this field are shown in the list below:

 RXSESSIONRUN   .H Header definition
 -----          --------------------
          0     SSF_TYPE_DEFAULT
 OS2FS    1     SSF_TYPE_FULLSCREEN
 OS2      2     SSF_TYPE_WINDOWABLEVIO
 PM       3     SSF_TYPE_PM
 DOSFS    4     SSF_TYPE_VDM
 DOS      7     SSF_TYPE_WINDOWEDVDM


 PgmControl (USHORT)
 -------------------
    An indicator which specifies the initial state for a windowed
    application.  This field is ignored for full-screen sessions.
    The bits in this field have the following values:

       Bit            Description
       ---            -----------
       15             SSF_CONTROL_SETPOS (0x8000)
                      Use specified size and position
       4-14           Reserved
       3              SSF_CONTROL_NOAUTOCLOSE (0x0008)
                      No Auto Close
 MIN   2              SSF_CONTROL_MINIMIZE (0x0004)
 MAX                  Minimize
       1              SSF_CONTROL_MAXIMIZE (0x0002)
                      Maximize
       0              SSF_CONTROL_INVISIBLE (0x0001)
                      Invisible
       0              SSF_CONTROL_VISIBLE (0x0000)
                      Visible

    Note:  The "No Auto Close" bit is used only for VIO Windowable
           applications, and is ignored for all other types of
           applications.
*/
#define INCL_REXXSAA
#define INCL_DOSSESMGR
#define INCL_DOSERRORS
#define INCL_DOSPROCESS
#define INCL_DOSQUEUES
#define INCL_DOSDATETIME
#define INCL_WIN
#define INCL_PM
#include <memory.h>
#include <rexxsaa.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <os2.h>
#define INVALID_ROUTINE 1
#define   VALID_ROUTINE 0
#ifdef __WATCOMC__
#define _ultoa utoa
#pragma aux RXSESSIONFOCUS "RXSESSIONFOCUS"
#pragma aux RXSESSIONHIDE  "RXSESSIONHIDE"
#pragma aux RXSESSIONMAX   "RXSESSIONMAX"
#pragma aux RXSESSIONMIN   "RXSESSIONMIN"
#pragma aux RXSESSIONRUN   "RXSESSIONRUN"
#pragma aux RXSESSIONTITLE "RXSESSIONTITLE"
#pragma aux RXSESSIONVISIBLE "RXSESSIONVISIBLE"
#pragma aux rxswhnd        "rxswhnd"
#pragma aux rxfocus        "rxfocus"
#pragma aux rxhide         "rxhide"
#pragma aux rxmax          "rxmax"
#pragma aux rxmin          "rxmin"
#pragma aux rxrnxgo        "rxrnxgo"
#pragma aux rxtitle        "rxtitle"
#pragma aux rxvisible      "rxvisible"
#endif

/* prototypes */
ULONG   rxfocus(PSZ pszch, PSZ pszch2);
ULONG   rxhide(ULONG ulfl);
ULONG   rxmax( ULONG ulfl);
ULONG   rxmin( ULONG ulfl);
ULONG   rxqtitle(PSZ pszch);
ULONG   rxrnxgo (
  PULONG pgmrc,       /* program return code      */
  PSZ pgmname,        /* target filespec          */
  PSZ wintitle,       /* switch list title string */
  PSZ parmstring,     /* parameters for program   */
  USHORT sesstype,    /* code of session type     */
  USHORT sessctl);    /* code of MAX/MIN window   */
HSWITCH rxswhnd(PSWCNTRL);
ULONG   rxtitle(PSZ wndtitle);
ULONG   rxvisible(ULONG ulvisible); /* 1=visible, 0=not    */

/*
Ŀ
 RXRLOADFUNCS RexxFunctionHandler            

*/
RexxFunctionHandler RXRLOADFUNCS;
ULONG RXRLOADFUNCS(
  PUCHAR    Name,
  ULONG     Argc,
  PRXSTRING Argv,
  PSZ       Queuename,
  PRXSTRING Retstr)
{
  PSZ   RxFncTable[] =
  { "RXSESSIONFOCUS",
    "RXSESSIONHIDE",
    "RXSESSIONMAX",
    "RXSESSIONMIN",
    "RXSESSIONQTITLE",
    "RXSESSIONRUN",
    "RXSESSIONTITLE",
    "RXSESSIONVISIBLE"
  };
  PSZ   RxFncDLL = "RXRNX";
  int entries = sizeof(RxFncTable)/sizeof(PSZ);
  MAKERXSTRING(*Retstr,"0",1);
  if (Argc > 0) return INVALID_ROUTINE;
  { int j;
    for (j=0; j<entries; j++)
      RexxRegisterFunctionDll(RxFncTable[j], RxFncDLL, RxFncTable[j]);
  }
  return VALID_ROUTINE;
}
/*
Ŀ
 RXSESSIONFOCUS  RexxFunctionHandler returning ULONG 

*/
RexxFunctionHandler RXSESSIONFOCUS;
ULONG RXSESSIONFOCUS(
  PUCHAR    Name,
  ULONG     Argc,
  PRXSTRING Argv,
  PSZ       Queuename,
  PRXSTRING Retstr)
{ if (Argc < 1 || Argc > 2) return INVALID_ROUTINE;
  { PSZ pszArg2 = Argc > 1 && (Argv[1].strptr[0] == '-' ||
                               Argv[1].strptr[0] == '/')&&
                              (Argv[1].strptr[1] == 'r' ||
                               Argv[1].strptr[1] == 'R') ?
                               Argv[1].strptr : "  ";
    _ultoa(rxfocus(Argv[0].strptr,pszArg2),Retstr->strptr,10);
    Retstr->strlength = strlen(Retstr->strptr);
  }
  return 0;
}
/*
Ŀ
 RXSESSIONHIDE  RexxFunctionHandler returning ULONG 

*/
RexxFunctionHandler RXSESSIONHIDE;
ULONG RXSESSIONHIDE(
  PUCHAR    Name,
  ULONG     Argc,
  PRXSTRING Argv,
  PSZ       Queuename,
  PRXSTRING Retstr)
{ if (Argc != 1) return INVALID_ROUTINE;
  _ultoa(rxhide(Argv[0].strptr[0] == '1' ? 1 : 0), Retstr->strptr, 10);
  Retstr->strlength = strlen(Retstr->strptr);
  return 0;
}
/*
Ŀ
 RXSESSIONMAX   RexxFunctionHandler returning ULONG 

*/
RexxFunctionHandler RXSESSIONMAX;
ULONG RXSESSIONMAX(
  PUCHAR    Name,
  ULONG     Argc,
  PRXSTRING Argv,
  PSZ       Queuename,
  PRXSTRING Retstr)
{ if (Argc != 1) return INVALID_ROUTINE;
  _ultoa(rxmax(Argv[0].strptr[0] == '1' ? 1 : 0), Retstr->strptr, 10);
  Retstr->strlength = strlen(Retstr->strptr);
  return 0;
}
/*
Ŀ
 RXSESSIONMIN   RexxFunctionHandler returning ULONG 

*/
RexxFunctionHandler RXSESSIONMIN;
ULONG RXSESSIONMIN(
  PUCHAR    Name,
  ULONG     Argc,
  PRXSTRING Argv,
  PSZ       Queuename,
  PRXSTRING Retstr)
{ if (Argc != 1) return INVALID_ROUTINE;
  _ultoa(rxmin(Argv[0].strptr[0] == '1' ? 1 : 0), Retstr->strptr, 10);
  Retstr->strlength = strlen(Retstr->strptr);
  return 0;
}
/*
Ŀ
 RXSESSIONRUN entry as RexxFunctionHandler returning ULONG 

*/
RexxFunctionHandler RXSESSIONRUN;
ULONG RXSESSIONRUN(
  PUCHAR    Name,            /* name of the function     */
  ULONG     Argc,            /* number of arguments      */
  PRXSTRING Argv,          /* list of argument strings */
  PSZ       Queuename,       /* current queue name       */
  PRXSTRING Retstr)          /* returned result string   */
{ ULONG ulrc                    = 0;
  UCHAR pgmname[CCHMAXPATHCOMP] = "";
  UCHAR wintitle[100]           = "";
  UCHAR parmstr[1024]           = "";
  USHORT sesstype               = SSF_TYPE_DEFAULT;
  USHORT sessctl                = SSF_CONTROL_VISIBLE;
  /* based on number of arguments, process and fall thru */
  switch (Argc)
  { case 5: /* session control */
    { if (RXVALIDSTRING(Argv[4]) && RXSTRLEN(Argv[4]) <= 3)
      { UCHAR c1 = Argv[4].strptr[2];
        /* MIN, MAX, or DEFAULT */
        switch (c1)
        { case 'N': sessctl = SSF_CONTROL_MINIMIZE; break;
          case 'X': sessctl = SSF_CONTROL_MAXIMIZE; break;
          default:  sessctl = SSF_CONTROL_VISIBLE;
    } } }
    case 4: /* session type */
    { if (RXVALIDSTRING(Argv[3]) && RXSTRLEN(Argv[3]) <= 5)
      { UCHAR c1 = Argv[3].strptr[0];
        /* OS2xx,DOSxx */
        switch (c1)
        { case 'O': /* OS2FS,OS2 */
          { UCHAR c1 = Argv[3].strptr[3];
            switch (c1)
            { case 'F': sesstype = SSF_TYPE_FULLSCREEN; break;
              default:  sesstype = SSF_TYPE_WINDOWABLEVIO;
            }
            break;
          }
          case 'D': /* DOSFS,DOS */
          { UCHAR c1 = Argv[3].strptr[3];
            switch (c1)
            { case 'F': sesstype = SSF_TYPE_VDM; break;
              default:  sesstype = SSF_TYPE_WINDOWEDVDM;
            }
            break;
          }
          case 'P': sesstype = SSF_TYPE_PM; break;
          default:  sesstype = SSF_TYPE_DEFAULT;
    } } }
    case 3: /* pass on parm string */
    { unsigned int len = RXSTRLEN(Argv[2]);
      if (RXVALIDSTRING(Argv[2]) && len <= sizeof(parmstr))
      { memmove(parmstr,RXSTRPTR(Argv[2]),len);
        parmstr[len] = '\0';
      }
    }
    case 2: /* pass on title of window session */
    { int len = RXSTRLEN(Argv[1]);
      if (RXVALIDSTRING(Argv[1]) && len <= sizeof(wintitle))
      { memmove(wintitle,RXSTRPTR(Argv[1]),len);
        wintitle[len] = '\0';
      }
    }
    case 1: /* pass on program name */
    { int len = RXSTRLEN(Argv[0]);
      if (RXVALIDSTRING(Argv[0]) && len <= sizeof(pgmname))
      { memmove(pgmname,RXSTRPTR(Argv[0]),len);
        pgmname[len] = '\0';
      }
      break;
    }
    default: return INVALID_ROUTINE;
  }
  { ULONG pgmrc;
    ulrc = rxrnxgo(&pgmrc,pgmname,wintitle,parmstr,sesstype,sessctl);
    { UCHAR result[32] = "";
      int j = sprintf(result,"%lu",pgmrc);
      MAKERXSTRING(*Retstr,result,j);
    }
    if (ulrc != NO_ERROR)
       fprintf(stderr,"RXSESSIONRUN rxrnxgo returned error RC=%lu\n",ulrc);
    return ulrc;
  }
}
/*
Ŀ
 RXSESSIONQTITLE RexxFunctionHandler returning ULONG 

*/
RexxFunctionHandler RXSESSIONQTITLE;
ULONG RXSESSIONQTITLE(
  PUCHAR    Name,
  ULONG     Argc,
  PRXSTRING Argv,
  PSZ       Queuename,
  PRXSTRING Retstr)
{ ULONG ultitlelen = MAXNAMEL;
  UCHAR szwintitle[MAXNAMEL+1] = "";
  if (Argc != 0) return INVALID_ROUTINE;
  rxqtitle(szwintitle);
  strcpy(Retstr->strptr,szwintitle);
  Retstr->strlength = strlen(Retstr->strptr);
  return 0;
}
/*
Ŀ
 RXSESSIONTITLE RexxFunctionHandler returning ULONG 

*/
RexxFunctionHandler RXSESSIONTITLE;
ULONG RXSESSIONTITLE(
  PUCHAR    Name,
  ULONG     Argc,
  PRXSTRING Argv,
  PSZ       Queuename,
  PRXSTRING Retstr)
{ ULONG ultitlelen = MAXNAMEL;
  UCHAR szwintitle[MAXNAMEL+1] = "";
  if (Argc != 1) return INVALID_ROUTINE;
  ultitlelen=strlen(Argv[0].strptr)>ultitlelen ?
     ultitlelen : strlen(Argv[0].strptr);
  memmove(szwintitle,Argv[0].strptr,ultitlelen);
  szwintitle[ultitlelen]='\0';
  _ultoa(rxtitle(szwintitle),Retstr->strptr,10);
  Retstr->strlength = strlen(Retstr->strptr);
  return 0;
}
/*
Ŀ
 RXSESSIONVISIBLE RexxFunctionHandler returning ULONG 

*/
RexxFunctionHandler RXSESSIONVISIBLE;
ULONG RXSESSIONVISIBLE(
  PUCHAR    Name,
  ULONG     Argc,
  PRXSTRING Argv,
  PSZ       Queuename,
  PRXSTRING Retstr)
{ if (Argc != 1) return INVALID_ROUTINE;
  _ultoa(rxvisible(Argv[0].strptr[0] == '0'? 0 : 1),Retstr->strptr,10);
  Retstr->strlength = strlen(Retstr->strptr);
  return 0;
}


/*
Ŀ
 rxrnxgo entry as function returning ULONG 

*/
ULONG rxrnxgo (
  PULONG pgmrc,       /* program return code      */
  PSZ pgmname,        /* target filespec          */
  PSZ wintitle,       /* switch list title string */
  PSZ parmstring,     /* parameters for program   */
  USHORT sesstype,    /* code of session type     */
  USHORT sessctl)     /* code of MAX/MIN window   */
{ DATETIME mydt;
  /* make a queue name using day and time of day */
  DosGetDateTime(&mydt);
  { UCHAR qname[64]="\\queues\\";
    sprintf(qname+8,"%01u",mydt.day);
    sprintf(qname+9,"%02u",mydt.hours);
    sprintf(qname+11,"%02u",mydt.minutes);
    sprintf(qname+13,"%02u.que",mydt.seconds);
    /* make a new queue for use as a termination queue */
    { HQUEUE htermq;
      APIRET rcqueue = DosCreateQueue(&htermq,
         QUE_FIFO | QUE_CONVERT_ADDRESS, qname);
      if (rcqueue != NO_ERROR)
      { fprintf(stderr,"rxrnxgo DosCreateQueue error, RC=%lu\n",rcqueue);
        return rcqueue;
      }
      { UCHAR objbuff[CCHMAXPATH]="";
        /* define and initialize SData */
#ifndef __WATCOMC__
          STARTDATA SData = {
          sizeof(STARTDATA), /* Length     */
          SSF_RELATED_CHILD, /* Related    */
          0, 0,        /* FgBg, TraceOpt   */
          wintitle,          /* PgmTitle   */
          pgmname,           /* PgmName    */
          parmstring,        /* PgmInputs  */
          qname,             /* TermQ      */
          0,                 /* Environment*/
          SSF_INHERTOPT_PARENT,
          sesstype,
          0, 0,    /* IconFile, PgmHandle  */
          sessctl,           /* PgmControl */
          0,0,0,0,0,         /* win coord  */
          objbuff,           /* buffer     */
          sizeof(objbuff)};  /* buffer len */
#else
          STARTDATA SData = {
          sizeof(STARTDATA), /* Length     */
          SSF_RELATED_CHILD, /* Related    */
          0, 0,        /* FgBg, TraceOpt   */
          NULL,              /* PgmTitle   */
          NULL,              /* PgmName    */
          NULL,              /* PgmInputs  */
          NULL,              /* TermQ      */
          0,                 /* Environment*/
          SSF_INHERTOPT_PARENT,
          0,
          0, 0,    /* IconFile, PgmHandle  */
          0,                 /* PgmControl */
          0,0,0,0,0,         /* win coord  */
          NULL,              /* buffer     */
          sizeof(objbuff)};  /* buffer len */
          SData.PgmTitle = wintitle;
          SData.PgmName = pgmname;
          SData.PgmInputs = parmstring;
          SData.SessionType = sesstype;
          SData.PgmControl = sessctl;
          SData.ObjectBuffer = objbuff;
          SData.TermQ = qname;
#endif
        /* call DosStartSession */
        { APIRET ulrc;
          PID    pid      = 0;
          ULONG  ulSessID = 0;
          ulrc = DosStartSession(&SData, &ulSessID, &pid);
          /* note rc 457 OK */
          if (ulrc != NO_ERROR && ulrc != 457)
          { fprintf(stderr,"rxrnxgo DosStartSession error, RC=%lu\n",ulrc);
            return ulrc;
          }
          { STATUSDATA sdata = {
              sizeof(STATUSDATA),
              SET_SESSION_UNCHANGED,
              SET_SESSION_BOND };
            /* set bond between parent & child */
            ulrc = DosSetSession(ulSessID,&sdata);
            if (ulrc != NO_ERROR)
              fprintf(stderr,"rxrnxgo DosSetSession error, RC=%lu\n",ulrc);
            /* set our own session not visible */
            /* call DosReadQueue & wait */
            { REQUESTDATA  Request = {0,0};
              ULONG        buflen;
              BYTE         bPriority;
              PVOID        pbuf;
              ulrc = DosReadQueue(htermq,&Request,
                &buflen, (VOID **) &pbuf, 0,
                DCWW_WAIT, &bPriority, (HEV) NULL);
              if (ulrc != NO_ERROR)
              { fprintf(stderr,"rxrnxgo DosReadQueue error, RC=%lu\n",ulrc);
                *pgmrc = 255;
              }
              else *pgmrc = Request.ulData;
              ulrc = DosFreeMem((VOID *)pbuf);
              if (ulrc != NO_ERROR)
                fprintf(stderr,"rxrnxgo DosFreeMem error, RC=%lu\n",ulrc);
              ulrc = DosCloseQueue(htermq);
              if (ulrc != NO_ERROR)
                fprintf(stderr,"rxrnxgo DosCloseQueue error, RC=%lu\n",ulrc);
  } } } } } }
  return 0;
}
/*
Ŀ
 rxfocus entry as function returning ULONG   

  args:  titlemask [,/r]
*/
#include "mskchk.h"
#define FOCUS_NOT_FOUND 2
#define FOCUS_TASK_ACTIVE 1
#define FOCUS_CHANGED 0

ULONG rxfocus(PSZ szwtitle, PSZ szFlag)
{ if (szwtitle[0] == '*' && szwtitle[1] == '\0')
  { SWCNTRL swc;
    return WinSwitchToProgram(rxswhnd(&swc));
  }
  { BOOL bTestOnly = (szFlag[0] == '/' || szFlag[0] == '-') &&
                     (szFlag[1] == 'r' || szFlag[1] == 'R');
    ULONG    ulretcode;
    ULONG    ulNrItems  = WinQuerySwitchList(NULLHANDLE,NULL,0);
    ULONG    ulcbBufLen = (ulNrItems * sizeof(SWENTRY)) +sizeof(ULONG);
    PSWBLOCK pswblkBlk  = (SWBLOCK *)malloc(ulcbBufLen);
    HSWITCH  hswhandle  = NULLHANDLE;
    WinQuerySwitchList(NULLHANDLE, pswblkBlk, ulcbBufLen);
    { PSWENTRY pSWEItem = (SWENTRY *)&(pswblkBlk->aswentry[0]);
      ULONG i;            /* subscript */
      PSWCNTRL pSWCItem;  /* one SWCNTRL */
      UCHAR szUCTitle[MAXNAMEL+4];    /* upper case title */
      ULONG ulstrlen = strlen(szwtitle);
      for (i = 0; i < ulNrItems; i++)
      { pSWCItem = (SWCNTRL *) &((pSWEItem+i)->swctl);
        if (pSWCItem->fbJump & SWL_JUMPABLE)
        { strcpy(szUCTitle,pSWCItem->szSwtitle); strupr(szUCTitle);
          if (! mskchk(szwtitle, ulstrlen, szUCTitle,
                strlen(szUCTitle), "?", "*"))
          { hswhandle = (pSWEItem+i)->hswitch; /* found match */
            break;
    } } } }
    free(pswblkBlk);
    if (bTestOnly)
    { if (hswhandle == NULLHANDLE) ulretcode = FOCUS_NOT_FOUND;
      else ulretcode = FOCUS_TASK_ACTIVE;
    }
    else
    { if (hswhandle == NULLHANDLE) ulretcode = FOCUS_NOT_FOUND;
      else ulretcode = (WinSwitchToProgram(hswhandle) == 0 ?
         FOCUS_CHANGED : FOCUS_NOT_FOUND);
    }
    return ulretcode;
} }
/*
Ŀ
 rxhide  entry as function returning ULONG   

*/
ULONG rxhide(ULONG ulfl)
{ SWCNTRL swc;
  ULONG ulrc;
  HSWITCH hswhnd = rxswhnd(&swc);
  ulrc = WinSetWindowPos(swc.hwnd,0,0,0,0,0,
         ulfl==1 ? SWP_HIDE : SWP_SHOW);
  return ulrc;
}
/*
Ŀ
 rxmax   entry as function returning ULONG   

*/
ULONG rxmax(ULONG ulfl)
{ SWCNTRL swc;
  HSWITCH hsw = rxswhnd(&swc);
  ULONG ulval = ulfl > 0 ? 0 : 1;
  WinSetWindowPos(swc.hwnd,0,0,0,0,0,
     ulfl > 0? SWP_MAXIMIZE : SWP_RESTORE);
  WinPostMsg((HWND) swc.hwnd,
     ( USHORT ) WM_SYSCOMMAND,
     ( MPARAM ) (ulfl > 0 ? SC_MAXIMIZE : SC_RESTORE),
     ( MPARAM ) 0);
  return 0;
}
/*
Ŀ
 rxmin   entry as function returning ULONG   

*/
ULONG rxmin(ULONG ulfl)
{ SWCNTRL swc;
  HSWITCH hsw = rxswhnd(&swc);
  ULONG ulval = ulfl > 0 ? 0 : 1;
  WinSetWindowPos(swc.hwnd,0,0,0,0,0,
     ulfl > 0? SWP_MINIMIZE : SWP_RESTORE);
  WinPostMsg((HWND) swc.hwnd,
     ( USHORT ) WM_SYSCOMMAND,
     ( MPARAM ) (ulfl > 0 ? SC_MINIMIZE : SC_RESTORE),
     ( MPARAM ) 0);
  return 0;
}
/*
Ŀ
 rxswhnd entry as function returning HSWITCH 
   returns  SWCNTRL                          

*/
HSWITCH rxswhnd (PSWCNTRL pswc)
{ PTIB    ptib;
  PPIB    ppib;
  DosGetInfoBlocks(&ptib,&ppib);
  { HSWITCH hsw;
    if ((hsw = WinQuerySwitchHandle(NULLHANDLE,ppib->pib_ulpid))!=NULLHANDLE)
    { if (WinQuerySwitchEntry(hsw,pswc) == NO_ERROR) return hsw;
      else return NULLHANDLE;
    }
    else return NULLHANDLE;
  }
}
/*
Ŀ
 rxqtitle entry as function returning ULONG 

*/
ULONG rxqtitle (
  PSZ wndtitle)
{ SWCNTRL swc;
  HSWITCH hswhnd = rxswhnd(&swc);
  strcpy(wndtitle,swc.szSwtitle);
  return 0;
}
/*
Ŀ
 rxtitle entry as function returning ULONG 

*/
ULONG rxtitle (
  PSZ wndtitle)
{ UCHAR swtitle[MAXNAMEL+1];
  ULONG ultitlen = strlen(wndtitle) > sizeof(swtitle)-1 ?
                   sizeof(swtitle)-1 : strlen(wndtitle);
  memmove(swtitle,wndtitle,ultitlen); swtitle[ultitlen] = '\0';
  { SWCNTRL swc;
    HSWITCH hswhnd = rxswhnd(&swc);
    ULONG ulrc;
    if (hswhnd == NULLHANDLE) return 1;
    strcpy(swc.szSwtitle,swtitle);
    /*
    if (! WinSetWindowText(swc.hwnd,swtitle))
    { printf("Failure to update window text in frame\n");
    }
    */
    ulrc = WinChangeSwitchEntry(hswhnd,&swc);
    return ulrc;
  }
}
/*
Ŀ
 rxvisible entry as function returning ULONG 

*/
ULONG rxvisible(ULONG ulvisible) /* 1=visible, 0=not */
{ SWCNTRL swc;
  HSWITCH hswhnd = rxswhnd(&swc);
  if (hswhnd == NULLHANDLE) return 1;
  swc.uchVisibility = ulvisible ? SWL_VISIBLE : SWL_INVISIBLE;
  WinShowWindow(swc.hwnd,ulvisible);
  return WinChangeSwitchEntry(hswhnd,&swc);
}
