/* An example of opening several (non MODAL) windows simultaneously. It
   creates a Main Window, and numerous Child Dialogs. Each window has
   only 1 Group in it. That's not to say that you can't have more Groups in a
   window, but just that it simplifies this example. Then it does a message
   loop, letting the user choose which window to interact with. Some of the
   windows have a Group which causes RXDLG to return (ie, END Flag set).
   Other windows have the RESULT flag set so that the user can end by pressing
   ESC or ENTER.  Of course, clicking upon a CLOSE BOX always causes RXDLG to
   return. We check the RXWIND variable and do some appropriate processing for
   that window -- namely, we just print out the returned value for that
   window's groups.
*/

/* Trap ERROR and FAILURE */
SIGNAL ON ERROR
SIGNAL ON FAILURE


/* ================ Create "Main Window" dialog =============== */
/* First Group is ENTRY */
RXTYPE.1 = 'ENTRY'

/* Use of a control in this group causes RXDLG to return */
RXFLAGS.1 = 'END'

/* Labels for each entry, and Groupbox. Note: 3rd entry has no label */
RXLABEL.1 = 'Range|Second||Fourth|All'

/* Variable name where the text "typed into" the entries is stored */
RXVAL.1 = 'TEXT'

/* TotalControls, ControlsPerLine, WidthOfControls */
RXINFO.1 = '4 2 45'

/* Position of Group */
RXX.1 = 65
RXY.1 = 70

/* The text for the 4 entries */
TEXT.1 = 'Hello'
TEXT.2 = 'Hi'
TEXT.3 = ' '         /* A blank entry initially */
TEXT.4 = 'More text'

/* Default size and position (also gives us sizing and max button) */
RXWIN1 = ''

/* Specify NOCLOSE (since we'll do the closing ourselves, and also
    RESULT so that RXDLG returns upon ESC or ENTER */
RXDLG 1 '"Main Window"' 'RXWIN1' 'NOCLOSE|RESULT'



/* ================ Create "PushButtons"  dialog =============== */
/* First Group is PUSH */
RXTYPE.1 = 'PUSH'

/* Use of a control in this group causes RXDLG to return */
RXFLAGS.1 = 'END'

/* Labels for each button, and Groupbox */
RXLABEL.1 = 'One|Two|Three|Four|Choices'

/* TotalControls, ControlsPerLine, WidthOfControls */
RXINFO.1 = '4 2 0'

/* Default choice for PushButton is nothing */
RXVAL.1 = ''

/* Position */
RXX.1 = 7
RXY.1 = 6

RXWIN2 = '190 108 8 8'
RXDLG 1 '"Pushbuttons"' 'RXWIN2' 'NOCLOSE|MIN|RESULT'




/* ================== Create "Spin Me" dialog ================= */
/* First Group is SPIN */
RXTYPE.1 = 'SPIN'

/* Use of a control in this group causes RXDLG to return */
RXFLAGS.1 = 'END'

/* Max, Min, Label for each slider, and groupbox */
RXLABEL.1 = '255 0 Range 1:| 20 10 Range 2:|Try us'

/* Values for each slider */
RXVAL.1 = '64 19'

/* TotalControls, ControlsPerLine, WidthOfControls */
RXINFO.1 = '2 1 70'

/* Position */
RXX.1 = 80
RXY.1 = 18

RXWIN3 = '190 104 30 30'
RXDLG 1 '"Spin Me"' 'RXWIN3' 'NOCLOSE|SIZE|RESULT'




/* ================= Create "Understand?" dialog ================ */
/* First Group is RESULT */
RXTYPE.1 = 'RESULT'

/* Return a 1 for OK, YES, RETRY, ENTER, etc. Return a 0 for
    CANCEL, NO, IGNORE, etc. No need to set 'END' since
    RESULTBUTTON group always causes RXDLG to return */
RXFLAGS.1 = 'BOOL'

/* No groupbox */
RXLABEL.1 = ' '

/* Give me "Yes" and "No" */
RXINFO.1 = '3 4'

/* Irrelevant for RESULTBUTTON */
RXVAL.1 = ''

/* Position */
RXX.1 = 3
RXY.1 = 3

RXWIN4 = '112 60 8 50'
RXDLG 1 '"Understand?"' 'RXWIN4' 'NOCLOSE|RESULT'




/* ============= Message loop on the dialog windows ============ */
more:
   /* Because we have more than 1 window open and have not specified MODAL
       for the active one, the window that is active when we call RXDLG may
       not be the same window that is active when RXDLG returns (ie, it may
       not be the same window that causes RXDLG to return and sets all of the
       RXVAL and any LIST/DROP BOX stem variables, RXID, RXSUBID, and RXWIND
       variables, and the Dimensions string for that window. For this
       reason, let's not specify any window title nor operation to RXDLG. This will
       cause RXDLG to pick up on the window that was last active,
       will interact with the user, and won't force any dialog closure (unless the
       user happens to have activated a window that wasn't created with
       NOCLOSE (here, all of our windows are NOCLOSE).
       Note that if all of the windows are closed (including the main window),
       and we don't specify a window title, this amounts to specifying a
       window title that isn't open. In this case, Rexx Dialog sets an ERROR (ie,
       "Can't find REXX dialog." error message), and we would branch to that
       label. In fact, our way out of here is to EXIT after we detect that the
       user has clicked on the main window CLOSE ICON (and we kill that
       window).
 */

   /* Wait on the window(s). We go to sleep while user manipulates dialogs,
       until such time as the user presses ESC or ENTER, or tries to close a dialog
       using its CLOSE ICON, or uses a RESULT Group or a Control within some
       Group that has its END Flag set. Other things that could cause RXDLG to
       return are a window timeout and if the user pressed a key in a window
       with its KEYS Flag set... but we haven't utilized those features */
   RXDLG  /* NOTE: No window title means "use Main Window". No operation
		   specified means an operation of 0 (ie, PROCESS) */

   /* RXWIND now specifies which window woke us up. (The window is still
      there because we specified NOCLOSE and didn't do an Operation that
      causes a window closure).
	 All of the RXVAL (and any LIST/DROP BOX stem) variables, RXID,
     RXSUBID, and RXWIND variables (and the Dimensions string for that
     window) have been setup according to the state of this window when RXDLG
     returned */

   /* Close that window (without setting up any variables again, ie,
       Operation 4). NOTE: You might not want to do this without
       checking RXWIND. After all, we could be closing our main window
       right now. Also, you might want to continue leaving this window
       open, or maybe only close it if the user pressed ESC or clicked the
       CLOSE ICON or selected a CANCEL/NO/IGNORE/OFF button in a
       RESULT Group. We haven't tested for those conditions yet.
       Actually, what I'm doing here is EXACTLY what Rexx Dialog would
       be doing if we created all of our windows without NOCLOSE, and
       we could eliminate this Operation. Try it */
   RXDLG '"'RXWIND'"' 4

   /* Did user press ESC or click on the CLOSE ICON? Note that we haven't set
      the KEYS flag of any window, so we don't need to check for particular
      negative values of RXID because the only 2 we ever receive here are
      for the ESC key and CLOSE ICON */
   IF RXID < 0 THEN DO
      RXSAY '"'RXWIND'" aborted.'
      IF RXWIND = 'Main Window' THEN EXIT
   END

   ELSE DO
   /* If user didn't abort, then display the value strings for this window.
       IMPORTANT: If he didn't abort, then he either pushed some SUCCESS (ie,
       OK, RETRY, ENTER, etc) or FAILURE (ie, CANCEL, IGNORE, NO, etc)
       button in a RESULT Group, or he used some Group whose 'END' Flag
       we set, or he pressed the ENTER key while the window itself had the
       focus, or a timeout occured if we setup that for this window, or some
       key was pressed if the window had its KEYS Flag set.
	   If we had a RESULT Group with a FAILURE type button in the window,
       and that caused RXDLG to return, normally we'd check the RXVAL of that
       Group, and if an odd number, abort any operation. (For BOOL, we'd
       abort on 0) */

       /* Did user press the ENTER key while the window had the focus?
	   Note that we haven't setup a timeout, nor set the KEYS flag
	   of any window, so we don't need to check RXSUBID. It's always
	   going to be 10 (ie, ENTER) here if RXID is 0 */
       IF RXID = 0 THEN DO
	   RXSAY 'Pressed ENTER on "'RXWIND'".'
       END

       /* Display the Group #, Control # of the control that caused RXDLG to return */
       ELSE RXSAY 'Group #'RXID', Control #'RXSUBID' ended "'RXWIND'" dialog.'

       SELECT
	   /* =============== Spin Me Dialog ================ */
	   WHEN RXWIND = "Spin Me" THEN DO
	   /* Display spin values for this group, one spin at a time */
	       DO i = 1 TO 2
		   PARSE VAR RXVAL.1 knob RXVAL.1
		   RXSAY '"'RXWIND'" Group 1, Knob #'i' = 'knob
	       END
	    END

	   /* ============== Pushbuttons Dialog ============== */
	   WHEN RXWIND = "Pushbuttons" THEN DO
	   /* NOTE: RXVAL could be "" for a PUSH/CHECK/RADIO button group if
	       the user didn't use any buttons and we didn't initially set any
	       of them (ie, RXVAL was a null string when the group was
	       created), or if there are ALL\NONE buttons and the user
	       selected ALL. By setting the RXVAL for a PUSH or RADIO BUTTON
	       group to "" before calling RXDLG to create a window, you can determine
	       whether the user selected any of the buttons */
	       IF RXVAL.1 > '' THEN RXSAY '"'RXWIND'" Group 1, pushed button #'RXVAL.1
	       ELSE RXSAY '"'RXWIND'" Group 1, no button pushed.'
	   END

	   /* ============== Understand? Dialog ============== */
	   WHEN RXWIND = "Understand?" THEN DO
	   /* See IMPORTANT above. We know that Group 1 in this dialog is a
	       RESULTBUTTON. Normally, we'd check the group's RXVAL, (in this
	       case, RXVAL.1), and if an odd number, abort any operation that
	       this dialog was supposed to provide us input for. Since we
	       asked for YES and NO buttons, and also specified BOOL result,
	       RXVAL will be 1 for YES and 0 for NO */
	       i = 'NO'
	       IF RXVAL.1 THEN i = 'YES'
	       RXSAY '"'RXWIND'" Group 1, result is 'i
	   END

	   /* ================ Main Window ================= */
	   WHEN RXWIND = "Main Window" THEN DO
	   /* Display each entry's text, one entry at a time */
	       DO i = 1 TO 4 /* There are 4 entries in this Group */
		   RXSAY '"'RXWIND'" Group 1, entry #'i' is "'TEXT.i'"'
	       END

	   /* NOTE: If you wanted a graceful end to the program (ie, so that
	       you don't get the "Can't find REXX dialog." error message),
	       you could do an EXIT right here. After all, we already closed
	       the main window (and therefore any children that were still
	       open), so if we try to do RXDLG again, that's when the FAILURE
	       occurs */
	       EXIT
	   END
       END
   END

/* Do another message loop */
SIGNAL more

/* ========================== Done ========================== */

FAILURE:
    /* NOTE: the variable RC contains the returned error message (not a number,
	unless we use RXERR to set up Rexx Dialog to return error numbers instead).
	Because we used the default severity level, Rexx Dialog has already displayed
	this error message to the enduser */
    EXIT
ERROR:
    /* NOTE: the variable RC contains the returned error message (not a number,
	unless we use RXERR to set up Rexx Dialog to return error numbers instead).
	Because we used the default severity level, Rexx Dialog has already displayed
	this error message to the enduser */
    EXIT
