/*-------------------------------------------------------------------------*\
 * VERSION CONTROL:
 *		$Id: rftp.cmd,v 1.3 1995/08/09 14:49:49 rodb Exp rodb $
 *		$Tabs: 4 7 $
 *-------------------------------------------------------------------------*
 * NAME
 *		rftp --- Command-line FTP using REXX rxftp.dll
 *
 * SYNOPSIS
 *		rftp -h host [-l uid] [-p passwd] -G|-P [-a|-b] [-HvV] l_file r_file
 *
 * DESCRIPTION
 *		The rftp command allows you to transfer a file between your PC and a 
 *		foreign host.
 *
 *		This REXX program requires the REXX File Transfer Protocol (FTP) 
 *		Application Program Interface (API) package supplied with IBM TCP/IP 
 *		Version 2.0 for OS/2 and OS/2 Warp Connect.
 *
 * OPTIONS
 *		-h host
 *				Specifies the foreign host to which you are connecting
 *		-l uid
 *				User ID specifies the name associated with you by the foreign
 *				host to which you are establishing a connection.  If not 
 *				specified, rftp will check the environment for the presence of 
 *				the variable USER, and if found use the value assigned to it.
 *		-p passwd
 *				Password specifies your unique password, which is associated 
 *				with your user ID by the foreign host to which you are 
 *				establishing a connection.  If not specified, rftp will check 
 *				the environment for the presence of the variable THE_WORD, and 
 *				if found use the value assigned to it.
 *		-G
 *				Get (receive) a file from the remote system
 *		-P
 *				Put (send) a file to the remote system
 *		-H
 *				Display help text and exit.  No other actions are taken.
 *		-a
 *				Sets the file transfer type to ASCII
 *		-b
 *				Sets the file transfer type to binary (default)
 *		-v
 *				Display version of RxFTP DLL and exit.  No other actions are 
 *				taken.
 *		-V
 *				Operate in verbose mode
 *		l_file
 *				The name of the file on the local PC
 *		r_file
 *				The name of the file on the foreign host
 *-------------------------------------------------------------------------*
 *	Copyright (c) 1994-95 Lawrence R Buchanan.  ALL RIGHTS RESERVED.
 *
 *	This program is free software; you are free to do whatever you want 
 * with it.  The only requirement is that if you use the getopt() option 
 * parsing subroutines in code that you distribute, that you also include 
 * the copyright messages that appear in the headers of those subroutines.
 *	
 *	This program is distributed in the hope that it will be useful, 
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
\*-------------------------------------------------------------------------*/


/*-------------------------------------------------------------------------*\
 * Initialize getopt. stem with commandline parameters.
 *
 * These statements MUST appear at the beginning of any program that uses 
 * getopt().
\*-------------------------------------------------------------------------*/
parse arg argv
retcd =  init_getopt( argv )
if retcd = -1 then do
	say getopt.!program ": Unbalanced quotation marks in command-line"
	exit -1
end


/* If no parameters issue usage message and exit. */
if getopt.0 = 0 then do
	call Usage
	exit 1
end


/* Begin Main program */

/* Setup program defaults. */
Verb      = ""
host_name = ""
login_id  = ""
password  = ""
AscOrBin  = "B"				/* (A)scii or (B)inary */
Verbose   = 0
prtver    = 0

rxFtpErr = 0
rxFtp.FTPSERVICE    = 10
rxFtp.FTPHOST       = 11
rxFtp.FTPSOCKET     = 12
rxFtp.FTPCONNECT    = 13
rxFtp.FTPLOGIN      = 14
rxFtp.FTPABORT      = 15
rxFtp.FTPLOCALFILE  = 16
rxFtp.FTPDATACONN   = 17
rxFtp.FTPCOMMAND    = 18
rxFtp.FTPPROXYTHIRD = 19
rxFtp.FTPNOPRIMARY  = 20


/* Get the option flags and arguments and set up the program environment. */
call DecodeSwitches

/* Register rxFtp DLL */
if RxFuncQuery( "FtpLoadFuncs" ) then do
	rc = RxFuncAdd( "FtpLoadFuncs", "RxFtp", "FtpLoadFuncs" )
	rc = FtpLoadFuncs( "Quiet" )
end

/* If -v specified, print RxFTP DLL version and exit. */
if prtver then do
	rc = FtpVersion( prtver )
	say "RxFtp DLL version is" prtver
	exit 0
end

/* If no files are given to be printed issue message and exit. */
if (getopt.0 - getopt.!optind) <> 1 then do
	say getopt.!program ": Missing local or remote file name"
	exit 2
end

/* If not specified, get the default login ID and password from the environment. */
if login_id = "" then
	login_id   = value( "USER",, "OS2ENVIRONMENT" )
if password = "" then
	password  = value( "THE_WORD",, "OS2ENVIRONMENT" )

/* Get the local/remote file names from getopt. */
i = getopt.!optind
local_file  = getopt.i
i = getopt.0
remote_file = getopt.i

if Verbose then do
	call time "R"
	say "Start Time:" time( "N" )
	say
	say "Local File = " local_file
	say "Host       = " host_name
	say "Host File  = " remote_file
	say "For User   = " login_id
	say "Copy Type  = " AscOrBin
end

rc = FtpSetUser( host_name, login_id, password )
if rc = 0 then do
	say getopt.!program ": Error setting host, userid, or password"
	exit 1
end

if Verb = "Get" then
	rc = FtpGet( local_file, remote_file, AscOrBin )
else
	rc = FtpPut( local_file, remote_file, AscOrBin )

/* Save the return codes. */
rxFtpRc  = rc
rxFtpErr = FTPERRNO

/* Terminate the FTP session and reset the host/userid/password. */
rc = FtpLogoff()

if Verbose then do
	ElapsedTime = time("E")
	say "Request Complete"
	say "End Time  :" time("N") "    Elapsed Time  :" ElapsedTime
	say
	say "Ftp"Verb"(): rc =" rxFtpRc "    FTPERRNO =" rxFtpErr
end

if rc = -1 & rxFtpErr = 0 then
	exit 0
else 
	exit rxFtpRc

/* End of Main */



/*-------------------------------------------------------------------------*\
 * NAME:
 *    DecodeSwitches - decodes command-line options
 *
 * PARAMETERS:
 *    *NONE
 *
 * RETURNS:
 *    *NONE
\*-------------------------------------------------------------------------*/
DecodeSwitches:
	errflg = 0
	optstr = "abGHh:p:Pu:vV"
	c = getopt(optstr)
	do while c <> -1
		select
			when c = "a" then 
				AscOrBin = "A"			/* ASCII transfer */

			when c = "b" then 
				AscOrBin = "B"			/* Binary transfer */

			when c = "G" then
				if Verb = "" then
					Verb = "Get"
				else
					errflg = 1

			when c = "H" then do
				call PrintHelp
				exit 1
				end

			when c = "h" then 
				host_name = getopt.!optarg

			when c = "p" then 
				password = getopt.!optarg

			when c = "P" then
				if Verb = "" then
					Verb = "Put"
				else
					errflg = 1

			when c = "u" then 
				login_id = getopt.!optarg

			when c = "v" then
				prtver = 1

			when c = "V" then 
				Verbose = 1

			otherwise 
				do
					call Usage
					exit 2
				end
		end
	
		if errflg then do
			say getopt.!program ":" getopt.!optarg "is an invalid argument for option" c 
			exit 2
		end
		c = getopt(optstr)
	end

return
/* End of DecodeSwitches */


/*-------------------------------------------------------------------------*\
 * NAME:
 *    PrintHelp - Print detailed help message.
 *
 * PARAMETERS:
 *    *NONE
 *
 * RETURNS:
 *    *NONE
\*-------------------------------------------------------------------------*/
PrintHelp:
	call Usage

	say
	say "       -h host"
	say "            Specifies the foreign host to which you are connecting"
	say "       -l uid"
	say "            Login ID specifies the name associated with you by the foreign"
	say "            host to which you are establishing a connection"
	say "       -p passwd"
	say "            Password specifies your unique password, which is associated with"
	say "            your user ID by the foreign host to which you are establishing a "
	say "            connection"
	say "       -G"
	say "            Get (receive) a file from the remote system"
	say "       -P"
	say "            Put (send) a file to the remote system"
	say "       -H"
	say "            Display help text and exit.  No other actions are taken."
	say "       -a"
	say "            Sets the file transfer type to ASCII"
	say "       -b"
	say "            Sets the file transfer type to binary (default)"
	say "       -v"
	say "            Display version of RxFTP DLL and exit.  No other actions are"
	say "            taken."
	say "       -V"
	say "            Operate in verbose mode"
	say "       l_file"
	say "            The name of the file on the local PC"
	say "       r_file"
	say "            The name of the file on the foreign host"
return
/* End of Usage */


/*-------------------------------------------------------------------------*\
 * NAME:
 *    Usage - Print usage message.
 *
 * PARAMETERS:
 *    *NONE
 *
 * RETURNS:
 *    *NONE
\*-------------------------------------------------------------------------*/
Usage:
	say "Usage:" getopt.!program "-h host [-l uid] [-p passwd] -G | -P [-a|-b] [-HvV] l_file r_file"
return
/* End of Usage */



/*-------------------------------------------------------------------------*\
 * NAME:
 *    getopt - Scan elements of stem variable getopt. for option characters
 *
 * PARAMETERS:
 *    optstr - Specifies a string of recognized flag letters. If a letter 
 *             is followed by a : (colon), the flag is expected to take a 
 *             parameter that may or may not be separated from it by one or 
 *             more spaces.
 *
 * RETURNS:
 *    The getopt subroutine returns the next flag letter specified on the 
 *    command line.  A value of -1 (negative one) is returned when all 
 *    command line flags have been parsed.
 *
 *    If the getopt subroutine encounters an option character that is not 
 *    contained in optstr, or a missing option argument, it returns a ? 
 *    (question mark).
 *
 * NOTES:
 *    The getopt. stem variable must have been prepared by a call to 
 *    init_getopt() prior to calling getopt().
 *
 *    See getopt(3r) for complete information.
 *-------------------------------------------------------------------------*
 *	Copyright (c) 1994-95 Lawrence R Buchanan.  ALL RIGHTS RESERVED.
 *
 *	This program is free software; you are free to do whatever you want 
 * with it.  The only requirement is that if you use this subroutine in 
 * code that you distribute, that you also include this copyright message.
 *	
 *	This program is distributed in the hope that it will be useful, 
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
\*-------------------------------------------------------------------------*/
getopt: procedure expose getopt.
   parse arg optstr

   NULL = "00"x
   i = getopt.!optind
   getopt.!optopt = ""
   getopt.!errmsg = ""

   if getopt.!sp = 1 then do
      if getopt.!optind > getopt.0 | ,
         substr(getopt.i, 1, 1, NULL) <> "-" | ,
         substr(getopt.i, 2, 1, NULL) = NULL then
         return -1
      else 
         if getopt.i =  "--" then do
            getopt.!optind = getopt.!optind + 1
            return -1
         end
   end

   c = substr(getopt.i, getopt.!sp+1, 1, NULL)
   getopt.!optopt = c
   cp = pos(c, optstr)

   if c = ":" | cp = 0 then do
      getopt.!errmsg = getopt.!program ": illegal option --" c
      if getopt.!opterr = 1 then 
         say getopt.!errmsg
      getopt.!sp = getopt.!sp + 1
      if substr(getopt.i, getopt.!sp+1, 1, NULL) = NULL then do
         getopt.!optind = getopt.!optind + 1
         getopt.!sp = 1
      end
      return "?"
   end

   cp = cp + 1
   if substr(optstr, cp, 1, NULL) = ":" then do
      if substr(getopt.i, getopt.!sp+2, 1, NULL) <> NULL then do
         getopt.!optarg = substr(getopt.i, getopt.!sp+2)
         getopt.!optind = getopt.!optind + 1
      end
      else do
         getopt.!optind = getopt.!optind + 1
         i = getopt.!optind
         if getopt.!optind > getopt.0 then do
            getopt.!errmsg = getopt.!program ": option requires an argument --" c
            if getopt.!opterr = 1 then 
               say getopt.!errmsg
            getopt.!sp = 1
            return "?"
         end
         else do
            getopt.!optarg = getopt.i
            getopt.!optind = getopt.!optind + 1
         end
      end

      getopt.!sp = 1
   end
   else do
      getopt.!sp = getopt.!sp + 1
      if substr(getopt.i, getopt.!sp+1, 1, NULL) = NULL then do
         getopt.!sp = 1
         getopt.!optind = getopt.!optind + 1
      end

      getopt.!optarg = ""
   end

return c
/* End of getopt */



/*-------------------------------------------------------------------------*\
 * NAME:
 *    init_getopt - Initialize getopt. stem variable for getopt()
 *
 * PARAMETERS:
 *    argv - Command line to be parsed.
 *
 * RETURNS:
 *    The init_getopt subroutine returns the number of arguments entered 
 *    on the command line.  If it detects unbalanced double quotes it 
 *    returns -1.
 *
 *    See init_getopt(3r) for complete information.
 *-------------------------------------------------------------------------*
 *	Copyright (c) 1994-95 Lawrence R Buchanan.  ALL RIGHTS RESERVED.
 *
 *	This program is free software; you are free to do whatever you want 
 * with it.  The only requirement is that if you use this subroutine in 
 * code that you distribute, that you also include this copyright message.
 *	
 *	This program is distributed in the hope that it will be useful, 
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
\*-------------------------------------------------------------------------*/
init_getopt: procedure expose getopt.
   parse arg argv

   /* Initialize variables used in getopt(). */
   getopt. = ""
   getopt.!opterr = 1
   getopt.!optind = 1
   getopt.!sp     = 1

   /* Place program name in getopt.!program. */
	parse source os . getopt.!program .
	if os = 'OS/2' then do
		getopt.!program = filespec('N', getopt.!program)
		i = lastpos('.', getopt.!program)
		if i > 0 then
			getopt.!program = delstr(getopt.!program, i)
	end

   /* Make sure the command line contains an even number of 
      double quotes.  If it doesn't, return -1 to indicate an error. */
   if init_getopt_count_quotes(argv) // 2 then do
      getopt.0 = 0
      return -1
   end

   i = 0
   /* Load command line arguments into getopt.1 through getopt.n. */
   do while argv <> ""
      i = i + 1
      parse var argv getopt.i argv

      /* If quoted argument, make sure we get all of it from the command line. */
      if pos('"', getopt.i) > 0 then do
         cnt = init_getopt_count_quotes(getopt.i)
         parse var getopt.i opt '"' tmparg
         getopt.i = opt || strip(tmparg, "T", '"')
         if cnt = 1 then do
            parse var argv remarg '"' argv
            getopt.i = getopt.i remarg
         end
      end
   end
   getopt.0 = i

return getopt.0
/* End of init_getopt */



/*-------------------------------------------------------------------------*\
 * NAME:
 *    init_getopt_count_quotes - Count number of double quotation marks (") 
 *                               in a string
 *
 * PARAMETERS:
 *    str - The string to count double quotation marks (") in
 *
 * RETURNS:
 *    The number of double quotation marks (") contained in the string.
 *-------------------------------------------------------------------------*
 *	Copyright (c) 1994-95 Lawrence R Buchanan.  ALL RIGHTS RESERVED.
 *
 *	This program is free software; you are free to do whatever you want 
 * with it.  The only requirement is that if you use this subroutine in 
 * code that you distribute, that you also include this copyright message.
 *	
 *	This program is distributed in the hope that it will be useful, 
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
\*-------------------------------------------------------------------------*/
init_getopt_count_quotes: procedure
	parse arg str
	
	cnt = 0
	pos = pos('"', str)
	do while pos > 0
		cnt = cnt + 1
		pos = pos('"', str, pos+1)
	end

return cnt
/* End of init_getopt_count_quotes */
