/* -------------  INSTALL.CMD - software installation 2/REXX script -------------


Date            Author                  Description

95/09/08        Klaus Brunner           minor problem in InsMkDir fixed
95/09/01        Klaus Brunner           first release
*/


ADDRESS CMD "@ECHO OFF"
ADDRESS CMD "ANSI ON >NUL"

/* ---------------- REQUIRED USER DATA BEGIN -------------------------------- */
/* -- program information */
InsProgName	= "ProgramTitle V.V"	/* program title */

/* -- program requirements */
InsTargetBytesRequired = 10000		/* free space in bytes required on target drive */
InsTargetOSRequired = 2.1		/* minimum OS/2 version required (Warp v3 = 2.30!)*/

InsStartableFromCMD = 1                 /* is program startable from CMD? */
InsProgramStartCall = "PROGRAM"		/* if yes, that's what user has to type */
/* ---------------- REQUIRED USER DATA END ---------------------------------- */


/* ---------------- OPTIONAL USER DATA BEGIN --------------------------------- */
/* -- screen information */
InsScreenHeader.1	= "[44;37m[2J[1B  Installation of "InsProgName
InsScreenHeader.2	= "  "

/* -- general path information */
InsSourcePath	= ""
InsTargetPath   = "C:\PROGRAM"		/* fill in default target path here */

/* -- installation options */
InsRestartRequired = 0			/* advise user to restart system afterwards */
                                        /* automatically set by InsConfigClose */
/* ---------------- OPTIONAL USER DATA END ----------------------------------- */

/* -- global error flag */
InsErrorFlag       = 0		/* set by failing Ins... functions, must be reset
                                   manually. */
                   
/* -- internal globals */
InsProgramObjectMade = 0        /* set by InsCreateProgramObject if a WPProgram object
                                   was successfully generated */

/* -- REXX options */
CALL ON HALT NAME InsHaltTrap                                   /* Ctrl-Break trap */
CALL RxFuncAdd "SysLoadFuncs", "RexxUtil", "SysLoadFuncs"       /* get RexxUtil */
CALL SysLoadFuncs

/* ------------------------------- MAIN PROGRAM ------------------------------ */
InsMainProgram:

        /* your code goes here */

CALL InsExit 0

/* ------------------------------- SUBPROGRAMS ------------------------------- */

/* --------------------------------- screens --------------------------------- */
/* Note: all screen procedures use globals and reset InsErrorFlag before
   returning to the caller. */

/* InsWelcomeScreen */
/* displays welcome text */
InsWelcomeScreen:
	CALL InsScreenInit
	SAY "	Welcome to the installation procedure of" InsProgName"!"
	SAY  
	SAY "	* This program will guide you through the process of copying the"
	SAY "	  necessary files to your hard disk, setting up your operating"
	SAY "	  system environment and making the program easily accessible."
	SAY  
	SAY "	* You may quit from this procedure at any time by pressing"
	SAY "	  CTRL+BREAK on your keyboard. Note that the installation"
	SAY "	  process must be completed to ensure proper function of"
	SAY "	  "InsProgName"."
	SAY  
	CALL InsGetEnter
        InsErrorFlag = 0
RETURN 1


/* InsPathScreen */
/* determines source path, gets target path string from user, checks for accuracy
   and disk space, creates subdirectory if necessary, checks OS version. Exits
   program on fatal errors. */
InsPathScreen:
        /* find out source path, i.e. location of this file */
        /* part of this was taken from a Warp BonusPak installation .CMD */
        PARSE SOURCE . . Argv0
        Argv0 = REVERSE(Argv0)
        InsSourcePath = InsFormatPath(REVERSE(SUBSTR(Argv0,POS("\",Argv0)+1)))

        DO UNTIL Success <> 0
                /* check OS, get target path from user */
                CALL InsScreenInit

                IF InsCheckOS(InsTargetOSRequired) = 0 THEN
                        CALL InsErrorScreen "OSERROR", "FATAL"

                SAY "	* Target Path"
                SAY
                SAY "	  Please enter the location where you want to install the"
                SAY "	  program files. The specified drive and directory must be"
                SAY "	  writable and should provide at least "TRUNC(InsTargetBytesRequired/1024+1),
                         "kilobytes of"
                SAY "	  free space. The default target path is "InsTargetPath"."
                SAY
                CALL CHAROUT ,"	  "                     /* align cursor position to text */

                UserPath = LINEIN()                     /* get user input */
                IF UserPath = "" THEN                   /* substitute w/ default if necessary */
                        UserPath = InsTargetPath

                /* clean and format path */
                UserPath = InsFormatPath(InsCleanString(UserPath))

                /* check path, try to create directory if it does not exist */
                Success = 1
                IF InsCheckSpace(UserPath, InsTargetBytesRequired) = 0 THEN
                        Success = 0
                ELSE IF InsCheckPath(UserPath) = 0 THEN
                        IF InsMakeDirectory(UserPath) = 0 THEN
                                Success = 0

                SAY
                IF Success = 0 THEN DO
                        SAY "	* Target Path Error"
                        SAY
                        SAY "	  The path you entered is nonexistent, does not provide enough"
                        SAY "	  space for installation or is write-protected. Please try again"
                        SAY "	  using a suitable path."
                        CALL InsGetEnter "Press ENTER to retry, or CTRL+BREAK to quit."
                END
                ELSE DO
                        SAY "	* Target Path Confirmation"
                        SAY
                        SAY "	  The program files will be copied to "UserPath"."
                        CALL InsGetEnter
                END
        END /* DO UNTIL Success <> 0 */
        InsTargetPath = UserPath
        InsErrorFlag = 0
RETURN 1


/* InsErrorScreen [ErrorCode], [SeverityCode] */
/* Displays an error message for ErrorCode. See code for valid ErrorCodes/SeverityCodes
   and associated text. If SeverityCode="FATAL", calls InsAbort */
InsErrorScreen:
        PARSE UPPER ARG ErrorCode, SeverityCode
        CALL InsScreenInit
        SAY "	* Installation Error"
        SAY
        SELECT
                WHEN ErrorCode="DISKERROR" THEN DO
                        SAY "	  An error occured during disk operations. This may be"
                        SAY "	  caused by incorrect path information, missing source"
                        SAY "	  files, missing target drive space, missing read/write"
                        SAY "	  permission (e.g. on network drives) or disk media"
                        SAY "	  problems."
                END
                WHEN ErrorCode="CONFIGERROR" THEN DO
                        SAY "	  An error occured while trying to change the CONFIG.SYS"
                        SAY "	  file. This may be caused by missing read/write permission,"
                        SAY "	  insufficient drive space, an unsupported operating system"
                        SAY "	  or disk media problems."
                END
                WHEN ErrorCode="OSERROR" THEN DO
                        SAY "	  The operating system version you are using is not supported"
                        SAY "	  by "InsProgName"."
                END

                /* insert your own ErrorCode handlers here... */

        OTHERWISE
                SAY "	  An unspecified error occured during installation."
        END

        SAY

        IF SeverityCode="FATAL" THEN DO
                CALL InsGetEnter "Press ENTER now to exit."
                CALL InsAbort
        END
        ELSE
                CALL InsGetEnter "Press ENTER to ignore, or CTRL-BREAK to quit."
        InsErrorFlag = 0
RETURN 1


/* InsCompletionScreen */
/* displays goodbye text, depending on some flags */
InsCompletionScreen:
	CALL InsScreenInit
	SAY "	Installation of "InsProgName" is now complete!"
	IF InsRestartRequired THEN DO
		SAY  
		SAY "	* To ensure proper function of the program, please perform a"
		SAY "	  system shutdown and restart the system."
	END
	IF InsProgramObjectMade THEN DO
		SAY  
		SAY "	* Start the program by double-clicking on the newly created icon"
		SAY "	  on your desktop. You may safely move, rename, copy or delete"
		SAY "	  this object without affecting the actual program files."
	END
	IF InsStartableFromCMD THEN DO
		SAY  
		SAY "	* From the command line, start the program by typing"
		SAY  
		SAY "		  "InsProgramStartCall
		SAY  
		SAY "	  followed by the ENTER key."
	END
	SAY
        CALL InsGetEnter("Press ENTER now to exit installation.")
        InsErrorFlag = 0
RETURN 1

/* --------------------------- screen related procedures --------------------- */

/* InsScreenInit */
/* clear screen, write header text */
InsScreenInit: PROCEDURE EXPOSE InsScreenHeader.
	SAY "[2J"             /* ANSI clear screen (alternate: CALL SysCls) */
	SAY InsScreenHeader.1
	SAY InsScreenHeader.2
	SAY  
RETURN 1

/* InsGetEnter [message] */
/* display message, get ENTER */
InsGetEnter:	PROCEDURE
	PARSE ARG message
	SAY  
	IF message = "" THEN
		SAY "	Press ENTER to continue, or CTRL+BREAK to quit."
	ELSE	
		SAY "	"message
	LINEIN()
RETURN 1

/* -------------------------- config.sys procedures -------------------------- */

/* InsConfigOpen <BackupName> */
/* This function will make a backup copy of config.sys (BackupName) and
   read its contents into the InsConfigArray global compound variable - affects
   InsErrorFlag. Note that the actual config.sys file remains untouched and is
   closed afterwards. */
InsConfigOpen: PROCEDURE EXPOSE InsErrorFlag InsConfigArray. InsConfigArrayStatus
        PARSE UPPER ARG BackupName

        BootDrive = InsFormatPath(InsGetBootDrive())    /* get boot drive */
        BackupFullName=BootDrive""BackupName

        /* copy file, open backup file */
        /* note: documentation says that STREAM() should return "READY". On my system,
           STREAM() returns "READY:". This to explain the LEFT(STREAM(..)) call. */

        CALL SysFileDelete BackupFullName
        DROP InsConfigArray.

        IF (InsFileCopy(BootDrive"CONFIG.SYS", BackupFullName) = 0) |,
           (LEFT(STREAM(BackupFullName, "C", "OPEN READ"), 5) <> "READY") THEN DO
                InsErrorFlag = 1
                RETURN 0
        END

        /* read from backup into InsConfigArray */
        i = 1
        DO WHILE LINES(BackupFullName)
                InsConfigArray.i=LINEIN(BackupFullName)
                i = i + 1
        END
        InsConfigArray.0 = i - 1           /* store no. of lines in InsConfigArray.0 */

        /* close backup file */
        IF LEFT(STREAM(BackupFullName, "C", "CLOSE"), 5) <> "READY" THEN DO
                InsErrorFlag = 1
                RETURN 0
        END
        InsConfigArrayStatus = "OPEN"
RETURN 1

/* InsConfigAppend <Statement> */
/* This function will append a line containing Statement to the InsConfigArray
   created by InsConfigOpen(). Use for things like "DEVICE=..." or "SET X=...".
   Note: See InsConfigAddPath() for changing of PATH/LIBPATH/DPATH...
   Affects InsErrorFlag.
*/
InsConfigAppend: PROCEDURE EXPOSE InsConfigArray. InsConfigArrayStatus
        PARSE ARG Statement

        IF InsConfigArrayStatus <> "OPEN" THEN DO
                InsErrorFlag = 1
                RETURN 0
        END

        Lines = InsConfigArray.0 + 1
        InsConfigArray.Lines = Statement
        InsConfigArray.0 = Lines
RETURN 1

/* InsConfigAddPath <TargetPathName>, <Path> */
/* This function will try to locate TargetPathName (PATH,LIBPATH,DPATH,HELP...) in
   the InsConfigArray, check if it already contains Path and if not, append it.
   If the TargetPathName cannot be found, a line like "SET TargetPathName=Path"
   will be appended to the array. Note: Use only fully qualified path strings
   including drive specification. Returns 1 if TargetPathName was found, 0 if
   a new statement had to be appended. Affects InsErrorFlag.
*/
InsConfigAddPath: PROCEDURE EXPOSE InsConfigArray. InsConfigArrayStatus
        PARSE UPPER ARG TargetPathName, Path
        Path = InsFormatPath(Path)      /* format path if necessary */
        Path = Path";"                  /* append ";" */

        IF InsConfigArrayStatus <> "OPEN" THEN DO
                InsErrorFlag = 1
                RETURN 0
        END
 
        Success = 0
        i = 1
        DO WHILE (Success = 0) & (i <= InsConfigArray.0)
                /* first, remove all trailing whitespace from original line */
                InsConfigArray.i=STRIP(InsConfigArray.i, "Trailing", " ") /* Spaces */
                InsConfigArray.i=STRIP(InsConfigArray.i, "Trailing", X2C(9)) /* Tabs */
                /* make a work copy of that original */
                CopiedLine=InsConfigArray.i
                /* now, convert all tabs in our copy to single space characters */
                CopiedLine = TRANSLATE(CopiedLine, " ", X2C(9)) 
 
                IF (TargetPathName <> "LIBPATH") THEN DO        /* search for SET */
                        IF WORD(CopiedLine, 1) = "SET" THEN
                                Expression = SUBWORD(CopiedLine, 2)
                        ELSE
                                Expression = ""
                END
                ELSE
                        Expression = CopiedLine                 /* no SET for LIBPATH */

                Expression = SPACE(Expression, 0)               /* strip blanks */
                PARSE UPPER VAR Expression Symbol "=" Value     /* split expression */
                IF Symbol = TargetPathName THEN DO              /* found right symbol? */
                        IF RIGHT(Value, 1) <> ";" THEN DO
                                Value = Value";"                /* add ; if necessary */
                                InsConfigArray.i = InsConfigArray.i";" /* can't be bad */
                        END

                        IF POS(Path, Value) = 0 THEN            /* already there? */
                                InsConfigArray.i = InsConfigArray.i""Path /* no, append */
                        
                        Success = 1                             /* now it has to be there */
                END
                i = i + 1
        END

        IF Success = 0 THEN             /* we didn't find our symbol? */
                CALL InsConfigAppend "SET "TargetPathName"="Path        /* then create it */
RETURN Success
 
/* InsConfigClose */
/* This function will write InsConfigArray to a temporary file, copy that file
   over config.sys and delete the temporary file afterwards - affects InsErrorFlag,
   InsRestartRequired */
InsConfigClose: PROCEDURE EXPOSE InsErrorFlag InsRestartRequired InsConfigArray. InsConfigArrayStatus
        
        BootDrive = InsFormatPath(InsGetBootDrive())    /* get boot drive */

        IF InsConfigArrayStatus <> "OPEN" THEN DO
                InsErrorFlag = 1
                RETURN 0
        END

        /* create a temporary file */
        TempFile=SysTempFileName(BootDrive"INST????.TMP")
        IF TempFile = "" THEN 
                TempFile = "."
        IF (LEFT(STREAM(TempFile, "C", "OPEN WRITE"), 5) <> "READY") THEN DO
                InsErrorFlag = 1
                RETURN 0
        END

        /* write InsConfigArray to temporary file */
        i = 1
        DO InsConfigArray.0
                IF LINEOUT(TempFile, InsConfigArray.i) <> 0 THEN
                        LEAVE
                i = i + 1
        END

        /* check for errors, close temporary file */
        IF (i <> (InsConfigArray.0 + 1)) |,
           (LEFT(STREAM(TempFile, "C", "CLOSE"), 5) <> "READY") THEN DO                      
                CALL SysFileDelete TempFile                
                InsErrorFlag = 1
                RETURN 0
        END

        /* copy temporary file over config.sys */
        IF (InsFileCopy(TempFile, BootDrive"CONFIG.SYS") = 0) THEN DO
                CALL SysFileDelete TempFile
                InsErrorFlag = 1
                RETURN 0
        END

        CALL SysFileDelete TempFile     /* delete temporary file */
        InsRestartRequired = 1          /* set flag */
        DROP InsConfigArrayStatus
RETURN 1

/* ------------------------ disk I/O related procedures ---------------------- */

/* InsCheckSpace <Path>, <SpaceRequired> */
/* checks if free drive space meets requirements - affects InsErrorFlag	*/
/* can also be used to check if a drive is accessible w/o nasty popups 	*/
InsCheckSpace: PROCEDURE EXPOSE InsErrorFlag 
	PARSE ARG Path, SpaceRequired
        freespace = ""
        DriveSpec = FILESPEC("Drive", InsFormatPath(Path))
        IF DriveSpec <> "" THEN
                freespace = SUBWORD(SysDriveInfo(DriveSpec), 2, 1)
	IF (freespace <> "") & (freespace >= SpaceRequired) THEN
                RETURN 1
        ELSE DO
                InsErrorFlag = 1
                RETURN 0
        END
RETURN

/* InsFileExists <filename> */
/* checks if specified file exists - affects InsErrorFlag */
InsFileExists: PROCEDURE EXPOSE InsErrorFlag
        PARSE ARG filename
        IF (STREAM(filename, "c", "QUERY EXISTS") <> "") THEN
                RETURN 1
        ELSE DO
		InsErrorFlag = 1
                RETURN 0
	END
RETURN      

/* InsFormatPath <Path> */
/* Checks Path for proper format, tries to correct if possible. returns null
   string if Path's format cannot be corrected. Does not affect InsErrorFlag. */
InsFormatPath: PROCEDURE        
        PARSE UPPER ARG Path

        /* Following code is rather ugly, but it works OK. The same effect
        could probably be achieved in a more elegant way using PARSE. */

        Path = SPACE(Path, 0)                   /* remove all blanks */
        IF LENGTH(Path) = 1 THEN                /* single letter? probably a drive */
		Path = Path":\"

        /* no full device names like com: allowed, just drive letters */
        IF (POS(":", Path) <> 0) & (POS(":", Path) > 2) THEN
                Path = DELSTR(Path, 1, POS(":", Path)-2)
        ELSE IF POS(":", Path) = 1 THEN         /* starting with ":" ? */
                RETURN ""                       /* yes, give up */
        /* ensure proper beginning of path (only "\" or A..Z allowed) */
        IF (VERIFY(LEFT(Path, 1), XRANGE("A", "Z")) <> 0) & (LEFT(Path, 1) <> "\") THEN
                RETURN ""                       /* not OK, give up */
        
	IF (RIGHT(Path, 2) <> ":\") & (RIGHT(Path, 1) = "\") THEN       /* trailing "\" ? */
		Path = DELSTR(Path, LENGTH(Path))
        ELSE IF RIGHT(Path, 1) = ":" THEN                               /* drive only? */
                Path = Path"\"                                          /* append "\" */
RETURN Path

/* InsFormatFileSpec <FileSpec> */
/* Checks FileSpec (may include full path spec.) for proper format, tries to
   correct if possible. Mainly used to eliminate double backslashes caused
   by merging of strings. */
InsFormatFileSpec: PROCEDURE
        PARSE ARG FileSpec
 
        FileSpec = SPACE(FileSpec, 0)                   /* strip blanks */
        
        DO WHILE POS("\\", FileSpec) <> 0               /* strip "\\" */
                FileSpec=DELSTR(FileSpec, POS("\\", FileSpec), 1)
        END
 
RETURN FileSpec
       
 
/* InsCheckPath <Path> */
/* check if specified path is valid and accessible - affects InsErrorFlag */      
InsCheckPath: PROCEDURE EXPOSE InsErrorFlag        
        PARSE ARG Path
	
	Path = InsFormatPath(Path)		/* format path (just to make sure) */

	/* check if drive is accessible */
	IF InsCheckSpace(Path, 0) = 0 THEN DO
		InsErrorFlag = 1
		RETURN 0
	END

	/* check if directory is accessible */
        CurrentDir = DIRECTORY()
        IF DIRECTORY(Path) = Path THEN DO
                CALL DIRECTORY CurrentDir
                RETURN 1
        END
        ELSE DO
                InsErrorFlag = 1
                RETURN 0
        END
RETURN

/* InsCreateProgramObject <ObjectTitle>, <ExeFile> [,WorkPath] [,Parameters] */
/* creates a WPProgram object on the desktop pointing to ExeFile
   affects InsErrorFlag, ProgramObjectMade */
InsCreateProgramObject: PROCEDURE EXPOSE InsErrorFlag InsProgramObjectMade
        PARSE ARG ObjectTitle, ExeFile, WorkPath, Parameters

        /* build a proper WinCreateObject setup string */
        SetupString = "EXENAME="ExeFile";STARTUPDIR="WorkPath";PARAMETERS="Parameters";"

        IF SysCreateObject("WPProgram", ObjectTitle, "<WP_DESKTOP>", SetupString, "REPLACE") = 1 THEN DO
                InsProgramObjectMade = 1
                RETURN 1
        END
        ELSE DO
                ProgramObjectMade = 0
                InsErrorFlag = 1
                CALL SysDestroyObject ObjectTitle
                RETURN 0
        END
RETURN

/* InsGetBootDrive */
/* returns OS/2 boot drive - affects InsErrorFlag */
InsGetBootDrive: PROCEDURE EXPOSE InsErrorFlag
        /* get bootdrive by scanning PATH for \os2\system directory - this
           was taken from the OS/2 Programmers Toolkit 2.1 installation procedure */
        BootDrive =  SUBSTR(TRANSLATE(VALUE('PATH',,'OS2ENVIRONMENT')),,
                        POS('\OS2\SYSTEM',TRANSLATE(VALUE('PATH',,'OS2ENVIRONMENT')))-2,2)

        IF BootDrive = "" THEN
                InsErrorFlag = 1        
RETURN BootDrive

/* InsFileCopy <Source>, <Target> */
/* this function interfaces to the COPY command - affects InsErrorFlag */
InsFileCopy: PROCEDURE EXPOSE InsErrorFlag
        PARSE ARG Source, Target
        Source = InsFormatFileSpec(Source)
        Target = InsFormatFileSpec(Target)
        CopyCall = "COPY "Source" "Target" /V 1> NUL 2> NUL"

        ADDRESS CMD CopyCall
        IF RC <> 0 THEN DO
                InsErrorFlag = 1
                RETURN 0
        END
RETURN 1

/* InsMakeDirectory <Path> */
/* This procedure works like the MD command. Affects InErrorFlag */
InsMakeDirectory: PROCEDURE EXPOSE InsErrorFlag
        PARSE ARG Path
        Path = InsFormatFileSpec(Path);  /* eliminate double \\ */
        IF SysMkDir(InsFormatPath(Path)) <> 0 THEN DO
                InsErrorFlag = 1
                RETURN 0
        END
RETURN 1


/* ------------------------- miscellaneous procedures ------------------------ */


/* InsCleanString <String> */
/* replaces all control characters (0x00-0x1F) in String w/ space characters (0x20) */
InsCleanString: PROCEDURE
        PARSE ARG String
        String = TRANSLATE(String, " ", XRANGE("00"x,"1F"x))
RETURN String

/* InsCheckOS <OSRequired> */
/* checks if OS version meets requirements - affects InsErrorFlag */
InsCheckOS: PROCEDURE EXPOSE InsErrorFlag
	PARSE ARG OSRequired
	IF SysOS2Ver() >= OSRequired THEN
		RETURN 1
	ELSE DO
		InsErrorFlag = 1
		RETURN 0
	END
RETURN

/* InsHaltTrap */
/* called upon SIGINT (ctrl-break) - see InsExit() */
InsHaltTrap: PROCEDURE
        CALL InsExit 1, "The installation process has been aborted prematurely on your request."
RETURN

/* InsAbort */
/* use if something fatal happened - see InsExit() */
InsAbort: PROCEDURE
	CALL InsExit 1, "The installation process has been aborted prematurely."
RETURN

/* InsExit [ErrorLevel], [Message] */
/* regular exit procedure - blanks screen, displays message, exits program */
InsExit: PROCEDURE
        PARSE ARG ErrorLevel, Message
        IF ErrorLevel="" THEN
                ErrorLevel = 0
        SAY  "[0m"    /* ANSI: all attributes off */
        CALL SysCls
        SAY
        SAY Message
        EXIT ErrorLevel
RETURN

/* --------------------------------- END OF FILE ----------------------------- */
