/*
 *  Rexx/Curses
 *  Copyright (C) 1998  Mark Hessling <M.Hessling@qut.edu.au>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define RXCURSES_VERSION "1.0.0"
#define RXCURSES_DATE "2 May 1998"

#define INCL_RXSHV    /* Shared variable support */
#define INCL_RXFUNC   /* External functions support */

#include "rxpack.h"

/*
 * Because Rexx interpreters include <windows.h> on Windows platforms
 * and it defines MOUSE_MOVED, we have to undef it here to avoid
 * conflict with the MOUSE_MOVED #define in curses.h
 */
#undef MOUSE_MOVED

#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_TERMIOS_H
# include <termios.h>
#endif
#include <assert.h>
#if defined(USE_XCURSES)
# define XCURSES 1
# include "xcurses.h"
char *XCursesProgramName="rxcurses";
#else
# include <curses.h>
#endif


#if defined(__PDCURSES__) && !defined(XCURSES)
# define HAVE_CURS_SET
# define HAVE_NAPMS
# define HAVE_KEYPAD
# define HAVE_KEYNAME
#endif

char *RxPackageName = "rxcurses";

#ifndef HAVE_KEYNAME
typedef struct 
{
   char *keyname;
   int keynumber;
} KeyName;

static KeyName RexxKeyNames[] = {
   { "KEY_BREAK",     KEY_BREAK },
   { "KEY_DOWN",      KEY_DOWN  },
   { "KEY_UP",        KEY_UP    },
   { "KEY_LEFT",      KEY_LEFT  },
   { "KEY_RIGHT",     KEY_RIGHT },
   { "KEY_HOME",      KEY_HOME  },
   { "KEY_BACKSPACE", KEY_BACKSPACE  },
   { "KEY_F(0)",      KEY_F0    },
   { "KEY_F(1)",      KEY_F0+1  },
   { "KEY_F(2)",      KEY_F0+2  },
   { "KEY_F(3)",      KEY_F0+3  },
   { "KEY_F(4)",      KEY_F0+4  },
   { "KEY_F(5)",      KEY_F0+5  },
   { "KEY_F(6)",      KEY_F0+6  },
   { "KEY_F(7)",      KEY_F0+7  },
   { "KEY_F(8)",      KEY_F0+8  },
   { "KEY_F(9)",      KEY_F0+9  },
   { "KEY_F(10)",     KEY_F0+10 },
   { "KEY_F(11)",     KEY_F0+11 },
   { "KEY_F(12)",     KEY_F0+12 },
   { "KEY_F(13)",     KEY_F0+13 },
   { "KEY_F(14)",     KEY_F0+14 },
   { "KEY_F(15)",     KEY_F0+15 },
   { "KEY_F(16)",     KEY_F0+16 },
   { "KEY_F(17)",     KEY_F0+17 },
   { "KEY_F(18)",     KEY_F0+18 },
   { "KEY_F(19)",     KEY_F0+19 },
   { "KEY_F(20)",     KEY_F0+20 },
   { "KEY_F(21)",     KEY_F0+21 },
   { "KEY_F(22)",     KEY_F0+22 },
   { "KEY_F(23)",     KEY_F0+23 },
   { "KEY_F(24)",     KEY_F0+24 },
   { "KEY_F(25)",     KEY_F0+25 },
   { "KEY_F(26)",     KEY_F0+26 },
   { "KEY_F(27)",     KEY_F0+27 },
   { "KEY_F(28)",     KEY_F0+28 },
   { "KEY_F(29)",     KEY_F0+29 },
   { "KEY_F(30)",     KEY_F0+30 },
   { "KEY_F(31)",     KEY_F0+31 },
   { "KEY_F(32)",     KEY_F0+32 },
   { "KEY_F(33)",     KEY_F0+33 },
   { "KEY_F(34)",     KEY_F0+34 },
   { "KEY_F(35)",     KEY_F0+35 },
   { "KEY_F(36)",     KEY_F0+36 },
   { "KEY_F(37)",     KEY_F0+37 },
   { "KEY_F(38)",     KEY_F0+38 },
   { "KEY_F(39)",     KEY_F0+39 },
   { "KEY_F(40)",     KEY_F0+40 },
   { "KEY_F(41)",     KEY_F0+41 },
   { "KEY_F(42)",     KEY_F0+42 },
   { "KEY_F(43)",     KEY_F0+43 },
   { "KEY_F(44)",     KEY_F0+44 },
   { "KEY_F(45)",     KEY_F0+45 },
   { "KEY_F(46)",     KEY_F0+46 },
   { "KEY_F(47)",     KEY_F0+47 },
   { "KEY_F(48)",     KEY_F0+48 },
   { "KEY_F(49)",     KEY_F0+49 },
   { "KEY_F(50)",     KEY_F0+50 },
   { "KEY_F(51)",     KEY_F0+51 },
   { "KEY_F(52)",     KEY_F0+52 },
   { "KEY_F(53)",     KEY_F0+53 },
   { "KEY_F(54)",     KEY_F0+54 },
   { "KEY_F(55)",     KEY_F0+55 },
   { "KEY_F(56)",     KEY_F0+56 },
   { "KEY_F(57)",     KEY_F0+57 },
   { "KEY_F(58)",     KEY_F0+58 },
   { "KEY_F(59)",     KEY_F0+59 },
   { "KEY_F(60)",     KEY_F0+60 },
   { "KEY_F(61)",     KEY_F0+61 },
   { "KEY_F(62)",     KEY_F0+62 },
   { "KEY_F(63)",     KEY_F0+63 },
   { "KEY_DL",        KEY_DL    },
   { "KEY_IL",        KEY_IL    },
   { "KEY_DC",        KEY_DC    },
   { "KEY_IC",        KEY_IC    },
   { "KEY_EIC",       KEY_EIC   },
   { "KEY_CLEAR",     KEY_CLEAR },
   { "KEY_EOS",       KEY_EOS   },
   { "KEY_EOL",       KEY_EOL   },
   { "KEY_SF",        KEY_SF    },
   { "KEY_SR",        KEY_SR    },
   { "KEY_NPAGE",     KEY_NPAGE },
   { "KEY_PPAGE",     KEY_PPAGE },
   { "KEY_STAB",      KEY_STAB  },
   { "KEY_CTAB",      KEY_CTAB  },
   { "KEY_CATAB",     KEY_CATAB },
   { "KEY_ENTER",     KEY_ENTER },
   { "KEY_SRESET",    KEY_SRESET},
   { "KEY_RESET",     KEY_RESET },
   { "KEY_PRINT",     KEY_PRINT },
   { "KEY_LL",        KEY_LL    },
   { "KEY_A1",        KEY_A1    },
   { "KEY_A3",        KEY_A3    },
   { "KEY_B2",        KEY_B2    },
   { "KEY_C1",        KEY_C1    },
   { "KEY_C3",        KEY_C3    },
   { "KEY_BTAB",      KEY_BTAB  },
   { NULL,            0         },
};
#endif

RexxFunctionHandler CurLoadFuncs;
RexxFunctionHandler CurDropFuncs;
RexxFunctionHandler CurVariable;
RexxFunctionHandler CurAddch;
RexxFunctionHandler CurWaddch;
RexxFunctionHandler CurAddstr;
RexxFunctionHandler CurWaddstr;
RexxFunctionHandler CurBox;
RexxFunctionHandler CurCbreak;
RexxFunctionHandler CurNocbreak;
RexxFunctionHandler CurClear;
RexxFunctionHandler CurClearok;
RexxFunctionHandler CurClrtobot;
RexxFunctionHandler CurClrtoeol;
RexxFunctionHandler CurWclrtobot;
RexxFunctionHandler CurWclrtoeol;
RexxFunctionHandler CurCrmode;
RexxFunctionHandler CurCursSet;
RexxFunctionHandler CurDelch;
RexxFunctionHandler CurMvdelch;
RexxFunctionHandler CurWdelch;
RexxFunctionHandler CurMvwdelch;
RexxFunctionHandler CurDeleteln;
RexxFunctionHandler CurWdeleteln;
RexxFunctionHandler CurDelwin;
RexxFunctionHandler CurEcho;
RexxFunctionHandler CurEndwin;
RexxFunctionHandler CurErase;
RexxFunctionHandler CurGetch;
RexxFunctionHandler CurWgetch;
RexxFunctionHandler CurGetstr;
RexxFunctionHandler CurGetyx;
RexxFunctionHandler CurInch;
RexxFunctionHandler CurWinch;
RexxFunctionHandler CurInitscr;
RexxFunctionHandler CurInsch;
RexxFunctionHandler CurInsertln;
RexxFunctionHandler CurKeypad;
RexxFunctionHandler CurMove;
RexxFunctionHandler CurWmove;
RexxFunctionHandler CurNapms;
RexxFunctionHandler CurNewwin;
RexxFunctionHandler CurNl;
RexxFunctionHandler CurNocrmode;
RexxFunctionHandler CurNoecho;
RexxFunctionHandler CurNoraw;
RexxFunctionHandler CurPrintw;
RexxFunctionHandler CurRaw;
RexxFunctionHandler CurRefresh;
RexxFunctionHandler CurWrefresh;
RexxFunctionHandler CurScanw;
RexxFunctionHandler CurStandend;
RexxFunctionHandler CurStandout;
RexxFunctionHandler CurAttrset;
RexxFunctionHandler CurWattrset;
RexxFunctionHandler CurAttron;
RexxFunctionHandler CurWattron;
RexxFunctionHandler CurAttroff;
RexxFunctionHandler CurWattroff;
RexxFunctionHandler CurStartColor;
RexxFunctionHandler CurHasColors;
RexxFunctionHandler CurHasAcs;
RexxFunctionHandler CurInitPair;

typedef struct cur_window *winptr ;
typedef struct cur_window 
{
   WINDOW *win ;
   struct cur_window *prev, *next ;
} cur_window_type ;

winptr first=NULL, last=NULL ;

/*-----------------------------------------------------------------------------
 * Table of CURSES Functions. Used to install/de-install functions.
 * If you change this table, don't forget to change the table at the end
 * of this file.
 *----------------------------------------------------------------------------*/
RexxFunction RxPackageFunctions[] = {
   { "CURADDCH",        CurAddch       ,"CurAddch"        },
   { "CURWADDCH",       CurWaddch      ,"CurWaddch"       },
   { "CURADDSTR",       CurAddstr      ,"CurAddstr"       },
   { "CURWADDSTR",      CurWaddstr     ,"CurWaddstr"      },
   { "CURBOX",          CurBox         ,"CurBox"          },
   { "CURCLEAR",        CurClear       ,"CurClear"        },
   { "CURCBREAK",       CurCbreak      ,"CurCbreak"       },
   { "CURNOCBREAK",     CurNocbreak    ,"CurNocbreak"     },
   { "CURCLEAROK",      CurClearok     ,"CurClearok"      },
   { "CURCLRTOBOT",     CurClrtobot    ,"CurClrtobot"     },
   { "CURCLRTOEOL",     CurClrtoeol    ,"CurClrtoeol"     },
   { "CURWCLRTOBOT",    CurWclrtobot   ,"CurWclrtobot"    },
   { "CURWCLRTOEOL",    CurWclrtoeol   ,"CurWclrtoeol"    },
   { "CURCRMODE",       CurCrmode      ,"CurCrmode"       },
   { "CURCURSSET",      CurCursSet     ,"CurCursSet"      },
   { "CURDELCH",        CurDelch       ,"CurDelch"        },
   { "CURMVDELCH",      CurMvdelch     ,"CurMvdelch"      },
   { "CURWDELCH",       CurWdelch      ,"CurWdelch"       },
   { "CURMVWDELCH",     CurMvwdelch    ,"CurMvwdelch"     },
   { "CURDELETELN",     CurDeleteln    ,"CurDeleteln"     },
   { "CURDELWIN",       CurDelwin      ,"CurDelwin"       },
   { "CURECHO",         CurEcho        ,"CurEcho"         },
   { "CURENDWIN",       CurEndwin      ,"CurEndwin"       },
   { "CURERASE",        CurErase       ,"CurErase"        },
   { "CURGETCH",        CurGetch       ,"CurGetch"        },
   { "CURWGETCH",       CurWgetch      ,"CurWgetch"       },
   { "CURGETSTR",       CurGetstr      ,"CurGetstr"       },
   { "CURGETYX",        CurGetyx       ,"CurGetyx"        },
   { "CURINCH",         CurInch        ,"CurInch"         },
   { "CURWINCH",        CurWinch       ,"CurWinch"        },
   { "CURINITSCR",      CurInitscr     ,"CurInitscr"      },
   { "CURINSCH",        CurInsch       ,"CurInsch"        },
   { "CURINSERTLN",     CurInsertln    ,"CurInsertln"     },
   { "CURKEYPAD",       CurKeypad      ,"CurKeypad"       },
   { "CURMOVE",         CurMove        ,"CurMove"         },
   { "CURWMOVE",        CurWmove       ,"CurWmove"        },
   { "CURNEWWIN",       CurNewwin      ,"CurNewwin"       },
   { "CURNL",           CurNl          ,"CurNl"           },
   { "CURNAPMS",        CurNapms       ,"CurNapms"        },
   { "CURNOCRMODE",     CurNocrmode    ,"CurNocrmode"     },
   { "CURNOECHO",       CurNoecho      ,"CurNoecho"       },
   { "CURNORAW",        CurNoraw       ,"CurNoraw"        },
   { "CURPRINTW",       CurPrintw      ,"CurPrintw"       },
   { "CURRAW",          CurRaw         ,"CurRaw"          },
   { "CURREFRESH",      CurRefresh     ,"CurRefresh"      },
   { "CURWREFRESH",     CurWrefresh    ,"CurWrefresh"     },
   { "CURSCANW",        CurScanw       ,"CurScanw"        },
   { "CURSTANDEND",     CurStandend    ,"CurStandend"     },
   { "CURSTANDOUT",     CurStandout    ,"CurStandout"     },
   { "CURATTRSET",      CurAttrset     ,"CurAttrset"      },
   { "CURWATTRSET",     CurWattrset    ,"CurWattrset"     },
   { "CURHASCOLORS",    CurHasColors   ,"CurHasColors"    },
   { "CURHASACS",       CurHasAcs      ,"CurHasAcs"       },
   { "CURSTARTCOLOR",   CurStartColor  ,"CurStartColor"   },
   { "CURINITPAIR",     CurInitPair    ,"CurInitPair"     },
   { "CURLOADFUNCS",    CurLoadFuncs   ,"CurLoadFuncs"    },
   { "CURDROPFUNCS",    CurDropFuncs   ,"CurDropFuncs"    },
   { "CURVARIABLE",     CurVariable    ,"CurVariable"     },
   { NULL, NULL, NULL }
};

typedef struct 
{
   char *name;
   long number;
} String2Number;

#ifdef A_COLOR

static String2Number ColorName[] =
{
   { "COLOR_BLACK",   COLOR_BLACK   },
   { "COLOR_BLUE",    COLOR_BLUE    },
   { "COLOR_RED",     COLOR_RED     },
   { "COLOR_CYAN",    COLOR_CYAN    },
   { "COLOR_MAGENTA", COLOR_MAGENTA },
   { "COLOR_GREEN",   COLOR_GREEN   },
   { "COLOR_YELLOW",  COLOR_YELLOW  },
   { "COLOR_WHITE",   COLOR_WHITE   },
   { NULL, 0 }
};
#endif

static String2Number AttrName[] =
{
   { "A_NORMAL",      A_NORMAL      },
   { "A_STANDOUT",    A_STANDOUT    },
   { "A_BOLD",        A_BOLD        },
   { "A_UNDERLINE",   A_UNDERLINE   },
   { "A_REVERSE",     A_REVERSE     },
#ifdef A_BLINK
   { "A_BLINK",       A_BLINK       },
#endif
#ifdef A_DIM
   { "A_DIM",         A_DIM         },
#endif
#ifdef A_INVIS
   { "A_INVIS",       A_INVIS       },
#endif
#ifdef A_PROTECT
   { "A_PROTECT",     A_PROTECT     },
#endif
   { NULL, 0 }
};


#ifndef HAVE_KEYNAME
char *mykeyname
    (int key)
{
   register int i=0;
    
   for (i = 0 ; ; i++ )
   {
      if (RexxKeyNames[i].keyname == NULL)
         return("NO NAME");
      if (RexxKeyNames[i].keynumber == key)
         return(RexxKeyNames[i].keyname);
   }
   return("NO NAME");
}
#endif

static int my_checkparam(int argc, int mini, int maxi)
{
 if (argc < mini
 ||  argc > maxi)
   return 1;
 else
   return 0;
}

#ifdef A_ALTCHARSET
static long acs_to_number( char *str, int len )
{
   /*
    * This code is horrible, but because of the way most acs character
    * sets are defined, this is the only way.
    */
#ifdef ACS_BLOCK
   if ( memcmpi( "ACS_BLOCK", str, len ) == 0 ) return ( ACS_BLOCK );
#endif
#ifdef ACS_DIAMOND
   if ( memcmpi( "ACS_DIAMOND", str, len ) == 0 ) return ( ACS_DIAMOND );
#endif
#ifdef ACS_CKBOARD
   if ( memcmpi( "ACS_CKBOARD", str, len ) == 0 ) return ( ACS_CKBOARD );
#endif
   return(0);
}
#endif


#ifdef A_COLOR
static int color_to_number( char *str, int len )
{
   register int i = 0;

   for ( i = 0; ColorName[i].name != NULL; i++ )
   {
      if ( memcmpi( str, ColorName[i].name, len ) == 0 )
         return ( int )ColorName[i].number;
   }
   return ( -1 );
}

static long color_pair_to_number( char *str, int len )
{
   int num = 0;

   if (len < 13 )
      return(-1);
   if ( strncmp( str, "COLOR_PAIR(", 11 ) != 0 )
      return(-1);
   if ( str[len-1] != ')' )
      return(-1);
   str[len-1] = '\0';
   if ( ( num = atoi( str + 11 ) ) == 0 )
      return(-1);

   return COLOR_PAIR( num );
}
#endif

static long attr_to_number( char *str, int len )
{  
   register int i = 0;

   for ( i = 0; AttrName[i].name != NULL; i++ )
   {
      if (memcmpi( AttrName[i].name, str, len ) == 0 )
         return AttrName[i].number;
   }
   return ( -1 );
}

static long get_attr(ULONG argc, RXSTRING argv[], int start, int end )
{
   register int i = 0;
   long tmp = 0;
   long attr = 0;

   for (i = start; i < end ; i++)
   {
      if ( argv[i].strlength == 0 )
         break;
      if ( ( tmp = attr_to_number( (char *)argv[i].strptr, argv[i].strlength ) ) == (-1) )
      {
#ifdef A_COLOR
         if ( ( tmp = color_pair_to_number ( (char *)argv[i].strptr, argv[i].strlength ) ) == (-1) )
            return(-1);
         else
            attr |= tmp;
#else
         return(-1);
#endif
      }
      else
         attr |= tmp;
   }
   return attr;
}


static WINDOW *getwindow( long index )
{
   winptr ptr ;
   
   for (ptr=first; ptr; ptr=ptr->next )
      if ((long)ptr->win == index)
         break ;

   if (!ptr)
   {
/*      fprintf( stderr, "Curses: Unknown window name: %s\n", name->value) ; */
      return NULL ;
   }
   return ptr->win ;
}


static void dropwindows()
{
   winptr ptr=first, nptr=first ;
 
   while(ptr)
   {
      nptr = ptr->next ;
      free( ptr ) ;
      ptr = nptr;
      if (!ptr)
         break;
   }
}

static void delwindow( long index )
{
   winptr ptr=first, nptr=first ;

   for (ptr=first; ptr; ptr=ptr->next )
   {
      if ((long)ptr->win == index)
      {
         if (ptr == first)
         {
            if (ptr == last)
            {
               first = last = NULL;
               free( ptr );
               return;
            }
            first = ptr->next;
            first->prev = NULL;
            free (ptr );
            return;
         }
         else if (ptr == last)
         {
            last = ptr->prev;
            last->next = NULL;
            free( ptr );
            return;
         }
         else
         {
            nptr = ptr;
            ptr->prev->next = ptr->next;
            ptr->next->prev = ptr->prev;
            free( nptr );
            return;
         }
      }
      break ;
   }
}

#if OLD
static void addwindow( WINDOW *win, char *name )
#else
static void addwindow( WINDOW *win )
#endif
{
   winptr ptr ;

   ptr = malloc( sizeof(struct cur_window) ) ;
#if OLD
   ptr->name.strptr = (char *)malloc(strlen(name)+1);
   strcpy(ptr->name.strptr,name) ;
#endif
   ptr->win = win ;
   ptr->next = NULL ;
   ptr->prev = last ;

   if (last)
      ptr->prev->next = last = ptr ;

   if (!first)
      first = last = ptr ;
}


/* here */
/*====== Here come the real interface functions to curses ======*/

/*====== Initiate and terminate ================================*/

RFH_RETURN_TYPE CurInitscr
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{ 
   char buf[50];

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam(argc, 0, 0 ) )
      return( 1 );
   initscr( ) ;
   sprintf( buf, "%d", LINES );
   if ( SetRexxVariable( "LINES", 5, buf, strlen( buf ) ) )
      return( 1 );
   sprintf( buf, "%d", COLS );
   if ( SetRexxVariable( "COLS", 4, buf, strlen( buf ) ) )
      return( 1 );
#ifdef A_COLOR
   if ( has_colors() )
   {
     sprintf( buf, "%d", COLORS );
     if ( SetRexxVariable( "COLORS", 6, buf, strlen( buf ) ) )
        return( 1 );
     sprintf( buf, "%d", COLOR_PAIRS );
     if ( SetRexxVariable( "COLOR_PAIRS", 11, buf, strlen( buf ) ) )
        return( 1 );
   }
#endif
   addwindow( stdscr ) ;
   addwindow( curscr ) ;
   sprintf( buf, "%ld", (long)stdscr );
   if ( SetRexxVariable( "STDSCR", 6, buf, strlen( buf ) ) )
      return( 1 );
   sprintf( buf, "%ld", (long)curscr );
   if ( SetRexxVariable( "CURSCR", 6, buf, strlen( buf ) ) )
      return( 1 );
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurEndwin
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
   /*
    * In case we have called CurEndwin BEFORE CurInitscr, don't call
    * endwin()
    */
   if ( first != NULL )
      endwin( ) ;
   dropwindows( ) ;
#if defined(XCURSES)
   XCursesExit( );
#endif
   return RxReturnString( retstr, "" ) ;
}


/*====== Cursor movements ======================================*/
   
RFH_RETURN_TYPE CurMove
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   int x, y ;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 2, 2 ) ) 
      return( 1 );
   argv[0].strptr[argv[0].strlength] = '\0';
   argv[1].strptr[argv[1].strlength] = '\0';
   y = atoi( (char *)argv[0].strptr ) ;
   x = atoi( (char *)argv[1].strptr ) ;
   move( y, x ) ;
   return RxReturnString( retstr, "" ) ;
}
   
RFH_RETURN_TYPE CurWmove
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   WINDOW *pwin ;
   int x, y ;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 3, 3 ) ) 
      return( 1 );
   argv[1].strptr[argv[1].strlength] = '\0';
   argv[2].strptr[argv[2].strlength] = '\0';
   y = atoi( (char *)argv[1].strptr ) ;
   x = atoi( (char *)argv[2].strptr ) ;
   pwin = getwindow( atol( (char *)argv[0].strptr ) ) ;
   wmove( pwin, y, x ) ;
   return RxReturnString( retstr, "" ) ;
}


RFH_RETURN_TYPE CurGetyx 
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   WINDOW *pwin ;
   int x, y ;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 3, 3 ) ) 
      return( 1 );
   pwin = getwindow( atol( (char *)argv[0].strptr ) ) ;

   getyx( pwin, y, x ) ;
   sprintf( (char *)retstr->strptr, "%d %d", y, x ) ;
   retstr->strlength = strlen( (char *)retstr->strptr ) ;
   return 0 ;
}


RFH_RETURN_TYPE CurInch 
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
   retstr->strptr[0] = (unsigned char)inch( ) ;
   retstr->strlength = 1 ;
   return 0 ;
}

RFH_RETURN_TYPE CurWinch
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   WINDOW *pwin ;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 1, 1 ) )
      return( 1 );
   pwin = getwindow( atol( (char *)argv[0].strptr ) ) ;
   retstr->strptr[0] = (unsigned char)winch( pwin ) ;
   retstr->strlength = 1 ;
   return 0 ;
}

   
RFH_RETURN_TYPE CurNapms
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   int ms ;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 1, 1 ) ) 
      return( 1 );
   argv[0].strptr[argv[0].strlength] = '\0';
   ms = atoi( (char *)argv[0].strptr ) ;
#ifdef HAVE_NAPMS
   napms( ms ) ;
#endif
   return RxReturnString( retstr, "" ) ;
}

   
RFH_RETURN_TYPE CurCursSet
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   int flag ;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 1, 1 ) ) 
      return( 1 );
   argv[0].strptr[argv[0].strlength] = '\0';
   flag = atoi( (char *)argv[0].strptr ) ;
#ifdef HAVE_CURS_SET
   curs_set( flag ) ;
#endif
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurLeaveok 
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   WINDOW *pwin ;
   int flag ;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 2, 2 ) ) 
      return( 1 );

   pwin = getwindow( atol( (char *)argv[0].strptr ) ) ;
   flag = atoi( (char *)argv[1].strptr ) ;
   if ( flag > 1 )
      return( 1 );

   leaveok( pwin, flag ) ;
   return RxReturnString( retstr, "" ) ;
}   
   


RFH_RETURN_TYPE CurAddch
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   long attr = 0;
   long ch = 0;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 1, 6 ) ) 
      return( 1 );
   if ( ( attr = get_attr( argc, argv, 1, argc ) ) == (-1) )
      return( 1 );
#ifdef A_ALTCHARSET
   argv[0].strptr[argv[0].strlength] = '\0';
   if (argv[0].strlength == 1)
      ch = argv[0].strptr[0] ;
   else
   {
      if ( ( ch = acs_to_number( (char *)argv[0].strptr, argv[0].strlength ) ) == 0)
         return( 1 );
   }
#else
   ch = argv[0].strptr[0] ;
#endif
   addch( ch | attr ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurWaddch
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   WINDOW *pwin ;
   long attr = 0;
   long ch = 0;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 2, 7 ) ) 
      return( 1 );
   pwin = getwindow( atol( (char *)argv[0].strptr ) ) ;
   if (( attr = get_attr( argc, argv, 2, argc )) == (-1) )
      return( 1 );
#ifdef A_ALTCHARSET
   argv[1].strptr[argv[1].strlength] = '\0';
   if (argv[1].strlength == 1)
      ch = argv[1].strptr[0] ;
   else
   {
      if ( ( ch = acs_to_number( (char *)argv[1].strptr, argv[1].strlength ) ) == 0 )
         return( 1 );
   }
#else
   ch = argv[1].strptr[0] ;
#endif
   waddch( pwin, ch | attr ) ;
   return RxReturnString( retstr, "" ) ;
}
 

RFH_RETURN_TYPE CurAddstr
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 1, 1 ) ) 
      return( 1 );

   argv[0].strptr[argv[0].strlength] = '\0';
   addstr( argv[0].strptr ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurWaddstr
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   WINDOW *pwin ;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 2, 2 ) ) 
      return( 1 );
   pwin = getwindow( atol( (char *)argv[0].strptr ) ) ;

   argv[1].strptr[argv[1].strlength] = '\0';
   waddstr( pwin, argv[1].strptr ) ;
   return RxReturnString( retstr, "" ) ;
}


RFH_RETURN_TYPE CurPrintw 
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   return RxReturnString( retstr, "" ) ;
}


RFH_RETURN_TYPE CurInsch
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 1, 1 ) ) 
      return( 1 );
   insch( argv[0].strptr[0]) ;
   return RxReturnString( retstr, "" ) ;
}


RFH_RETURN_TYPE CurInsertln
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
   insertln( ) ;
   return RxReturnString( retstr, "" ) ;
}


RFH_RETURN_TYPE CurBox
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   WINDOW *pwin ;
   char hor=0, vert=0 ;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 1, 3 ) ) 
      return( 1 );
   pwin = getwindow( atol( (char *)argv[0].strptr ) ) ;
   switch(argc)
   {
      case 3:
         if ( argv[2].strlength > 1 ) return( 1 );
         if ( argv[2].strlength == 1 )
            hor = argv[2].strptr[0];
         /* fall through */
      case 2:
         if ( argv[1].strlength > 1 ) return( 1 );
         if ( argv[1].strlength == 1 )
            vert = argv[1].strptr[0];
         break;
      default:
         break;
   }

   box( pwin, vert, hor ) ;
   return RxReturnString( retstr, "" ) ;
}


RFH_RETURN_TYPE CurRefresh
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
   refresh() ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurWrefresh
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   WINDOW *pwin ;
 
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 1, 1 ) ) 
      return( 1 );

   pwin = getwindow( atol( (char *)argv[0].strptr ) ) ;
   wrefresh( pwin ) ;
   return RxReturnString( retstr, "" ) ;
}


RFH_RETURN_TYPE CurKeypad
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   int flag = 0;
   WINDOW *pwin ;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 2, 2 ) ) 
      return( 1 );
   pwin = getwindow( atol( (char *)argv[0].strptr ) ) ;
   flag = atoi( (char *)argv[1].strptr );
   if ( flag != 0 &&  flag != 1 ) 
      return( 1 );
#ifdef HAVE_KEYPAD
   keypad( pwin, flag );
#endif
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurGetch 
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   int key = 0;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
   key = getch( );
#ifdef KEY_MIN
   if ( key >= KEY_MIN )
#else
   if ( key >= 256 )
#endif
   {
#ifdef HAVE_KEYNAME
      retstr->strptr = (RXSTRING_STRPTR_TYPE)keyname( key );
#else
      retstr->strptr = (RXSTRING_STRPTR_TYPE)mykeyname( key );
#endif
      retstr->strlength = strlen( (char *)retstr->strptr );
   }
   else
   {
      retstr->strptr[0] = key;
      retstr->strlength = 1 ;
   }
   return 0;
}


RFH_RETURN_TYPE CurWgetch
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   int key = 0;
   WINDOW *pwin ;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 1, 1 ) ) 
      return( 1 );
   pwin = getwindow( atol( (char *)argv[0].strptr ) ) ;
   key = wgetch(pwin);
#ifdef KEY_MIN
   if ( key >= KEY_MIN )
#else
   if ( key >= 256 )
#endif
   {
#ifdef HAVE_KEYNAME
      retstr->strptr = (RXSTRING_STRPTR_TYPE)keyname( key );
#else
      retstr->strptr = (RXSTRING_STRPTR_TYPE)mykeyname( key );
#endif
      retstr->strlength = strlen( (char *)retstr->strptr );
   }
   else
   {
      retstr->strptr[0] = key;
      retstr->strlength = 1 ;
   }
   return 0;
}


RFH_RETURN_TYPE CurGetstr 
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   int i ;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
   i = getstr( retstr->strptr );
   retstr->strlength = i;
   return 0;
}


RFH_RETURN_TYPE CurScanw
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   return RxReturnString( retstr, "" ) ;
}


RFH_RETURN_TYPE CurCrmode
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
#if defined(XCURSES)
#else
   crmode( ) ;
#endif
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurNocrmode
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
#if defined(XCURSES)
#else
   nocrmode( ) ;
#endif
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurCbreak
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
   cbreak( );
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurNocbreak
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) )
      return( 1 );
   nocbreak( );
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurRaw
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) )
      return( 1 );
   raw( ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurNoraw
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) )
      return( 1 );
   noraw( ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurEcho
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) )
      return( 1 );
   echo( ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurNoecho
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) )
      return( 1 );
   noecho( ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurNl
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
   nl( ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurNonl
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) )
      return( 1 );
   nonl( ) ;
   return RxReturnString( retstr, "" ) ;
}


RFH_RETURN_TYPE CurDelch
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
   delch( ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurMvdelch
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   int y, x;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 2, 2 ) ) 
      return( 1 );
   y = atoi( (char *)argv[0].strptr ) ;
   x = atoi( (char *)argv[1].strptr ) ;
   mvdelch( y, x ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurWdelch
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   WINDOW *pwin ;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 1, 1 ) ) 
      return( 1 );
   pwin = getwindow( atol( (char *)argv[0].strptr ) ) ;
   wdelch( pwin ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurMvwdelch
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   WINDOW *pwin ;
   int y, x;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 3, 3 ) ) 
      return( 1 );
   y = atoi( (char *)argv[0].strptr ) ;
   x = atoi( (char *)argv[1].strptr ) ;
   pwin = getwindow( atol( (char *)argv[2].strptr ) ) ;
   mvwdelch( pwin, y, x ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurDeleteln
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
   deleteln( ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurWdeleteln
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   WINDOW *pwin ;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 1, 1 ) ) return( 1 );
   pwin = getwindow( atol( (char *)argv[0].strptr ) ) ;
   wdeleteln( pwin ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurClrtoeol
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
   clrtoeol( ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurWclrtoeol
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   WINDOW *pwin ;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 1, 1 ) ) 
      return( 1 );
   pwin = getwindow( atol( (char *)argv[0].strptr ) ) ;
   wclrtoeol( pwin ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurClrtobot
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
   clrtobot( ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurWclrtobot
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   WINDOW *pwin ;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
   pwin = getwindow( atol( (char *)argv[0].strptr ) ) ;
   wclrtobot( pwin ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurErase
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
   erase( ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurClearok
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   WINDOW *pwin ;
   int flag ;
          
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 2, 2 ) ) 
      return( 1 );
   pwin = getwindow( atol( (char *)argv[0].strptr ) ) ;
   if ( argv[1].strlength != 1 )
      return( 1 );
   flag = atoi( (char *)argv[1].strptr );
   if ( flag > 1 )
      return( 1 );

   clearok( pwin, ( bool )flag ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurClear
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
   clear( ) ;
   return RxReturnString( retstr, "" ) ;
}

/*====== Routines for handling attributes ======*/

RFH_RETURN_TYPE CurStandout
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
   standout( ) ;
   return RxReturnString( retstr, "" ) ;
}


RFH_RETURN_TYPE CurStandend
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
   standend( ) ;
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurAttrset
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   long attr = 0;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 1, 5 ) ) 
      return( 1 );
   if (( attr = get_attr( argc, argv, 0, argc )) == (-1) )
      return( 1 );
   attrset( attr );
   return RxReturnString( retstr, "OK" ) ;
}

RFH_RETURN_TYPE CurWattrset
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   WINDOW *pwin ;
   long attr = 0;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 2, 6 ) ) 
      return( 1 );
   pwin = getwindow( atol( (char *)argv[0].strptr ) ) ;
   if (( attr = get_attr( argc, argv, 1, argc )) == (-1) )
      return( 1 );
   wattrset( pwin, attr );
   return RxReturnString( retstr, "OK" ) ;
}

RFH_RETURN_TYPE CurAttron
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   long attr = 0;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 1, 5 ) ) 
      return( 1 );
   if (( attr = get_attr( argc, argv, 0, argc )) == (-1) )
      return( 1 );
   attron( attr );
   return RxReturnString( retstr, "OK" ) ;
}

RFH_RETURN_TYPE CurWattron
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   WINDOW *pwin ;
   long attr = 0;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 2, 6 ) ) 
      return( 1 );
   pwin = getwindow( atol( (char *)argv[0].strptr ) ) ;
   if (( attr = get_attr( argc, argv, 1, argc )) == (-1) )
      return( 1 );
   wattron( pwin, attr );
   return RxReturnString( retstr, "OK" ) ;
}

RFH_RETURN_TYPE CurAttroff
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   long attr = 0;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 1, 5 ) ) 
      return( 1 );
   if (( attr = get_attr( argc, argv, 0, argc )) == (-1) )
      return( 1 );
   attroff( attr );
   return RxReturnString( retstr, "OK" ) ;
}

RFH_RETURN_TYPE CurWattroff
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   WINDOW *pwin ;
   long attr = 0;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 2, 6 ) ) 
      return( 1 );
   pwin = getwindow( atol( (char *)argv[0].strptr ) ) ;
   if (( attr = get_attr( argc, argv, 1, argc )) == (-1) )
      return( 1 );
   wattroff( pwin, attr );
   return RxReturnString( retstr, "OK" ) ;
}


RFH_RETURN_TYPE CurHasAcs
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   int acschars = 0;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
#ifdef ACS_BLOCK
   acschars++;
#endif
#ifdef ACS_DIAMOND
   acschars++;
#endif
#ifdef ACS_CKBOARD
   acschars++;
#endif

   if ( acschars )
      return RxReturnNumber( retstr, 1 ) ;
   else
      return RxReturnNumber( retstr, 0 ) ;
}


RFH_RETURN_TYPE CurHasColors
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
#ifdef A_COLOR
   if ( has_colors( ) )
      return RxReturnNumber( retstr, 1 ) ;
   return RxReturnNumber( retstr, 0 ) ;
#else
   return RxReturnNumber( retstr, 0 ) ;
#endif
}


RFH_RETURN_TYPE CurStartColor
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 0, 0 ) ) 
      return( 1 );
#ifdef A_COLOR
   if ( has_colors( ) )
   {
      start_color( );
      return RxReturnString( retstr, "OK" ) ;
   }
   return RxReturnString( retstr, "ERR" ) ;
#else
   return RxReturnString( retstr, "ERR" ) ;
#endif
}


RFH_RETURN_TYPE CurInitPair
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   int pairnum, fore, back;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 3, 3 ) ) 
      return( 1 );
#ifdef A_COLOR
   if ( has_colors( ) )
   {
      if ( ( fore = color_to_number( (char *)argv[1].strptr, argv[1].strlength ) ) == -1 )
         return( 1 );
      if ( ( back = color_to_number( (char *)argv[2].strptr, argv[2].strlength ) ) == -1 )
         return( 1 );
      argv[0].strptr[argv[0].strlength] = '\0';
      if ( ( pairnum = atoi( (char *)argv[0].strptr ) ) == 0 )
         return( 1 );
      init_pair( ( short )pairnum, ( short )fore, ( short )back );
      return RxReturnString( retstr, "OK" ) ;
   }
   return RxReturnString( retstr, "ERR" ) ;
#else
   return RxReturnString( retstr, "ERR" ) ;
#endif
}



/*====== Routines for handling new windows ======*/

RFH_RETURN_TYPE CurNewwin
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   WINDOW *pwin ;
   int lines, cols, begy, begx;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 4, 4 ) ) 
      return( 1 );
   lines = atoi( (char *)argv[0].strptr );
   cols  = atoi( (char *)argv[1].strptr );
   begy  = atoi( (char *)argv[2].strptr );
   begx  = atoi( (char *)argv[3].strptr );
   pwin = newwin( lines, cols, begy, begx );
   if ( pwin )
   {
      addwindow( pwin );
      return RxReturnNumber( retstr, (long)pwin );
   }
   return RxReturnString( retstr, "" ) ;
}

RFH_RETURN_TYPE CurDelwin
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   WINDOW *pwin ;
   long index;

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 1, 1 ) ) 
      return( 1 );
   index =  atol( (char *)argv[0].strptr ) ;
   pwin = getwindow( index ) ;
   delwin( pwin );
   delwindow( index );
   return RxReturnString( retstr, "" ) ;
}

/*====== Routines for handling registration of functions ======*/


RFH_RETURN_TYPE CurLoadFuncs
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   ULONG rc = 0L;

#if defined(DYNAMIC_LIBRARY)
   rc = InitRxPackage( NULL );
   if ( !QueryRxFunction( "CURINITSCR" ) )
   {
      /* 
       * Register all external functions
       */
      if ( !rc )
         rc = RegisterRxFunctions( );
   }
#endif
   return RxReturnNumber( retstr, rc );
}

RFH_RETURN_TYPE CurDropFuncs
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   ULONG rc = 0L;

   rc = DeregisterRxFunctions( );
   return RxReturnNumber( retstr, rc );
}


RFH_RETURN_TYPE CurVariable
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   ULONG rc = 0L;
   char buf[50];

   FunctionPrologue( (char *)name, argc, argv );
   if ( my_checkparam( argc, 1, 2 ) )
      return( 1 );
   if ( memcmp( "DEBUG", argv[0].strptr, argv[0].strlength ) == 0 )
   {
      if ( argc == 1 )
      {
         sprintf( buf, "%d", RxGetRunFlags( ) );
         return RxReturnString( retstr, buf );
      }
      else
      {
         RxSetRunFlags ( atoi( (char *)argv[1].strptr ) );
         return RxReturnNumber( retstr, 0 );
      }
   }
   else if ( memcmp( "VERSION", argv[0].strptr, argv[0].strlength ) == 0 )
   {
      if ( argc == 1 )
      {
         sprintf( buf, "%s %s %s", RxPackageName, RXCURSES_VERSION, RXCURSES_DATE );
         return RxReturnString( retstr, buf );
      }
      else
         return RxReturnString( retstr, "ERROR: Cannot set variable; VERSION" );
   }
   else if ( memcmp( "DEBUGFILE", argv[0].strptr, argv[0].strlength ) == 0 )
   {
      if ( argc == 1 )
         return RxReturnString( retstr, RxGetTraceFile() );
      else
      {
         rc = RxSetTraceFile( (char *)argv[1].strptr );
         return RxReturnNumber( retstr, rc );
      }
   }
   sprintf( buf, "ERROR: Invalid variable; %s", argv[0].strptr );
   return RxReturnString( retstr, buf );
}


/*
 * The following functions are used in rxpackage.c
 */

/*-----------------------------------------------------------------------------
 * Print a usage message.
 *----------------------------------------------------------------------------*/
void usage

#ifdef HAVE_PROTO
   (void)
#else
   ()
#endif

{
   (void)fprintf(stderr,
      "\nVersion: %s %s %s\n\nUsage:   %s [-h]\n         %s [-idvf<trace file>] [Rexx program name]\n\n",
      RxPackageName,
      RXCURSES_VERSION,
      RXCURSES_DATE,
      RxPackageName,
      RxPackageName);
   exit( 1 );
}

/*-----------------------------------------------------------------------------
 * Execute any initialisation
 *----------------------------------------------------------------------------*/
int InitialisePackage

#ifdef HAVE_PROTO
   ( void )
#else
   ( )
#endif

{
   return 0;
}

/*-----------------------------------------------------------------------------
 * Execute any termination
 *----------------------------------------------------------------------------*/
int TerminatePackage

#ifdef HAVE_PROTO
   ( void )
#else
   ( )
#endif

{
   return 0;
}

#if defined(USE_REXX6000)
/*
 * This function is used as the entry point for the REXX/6000
 * interpreter
 * If you change this table, don't forget to change the table at the
 * start of this file.
 */
USHORT InitFunc( RXFUNCBLOCK **FuncBlock )
{
   static RXFUNCBLOCK funcarray[] =
   {
      { "CURADDCH",        CurAddch       ,NULL },
      { "CURWADDCH",       CurWaddch      ,NULL },
      { "CURADDSTR",       CurAddstr      ,NULL },
      { "CURWADDSTR",      CurWaddstr     ,NULL },
      { "CURBOX",          CurBox         ,NULL },
      { "CURCLEAR",        CurClear       ,NULL },
      { "CURCBREAK",       CurCbreak      ,NULL },
      { "CURNOCBREAK",     CurNocbreak    ,NULL },
      { "CURCLEAROK",      CurClearok     ,NULL },
      { "CURCLRTOBOT",     CurClrtobot    ,NULL },
      { "CURCLRTOEOL",     CurClrtoeol    ,NULL },
      { "CURWCLRTOBOT",    CurWclrtobot   ,NULL },
      { "CURWCLRTOEOL",    CurWclrtoeol   ,NULL },
      { "CURCRMODE",       CurCrmode      ,NULL },
      { "CURCURSSET",      CurCursSet     ,NULL },
      { "CURDELCH",        CurDelch       ,NULL },
      { "CURMVDELCH",      CurMvdelch     ,NULL },
      { "CURWDELCH",       CurWdelch      ,NULL },
      { "CURMVWDELCH",     CurMvwdelch    ,NULL },
      { "CURDELETELN",     CurDeleteln    ,NULL },
      { "CURDELWIN",       CurDelwin      ,NULL },
      { "CURECHO",         CurEcho        ,NULL },
      { "CURENDWIN",       CurEndwin      ,NULL },
      { "CURERASE",        CurErase       ,NULL },
      { "CURGETCH",        CurGetch       ,NULL },
      { "CURWGETCH",       CurWgetch      ,NULL },
      { "CURGETSTR",       CurGetstr      ,NULL },
      { "CURGETYX",        CurGetyx       ,NULL },
      { "CURINCH",         CurInch        ,NULL },
      { "CURWINCH",        CurWinch       ,NULL },
      { "CURINITSCR",      CurInitscr     ,NULL },
      { "CURINSCH",        CurInsch       ,NULL },
      { "CURINSERTLN",     CurInsertln    ,NULL },
      { "CURKEYPAD",       CurKeypad      ,NULL },
      { "CURMOVE",         CurMove        ,NULL },
      { "CURWMOVE",        CurWmove       ,NULL },
      { "CURNEWWIN",       CurNewwin      ,NULL },
      { "CURNL",           CurNl          ,NULL },
      { "CURNAPMS",        CurNapms       ,NULL },
      { "CURNOCRMODE",     CurNocrmode    ,NULL },
      { "CURNOECHO",       CurNoecho      ,NULL },
      { "CURNORAW",        CurNoraw       ,NULL },
      { "CURPRINTW",       CurPrintw      ,NULL },
      { "CURRAW",          CurRaw         ,NULL },
      { "CURREFRESH",      CurRefresh     ,NULL },
      { "CURWREFRESH",     CurWrefresh    ,NULL },
      { "CURSCANW",        CurScanw       ,NULL },
      { "CURSTANDEND",     CurStandend    ,NULL },
      { "CURSTANDOUT",     CurStandout    ,NULL },
      { "CURATTRSET",      CurAttrset     ,NULL },
      { "CURWATTRSET",     CurWattrset    ,NULL },
      { "CURHASCOLORS",    CurHasColors   ,NULL },
      { "CURHASACS",       CurHasAcs      ,NULL },
      { "CURSTARTCOLOR",   CurStartColor  ,NULL },
      { "CURINITPAIR",     CurInitPair    ,NULL },
      { "CURLOADFUNCS",    CurLoadFuncs   ,NULL },
      { "CURDROPFUNCS",    CurDropFuncs   ,NULL },
      { "CURVARIABLE",     CurVariable    ,NULL },
      { NULL, NULL, NULL }
   } ;
   *FuncBlock = funcarray;
   return (USHORT)0;
}
#endif
