#ifndef lint
static char *RCSid = "$Id";
#endif
/*
 *  The Regina Rexx Interpreter
 *  Copyright (C) 1993-1994  Anders Christensen <anders@pvv.unit.no>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * This file implements the client part of the SAA API when Regina
 * is linked into a program using SAA API. There is one routine for
 * each of the functions in SAA API, and the functionality is partly
 * implemented here, and partly by calling subroutines in Regina.
 * Note that the interface to Regina is as simple as possible, so that
 * a multitude of different transport mechanisms can be used (although
 * normal linking is probably the most common.
 *
 * The following SAA API functions is defined in this source file:
 *
 *    RexxStart()               --- execute Rexx code
 *    RexxRegisterSubcomExe()   --- register subcommand handler
 *    RexxRegisterSubcomDll()   --- ditto (not yet implemented)
 *    RexxQuerySubcom()         --- query subcommand handler
 *    RexxDeregisterSubcom()    --- deregister subcommand handler
 *    RexxVariablePool()        --- handle Rexx variable manipulation
 *    RexxRegisterExitExe()     --- register exit handler
 *    RexxRegisterExitDll()     --- ditto (not yet implemented)
 *    RexxDeregisterExit()      --- deregister exit handler
 *    RexxQueryExit()           --- query exit handler
 *    RexxRegisterFunctionExe() --- register external function handler
 *    RexxRegisterFunctionDll() --- ditto (from dynamic library)
 *    RexxQueryFunction()       --- query external function
 *    RexxDeregisterFunction()  --- deregister external function
 *    RexxSetHalt()             --- set Halt and Trace
 *    RexxFreeMemory()          --- free memory allocated by Rexx API
 *    RexxAllocMemory()         --- allocate memory to be freed by Rexx API
 *
 * These functions are all defined in the doc for SAA API. In addition,
 * a number of calls in Regina are called, as well as a number of calls
 * are defined for use by Regina. These all start with the prefix Ifc.
 * First the one defined in rexxsaa.c, which can be called from other
 * parts of Regina:
 *
 *    IfcSubCmd()      --- invoke a subcommand
 *    IfcDoExit()      --- invoke a system exit handler
 *    IfcExecFunc()    --- invoke an external function handler
 *    IfcExecFuncDll() --- invoke an external function handler in a DLL
 *
 * Then the functions which are defined elsewhere, which can be called
 * by this source code:
 *
 *    IfcStartUp()    --- initialize Regina
 *    IfcExecScript() --- start to execute Rexx code
 *    IfcVarPool()    --- handle a variable manipulation request
 *    IfcRegFunc()    --- register an external function name
 *    IfcDelFunc()    --- deregister an external function name
 *    IfcQueryFunc()  --- queries an external function name
 *
 * All these routines are properly defined in the documentation for
 * Regina. Other than the functions listed, the code in this file has
 * been isolated as far as possible, and no functions specific to
 * Regina is used, not even for memory allocation.
 */

/*
 * We need to define these symbols in order to get the proper macros,
 * datatypes, and declaration when including rexxsaa.h.
 */
#define INCL_RXSHV
#define INCL_RXSUBCOM
#define INCL_RXFUNC
#define INCL_RXSYSEXIT
#define INCL_RXARI


#if defined(HAVE_WINMULTITHREADING)
# include "header.h"
# define UNION_EXIT
#else
# ifdef HAVE_CONFIG_H
#  include "config.h"
# endif
#endif
/*
 * The rexxsaa.h header file defines the interface between this file and
 * the client program which uses SAA API. The rxiface.h header file
 * defines the interface between this file and Regina.
 */
#include "configur.h"
#include "rexxsaa.h"
#include "rxiface.h"
#include "defs.h"
#if defined(HAVE_WINMULTITHREADING)
# include "rexx.h"
#endif

#include <limits.h>
#include <stdio.h>
#include <string.h>
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_ASSERT_H
# include <assert.h>
#endif
#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>
#include <ctype.h>
#include <setjmp.h>

#if defined(HAVE_WINMULTITHREADING)
# if !defined(_MSC_VER)
#pragma data_seg(dseg1)
#endif
int noexternfunc=0;
#pragma data_seg()
struct funcbox2 {
   struct funcbox2 *next, *prev ;
   PSZ name ;
   RexxFunctionHandler *entry ;
   int hash ;
} *extfuncs[133] = { NULL } ;
char *extfuncName = 0L;
char *extfuncDllName = 0L;
char *extfuncEntryname = 0L;
HANDLE hextfuncName = 0L;
HANDLE hextfuncDllName = 0L;
HANDLE hextfuncEntryname = 0L;
void AddFuncName(char **p,char * Name);
void RemFuncName(char **pp,char * Name,char **qq);
BOOL IsFuncName(char **p,char * Name);
char *GetFuncName(char **pp,int i);
void AllocShared(void);
char *ReAllocShared(char *o,int size);
extern char *months[], *WeekDays[], *signalnames[];
extern int DaysInYear[], MonthDays[] ;
extern globalext *G;
extern long dwsysteminfo;
# if defined(_MSC_VER)
__declspec(dllexport) globalext SG;
__declspec(dllexport) int InitFlag=0 ;
#else
globalext SG;
int InitFlag=0 ;
#endif
int APIENTRY InitG2(globalext *G,int setting);
int APIENTRY IfcStartRexx( int setting );
void AddFunctions(void);
int __stdcall InitG(void);
int __stdcall InitG3(void);
#else
static int InitFlag=0 ;
struct funcbox2 {
   struct funcbox2 *next, *prev ;
   PSZ name ;
   RexxFunctionHandler *entry ;
   int hash ;
} *extfuncs[133] = { NULL } ;
#endif
static struct funcbox2 *findfunc( char *name );
static int addfunc2( PSZ name, RexxFunctionHandler *EntryPoint );
static int delfunc2( char *name );
extern void set_rexx_halt( void ) ;
extern void starttrace( void ) ;
extern void purge_filetable( void ) ;

int dummyzero=0;
#define HASH_MODULUS (sizeof(extfuncs) / sizeof(extfuncs[0]))

/*
 * The struct EnvBox datatype holds the definition of any subcommand
 * handler (i.e. an 'environment'). It is intended as an double-linked
 * list of entries, though performence concerns may force a change in
 * this structure later. It contains the name of the environment, and
 * an eight byte dataarea of user defined data. The same datastructure
 * is also used for holding the symbol table of external functions.
 *
 * This may prove a problem in the future, since the number of external
 * functions are generally much larger than the number of subcommand
 * handlers. Thus, different datastructures may be necessary to acheive
 * maximum performance.
 */
struct EnvBox
{
   struct EnvBox *prev, *next ;    /* double linked list pointers */
   char *EnvName ;                 /* environment/function name */
   unsigned char UserData[8] ;     /* user defined data area */
   union {
     PFN EntryPnt ;                /* external function entry point */
     RexxSubcomHandler *SubCom ;   /* subcommand handler entry point */
   } u ;
} ;

struct ExitHandlers
{
 RexxExitHandler *(Handlers[RXNOOFEXITS]) ; /* for RexxRegisterExitExe */
#if 0
 RXSTRING EntryPoint[RXNOOFEXITS];          /* for RexxRegisterExitDll */
 RXSTRING ModuleName[RXNOOFEXITS];          /* for RexxRegisterExitDll */
#endif
 struct ExitHandlers *prev ;
} ;
#if !defined(HAVE_WINMULTITHREADING)
struct ExitHandlers *CurrentHandlers=NULL ;
#endif

/*
 * The Interpreting variable is set when at least one level of
 * interpretation is currently active; i.e. at least one invokation of
 * RexxStart() is active in the call-stack. This variable is checked
 * later, when one wants to see whether a call to retrieve data values
 * would succeed.
 */
static int Interpreting=0 ;


/*
 * The following MAP_TYPE() macro maps from the SAA API macros holding
 * the type of an invocation (function, subroutine or command), to its
 * equivalent value in the internal interface of Regina (as defined in
 * the file rxiface.h
 */
#define MAP_TYPE(a) ((a)==RXCOMMAND ? RX_TYPE_COMMAND : \
              (a)==RXFUNCTION ? RX_TYPE_FUNCTION : RX_TYPE_SUBROUTINE)


/*
 * The FillReq() function takes as parameter a pointer to a VarPool()
 * request structure variable, and the definition of a string, and
 * fill the content of the string into the request block. Note that the
 * third parameter is gobbled up, so it can not be used or released by
 * the calling function afterwards. Also, there are two macros defined,
 * which gives a better access to the contents of the function
 */
#define FillReqName(a,b,c) FillReq(a,b,c,1)
#define FillReqValue(a,b,c) FillReq(a,b,c,0)

static void FillReq( PSHVBLOCK Req, ULONG Length, char *String, int type )
{
   RXSTRING *string=NULL ;
   ULONG *strlen=NULL ;
   ULONG SaveLength=0;

   string = type ? &(Req->shvname) : &(Req->shvvalue) ;
   strlen = type ? &(Req->shvnamelen) : &(Req->shvvaluelen) ;

   /*
    * If the string is undefined, set ->strptr to NULL. It is not required
    * that the lengths parameters are set to zero, but I'll do this as
    * nice gest to the users; someone is probably going to believe that
    * this is how the data is returned.
    */
   if (Length == RX_NO_STRING)
   {
      string->strptr = NULL ;
      *strlen = string->strlength = 0 ;
      return ;
   }

   /*
    * If a string was supplied, use it, else allocate sufficient space.
    * The then part of the if will just copy the data to the user-supplied
    * return string data area, noting a truncation is one occurred.
    */
   if (string->strptr)
   {
      if (*strlen<Length)
      {
          Req->shvret |= RXSHV_TRUNC ;
          SaveLength = Length;
          Length = *strlen ;
          *strlen = SaveLength;
      }
      else
         *strlen = Length ;
      memcpy(string->strptr, String, Length ) ;
      string->strlength = Length ;
   }
   else
   {
      /*
       * The else part of the if will allocate new space for the data, and
       * fills in the data, or return a memory fault if data could not
       * properly be allocated.
       * MH - 6-Feb-97
       * Some mallocs do not allow for a size of 0, so simply set the
       * return length to 0 and don't attempt to do the malloc.
       */
      if (Length)
      {
         string->strptr = malloc( Length ) ;
         if (string->strptr)
         {
            memcpy( string->strptr, String, Length ) ;
            string->strlength = Length ;
            *strlen = Length ;
         }
         else
            Req->shvret |= RXSHV_MEMFL ;
      }
      else
      {
       /* allocate at least 1 byte */
       string->strptr = malloc( 1 ) ;
       if (string->strptr)
       {
          /* no copy needed                    */
          string->strlength = Length ;
          *strlen = Length ;
       }
       else
          Req->shvret |= RXSHV_MEMFL ;
      }
   }
}

/*
 * This function initializes the Rexx interpreter. Actually, most of the
 * job is performed by IfcStartUp(). This module mainly performs three
 * functions: (1) veryfy parameters, (2) prevent reinitialization, and
 * (3) to verify that this module and the Rexx interpreter implements the
 * same version of the interface.
 *
 * Zero is returned if no errors occurred, while non-zero return code
 * indicates an error.
 */
static int StartupRexx( char *EnvName )
{
#if !defined(HAVE_WINMULTITHREADING)
   int rc=0, Maj=0, Min=0 ;

   if (InitFlag)
      return 1 ;

   InitFlag = 1 ;
   /*
    * First parameter ignored, but keep it for compatibility
    */
   rc = IfcStartUp( NULL, &Maj, &Min ) ;

   if (Maj != RXVERSION_MAJ || Min != RXVERSION_MIN)
      return 1 ;

   return rc ;
#else
   int rc=0 ;

   if (InitFlag)
      return 1 ;
   InitFlag=1;
   rc=InitG2(&SG,2);
   return rc ;
#endif
}

/* ========================================================================
 * Here starts the section for maintaining the list of environments
 * supported by this environment. There are several routines using
 * the functions in this section, the routines defined here are:
 *
 *  FindEnv()  --- retrieves a pointer to a environment box.
 *
 * Actually, it used to be more, one to insert and one to delete.
 * However, in order to save code, these was moved into the routines
 * where they were called (they were used only once). The functions
 * into which the code was moved are RexxRegisterSubcomExe(), and
 * RexxDeregisterSubcom(). To improve modularization, and to
 * ease the introduction of a new datastructure, the code should
 * probably be extracted and inserted in this section.
 */

/*
 * First there is the name of the pointer to the head of the list
 * containing all the environments.
 */
static struct EnvBox *FirstEnv=NULL ;
static struct EnvBox *FirstExit=NULL ;

#define BOX_IS_ENVIR 0
#define BOX_IS_EXIT 1

/*
 * Find a particular environment, and return a pointer to a struct
 * containing information about that environment. If it is not found,
 * a pointer to NULL is returned.
 */
#define FindEnvir(a,b) FindBox(a,b,BOX_IS_ENVIR)
#define FindExit(a,b) FindBox(a,b,BOX_IS_EXIT)

static struct EnvBox *FindBox( char *Env, int EnvLen, int type )
{
   struct EnvBox *bptr = ((type==BOX_IS_ENVIR) ? FirstEnv : FirstExit) ;

   assert( Env ) ;
   for (; bptr; bptr=bptr->next)
      if (!memcmp(bptr->EnvName,Env,EnvLen))
         return bptr ;

   return NULL ;
}

#define AddEnvir(a,b,c,d) AddBox(a,b,c,d,BOX_IS_ENVIR)
#define AddExit(a,b,c,d) AddBox(a,b,c,d,BOX_IS_EXIT)

static struct EnvBox *AddBox( char *EnvName, int EnvLen, void *UserArea,
                              PFN EntryPoint, int type )
{
   struct EnvBox *NewBox=NULL ;
   struct EnvBox **first=NULL ;

   first = (type==BOX_IS_ENVIR) ? &FirstEnv : &FirstExit ;
   NewBox = malloc( sizeof( struct EnvBox )) ;
   if (!NewBox)
      return NULL ;

   NewBox->EnvName = malloc( EnvLen+1 ) ;
   if (!NewBox->EnvName)
   {
      free( NewBox ) ;
      return NULL ;
   }

   NewBox->prev = NULL ;
   NewBox->next = (*first) ;
   if (*first)
      (*first)->prev = NewBox ;
   (*first) = NewBox ;

   memcpy( NewBox->EnvName, EnvName , EnvLen ) ;
   NewBox->EnvName[EnvLen] = '\0';
   NewBox->u.EntryPnt = EntryPoint ;
   if (UserArea)
      memcpy( NewBox->UserData, UserArea, 8 ) ;
   else
      memset( NewBox->UserData, 0x00, 8 ) ;

   return NewBox ;
 }


#define RemoveExit(a,b) RemoveBox(a,b,BOX_IS_EXIT)
#define RemoveEnvir(a,b) RemoveBox(a,b,BOX_IS_ENVIR)

static int RemoveBox( char *EnvName, int EnvLen, int type )
{
   struct EnvBox *OldBox=NULL ;
   struct EnvBox **First=NULL ;

   OldBox = FindBox( EnvName, EnvLen, type ) ;
   if (OldBox)
   {
      First = (type==BOX_IS_ENVIR) ? &FirstEnv : &FirstExit ;
      if (OldBox->prev)
         OldBox->prev->next = OldBox->next ;
      if (OldBox->next)
         OldBox->next->prev = OldBox->prev ;
      if ((*First)==OldBox)
         (*First) = OldBox->prev ;

      free( OldBox->EnvName ) ;
      free( OldBox ) ;
      return 0 ;
   }
   return 1 ;
}



int IfcSubCmd( int EnvLen, char *EnvStr, int CmdLen, char *CmdStr,
               int *RetLen, char **RetStr )
{
   RXSTRING Cmd, Ret ;
#ifdef FGC
   /* FGC: result must be available in  */
   /*      calling function SubCom in   */
   /*      client.c --> static          */
   static char result[RXAUTOBUFLEN] ;
#else
   char result[RXAUTOBUFLEN] ;
#endif
   char *OldResult= NULL ;
   USHORT Flags=0 ;
   int Length=0 ;
   char *EnvNam=NULL, *Command=NULL ;
   struct EnvBox *Envir=NULL ;
   int rvalue=0, RCode=0, rc=RXEXIT_NOT_HANDLED ;
#ifdef UNION_EXIT
   EXIT ParBox ;
#else
   RXCMDHST_PARM cmdhst;
   PUCHAR parm=NULL;
#endif
#if defined(HAVE_WINMULTITHREADING)
   struct ExitHandlers *CurrentHandlers=NULL ;
#include "multi.h"
   CurrentHandlers=(struct ExitHandlers *)CH;
#endif

   EnvNam = EnvStr ;
   Length = CmdLen ;
   Command = CmdStr ;
   /*
    * Terminate the command string with nul character
    */
   if (Command && CmdLen)
      Command[Length] = '\0';
   if ( CurrentHandlers
   &&   CurrentHandlers->Handlers[RXCMD] )
     {
#ifdef UNION_EXIT
      ParBox.cmdhst.rxcmd_flags.rxfcfail = 0;
      ParBox.cmdhst.rxcmd_flags.rxfcerr = 0;
      ParBox.cmdhst.rxcmd_command.strlength = Length ;
      ParBox.cmdhst.rxcmd_command.strptr = Command ;
      ParBox.cmdhst.rxcmd_address = (unsigned char *)EnvStr ;
      ParBox.cmdhst.rxcmd_addressl = (USHORT) EnvLen ;
      MAKERXSTRING( Ret, result, RXAUTOBUFLEN) ;
      OldResult = result ;
      ParBox.cmdhst.rxcmd_retc.strlength = Ret.strlength;
      ParBox.cmdhst.rxcmd_retc.strptr = Ret.strptr;
      ParBox.cmdhst.rxcmd_dll = NULL;
      ParBox.cmdhst.rxcmd_dll_len = 0;
      rc = (*(CurrentHandlers->Handlers[RXCMD]))(RXCMD, RXCMDHST, &ParBox);
      assert( rc==RXEXIT_HANDLED || rc==RXEXIT_NOT_HANDLED ||
              rc==RXEXIT_RAISE_ERROR ) ;
      *RetLen = ParBox.cmdhst.rxcmd_retc.strlength ;
      *RetStr = ParBox.cmdhst.rxcmd_retc.strptr ;
      if (ParBox.cmdhst.rxcmd_flags.rxfcerr)
         RCode = RXFLAG_ERROR ;
      else if (ParBox.cmdhst.rxcmd_flags.rxfcfail)
         RCode = RXFLAG_FAILURE ;
      else
         RCode = RXFLAG_OK;
      Ret.strlength = ParBox.cmdhst.rxcmd_retc.strlength;
#else
      cmdhst.rxcmd_flags.rxfcfail = 0;
      cmdhst.rxcmd_flags.rxfcerr = 0;
      cmdhst.rxcmd_command.strlength = Length ;
      cmdhst.rxcmd_command.strptr = Command ;
      cmdhst.rxcmd_address = (unsigned char *)EnvStr ;
      cmdhst.rxcmd_addressl = (USHORT) EnvLen ;
      MAKERXSTRING( Ret, result, RXAUTOBUFLEN) ;
      OldResult = result ;
      cmdhst.rxcmd_retc.strlength = Ret.strlength;
      cmdhst.rxcmd_retc.strptr = Ret.strptr;
      cmdhst.rxcmd_dll = NULL;
      cmdhst.rxcmd_dll_len = 0;
      parm = (PUCHAR)&cmdhst;
      rc = (*(CurrentHandlers->Handlers[RXCMD]))(RXCMD, RXCMDHST, parm);
      assert( rc==RXEXIT_HANDLED || rc==RXEXIT_NOT_HANDLED ||
              rc==RXEXIT_RAISE_ERROR ) ;
/*      cmdhst = (RXCMDHST_PARM *)parm; */
      *RetLen = cmdhst.rxcmd_retc.strlength ;
      *RetStr = cmdhst.rxcmd_retc.strptr ;
      if (cmdhst.rxcmd_flags.rxfcerr)
         RCode = RXFLAG_ERROR ;
      else if (cmdhst.rxcmd_flags.rxfcfail)
         RCode = RXFLAG_FAILURE ;
      else
         RCode = RXFLAG_OK;
      Ret.strlength = cmdhst.rxcmd_retc.strlength;
#endif
     }
   if (rc == RXEXIT_NOT_HANDLED)
     {
      Envir = FindEnvir( EnvNam , EnvLen ) ;
      if (Envir)
      {
         MAKERXSTRING( Cmd, Command, Length ) ;
         MAKERXSTRING( Ret, result, RXAUTOBUFLEN) ;
         OldResult = result ;
         rvalue = (*(Envir->u.SubCom))( &Cmd, &Flags, &Ret ) ;
         if (Flags==RXSUBCOM_OK)
            RCode = RXFLAG_OK ;
         else if (Flags==RXSUBCOM_ERROR)
            RCode = RXFLAG_ERROR ;
         else if (Flags==RXSUBCOM_FAILURE)
            RCode = RXFLAG_FAILURE ;
         else
            assert( dummyzero ) ;
      }
      else
      {
         RCode = RXFLAG_NOTREG ;
         Ret.strlength = 0 ;
      }
     }

   if (Ret.strlength)
   {
      *RetLen = Ret.strlength ;
      *RetStr = Ret.strptr ;
   }
   else
   {
      *RetLen = 1 ;
      *RetStr = "0" ;
   }

   if (Ret.strlength && OldResult != Ret.strptr)
      free( Ret.strptr ) ;

#include "unmulti.h"
   return RCode ;
}


/*
 * There is a macro for this now...
static struct EnvBox *FindExit( char *Env , int EnvLen )
{
   struct EnvBox *bptr=NULL ;

   assert( Env ) ;
   for (bptr=FirstExit; bptr; bptr=bptr->next)
      if (!memcmp(bptr->EnvName,Env,EnvLen))
         return bptr ;

   return NULL ; }
*/



int IfcDoExit( int Code, int *RLength, char **RString )
{
   int rc=0;
   LONG SubCode=0, MainCode=0 ;
#ifdef UNION_EXIT
   EXIT ParBox ;
#else
   RXSIOSAY_PARM siosay;
   RXSIOTRD_PARM siotrd;
   RXSIODTR_PARM siodtr;
   PEXIT parm=NULL;
#endif
#if defined(HAVE_WINMULTITHREADING)
   struct ExitHandlers *CurrentHandlers=NULL ;
#include "multi.h"
   CurrentHandlers=(struct ExitHandlers *)CH;
#endif

   switch (Code)
   {
      case RX_EXIT_STDERR:
      case RX_EXIT_STDOUT:
#ifdef UNION_EXIT
         ParBox.siosay.rxsio_string.strptr = *RString ;
         ParBox.siosay.rxsio_string.strlength = *RLength ;
#else
         siosay.rxsio_string.strptr = *RString ;
         siosay.rxsio_string.strlength = *RLength ;
         /*
          * nul terminate the string
          */
         siosay.rxsio_string.strptr[siosay.rxsio_string.strlength] = '\0';
         parm = (PEXIT)&siosay;
#endif
         SubCode = (Code==RX_EXIT_STDOUT) ? RXSIOSAY : RXSIOTRC ;
         MainCode = RXSIO ;

         break ;

      case RX_EXIT_TRCIN:
#ifdef UNION_EXIT
         ParBox.siodtr.rxsiodtr_retc.strlength = *RLength ;
         ParBox.siodtr.rxsiodtr_retc.strptr = *RString ;
#else
         siodtr.rxsiodtr_retc.strlength = *RLength ;
         siodtr.rxsiodtr_retc.strptr = *RString ;
         parm = (PEXIT)&siodtr;
#endif
         SubCode = RXSIODTR ;
         MainCode = RXSIO ;
         break ;

      case RX_EXIT_PULL:
#ifdef UNION_EXIT
         ParBox.siotrd.rxsiotrd_retc.strlength = *RLength ;
         ParBox.siotrd.rxsiotrd_retc.strptr = *RString ;
#else
         siotrd.rxsiotrd_retc.strlength = *RLength ;
         siotrd.rxsiotrd_retc.strptr = *RString ;
         parm = (PEXIT)&siotrd;
#endif
         SubCode = RXSIOTRD ;
         MainCode = RXSIO ;
         break ;

      case RX_EXIT_INIT:
         MainCode = RXINI ;
         SubCode = RXINIEXT ;
         break ;

      default:
         assert( dummyzero ) ;
         /* fallthrough; to avoid compiler warnings */

      case RX_EXIT_TERMIN:
         MainCode = RXTER ;
         SubCode = RXTEREXT ;
         break ;
   }

   assert( CurrentHandlers->Handlers[MainCode] ) ;

#ifdef UNION_EXIT
{
#if defined(HAVE_WINMULTITHREADING)
LONG (* APIENTRY SystemExit)(LONG,LONG,PEXIT);
       SystemExit=(LONG (* APIENTRY)(LONG,LONG,PEXIT))CurrentHandlers->Handlers[MainCode];
   rc = (*(SystemExit))((LONG)MainCode,(LONG)SubCode,(PEXIT)&ParBox);
#else
   rc = (*(CurrentHandlers->Handlers[MainCode]))(MainCode, SubCode, &ParBox);
#endif
}
#else
   rc = (*(CurrentHandlers->Handlers[MainCode]))(MainCode, SubCode, parm);
#endif
   assert( rc==RXEXIT_HANDLED || rc==RXEXIT_NOT_HANDLED ||
           rc==RXEXIT_RAISE_ERROR ) ;

   switch (Code)
   {
     case RX_EXIT_STDERR:
     case RX_EXIT_STDOUT:
     case RX_EXIT_INIT:
     case RX_EXIT_TERMIN:
         break ;

     case RX_EXIT_TRCIN:
#ifdef UNION_EXIT
         *RLength = ParBox.siodtr.rxsiodtr_retc.strlength ;
         *RString = ParBox.siodtr.rxsiodtr_retc.strptr ;
#else
/*         siodtr = (RXSIODTR_PARM*)parm; */
         *RLength = siodtr.rxsiodtr_retc.strlength ;
         *RString = siodtr.rxsiodtr_retc.strptr ;
#endif
         break ;

     case RX_EXIT_PULL:
#ifdef UNION_EXIT
         *RLength = ParBox.siotrd.rxsiotrd_retc.strlength ;
         *RString = ParBox.siotrd.rxsiotrd_retc.strptr ;
#else
/*         siotrd = (RXSIOTRD_PARM*)parm; */
         *RLength = siotrd.rxsiotrd_retc.strlength ;
         *RString = siotrd.rxsiotrd_retc.strptr ;
#endif
         break ;

      default:
         assert( dummyzero ) ;
   }

   if (rc==RXEXIT_HANDLED)
      rc = RX_HOOK_NOPE ;
   else if (rc==RXEXIT_NOT_HANDLED)
      rc = RX_HOOK_GO_ON ;
   else if (rc==RXEXIT_RAISE_ERROR)
      rc = RX_HOOK_ERROR ;

   if (RString && !(*RString))
   {
      *RLength = RX_NO_STRING ;
   }
   return rc ;
#include "unmulti.h"
}

/* ================================================================ */
/* ================ general purpose API functions ================= */

APIRET APIENTRY RexxFreeMemory(
   PVOID        MemoryBlock )
{
#if defined(HAVE_WINMULTITHREADING)
   GlobalFree( (HGLOBAL)MemoryBlock );
#else
   free( MemoryBlock );
#endif
   return 0;
}

PVOID APIENTRY RexxAllocateMemory(
   int         size )
{
#if defined(HAVE_WINMULTITHREADING)
   return (PVOID)(GlobalAlloc( GMEM_FIXED, size ));
#else
   return(PVOID)(malloc( size ));
#endif
}

/* ================================================================ */
/* ================ in order to start Rexx scripts ================ */

APIRET APIENTRY RexxStart(
   LONG         ArgCount,
   PRXSTRING    ArgList,
   PSZ          ProgName,
   PRXSTRING    Instore,
   PSZ          EnvName,
   LONG         CallType,
   PRXSYSEXIT   Exits,
   PSHORT       ReturnCode,
   PRXSTRING    Result )
{
   int cnt=0, RLength=0 ;
   char *RString=NULL ;
   int ParLengths[32] ;
   char *ParStrings[32] ;
   int ExitFlags=0 ;
   int EnvNamLen=0 ;
   char *EnvNamStr=NULL ;
   char *WherePtr=NULL ;
   int WhereCode=0, WhereLen=0, rc=0 ;
   struct ExitHandlers *Handlers=NULL ;
   RexxExitHandler *handler=NULL ;
   struct EnvBox *EnvPtr=NULL ;
   LONG ResValue=0L ;
   int instore_idx = 0;
#if defined(HAVE_WINMULTITHREADING)
   int i;
   char *memptr;
   char *ProgramName;
   struct ExitHandlers *CurrentHandlers=NULL ;
   int IsAlloc,s ,r;
   globalext *G;
   DWORD Size;
   extern globalext SG;
#else
   int drop_ipret=0 ;
   PSZ ProgramName=ProgName;
#endif

#if defined(HAVE_WINMULTITHREADING)
   ProgramName=(char *)LocalAlloc(LPTR, strlen(ProgName)+1);
   strcpy(ProgramName,ProgName);
   CharUpper(ProgramName);
#endif

   if (Instore)
   {
/* MH 2602     if (Instore[1].strptr && Instore[1].strlength < sizeof(int)) */
      if (Instore[1].strptr && Instore[1].strlength < 1)
         return RX_START_BADP ;
   }

   if (CallType!=RXCOMMAND && CallType!=RXFUNCTION && CallType!=RXSUBROUTINE)
      return RX_START_BADP ;

   if (CallType==RXCOMMAND && ArgCount>1)
      return RX_START_TOOMANYP ;

   if (!InitFlag)
      StartupRexx( EnvName ) ;

#if defined(HAVE_WINMULTITHREADING)
    Interpreting++;
#include "domulti.h"
    IsAlloc=0;
    s=sizeof(struct globalvar);
    G = (globalext *)TlsGetValue(dwsysteminfo);
    if ((G == (globalext *)0L) || (G == (globalext *)&SG))
    {
       s=sizeof(struct globalvar);
       G = (globalext *) LocalAlloc(LPTR, s);
       IsAlloc=1;
       InitG2(G,2);
       countthreads=1;
       InUse=1;
       r=TlsSetValue(dwsysteminfo, (void *)G);
    }
    else
    {
       countthreads++;
    }
#else
   if (Interpreting==0)
      drop_ipret = Interpreting = 1 ;
#endif

   for (cnt=0; cnt<ArgCount; cnt++)
   {
      ParLengths[cnt] = ArgList[cnt].strlength ;
      ParStrings[cnt] = ArgList[cnt].strptr ;
      if (ParStrings[cnt]==NULL)
         ParLengths[cnt] = RX_NO_STRING ;
   }

   Handlers = malloc( sizeof( struct ExitHandlers )) ;
#if !defined(HAVE_WINMULTITHREADING)
   Handlers->prev = CurrentHandlers ;
#else
   Handlers->prev = CH ;
   CH = (void *)Handlers ;
#endif
   CurrentHandlers = Handlers ;
   for (cnt=0; cnt<RXNOOFEXITS; cnt++)
      CurrentHandlers->Handlers[cnt] = NULL ;

   ExitFlags = 0x00000000 ;
   for (cnt=0; Exits && Exits->sysexit_code!=RXENDLST; Exits++ )
   {
      EnvPtr = FindExit( Exits->sysexit_name , strlen(Exits->sysexit_name) ) ;
      if (!EnvPtr)
         continue ;

      /* Sigh ... Definition requires some strange casting */
      handler = (RexxExitHandler*)(EnvPtr->u.EntryPnt) ;
      switch (Exits->sysexit_code)
      {
          case RXSIO:
             ExitFlags |= (1<<RX_EXIT_STDOUT) | (1<<RX_EXIT_STDERR) |
                          (1<<RX_EXIT_TRCIN) | (1<<RX_EXIT_PULL) ;
             CurrentHandlers->Handlers[RXSIO] = handler ;
             break ;

          case RXINI:
             ExitFlags |= (1<<RX_EXIT_INIT) ;
             CurrentHandlers->Handlers[RXINI] = handler ;
             break ;

          case RXTER:
             ExitFlags |= (1<<RX_EXIT_TERMIN) ;
             CurrentHandlers->Handlers[RXTER] = handler ;
             break ;

          case RXCMD:
             ExitFlags |= (1<<RX_EXIT_SUBCOM) ;
             CurrentHandlers->Handlers[RXCMD] = handler ;
             break ;

          case RXFNC:
             ExitFlags |= (1<<RX_EXIT_FUNC) ;
             CurrentHandlers->Handlers[RXFNC] = handler ;
             break ;

          default:
             assert( dummyzero ) ;
      }
   }

   if (EnvName)
   {
      EnvNamLen = strlen(EnvName) ;
      EnvNamStr = EnvName ;
   }
   else
   {
      EnvNamLen = RX_NO_STRING ;
      EnvNamStr = NULL ;
   }

   WherePtr = NULL ;
   WhereLen = 0 ;
   if (Instore && Instore[1].strptr)
   {
      WhereCode = RX_TYPE_INSTORE ;
      WherePtr = Instore[1].strptr ;
/* MH2602      WhereLen = sizeof(int) ; */
      WhereLen = Instore[1].strlength ;
   }
   else if (Instore && Instore[0].strptr)
   {
      WhereCode = RX_TYPE_SOURCE ;
      WherePtr = Instore[0].strptr ;
      WhereLen = Instore[0].strlength ;
   }
   else if (Instore)
      WhereCode = RX_TYPE_MACRO ;
   else
      WhereCode = RX_TYPE_EXTERNAL ;
   starttrace() ;

   rc = IfcExecScript( strlen(ProgramName), ProgramName,
          ArgCount, ParLengths, ParStrings, MAP_TYPE(CallType),
          ExitFlags, EnvNamLen, EnvNamStr,  WhereCode, WherePtr,
          WhereLen, &RLength, &RString, &instore_idx) ;
#if !defined(HAVE_WINMULTITHREADING)
   Handlers = CurrentHandlers ;
   CurrentHandlers = Handlers->prev ;
#else
   Handlers = CH ;
   CH = Handlers->prev ;
#endif
   free( Handlers ) ;

   if (WhereCode == RX_TYPE_SOURCE)
   {
#if defined(HAVE_WIN32GUI)
      Instore[1].strptr = GlobalAlloc(GMEM_FIXED|GMEM_ZEROINIT,20);
#else
      Instore[1].strptr = malloc(20);
#endif
      sprintf(Instore[1].strptr,"%d",instore_idx);
      Instore[1].strlength = strlen(Instore[1].strptr);
   }

   if (RLength!=RX_NO_STRING)
      ResValue = atoi( RString ) ;
   else
      ResValue = 0 ;

   if (ReturnCode)
      *ReturnCode = (SHORT) ResValue ;   /* FGC */

   if (Result)
   {
      if (!Result->strptr || (int)Result->strlength>=RLength+1)
      {
         Result->strlength = RLength ;
#if defined(HAVE_WIN32GUI)
         Result->strptr = GlobalAlloc(GMEM_FIXED|GMEM_ZEROINIT,RLength+1);
         memcpy(Result->strptr,RString,RLength);
#else
         Result->strptr = RString ;
#endif
      }
      else
      {
         Result->strlength = RLength ;
         memcpy( Result->strptr, RString, RLength+1 ) ;
         free( RString ) ;
      }
   }

#if defined(HAVE_WINMULTITHREADING)
   G = (globalext *)TlsGetValue(dwsysteminfo);
   if (G != (globalext *)0L)
   {
      countthreads--;
      if ((countthreads==0) && (IsAlloc==1))
      {
         InUse=0;
         r=TlsSetValue(dwsysteminfo, (void *)0L);
         LocalFree((HLOCAL) G);
      }
   }
   Interpreting--;
   LocalFree((HLOCAL)ProgramName);
#else
   if (drop_ipret)
      Interpreting = 0 ;

   /*
    * Close all open files and reset Regina's internal file table
    */
   purge_filetable();
#endif

   return rc ;
#include "unmulti.h"
}



/* ============================================================= */
/* subcom handler subsystem */

APIRET APIENTRY RexxRegisterSubcomExe(
   PSZ EnvName,
   PFN EntryPoint,
   PUCHAR UserArea )
{
   int EnvLen=0 ;

#if defined(HAVE_WINMULTITHREADING)
   if (!InitFlag)
      StartupRexx( "none" ) ;
#endif
   /*
    * Perform sanity check on the parameters; UserArea may be NULL
    */
   if (!EnvName || !EntryPoint)
      return RXSUBCOM_BADTYPE ;

   EnvLen = strlen( EnvName ) ;
   if (EnvLen>MAXENVNAMELEN)
      return RXSUBCOM_NOTREG ;

   if (FindEnvir( EnvName, EnvLen ))
      return RXSUBCOM_NOTREG ;

   if (!AddEnvir( EnvName, strlen(EnvName) , UserArea, EntryPoint ))
      return RXSUBCOM_NOEMEM ;

   return RXSUBCOM_OK ;
}


APIRET APIENTRY RexxRegisterSubcomDll(
   PSZ EnvName,
   PSZ ModuleName,
   PSZ ProcedureName,
   PUCHAR UserArea,
   ULONG DropAuth )
{
   /* not yet functional */
#if defined(HAVE_WINMULTITHREADING)
   if (!InitFlag)
      StartupRexx( "none" ) ;
#endif
   return RXSUBCOM_NOTREG ;
}


APIRET APIENTRY RexxQuerySubcom(
   PSZ EnvName,
   PSZ ModuleName,
   PUSHORT Flag,      /* Who knows what this is used for ... */
   PUCHAR UserWord )
{
   int ret=0 ;
   struct EnvBox *eptr=NULL ;

#if defined(HAVE_WINMULTITHREADING)
   if (!InitFlag)
      StartupRexx( "none" ) ;
#endif
   if (!EnvName)
      return RXSUBCOM_BADTYPE ;

   if (ModuleName)
      return RXSUBCOM_BADTYPE ;   /* not yet functional */

   eptr = FindEnvir( EnvName , strlen(EnvName) ) ;
   if (eptr)
   {
      ret = RXSUBCOM_OK ;
      if (UserWord)
         memcpy( UserWord, eptr->UserData, 8 ) ;
   }
   else
      ret = RXSUBCOM_NOTREG ;

   *Flag = 0;               /* what else to give it ? */

   return ret ;
}

APIRET APIENTRY RexxDeregisterSubcom(
   PSZ EnvName,
   PSZ ModuleName )
{
#if defined(HAVE_WINMULTITHREADING)
   if (!InitFlag)
      StartupRexx( "none" ) ;
#endif
   if (!EnvName)
      return RXSUBCOM_BADTYPE ;

   if (ModuleName)
      return RXSUBCOM_BADTYPE ;  /* not yet functional */

   if (RemoveEnvir( EnvName , strlen(EnvName) ))
      return RXSUBCOM_NOTREG ;

   return RXSUBCOM_OK ;
}



/* ============================================================ */
/* Variable subsystem */


APIRET APIENTRY RexxVariablePool(
   PSHVBLOCK RequestBlockList )
{
   int Code=0, RetCode=0 ;
   int Lengths[2] ;
   int rc=0 ;
   char *Strings[2] ;
   PSHVBLOCK Req=RequestBlockList ;
#if !defined(HAVE_WINMULTITHREADING)
extern int var_indicator ;
#else
# include "multi.h"
#endif

#if !defined(HAVE_WINMULTITHREADING)
   if (Interpreting==0)
      return RXSHV_NOAVL ;
#endif
   RetCode = 0 ;

   /* Probably unneeded, but it won't do any harm */
   if (!InitFlag)
      StartupRexx( "none" ) ;

   for (;Req;Req=Req->shvnext)
   {
      switch (Req->shvcode)
      {
         case RXSHV_SYDRO:
         case RXSHV_SYSET:
         case RXSHV_DROPV:                              /* MH 26-12-95 */
         case RXSHV_SET:                                /* MH 26-12-95 */
         {
            Lengths[0] = Req->shvname.strlength ;
            Strings[0] = Req->shvname.strptr ;
            if (Req->shvcode==RXSHV_SYSET               /* MH 26-12-95 */
            ||  Req->shvcode==RXSHV_SET)                /* MH 26-12-95 */
            {
               Lengths[1] = Req->shvvalue.strlength ;
               Strings[1] = Req->shvvalue.strptr ;
            }
            else
               Lengths[1] = RX_NO_STRING ;

            Code = IfcVarPool( RX_SETSVAR, Lengths, Strings ) ;

            Req->shvret = RXSHV_OK ;
            if (Code==RX_CODE_NOVALUE)
               Req->shvret |= RXSHV_NEWV ;
            else if (Code==RX_CODE_INVNAME)
               Req->shvret |= RXSHV_BADN ;
            else if (Code!=RXSHV_OK)
               assert( dummyzero ) ;
            var_indicator=0;
            break ;
         }
         case RXSHV_SYFET:
         case RXSHV_FETCH:                              /* MH 26-12-95 */
         {
            Lengths[0] = Req->shvname.strlength ;
            Strings[0] = Req->shvname.strptr ;
            Lengths[1] = Req->shvvalue.strlength ;
            Strings[1] = Req->shvvalue.strptr ;
            Code = IfcVarPool( RX_GETSVAR, Lengths, Strings ) ;

            Req->shvret = RXSHV_OK ;
            if (Code==RX_CODE_NOVALUE)
               Req->shvret |= RXSHV_NEWV ;
            else if (Code==RX_CODE_INVNAME)
               Req->shvret |= RXSHV_BADN ;
            else if (Code!=RXSHV_OK)
               assert( dummyzero ) ;
            FillReqValue( Req, Lengths[1], Strings[1] ) ;
            var_indicator=0;
            break ;
         }

         case RXSHV_PRIV:
         {
            Req->shvret = RXSHV_OK ;
            if (Req->shvname.strlength==4 && Req->shvname.strptr &&
                !strncmp(Req->shvname.strptr, "PARM", 4 ))
            {
               rc = IfcVarPool( RX_CODE_PARAMS, Lengths, Strings ) ;
               FillReqValue( Req, Lengths[0], Strings[0] ) ;
            }

            else if (Req->shvname.strlength>=5 && Req->shvname.strptr &&
                !strncmp(Req->shvname.strptr, "PARM.", 5 ))
            {
               Lengths[0] = Req->shvname.strlength - 5 ;
               Strings[0] = Req->shvname.strptr + 5 ;

               rc = IfcVarPool( RX_CODE_PARAM, Lengths, Strings ) ;
               if (rc == RX_CODE_OK)
                  FillReqValue( Req, Lengths[1], Strings[1] ) ;
               else
                  Req->shvret |= RXSHV_BADN ;
            }

            else
            {
               int Code=0 ;
               if (Req->shvname.strptr)
               {
                  if (Req->shvname.strlength==7 &&
                         !memcmp(Req->shvname.strptr, "QUENAME", 7))
                  {
                     Code = RX_CODE_QUEUE ;
                  }
                  else if (Req->shvname.strlength==7 &&
                         !memcmp(Req->shvname.strptr, "VERSION", 7))
                  {
                     Code = RX_CODE_VERSION ;
                  }
                  else if (Req->shvname.strlength==6 &&
                         !memcmp(Req->shvname.strptr, "SOURCE", 6))
                  {
                     Code = RX_CODE_SOURCE ;
                  }
                  else
                     Req->shvret |= RXSHV_BADN ;

                  if (!Req->shvret | RXSHV_BADN)
                  {
                     rc=IfcVarPool(Code, Lengths, Strings ) ;
                     FillReqValue( Req, Lengths[0], Strings[0] ) ;
                  }
               }
               else
                  Req->shvret |= RXSHV_BADN ;
            }
            break ;
         }

         case RXSHV_NEXTV:
         {
            int Items ;

            Req->shvret = RXSHV_OK ;
            Items = IfcVarPool( RX_NEXTVAR, Lengths, Strings ) ;
            assert( Items==0 || Items==2 ) ;

            if (Items==2)
            {
               FillReqValue( Req, Lengths[1], Strings[1] ) ;
               FillReqName( Req, Lengths[0], Strings[0] ) ;
            }
            else
               Req->shvret |= RXSHV_LVAR ;

            break ;
         }

         default:
            Req->shvret = RXSHV_BADF ;
      }

      RetCode |= ( Req->shvret & 0x007f ) ;
   }

   return RetCode ;
#include "unmulti.h"
}



/* ================================================================ */
/* system exit handler subsystem */

APIRET APIENTRY RexxRegisterExitExe(
   PSZ EnvName,
   PFN EntryPoint,
   PUCHAR UserArea )
{
   int EnvLen=0 ;
#if defined(HAVE_WINMULTITHREADING)
# include "multi.h"
   if (!InitFlag)
      StartupRexx( "none" ) ;
#endif
   /*
    * Perform sanity check on the parameters; UserArea may be NULL
    */
   if (!EnvName || !EntryPoint)
      return RXEXIT_BADTYPE ;

   EnvLen = strlen( EnvName ) ;
   if (EnvLen>MAXENVNAMELEN)
      return RXEXIT_NOTREG ;

   if (FindExit( EnvName, EnvLen ))
      return RXEXIT_NOTREG ;

   if (!AddExit( EnvName, EnvLen, UserArea, EntryPoint ))
      return RXEXIT_NOEMEM ;

   return RXEXIT_OK ;
#include "unmulti.h"
}

APIRET APIENTRY RexxRegisterExitDll(
   PSZ EnvName,
   PSZ ModuleName,
   PSZ ProcedureName,
   PUCHAR UserArea,
   ULONG DropAuth )
{
#if defined(HAVE_WINMULTITHREADING)
   if (!InitFlag)
      StartupRexx( "none" ) ;
#endif
   /* not yet functional */
   return RXEXIT_NOTREG ;
}


APIRET APIENTRY RexxDeregisterExit(
   PSZ EnvName,
   PSZ ModuleName )
{
#if defined(HAVE_WINMULTITHREADING)
   if (!InitFlag)
      StartupRexx( "none" ) ;
#endif
   if (!EnvName)
      return RXEXIT_BADTYPE ;

   if (ModuleName)
      return RXEXIT_BADTYPE ;

   if (RemoveExit(EnvName, strlen(EnvName) ))
      return RXEXIT_NOTREG ;

   return RXEXIT_OK ;
}


APIRET APIENTRY RexxQueryExit(
   PSZ EnvName,
   PSZ ModuleName,
   PUSHORT Flag,
   PUCHAR UserArea)
{
#if defined(HAVE_WINMULTITHREADING)
   if (!InitFlag)
      StartupRexx( "none" ) ;
#endif
   assert( EnvName );
   return( RXEXIT_NOTREG );
/*
   return ( FindExit( Name, strlen( Name ) ) ) ? RXEXIT_NOTREG : RXEXIT_OK ;
*/
}


/* ================================================================= */
/* What is this .... ? */


void DosFreeMem( void *ptr )
{
/*
   SendNumber( RX_DROP_INSTORE ) ;
   SendNumber( *((int*)(ptr)) ) ;
   WaitForNumber() ;
 */
}


/* =================================================================== */

/*
 * This section contains the support for the external functions
 */


/*
 * Calulates a hashvalue for 'name', for use as index into the array of
 * functions that have been loaded from external function packages.
 */
static int hashvalue( char *name )
{
   int result=0 ;
   char c;

   while ((c = *name++) != '\0')
      result += c ;

   return result ;
}

static struct funcbox2 *findfunc( char *name )
{
   struct funcbox2 *fptr=NULL ;
   int hashbox, hash ;

   hash = hashvalue( name ) ;
   hashbox = hash % HASH_MODULUS ;
   for (fptr=extfuncs[hashbox]; fptr; fptr=fptr->prev)
      if (fptr->hash == hash)
         if (!strcmp(name, fptr->name))
            return fptr ;

   return NULL ;
}

static int delfunc2( char *name )
{
   struct funcbox2 *old=NULL ;
   int hashbox ;

   old = findfunc( name ) ;
   if (!old)
      return RXFUNC_NOTREG ;

   hashbox = hashvalue( name ) % HASH_MODULUS ;
   free( old->name ) ;
   if (old==extfuncs[hashbox])
      extfuncs[hashbox] = old->prev ;
   else
      old->next->prev = old->prev ;

   if (old->prev)
      old->prev->next = old->next ;

   free( old ) ;
   return RXFUNC_OK ;
}

static int addfunc2( PSZ name, RexxFunctionHandler *EntryPoint )
{
   struct funcbox2 *new=NULL ;
   int hashbox, hash ;

   if (findfunc( name ))
      return RXFUNC_DEFINED ;

   new = malloc( sizeof(struct funcbox2 )) ;
   if (!new)
      return RXFUNC_NOMEM ;

   new->name = malloc( strlen( name )+1 ) ;
   if (!new->name)
   {
      free( new ) ;
      return RXFUNC_NOMEM ;
   }

   strcpy( new->name, name ) ;
   hash = hashvalue( new->name ) ;
   hashbox = hash % HASH_MODULUS ;

   new->entry = EntryPoint ;
   new->next = NULL ;
   new->hash = hash ;
   new->prev = extfuncs[hashbox] ;
   if (extfuncs[hashbox])
      extfuncs[hashbox]->next = new ;
   extfuncs[hashbox] = new ;

   return RXFUNC_OK ;
}


APIRET APIENTRY RexxRegisterFunctionExe( PSZ Name, PFN EntryPoint )
{
   int code=0 ;
   int i=0;
   char *upper_name=NULL;

#if defined(HAVE_WINMULTITHREADING)
   if (!InitFlag)
      StartupRexx( "none" ) ;
#endif
   assert( Name );
   if ((upper_name = (char *)malloc(strlen(Name)+1)) == NULL)
      return(RXFUNC_NOMEM);
   strcpy(upper_name,Name);
   for (i=0;i<(int) strlen(upper_name);i++)
     *(upper_name+i) = (char) toupper(*(upper_name+i));
   code = addfunc2( upper_name, (RexxFunctionHandler*)EntryPoint ) ;
   if (code)
   {
      free(upper_name);
      return code ;
   }

   if (!InitFlag)
      StartupRexx( "none" ) ;

   code = IfcRegFunc( upper_name ) ;
   free(upper_name);
   assert( code==RX_CODE_OK ) ;

   return RXFUNC_OK ;
}

APIRET APIENTRY RexxRegisterFunctionDll( PSZ ExternalName, PSZ LibraryName, PSZ InternalName )
{
#if defined(HAVE_WINMULTITHREADING)
   int i,code,f ;
   HINSTANCE hInst;
   PFN EntryPoint;
   if (!InitFlag)
      StartupRexx( "none" ) ;
#endif
   assert( ExternalName );
   assert( LibraryName );
   assert( InternalName );
#if defined(HAVE_WINMULTITHREADING)
   if (code==0)
   {
      f=IsFuncName(&extfuncName,ExternalName);
      if (f==0)
      {
         AddFuncName(&extfuncName,ExternalName);
         AddFuncName(&extfuncDllName,LibraryName);
         AddFuncName(&extfuncEntryname,InternalName);
      }
   }
   return ((ULONG)code);
#else
   return ((ULONG)rexxsaa_rxfuncdlladd(ExternalName,LibraryName,InternalName));
#endif

}

APIRET APIENTRY RexxQueryFunction( PSZ Name )
{
   assert( Name );
#if defined(HAVE_WINMULTITHREADING)
   if (!InitFlag)
      StartupRexx( "none" ) ;
#endif
   return ( IfcQueryFunc( Name ) ) ? RXFUNC_NOTREG : RXFUNC_OK ;
}


APIRET APIENTRY RexxDeregisterFunction( PSZ Name )
{
   int rc;
#include "multi.h"

   if (!InitFlag)
      StartupRexx( "none" ) ;
#if defined(HAVE_WINMULTITHREADING)
   delfunc2(Name);
   code = IfcDelFunc( Name ) ;
   if (code==0)
   {
      RemFuncName(&extfuncName,Name,&extfuncDllName);
      RemFuncName(&extfuncName,Name,&extfuncEntryname);
      RemFuncName(&extfuncName,Name,&extfuncName);
   }
   return (code) ? RXFUNC_NOTREG : RXFUNC_OK ;
#else
   if ((rc = delfunc2(Name)) != RXFUNC_OK)
      return rc;
   return (IfcDelFunc(Name)) ? RXFUNC_NOTREG : RXFUNC_OK ;
#endif
#include "unmulti.h"
}

int IfcFunctionExit(PSZ Name, int Params, RXSTRING *params, PSZ queuename, PRXSTRING Retstr, int *RCode, char called)
{
   int rc=0 ;
#ifdef UNION_EXIT
   EXIT ParBox ;
#else
   RXFNCCAL_PARM fnccal;
   PUCHAR parm=NULL;
#endif
#if defined(HAVE_WINMULTITHREADING)
   struct ExitHandlers *CurrentHandlers=NULL ;
#include "multi.h"
   CurrentHandlers=(struct ExitHandlers *)CH;
#endif

   if ( CurrentHandlers
   &&   CurrentHandlers->Handlers[RXFNC] )
   {
#ifdef UNION_EXIT
      ParBox.fnccal.rxfnc_flags.rxfferr = 0;
      ParBox.fnccal.rxfnc_flags.rxffnfnd = 0;
      ParBox.fnccal.rxfnc_flags.rxffsub = (called) ? 1 : 0;
      ParBox.fnccal.rxfnc_name = (unsigned char *)Name;
      ParBox.fnccal.rxfnc_namel = strlen(Name);
      ParBox.fnccal.rxfnc_que = (unsigned char *)queuename;
      ParBox.fnccal.rxfnc_quel = strlen(queuename);
      ParBox.fnccal.rxfnc_argc = Params;
      ParBox.fnccal.rxfnc_argv = params;
      ParBox.fnccal.rxfnc_retc.strptr = Retstr->strptr;
      ParBox.fnccal.rxfnc_retc.strlength = Retstr->strlength;
      rc = (*(CurrentHandlers->Handlers[RXFNC]))(RXFNC, RXFNCCAL, &ParBox);
      assert( rc==RXEXIT_HANDLED || rc==RXEXIT_NOT_HANDLED ||
              rc==RXEXIT_RAISE_ERROR ) ;
      if (rc == RXEXIT_HANDLED)
      {
         if (ParBox.fnccal.rxfnc_flags.rxfferr)
            *RCode = RXFLAG_ERROR ;
         else if (ParBox.fnccal.rxfnc_flags.rxffnfnd)
            *RCode = RXFLAG_FAILURE ;
         else
            *RCode = RXFLAG_OK;
      }
      return(rc);
#else
      fnccal.rxfnc_flags.rxfferr = 0;
      fnccal.rxfnc_flags.rxffnfnd = 0;
      fnccal.rxfnc_flags.rxffsub = (called) ? 1 : 0;
      fnccal.rxfnc_name = (unsigned char *)Name;
      fnccal.rxfnc_namel = (USHORT) strlen(Name);
      fnccal.rxfnc_que = (unsigned char *)queuename;
      fnccal.rxfnc_quel = (USHORT) strlen(queuename);
      fnccal.rxfnc_argc = (USHORT) Params;
      fnccal.rxfnc_argv = params;
      fnccal.rxfnc_retc.strptr = Retstr->strptr;
      fnccal.rxfnc_retc.strlength = Retstr->strlength;
      parm = (PUCHAR)&fnccal;
      rc = (*(CurrentHandlers->Handlers[RXFNC]))(RXFNC, RXFNCCAL, parm);
      assert( rc==RXEXIT_HANDLED || rc==RXEXIT_NOT_HANDLED ||
              rc==RXEXIT_RAISE_ERROR ) ;
      if (rc == RXEXIT_HANDLED)
      {
/*         fnccal = (RXFNCCAL_PARM *)parm; */
         if (fnccal.rxfnc_flags.rxfferr)
            *RCode = RXFLAG_ERROR ;
         else if (fnccal.rxfnc_flags.rxffnfnd)
            *RCode = RXFLAG_FAILURE ;
         else
            *RCode = RXFLAG_OK;
      }
      return(rc);
#endif
   }
   else
   {
      return (RXEXIT_NOT_HANDLED);
   }
#include "unmulti.h"
}


int IfcExecFunc( PFN Func, PSZ Name, int Params, int *Lengths, char *Strings[],
                 int *RetLength, char **RetString, int *RC, char exitonly, char called )
{
#if defined(HAVE_WINMULTITHREADING)
   ULONG (* APIENTRY SystemFunc)(CHAR *,ULONG,RXSTRING *,CHAR *,RXSTRING *);
#endif
   static char foo='x' ;
   struct funcbox2 *fptr=NULL ;
   int i=0, length=0, rc=0, RCode=0 ;
   static char *killme=NULL ;
   RXSTRING *params=NULL ;
   RXSTRING retstr ;
   static char retdata[RXAUTOBUFLEN] ;      /* MH 26-12-95 */

   assert( Name ) ;
   assert( Params >= 0 ) ;

   if (killme)
   {
#if defined(HAVE_WIN32GUI)
      GlobalFree((HGLOBAL)killme);
#else
      free( killme ) ;
#endif
      killme = NULL ;
   }

   params = malloc( sizeof(RXSTRING)*Params ) ;
   for (i=0; i<Params; i++)
   {
      length = Lengths[i] ;
      if (length==RX_NO_STRING)
      {
         params[i].strptr = NULL ;
         params[i].strlength = 0 ;
      }
      else if (length==0)
      {
         params[i].strptr = &foo ;
         params[i].strlength = 0 ;
      }
      else
      {
         assert( length>0 ) ;
         params[i].strptr = Strings[i] ;
         params[i].strlength = length ;
      }
   }

   retstr.strptr = retdata ;
   retstr.strlength = RXAUTOBUFLEN;        /* MH 26-12-95 */

   rc = IfcFunctionExit( Name, Params, params, "default", &retstr, &RCode, called );
   switch(rc)
   {
      case RXEXIT_NOT_HANDLED:
         if (exitonly)
         {
           *RC = ERR_ROUTINE_NOT_FOUND;
         }
         else
         {
            if (Func == NULL)
            {
               if ((fptr=findfunc( Name )) == NULL)
                  return RX_CODE_NOSUCH ;
#if defined(HAVE_WINMULTITHREADING)
               SystemFunc=(ULONG (* APIENTRY)(CHAR *,ULONG,RXSTRING *,CHAR *,RXSTRING *))fptr->entry;
               rc = (*(SystemFunc))( Name, Params, params, "default", &retstr ) ;
#else
               rc = (*(fptr->entry))( (PUCHAR)Name, Params, params, "default", &retstr ) ;
#endif
            }
            else
            {
#if defined(HAVE_WINMULTITHREADING)
               SystemFunc=(ULONG (* APIENTRY)(CHAR *,ULONG,RXSTRING *,CHAR *,RXSTRING *))Func;
               rc = (*(SystemFunc))( Name, Params, params, "default", &retstr ) ;
#else
               rc = (*(Func))( Name, Params, params, "default", &retstr ) ;
#endif
            }
            if (rc)
               *RC = ERR_INCORRECT_CALL;
            else
               *RC = 0;
         }
         break;
      case RXEXIT_HANDLED:
         if (RCode == RXFLAG_ERROR)
            *RC = ERR_INCORRECT_CALL;
         else if (RCode == RXFLAG_FAILURE)
            *RC = ERR_ROUTINE_NOT_FOUND;
         else
            *RC = 0;
         break;
      case RXEXIT_RAISE_ERROR:
         *RC = ERR_SYSTEM_FAILURE;
         break;
   }

/* for (i=0; i<Params; i++)
      if (params[i].strptr && params[i].strlength)
         free( params[i].strptr ) ;
*/
   free( params ) ;

   if (!(*RC) && retstr.strptr)
   {
      *RetLength = retstr.strlength ;
      *RetString = retstr.strptr ;
   }
   else
      *RetLength = RX_NO_STRING ;

   if (retstr.strptr && retstr.strptr != retdata)
      killme = retstr.strptr ;

   return RX_CODE_OK ;
}

#if 0
int IfcExecFuncDll( PFN Func, PSZ Name, int Params, int *Lengths, char *Strings[],
                 int *RetLength, char **RetString, int *RC )
{
   static char foo='x' ;
   struct funcbox *fptr=NULL ;
   int i=0, length=0, rc=0, RCode=0 ;
   static char *killme=NULL ;
   RXSTRING *params=NULL ;
   RXSTRING retstr ;
   static char retdata[RXAUTOBUFLEN] ;      /* MH 26-12-95 */

   assert( Func ) ;
   assert( Params >= 0 ) ;

   if (killme)
   {
      free( killme ) ;
      killme = NULL ;
   }

   params = malloc( sizeof(RXSTRING)*Params ) ;
   for (i=0; i<Params; i++)
   {
      length = Lengths[i] ;
      if (length==RX_NO_STRING)
      {
         params[i].strptr = NULL ;
         params[i].strlength = 0 ;
      }
      else if (length==0)
      {
         params[i].strptr = &foo ;
         params[i].strlength = 0 ;
      }
      else
      {
         assert( length>0 ) ;
         params[i].strptr = Strings[i] ;
         params[i].strlength = length ;
      }
   }

   retstr.strptr = retdata ;
   retstr.strlength = RXAUTOBUFLEN;        /* MH 26-12-95 */

   rc = IfcFunctionExit( Name, Params, params, "default", &retstr, &RCode );
   switch(rc)
     {
      case RXEXIT_NOT_HANDLED:
      *RC = (*(Func))( Name, Params, params, "default", &retstr ) ;
           break;
      case RXEXIT_HANDLED:
           if (RCode == RXFLAG_ERROR)
              *RC = 40;
           else if (RCode == RXFLAG_FAILURE)
              *RC = ERR_ROUTINE_NOT_FOUND;
           else
              *RC = 0;
           break;
      case RXEXIT_RAISE_ERROR:
           *RC = 48;
           break;
     }

/* for (i=0; i<Params; i++)
      if (params[i].strptr && params[i].strlength)
         free( params[i].strptr ) ;
*/
   free( params ) ;

   if (!(*RC) && retstr.strptr)
   {
      *RetLength = retstr.strlength ;
      *RetString = retstr.strptr ;
   }
   else
   {
      *RetLength = RX_NO_STRING ;
   }

   if (retstr.strptr && retstr.strptr != retdata)
      killme = retstr.strptr ;

   return RX_CODE_OK ;
}
#endif

int APIENTRY IfcStartRexx( int setting )
{
#if defined(HAVE_WINMULTITHREADING)
   int i,r;
   DWORD Size;
   globalext *G;
   if (InitFlag!=0)
   {
      G=&SG;
      r=TlsSetValue(dwsysteminfo, (void *)G);
      return 0;
   }
   if (setting==0) return 0;
   if (setting==2)
   {
     StartupRexx( "none" ) ;
     setting=1;
   }
   else
   {
     InitFlag=1;
     InitG2(&SG,1);
   }
#else
 InitFlag = Interpreting = setting;
#endif
 return(0);
}

int IfcHaveFunctionExit(void)
{
#if defined(HAVE_WINMULTITHREADING)
   struct ExitHandlers *CurrentHandlers=NULL ;
# include "multi.h"
   CurrentHandlers=(struct ExitHandlers *)CH;
#endif
   if ( CurrentHandlers
   &&   CurrentHandlers->Handlers[RXFNC] )
     return 1;
   else
     return 0;
#include "unmulti.h"
}

/* ============================================================= */
/* Asynchronous Rexx API interface */

APIRET APIENTRY RexxSetHalt(
   LONG dummyProcess,
   LONG dummyThread )
{
   /*
    * Perform sanity check on the parameters; is process id me ?
    */
   set_rexx_halt();
   return RXARI_OK ;
}


#if defined(HAVE_WINMULTITHREADING)
/* ============================================================= */
/* Windows multi-threading code */

int __stdcall InitG(void)
{
    IfcStartRexx(1);
    return 0;
}
int APIENTRY InitG2(globalext *G,int setting)
{
   int rc=0, Maj=0, Min=0 ;
   int i;
   DWORD Size;
   BOOL r;
   if(dwsysteminfo==-1L)
     {
      dwsysteminfo = TlsAlloc();
     }
   if(setting!=3)
      {
      Size=sizeof(struct globalvar);
      memset((void *)G,(int)0,(size_t)Size);
      }
   G->yy_init = 1;  // whether we need to initialize
   G->kill_next_space=1 ;
   G->extnextline = -1;
   G->linenr=1 ;
   G->loopcnt=1 ;
   G->lasttracedline = -1 ;
   G->nextstart=1 ;
   G->nextline=1 ;
   G->var_rc.len=2;
   G->var_rc.max=2;
#ifdef CHECK_MEMORY
   G->var_rc.value="RC";
#else
   strcpy(G->var_rc.value,"RC");
#endif
   G->current_valid=1 ;
   G->next_current_valid=2 ;
   for(i=0;i<6;i++)
      {
      G->signalnames[i]=signalnames[i] ;
      }
   for(i=0;i<7;i++)
      {
      G->WeekDays[i]=WeekDays[i] ;
      }
   for(i=0;i<12;i++)
      {
      G->months[i]=months[i] ;
      G->DaysInYear[i]=DaysInYear[i] ;
      G->MonthDays[i]=MonthDays[i] ;
      }
   if(G==&SG)
      {
      AllocShared();
      }
   r=TlsSetValue(dwsysteminfo, (void *)G);
   /*
    * First parameter ignored, but keep it for compatibility
    */
   if(setting==2)
      {
      rc = IfcStartUp( NULL, &Maj, &Min ) ;

      if (Maj != RXVERSION_MAJ || Min != RXVERSION_MIN)
      return 1 ;
      }
   return 0;
}
void AddFunctions(void)
{
int i;
//re-register any pre-registered functions
   RexxRegisterFunctionDll("RXFUNCADD","REXX","RexxAddFunctionDll");
   RexxRegisterFunctionDll("RXFUNCDROP","REXX","RexxDropFunctionDll");
   RexxRegisterFunctionDll("RXFUNCQUERY","REXX","RexxQueryFunctionDll");
   for(i=3;i<noexternfunc;i++)
      {
      RexxRegisterFunctionDll(GetFuncName(&extfuncName,i),GetFuncName(&extfuncDllName,i),GetFuncName(&extfuncEntryname,i));
      }
}
void AddFuncName(char **p,char * Name)
{
int total,size;
char *r;
char *o;
char *q;
char N[20];
   o=*p;
   q=o;
   size=strlen(q);
   total=0;
   while(size!=0)
      {
      total+=size+1;
      q+=size+1;
      size=strlen(q);
      }
   size=total+strlen(Name)+2;
   r=ReAllocShared(o,size);
   o=r;
   r+=total;
   strcpy(r,Name);
   r+=strlen(Name)+1;
   *r='\0';
   *p=o;
   if(p==&extfuncName)
   noexternfunc++;
   return;
}
BOOL IsFuncName(char **pp,char * Name)
{
BOOL f;
int total,size;
char *r;
char *p;
   f=0;
   p=*pp;
   if(!strcmp(Name,p))
      return 1;
   size=strlen(p);
   total=0;
   while((size!=0) && (f==0))
      {
      total+=size+1;
      p+=size+1;
      size=strlen(p);
      if(!strcmp(Name,p))
         f=1;
      }
   return f;
}
char *GetFuncName(char **pp,int i)
{
BOOL f;
int size;
char *r;
char *p;
   f=0;
   p=*pp;
   if(i==0)
      return p;
   size=strlen(p);
   while((size!=0) && (i>0))
      {
      p+=size+1;
      size=strlen(p);
      i--;
      }
   return p;
}
void RemFuncName(char **pp,char * Name,char **qq)
{
BOOL f;
int total,size,total1,total2,cnt;
char *o;
char *r;
char *s;
char *p;
char *q;
   p=*pp;
   q=*qq;
   f=0;
   cnt=0;
   if(!strcmp(Name,p))
     {
     f=1;
     }
   if(f==0)
      {
      size=strlen(p);
      while((size!=0) && (f==0))
         {
         p+=size+1;
         cnt++;
         size=strlen(p);
         if(!strcmp(Name,p))
            f=1;
         }
      }
   if(f==0)return;
   o=q;
   total=0;
   while(cnt>0)
      {
      size=strlen(q);
      total+=size+1;
      q+=size+1;
      cnt--;
      }
   total1=total;
   r=q;
   size=strlen(q);
   q+=size+1;
   size=strlen(q);
   s=q;
   total=0;
   while(size!=0)
      {
      total+=size+1;
      q+=size+1;
      size=strlen(q);
      }
   total2=total;
   memcpy(r,s,total2);
   size=total1+total2+1;
   r=ReAllocShared(o,size);
   o=r;
   r+=size-1;
   *r='\0';
   *qq=o;
   if(qq==&extfuncName)
   noexternfunc--;
   return;
}
void AllocShared(void)
{
int size;
char *p;
      if(extfuncName==0L)
         {
         hextfuncName=OpenFileMapping(FILE_MAP_ALL_ACCESS,0L,(LPCTSTR)"REXXEXTFNCTS");
         if (hextfuncName==0L)
            {
            hextfuncName = CreateFileMapping((HANDLE)INFINITE,NULL,PAGE_READWRITE,0,25L,"REXXEXTFNCTS");
            }
         extfuncName = (char *)MapViewOfFile(hextfuncName, FILE_MAP_ALL_ACCESS, 0, 0, 0);
         p=extfuncName;
         size=strlen(p);
         noexternfunc=0;
         while(size!=0)
            {
            noexternfunc++;
            p+=size+1;
            size=strlen(p);
            }
         }
      if(extfuncDllName==0L)
         {
         hextfuncDllName=OpenFileMapping(FILE_MAP_ALL_ACCESS,0L,(LPCTSTR)"REXXEXTFNCTD");
         if (hextfuncDllName==0L)
            {
            hextfuncDllName = CreateFileMapping((HANDLE)INFINITE,NULL,PAGE_READWRITE,0,25L,"REXXEXTFNCTD");
            }
         extfuncDllName = (char *)MapViewOfFile(hextfuncDllName, FILE_MAP_ALL_ACCESS, 0, 0, 0);
         }
      if(extfuncEntryname==0L)
         {
         hextfuncEntryname=OpenFileMapping(FILE_MAP_ALL_ACCESS,0L,(LPCTSTR)"REXXEXTFNCTE");
         if (hextfuncEntryname==0L)
            {
            hextfuncEntryname = CreateFileMapping((HANDLE)INFINITE,NULL,PAGE_READWRITE,0,25L,"REXXEXTFNCTE");
            }
         extfuncEntryname = (char *)MapViewOfFile(hextfuncEntryname, FILE_MAP_ALL_ACCESS, 0, 0, 0);
         }
}
char *ReAllocShared(char *o,int size)
{
char *r;
      UnmapViewOfFile(o);
      if(extfuncName==o)
         {
         hextfuncName = CreateFileMapping((HANDLE)INFINITE,NULL,PAGE_READWRITE,0,size,"REXXEXTFNCTS");
         if (hextfuncName!= NULL)
            {
            r = (char *)MapViewOfFile(hextfuncName, FILE_MAP_ALL_ACCESS, 0, 0, 0);
            }
         }
      else if(extfuncDllName==o)
         {
         hextfuncDllName = CreateFileMapping((HANDLE)INFINITE,NULL,PAGE_READWRITE,0,size,"REXXEXTFNCTD");
         if (hextfuncDllName!= NULL)
            {
            r = (char *)MapViewOfFile(hextfuncDllName, FILE_MAP_ALL_ACCESS, 0, 0, 0);
            }
         }
      else if(extfuncEntryname==o)
         {
         hextfuncEntryname = CreateFileMapping((HANDLE)INFINITE,NULL,PAGE_READWRITE,0,size,"REXXEXTFNCTE");
         if (hextfuncEntryname!= NULL)
            {
            r = (char *)MapViewOfFile(hextfuncEntryname, FILE_MAP_ALL_ACCESS, 0, 0, 0);
            }
         }
      return r;
}
int __stdcall InitG3(void)
{
    IfcStartRexx(2);
    return 0;
}

#endif
