/****************************************************************/
/*                                                              */
/*      MODUL:  mouse.c                                         */
/*                                                              */
/*      FUNKTIONEN:                                             */
/*       - Fenster ermitteln, in dem Position liegt (pos_to_win)*/
/*       - Zum Fenster wechseln, in dem Position liegt (sw_pos) */
/*       - Liegt Position innerhalb von akt_winp? (in_akt_win)  */
/*       - Mauscursor verstecken und wieder zeigen (hide_show)  */
/*       - Maus-Status lesen (get_mouse)                        */
/*       - Fenster mit Maus bewegen (mouse_movewin)             */
/*       - Fenstergre mit Maus ndern (mouse_sizewin)         */
/*       - Bildschirm gem Mausposition scrollen (check_scroll)*/
/*       - Cursor setzen oder Block (set_cursor_or_mark_block)  */
/*       - Mausposition in Schaltfache wandeln (pos_to_button) */
/*       - Fenster mit Maus schlieen (mouse_win_zu)            */
/*       - Mit Maus nach rechts scrollen (msr)                  */
/*       - Mit Maus nach links scrollen (msl)                   */
/*       - Mit Maus nach oben scrollen (msu)                    */
/*       - Mit Maus nach unten scrollen (msd)                   */
/*       - Maus-Routine (mouse_routine)                         */
/*       - Maus-Routine fr Funktion ja_nein (jn_mouse_routine) */
/*       - Mouse-Interrupt aktivieren/sperren (mouse_jn_init)   */
/*       - Mouse-Interrupt aktivieren/sperren (set_mouse_int)   */
/*       - Maus initialisieren (init_mouse)                     */
/*                                                              */
/****************************************************************/

#pragma inline

#include "defs.h"
#include <dos.h>

typedef struct mouse_struct
{
  int button_status,  /* Knopfstatus: Bit 0 links, Bit 1 rechts, Bit 2 Mitte */
      x,              /* X-Position (0..COLS-1) */
      y;              /* Y-Position (0..LINES-1) */
} mouse_type;

extern (*funktion[])(),
       do_win_zu();
extern short int mc_index;
extern comm_typ *comm_feld;

int mouse_jn;         /* Variable fr Bedienung der Funktion ja_nein mit
			 der Maus. Kann die Werte NO_KLICK, KLICK_RIGHT und
			 KLICK_LEFT annehmen. */

/******************************************************************************

 Funktion     : Fenster ermitteln, in dem Position liegt (pos_to_win)
 --------------

 Parameter    : x           :
                  Typ          : int
                  Wertebereich : 0-COLS
                  Bedeutung    : X-Achse der Position

                y           :
                  Typ          : int
                  Wertebereich : 0-LINES
                  Bedeutung    : Y-Achse der Position

 Ergebnis     :
*                  Typ          : win_typ *
*                  Wertebereich : Pointer auf Fensterstruktur
*                  Bedeutung    : Zeigt auf das sichtbare Fenster, in dem die
*                                 angegebene Position liegt
*
* Beschreibung : Es werden in der Reihenfolge. in der sie beim Refresh
*                gezeichnet wrden, alle Fenster durchgegangen und der Zeiger
*                auf das Fenster gemerkt, falls die angegebene Position in
*                dem getesteten Fenster liegt. So erhlt man das Fenster,
*                in dem die bergebene Position liegt.
*
******************************************************************************/

win_typ *pos_to_win(x,y)
int x,y;
{
  win_typ *w,          /* Pointer zum Durchlaufen der Fensterliste */
	  *res = NULL; /* Pointe auf Fenster, in dem Position liegt */

  for(w=akt_winp->next->next; w != akt_winp->next; w=w->next)
    /* Alle Fenster durchlaufen */
    if(x >= w->x && x <= w->x + w->dx + 1
    && y >= w->y && y <= w->y + w->dy + 1)
      res = w;  /* Position liegt im Fenster, Fenster merken */
  return(res);
}

/******************************************************************************
*
* Funktion     : Zum Fenster wechseln, in dem Position liegt (sw_pos)
* --------------
*
* Parameter    : x           :
*                  Typ          : int
*                  Wertebereich : 0-COLS
*                  Bedeutung    : X-Achse der Position
*
*                y           :
*                  Typ          : int
*                  Wertebereich : 0-LINES
*                  Bedeutung    : Y-Achse der Position
*
* Ergebnis     :
*                  Typ          : int
*                  Wertebereich : TRUE, FALSE
*                  Bedeutung    : TRUE: Fenster gefunden, FALSE: nicht gefunden
*
* Beschreibung : Mittels der Funktion pos_to_win wird das Fenster ermittelt,
*                in dem die Position liegt. Wird ein solches Fenster gefunden,
*                so wird es zum aktuellen gemacht und TRUE geliefert, ansonsten
*                wird FALSE geliefert.
*
******************************************************************************/

int sw_pos(x,y)
int x,y;
{
  win_typ *w, /* Zeiger auf zu aktivierendes Fenster */
	  *dummy;

  if(w = pos_to_win(x,y))  /* Liegt die Position in einem sichtbaren Fenster ? */
  {
    if(w != akt_winp)
    {
      dummy = akt_winp->next;   /* Dann gefundenes Fenster zum aktuellen */
      akt_winp->next = w;       /* machen, indem man es vor dummy in     */
      w->prev->next = w->next;  /* die Fensterliste einhaengt.           */
      w->next->prev = w->prev;
      w->prev = akt_winp;
      w->next = dummy;
      dummy->prev = w;
      akt_winp = w;
    }
    return(TRUE);
  }
  return(FALSE); /* Position lag nicht in einem Fenster */
}

/******************************************************************************
*
* Funktion     : Liegt Position innerhalb von akt_winp? (in_akt_win)
* --------------
*
* Parameter    : x           :
*                  Typ          : int
*                  Wertebereich : 0-MAXINT
*                  Bedeutung    : X-Position, die zu testen ist
*
*                y           :
*                  Typ          : int
*                  Wertebereich : 0-MAXINT
*                  Bedeutung    : Y-Position, die zu testen ist
*
* Ergebnis     :
*                  Typ          : int
*                  Wertebereich : TRUE, FALSE
*                  Bedeutung    : TRUE: Position liegt im aktuellen Fenster
*                                 FALSE: Position liegt nicht im aktuellen
*                                        Fenster
*
* Beschreibung : Die bergebenen Koordinaten werden mit den Koordinaten
*                des aktuellen Fensters verglichen. Liegen sie innerhalb,
*                also in der Textflche, so wird TRUE, sonst FALSE ge-
*                liefert.
*
******************************************************************************/

int in_akt_win(x,y)
int x,y;
{
  return(x > akt_winp->x && x <= akt_winp->x+akt_winp->dx
  && y > akt_winp->y && y <= akt_winp->y+akt_winp->dy);
}

/******************************************************************************
*
* Funktion     : Mauscursor verstecken und wieder zeigen (hide_show)
* --------------
*
* Parameter    : hs          :
*                  Typ          : int
*                  Wertebereich : MOUSE_SHOW, MOUSE_HIDE,
*                                 MOUSE_SHOW | MOUSE_HIDE
*                  Bedeutung    : MOUSE_HIDE : Mauszeiger verstecken
                                 MOUSE_SHOW : Mauszeiger anzeigen
*
* Beschreibung : Der Mauscursor wird mit der Funktion 2 des Mausinterrupts
*                versteckt, falls MOUSE_HIDE gesetzt war, und anschlieend
*                mit der Funktion 1 wieder dargestellt, falls MOUSE_SHOW
*                gesetzt war. War der Mauscursor also vorher eingeschal-
*                tet, so ist dann garantiert, da er auch sichtbar ist.
*                Das ist wichtig, falls nach einem Refresh der Mauscursor
*                sichtbar sein soll.
*
******************************************************************************/

void hide_show(hs)
int hs;
{
  union  REGS  regs;

  if(hs & MOUSE_HIDE)
  {
    regs.x.ax = 2;
    int86(51,&regs,&regs);
  }
  if(hs & MOUSE_SHOW)
  {
    regs.x.ax = 1;
    int86(51,&regs,&regs);
  }
}

/*******************************************************************************

* Funktion     : Maus-Status lesen (get_mouse)
* --------------
*
* Ergebnis     :
*                  Typ          : *mouse_type
*                  Wertebereich : Pointer auf Maus-Struktur
*                  Bedeutung    : Zeigt auf Struktur, die die aktuellen
*                                 Mausdaten enthlt.
*
* Beschreibung : Mittels der Funktion 3 des Mausinterrupts wird der Mausstatus
*                eingelesen.
*
******************************************************************************/

mouse_type *get_mouse()
{
  static mouse_type m; /* zur Ablage des Maus-Status */
  union  REGS  regs;

  regs.x.ax  = 3;
  int86(51,&regs,&regs);
  m.button_status = regs.x.bx;
  m.x             = regs.x.cx / 8;
  m.y             = regs.x.dx / 8;
  return(&m);
}

/******************************************************************************
*
* Funktion     : Fenster mit Maus bewegen (mouse_movewin)
* --------------
*
* Parameter    : m           :
*                  Typ          : mouse_type *
*                  Wertebereich : Pointer aus Mausstruktur
*                  Bedeutung    : Mauszustand mit Position und Knopfdrcken
*
* Beschreibung : Das aktuelle Fenster wird durch die Maus verschoben.
*                Die Mauskoordinaten werden solange geprft, bis der
*                linke Mausknopf nicht mehr gedrckt ist. Dann wird
*                das Fenster in seiner dortigen Position gezeichnet.
*
******************************************************************************/

void mouse_movewin(m)
mouse_type *m;
{
  int old_x = m->x, /* Ablage fr alte Mauskoordinaten */
      old_y = m->y,
      moved = FALSE;

  /* Wenn rechter Mausknopf gedrckt wurde, dann Fenster in den Hintergrund
     schieben. Bei linkem Mausknopf evtl. Fenster verschieben. */
  if(m->button_status & 2) /* rechter Knopf ? */
  {
    hide_show(MOUSE_HIDE); /* Mauszeiger verstecken */
    push_win_back(); /* Fenster zum hintersten machen */
    do_refresh();
    hide_show(MOUSE_SHOW); /* Mauszeiger wieder anzeigen */
  }
  else /* linker Mausknopf war gedrckt */
  {
    /* warten, bis entweder der Mausknopf losgelassen wird, oder */
    /* die Maus bewegt wird. */
    while((m=get_mouse())->button_status & 1)
    {
      if(m->x != old_x || m->y != old_y)  /* Maus wurde mit gedrcktem Knopf */
      {                                   /* bewegt. */
	hide_show(MOUSE_HIDE);
	if(!moved)
	{
	  moved = TRUE;
	  werase (akt_winp->winp);  /* Fensterinhalt auf Bildschirm loeschen */
	  wrefresh(akt_winp->winp); /* damit kein Fehler beim Verschieben passiert */
	  cpwins2stdscr(); /* Alle Fenster ausser akt. nach stdscr kopieren */
	}
	else
	  eckenhw();                 /* Ecken des Fensters demarkieren      */
	if(m->x > old_x)
	  win_right(m->x - old_x);
	else
	  win_left(old_x - m->x);
	if(m->y > old_y)
	  win_down(m->y - old_y);
	else
	  win_up(old_y - m->y);
	old_x = m->x; /* Mauskoordinaten merken */
	old_y = m->y;
	eckenhw();                   /* Ecken markieren */
	hide_show(MOUSE_SHOW);
      }
    }
    if(moved)    /* Falls Fenster verschoben wurde, Ecken wieder lschen */
    {            /* und Fenster neu zeichnen */
      hide_show(MOUSE_HIDE);
      eckenhw();
      erase();
      refresh();
      mvwin (akt_winp->winp,akt_winp->y,akt_winp->x); /* Fenster verschieben */
      do_refresh();                   /* Alle Fenster neu zeichnen           */
      hide_show(MOUSE_SHOW);
    }
  }
}

/******************************************************************************
*
* Funktion     : Fenstergre mit Maus ndern (mouse_sizewin)
* --------------
*
* Parameter    : m           :
*                  Typ          : mouse_type *
*                  Wertebereich : Pointer aus Mausstruktur
*                  Bedeutung    : Mauszustand mit Position und Knopfdrcken
*
* Beschreibung : Das aktuelle Fenster wird durch die Maus vergrert
*                oder verkleinert.
*                Die Mauskoordinaten werden solange geprft, bis der
*                linke Mausknopf nicht mehr gedrckt ist. Dann wird
*                das Fenster in seiner dortigen Gre gezeichnet.
*
******************************************************************************/

void mouse_sizewin(m)
mouse_type *m;
{
  int old_x = m->x, /* Ablage fr alte Mauskoordinaten */
      old_y = m->y,
      moved = FALSE;

  /* warten, bis entweder der Mausknopf losgelassen wird, oder */
  /* die Maus bewegt wird. */
  while((m=get_mouse())->button_status & 3)
  {
    if(m->x != old_x || m->y != old_y)  /* Maus wurde mit gedrcktem Knopf */
    {                                   /* bewegt. */
      hide_show(MOUSE_HIDE);
      if(!moved)
      {
	moved = TRUE;
	werase (akt_winp->winp);  /* Fensterinhalt auf Bildschirm loeschen */
	wrefresh(akt_winp->winp); /* damit kein Fehler beim Verschieben passiert */
	cpwins2stdscr(); /* Alle Fenster ausser akt. nach stdscr kopieren */
      }
      else
	eckenhw();                 /* Ecken des Fensters demarkieren      */
      if(m->x > old_x)
	size_right(m->x - old_x);
      else
	size_left(old_x - m->x);
      if(m->y > old_y)
	size_down(m->y - old_y);
      else
	size_up(old_y - m->y);
      old_x = m->x; /* Mauskoordinaten merken */
      old_y = m->y;
      eckenhw();                   /* Ecken markieren */
      hide_show(MOUSE_SHOW);
    }
  }
  if(moved)    /* Falls Fenster verschoben wurde, Ecken wieder lschen */
  {            /* und Fenster neu zeichnen */
    hide_show(MOUSE_HIDE);
    eckenhw();
    erase();                  /* gedrueckt wurde (RETURN) */
    refresh();                /* Fensterinhalt loeschen */
    delwin(akt_winp->winp);   /* Fenster mit Curses loeschen und neu anlegen */
    akt_winp->winp=newwin(akt_winp->dy+2,akt_winp->dx+2,akt_winp->y,akt_winp->x);
    init_win();               /* Fensterattribute setzen (raw etc.) */

    /* Falls Cursor jetzt ausserhalb des Fensters steht, Cursorposition anpassen */
    if (akt_winp->ws_line+akt_winp->dy <= akt_winp->textline)
      gotox(akt_winp->ws_line+akt_winp->dy-1);
    if (akt_winp->ws_col+akt_winp->dx <= akt_winp->screencol)
      akt_winp->screencol = akt_winp->ws_col+akt_winp->dx-1;
    do_refresh();  /* Alle Fenster neu zeichnen */
    hide_show(MOUSE_SHOW);
  }
}

/******************************************************************************
*
* Funktion     : Bildschirm gem Mausposition scrollen (check_scroll)
* --------------
*
* Parameter    : m           :
*                  Typ          : mouse_type *
*                  Wertebereich : Pointer auf Mausstruktur
*                  Bedeutung    : Mausstatus
*
* Beschreibung : Die Mausposition liegt auerhalb des aktuellen Fensters.
*                Der Fensterinhalt soll in Richtung des Mauscursors an-
*                gepat werden. Steht also beispielsweise die Maus unter-
*                halb des aktuellen Fensters, so soll der Text im Fenster
*                nach oben geschoben werden (damit z.B. auch fensterber-
*                greifende Blcke markiert werden knnen.
*
******************************************************************************/

void check_scroll(m)
mouse_type *m;
{
  /* Zuerst die Verschiebung nach oben/unten testen */
  if(m->y < akt_winp->y+1
  && akt_winp->ws_line) /* Verschiebung des Inhalts nach unten */
  {
    akt_winp->ws_line--; /* dann Nummer der ersten sichtbaren Zeile erhoehen */
    text_down(0);        /* gesamten Fenstertext um 1 Zeile nach unten */
    up();                /* Cursor um 1 Zeile nach oben bewegen */
    setz_cursor(W_AKT);       /* Cursor an richtige Position setzen */
  }
  else
    if(m->y > akt_winp->y+akt_winp->dy /* Verschiebung der Inhalts nach */
    && akt_winp->ws_line < akt_winp->maxline) /* oben */
    {
      akt_winp->ws_line++; /* dann Nummer der ersten sichtbaren Zeile erhoehen */
      text_up(0);          /* gesamten Fenstertext um 1 Zeile nach oben */
      down();              /* Cursor um 1 Zeile nach unten bewegen */
      setz_cursor(W_AKT);       /* Cursor an richtige Position setzen */
    }

  /* Jetzt Verschiebung nach rechts/links testen */
  if(m->x < akt_winp->x+1
  && akt_winp->ws_col)  /* Verschiebung des Inhalts nach rechts */
  {
    text_right();
    left();
    setz_cursor(W_AKT);       /* Cursor an richtige Position setzen */
  }
  else
    if(m->x > akt_winp->x+akt_winp->dx
    && akt_winp->ws_col < MAXLENGTH)
    {
      text_left();
      right();
      setz_cursor(W_AKT);       /* Cursor an richtige Position setzen */
    }
}

/******************************************************************************
*
* Funktion     : Cursor setzen oder Block markieren (set_cursor_or_mark_block)
* --------------
*
* Parameter    : m           :
*                  Typ          : mouse_type *
*                  Wertebereich : Pointer auf Mausstruktur
*                  Bedeutung    : Mauszustand
*
* Beschreibung : Wird der Mausknopf ohne Mausbewegung losgelassen, dann
*                wird lediglich der Cursor auf die Mausposition gesetzt.
*                Wird die Maus jedoch mit gedrcktem Mausknopf bewegt,
*                dann wird damit ein Block markiert.
*
******************************************************************************/

void set_cursor_or_mark_block(m)
mouse_type *m;
{
  int old_x = m->x, /* Ablage fr alte Mauskoordinaten */
      old_y = m->y,
      old_button = m->button_status, /* Zwischenspeicher fr Knpfe */
      moved = FALSE;

  /* zunchst wird der Cursor an die Mausposition gesetzt */
  gotox(akt_winp->ws_line + m->y - 1 - akt_winp->y);
  akt_winp->screencol = akt_winp->ws_col + m->x - 1 - akt_winp->x;
  setz_cursor(W_AKT);
  /* warten, bis entweder der Mausknopf losgelassen wird, oder */
  /* die Maus bewegt wird. */
  while((m=get_mouse())->button_status & 3)
  {
    if(in_akt_win(m->x,m->y))         /* Innerhalb des Fenster ? */
    {
      if(m->x != old_x || m->y != old_y)  /* Maus wurde mit gedrcktem Knopf */
      {                                   /* bewegt: */
	hide_show(MOUSE_HIDE);
	if(!moved) /* Falls erste Bewegung: */
	{
	  moved = TRUE;
	  /* Jetzt Blockanfang auf die alte Cursorposition setzen */
	  do_blstart();
	  /* Dann Cursor auf neue Position setzen */
	  gotox(akt_winp->ws_line + m->y - 1 - akt_winp->y);
	  akt_winp->screencol = akt_winp->ws_col + m->x - 1 - akt_winp->x;
	  setz_cursor(W_AKT);
	}
	else
	{
	  gotox(akt_winp->ws_line + m->y - 1 - akt_winp->y);
	  akt_winp->screencol = akt_winp->ws_col + m->x - 1 - akt_winp->x;
	  setz_cursor(W_AKT);
	}
	if(m->button_status & 1) /* Bei linkem Knopf normaler Block */
	  do_blnormend();
	else                     /* Bei rechtem Knopf rechteckiger Block */
	  do_blrechtend();
	hide_show(MOUSE_SHOW);
      }
    }
    else /* Maus steht auerhalb des Fensters */
    { /* Jetzt mu evtl. der Fensterinhalt gescrollt werden. */
      hide_show(MOUSE_HIDE);
      check_scroll(m);
      hide_show(MOUSE_SHOW);   /* Mauscursor wieder sichtbar machen */
    }
    old_x = m->x; /* Mauskoordinaten merken */
    old_y = m->y;
  }
  if(!moved)
    if(old_button & 2) /* Wenn der Mauszeiger nicht mit */
    { /* gedrcktem Knopf bewegt wurde und der rechte Knopf gedrckt war: */
      hide_show(MOUSE_HIDE);
      if(block_defined()) /* Wenn im aktuellen Fenster ein Block markiert ist, */
      {                    /* und neben dem rechten noch der mittlere Knopf */
	if(old_button & 4) /* gedrckt war, dann Block lschen */
	  do_blweg();
	else             /* Wenn nur der rechte Knopf gedrckt war, */
	{
	  do_blcut();    /* dann diesen Cutten */
	  do_blunmark(); /* und anschlieend unmarkieren */
	}
      }
      else          /* Ist jedoch kein Block markiert, dann Pasten */
	do_blpaste();
      hide_show(MOUSE_SHOW);
    }
    else
      if((old_button & 5) == 5) /* linker und mittlerer Knopf gedrckt ? */
      {
	hide_show(MOUSE_HIDE);
	do_blmove();
	hide_show(MOUSE_SHOW);
      }
}

/******************************************************************************
*
* Funktion     : Mausposition in Schaltfache wandeln (pos_to_button)
* --------------
*
* Parameter    : m           :
*                  Typ          : mouse_type *
*                  Wertebereich : Pointer auf Mausstruktur
*                  Bedeutung    : Mausstatus
*
* Ergebnis     :
*                  Typ          : int
*                  Wertebereich : MOUSE_...
*                  Bedeutung    : Bezeichner der Schaltflche
*
* Beschreibung : Die Mausposition wird mit der Lage der Schaltflchen
*                verglichen und die entsprechende Schaltflche
*                zurckgeliefert.
*
******************************************************************************/

int pos_to_button(m)
mouse_type *m;
{
  if(in_akt_win(m->x,m->y))
    return(MOUSE_TEXT); /* Maus steht im Textbereich */
  if(m->x == akt_winp->x && m->y == akt_winp->y)
    return(MOUSE_CLOSE); /* linke obere Ecke: Schlieknopf */
  if(m->x == akt_winp->x+akt_winp->dx+1 && m->y == akt_winp->y)
    return(MOUSE_TOG_SIZE);
  if(m->x == akt_winp->x+akt_winp->dx+1 
  && m->y == akt_winp->y+akt_winp->dy+1)
    return(MOUSE_SIZE);
  if(m->y == akt_winp->y)
    return(MOUSE_KOPF);
  if(m->x == akt_winp->x+akt_winp->dx+1)
    if(m->y == akt_winp->y+1)
      return(MOUSE_SCROLL_DOWN);
    else
      if(m->y == akt_winp->y+akt_winp->dy)
	return(MOUSE_SCROLL_UP);
  if(m->y == akt_winp->y+akt_winp->dy+1)
    if(m->x == akt_winp->x+1)
      return(MOUSE_SCROLL_RIGHT);
    else
      if(m->x == akt_winp->x+akt_winp->dx)
	return(MOUSE_SCROLL_LEFT);
  return(MOUSE_RAHMEN);
}

/******************************************************************************
*
* Funktion     : Fenster mit Maus schlieen (mouse_win_zu)
* --------------
*
* Beschreibung : Offensichtlich kann whrend der Maus-Routine keine andere
*                Maus-Routine installiert werden. Das hat zur Folge, da
*                beim Ausfhren der Mausfunktion "Fenster schlieen" im
*                Falle der Ausfhrung der Funktion ja_nein dort keine
*                neue Maus-Routine installiert werden kann, wenn die alte
*                Maus-Routine noch luft.
*                Die uerst unschne Lsung, die nach Verbesserung schreit,
*                ist folgende:
*                In den Tastaturpuffer wird der Code fr das Kommando
*                "Fenster schlieen" geschrieben.
*                Dann wird die Maus-Routine verlassen. Whrend der Ausfhrung
*                des Kommandos ist die alte Maus-Routine sowieso gesperrt und
                die neue kann bedenkenlos installiert werden.
*
******************************************************************************/

void mouse_win_zu()
{
  int kom_nr=0, /* Nummer des Kommandos do_win_zu */
      i;        /* Zum Durchlaufen des Kommandostrings */
  short int *key_buff_in = (short int *) 0x41C,
	    *key_buff    = (short int *) 0x41E; /* Tastaturpuffer */

  /* Zuerst den Index des Kommandos "Fenster schlieen" ermitteln */
  while(funktion[kom_nr] != do_win_zu && kom_nr++ <= mc_index);

  /* Jetzt Schreibadresse in Tastaturpuffer ermitteln: */
  key_buff += (*key_buff_in - 0x1E) / 2;
  for(i=0;i<comm_feld[kom_nr].blen;i++)
  {
    *key_buff++ = comm_feld[kom_nr].befehl[i];
    if(key_buff == (short int *) 0x43E) /* Am Ende wrap around */
      key_buff = (short int *) 0x41E;
  }
  *key_buff_in = FP_OFF(key_buff) - 0x400;
}

/******************************************************************************
*
* Funktion     : Mit Maus nach rechts scrollen (msr)
* --------------
*
* Beschreibung : Der Mauszeiger wird versteckt. Dann wird der Text im Fenster
*                nach rechts geschoben und ntigenfalls der Cursor angepat.
*                Dann wird setz_cursor aufgerufem imd der Mauszeiger wieder
*                angezeigt.
*                Der Verschiebe-Vorgang wird solange wiederholt, bis kein
*                Mausknopf mehr gedrckt ist.
*
******************************************************************************/

void msr()
{
  hide_show(MOUSE_HIDE); /* Mauszeiger verstecken */
  while(get_mouse()->button_status & 3)
  {
    text_right();          /* Text nach rechts scrollen */
    /* Falls Cursor am rechten Rand "rausgefallen", dann eins nach links */
    if(akt_winp->screencol >= akt_winp->ws_col + akt_winp->dx)
      left();
    setz_cursor(W_AKT);
  }
  hide_show(MOUSE_SHOW);
}

/******************************************************************************
*
* Funktion     : Mit Maus nach links scrollen (msl)
* --------------
*
* Beschreibung : Der Mauszeiger wird versteckt. Dann wird der Text im Fenster
*                nach links geschoben und ntigenfalls der Cursor angepat.
*                Dann wird setz_cursor aufgerufem imd der Mauszeiger wieder
*                angezeigt.
*                Der Verschiebe-Vorgang wird solange wiederholt, bis kein
*                Mausknopf mehr gedrckt ist.
*
******************************************************************************/

void msl()
{
  hide_show(MOUSE_HIDE); /* Mauszeiger verstecken */
  while(get_mouse()->button_status & 3)
  {
    text_left();          /* Text nach rechts scrollen */
    /* Falls Cursor am linken Rand "rausgefallen", dann eins nach rechts */
    if(akt_winp->screencol < akt_winp->ws_col)
      right();
    setz_cursor(W_AKT);
  }
  hide_show(MOUSE_SHOW);
}

/******************************************************************************
*
* Funktion     : Mit Maus nach oben scrollen (msu)
* --------------
*
* Beschreibung : Der Mauszeiger wird versteckt. Dann wird der Text im Fenster
*                nach oben geschoben und ntigenfalls der Cursor angepat.
*                Dann wird setz_cursor aufgerufem imd der Mauszeiger wieder
*                angezeigt.
*                Der Verschiebe-Vorgang wird solange wiederholt, bis kein
*                Mausknopf mehr gedrckt ist.
*
******************************************************************************/

void msu()
{
  hide_show(MOUSE_HIDE); /* Mauszeiger verstecken */
  while(get_mouse()->button_status & 3 && akt_winp->ws_line < akt_winp->maxline)
  {
    akt_winp->ws_line++; /* dann Nummer der ersten sichtbaren Zeile erhoehen */
    text_up(0);          /* gesamten Fenstertext um 1 Zeile nach oben */
    if(akt_winp->textline < akt_winp->ws_line) /* Falls Cursor in oberster */
      down();            /* Zeile, dann Cursor um 1 Zeile nach unten bewegen */
    setz_cursor(W_AKT);       /* Cursor an richtige Position setzen */
  }
  hide_show(MOUSE_SHOW);
}

/******************************************************************************
*
* Funktion     : Mit Maus nach unten scrollen (msd)
* --------------
*
* Beschreibung : Der Mauszeiger wird versteckt. Dann wird der Text im Fenster
*                nach unten geschoben und ntigenfalls der Cursor angepat.
*                Dann wird setz_cursor aufgerufem imd der Mauszeiger wieder
*                angezeigt.
*                Der Verschiebe-Vorgang wird solange wiederholt, bis kein
*                Mausknopf mehr gedrckt ist.
*
******************************************************************************/

void msd()
{
  hide_show(MOUSE_HIDE); /* Mauszeiger verstecken */
  while(get_mouse()->button_status & 3 && akt_winp->ws_line)
  {
    akt_winp->ws_line--; /* Nummer der ersten sichtbaren Zeile dekrementieren */
    text_down(0);        /* Text ab Zeile 0 (ganzes Fenster) um 1 nach unten  */
    if(akt_winp->textline >= akt_winp->ws_line + akt_winp->dy)
      up(); /* Stand Cursor in letzter Schirmzeile, Cursor 1 Zeile hoch */
    setz_cursor(W_AKT);        /* Cursor an richtige Position */
  }
  hide_show(MOUSE_SHOW);
}

/******************************************************************************
*
* Funktion     : Maus-Routine (mouse_routine)
* --------------
*
* Beschreibung : Diese Routine wird angesrpugen, wenn mit set_mouse_int eine
*                Event-Mask ungleich 0 eingetragen wurde und das entsprechende
*                Ereignis eingetreten ist.
*                Diese Funktion fhrt dann die entsprechenden Mausfunktionen
*                aus.
*
******************************************************************************/

void mouse_routine()
{
  mouse_type *m;    /* Mausstatus */

  asm push ds;
  asm push es;
  asm mov  ax,DGROUP;
  asm mov  es,ax;
  asm mov  ds,ax;

/*  outportb(32,(char) 32); Interrupt fr beendet erklren, (nicht auf 386)*/
	  /* damit z.B. im Schneider PC Maus wieder erkannt wird. */
  check_buff();     /* Pufferinhalt in Text uebernehmen       */
  m = get_mouse();  /* Mauskoordinaten und Knopfstatus einlesen */
  if(pos_to_win(m->x,m->y) != akt_winp) /* Klick auf aktuelles Fenster? */
  {                     /* Nein: */
    hide_show(MOUSE_HIDE); /* Maus solange verstecken */
    kopf(W_NOTAKT);     /* Altes Fenster als inaktiv markieren */
    rahmen(W_NOTAKT);
    wrefresh(akt_winp->winp);
    sw_pos(m->x,m->y);  /* Zum Fenster wechseln */
    show_win(W_AKT);    /* Fenster neu zeichnen */
    hide_show(MOUSE_SHOW); /* Maus wieder anzeigen */
  }
  switch(pos_to_button(m))
  {
    case MOUSE_TEXT        : set_cursor_or_mark_block(m); break;
    case MOUSE_KOPF        : mouse_movewin(m); break;
    case MOUSE_SIZE        : mouse_sizewin(m); break;
    case MOUSE_CLOSE       : mouse_win_zu();
			     break;
    case MOUSE_TOG_SIZE    : hide_show(MOUSE_HIDE);
			     do_toggle_size();
			     hide_show(MOUSE_SHOW);
			     break;
    case MOUSE_SCROLL_UP   : msu(); break;
    case MOUSE_SCROLL_DOWN : msd(); break;
    case MOUSE_SCROLL_RIGHT: msr(); break;
    case MOUSE_SCROLL_LEFT : msl(); break;
  }

  asm pop  es;
  asm pop  ds;
}

/******************************************************************************
*
* Funktion     : Maus-Routine fr Funktion ja_nein (jn_mouse_routine)
* --------------
*
* Beschreibung : Diese Routine wird angesrpugen, wenn mit mouse_jn_init die
*                Maus aktiviert wurde und das entsprechende Ereignis
*                eingetreten ist.
*                Diese Funktion setzt dann gem dem gedrckten Mausknopf
*                die globale Variable mouse_jn auf KLICK_LEFT bzw.
*                KLICK_RIGHT. 
*
******************************************************************************/

void jn_mouse_routine()
{
  int event; /* Fr Ablage des eingetretenen Ereignisses */

  asm push ds;
  asm push es;
  asm mov  bx,DGROUP;
  asm mov  es,bx;
  asm mov  ds,bx;
  event = _AX;
  mouse_jn = event & 8 ? KLICK_RIGHT : KLICK_LEFT;
  asm pop  es;
  asm pop  ds;
}

/******************************************************************************
*
* Funktion     : Mouse-Interrupt aktivieren/sperren (mouse_jn_init)
* --------------
*
* Parameter    : on_off    :
*                  Typ          : int
*                  Wertebereich : TRUE, FALSE
*                  Bedeutung    : TRUE : jn_mouse_routine aktivieren
*                                 FALSE: jn_mouse_routine deaktivieren
*
* Beschreibung : Mittels des Mausinterrupts 51, Funktion 12 wird die
*                Funktion jn_mouse_routine als anzuspringende Funktion
*                eingetragen. Die Event-Mask wird auf rechten Knopf +
*                linken Knopf gesetzt, falls aktiviert wird, auf 0 sonst.
*
******************************************************************************/

void mouse_jn_init(on_off)
int on_off;
{
  union  REGS  regs;
  struct SREGS sregs;

  /* Mauszeiger lschen, falls Maus demaskiert wird, sonst */
  /* Mauszeiger anzeigen */
  if(on_off) /* Bei Initialisierung: */
    mouse_jn = NO_KLICK; /* Variable initialisieren */
  hide_show(on_off ? MOUSE_SHOW : MOUSE_HIDE);
  regs.x.ax  = 12;
  regs.x.cx  = on_off ? MOUSE_MASK : 0;
  regs.x.dx  = FP_OFF(jn_mouse_routine);
  sregs.es   = FP_SEG(jn_mouse_routine);
  int86x(51,&regs,&regs,&sregs);
}

/******************************************************************************
*
* Funktion     : Mouse-Interrupt aktivieren/sperren (set_mouse_int)
* --------------
*
* Parameter    : event_mask :
*                  Typ          : int
*                  Wertebereich : 0-31
*                  Bedeutung    : Bit 0 : Mausbewegung melden
*                                 Bit 1 : Druck linker Mausknopf melden
*                                 Bit 2 : Lsen linker Mausknopf melden
*                                 Bit 3 : Druck rechter Mausknopf melden
*                                 Bit 4 : Lsen linker Mausknopf melden
*
* Beschreibung : Mittels des Mausinterrupts 51, Funktion 12 wird die
*                Funktion mouse_routine als anzuspringende Funktion
*                eingetragen. Die Event-Mask wird gesetzt.
*
******************************************************************************/

void set_mouse_int(event_mask)
int event_mask;
{
  union  REGS  regs;
  struct SREGS sregs;

  /* Mauszeiger lschen, falls Maus demaskiert wird, sonst */
  /* Mauszeiger anzeigen */
  hide_show(event_mask ? MOUSE_SHOW : MOUSE_HIDE);
  regs.x.ax  = 12;
  regs.x.cx  = event_mask; /* Linken Mausknopf melden */
  regs.x.dx  = FP_OFF(mouse_routine);
  sregs.es   = FP_SEG(mouse_routine);
  int86x(51,&regs,&regs,&sregs);
}

/******************************************************************************
*
* Funktion     : Maus initialisieren (init_mouse)
* --------------
*
* Beschreibung : Es werden die Attribute des Maus-Cursors festgelegt, der
*                Maus-Cursor wird eingeschaltet, die Maus-Routine eingebunden
*                und so eingestellt, da sie durch Druck auf den linken
*                Mausknopf aktiviert wird.
*
******************************************************************************/

void init_mouse()
{
  /* Ruft die Maus-Funktion 12 auf, wodurch der Vektor der User-Routine */
  /* auf die Funktion mouse_routine gesetzt werden soll. */
  union  REGS  regs;
  struct SREGS sregs;

  regs.x.ax  = 10;  /* Set Text Cursor */
  regs.x.bx  = 0;   /* Software-Cursor */
  regs.x.cx  = 0x77ff;  /* Screen Mask */
  regs.x.dx  = 0x7700;  /* Cursor Mask */
  int86(51,&regs,&regs);
  regs.x.ax  = 7;   /* Set Minimum and Maximum X-Cursor Position */
  regs.x.cx  = 0;
  regs.x.dx  = 8*(COLS-1);
  int86(51,&regs,&regs);
  regs.x.ax  = 8;   /* Set Minimum and Maximum Y-Cursor Position */
  regs.x.cx  = 0;
  regs.x.dx  = 8*(LINES-1);
  int86(51,&regs,&regs);
}
