/*****************************************************************************
 * $Id: tiff2sff.c,v 1.4 1996/04/11 00:13:45 ak Exp $
 *****************************************************************************
 * $Log: tiff2sff.c,v $
 * Revision 1.4  1996/04/11 00:13:45  ak
 * *** empty log message ***
 *
 * Revision 1.3  1996/04/10 22:58:46  ak
 * CR
 *
 * Revision 1.2  1996/04/10 22:57:18  ak
 * *** empty log message ***
 *
 *****************************************************************************/

static char *rcsid = "$Id: tiff2sff.c,v 1.4 1996/04/11 00:13:45 ak Exp $";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <tiff/tiffio.h>
#include <tiff/t4.h>

typedef unsigned char  u_char;
typedef unsigned short u_short;
typedef unsigned int   u_int;
typedef unsigned long  u_long;

typedef struct {
    char    sff_id[4];
    u_char  version;
    u_char  reserved;
    u_short user_info;
    u_short num_pages;
    u_short first_page;
    u_long  last_page;
    u_long  file_size;
} DocHeader;

typedef struct{
    u_char  vert_res;
    u_char  horiz_res;
    u_char  coding;
    u_char  specials;
    u_short linelen;
    u_short pagelen;
    u_long  prev_page;
    u_long  next_page;
} PageHeader;

typedef struct Row {
    int            size;
    char *         data;
} Row;

typedef struct Page {
    PageHeader     h;
    struct Page *  next;
    Row *          rows;
    u_long         offset;
} Page;

DocHeader	header;
Page *		page0;

/*****************************************************************************/
    
static const u_char zeroruns[256] = {
    8, 7, 6, 6, 5, 5, 5, 5, 4, 4, 4, 4, 4, 4, 4, 4,	/* 0x00 - 0x0f */
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,	/* 0x10 - 0x1f */
    2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,	/* 0x20 - 0x2f */
    2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,	/* 0x30 - 0x3f */
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 0x40 - 0x4f */
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 0x50 - 0x5f */
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 0x60 - 0x6f */
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 0x70 - 0x7f */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 0x80 - 0x8f */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 0x90 - 0x9f */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 0xa0 - 0xaf */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 0xb0 - 0xbf */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 0xc0 - 0xcf */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 0xd0 - 0xdf */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 0xe0 - 0xef */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 0xf0 - 0xff */
};
static const u_char oneruns[256] = {
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 0x00 - 0x0f */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 0x10 - 0x1f */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 0x20 - 0x2f */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 0x30 - 0x3f */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 0x40 - 0x4f */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 0x50 - 0x5f */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 0x60 - 0x6f */
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,	/* 0x70 - 0x7f */
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 0x80 - 0x8f */
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 0x90 - 0x9f */
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 0xa0 - 0xaf */
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,	/* 0xb0 - 0xbf */
    2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,	/* 0xc0 - 0xcf */
    2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,	/* 0xd0 - 0xdf */
    3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,	/* 0xe0 - 0xef */
    4, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 6, 6, 7, 8,	/* 0xf0 - 0xff */
};

/*
 * Find a span of ones or zeros using the supplied
 * table.  The byte-aligned start of the bit string
 * is supplied along with the start+end bit indices.
 * The table gives the number of consecutive ones or
 * zeros starting from the msb and is indexed by byte
 * value.
 */
static int
findspan(u_char **bufp, int bs, int be, u_char const *tab)
{
    u_char *buf = *bufp;
    int bits = be - bs;
    int n, span;

    /*
     * Check partial byte on lhs.
     */
    if (bits > 0 && (n = (bs & 7))) {
	span = tab[(*buf << n) & 0xff];
	if (span > 8-n)		/* table value too generous */
	    span = 8-n;
	if (span > bits)	/* constrain span to bit range */
	    span = bits;
	if (n+span < 8)		/* doesn't extend to edge of byte */
	    goto done;
	bits -= span;
	buf++;
    } else
	span = 0;

    /*
     * Scan full bytes for all 1's or all 0's.
     */
    while (bits >= 8) {
	n = tab[*buf];
	span += n;
	bits -= n;
	if (n < 8)		/* end of run */
	    goto done;
	buf++;
    }

    /*
     * Check partial byte on rhs.
     */
    if (bits > 0) {
	n = tab[*buf];
	span += (n > bits ? bits : n);
    }
done:
    *bufp = buf;
    return span;
}

typedef struct {
    u_int    bit;
    u_int    byte;
    u_char * buf;
} BitBuf;

/*
 * Write a variable-length bit-value to
 * the output stream.  Values are
 * assumed to be at most 16 bits.
 */
static void
putbits(BitBuf *bitbuf, u_int bits, u_int length)
{
    static const int mask[9] = { 0x00, 0x01, 0x03, 0x07, 0x0f, 0x1f, 0x3f, 0x7f, 0xff };
    while (length > bitbuf->bit) {
	bitbuf->buf[bitbuf->byte] |= bits >> (length - bitbuf->bit);
	length -= bitbuf->bit;
	++bitbuf->byte;
	bitbuf->bit = 8;
    }
    bitbuf->buf[bitbuf->byte] |= (bits & mask[length]) << (bitbuf->bit - length);
    bitbuf->bit -= length;
    if (bitbuf->bit == 0) {
	++bitbuf->byte;
	bitbuf->bit = 8;
    }
}

/*
 * Write the sequence of codes that describes
 * the specified span of zero's or one's.  The
 * appropriate table that holds the make-up and
 * terminating codes is supplied.
 */
static void
putspan(BitBuf *bitbuf, int span, tableentry const *tab)
{
    while (span >= 2624) {
	tableentry const *te = &tab[63 + (2560>>6)];
	putbits(bitbuf, te->code, te->length);
	span -= te->runlen;
    }
    if (span >= 64) {
	tableentry const *te = &tab[63 + (span>>6)];
	putbits(bitbuf, te->code, te->length);
	span -= te->runlen;
    }
    putbits(bitbuf, tab[span].code, tab[span].length);
}

static Row
bitmap2fax(u_char *buf, int nbits)
{
    int bs, span;
    BitBuf bitbuf;
    Row row;

    bitbuf.bit  = 8;
    bitbuf.byte = 0;
    bitbuf.buf  = alloca(nbits);
    memset(bitbuf.buf, 0, nbits);

    for (bs = 0;;) {
	span = findspan(&buf, bs, nbits, zeroruns);	/* white span */
	putspan(&bitbuf, span, TIFFFaxWhiteCodes);
	bs += span;
	if (bs >= nbits)
	    break;
	span = findspan(&buf, bs, nbits, oneruns);	/* black span */
	putspan(&bitbuf, span, TIFFFaxBlackCodes);
	bs += span;
	if (bs >= nbits)
	    break;
    }
    if (bitbuf.bit != 8)
	++bitbuf.byte;

    row.size = bitbuf.byte;
    row.data = malloc(row.size);
    memcpy(row.data, bitbuf.buf, row.size);
    return row;
}

/*****************************************************************************/

void
read_TIFF(char *name)
{
    TIFF *tif;

    u_long imageWidth, imageLength;
    float xResolution, yResolution;

    int row, size;
    Page *page, *page2;
    u_char *buf;

    tif = TIFFOpen(name, "rb");
    if (!tif) {
	perror(name);
	exit(1);
    }

    memset(&header, 0, sizeof(DocHeader));
    memcpy(header.sff_id, "Sfff", 4);
    header.version = 1;

    page0 = NULL;

    do {

    	page2 = calloc(1, sizeof(Page));
	if (page0)
	    page->next = page2;
	else
	    page0 = page2;
	page = page2;
	++header.num_pages;

	TIFFGetField(tif, TIFFTAG_IMAGEWIDTH,  &imageWidth);
	TIFFGetField(tif, TIFFTAG_IMAGELENGTH, &imageLength);
	TIFFGetField(tif, TIFFTAG_XRESOLUTION, &xResolution);
	TIFFGetField(tif, TIFFTAG_YRESOLUTION, &yResolution);

	page->h.vert_res  = (yResolution > 120.0) ? 1 : 0;
	page->h.horiz_res = 0; // 204 dpi
	page->h.coding    = 0; // MH Huffman
	page->h.specials  = 0;
	page->h.linelen   = imageWidth;
	page->h.pagelen   = imageLength;

	size = TIFFScanlineSize(tif);
	buf = malloc(size);

	page->rows = calloc(page->h.pagelen, sizeof(Row));

	for (row = 0; row < imageLength; row++) {
	    TIFFReadScanline(tif, buf, row, 0);
	    page->rows[row] = bitmap2fax(buf, page->h.linelen);
	}

	free(buf);

    } while (TIFFReadDirectory(tif));

    TIFFClose(tif);
}

/*****************************************************************************/

void
write_SFF(char *name)
{
    FILE *file;
    u_long offset;
    Page *page, *ppage;
    int row;

    file = fopen(name, "wb");
    if (!file) {
	perror(name);
	exit(1);
    }

    offset = sizeof(DocHeader);
    header.first_page = offset;

    ppage = NULL;
    for (page = page0; page; page = page->next) {
	if (ppage) {
	    ppage->h.next_page = offset;
	    page->h.prev_page = ppage->offset;
	} else
	    page0->h.prev_page = 1;
	ppage = page;

	page->offset = offset;

	offset += 2 + sizeof(PageHeader);
	for (row = 0; row < page->h.pagelen; ++row)
	    offset += 1 + page->rows[row].size;
    }
    offset += 2;

    header.last_page = ppage->offset;
    header.file_size = offset;

    fwrite(&header, sizeof(DocHeader), 1, file);

    for (page = page0; page; page = page->next) {
	fputc(254, file);
	fputc(sizeof(PageHeader), file);
	fwrite(&page->h, sizeof(PageHeader), 1, file);
	for (row = 0; row < page->h.pagelen; ++row) {
	    fputc(page->rows[row].size, file);
	    fwrite(page->rows[row].data, 1, page->rows[row].size, file);
	}
    }

    fputc(254, file);
    fputc(0, file);

    fclose(file);
}

/*****************************************************************************/

void
print_data(void)
{
    Page *page;
    int pageno, row, i;

    pageno = 1;
    for (page = page0; page; page = page->next) {
	printf("Page %d:\n", pageno);
	for (row = 0; row < page->h.pagelen; ++row) {
	    for (i = 0; i < page->rows[row].size; ++i)
		printf(" %02X", page->rows[row].data[i]);
	    printf("\n");
	}
	printf("\n");
	++pageno;
    }
}

main(int argc, char **argv)
{
    if (argc != 3) {
	fprintf(stderr,
		"TIFF-FaxG3 nach SFF Konverter.\n"
		"Copyright (c) Andreas Kaiser 1996\n"
		"\n"
		"Aufruf: tiff2sff tiff-file sff-file\n");
	exit(1);
    }

    read_TIFF(argv[1]);
    write_SFF(argv[2]);

    return 0;
}

