/* sane - Scanner Access Now Easy.
   Copyright (C) 1996, 1997 David Mosberger-Tang
   This file is part of the SANE package.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.

   As a special exception, the authors of SANE give permission for
   additional uses of the libraries contained in this release of SANE.

   The exception is that, if you link a SANE library with other files
   to produce an executable, this does not by itself cause the
   resulting executable to be covered by the GNU General Public
   License.  Your use of that executable is in no way restricted on
   account of linking the SANE library code into it.

   This exception does not, however, invalidate any other reasons why
   the executable file might be covered by the GNU General Public
   License.

   If you submit changes to SANE to the maintainers to be included in
   a subsequent release, you agree by submitting the changes that
   those changes may be distributed with this exception intact.

   If you write modifications of your own for SANE, it is your choice
   whether to permit this exception to apply to your modifications.
   If you do not wish that, delete this exception notice.

   This file provides a generic SCSI interface.  */

#include <sane/config.h>

#include <lalloca.h>	/* MUST come first for AIX! */

#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/ioctl.h>
#include <sys/param.h>
#include <sys/types.h>

#define STUBBED_INTERFACE	0
#define LINUX_INTERFACE		1
#define BSD_INTERFACE		2
#define	HPUX_INTERFACE		3
#define OPENSTEP_INTERFACE	4
#define DECUNIX_INTERFACE	5
#define SCO_OS5_INTERFACE	6
#define IRIX_INTERFACE		7
#define SOLARIS_INTERFACE	8
#define OS2_INTERFACE		9

#ifdef HAVE_SCSI_SG_H
# define USE LINUX_INTERFACE
# include <scsi/sg.h>
#else
# ifdef HAVE__USR_SRC_LINUX_INCLUDE_SCSI_SG_H
#  define USE LINUX_INTERFACE
#  include "/usr/src/linux/include/scsi/sg.h"
# else
#  ifdef HAVE_SYS_SCSICMD
#   define USE SCSO_OS5_INTERFACE
#   include <sys/scsi.h>
#   include <sys/scsicmd.h>
#  else
#   ifdef HAVE_SYS_SCSIIO_H
#    define USE BSD_INTERFACE
#    include <sys/scsiio.h>
#    ifdef HAVE_SCSI_H
#     include <scsi.h>
#    endif
#   else
#    ifdef HAVE_BSD_DEV_SCSIREG_H
#     define USE OPENSTEP_INTERFACE
#     include <bsd/dev/scsireg.h>
#    else
#     ifdef HAVE_IO_CAM_CAM_H
#      define USE DECUNIX_INTERFACE
#      include <io/common/iotypes.h>
#      include <io/cam/cam.h>
#      include <io/cam/dec_cam.h>
#      include <io/cam/uagt.h>
#      include <io/cam/scsi_all.h>
#     else
#      ifdef HAVE_SYS_DSREQ_H
#	define USE IRIX_INTERFACE
#	include <sys/dsreq.h>
#      else
#       ifdef HAVE_SYS_SCSI_H
#	 include <sys/scsi.h>
#	 ifdef SCTL_READ
#	  define USE HPUX_INTERFACE
#	 else
	  /* This happens for AIX and possibly other platforms... */
#	 endif
#       else
#        ifdef HAVE_OS2_H
#         define USE OS2_INTERFACE
#         define INCL_DOSFILEMGR
#         define INCL_DOS
#         define INCL_DOSDEVICES
#         define INCL_DOSDEVIOCTL
#         define INCL_DOSSEMAPHORES
#         define INCL_DOSMEMMGR
#         include <os2.h>
#         include "srb.h"
#        endif
#       endif
#      endif
#     endif
#    endif
#   endif
#  endif
# endif
#endif

#ifndef USE
# define USE STUBBED_INTERFACE
#endif

#include <sane/sanei_scsi.h>
#include <sane/sanei.h>

#define BACKEND_NAME	sanei_scsi
#include <sane/sanei_debug.h>


#ifdef SG_BIG_BUFF
int sanei_scsi_max_request_size = SG_BIG_BUFF;
#else
# ifdef __FreeBSD__
int sanei_scsi_max_request_size =  32 * 1024; /* FreeBSD is limited to this */
# elif USE == IRIX_INTERFACE
/* Actually, the limit varies from 256MB to 1GB :) */
int sanei_scsi_max_request_size = 8 * 1024 * 1024;
# elif USE == OS2_INTERFACE
int sanei_scsi_max_request_size =  32 * 1024; /* OS/2 limited to 64K */
# else
int sanei_scsi_max_request_size = 120 * 1024; /* works at least for OpenStep */
# endif
#endif

static int cam_fd = -1;		/* used for SCSI CAM based interfaces */
static struct
  {
    u_int in_use  : 1;		/* is this fd_info in use? */
    u_int fake_fd : 1;		/* is this a fake file descriptor? */
    u_int bus, target, lun;	/* nexus info; used for some interfaces only */
    SANEI_SCSI_Sense_Handler sense_handler;
  }
*fd_info;

static u_char cdb_sizes[8] = {6, 10, 10, 12, 12, 12, 10, 10};
#define CDB_SIZE(opcode)	cdb_sizes[(((opcode) >> 5) & 7)]

#if USE == OS2_INTERFACE

/* Driver info:  */
static HEV postSema = 0; /* Event Semaphore for posting SRB completion */
static HFILE driver_handle = 0;		/* file handle for device driver */
static PVOID aspi_buf = 0;		/* Big data buffer locked by driver. */
static int aspi_ref_count = 0;		/* # of fds using ASPI */

/* Open OS2 ASPI driver.

   Output: 0 if error, which is reported.  */
static int
open_aspi (void)
{
  ULONG rc;
  ULONG ActionTaken;
  USHORT openSemaReturn;
  USHORT lockSegmentReturn;
  unsigned long cbreturn;
  unsigned long cbParam;

  if (driver_handle)
    return 1;					/* Already open. */
  rc = DosAllocMem (&aspi_buf, sanei_scsi_max_request_size,
		    OBJ_TILE | PAG_READ | PAG_WRITE | PAG_COMMIT);
  if (rc)
    {
      DBG(1, "open_aspi: can't allocate memory\n");
      return 0;
    }
  rc = DosOpen ((PSZ) "aspirou$",		/* open driver */
		&driver_handle,
		&ActionTaken,
		0,
		0,
		FILE_OPEN,
		OPEN_SHARE_DENYREADWRITE | OPEN_ACCESS_READWRITE,
		NULL);
  if (rc)
    {
      /* opening failed -> return false */
      DBG(1, "open_aspi:  opening failed.\n");
      return 0;
    }
  rc = DosCreateEventSem (NULL, &postSema,	/* create event semaphore */
			  DC_SEM_SHARED, 0);
  if (rc)
    {
      /* DosCreateEventSem failed */
      DBG(1, "open_aspi:  couldn't create semaphore.\n");
      return 0;
    }
  rc = DosDevIOCtl (driver_handle, 0x92, 0x03,	    /* pass semaphore handle */
		    (void*) &postSema, sizeof(HEV),	/* to driver */
		    &cbParam, (void*) &openSemaReturn,
		    sizeof (USHORT), &cbreturn);
  if (rc || openSemaReturn)
    {
      DBG(1, "open_aspi:  couldn't set semaphore.\n");
      return 0;
    }

  /* Lock aspi_buf. */
  rc = DosDevIOCtl (driver_handle, 0x92, 0x04,	/* pass aspi_buf pointer */
		    (void*) aspi_buf, sizeof (PVOID),	/* to driver */
		    &cbParam, (void*) &lockSegmentReturn,
		    sizeof (USHORT), &cbreturn);
  if (rc || lockSegmentReturn)
    {
      /* DosDevIOCtl failed */
      DBG(1, "open_aspi:  Can't lock buffer.\n");
      return 0;
    }
  return 1;
}

/* Close driver and free everything.  */

static void
close_aspi (void)
{
  if (postSema)
    DosCloseEventSem (postSema);	/* Close event semaphore. */
  postSema = 0;
  if (driver_handle)			/* Close driver. */
    DosClose (driver_handle);
  driver_handle = 0;
  if (aspi_buf)				/* Free buffer. */
    DosFreeMem(aspi_buf);
  aspi_buf = 0;
}

#endif /* USE_OS2_INTERFACE */

SANE_Status
sanei_scsi_open (const char * dev, int * fdp, SANEI_SCSI_Sense_Handler handler)
{
  static int num_alloced = 0;
  u_int bus = 0, target = 0, lun = 0, fake_fd = 0;
  int fd;

  DBG_INIT();

#if USE == OS2_INTERFACE
  if (sscanf (dev, "b%dt%dl%d", &bus, &target, &lun) != 3)
    {
      DBG(1, "sanei_scsi_open: device name %s is not a valid\n",
	  strerror (errno));
      return SANE_STATUS_INVAL;
    }
  if (!open_aspi ())
    {
      /* Open driver if necessary. */
      close_aspi ();
      return SANE_STATUS_INVAL;
    } 

  ++aspi_ref_count;

  /* Find fake fd. */
  for (fd = 0; fd < num_alloced; ++fd)
    if (!fd_info[fd].in_use)
      break;
  fake_fd = 1;
#elif USE == DECUNIX_INTERFACE
  {
    UAGT_CAM_SCAN cam_scan;

    if (sscanf (dev, "b%dt%dl%d", &bus, &target, &lun) != 3)
      {
	DBG(1, "sanei_scsi_open: device name %s is not a valid\n",
	    strerror (errno));
	return SANE_STATUS_INVAL;
      }

    if (cam_fd < 0)
      {
	cam_fd = open ("/dev/cam", O_RDWR);
	if (cam_fd < 0)
	  {
	    DBG(1, "sanei_scsi_open: open(/dev/cam) failed: %s\n",
		strerror (errno));
	    return SANE_STATUS_INVAL;
	  }
      }
    cam_scan.ucs_bus = bus;
    cam_scan.ucs_target = target;
    cam_scan.ucs_lun = lun;
    if (ioctl (cam_fd, UAGT_CAM_SINGLE_SCAN, &cam_scan) < 0)
      {
	DBG(1, "sanei_scsi_open: ioctl(UAGT_CAM_SINGLE_SCAN) failed: %s\n",
	    strerror (errno));
	return SANE_STATUS_INVAL;
      }

    for (fd = 0; fd < num_alloced; ++fd)
      if (!fd_info[fd].in_use)
	break;
  }
#else /* !DECUNIX_INTERFACE */
# ifdef SGIOCSTL
  {
    char * real_dev;
    size_t len;

    /* OpenStep is a bit broken in that the device name refers to a scsi
       _bus_, not an individual scsi device.  Hence, SANE has to fudge
       with the device name so we know which target to connect to.  For
       this purpose, we use the last character in the device name as the
       target index.  'a' is target 0, 'b', target 1, and so on... */

    len = strlen (dev);
    if (len <= 1)
      {
	DBG(1, "sanei_scsi_open: devicename `%s' too short\n", dev);
	return SANE_STATUS_INVAL;
      }

    real_dev = strdup (dev);
    real_dev[len - 1] = '\0';

    target = dev[len - 1] - 'a';
    if (target > 7)
      {
	DBG(1, "sanei_scsi_open: `%c' is not a valid target id\n",
	    dev[len - 1]);
	return SANE_STATUS_INVAL;
      }
    dev = real_dev;
  }
# endif /* SGIOCSTL */

  fd = open (dev, O_RDWR | O_EXCL);
  if (fd < 0)
    {
      DBG(1, "sanei_scsi_open: open() failed: %s\n", strerror (errno));
      return SANE_STATUS_INVAL;
    }

# ifdef SG_SET_TIMEOUT
  /* Set large timeout since some scanners are slow but do not
     disconnect... ;-( */
  {
    int timeout;
    timeout = 10*60*HZ;		/* how about 10 minutes? ;-) */
    ioctl(fd, SG_SET_TIMEOUT, &timeout);
  }
# endif

# ifdef SGIOCSTL
  {
    struct scsi_adr sa;

    free ((void *) dev);

    sa.sa_target = target;
    sa.sa_lun = 0;
    if (ioctl (fd, SGIOCSTL, &sa) == -1)
      {
	DBG(1, "sanei_scsi_open: failed to attach to target: %u (%s)\n",
	    sa.sa_target, strerror (errno));
	return SANE_STATUS_INVAL;
      }
  }
# endif /* SGIOCSTL */
#endif /* !DECUNIX_INTERFACE */

  if (fd >= num_alloced)
    {
      size_t new_size, old_size;

      old_size = num_alloced * sizeof (fd_info[0]);
      num_alloced = fd + 8;
      new_size = num_alloced * sizeof (fd_info[0]);
      if (fd_info)
	fd_info = realloc (fd_info, new_size);
      else
	fd_info = malloc (new_size);
      memset ((char *) fd_info + old_size, 0, new_size - old_size);
      if (!fd_info)
	{
	  if (!fake_fd)
	    close (fd);
	  return SANE_STATUS_NO_MEM;
	}
    }
  fd_info[fd].sense_handler = handler;
  fd_info[fd].fake_fd = fake_fd;
  fd_info[fd].bus = bus;
  fd_info[fd].target = target;
  fd_info[fd].lun = lun;

  if (fdp)
    *fdp = fd;
  return SANE_STATUS_GOOD;
}

void
sanei_scsi_close (int fd)
{
  fd_info[fd].in_use = 0;
  fd_info[fd].sense_handler = 0;
  if (!fd_info[fd].fake_fd)
    close (fd);

#if USE == OS2_INTERFACE
  if (--aspi_ref_count <= 0)
    close_aspi ();
#endif /* USE == OS2_INTERFACE */
}


#if USE == LINUX_INTERFACE

#include <signal.h>
#include <sys/time.h>

#define WE_HAVE_ASYNC_SCSI

static int need_init = 1;
static sigset_t all_signals;

#define ATOMIC(s)					\
do							\
  {							\
    sigset_t old_mask;					\
							\
    if (need_init)					\
      {							\
	need_init = 0;					\
	sigfillset (&all_signals);			\
      }							\
    sigprocmask (SIG_BLOCK, &all_signals, &old_mask);	\
    {s;}						\
    sigprocmask (SIG_SETMASK, &old_mask, 0);		\
  }							\
while (0)

static struct req
  {
    int				fd;
    u_int			running : 1,
				done    : 1;
    SANE_Status			status;
    size_t *			dst_len;
    void *			dst;
    struct
      {
	struct sg_header	hdr;
	u_int8_t		data[SG_BIG_BUFF];
      }				cdb;
    struct req *		next;
  }
*qhead, *qtail, *free_list;

static void
issue (struct req *req)
{
  ssize_t nwritten;

  if (!req || req->running)
    return;

  DBG(4, "sanei_scsi.issue: %p\n", req);

  ATOMIC(req->running = 1;
	 nwritten = write (req->fd, &req->cdb, req->cdb.hdr.pack_len));

  if (nwritten != req->cdb.hdr.pack_len)
    {
      DBG(1, "sanei_scsi.issue: bad write (errno=%s)\n",
	  strerror (errno));
      req->done = 1;
      if (errno == ENOMEM)
	{
	  DBG(1, "sanei_scsi.issue: SG_BIG_BUF inconsistency?  "
	      "Check file PROBLEMS.\n");
	  req->status = SANE_STATUS_NO_MEM;
	}
      else
	req->status = SANE_STATUS_IO_ERROR;
    }
}

void
sanei_scsi_req_flush_all (void)
{
  struct req *req, *next_req;

  for (req = qhead; req; req = next_req)
    {
      if (req->running && !req->done)
	read (req->fd, &req->cdb, req->cdb.hdr.reply_len);
      next_req = req->next;

      req->next = free_list;
      free_list = req;
    }
  qhead = qtail = 0;
}

SANE_Status
sanei_scsi_req_enter (int fd, const void * src, size_t src_size,
		      void * dst, size_t * dst_size, void **idp)
{
  struct req * req;

  if (free_list)
    {
      req = free_list;
      free_list = req->next;
      req->next = 0;
    }
  else
    {
      req = malloc (sizeof (*req));
      if (!req)
	{
	  DBG(1, "sanei_scsi_req_enter: failed to malloc %lu bytes\n",
	      (u_long) sizeof (*req));
	  return SANE_STATUS_NO_MEM;
	}
    }
  req->fd = fd;
  req->running = 0;
  req->done = 0;
  req->status = SANE_STATUS_GOOD;
  req->dst = dst;
  req->dst_len = dst_size;
  memset (&req->cdb.hdr, 0, sizeof (req->cdb.hdr));
  req->cdb.hdr.pack_len = src_size + sizeof (req->cdb.hdr);
  req->cdb.hdr.reply_len = (dst_size ? *dst_size : 0) + sizeof (req->cdb.hdr);
  memcpy (&req->cdb.data, src, src_size);

  req->next = 0;
  ATOMIC(if (qtail)
	   {
	     qtail->next = req;
	     qtail = req;
	   }
	 else
	   qhead = qtail = req);

  DBG(4, "scsi_req_enter: entered %p\n", req);

  *idp = req;
  return SANE_STATUS_GOOD;
}

SANE_Status
sanei_scsi_req_wait (void *id)
{
  SANE_Status status;
  struct req * req = id;
  ssize_t nread = 0;

  assert (req == qhead);	/* we don't support out-of-order completion */

  DBG(4, "sanei_scsi_req_wait: waiting for %p\n", req);

  issue (req);			/* ensure the command is running */
  if (req->done)
    {
      issue (req->next);	/* issue next command, if any */
      status = req->status;
    }
  else
    {
      fd_set readable;

      /* wait for command completion: */
      FD_ZERO(&readable);
      FD_SET(req->fd, &readable);
      select (req->fd + 1, &readable, 0, 0, 0);

      /* now atomically read result and set DONE: */
      ATOMIC(nread = read (req->fd, &req->cdb, req->cdb.hdr.reply_len);
	     req->done = 1);

      /* Now issue next command asap, if any.  We can't do this
	 earlier since the Linux kernel has space for just one big
	 buffer.  */
      issue (req->next);

      DBG(4, "sanei_scsi_req_wait: read %ld bytes\n", (long) nread);

      if (nread < 0)
	{
	  DBG(1, "sanei_scsi_req_wait: read returned %ld (errno=%d)\n",
	      (long) nread, errno);
	  status = SANE_STATUS_IO_ERROR;
	}
      else
	{
	  nread -= sizeof (req->cdb.hdr);

	  if (req->cdb.hdr.result == 0)
	    {
	      if (req->dst)
		memcpy (req->dst, req->cdb.data, nread);

	      if (req->dst_len)
		*req->dst_len = nread;

	      status = SANE_STATUS_GOOD;
	    }
	  else
	    {
	      SANEI_SCSI_Sense_Handler handler
		= fd_info[req->fd].sense_handler;

	      DBG(1, "sanei_scsi_req_wait: SCSI command failed: %s\n",
		  strerror (req->cdb.hdr.result));

	      if (req->cdb.hdr.result == EBUSY)
		status = SANE_STATUS_DEVICE_BUSY;
	      else if ((req->cdb.hdr.sense_buffer[0] & 0x80) && handler)
		status = (*handler) (req->fd, req->cdb.hdr.sense_buffer);
	      else
		status = SANE_STATUS_IO_ERROR;
	    }
	}
    }


  /* dequeue and release processed request: */
  ATOMIC(qhead = qhead->next;
	 if (!qhead)
	   qtail = 0;
	 req->next = free_list;
	 free_list = req);
  return status;
}

SANE_Status
sanei_scsi_cmd (int fd, const void * src, size_t src_size,
		void * dst, size_t * dst_size)
{
  SANE_Status status;
  void *id;

  status = sanei_scsi_req_enter (fd, src, src_size, dst, dst_size, &id);
  if (status != SANE_STATUS_GOOD)
    return status;
  return sanei_scsi_req_wait (id);
}

#endif /* USE == LINUX_INTERFACE */


#if USE == BSD_INTERFACE

# ifndef HAVE_SCSIREQ_ENTER
static int
scsireq_enter (int fd, scsireq_t *hdr)
{
  return ioctl (fd, SCIOCCOMMAND, hdr);
}
# endif /* !HAVE_SCSIREQ_ENTER */

SANE_Status
sanei_scsi_cmd (int fd, const void * src, size_t src_size,
		void * dst, size_t * dst_size)
{
  size_t cdb_size;
  scsireq_t hdr;
  int result;

  cdb_size = CDB_SIZE (*(u_char *) src);

  memset (&hdr, 0, sizeof (hdr));
  memcpy (hdr.cmd, src, cdb_size);
  if (dst_size && *dst_size)
    {
      assert (cdb_size == src_size);
      hdr.flags = SCCMD_READ;
      hdr.databuf = dst;
      hdr.datalen = *dst_size;
    }
  else
    {
      assert (cdb_size <= src_size);
      hdr.flags = SCCMD_WRITE;
      hdr.databuf = (char *) src + cdb_size;
      hdr.datalen = src_size;
    }
  hdr.timeout = 60000;			/* 1 minute timeout */
  hdr.cmdlen = cdb_size;
  hdr.senselen = sizeof (hdr.sense);

  result = scsireq_enter (fd, &hdr);
  if (result < 0)
    {
      DBG(1, "sanei_scsi_cmd: scsi_reqenter() failed: %s\n", strerror (errno));
      return SANE_STATUS_IO_ERROR;
    }
  if (hdr.retsts != SCCMD_OK)
    {
      DBG(1, "sanei_scsi_cmd: scsi returned with status %d\n", hdr.retsts);
      switch (hdr.retsts)
	{
	case SCCMD_TIMEOUT:
	case SCCMD_BUSY:
	  return SANE_STATUS_DEVICE_BUSY;

	case SCCMD_SENSE:
	  if (fd_info[fd].sense_handler)
	    return (*fd_info[fd].sense_handler) (fd, &hdr.sense[0]);
	  /* fall through */
	default:
	  return SANE_STATUS_IO_ERROR;
	}
    }

  if (dst_size)
    *dst_size = hdr.datalen_used;

  return SANE_STATUS_GOOD;
}
#endif /* USE == BSD_INTERFACE */


#if USE == HPUX_INTERFACE
/* XXX untested code! */
SANE_Status
sanei_scsi_cmd (int fd, const void * src, size_t src_size,
		void * dst, size_t * dst_size)
{
  struct sctl_io hdr;
  size_t cdb_size;

  cdb_size = CDB_SIZE (*(u_char *) src);

  memset (&hdr, 0, sizeof (hdr));
  memcpy (hdr.cdb, src, src_size);
  if (dst_size && *dst_size)
    {
      assert (cdb_size == src_size);
      hdr.flags = SCTL_READ;
      hdr.data = dst;
      hdr.data_length = *dst_size;
    }
  else
    {
      assert (cdb_size <= src_size);
      hdr.data = (char *) src + cdb_size;
      hdr.data_length = src_size - cdb_size;
    }
  hdr.cdb_length = cdb_size;
  hdr.max_msecs = 60000;	/* 1 minute timeout */
  if (ioctl (fd, SIOC_IO, &hdr) < 0)
    {
      DBG(1, "sanei_scsi_cmd: ioctl(SIOC_IO) failed: %s\n",
	  strerror (errno));
      return SANE_STATUS_IO_ERROR;
    }
  if (hdr.cdb_status)
    DBG(1, "sanei_scsi_cmd: SCSI completed with cdb_status=%d\n",
	hdr.cdb_status);
  if (dst_size)
    *dst_size = hdr.data_xfer;

  if (hdr.sense_xfer > 0 && (hdr.sense[0] & 0x80) && fd_info[fd].sense_handler)
    return (*fd_info[fd].sense_handler) (fd, hdr.sense);
  return SANE_STATUS_GOOD;
}
#endif /* USE == HPUX_INTERFACE */


#if USE == OPENSTEP_INTERFACE
SANE_Status
sanei_scsi_cmd (int fd, const void * src, size_t src_size,
		void * dst, size_t * dst_size)
{
  struct scsi_req hdr;
  size_t cdb_size;

  cdb_size = CDB_SIZE (*(u_char *) src);

  memset (&hdr, 0, sizeof (hdr));
  memcpy (&hdr.sr_cdb, src, cdb_size);
  if (dst_size && *dst_size)
    {
      assert (cdb_size == src_size);
      hdr.sr_dma_dir = SR_DMA_RD;
      hdr.sr_addr = dst;
      hdr.sr_dma_max = *dst_size;
    }
  else
    {
      assert (cdb_size <= src_size);
      hdr.sr_dma_dir = SR_DMA_WR;
      hdr.sr_addr = (char *) src + cdb_size;
      hdr.sr_dma_max = src_size - cdb_size;
    }
  hdr.sr_ioto = 60;		/* I/O timeout in seconds */

  if (ioctl (fd, SGIOCREQ, &hdr) == -1)
    {
      DBG(1, "sanei_scsi_cmd: ioctl(SGIOCREQ) failed: %s\n", strerror (errno));
      return SANE_STATUS_IO_ERROR;
    }
  if (hdr.sr_io_status != 1)
    DBG(1, "sanei_scsi_cmd: SGIOCREQ completed with sr_io_status=%d\n",
	hdr.sr_io_status);
  if (hdr.sr_scsi_status == SR_IOST_CHKSV && fd_info[fd].sense_handler)
    return (*fd_info[fd].sense_handler) (fd, (u_char *) &hdr.sr_esense);
  if (dst_size)
    *dst_size = hdr.sr_dma_xfr;
  return SANE_STATUS_GOOD;
}
#endif /* USE == OPENSTEP_INTERFACE */


#if USE == DECUNIX_INTERFACE
SANE_Status
sanei_scsi_cmd (int fd, const void * src, size_t src_size,
		void * dst, size_t * dst_size)
{
  u_char sense[64];
  UAGT_CAM_CCB hdr;
  CCB_SCSIIO ccb;
  size_t cdb_size;

  cdb_size = CDB_SIZE (*(u_char *) src);

  memset (ccb, 0, sizeof (ccb));
  ccb.cam_ch.my_addr	    = (CCB_HEADER *) &ccb;
  ccb.cam_ch.cam_ccb_len    = sizeof (ccb);
  ccb.cam_ch.cam_func_code  = XPT_SCSI_IO;
  ccb.cam_ch.cam_path_id    = fd_info[fd].bus;
  ccb.cam_ch.cam_target_id  = fd_info[fd].target;
  ccb.cam_ch.cam_target_lun = fd_info[fd].lun;
  ccb.cam_ch.cam_flags	    = 0;

  if (dst_size && *dst_size)
    {
      assert (cdb_size == src_size);
      ccb.cam_ch.cam_flags |= CAM_DIR_IN;
      ccb.cam_data_ptr = (u_char *) dst;
      ccb.cam_dxfer_len = *dst_size;
    }
  else
    {
      assert (cdb_size <= src_size);
      if (cdb_size == src_size)
	ccb.cam_ch.cam_flags |= CAM_DIR_NONE;
      else
	ccb.cam_ch.cam_flags |= CAM_DIR_OUT;
      ccb.cam_data_ptr = (u_char *) src + cdb_size;
      ccb.cam_dxfer_len = src_size - cdb_size;
    }
  ccb.cam_timeout = 60;		/* set timeout in seconds */
  ccb.cam_cdb_len = cdb_size;
  memcpy (&ccb.cam_cdb_io.cam_cdb_bytes[0], src, cdb_size);

  memset (&hdr, 0, sizeof (hdr));
  hdr.uagt_ccb = (CCB_HEADER *) &ccb;
  hdr.uagt_buffer = ccb.cam_data_ptr;
  hdr.uagt_buflen = ccb.cam_dxfer_len;
  hdr.uagt_snsbuf = sense;
  hdr.uagt_snslen = sizeof (sense);
  hdr.uagt_cdb = 0;		/* indicate that CDB is in CCB */
  hdr.uagt_cdblen = 0;

  if (ioctl (cam_fd, UAGT_CAM_IO, &hdr) < 0)
    {
      DBG(1, "sanei_scsi_cmd: ioctl(UAGT_CAM_IO) failed: %s\n",
	  strerror (errno));
      return SANE_STATUS_IO_ERROR;
    }
  if (ccb.cam_ch.cam_status != CAM_REQ_CMP)
    {
      DBG(1, "sanei_scsi_cmd: UAGT_CAM_IO completed with cam_status=%d\n",
	  ccb.cam_ch.cam_status);

      if (ccb.cam_ch.cam_status == CAM_AUTOSNS_VALID
	  && fd_info[fd].sense_handler)
	return (*fd_info[fd].sense_handler) (fd, sense);
      else
	return SANE_STATUS_INVAL;
    }
  if (dst_size)
    *dst_size = ccb.cam_dxfer_len;
  return SANE_STATUS_GOOD;
}
#endif /* USE == DECUNIX_INTERFACE */


#if USE == SCO_OS5_INTERFACE
SANE_Status
sanei_scsi_cmd (int fd, const void * src, size_t src_size,
		void * dst, size_t * dst_size)
{
  static u_char sense_buffer[256];
  struct scsicmd2 sc2;
  struct scsicmd *sc;
  int cdb_size;
  int opcode;
  int i;

  if (fd < 0)
    return SANE_STATUS_IO_ERROR;

  memset (&sc2, 0, sizeof (sc2));
  sc = &sc2.cmd;
  sc2.sense_len = sizeof (sense_buffer);
  sc2.sense_ptr = sense_buffer;

  cdb_size = CDB_SIZE (*(u_char *) src);
  if (dst_size && *dst_size)
    {
      sc->is_write = 0;
      sc->data_ptr = dst;
      sc->data_len = *dst_size;
    }
  else
    {
      sc->data_len = src_size - cdb_size;
      sc->data_ptr = (char *) src + cdb_size;
      sc->is_write = 1;
    }
  memcpy (sc->cdb, src, cdb_size);
  sc->cdb_len = cdb_size;

  /* Send the command down via the "pass-through" interface */
  if (ioctl (fd, SCSIUSERCMD2, &sc2) < 0)
    {
      DBG(1, "sanei_scsi_cmd: ioctl(SCSIUSERCMD2) failed: %s\n",
	  strerror (errno));
      return SANE_STATUS_IO_ERROR;
    }
  if (sc->host_sts || sc->target_sts)
    {
      DBG(1, "sanei_scsi_cmd: SCSIUSERCMD2 completed with "
	  "host_sts=%x, target_sts=%x\n", sc->host_sts, sc->target_sts);
      if (fd_info[fd].sense_handler)
	return (*fd_info[fd].sense_handler) (fd, sense_buffer);
      return SANE_STATUS_IO_ERROR;
    }
  return SANE_STATUS_GOOD;
}
#endif /* USE == SCO_OS5_INTERFACE */

#if USE == OS2_INTERFACE
/* XXX untested code! */
SANE_Status
sanei_scsi_cmd (int fd, const void * src, size_t src_size,
		void * dst, size_t * dst_size)
{
  ULONG rc;					/* Returns. */
  unsigned long cbreturn;
  unsigned long cbParam;
  SRB srb;					/* SCSI Request Block */
  ULONG count=0;				/* For semaphore. */
  size_t cdb_size;

  memset ((char *) &srb, 0, sizeof (srb));	/* Okay, I'm paranoid. */
  cdb_size = CDB_SIZE (*(u_char *) src);	/* Size of command block. */
  srb.cmd = SRB_Command;			/* execute SCSI cmd */
  srb.ha_num = fd_info[fd].bus;			/* host adapter number */
  srb.u.cmd.target = fd_info[fd].target;	/* Target SCSI ID */
  srb.u.cmd.lun = fd_info[fd].lun;		/* Target SCSI LUN */
  srb.flags = SRB_Post;				/* posting enabled */
  if (dst_size && *dst_size)
    {
      /* Reading. */
      assert (*dst_size <= sanei_scsi_max_request_size);
      assert (cdb_size == src_size);
      srb.u.cmd.data_len = *dst_size;
      srb.flags |= SRB_Read;
    }
  else
    {
      /* Writing. */
      srb.u.cmd.data_len = src_size - cdb_size;
      assert (cdb_size <= src_size);
      assert (srb.u.cmd.data_len <= sanei_scsi_max_request_size);
      if (srb.u.cmd.data_len)
	srb.flags |= SRB_Write;
      else
	srb.flags |= SRB_NoTransfer;
      memcpy (aspi_buf, (char *) src + cdb_size, srb.u.cmd.data_len);
    }
  srb.u.cmd.sense_len = 16;			/* length of sense buffer */
  srb.u.cmd.data_ptr = NULL; /* pointer to data buffer already registered */
  srb.u.cmd.link_ptr = NULL;			/* pointer to next SRB */
  srb.u.cmd.cdb_len = cdb_size;			/* SCSI command length */
  memcpy (&srb.u.cmd.cdb_st[0], (char *) src, cdb_size);

  /* Do the command. */
  rc = DosDevIOCtl (driver_handle, 0x92, 0x02, (void*) &srb,
		    sizeof (SRB), &cbParam,
		    (void*) &srb, sizeof (SRB), &cbreturn);

  if (rc)
    {
      DBG(1, "sanei_scsi_cmd: DosDevIOCtl failed.\n");
      return SANE_STATUS_IO_ERROR;
    }
  if (DosWaitEventSem (postSema, -1) ||		/* wait forever for sema. */
      DosResetEventSem (postSema, &count))	/* reset semaphore. */
    {
      DBG(1, "sanei_scsi_cmd:  semaphore failure.\n");
      return SANE_STATUS_IO_ERROR;
    }

  /* Get sense data if available. */
  if ((srb.status == SRB_Aborted || srb.status == SRB_Error) &&
      srb.u.cmd.target_status == SRB_CheckStatus
      && fd_info[fd].sense_handler != 0)
    {
      SANEI_SCSI_Sense_Handler s_handler = fd_info[fd].sense_handler;
      return (*s_handler) (fd, &srb.u.cmd.cdb_st[cdb_size]);
    }
  if (srb.status != SRB_Done ||
      srb.u.cmd.ha_status != SRB_NoError ||
      srb.u.cmd.target_status != SRB_NoStatus)
    DBG(1, "sanei_scsi_cmd:  command 0x%02x failed.\n", srb.u.cmd.cdb_st[0]);
  if (dst_size && *dst_size)				/* Reading? */
    memcpy ((char *) dst, aspi_buf, *dst_size);
  return SANE_STATUS_GOOD;
}
#endif /* USE == OS2_INTERFACE */

#if USE == STUBBED_INTERFACE
SANE_Status
sanei_scsi_cmd (int fd, const void * src, size_t src_size,
		void * dst, size_t * dst_size)
{
  return SANE_STATUS_UNSUPPORTED;
}
#endif /* USE == STUBBED_INTERFACE */

#if USE == IRIX_INTERFACE
SANE_Status
sanei_scsi_cmd (int fd, const void *src, size_t src_size,
		void *dst, size_t *dst_size)
{
  dsreq_t scsi_req;		/* SCSI request */
  u_char sensebuf[1024];	/* Request sense buffer */
  size_t cdb_size;		/* Size of SCSI command */

  cdb_size = CDB_SIZE(*(u_char *)src);

  memset(&scsi_req, 0, sizeof(scsi_req));

  if (dst != NULL)
    {
      /*
       * SCSI command returning/reading data...
       */
      scsi_req.ds_flags    = DSRQ_READ | DSRQ_SENSE;
      scsi_req.ds_time     = 120 * 1000;
      scsi_req.ds_cmdbuf   = (caddr_t)src;
      scsi_req.ds_cmdlen   = cdb_size;
      scsi_req.ds_databuf  = (caddr_t)dst;
      scsi_req.ds_datalen  = *dst_size;
      scsi_req.ds_sensebuf = (caddr_t)sensebuf;
      scsi_req.ds_senselen = sizeof (sensebuf);
    }
  else
    {
      /*
       * SCSI command sending/writing data...
       */
      scsi_req.ds_flags    = DSRQ_WRITE | DSRQ_SENSE;
      scsi_req.ds_time     = 120 * 1000;
      scsi_req.ds_cmdbuf   = (caddr_t)src;
      scsi_req.ds_cmdlen   = cdb_size;
      scsi_req.ds_databuf  = (caddr_t)src + cdb_size;
      scsi_req.ds_datalen  = src_size - cdb_size;
      scsi_req.ds_sensebuf = (caddr_t)sensebuf;
      scsi_req.ds_senselen = sizeof (sensebuf);
    }
  if (ioctl (fd, DS_ENTER, &scsi_req) < 0)
    return SANE_STATUS_IO_ERROR;

  if (dst_size != NULL)
    *dst_size = scsi_req.ds_datasent;

  if (scsi_req.ds_status != 0)
    {
      if (scsi_req.ds_status == STA_BUSY)
	return SANE_STATUS_DEVICE_BUSY;
      else if ((sensebuf[0] & 0x80) && fd_info[fd].sense_handler)
	return (*fd_info[fd].sense_handler)(fd, sensebuf);
      else
	return SANE_STATUS_IO_ERROR;
    }
  return SANE_STATUS_GOOD;
}
#endif /* USE == IRIX_INTERFACE */

#ifndef WE_HAVE_ASYNC_SCSI

SANE_Status
sanei_scsi_req_enter (int fd, const void * src, size_t src_size,
		      void * dst, size_t * dst_size, void **idp)
{
  return sanei_scsi_cmd (fd, src, src_size, dst, dst_size);
}

SANE_Status
sanei_scsi_req_wait (void *id)
{
  return SANE_STATUS_GOOD;
}

void
sanei_scsi_req_flush_all (void)
{
}

#endif /* WE_HAVE_ASYNC_SCSI */
