/* ------------------------------------------------------------------------- */

/* umax-scsi.c: low-level scanner driver for UMAX scanners.
  
   Copyright (C) 1996-1997 Michael K. Johnson
   Copyright (C) 1997 Oliver Rauch

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.

   As a special exception, the authors of SANE give permission for
   additional uses of the libraries contained in this release of SANE.

   The exception is that, if you link a SANE library with other files
   to produce an executable, this does not by itself cause the
   resulting executable to be covered by the GNU General Public
   License.  Your use of that executable is in no way restricted on
   account of linking the SANE library code into it.

   This exception does not, however, invalidate any other reasons why
   the executable file might be covered by the GNU General Public
   License.

   If you submit changes to SANE to the maintainers to be included in
   a subsequent release, you agree by submitting the changes that
   those changes may be distributed with this exception intact.

   If you write modifications of your own for SANE, it is your choice
   whether to permit this exception to apply to your modifications.
   If you do not wish that, delete this exception notice.
   
 */

/* ------------------------------- INCLUDES -------------------------------- */

#include <sys/types.h>
#include <unistd.h>
#include <string.h>
#include <assert.h>
#include <signal.h>
#include <stdlib.h>
#include <math.h>
#include <stdio.h>
#include <getopt.h>


#ifndef UMAX_TO_SANE                                    /* ONLY FOR UMAX-CLI */

#  include <scsi/scsi.h>
#  include <scsi/sg.h>
#  include <fcntl.h>
#  include "scsicmd.h"        
#  include "debug.h"
#  define UMAX_SCSI_MAX_REQUEST_SIZE SG_BIG_BUFF
#  define sense_handler NULL

#else                                               /* ONLY FOR SANE-BACKEND */

#  include "sane/sanei_scsi.h"
#  include "sane/sanei_debug.h"
#  include <sane/config.h>           /* get inline #define'd if necessary... */

#  define do_scsi_cmd(fd, cmd, cmd_len, out, out_len) \
       sanei_scsi_cmd(fd, cmd, cmd_len, out, out_len)
#  define umax_do_scsi_open(dev, fd, handler) sanei_scsi_open(dev, fd, handler)
#  define umax_do_scsi_close(fd)              sanei_scsi_close(fd)
#  define UMAX_SCSI_MAX_REQUEST_SIZE          sanei_scsi_max_request_size

#endif /* UMAX_TO_SANE */


#include "umax-scsidef.h"
#include "umax-scanner.h"
#include "umax-struct.h"


/* ---------------------------- MATH-HELPERS ------------------------------- */


#define min(a,b) (((a)<(b))?(a):(b))
#define max(a,b) (((a)>(b))?(a):(b))
#define inrange(minimum, val, maximum) (min(maximum,max(val,minimum)))


/* ---------------------------- CBHS_CORRECT ------------------------------- */


static int umax_cbhs_correct(minimum, cbhs, maximum)
{
  int range=maximum-minimum+1;
  if (range == 256) { return cbhs; }

 return (int)( (cbhs/256.0)*range + minimum );
}


/* ----------------------- for umax-cli: ----------------------------------- */


#ifndef UMAX_TO_SANE

  /* ------------------------- UMAX_DO_SCSI_OPEN  -------------------------- */

  int umax_do_scsi_open(const char * device, int * sfd, void * handler)
  {
   int fd;

    fd=open(device, O_RDWR);
    if (fd < 0) { return -1; }
    *sfd = fd;
    return 0;
  }


  /* ------------------------- UMAX_DO_SCSI_CLOSE  ------------------------- */

  void umax_do_scsi_close(int sfd)
  {
    close(sfd);
    sfd = -1;
  }

#endif /* UMAX_TO_SANE */



/* --------------------------- UMAX_DO_SCSI_CMD  --------------------------- */


#ifdef UMAX_TO_SANE
static
#endif
int umax_do_scsi_cmd(int fd,char * cmd,int cmd_len,char * out,size_t out_len)
{
 sigset_t signal_set;
 int ret;

  sigemptyset(&signal_set);
  sigaddset(&signal_set,SIGINT);          /* block signals INT,TERM and ABRT */
  sigaddset(&signal_set,SIGTERM);
  sigaddset(&signal_set,SIGABRT);

  sigprocmask(SIG_BLOCK,&signal_set,NULL);
  ret = do_scsi_cmd(fd, cmd, cmd_len, out, &out_len);
  sigprocmask(SIG_UNBLOCK,&signal_set,NULL);                 /* free signals */

 return ret;
}


/* --------------------- UMAX GET DATA BUFFER STATUS ----------------------- */


#ifdef UMAX_TO_SANE
static
#endif
int umax_get_data_buffer_status(US *us)
{
 int status;
  DBG(5,"get_data_buffer_status\n");
  set_GDBS_wait(get_data_buffer_status.cmd,1);      /* wait for scanned data */
  status = umax_do_scsi_cmd(us->sfd, get_data_buffer_status.cmd,
                            get_data_buffer_status.size,
                            us->buffer, gdbs_return_block_size);

  if (status) { return status; }

 return 0;
}


/* ------------------------ UMAX DO REQUEST SENSE -------------------------- */


#ifdef UMAX_TO_SANE
static
#endif
void umax_do_request_sense(US *us)
{
   DBG(5,"do_request_sense\n");
   set_RS_allocation_length(request_sense.cmd, rs_return_block_size); 
   umax_do_scsi_cmd(us->sfd, request_sense.cmd, request_sense.size,
                    us->buffer, rs_return_block_size);
}


/* -------------------------- UMAX TEST WARMUP ----------------------------- */


#ifdef UMAX_TO_SANE
static
#endif
int umax_test_warmup(US *us)
{

/* scsi-generic-driver only asks for request_sense, if error code
   is CHECK CONDITION, but in this case it should be BUSY, so we
   have to do a request_sense */

  umax_do_request_sense(us);                                /* request_sense */

return ( ( get_RS_sense_key(us->buffer) == 0x09) &&
         ( get_RS_ASC(us->buffer) == 0x80) &&
         ( get_RS_ASCQ(us->buffer) ==0x01) );
}


/* -------------------------- UMAX WAIT SCANNER ---------------------------- */


/* wait_scanner should spin until TEST_INIT_READY returns 0 (GOOD)
   returns 0 on success,
   returns -1 on error.  */

#ifdef UMAX_TO_SANE
static
#endif
int umax_wait_scanner(US *us)
{
  int ret;
  int cnt = 0;

  do
  {
      if (cnt > 100)                                   /* maximal 100 tests */
      {
        DBG(2, "scanner does not get ready\n");
#ifndef UMAX_TO_SANE
        umax_do_request_sense(us);         /* may be this has to be removed */
        print_sense(us->buffer);                 /* no: print sense */
#endif
        return -1;
      }
                                                         /* test unit ready */
      ret = umax_do_scsi_cmd(us->sfd, test_unit_ready.cmd,test_unit_ready.size,
                             NULL, 0);
      cnt++;

      if (ret !=0)
      {
        if (cnt==1)                                          /* print reason */
        {
          if (umax_test_warmup(us))                        /* test if warmup */
          { DBG(3,"waiting for lamp warmup\n"); }
          else
          { DBG(3,"scanner is busy, waiting ...\n"); }
        }
        usleep(100000);                                  /* wait 0.1 seconds */
      } /* if ret != 0 */

  } while (ret != 0);                         /* until scanner reports ready */

  DBG(4,"scanner ready\n"); 

  return 0;
}

#define WAIT_SCANNER { int ret = umax_wait_scanner(us); if (ret) return ret; }


/* -------------------- UMAX EMERGENCY GIVE SCANNER ------------------------ */


#ifndef UMAX_TO_SANE
static US *emergency_us;

void umax_emergency_give_scanner(int i)
{
  US *us=emergency_us;

  if (emergency_us->verbose)
  { DBG(1, "emergency: trying to release scanner ...\n"); }

  umax_wait_scanner(emergency_us);
  umax_do_scsi_cmd(emergency_us->sfd, object_position.cmd,
                   object_position.size, NULL, 0);

  umax_wait_scanner(emergency_us);
  umax_do_scsi_cmd(emergency_us->sfd, release_unit.cmd,
                   release_unit.size, NULL, 0);

  if (emergency_us->verbose) { DBG(1, "scanner released\n"); }

  exit(1);
}
#endif


/* ------------------------- UMAX GRAB SCANNER ----------------------------- */


/* umax_grab_scanner should go through the following command sequence:
 * TEST UNIT READY
 *     CHECK CONDITION  \
 * REQUEST SENSE         > These should be handled automagically by
 *     UNIT ATTENTION   /  the kernel if they happen (powerup/reset)
 * TEST UNIT READY
 *     GOOD
 * RESERVE UNIT
 *     GOOD
 * 
 * It is then responsible for installing appropriate signal handlers
 * to call emergency_give_scanner() if user aborts.
 */

#ifdef UMAX_TO_SANE
static
#endif
int umax_grab_scanner(US *us)
{
 int ret;

  DBG(4,"grabbing scanner\n");

  WAIT_SCANNER;   /* wait for scanner ready, if not print sense and return 1 */
  ret = umax_do_scsi_cmd(us->sfd, reserve_unit.cmd, reserve_unit.size, NULL, 0);
  if (ret) { return ret; }

  DBG(3, "scanner reserved\n");

#ifndef UMAX_TO_SANE
  /* And if we abort? */
  emergency_us = us;
  signal(SIGINT,  umax_emergency_give_scanner);
  signal(SIGTERM, umax_emergency_give_scanner);
  signal(SIGABRT, umax_emergency_give_scanner);
  signal(SIGILL,  umax_emergency_give_scanner);
  signal(SIGSEGV, umax_emergency_give_scanner);
#endif

  return 0;
}


/* ------------------- UMAX REPOSITION SCANNER ----------------------------- */


#ifdef UMAX_TO_SANE
static
#endif
int umax_reposition_scanner(US *us)
{
 int ret;

  DBG(4, "trying to reposition scanner ...\n");
  ret = umax_do_scsi_cmd(us->sfd, object_position.cmd, object_position.size, NULL, 0);
  if (ret) { return ret; }
  WAIT_SCANNER;
  DBG(3, "scanner repositioned\n");
  return 0;
}


/* ------------------------- UMAX GIVE SCANNER ----------------------------- */


/* umax_give_scanner should go through the following sequence:
 * OBJECT POSITION
 *     GOOD
 * RELEASE UNIT
 *     GOOD
 */

#ifdef UMAX_TO_SANE
static
#endif
int umax_give_scanner(US *us)
{
 int ret;

  DBG(4, "trying to release scanner ...\n");
  ret = umax_do_scsi_cmd(us->sfd, object_position.cmd, object_position.size, NULL, 0);
  if (ret)  { return ret; }
  WAIT_SCANNER;
  ret = umax_do_scsi_cmd(us->sfd, release_unit.cmd, release_unit.size, NULL, 0);
  if (ret)  { return ret; }
  DBG(3, "scanner released\n");
  return 0;
}


/* ------------------------ UMAX SEND GAMMA DATA --------------------------- */


#ifdef UMAX_TO_SANE
static
#endif
void umax_send_gamma_data(US *us, char *data, int color)
{
 char *dest;
 int length;

  DBG(5, "send_gamma_data\n");
  memcpy(us->buffer, send.cmd, send.size);                          /* send */
  set_S_datatype_code(us->buffer, S_datatype_gamma);  /* gamma curve */
  set_S_xfer_length(us->buffer, color*256+gamma_DCF2.size); /* bytes */

  dest=us->buffer + send.size;
  memcpy(dest, gamma_DCF2.cmd, gamma_DCF2.size);     /* gamma format type 2 */

  set_DCF2_gamma_color(dest, DCF2_gamma_color_grey);            /* greyscale */
  if ( (us->colormode == RGB) && (us->three_pass != 0) )     /* 3 pass color */
  { set_DCF2_gamma_color(dest, us->three_pass_color); }         /*  set color*/

  if (color == 1)
  { set_DCF2_gamma_lines(dest, DCF2_gamma_one_line); }
  else
  { set_DCF2_gamma_lines(dest, DCF2_gamma_three_lines); }

  set_DCF2_gamma_input_bits(dest, us->gamma_input_bits);
  set_DCF2_gamma_output_bits(dest, us->bits_per_pixel);

  dest=us->buffer + send.size + gamma_DCF2.size;            /* write to dest */

  if (us->gamma_input_bits == 1)
  { length = 256; }                                          /* 8 input bits */
  else
  { length = 1024; }                                        /* 10 input bits */
  if (us->bits_per_pixel != 1)          /* more than 8 output bits per pixel */
  { length = length*2; }                                 /* = 2 output bytes */

  memcpy(dest, data, color*length);

  umax_do_scsi_cmd(us->sfd, us->buffer, send.size+gamma_DCF2.size+length*color,
                   NULL, 0);
}


/* --------------------- UMAX SEND HALFTONE PATTERN ------------------------ */


#ifdef UMAX_TO_SANE
static
#endif
void umax_send_halftone_pattern(US *us, char *data, int size)
{
 char *dest;

  DBG(5, "send_halftone_pattern\n");
  memcpy(us->buffer, send.cmd, send.size);                           /* send */
  set_S_datatype_code(us->buffer, S_datatype_halftone);  /* halftone-pattern */
  set_S_xfer_length(us->buffer, size * size);                       /* bytes */

  dest=us->buffer + send.size;
  memcpy(dest, data, size * size);                              /* copy data */

  umax_do_scsi_cmd(us->sfd, us->buffer, send.size + size * size, NULL, 0);
}


/* --------------------------- UMAX CORRECT LIGHT -------------------------- */


#ifdef UMAX_TO_SANE
static
#endif
int umax_correct_light(int light, int analog_gamma_byte)
{ 
  /* calculates highlight and shadow if analog gamma is set */
  double analog_gamma;
  analog_gamma=analog_gamma_table[analog_gamma_byte];
  return( (int) 255 * pow(  ((double) light)/255.0 , (1.0/analog_gamma) )+.5 );
}


/* -------------------------- UMAX SET WINDOW PARAM ------------------------ */


/* set_window_param sets all the window parameters.  This
 * means building a fairly complicated SCSI command before
 * sending it...
 */

#ifdef UMAX_TO_SANE
static
#endif
int umax_set_window_param(US *us)
{
 int num_dblocks = 1; /* number of window descriptor blocks, usually 1 or 3 */

 unsigned char buffer_r[max_WDB_size],
               buffer_g[max_WDB_size],
               buffer_b[max_WDB_size];

  DBG(5, "set_window_param\n");
  memset(buffer_r , '\0', max_WDB_size);                   /* clear buffer */
  set_WDB_length(us->wdb_len);           /* length of win descriptor block */
  memcpy(buffer_r,window_descriptor_block.cmd,
         window_descriptor_block.size);                /* copy preset data */

  set_WD_wid(buffer_r, 0);                            /* window identifier */
  set_WD_auto(buffer_r, us->set_auto);    /* 0 or 1: don't know what it is */

                                            /* geometry */
  set_WD_Xres(buffer_r, us->x_resolution);          /* x resolution in dpi */
  set_WD_Yres(buffer_r, us->y_resolution);          /* y resolution in dpi */
  set_WD_ULX(buffer_r, us->upper_left_x);                   /* left_edge x */
  set_WD_ULY(buffer_r, us->upper_left_y);                  /* upper_edge y */
  set_WD_width(buffer_r, us->phys_width);                         /* width */
  set_WD_length(buffer_r, us->phys_height);                      /* height */

                                                     /* BTC */
  set_WD_brightness(buffer_r, us->brightness);/* brightness, only halftone */
  set_WD_threshold(buffer_r, us->threshold);    /* threshold, only lineart */
  set_WD_contrast(buffer_r, us->contrast);      /* contrast, only halftone */
    
                            /* scanmode, preset to LINEART */  
  set_WD_composition(buffer_r, WD_comp_lineart);      /* image composition */
                                                          /* = (scan-mode) */
  set_WD_bitsperpixel(buffer_r, WD_bits_1);         /* bits/pixel (1,8,10) */
  set_WD_halftone(buffer_r, us->halftone);      /* select halftone-pattern */
  set_WD_RIF(buffer_r, us->reverse);    /* reverse, invert black and white */
  set_WD_speed(buffer_r, us->WD_speed);                       /* set speed */
  set_WD_select_color(buffer_r, WD_color_grey);  /* color for window-block */ 

           /* set highlight and shadow in dependence of analog gamma */ 
  set_WD_highlight(buffer_r,
                   umax_correct_light(us->highlight_r,us->analog_gamma_r));
  set_WD_shadow(buffer_r, 
                umax_correct_light(us->shadow_r,us->analog_gamma_r));

                                           /* scan options */
         /* paper length not used !!! */
  set_WD_gamma(buffer_r, us->digital_gamma_r);        /* set digital gamma */ 
  set_WD_module(buffer_r, us->module);          /* flatbed or transparency */ 
  set_WD_CBHS(buffer_r, us->cbhs_range);                      /* 50 or 255 */ 
  set_WD_RMIF(buffer_r, us->reverse_multi);        /* reverse color-values */
  set_WD_DOR(buffer_r, us->dor);                 /* double-resolution-mode */ 
  set_WD_scan_exposure_level(buffer_r,
             us->exposure_time_scan_r);              /* scan exposure time */
  set_WD_calibration_exposure_level(buffer_r,
             us->exposure_time_calibration_r); /* calibration exposure time*/
       /* batch-scan not used in the moment ------------------------------ */
  set_WD_line_arrangement(buffer_r,
             WD_line_arrengement_by_fw);   /* line arrangement by firmware */
  set_WD_warmup(buffer_r, us->warmup);

  if (us->calibration == 1)
  { set_WD_calibration(buffer_r, WD_calibration_image); }   /* image calib */
  else
  { set_WD_calibration(buffer_r, WD_calibration_ignore); } /* ignore calib */

  set_WD_color_sequence(buffer_r,WD_color_sequence_RGB);   /* sequence RGB */
  set_WD_color_ordering(buffer_r,
            WD_color_ordering_pixel); /* set to pixel for pbm,pgm,pnm-file */
  set_WD_analog_gamma(buffer_r, us->analog_gamma_r );      /* analog gamma */
  set_WD_lamp_c_density(buffer_r, us->c_density); /* calirat. lamp density */
  set_WD_lamp_s_density(buffer_r, us->s_density);     /* scan lamp density */
     /* upper left y for batch scan not used ----------------------------- */
  set_WD_pixel_count(buffer_r, us->width_in_pixels);        /* pixel count */
  set_WD_line_count(buffer_r, us->height_in_pixels);         /* line count */
  set_WD_x_coordinate_base(buffer_r, us->x_coordinate_base); /* dpi (1200) */
  set_WD_y_coordinate_base(buffer_r, us->y_coordinate_base); /* dpi (1200) */
  set_WD_calibration_data_lines(buffer_r, 0);                /* ?????????? */


  switch(us->colormode)
  {
     case LINEART: 
      set_WD_composition(buffer_r, WD_comp_lineart);
      set_WD_bitsperpixel(buffer_r, WD_bits_1);
      set_WD_select_color(buffer_r, WD_color_grey);
      if (us->calibration == 2)
      { set_WD_calibration(buffer_r, WD_calibration_lineart); }
     break;

     case HALFTONE: 
      set_WD_composition(buffer_r, WD_comp_dithered);
      set_WD_bitsperpixel(buffer_r, WD_bits_1);
      set_WD_select_color(buffer_r, WD_color_grey);
      if (us->calibration == 2)
      { set_WD_calibration(buffer_r, WD_calibration_dither); }
     break;

     case GREYSCALE:
      set_WD_composition(buffer_r, WD_comp_grey);

      if ( us->bits_per_pixel == 1)                      /* 8 bits per pixel */
      { set_WD_bitsperpixel(buffer_r, 8); }
      else 
      {
        set_WD_bitsperpixel(buffer_r, 10);              /* 10 bits per pixel */
        set_WD_HBT(buffer_r, WD_HBT_HBF);                  /* highbyte first */
      }

      set_WD_select_color(buffer_r, WD_color_grey);
      if (us->calibration == 2)
      { set_WD_calibration(buffer_r, WD_calibration_grey); }
     break;

     case RGB_LINEART:
     case RGB_HALFTONE:
     case RGB:
      if (us->colormode == RGB_LINEART )
      {
        set_WD_composition(buffer_r, WD_comp_rgb_bilevel);
        set_WD_bitsperpixel(buffer_r, WD_bits_1);
        if (us->calibration == 2)
        { set_WD_calibration(buffer_r, WD_calibration_lineart); }
      }
      else if (us->colormode == RGB_HALFTONE )
      {
        set_WD_composition(buffer_r, WD_comp_rgb_dithered);
        set_WD_bitsperpixel(buffer_r, WD_bits_1);
        if (us->calibration == 2)
        { set_WD_calibration(buffer_r, WD_calibration_dither); }
      }
      else /* RGB */
      {
        set_WD_composition(buffer_r, WD_comp_rgb_full);

        if ( us->bits_per_pixel == 1)                   /* 24 bits per pixel */
        { set_WD_bitsperpixel(buffer_r, 8); }
        else 
        {
          set_WD_bitsperpixel(buffer_r, 10);            /* 30 bits per pixel */
          set_WD_HBT(buffer_r, WD_HBT_HBF);                /* highbyte first */
	}

        if (us->calibration == 2)
        { set_WD_calibration(buffer_r, WD_calibration_rgb); }
      }

      if (us->three_pass == 0)
      {                                                       /* singlepass */
        num_dblocks = 3;

        memcpy(buffer_g, buffer_r, max_WDB_size);      /* copy WDB for green */
        memcpy(buffer_b, buffer_r, max_WDB_size);       /* copy WDB for blue */

        set_WD_wid(buffer_r, WD_wid_red);               /* window identifier */
        set_WD_wid(buffer_g, WD_wid_green); 
        set_WD_wid(buffer_b, WD_wid_blue);

        set_WD_select_color(buffer_r, WD_color_red);                /* color */
        set_WD_select_color(buffer_g, WD_color_green);
        set_WD_select_color(buffer_b, WD_color_blue);

        set_WD_gamma(buffer_r, us->digital_gamma_r);        /* digital gamma */
        set_WD_gamma(buffer_g, us->digital_gamma_g);
        set_WD_gamma(buffer_b, us->digital_gamma_b);

        set_WD_analog_gamma(buffer_r, us->analog_gamma_r);   /* analog gamma */
        set_WD_analog_gamma(buffer_g, us->analog_gamma_g);
        set_WD_analog_gamma(buffer_b, us->analog_gamma_b);

                              /* set highlight in dependence of analog gamma */ 
        set_WD_highlight(buffer_r,
                   umax_correct_light(us->highlight_r,us->analog_gamma_r));
        set_WD_highlight(buffer_g,
                   umax_correct_light(us->highlight_g,us->analog_gamma_g));
        set_WD_highlight(buffer_b,
                   umax_correct_light(us->highlight_g,us->analog_gamma_b));

                              /* set highlight in dependence of analog gamma */ 
        set_WD_shadow(buffer_r, 
                umax_correct_light(us->shadow_r,us->analog_gamma_r));
        set_WD_shadow(buffer_g, 
                umax_correct_light(us->shadow_g,us->analog_gamma_g));
        set_WD_shadow(buffer_b, 
                umax_correct_light(us->shadow_b,us->analog_gamma_b));


        set_WD_scan_exposure_level(buffer_r,
               us->exposure_time_scan_r);          /* scan exposure time red */
        set_WD_calibration_exposure_level(buffer_r,
               us->exposure_time_calibration_r); /* calibration exposure time*/

        set_WD_scan_exposure_level(buffer_g,
               us->exposure_time_scan_g);        /* scan exposure time green */
        set_WD_calibration_exposure_level(buffer_g,
               us->exposure_time_calibration_g); /* calibration exposure time*/

        set_WD_scan_exposure_level(buffer_b,
               us->exposure_time_scan_b);         /* scan exposure time blue */
        set_WD_calibration_exposure_level(buffer_b,
               us->exposure_time_calibration_b); /* calibration exposure time*/
      }
      else
      {                                                         /* threepass */
        set_WD_wid(buffer_r, 0);                        /* window identifier */
        set_WD_color_ordering(buffer_r, WD_color_ordering_plane);    /* ???? */

        if (us->colormode == RGB_LINEART )
        { set_WD_composition(buffer_r, WD_comp_lineart); } /* only one color */
        else if (us->colormode == RGB_HALFTONE )
        { set_WD_composition(buffer_r, WD_comp_dithered); }/* only one color */
        else /* RGB */
        { set_WD_composition(buffer_r, WD_comp_grey); }    /* only one color */

        switch (us->three_pass_color)
        {
        case WD_wid_red:
           set_WD_select_color(buffer_r, WD_color_red);         /* color red */
           set_WD_gamma(buffer_r, us->digital_gamma_r);     /* digital gamma */
           set_WD_analog_gamma(buffer_r,us->analog_gamma_r); /* analog gamma */
           set_WD_highlight(buffer_r,
                      umax_correct_light(us->highlight_r, us->analog_gamma_r));
           set_WD_shadow(buffer_r,
                      umax_correct_light(us->shadow_r, us->analog_gamma_r)); 
           set_WD_scan_exposure_level(buffer_r,
               us->exposure_time_scan_r);          /* scan exposure time red */
           set_WD_calibration_exposure_level(buffer_r,
               us->exposure_time_calibration_r); /* calibration exposure time*/
           break;

        case WD_wid_green:
           set_WD_select_color(buffer_r, WD_color_green);     /* color green */
           set_WD_gamma(buffer_r, us->digital_gamma_g);     /* digital gamma */
           set_WD_analog_gamma(buffer_r,us->analog_gamma_g); /* analog gamma */
           set_WD_highlight(buffer_r,
                      umax_correct_light(us->highlight_g, us->analog_gamma_g));
           set_WD_shadow(buffer_r,
                      umax_correct_light(us->shadow_g, us->analog_gamma_g));
           set_WD_scan_exposure_level(buffer_r,
               us->exposure_time_scan_g);        /* scan exposure time green */
           set_WD_calibration_exposure_level(buffer_r,
               us->exposure_time_calibration_g); /* calibration exposure time*/
           break;

        case WD_wid_blue:
           set_WD_select_color(buffer_r, WD_color_blue);       /* color blue */
           set_WD_gamma(buffer_r, us->digital_gamma_b);     /* digital gamma */
           set_WD_analog_gamma(buffer_r,us->analog_gamma_b); /* analog gamma */
           set_WD_highlight(buffer_r,
                      umax_correct_light(us->highlight_b, us->analog_gamma_b));
           set_WD_shadow(buffer_r,
                      umax_correct_light(us->shadow_b, us->analog_gamma_b));
           set_WD_scan_exposure_level(buffer_r,
               us->exposure_time_scan_b);         /* scan exposure time blue */
           set_WD_calibration_exposure_level(buffer_r,
               us->exposure_time_calibration_b); /* calibration exposure time*/
           break;

        } /* switch us->three_pass_color */

      } /* if (single_pass) else (three_pass) */
     break;
  } /* switch us->colormode, case RGB */

                                                      /* prepare SCSI-BUFFER */
    memcpy(us->buffer, set_window.cmd, set_window.size);  /* SET-WINDOW cmd */
    memcpy(WPDB_OFF(us->buffer),                        /* add WPDB */
            window_parameter_data_block.cmd, 
            window_parameter_data_block.size);
    set_WPDB_wdbnum(WPDB_OFF(us->buffer), num_dblocks);/* set WD_len */
    memcpy(WDB_OFF(us->buffer,1), buffer_r,                 /* add WD_block */
            window_descriptor_block.size); 

    if ( num_dblocks == 3)                              /* if singelpass RGB */
    {
       memcpy(WDB_OFF(us->buffer,2), buffer_g,
               window_descriptor_block.size);                   /* add green */
       memcpy(WDB_OFF(us->buffer,3), buffer_b,
               window_descriptor_block.size);                    /* add blue */
    }


    DBG(4, "window descriptor block created with %d bytes\n", us->wdb_len);

    set_SW_xferlen(us->buffer, (window_parameter_data_block.size +
                            (window_descriptor_block.size * num_dblocks)));

    {
     int ret;

      ret = umax_do_scsi_cmd(us->sfd, us->buffer, set_window.size +
                       window_parameter_data_block.size +
                       (window_descriptor_block.size * num_dblocks),
                       NULL, 0);
      if (ret) { return ret; }
    }

    DBG(3, "window(s) set\n"); 

return (umax_wait_scanner(us));            /* wait for scanner, lamp warmup */
}


/* -------------------------- UMAX START SCAN ------------------------------ */


#ifdef UMAX_TO_SANE
static
#endif
int umax_start_scan(US *us)
{
 int size = 1;

  DBG(3,"starting scan\n");

  set_SC_quality(    scan.cmd, us->quality);    /*  1=qual, 0=fast */
  set_SC_adf(        scan.cmd, us->adf);      /* ADF, 0=off, 1=use */
  set_SC_preview(    scan.cmd, us->preview);          /* 1=preview */

  if (us->RGB_PREVIEW_PATCH != 0)
  {
    /* in RGB-mode set preview bit, eg. for UMAX S6E */
    if (us->colormode == RGB) { set_SC_preview(    scan.cmd, 1); }
  }
  
  set_SC_wid(        scan.cmd, 1, 0);         /* Window-Identifier */

  if (us->START_SCAN_PATCH != 0)
  {
    if ( (us->three_pass == 0) && (us->colormode == RGB) ) /* singlepass rgb */
    {
      size = 3;
      set_SC_wid(scan.cmd, 1, WD_wid_red);
      set_SC_wid(scan.cmd, 2, WD_wid_green);
      set_SC_wid(scan.cmd, 3, WD_wid_blue);
    }
  }

  set_SC_xfer_length(scan.cmd, size);           /* following Bytes */

return umax_do_scsi_cmd(us->sfd, scan.cmd, scan.size + size, NULL, 0);
}


/* ------------------------ UMAX READ DATA BLOCK --------------------------- */


#ifdef UMAX_TO_SANE
static
#endif
int umax_read_data_block(US *us, unsigned int length)
{
   DBG(5,"read_data_block\n");
   WAIT_SCANNER;
   set_R_xfer_length(sread.cmd, length);
   if (umax_do_scsi_cmd(us->sfd, sread.cmd, sread.size,
                          us->buffer, length) != 0) { return -1; } 
return length;
}


/* -------------------------- UMAX DO INQUIRY ------------------------------ */


#ifdef UMAX_TO_SANE
static
#endif
void umax_do_inquiry(US *us)
{
 int size;

  DBG(5,"do_inquiry\n");
  memset(us->buffer, '\0', 256);                    /* clear buffer */

              /* first get only 5 bytes to get size of inquiry_return_block */
  set_inquiry_return_size(inquiry.cmd, 5);
  umax_do_scsi_cmd(us->sfd, inquiry.cmd, inquiry.size, us->buffer, 5);

  size = get_inquiry_additional_length(us->buffer) + 5;

                                       /* then get inquiry with actual size */
  set_inquiry_return_size(inquiry.cmd, size);
  umax_do_scsi_cmd(us->sfd, inquiry.cmd, inquiry.size, us->buffer, size);
}



/* ------------------------ UMAX IDENTIFY SCANNER -------------------------- */



#ifdef UMAX_TO_SANE
static
#endif
int umax_identify_scanner(US *us)
{
 char vendor[9];
 char product[0x11];
 char version[5];
 char *pp;
 int  i;

  DBG(5,"identify_scanner\n");
  umax_do_inquiry(us);                                       /* get inquiry */
  if (get_inquiry_periph_devtype(us->buffer) !=        /* scanner ? */
      IN_periph_devtype_scanner) { return 1; }    /* no, continue searching */ 

  get_inquiry_vendor( us->buffer, vendor);
  get_inquiry_product(us->buffer, product);
  get_inquiry_version(us->buffer, version);
  if (strncmp("UMAX    ", vendor, 8)) { return 1; }   /* Not a UMAX product */

  pp = &vendor[8];
  vendor[8] = ' ';
  while (*pp == ' ') { *pp-- = '\0'; }

  pp = &product[0x10];
  product[0x10] = ' ';
  while (*(pp - 1) == ' ') { *pp-- = '\0'; } /* leave one blank at the end! */
  
  pp = &version[4];
  version[4] = ' ';
  while (*pp == ' ') { *pp-- = '\0'; }
  
  DBG(3, "Found UMAX scanner %sversion %s on device %s\n",
                     product, version, us->devicename);

  /* look for scanners that do not give all inquiry-informations */
  /* and if possible use driver-known inquiry-data  */

  if (get_inquiry_additional_length(us->buffer)>=0x8f)
  {
    /* Now identify full supported scanners */
    for(i=0; i<known_scanners; i++)
    {
      if (!strncmp(product, scanner_str[i], strlen(scanner_str[i])))
      {return 0;} 
    }
    if (us->cont != 0) { return 0; }         /* return if --contionue is set */

    DBG(2, "WARNING: UMAX scanner %s version %s on device %s\n"
     "is currently an unrecognized device, but inquiry seems ok.\n"
     "You may continue with option --continue - but do it on your own risk!\n"
     "Please do 'umax -I 2>text' and send it to\n"
     "Oliver.Rauch@Wolfsburg.DE\n",
     product, version, us->devicename);
    return 1; /* no supported scanner */
  }
  else /* inquiry-data not complete */
  {
    inquiry_blk inq_data;
    for(i=0; i<known_inquiry; i++)
    {
      inq_data = *inquiry_table[i];
      if (!strncmp(product, inq_data.scanner, strlen(inq_data.scanner))) 
      {
	DBG(4, "inquiry-block-length: %d\n"
	       "using driver-internal inquiry-data for this scanner!\n",
               get_inquiry_additional_length(us->buffer)+5);

        /* copy driver-defined inquiry-data into inquiry-block */
        memcpy( us->buffer+0x24,
	        inq_data.inquiry,
		inq_data.inquiry_len-0x24);

        set_inquiry_length(us->buffer,inq_data.inquiry_len); 

        return 0; /* ok */
      }
    }
  }
  DBG(1, "ERROR: UMAX scanner %s version %s on device %s\n"
         "is currently an unrecognized device, and inquiry is too short,\n"
         "so we are not able to continue!\n"
         "Please contact Oliver.Rauch@Wolfsburg.DE\n",
         product, version, us->devicename);

return 1; /* short inquiry-block, continue not possible !!! */
}



/* -------------------------- UMAX OPEN SCANNER ---------------------------- */


/* open_scanner tries to open the scanner.  It returns 0 on success,
 * and negative numbers on failure.
 * If no device name is specified on the command line, it hunts for
 * the first UMAX scanner on an sg device.
 * If a device name is specified on the command line, it trusts it
 * and does not test it as long as it can be opened.
 * If verbose output is enabled and it is searching for a scanner,
 * it tells the world where it found the scanner.
 */

#ifndef UMAX_TO_SANE
int umax_open_scanner(US *us)
{
 int i;
 char filename[64]; /* only needs to hold "/dev/scanner"; this is overkill */
 char s[] = "abcdefgh01234567";

  DBG(5,"open_scanner\n");
  if (us->devicename && strlen(us->devicename)) 
  {                                                /* take specified device */
    if (umax_do_scsi_open(us->devicename, &(us->sfd), sense_handler)) { return 1; }
    if (umax_identify_scanner(us)==0)  { return 0; }
  }
  else                                                 /* probe for scanner */
  {
    sprintf(filename, "/dev/scanner");
    us->devicename=filename;
    if (umax_do_scsi_open(us->devicename, &(us->sfd), sense_handler) == 0)
    { if (umax_identify_scanner(us) == 0) { return 0; } }

    for (i = 0; i < sizeof(s)-1; i++)
    {
      sprintf(filename, "/dev/sg%c", s[i]);
      us->devicename=filename;
      if (umax_do_scsi_open(us->devicename, &(us->sfd), sense_handler) == 0)
      { if (umax_identify_scanner(us) == 0) { return 0; } }
    }
  }

  /* if we reach here, we haven't found a UMAX scanner on any device */

  umax_do_scsi_close(us->sfd);               /* no scanner: close device */
  DBG(1, "No UMAX scanner found\n"); 

return -2;
}
#endif

/* ---------------------------- UMAX TRIM BUFSIZE -------------------------- */



/* For scanners that need neatly-sized reads, use code like this:
 * us->row_bufsize = trim_bufsize(us->bufsize, us->row_len);
 */

#ifdef UMAX_TO_SANE
static
#endif
void umax_trim_rowbufsize(US *us)
{
  us->row_bufsize =  (us->row_bufsize < us->row_len) ? us->row_bufsize
                      : us->row_bufsize - (us->row_bufsize % us->row_len);
  DBG(5,"trim_bufsize to %d\n",us->row_bufsize);
}


/* ------------------------ UMAX BOUND SCANAREA  ---------------------------- */


#ifdef UMAX_TO_SANE
static
#endif
int umax_bound_scanarea(US *us)
{
 double inquiry_width;
 double inquiry_length;
 int bytes;

  DBG(5,"bound_scanarea\n");

  /* get witdh and length of scanarea in dependence of scanmode */

  if (us->module == WD_module_flatbed)
  {
     inquiry_width =us->inquiry_fb_width;
     inquiry_length=us->inquiry_fb_length;
  }
  else /* WD_module_transparency */
  {
     inquiry_width =us->inquiry_uta_width;
     inquiry_length=us->inquiry_uta_length;
  }

  if (us->dor != 0)
  {
     inquiry_width =us->inquiry_dor_width;
     inquiry_length=us->inquiry_dor_length;
  }

  if ((inquiry_width > 0) && (inquiry_length > 0))
  {
    us->width =inquiry_width;
    us->length=inquiry_length;
  }


  /* bound_resolution limits the resolution to what the scanner is capable of */
  if ( (us->x_resolution <= 0) || (us->x_resolution > us->inquiry_x_res) )
  { us->x_resolution = us->inquiry_x_res; }
  if ( (us->y_resolution <= 0) || (us->y_resolution > us->inquiry_y_res) )
  { us->y_resolution = us->inquiry_y_res; }

/* limit the size to what the scanner can scan.
 * this is particularly important because the scanners don't have
 * built-in checks and will happily grind their gears if this is exceeded.
 */

/* bound width and height to stay within scanner, so that
 * -R n for n < resolution doesn't confuse the scanner */

  {
   int maxwidth;

    maxwidth = us->x_resolution * (us->width - (us->upper_left_x / 1200.0) ) ;
    if ( (us->width_in_pixels <= 0) || (us->width_in_pixels > maxwidth) )
    { us->width_in_pixels = maxwidth; }
  }

  {
   int maxheight;

    maxheight = us->y_resolution * (us->length - (us->upper_left_y / 1200.0) );
    if ( (us->height_in_pixels <= 0) || (us->height_in_pixels > maxheight) )
    { us->height_in_pixels = maxheight; }
  }


/* Now calculate the absolute width */
  us->phys_width  = (us->width_in_pixels ) * 1200 / us->x_resolution;
  us->phys_height = (us->height_in_pixels) * 1200 / us->y_resolution;

  if ((us->phys_width <= 0) || (us->phys_height <= 0))
  { DBG(1,"ERROR: the geometric values are confusing me\n"); return 1; }

/* bound_linelengths_for_colormode gives sane line lengths for each
 * color model.  It might not be strictly necessary; I'm not sure. */

  if (us->bits_per_pixel == 1)
  { bytes = 1; }
  else
  { bytes = 2; }

  switch(us->colormode)
  {
   case LINEART:
     /* make sure line width divides evenly by 8... */
     us->width_in_pixels -= us->width_in_pixels % 8;
     us->phys_width = 1200 * us->width_in_pixels / us->x_resolution;
     us->row_len = (us->width_in_pixels / 8);
    break;

   case HALFTONE:
     /* don't know if it is right : */
     us->width_in_pixels -= us->width_in_pixels % 8;
     us->phys_width = 1200 * us->width_in_pixels / us->x_resolution;
     us->row_len = (us->width_in_pixels / 8);
    break;

   case GREYSCALE:
     /* one byte per pixel */
     us->row_len = us->width_in_pixels * bytes;
    break;

   case RGB_LINEART:
   case RGB_HALFTONE:
     us->width_in_pixels -= us->width_in_pixels % 8;
     us->phys_width = 1200 * us->width_in_pixels / us->x_resolution;
     if (us->three_pass)
     { us->row_len = us->width_in_pixels / 8 ; }
     else
     { us->row_len = (us->width_in_pixels / 8 ) * 3; }
    break;

   case RGB:
     /* three (24bpp) or six (30bpp) bytes per pixel */
     if (us->three_pass)
     { us->row_len = us->width_in_pixels * bytes; }
     else
     { us->row_len = us->width_in_pixels * 3 * bytes; }
    break;
  }


return 0;
}


/* ---------------------- UMAX CALCULATE EXPOSURE TIME --------------------- */

  
#ifdef UMAX_TO_SANE
static
#endif
void umax_calculate_exposure_time(US *us, int def,int *value)
{
 int level;

  DBG(5,"calculate_exposure_time\n");
  if ( (*value))
  {
    if ( (*value) == -1 ) { (*value) = def; }
    else
    {
      level = (*value) / us->inquiry_exposure_time_step_unit;
      (*value) = inrange(us->use_exposure_time_min,
                  level,
		  us->inquiry_exposure_time_max);
    }
  }
}

			  
/* ------------------------ UMAX CHECK VALUES ----------------------------- */


#ifdef UMAX_TO_SANE
static
#endif
int umax_check_values(US *us)
{
  DBG(5,"check_values\n");

  /* ------------------------------- resolution ------------------------ */

  if (us->x_resolution <= 0)
  { DBG(1,"ERROR: no x-resolution given\n"); return(1); }

  if (us->y_resolution <= 0)
  { DBG(1,"ERROR: no y-resolution given\n"); return(1); }

  /* ------------------------------- scanarea ------------------------ */

  if (us->width <= 0)
  { DBG(1, "ERROR: no width of scanarea given\n"); return(1); }

  if (us->length <= 0) 
  { DBG(1, "ERROR: no height of scanarea given\n"); return(1); }
  
  /* ------------------------------- wdb length ------------------------ */

  if (us->wdb_len <= 0)
  {
    us->wdb_len = us->inquiry_wdb_len; 
    if (us->wdb_len <= 0)
    {
      DBG(1,"ERROR: wdb-length not given\n");
      return(1);
    }
  }

  if (us->wdb_len > used_WDB_size)
  {
    DBG(2,"WARNING:window descriptor block too long, will be shortned!\n");
    us->wdb_len = used_WDB_size;
  }

  /* ----------------------------- cbhs-range ----------------------------- */

  us->threshold   = umax_cbhs_correct(us->inquiry_cbhs_min,
                                us->threshold , us->inquiry_cbhs_max);
  us->contrast    = umax_cbhs_correct(us->inquiry_cbhs_min,
                                us->contrast  , us->inquiry_cbhs_max);
  us->brightness  = umax_cbhs_correct(us->inquiry_cbhs_min,
                                us->brightness, us->inquiry_cbhs_max);

  us->highlight_r = umax_cbhs_correct(us->inquiry_cbhs_min+1,
                                us->highlight_r , us->inquiry_cbhs_max);
  us->highlight_g = umax_cbhs_correct(us->inquiry_cbhs_min+1,
                                us->highlight_g , us->inquiry_cbhs_max);
  us->highlight_b = umax_cbhs_correct(us->inquiry_cbhs_min+1,
                                us->highlight_b , us->inquiry_cbhs_max);

  us->shadow_r    = umax_cbhs_correct(us->inquiry_cbhs_min,
                                us->shadow_r    , us->inquiry_cbhs_max-1);
  us->shadow_g    = umax_cbhs_correct(us->inquiry_cbhs_min,
                                us->shadow_g    , us->inquiry_cbhs_max-1);
  us->shadow_b    = umax_cbhs_correct(us->inquiry_cbhs_min,
                                us->shadow_b    , us->inquiry_cbhs_max-1);

  if (us->shadow_r >= us->highlight_r) { us->shadow_r = us->highlight_r-1; }
  if (us->shadow_g >= us->highlight_g) { us->shadow_g = us->highlight_g-1; }
  if (us->shadow_b >= us->highlight_b) { us->shadow_b = us->highlight_b-1; }

  /* --------------------------- quality calibration ---------------------- */

  if (us->inquiry_quality_ctrl == 0)
  {
    if (us->quality)
    {
      DBG(2, "WARNING: quality calibration not supported by scanner\n");
      us->quality = 0;
    }
  }

  /* -------------------------------- preview ----------------------------- */

  if (us->inquiry_preview == 0)
  {
    if (us->preview)
    {
      DBG(2, "WARNING: fast preview function not supported by scanner\n");
      us->preview = 0;
    }
  }

  /* --------------------------- lamp intensity control ------------------- */

  if (us->inquiry_lamp_ctrl == 0)
  {
    if (us->c_density || us->s_density)
    { DBG(2, "WARNING: scanner doesn't support lamp intensity control\n"); }
    us->c_density = us->s_density = 0;
  }

  /* --------------------------------- uta --------------------------------- */

  if (us->uta != 0) 
  {
    us->module = WD_module_transparency;
    if ( (us->inquiry_uta == 0) || (us->inquiry_transavail == 0) )
    {
      DBG(1, "ERROR: TRANSPARENCY-MODE NOT SUPPORTED BY SCANNER, ABORTING\n");
      return(1);
    }
  }

  /* --------------------------------- adf --------------------------------- */

  if (us->adf != 0) 
  {
    if (us->inquiry_adf == 0)
    {
      DBG(1,"ERROR: ADF-MODE NOT SUPPORTED BY SCANNER, ABORTING\n");
      return(1);
    }
  }

  /* --------------------------------- dor --------------------------------- */

  if (us->dor != 0)
  {
    if (us->inquiry_dor == 0)
    {
       DBG(1, "ERROR: double optical resolution not supported by scanner\n");
       return(1); 
    }
  }

  /* -------------------------------- reverse ------------------------------ */

  if (us->reverse != 0)
  {
    if ((us->colormode == LINEART) ||
        (us->colormode == HALFTONE) ||
        (us->colormode == RGB_LINEART) ||
        (us->colormode == RGB_HALFTONE) )
    {
      if (us->inquiry_reverse == 0)
      {
         DBG(1, "ERROR: reverse for bi-level-image not supported\n");
         return(1);
      }
    }
    else
    { us->reverse = 0; }
  }

  if (us->reverse_multi != 0)
  {
    if ((us->colormode == RGB) || (us->colormode == GREYSCALE) )
    {
      if (us->inquiry_reverse_multi == 0)
      {
         DBG(1, "ERROR: reverse for multi-level-image not supported\n");
         return(1);
      }
    }
    else
    { us->reverse_multi = 0; }
  }


  /* ----------------------------- analog gamma ---------------------------- */

  if (us->inquiry_analog_gamma == 0)
  {
    if (us->analog_gamma_r + us->analog_gamma_g + us->analog_gamma_b != 0)
    { DBG(2,"WARNING: analog gamma correction not supported by scanner!\n"); }
    us->analog_gamma_r = us->analog_gamma_g = us->analog_gamma_b = 0;
  }

  /* ---------------------------- digital gamma ---------------------------- */

  if ( (us->digital_gamma_r == 0) || (us->digital_gamma_g == 0) ||
       (us->digital_gamma_b == 0) )
  {
    if (us->inquiry_gamma_dwload == 0)
    {
      DBG(2, "WARNING: gamma download not available\n");
      us->digital_gamma_r = us->digital_gamma_g = us->digital_gamma_b = 15;
      us->gamma_r = us->gamma_g = us->gamma_b = 1.0;
    }
    
  }

  /* ---------------------------- speed and smear  ------------------------- */
  
  if (us->slow != 0)
  {us->WD_speed=WD_speed_slow;}
  else
  {us->WD_speed=WD_speed_fast;}

  if (us->smear != 0) {us->WD_speed+=WD_speed_smear;}
  
  /* ------------------------------ test 10 bpp  --------------------------- */
  
  if ( ( (us->inquiry_GIB | 1) & us->gamma_input_bits) == 0 )
  {
    DBG(2,"WARNING: selected gamma input bits not supported, gamma ignored\n");
    us->gamma_input_bits = 1;
    us->digital_gamma_r = us->digital_gamma_g = us->digital_gamma_b = 15;
    us->gamma_r = us->gamma_g = us->gamma_b = 1.0;
  }

  if ( ( (us->inquiry_GOB | 1) & us->bits_per_pixel) == 0 )
  {
    DBG(1,"ERROR: selected bits per pixel not supported\n");
    return(1);
  }
  

  /* ----------------------------- exposure times -------------------------- */

  switch(us->colormode)
  {
  case LINEART:     /* ------------ LINEART ------------- */
  case RGB_LINEART: /* ---------- RGB_LINEART ----------- */
   us->use_exposure_time_min = us->inquiry_exposure_time_l_min;
   if (us->module == WD_module_flatbed) 
   { us->use_exposure_time_def_r = us->inquiry_exposure_time_l_fb_def; }
   else
   { us->use_exposure_time_def_r = us->inquiry_exposure_time_l_uta_def; }
   if (us->inquiry_lineart == 0)
   {
    DBG(1,"ERROR: LINEART-MODE NOT SUPPORTED BY SCANNER, ABORTING\n");
    return(1);
   }
   break;

  case HALFTONE:     /* ----------- HALFTONE------------ */
  case RGB_HALFTONE: /* --------- RGB_HALFTONE---------- */
   us->use_exposure_time_min = us->inquiry_exposure_time_h_min;
   if (us->module == WD_module_flatbed) 
   { us->use_exposure_time_def_r = us->inquiry_exposure_time_h_fb_def; }
   else
   { us->use_exposure_time_def_r = us->inquiry_exposure_time_h_uta_def; }
   if (us->inquiry_halftone == 0)
   {
    DBG(1,"ERROR: HALFTONE-MODE NOT SUPPORTED BY SCANNER, ABORTING\n");
    return(1);
   }
  break;

  case GREYSCALE: /* ---------- GREYSCALE ------------- */
   us->use_exposure_time_min = us->inquiry_exposure_time_g_min;
   if (us->module == WD_module_flatbed) 
   { us->use_exposure_time_def_r = us->inquiry_exposure_time_g_fb_def; }
   else
   { us->use_exposure_time_def_r = us->inquiry_exposure_time_g_uta_def; }
   if (us->inquiry_grey == 0)
   {
    DBG(1, "ERROR: GREYSCALE-MODE NOT SUPPORTED BY SCANNER, ABORTING\n");
    return(1);
   }
   break;

  case RGB: /* ----------------- COLOR ---------- */
   us->use_exposure_time_min = us->inquiry_exposure_time_c_min;
   if (us->module == WD_module_flatbed) 
   {
    us->use_exposure_time_def_r= us->inquiry_exposure_time_c_fb_def_r;
    us->use_exposure_time_def_g= us->inquiry_exposure_time_c_fb_def_g;
    us->use_exposure_time_def_b= us->inquiry_exposure_time_c_fb_def_b;
   }
   else
   {
    us->use_exposure_time_def_r= us->inquiry_exposure_time_c_uta_def_r;
    us->use_exposure_time_def_g= us->inquiry_exposure_time_c_uta_def_g;
    us->use_exposure_time_def_b= us->inquiry_exposure_time_c_uta_def_b;
   }

   if (us->inquiry_color == 0)
   {
    DBG(1,"ERROR: COLOR-MODE NOT SUPPORTED BY SCANNER, ABORTING\n");
    return(1);
   }
   if (us->inquiry_one_pass_color)
   { DBG(3,"using one pass scanning mode\n"); }
   else
   {
     DBG(3,"using three pass scanning mode\n");
     us->three_pass=1;
   }
   break;
  } /* switch */



  if ( us->inquiry_exposure_adj )
  {
  umax_calculate_exposure_time(us,
                          us->use_exposure_time_def_r,
                          &us->exposure_time_calibration_r);
  umax_calculate_exposure_time(us,
                          us->use_exposure_time_def_g,
                          &us->exposure_time_calibration_g);
  umax_calculate_exposure_time(us,
                          us->use_exposure_time_def_b,
                          &us->exposure_time_calibration_b);

  umax_calculate_exposure_time(us,
                          us->use_exposure_time_def_r,
                          &us->exposure_time_scan_r);
  umax_calculate_exposure_time(us,
                          us->use_exposure_time_def_g,
                          &us->exposure_time_scan_g);
  umax_calculate_exposure_time(us,
                          us->use_exposure_time_def_b,
                          &us->exposure_time_scan_b);
  }
  else
  {
    us->exposure_time_calibration_r = us->exposure_time_calibration_g =
    us->exposure_time_calibration_b = us->exposure_time_scan_r =
    us->exposure_time_scan_g = us->exposure_time_scan_b = 0;
  }

return(0);
}


/* ----------------------- UMAX GET INQUIRY VALUES ------------------------- */


#ifdef UMAX_TO_SANE
static
#endif
void umax_get_inquiry_values(US *us)
{
 char * inquiry_block;

  DBG(5,"get_inquiry_values\n");

  inquiry_block   = us->buffer;
  us->inquiry_len = get_inquiry_additional_length(us->buffer)+5;
  us->cbhs_range  = us->inquiry_cbhs = get_inquiry_CBHS(inquiry_block);
  if (us->cbhs_range > IN_CBHS_255) { us->cbhs_range = IN_CBHS_255; }
  if (us->cbhs_range == IN_CBHS_50)
  {
    us->inquiry_cbhs_min = 78;
    us->inquiry_cbhs_max = 178;
  }

  get_inquiry_vendor( inquiry_block, us->vendor);  us->vendor[8]  ='\0';
  get_inquiry_product(inquiry_block, us->product); us->product[16]='\0';
  get_inquiry_version(inquiry_block, us->version); us->version[4] ='\0';

  us->inquiry_quality_ctrl = get_inquiry_fw_quality(inquiry_block);
  us->inquiry_preview      = get_inquiry_fw_fast_preview(inquiry_block);
  us->inquiry_lamp_ctrl    = get_inquiry_fw_lamp_int_cont(inquiry_block);
  us->inquiry_calibration  = get_inquiry_fw_calibration(inquiry_block);
  us->inquiry_transavail   = get_inquiry_transavail(inquiry_block);
  us->inquiry_adfmode      = get_inquiry_scanmode(inquiry_block);

  if (us->inquiry_len<=0x8f)
  { DBG(2, "WARNING: inquiry return block is unexpected short.\n"); }

  us->inquiry_uta          = get_inquiry_sc_uta(inquiry_block);
  us->inquiry_adf          = get_inquiry_sc_adf(inquiry_block);

  us->inquiry_one_pass_color  = get_inquiry_sc_one_pass_color(inquiry_block);
  us->inquiry_three_pass_color= get_inquiry_sc_three_pass_color(inquiry_block);
  us->inquiry_color        = get_inquiry_sc_color(inquiry_block);
  us->inquiry_grey         = get_inquiry_sc_grey(inquiry_block);
  us->inquiry_halftone     = get_inquiry_sc_halftone(inquiry_block);
  us->inquiry_lineart      = get_inquiry_sc_lineart(inquiry_block);

  us->inquiry_exposure_adj = get_inquiry_fw_adjust_exposure_tf(inquiry_block);
  us->inquiry_exposure_time_step_unit =
     get_inquiry_exposure_time_step_unit(inquiry_block);
  us->inquiry_exposure_time_max=get_inquiry_exposure_time_max(inquiry_block);

             /* --- lineart --- */
  us->inquiry_exposure_time_l_min =
     get_inquiry_exposure_time_lhg_min(inquiry_block);
  us->inquiry_exposure_time_l_fb_def =
     get_inquiry_exposure_time_lh_def_fb(inquiry_block);
  us->inquiry_exposure_time_l_uta_def =
       get_inquiry_exposure_time_lh_def_uta(inquiry_block);

             /* --- halftone --- */
  us->inquiry_exposure_time_h_min =
     get_inquiry_exposure_time_lhg_min(inquiry_block);
  us->inquiry_exposure_time_h_fb_def =
     get_inquiry_exposure_time_lh_def_fb(inquiry_block);
  us->inquiry_exposure_time_h_uta_def =
     get_inquiry_exposure_time_lh_def_uta(inquiry_block);

             /* --- greyscale --- */
  us->inquiry_exposure_time_g_min =
     get_inquiry_exposure_time_lhg_min(inquiry_block);
  us->inquiry_exposure_time_g_fb_def =
     get_inquiry_exposure_time_grey_def_fb(inquiry_block);
  us->inquiry_exposure_time_g_uta_def =
      get_inquiry_exposure_time_grey_def_uta(inquiry_block);

             /* --- color --- */
  us->inquiry_exposure_time_c_min =
     get_inquiry_exposure_time_color_min(inquiry_block);
  us->inquiry_exposure_time_c_fb_def_r=
    get_inquiry_exposure_time_def_r_fb(inquiry_block);
  us->inquiry_exposure_time_c_fb_def_g=
    get_inquiry_exposure_time_def_g_fb(inquiry_block);
  us->inquiry_exposure_time_c_fb_def_b=
    get_inquiry_exposure_time_def_g_fb(inquiry_block);
  us->inquiry_exposure_time_c_uta_def_r=
    get_inquiry_exposure_time_def_r_uta(inquiry_block);
  us->inquiry_exposure_time_c_uta_def_g=
    get_inquiry_exposure_time_def_g_uta(inquiry_block);
  us->inquiry_exposure_time_c_uta_def_b=
    get_inquiry_exposure_time_def_b_uta(inquiry_block);


  us->inquiry_dor          = get_inquiry_sc_double_res(inquiry_block);
  us->inquiry_reverse      = get_inquiry_sc_bi_image_reverse(inquiry_block);
  us->inquiry_reverse_multi= get_inquiry_sc_multi_image_reverse(inquiry_block);
  us->inquiry_shadow       = 1 - get_inquiry_sc_no_shadow(inquiry_block);
  us->inquiry_highlight    = 1 - get_inquiry_sc_no_highlight(inquiry_block);
  us->inquiry_analog_gamma = get_inquiry_analog_gamma(inquiry_block);
  us->inquiry_gamma_dwload =get_inquiry_gamma_download_available(inquiry_block);

  /* optical resolution = [0x73] * 100 + [0x94] , 0x94 is not always defined */
   us->inquiry_optical_res = 100*get_inquiry_max_opt_res(inquiry_block);
   if (us->inquiry_len >= 0x94)
   {
     us->inquiry_optical_res+=
        get_inquiry_optical_resolution_residue(inquiry_block);
   }

   /* x resolution = [0x74] * 100 + [0x95] , 0x95 is not always defined */
   us->inquiry_x_res = 100*get_inquiry_max_x_res(inquiry_block);
   if (us->inquiry_len >= 0x95)
   {us->inquiry_x_res+= get_inquiry_x_resolution_residue(inquiry_block);};

   /* y resolution = [0x75] * 100 + [0x96] , 0x96 is not always defined */
   us->inquiry_y_res = 100*get_inquiry_max_y_res(inquiry_block);
   if (us->inquiry_len >= 0x96)
   {us->inquiry_y_res+= get_inquiry_y_resolution_residue(inquiry_block);}

  us->inquiry_fb_width   =
      (double)get_inquiry_fb_max_scan_width(inquiry_block)*0.01;
  us->inquiry_fb_length  =
      (double)get_inquiry_fb_max_scan_length(inquiry_block)*0.01;

  us->inquiry_uta_width  =
     (double)get_inquiry_uta_max_scan_width(inquiry_block)*0.01;
  us->inquiry_uta_length =
     (double)get_inquiry_uta_max_scan_length(inquiry_block)*0.01;

  us->inquiry_dor_width  =
     (double)get_inquiry_dor_max_scan_width(inquiry_block)*0.01;
  us->inquiry_dor_length =
     (double)get_inquiry_dor_max_scan_length(inquiry_block)*0.01;

  us->inquiry_max_warmup_time=
          get_inquiry_lamp_warmup_maximum_time(inquiry_block)*2;

  us->inquiry_wdb_len = get_inquiry_wdb_length(inquiry_block);

  us->inquiry_GIB = get_inquiry_gib(inquiry_block);
  us->inquiry_GOB = get_inquiry_gob(inquiry_block);

  return;
}


/* -------------------- UMAX CALCULATE ANALOG GAMMA ------------------------ */


#ifdef UMAX_TO_SANE
static
#endif
int umax_calculate_analog_gamma(double value)
{
 int gamma;

  if (value < 1.0)
   { value=1.0; }

  if (value > 2.0)
   { value=2.0; }

  gamma=0;                     /* select gamma_value from analog_gamma_table */
  while (value>analog_gamma_table[gamma]) { gamma++; } 
  if (gamma)
  {
    if ((analog_gamma_table[gamma-1] + analog_gamma_table[gamma]) /2 > value)
    { gamma--;}
  }
  
return(gamma);
}


/* ------------------------ UMAX GET ANALOG GAMMA -------------------------- */


#ifndef UMAX_TO_SANE
double umax_get_analog_gamma(int value)
{
  return(analog_gamma_table[value]);
}
#endif


/* ----------------------- UMAX INITIALIZE VALUES -------------------------- */


#ifdef UMAX_TO_SANE
static
#endif
void umax_initialize_values(US *us)
{
  DBG(5,"initialize_values\n");
  /* Initialize us structure */

  us->cont                = 0;      /* do not continue if scanner is unknown */
  us->verbose             = 0;                   /* 1=verbose,2=very verbose */
  us->three_pass          = 0;                    /* 1 if threepas_mode only */
  us->row_len             = -1;
  us->max_value           = 255;                           /* for pnm-header */

  us->wdb_len             = 0;
  us->width_in_pixels     = 0;                       /* scan width in pixels */
  us->height_in_pixels    = 0;                      /* scan height in pixels */
  us->width               = 0;
  us->length              = 0;
  us->phys_width          = 0;               /* width in inch at 1200pt/inch */
  us->phys_height         = 0;              /* height in inch at 1200pt/inch */
  us->x_resolution        = 0;
  us->y_resolution        = 0;
  us->upper_left_x        = 0;                             /* at 1200pt/inch */
  us->upper_left_y        = 0;                             /* at 1200pt/inch */
  us->x_coordinate_base   = 1200;               /* these are the 1200pt/inch */
  us->y_coordinate_base   = 1200;               /* these are the 1200pt/inch */

  us->bits_per_pixel      = 1;                /* 1 = 8/24 bpp, 4 = 10/30 bpp */
  us->gamma_input_bits    = 1;                /* 1 = 8/24 bpp, 4 = 10/30 bpp */
  us->set_auto            = 0;                                     /* 0 or 1 */
  us->preview             = 0;                              /* 1 for preview */
  us->quality             = 0;                        /* quality calibration */
  us->warmup              = 0;                                 /* warmup-bit */
  us->colormode           = 0;         /* LINEART,HALFTONE, GREYSCALE or RGB */
  us->adf                 = 0;                     /* 1 if adf shall be used */
  us->uta                 = 0;                     /* 1 if uta shall be used */
  us->module              = WD_module_flatbed;
  us->cbhs_range          = WD_CBHS_255;
  us->dor                 = 0;
  us->halftone            = WD_halftone_8x8_1;
  us->reverse             = 0;
  us->reverse_multi       = 0;
  us->slow                = 0;
  us->smear               = 0;
  us->calibration         = 0;

  us->exposure_time_calibration_r  = 0;          /* use this for calibration */
  us->exposure_time_calibration_g  = 0;          /* use this for calibration */
  us->exposure_time_calibration_b  = 0;          /* use this for calibration */
  us->exposure_time_scan_r         = 0;                 /* use this for scan */
  us->exposure_time_scan_g         = 0;                 /* use this for scan */
  us->exposure_time_scan_b         = 0;                 /* use this for scan */

  us->c_density           = WD_lamp_c_density_auto;  /* calibr. lamp density */
  us->s_density           = WD_lamp_s_density_auto;/* next scan lamp density */

  us->threshold           = 128;               /* threshold for lineart mode */
  us->brightness          = 128;             /* brightness for halftone mode */
  us->contrast            = 128;               /* contrast for halftone mode */
  us->highlight_r         = 255;                  /* highlight for each mode */
  us->highlight_g         = 255;                  /* highlight for each mode */
  us->highlight_b         = 255;                  /* highlight for each mode */
  us->shadow_r            = 0;                       /* shadow for each mode */
  us->shadow_g            = 0;                       /* shadow for each mode */
  us->shadow_b            = 0;                       /* shadow for each mode */

  us->digital_gamma_r     = WD_gamma_normal;
  us->digital_gamma_g     = WD_gamma_normal;
  us->digital_gamma_b     = WD_gamma_normal;
  us->gamma_r             = 1.0;
  us->gamma_g             = 1.0;
  us->gamma_b             = 1.0;
  us->analog_gamma_r      = 0;       /* analog gamma for red and grey to 1.0 */
  us->analog_gamma_g      = 0;              /* analog gamma for green to 1.0 */
  us->analog_gamma_b      = 0;               /* analog gamma for blue to 1.0 */

  us->RGB_PREVIEW_PATCH   = 0;                         /* patch for umax s6e */
  us->START_SCAN_PATCH    = 0;

}


/* ------------------------------- UMAX INIT ------------------------------- */


#ifdef UMAX_TO_SANE
static
#endif
void umax_init(US *us)
{
  DBG(5,"init\n");

  us->devicename          = NULL;

  us->ofd                 = STDOUT_FILENO;   /* output fd, stdout by default */
  us->sfd                 = -1;

  us->bufsize             = UMAX_SCSI_MAX_REQUEST_SIZE;
  us->buffer              = malloc(us->bufsize);          /* allocate buffer */

  us->inquiry_len         = 0;
  us->inquiry_wdb_len     = -1;
  us->inquiry_optical_res = -1;
  us->inquiry_x_res       = -1;
  us->inquiry_y_res       = -1;
  us->inquiry_fb_width    = -1;
  us->inquiry_fb_length   = -1;
  us->inquiry_uta_width   = -1;
  us->inquiry_uta_length  = -1;
  us->inquiry_dor_width   = -1;
  us->inquiry_dor_length  = -1;
  us->inquiry_exposure_adj           = 0;
  us->inquiry_exposure_time_step_unit= -1;/* exposure time unit in micro sec */
  us->inquiry_exposure_time_max      = -1;          /* exposure time maximum */
  us->inquiry_exposure_time_l_min    = -1;         /*  exposure time minimum */
  us->inquiry_exposure_time_l_fb_def = -1;          /* exposure time default */
  us->inquiry_exposure_time_l_uta_def= -1;          /* exposure time default */
  us->inquiry_exposure_time_h_min    = -1;         /*  exposure time minimum */
  us->inquiry_exposure_time_h_fb_def = -1;          /* exposure time default */
  us->inquiry_exposure_time_h_uta_def= -1;          /* exposure time default */
  us->inquiry_exposure_time_g_min    = -1;         /*  exposure time minimum */
  us->inquiry_exposure_time_g_fb_def = -1;          /* exposure time default */
  us->inquiry_exposure_time_g_uta_def= -1;          /* exposure time default */
  us->inquiry_exposure_time_c_min    = -1;         /*  exposure time minimum */
  us->inquiry_exposure_time_c_fb_def_r = -1;        /* exposure time default */
  us->inquiry_exposure_time_c_fb_def_g = -1;        /* exposure time default */
  us->inquiry_exposure_time_c_fb_def_b = -1;        /* exposure time default */
  us->inquiry_exposure_time_c_uta_def_r= -1;        /* exposure time default */
  us->inquiry_exposure_time_c_uta_def_g= -1;        /* exposure time default */
  us->inquiry_exposure_time_c_uta_def_b= -1;        /* exposure time default */
  us->inquiry_max_warmup_time        = 0;             /* maximum warmup time */
  us->inquiry_cbhs                   = WD_CBHS_255;
  us->inquiry_cbhs_min               = 0;
  us->inquiry_cbhs_max               = 255;
  us->inquiry_quality_ctrl           = 0;
  us->inquiry_preview                = 0;
  us->inquiry_lamp_ctrl              = 0;
  us->inquiry_transavail             = 0;
  us->inquiry_uta                    = 0;
  us->inquiry_adfmode                = 0;
  us->inquiry_adf                    = 0;
  us->inquiry_dor                    = 0;
  us->inquiry_reverse                = 0;
  us->inquiry_reverse_multi          = 0;
  us->inquiry_analog_gamma           = 0;
  us->inquiry_gamma_dwload           = 0;
  us->inquiry_one_pass_color         = 0;
  us->inquiry_three_pass_color       = 0;
  us->inquiry_color                  = 0;
  us->inquiry_grey                   = 0;
  us->inquiry_halftone               = 0;
  us->inquiry_lineart                = 0;
  us->inquiry_calibration            = 1;
  us->inquiry_shadow                 = 0;
  us->inquiry_highlight              = 0;
}
