/***************************************************************************
 * SANE - Scanner Access Now Easy.

   microtek.c 

   This file (C) 1997 Matthew Marjanovic

   This file is part of the SANE package.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.

   As a special exception, the authors of SANE give permission for
   additional uses of the libraries contained in this release of SANE.

   The exception is that, if you link a SANE library with other files
   to produce an executable, this does not by itself cause the
   resulting executable to be covered by the GNU General Public
   License.  Your use of that executable is in no way restricted on
   account of linking the SANE library code into it.

   This exception does not, however, invalidate any other reasons why
   the executable file might be covered by the GNU General Public
   License.

   If you submit changes to SANE to the maintainers to be included in
   a subsequent release, you agree by submitting the changes that
   those changes may be distributed with this exception intact.

   If you write modifications of your own for SANE, it is your choice
   whether to permit this exception to apply to your modifications.
   If you do not wish that, delete this exception notice.

 ***************************************************************************

   This file implements a SANE backend for Microtek scanners.

   (feedback to:  mtek-bugs@mir.com)

 ***************************************************************************/



#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>

#include <sane/sane.h>
#include <sane/sanei.h>
#include <sane/sanei_scsi.h>
#include <sane/saneopts.h>

#define BACKEND_NAME microtek
#include <sane/sanei_backend.h>

#include "microtek.h"

#include <sane/sanei_config.h>
#define PATH_MICROTEK_CONFIG "microtek.conf"

#ifndef PATH_MAX
# define PATH_MAX	1024
#endif
#define MM_PER_INCH	25.4

#define SCSI_BUFF_SIZE sanei_scsi_max_request_size /* XXXXXXXXXXx */

#define M_LINEART  "Lineart"
#define M_HALFTONE "Halftone"
#define M_GRAY     "Gray"
#define M_COLOR    "Color"


static int num_devices = 0;
static Microtek_Device *first_dev = NULL;     /* list of known devices */
static Microtek_Scanner *first_handle = NULL; /* list of open scanners */



/* These are for the E6.  Does this hold for other models? */
static SANE_String_Const halftone_mode_list[13] = {
  " 1 53-dot screen (53 gray levels)",
  " 2 Horiz. screen (65 gray levels)",
  " 3 Vert. screen (65 gray levels)",
  " 4 Mixed page (33 gray levels)",
  " 5 71-dot screen (29 gray levels)",
  " 6 60-dot #1 (26 gray levels)",
  " 7 60-dot #2 (26 gray levels)",
  " 8 Fine detail #1 (17 gray levels)",
  " 9 Fine detail #2 (17 gray levels)",
  "10 Slant line (17 gray levels)",
  "11 Posterizing (10 gray levels)",
  "12 High Contrast (5 gray levels)",
  NULL
};


static SANE_Range geo_range50 =
{ SANE_FIX(0), SANE_FIX(5.0 * MM_PER_INCH), SANE_FIX(0) };
static SANE_Range geo_range80 =
{ SANE_FIX(0), SANE_FIX(8.0 * MM_PER_INCH), SANE_FIX(0) };
static SANE_Range geo_range83 =
{ SANE_FIX(0), SANE_FIX(8.3 * MM_PER_INCH), SANE_FIX(0) };
static SANE_Range geo_range85 =
{ SANE_FIX(0), SANE_FIX(/*XXXXX8.5*/ 8.45 * MM_PER_INCH), SANE_FIX(0) };
static SANE_Range geo_range100 =
{ SANE_FIX(0), SANE_FIX(10.0 * MM_PER_INCH), SANE_FIX(0) };
static SANE_Range geo_range110 =
{ SANE_FIX(0), SANE_FIX(11.0 * MM_PER_INCH), SANE_FIX(0) };
static SANE_Range geo_range1169 =
{ SANE_FIX(0), SANE_FIX(11.69 * MM_PER_INCH), SANE_FIX(0) };
static SANE_Range geo_range130 =
{ SANE_FIX(0), SANE_FIX(13.0 * MM_PER_INCH), SANE_FIX(0) };
static SANE_Range geo_range140 =
{ SANE_FIX(0), SANE_FIX(14.0 * MM_PER_INCH), SANE_FIX(0) };
static SANE_Range geo_range35M =
{ SANE_FIX(0), SANE_FIX(35.0), SANE_FIX(0) };
static SANE_Range geo_range36M =
{ SANE_FIX(0), SANE_FIX(36.0), SANE_FIX(0) };

static SANE_Range speed_range = {1, 7, 1};

/*static SANE_Range brightness_range = {-100, 100, 1};*/
static SANE_Range exposure_range = {-18, 21, 3};
static SANE_Range contrast_range = {-42, 49, 7};
static SANE_Range u8_range = {0, 255, 1};




static size_t max_string_size (const SANE_String_Const strings[])
{
  size_t size, max_size = 0;
  int i;

  for (i = 0; strings[i]; ++i) {
    size = strlen(strings[i]) + 1;
    if (size > max_size) max_size = size;
  }
  return max_size;
}



static SANE_Status
sense_handler (int scsi_fd, u_char *sense) 
{
  switch(sense[0]) {
  case 0x00:
    return SANE_STATUS_GOOD;
  case 0x81:           /* COMMAND/DATA ERROR */
    /*
    if (sense[1] & 0x01) error(ERR_SCSICMD);
    if (sense[1] & 0x02) error(ERR_TOOMANY);
    return(CD_ERROR);
    */
  case 0x82 :           /* SCANNER HARDWARE ERROR */
    /*
    if (sense[1] & 0x01) error(ERR_CPURAMFAIL);
    if (sense[1] & 0x02) error(ERR_SYSRAMFAIL);
    if (sense[1] & 0x04) error(ERR_IMGRAMFAIL);
    if (sense[1] & 0x10) error(ERR_CALIBRATE);
    if (sense[1] & 0x20) error(ERR_LAMPFAIL);
    if (sense[1] & 0x40) error(ERR_MOTORFAIL);
    if (sense[1] & 0x80) error(ERR_FEEDERFAIL);
    if (sense[2] & 0x01) error(ERR_POWERFAIL);
    if (sense[2] & 0x02) error(ERR_ILAMPFAIL);
    if (sense[2] & 0x04) error(ERR_IMOTORFAIL);
    if (sense[2] & 0x08) error(ERR_PAPERFAIL);
    if (sense[2] & 0x10) error(ERR_FILTERFAIL);
    return(HW_ERROR);                      
    */
  case 0x83 :           /* OPERATION ERROR */
    /*
    if (sense[1] & 0x01) error(ERR_ILLGRAIN);
    if (sense[1] & 0x02) error(ERR_ILLRES);
    if (sense[1] & 0x04) error(ERR_ILLCOORD);
    if (sense[1] & 0x10) error(ERR_ILLCNTR);
    if (sense[1] & 0x20) error(ERR_ILLLENGTH);
    if (sense[1] & 0x40) error(ERR_ILLADJUST);
    if (sense[1] & 0x80) error(ERR_ILLEXPOSE);
    if (sense[2] & 0x01) error(ERR_ILLFILTER);
    if (sense[2] & 0x02) error(ERR_NOPAPER);
    if (sense[2] & 0x04) error(ERR_ILLTABLE);
    if (sense[2] & 0x08) error(ERR_ILLOFFSET);
    if (sense[2] & 0x10) error(ERR_ILLBPP);
    return(OP_ERROR);
    */
  default :
    /*
    DBG(23, "%s: sense = %02x %02x %02x %02x.\n", myname, 
            sense[0], sense[1], sense[2], sense[3]);*/
    DBG(23, "bogus sense error\n");
    return SANE_STATUS_IO_ERROR;
  }
  return SANE_STATUS_GOOD;
}






static SANE_Status init_options(Microtek_Scanner *ms)
{
  int i;
  SANE_Option_Descriptor *sod = ms->sod;
  Microtek_Option_Value *val = ms->val;

  DBG(10, "init_options\n");

  memset(ms->sod, 0, sizeof(ms->sod));
  memset(ms->val, 0, sizeof(ms->val));
  /* default:  software selectable word options... */
  for (i=0; i<NUM_OPTIONS; i++) {
    sod[i].size = sizeof(SANE_Word);
    sod[i].cap = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
  }

  sod[OPT_NUM_OPTS].name =   SANE_NAME_NUM_OPTIONS;
  sod[OPT_NUM_OPTS].title =  SANE_TITLE_NUM_OPTIONS;
  sod[OPT_NUM_OPTS].desc =   SANE_DESC_NUM_OPTIONS;
  sod[OPT_NUM_OPTS].type =   SANE_TYPE_INT;
  sod[OPT_NUM_OPTS].unit =   SANE_UNIT_NONE;
  sod[OPT_NUM_OPTS].size =   sizeof (SANE_Word);
  sod[OPT_NUM_OPTS].cap =    SANE_CAP_SOFT_DETECT;
  sod[OPT_NUM_OPTS].constraint_type = SANE_CONSTRAINT_NONE;

  DBG(23, "init_options:  initopt others...");

  /* The Scan Mode Group */
  sod[OPT_MODE_GROUP].name  = "";
  sod[OPT_MODE_GROUP].title = "Scan Mode";
  sod[OPT_MODE_GROUP].desc  = "";
  sod[OPT_MODE_GROUP].type  = SANE_TYPE_GROUP;
  sod[OPT_MODE_GROUP].cap   = 0;
  sod[OPT_MODE_GROUP].constraint_type = SANE_CONSTRAINT_NONE;

  sod[OPT_MODE].name = SANE_NAME_SCAN_MODE;
  sod[OPT_MODE].title = SANE_TITLE_SCAN_MODE;
  sod[OPT_MODE].desc = SANE_DESC_SCAN_MODE;
  sod[OPT_MODE].type = SANE_TYPE_STRING;
  sod[OPT_MODE].constraint_type = SANE_CONSTRAINT_STRING_LIST;
  {
    SANE_String_Const *mode_list;
    mode_list = (SANE_String_Const *) malloc(5 * sizeof(SANE_String_Const));
    if (mode_list == NULL) return SANE_STATUS_NO_MEM;
    i = 0;
    if (ms->dev->info.modes & MI_MODES_COLOR)    mode_list[i++] = M_COLOR;
    if (ms->dev->info.modes & MI_MODES_GRAY)     mode_list[i++] = M_GRAY;
    if (ms->dev->info.modes & MI_MODES_HALFTONE) mode_list[i++] = M_HALFTONE;
    if (ms->dev->info.modes & MI_MODES_LINEART)  mode_list[i++] = M_LINEART;
    mode_list[i] = NULL;
    sod[OPT_MODE].constraint.string_list = mode_list;
    sod[OPT_MODE].size                   = max_string_size(mode_list);
    val[OPT_MODE].s = strdup(mode_list[0]);
  }

  sod[OPT_RESOLUTION].name  = SANE_NAME_SCAN_RESOLUTION;
  sod[OPT_RESOLUTION].title = SANE_TITLE_SCAN_RESOLUTION;
  sod[OPT_RESOLUTION].desc  = SANE_DESC_SCAN_RESOLUTION;
  sod[OPT_RESOLUTION].type  = SANE_TYPE_FIXED;
  sod[OPT_RESOLUTION].unit  = SANE_UNIT_DPI;
  sod[OPT_RESOLUTION].constraint_type  = SANE_CONSTRAINT_RANGE;
  {
    SANE_Range *range;
    range = (SANE_Range *) malloc(sizeof(SANE_Range));
    if (range == NULL) return SANE_STATUS_NO_MEM;
    range->max = SANE_FIX(ms->dev->info.base_resolution);
    /*range->max = SANE_FIX(600);   not true! fix XXXXXXXXXXXXX */
    if (ms->dev->info.res_step & MI_RESSTEP_1PER) {
      DBG(23, "init_options:  quant yes\n");
      range->min = SANE_FIX( 600 / 100 );
      range->quant = SANE_FIX( 600 / 100 );
    } else {
      /* XXXXXXXXXXX */
      DBG(23, "init_options:  quant no\n");
      range->quant = SANE_FIX( 0 );
    }
    sod[OPT_RESOLUTION].constraint.range = range;
  }
  val[OPT_RESOLUTION].w     = SANE_FIX(100);

  sod[OPT_HALFTONE_PATTERN].name = SANE_NAME_HALFTONE;
  sod[OPT_HALFTONE_PATTERN].title = SANE_TITLE_HALFTONE;
  sod[OPT_HALFTONE_PATTERN].desc = SANE_DESC_HALFTONE_PATTERN;
  sod[OPT_HALFTONE_PATTERN].type = SANE_TYPE_STRING;
  sod[OPT_HALFTONE_PATTERN].size = max_string_size(halftone_mode_list);
  sod[OPT_HALFTONE_PATTERN].cap  |= SANE_CAP_INACTIVE;
  sod[OPT_HALFTONE_PATTERN].constraint_type = SANE_CONSTRAINT_STRING_LIST;
  sod[OPT_HALFTONE_PATTERN].constraint.string_list = halftone_mode_list;
  val[OPT_HALFTONE_PATTERN].s = strdup(halftone_mode_list[0]);

  sod[OPT_NEGATIVE].name  = SANE_NAME_NEGATIVE;
  sod[OPT_NEGATIVE].title = SANE_TITLE_NEGATIVE;
  sod[OPT_NEGATIVE].desc  = SANE_DESC_NEGATIVE;
  sod[OPT_NEGATIVE].type  = SANE_TYPE_BOOL;
  sod[OPT_NEGATIVE].cap   |= 
    (ms->dev->info.modes & MI_MODES_NEGATIVE) ? 0 : SANE_CAP_INACTIVE;
  val[OPT_NEGATIVE].w     = SANE_FALSE;

  sod[OPT_SPEED].name  = SANE_NAME_SCAN_SPEED;
  sod[OPT_SPEED].title = SANE_TITLE_SCAN_SPEED;
  sod[OPT_SPEED].desc  = SANE_DESC_SCAN_SPEED;
  sod[OPT_SPEED].type  = SANE_TYPE_INT;
  sod[OPT_SPEED].unit  = SANE_UNIT_NONE;
  sod[OPT_SPEED].size  = sizeof(SANE_Word);
  /*  sod[OPT_SPEED].cap   = 0;*/
  sod[OPT_SPEED].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_SPEED].constraint      = &speed_range;
  val[OPT_SPEED].w     = 1;

  sod[OPT_PREVIEW].name  = SANE_NAME_PREVIEW;
  sod[OPT_PREVIEW].title = SANE_TITLE_PREVIEW;
  sod[OPT_PREVIEW].desc  = SANE_DESC_PREVIEW;
  sod[OPT_PREVIEW].type  = SANE_TYPE_BOOL; 
  sod[OPT_PREVIEW].unit  = SANE_UNIT_NONE;
  sod[OPT_PREVIEW].size  = sizeof(SANE_Word);
  /*  sod[OPT_PREVIEW].cap   = SANE_CAP_SOFT_DETECT | SANE_CAP_SOFT_SELECT;*/
  /*  sod[OPT_PREVIEW].constraint_type = 0;
  sod[OPT_PREVIEW].constraint      = 0;*/
  val[OPT_PREVIEW].w     = SANE_FALSE;


  sod[OPT_GEOMETRY_GROUP].name  = "";
  sod[OPT_GEOMETRY_GROUP].title = "Geometry";
  sod[OPT_GEOMETRY_GROUP].desc  = "";
  sod[OPT_GEOMETRY_GROUP].type  = SANE_TYPE_GROUP;
  sod[OPT_GEOMETRY_GROUP].cap   = 0;
  sod[OPT_GEOMETRY_GROUP].constraint_type = SANE_CONSTRAINT_NONE;


  sod[OPT_TL_X].name  = SANE_NAME_SCAN_TL_X;
  sod[OPT_TL_X].title = SANE_TITLE_SCAN_TL_X;
  sod[OPT_TL_X].desc  = SANE_DESC_SCAN_TL_X;
  sod[OPT_TL_X].type  = SANE_TYPE_FIXED;
  sod[OPT_TL_X].unit  = SANE_UNIT_MM;
  sod[OPT_TL_X].size  = sizeof(SANE_Word);
  /*  sod[OPT_TL_X].cap   = 0;*/
  sod[OPT_TL_X].constraint_type = SANE_CONSTRAINT_RANGE;
  val[OPT_TL_X].w     = 0;

  sod[OPT_TL_Y].name  = SANE_NAME_SCAN_TL_Y;
  sod[OPT_TL_Y].title = SANE_TITLE_SCAN_TL_Y;
  sod[OPT_TL_Y].desc  = SANE_DESC_SCAN_TL_Y;
  sod[OPT_TL_Y].type  = SANE_TYPE_FIXED;
  sod[OPT_TL_Y].unit  = SANE_UNIT_MM;
  sod[OPT_TL_Y].size  = sizeof(SANE_Word);
  /*  sod[OPT_TL_Y].cap   = 0;*/
  sod[OPT_TL_Y].constraint_type = SANE_CONSTRAINT_RANGE;
  val[OPT_TL_Y].w     = 0;

  sod[OPT_BR_X].name  = SANE_NAME_SCAN_BR_X;
  sod[OPT_BR_X].title = SANE_TITLE_SCAN_BR_X;
  sod[OPT_BR_X].desc  = SANE_DESC_SCAN_BR_X;
  sod[OPT_BR_X].type  = SANE_TYPE_FIXED;
  sod[OPT_BR_X].unit  = SANE_UNIT_MM;
  sod[OPT_BR_X].size  = sizeof(SANE_Word);
  /*  sod[OPT_BR_X].cap   = 0;*/
  sod[OPT_BR_X].constraint_type = SANE_CONSTRAINT_RANGE;
  val[OPT_BR_X].w     = 0;

  sod[OPT_BR_Y].name  = SANE_NAME_SCAN_BR_Y;
  sod[OPT_BR_Y].title = SANE_TITLE_SCAN_BR_Y;
  sod[OPT_BR_Y].desc  = SANE_DESC_SCAN_BR_Y;
  sod[OPT_BR_Y].type  = SANE_TYPE_FIXED;
  sod[OPT_BR_Y].unit  = SANE_UNIT_MM;
  sod[OPT_BR_Y].size  = sizeof(SANE_Word);
  /*  sod[OPT_BR_Y].cap   = 0;*/
  sod[OPT_BR_Y].constraint_type = SANE_CONSTRAINT_RANGE;
  val[OPT_BR_Y].w     = 0;

  switch (ms->dev->info.doc_size_code) {
  case 0x00:
    sod[OPT_TL_X].constraint = sod[OPT_BR_X].constraint = &geo_range85;
    sod[OPT_TL_Y].constraint = sod[OPT_BR_Y].constraint = &geo_range140;
    break;
  case 0x01:
    sod[OPT_TL_X].constraint = sod[OPT_BR_X].constraint = &geo_range85;
    sod[OPT_TL_Y].constraint = sod[OPT_BR_Y].constraint = &geo_range110;
    break;
  case 0x02:
    sod[OPT_TL_X].constraint = sod[OPT_BR_X].constraint = &geo_range85;
    sod[OPT_TL_Y].constraint = sod[OPT_BR_Y].constraint = &geo_range1169;
    break;
  case 0x03:
    sod[OPT_TL_X].constraint = sod[OPT_BR_X].constraint = &geo_range85;
    sod[OPT_TL_Y].constraint = sod[OPT_BR_Y].constraint = &geo_range130;
    break;
  case 0x04:
    sod[OPT_TL_X].constraint = sod[OPT_BR_X].constraint = &geo_range80;
    sod[OPT_TL_Y].constraint = sod[OPT_BR_Y].constraint = &geo_range100;
    break;
  case 0x05:
    sod[OPT_TL_X].constraint = sod[OPT_BR_X].constraint = &geo_range83;
    sod[OPT_TL_Y].constraint = sod[OPT_BR_Y].constraint = &geo_range140;
    break;
  case 0x80:
    sod[OPT_TL_X].constraint = sod[OPT_BR_X].constraint = &geo_range35M;
    sod[OPT_TL_Y].constraint = sod[OPT_BR_Y].constraint = &geo_range35M;
    break;
  case 0x81:
    sod[OPT_TL_X].constraint = sod[OPT_BR_X].constraint = &geo_range50;
    sod[OPT_TL_Y].constraint = sod[OPT_BR_Y].constraint = &geo_range50;
    break;
  case 0x82:
    sod[OPT_TL_X].constraint = sod[OPT_BR_X].constraint = &geo_range36M;
    sod[OPT_TL_Y].constraint = sod[OPT_BR_Y].constraint = &geo_range36M;
    break;
  default:
    DBG(23, "init_options:  YYIKES\n");  /* XXXXXXXXXXX */
  }


  sod[OPT_ENHANCEMENT_GROUP].name  = "";
  sod[OPT_ENHANCEMENT_GROUP].title = "Enhancement";
  sod[OPT_ENHANCEMENT_GROUP].desc  = "";
  sod[OPT_ENHANCEMENT_GROUP].type  = SANE_TYPE_GROUP;
  sod[OPT_ENHANCEMENT_GROUP].cap   = 0;
  sod[OPT_ENHANCEMENT_GROUP].constraint_type = SANE_CONSTRAINT_NONE;

  /* XXXXXXXXXX */
  sod[OPT_BRIGHTNESS].name  = SANE_NAME_BRIGHTNESS;
  sod[OPT_BRIGHTNESS].title = SANE_TITLE_BRIGHTNESS;
  sod[OPT_BRIGHTNESS].desc  = SANE_DESC_BRIGHTNESS;
  sod[OPT_BRIGHTNESS].type  = SANE_TYPE_INT;
  sod[OPT_BRIGHTNESS].unit  = SANE_UNIT_PERCENT;
  sod[OPT_BRIGHTNESS].size  = sizeof(SANE_Word);
  /*  sod[OPT_BRIGHTNESS].cap   = 0;*/
  sod[OPT_BRIGHTNESS].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_BRIGHTNESS].constraint      = &exposure_range;
  val[OPT_BRIGHTNESS].w     = 0;

  sod[OPT_CONTRAST].name  = SANE_NAME_CONTRAST;
  sod[OPT_CONTRAST].title = SANE_TITLE_CONTRAST;
  sod[OPT_CONTRAST].desc  = SANE_DESC_CONTRAST;
  sod[OPT_CONTRAST].type  = SANE_TYPE_INT;
  sod[OPT_CONTRAST].unit  = SANE_UNIT_PERCENT;
  sod[OPT_CONTRAST].size  = sizeof(SANE_Word);
  /*  sod[OPT_CONTRAST].cap   = 0;*/
  sod[OPT_CONTRAST].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_CONTRAST].constraint      = &contrast_range;
  val[OPT_CONTRAST].w     = 0;


  sod[OPT_HIGHLIGHT].name  = SANE_NAME_WHITE_LEVEL;
  sod[OPT_HIGHLIGHT].title = SANE_TITLE_WHITE_LEVEL;
  sod[OPT_HIGHLIGHT].desc  = SANE_DESC_WHITE_LEVEL;
  sod[OPT_HIGHLIGHT].type  = SANE_TYPE_INT;
  sod[OPT_HIGHLIGHT].unit  = SANE_UNIT_NONE;
  sod[OPT_HIGHLIGHT].size  = sizeof(SANE_Word);
  sod[OPT_HIGHLIGHT].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_HIGHLIGHT].constraint      = &u8_range;
  val[OPT_HIGHLIGHT].w     = 255;

  sod[OPT_SHADOW].name  = SANE_NAME_BLACK_LEVEL;
  sod[OPT_SHADOW].title = SANE_TITLE_BLACK_LEVEL;
  sod[OPT_SHADOW].desc  = SANE_DESC_BLACK_LEVEL;
  sod[OPT_SHADOW].type  = SANE_TYPE_INT;
  sod[OPT_SHADOW].unit  = SANE_UNIT_NONE;
  sod[OPT_SHADOW].size  = sizeof(SANE_Word);
  sod[OPT_SHADOW].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_SHADOW].constraint      = &u8_range;
  val[OPT_SHADOW].w     = 0;

  if (ms->dev->info.enhance_cap & MI_ENH_CAP_SHADOW) {
    sod[OPT_HIGHLIGHT].cap |= SANE_CAP_ADVANCED;
    sod[OPT_SHADOW].cap    |= SANE_CAP_ADVANCED;
  } else {
    sod[OPT_HIGHLIGHT].cap |= SANE_CAP_INACTIVE;
    sod[OPT_SHADOW].cap    |= SANE_CAP_INACTIVE;
  } 
   
  sod[OPT_MIDTONE].name  = "midtone";
  sod[OPT_MIDTONE].title = "Midtone Level";
  sod[OPT_MIDTONE].desc  = "Midtone Level";
  sod[OPT_MIDTONE].type  = SANE_TYPE_INT;
  sod[OPT_MIDTONE].unit  = SANE_UNIT_NONE;
  sod[OPT_MIDTONE].size  = sizeof(SANE_Word);
  sod[OPT_MIDTONE].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_MIDTONE].constraint      = &u8_range;
  val[OPT_MIDTONE].w     = 128;
  if (ms->dev->info.enhance_cap & MI_ENH_CAP_MIDTONE) {
    sod[OPT_MIDTONE].cap   |= SANE_CAP_ADVANCED;
  } else {
    sod[OPT_MIDTONE].cap   |= SANE_CAP_INACTIVE;
  }

  if ((strcmp(M_COLOR, val[OPT_MODE].s)) &&
      (strcmp(M_GRAY, val[OPT_MODE].s))) {
    sod[OPT_HIGHLIGHT].cap |= SANE_CAP_INACTIVE;
    sod[OPT_SHADOW].cap    |= SANE_CAP_INACTIVE;
    sod[OPT_MIDTONE].cap   |= SANE_CAP_INACTIVE;
  }

  sod[OPT_CUSTOM_GAMMA].name  = SANE_NAME_CUSTOM_GAMMA;
  sod[OPT_CUSTOM_GAMMA].title = SANE_TITLE_CUSTOM_GAMMA;
  sod[OPT_CUSTOM_GAMMA].desc  = SANE_DESC_CUSTOM_GAMMA;
  sod[OPT_CUSTOM_GAMMA].type  = SANE_TYPE_BOOL;
  val[OPT_CUSTOM_GAMMA].w     = SANE_FALSE;

  sod[OPT_GAMMA_BIND].name  = SANE_NAME_ANALOG_GAMMA_BIND;
  sod[OPT_GAMMA_BIND].title = SANE_TITLE_ANALOG_GAMMA_BIND;
  sod[OPT_GAMMA_BIND].desc  = SANE_DESC_ANALOG_GAMMA_BIND;
  sod[OPT_GAMMA_BIND].type  = SANE_TYPE_BOOL;
  val[OPT_GAMMA_BIND].w     = SANE_TRUE;

  sod[OPT_GAMMA_VECTOR].name  = SANE_NAME_GAMMA_VECTOR;
  sod[OPT_GAMMA_VECTOR].title = SANE_TITLE_GAMMA_VECTOR;
  sod[OPT_GAMMA_VECTOR].desc  = SANE_DESC_GAMMA_VECTOR;
  sod[OPT_GAMMA_VECTOR].type  = SANE_TYPE_INT;
  sod[OPT_GAMMA_VECTOR].unit  = SANE_UNIT_NONE;
  sod[OPT_GAMMA_VECTOR].size  = 256 * sizeof(SANE_Word);
  sod[OPT_GAMMA_VECTOR].cap   |= SANE_CAP_INACTIVE;
  sod[OPT_GAMMA_VECTOR].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_GAMMA_VECTOR].constraint      = &u8_range;
  val[OPT_GAMMA_VECTOR].wa     = &(ms->gamma_table[0][0]);

  sod[OPT_GAMMA_VECTOR_R].name  = SANE_NAME_GAMMA_VECTOR_R;
  sod[OPT_GAMMA_VECTOR_R].title = SANE_TITLE_GAMMA_VECTOR_R;
  sod[OPT_GAMMA_VECTOR_R].desc  = SANE_DESC_GAMMA_VECTOR_R;
  sod[OPT_GAMMA_VECTOR_R].type  = SANE_TYPE_INT;
  sod[OPT_GAMMA_VECTOR_R].unit  = SANE_UNIT_NONE;
  sod[OPT_GAMMA_VECTOR_R].size  = 256 * sizeof(SANE_Word);
  sod[OPT_GAMMA_VECTOR_R].cap   |= SANE_CAP_INACTIVE;
  /*  sod[OPT_GAMMA_VECTOR_R].cap   = 0;*/
  sod[OPT_GAMMA_VECTOR_R].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_GAMMA_VECTOR_R].constraint      = &u8_range;
  val[OPT_GAMMA_VECTOR_R].wa     = &(ms->gamma_table[1][0]);

  sod[OPT_GAMMA_VECTOR_G].name  = SANE_NAME_GAMMA_VECTOR_G;
  sod[OPT_GAMMA_VECTOR_G].title = SANE_TITLE_GAMMA_VECTOR_G;
  sod[OPT_GAMMA_VECTOR_G].desc  = SANE_DESC_GAMMA_VECTOR_G;
  sod[OPT_GAMMA_VECTOR_G].type  = SANE_TYPE_INT;
  sod[OPT_GAMMA_VECTOR_G].unit  = SANE_UNIT_NONE;
  sod[OPT_GAMMA_VECTOR_G].size  = 256 * sizeof(SANE_Word);
  sod[OPT_GAMMA_VECTOR_G].cap   |= SANE_CAP_INACTIVE;
  /*  sod[OPT_GAMMA_VECTOR_G].cap   = 0;*/
  sod[OPT_GAMMA_VECTOR_G].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_GAMMA_VECTOR_G].constraint      = &u8_range;
  val[OPT_GAMMA_VECTOR_G].wa     = &(ms->gamma_table[2][0]);

  sod[OPT_GAMMA_VECTOR_B].name  = SANE_NAME_GAMMA_VECTOR_B;
  sod[OPT_GAMMA_VECTOR_B].title = SANE_TITLE_GAMMA_VECTOR_B;
  sod[OPT_GAMMA_VECTOR_B].desc  = SANE_DESC_GAMMA_VECTOR_B;
  sod[OPT_GAMMA_VECTOR_B].type  = SANE_TYPE_INT;
  sod[OPT_GAMMA_VECTOR_B].unit  = SANE_UNIT_NONE;
  sod[OPT_GAMMA_VECTOR_B].size  = 256 * sizeof(SANE_Word);
  sod[OPT_GAMMA_VECTOR_B].cap   |= SANE_CAP_INACTIVE;
  sod[OPT_GAMMA_VECTOR_B].constraint_type = SANE_CONSTRAINT_RANGE;
  sod[OPT_GAMMA_VECTOR_B].constraint      = &u8_range;
  val[OPT_GAMMA_VECTOR_B].wa     = &(ms->gamma_table[3][0]);


  /*
  sod[OPT_].name  = SANE_NAME_;
  sod[OPT_].title = SANE_TITLE_;
  sod[OPT_].desc  = SANE_DESC_;
  sod[OPT_].type  = SANE_TYPE_;
  sod[OPT_].unit  = SANE_UNIT_NONE;
  sod[OPT_].size  = sizeof(SANE_Word);
  sod[OPT_].cap   = 0;
  sod[OPT_].constraint_type = SANE_CONSTRAINT_NONE;
  sod[OPT_].constraint      = ;
  val[OPT_].w     = ;
  */
  DBG(23, "init_options:  done.\n");


  return SANE_STATUS_GOOD;
}




static SANE_Status
parse_inquiry(Microtek_Info *mi, unsigned char *result)
{
  DBG(23, "parse_inquiry...");
  strncpy(mi->vendor_id, &result[8], 8);
  strncpy(mi->model_name, &result[16], 16);
  strncpy(mi->revision_num, &result[32], 4);
  mi->vendor_id[8]   = 0;
  mi->model_name[16] = 0;
  mi->revision_num[5] = 0;
  
  mi->device_type                = (SANE_Byte)(result[0] & 0x1f);
  mi->SCSI_firmware_ver_major    = (SANE_Byte)((result[1] & 0xf0) >> 4);
  mi->SCSI_firmware_ver_minor    = (SANE_Byte)(result[1] & 0x0f);
  mi->scanner_firmware_ver_major = (SANE_Byte)((result[2] & 0xf0) >> 4);
  mi->scanner_firmware_ver_minor = (SANE_Byte)(result[2] & 0x0f);
  mi->response_data_format       = (SANE_Byte)(result[3]);
  
  mi->res_step                   = (SANE_Byte)(result[56] & 0x03);
  mi->modes                      = (SANE_Byte)(result[57]);
  mi->pattern_count              = (SANE_Int)(result[58] & 0x7f);
  mi->pattern_dwnld              = (SANE_Byte)(result[58] & 0x80) > 0;
  mi->feed_type                  = (SANE_Byte)(result[59] & 0x0F);
  mi->compress_type              = (SANE_Byte)(result[59] & 0x30);
  mi->unit_type                  = (SANE_Byte)(result[59] & 0xC0);
  
  mi->doc_size_code              = (SANE_Byte)result[60];
  /*    size_unit = INCH; XXXXXXXXXX*/
  switch (mi->doc_size_code) {
  case 0x00:
    mi->max_x = SANE_FIX(8.5); mi->max_y = SANE_FIX(14.0); break;
  case 0x01:   
    mi->max_x = SANE_FIX(8.5); mi->max_y = SANE_FIX(11.0); break;
  case 0x02:
    mi->max_x = SANE_FIX(8.5); mi->max_y = SANE_FIX(11.69); break;
  case 0x03:
    mi->max_x = SANE_FIX(8.5); mi->max_y = SANE_FIX(13.0); break;
  case 0x04:
    mi->max_x = SANE_FIX(8.0); mi->max_y = SANE_FIX(10.0); break;
  case 0x05:
    mi->max_x = SANE_FIX(8.3); mi->max_y = SANE_FIX(14.0); break;
  case 0x80:
    /* Slide format, size is mm */
    mi->max_x = SANE_FIX(35.0 / MM_PER_INCH);
    mi->max_y = SANE_FIX(35.0 / MM_PER_INCH);  
      break;
  case 0x81:
    mi->max_x = SANE_FIX(5.0); mi->max_y = SANE_FIX(5.0); break;
  case 0x82:
    /* Slide format, size is mm */
    mi->max_x = SANE_FIX(36.0 / MM_PER_INCH);
    mi->max_y = SANE_FIX(36.0 / MM_PER_INCH); 
    break;
  default:
    /* Undefined document format code */
      mi->max_x =  SANE_FIX(0.0); mi->max_y =  SANE_FIX(0.0);
  }
  
  mi->cont_settings              = (SANE_Int)((result[61] & 0xf0) >> 4);
  mi->exp_settings               = (SANE_Int)(result[61] & 0x0f);
  mi->model_code                 = (SANE_Byte)(result[62]);
  switch (mi->model_code) {
  case 0x50:
  case 0x54:
  case 0x55:
  case 0x57:
  case 0x58:
  case 0x5f:
  case 0x56: /* ??? XXXXXXXXX */
    mi->base_resolution = 300;
    break;
  case 0x59: /* ??? XXXXXXXXX */
  case 0x63:
  case 0x66:
    mi->base_resolution = 600;
    break;
  case 0x51: /* ??? XXXXXXXXX */
  case 0x52:
    mi->base_resolution = 1950;
    break;
  default:
    /* ??? XXXXXXXXXXX */
    break;
  }

  mi->source_options             = (SANE_Byte)(result[63]); 
  
  mi->expanded_resolution        = (result[64] & 0x01);
  mi->enhance_cap                = (result[65] & 0x03);
  
  switch (result[66] & 0x0F) {
  case 0x00: mi->max_lookup_size =     0; break;
  case 0x01: mi->max_lookup_size =   256; break;
  case 0x03: mi->max_lookup_size =  1024; break;
  case 0x05: mi->max_lookup_size =  4096; break;
  case 0x09: mi->max_lookup_size = 65536; break;
  default: DBG(23, "parse_inquiry:  BAD!!!!!!!!!!! XXXXXXXXXXXXXXXXXXXX");
  }
  
  switch (result[66] >> 5) {
  case 0x00: mi->max_gamma_val =   255;  mi->gamma_size = 1;  break;
  case 0x01: mi->max_gamma_val =  1023;  mi->gamma_size = 2;  break;
  case 0x02: mi->max_gamma_val =  4095;  mi->gamma_size = 2;  break;
  case 0x03: mi->max_gamma_val = 65535;  mi->gamma_size = 2;  break;
  default:   mi->max_gamma_val =     0;  mi->gamma_size = 0;
  }
  
  mi->fast_color_preview         = (SANE_Byte)(result[67] & 0x01);
  mi->xfer_format_select         = (SANE_Byte)(result[68] & 0x01);
  mi->color_sequence             = (SANE_Byte)(result[69] & 0x7f);
  mi->does_3pass                 = (SANE_Byte)(result[69] & 0x80);
  mi->does_mode1                 = (SANE_Byte)(result[71] & 0x01);
  
  
  mi->contrast_vals              = (SANE_Int)(result[72]);
  mi->min_contrast = -42;
  mi->max_contrast =  49;
  if (mi->contrast_vals)
    mi->max_contrast = (mi->contrast_vals * 7) - 49;
  
  mi->exposure_vals              = (SANE_Int)(result[73]);
  mi->min_exposure  = -18;
  mi->max_exposure  =  21;
  if (mi->exposure_vals)
    mi->max_exposure  = (mi->exposure_vals * 3) - 21;
  
  mi->bit_formats                = (SANE_Byte)(result[74] & 0x0F);
  mi->extra_cap                  = (SANE_Byte)(result[75] & 0x07);
  
  return SANE_STATUS_GOOD;
}


static SANE_Status 
dump_inquiry(Microtek_Info *mi, unsigned char *result)
{
  int i;

  DBG(23, "dump_inquiry:  dumping device info....\n");
  fprintf(stderr, "========== Scanner Inquiry Block ==========\n");
  for (i=0; i<96; i++) {
    if (!(i % 16) && (i)) fprintf(stderr, "\n");
    fprintf(stderr, "%02x ", (int)result[i]);
  }
  fprintf(stderr, "\n\n");

  fprintf(stderr, "========== Scanner Inquiry Report ==========\n");
  fprintf(stderr, "===== Scanner ID...\n");
  fprintf(stderr, "Device Type Code: 0x%02x\n", mi->device_type);
  fprintf(stderr, "Model Code: 0x%02x\n", mi->model_code);
  fprintf(stderr, "Vendor Name: '%s'   Model Name: '%s'\n",
	  mi->vendor_id, mi->model_name);
  fprintf(stderr, "Firmware Rev: '%s'\n", mi->revision_num);
  fprintf(stderr, 
	  "SCSI F/W version: %1d.%1d     Scanner F/W version: %1d.%1d\n",
	  mi->SCSI_firmware_ver_major, mi->SCSI_firmware_ver_minor,
	  mi->scanner_firmware_ver_major, mi->scanner_firmware_ver_minor);
  fprintf(stderr, "Response data format: 0x%02x\n", mi->response_data_format);
  
  fprintf(stderr, "===== Imaging Capabilities...\n");
  fprintf(stderr, "Modes:  %s%s%s%s%s%s%s\n",
	  (mi->modes & MI_MODES_LINEART) ? "Lineart " : "",
	  (mi->modes & MI_MODES_HALFTONE) ? "Halftone " : "",
	  (mi->modes & MI_MODES_GRAY) ? "Gray " : "",
	  (mi->modes & MI_MODES_COLOR) ? "Color " : "",
	  (mi->modes & MI_MODES_TRANSMSV) ? "(X-msv) " : "",
	  (mi->modes & MI_MODES_ONEPASS) ? "(OnePass) " : "",
	  (mi->modes & MI_MODES_NEGATIVE) ? "(Negative) " : "");
  fprintf(stderr, 
	  "Resolution Step Sizes: %s%s    Expanded Resolution Support? %s\n",
	  (mi->res_step & MI_RESSTEP_1PER) ? "1% " : "",
	  (mi->res_step & MI_RESSTEP_5PER) ? "5%" : "",
	  (mi->expanded_resolution) ? "yes" : "no");
  fprintf(stderr, "Supported Bits Per Sample: %s8 %s%s%s\n",
	  (mi->bit_formats & MI_FMT_CAP_4BPP) ? "4 " : "",
	  (mi->bit_formats & MI_FMT_CAP_10BPP) ? "10 " : "",
	  (mi->bit_formats & MI_FMT_CAP_12BPP) ? "12 " : "",
	  (mi->bit_formats & MI_FMT_CAP_16BPP) ? "16 " : "");
  fprintf(stderr, "Max. document size code: 0x%02x\n",
	  mi->doc_size_code);
  /* max x, max y XXXXXXXX */
  fprintf(stderr, "Frame units:  %s%s\n",
	  (mi->unit_type & MI_UNIT_PIXELS) ? "pixels  " : "",
	  (mi->unit_type & MI_UNIT_8TH_INCH) ? "1/8\"'s " : "");
  fprintf(stderr, "# of built-in halftones: %d   Downloadable patterns? %s\n",
	  mi->pattern_count, (mi->pattern_dwnld) ? "Yes" : "No");

  fprintf(stderr, "Data Compression: %s%s\n",
	  (mi->compress_type & MI_COMPRSS_HUFF) ? "huffman " : "",
	  (mi->compress_type & MI_COMPRSS_RD) ? "read-data " : "");
  fprintf(stderr, "Contrast Settings: %d   Exposure Settings: %d\n",
	  mi->cont_settings, mi->exp_settings);
  fprintf(stderr, "Adjustable Shadow/Highlight? %s   Adjustable Midtone? %s\n",
	  (mi->enhance_cap & MI_ENH_CAP_SHADOW) ? "yes" : "no ",
	  (mi->enhance_cap & MI_ENH_CAP_MIDTONE) ? "yes" : "no ");
  fprintf(stderr, "Digital brightness/offset? %s\n",
	  (mi->extra_cap & MI_EXCAP_OFF_CTL) ? "yes" : "no");
  fprintf(stderr, 
	  "Gamma Table Size: %d entries of %d bytes (max. value: %d)\n",
	  mi->max_lookup_size, mi->gamma_size, mi->max_gamma_val);

  fprintf(stderr, "===== Source Options...\n");
  fprintf(stderr, "Feed type:  %s%s   ADF support? %s\n",
	  (mi->feed_type & MI_FEED_FLATBED) ? "flatbed " : "",
	  (mi->feed_type & MI_FEED_EDGEFEED) ? "edge-feed " : "",
	  (mi->feed_type & MI_FEED_AUTOSUPP) ? "yes" : "no");  
  fprintf(stderr, "Document Feeder Support? %s   Feeder Backtracking? %s\n",
	  (mi->source_options & MI_SRC_FEED_SUPP) ? "yes" : "no ",
	  (mi->source_options & MI_SRC_FEED_BT) ? "yes" : "no ");
  fprintf(stderr, "Feeder Installed? %s          Feeder Ready? %s\n",
	  (mi->source_options & MI_SRC_HAS_FEED) ? "yes" : "no ",
	  (mi->source_options & MI_SRC_FEED_RDY) ? "yes" : "no ");
  fprintf(stderr, "Transparency Adapter Installed? %s\n",
	  (mi->source_options & MI_SRC_HAS_TRANS) ? "yes" : "no ");
  /* GET_TRANS GET_FEED XXXXXXXXX */
  /* mt_SWslct ???? XXXXXXXXXXX */
  /*#define DOC_ON_FLATBED 0x00
    #define DOC_IN_FEEDER  0x01
    #define TRANSPARENCY   0x10
    */
  fprintf(stderr, "Fast Color Prescan? %s\n",
	  (mi->fast_color_preview) ? "yes" : "no");
  fprintf(stderr, "Selectable Transfer Format? %s\n",
	  (mi->xfer_format_select) ? "yes" : "no");
  fprintf(stderr, "Color Transfer Sequence: ");
  switch (mi->color_sequence) {
  case MI_COLSEQ_PLANE: 
    fprintf(stderr, "plane-by-plane (3-pass)\n"); break;
  case MI_COLSEQ_PIXEL: 
    fprintf(stderr, "pixel-by-pixel RGB\n"); break;
  case MI_COLSEQ_RGB:
    fprintf(stderr, "line-by-line, R-G-B sequence\n"); break;
  case MI_COLSEQ_NONRGB:
    fprintf(stderr, "line-by-line, non-sequential with headers\n"); break;
  default:
    fprintf(stderr, "UNKNOWN CODE (0x%02x)\n", mi->color_sequence);
  }
  /*    if (mt_OnePass)
      fprintf(stderr, "Three pass scanning%s supported\n", mt_3pass ? "" : " not");*/
  fprintf(stderr, "ModeSelect-1 and ModeSense-1 Support? %s\n",
	  (mi->does_mode1) ? "yes" : "no");
  fprintf(stderr, "Can Disable Linearization Table? %s\n",
	  (mi->extra_cap & MI_EXCAP_DIS_LNTBL) ? "yes" : "no");
  fprintf(stderr, "Can Disable Start-of-Scan Recalibration? %s\n",
	  (mi->extra_cap & MI_EXCAP_DIS_RECAL) ? "yes" : "no");


  /*
    fprintf(stderr, "cntr_vals = %d, min_cntr = %d, max_cntr = %d\n",
    cntr_vals, min_cntr, max_cntr);
    fprintf(stderr, "exp_vals = %d, min_exp = %d, max_exp = %d\n",
    exp_vals, min_exp, max_exp);
    */
  
  return SANE_STATUS_GOOD;
}





static SANE_Status
id_microtek(u_int8_t *result, char **model_string)
{
  DBG(23, "id_microtek...\n");
  if (!(strncmp("MICROTEK", &(result[8]), 8)) ||
      !(strncmp("        ", &(result[8]), 8)) ) {
    switch (result[62]) {
    case 0x50 :
      *model_string = "ScanMaker II/IIXE";  break;
    case 0x51 :
      *model_string = "ScanMaker 45t";      break;
    case 0x52 :
      *model_string = "ScanMaker 35t";      break;
    case 0x54 :
      *model_string = "ScanMaker IISP";     break;
    case 0x55 :
      *model_string = "ScanMaker IIER";     break;
    case 0x56 :
      *model_string = "ScanMaker A3t";      break;
    case 0x57 :
      *model_string = "ScanMaker IIHR";     break;
    case 0x58 :
      *model_string = "ScanMaker IIG";      break;
    case 0x59 :
      *model_string = "ScanMaker III";      break;
    case 0x5f :
      *model_string = "ScanMaker E3";       break;
    case 0x63 :
    case 0x66 :
      *model_string = "ScanMaker E6";       break;
    default :
      return SANE_STATUS_INVAL;
    }
    return SANE_STATUS_GOOD;
  }
  DBG(23, "id_microtek:  not microtek:  %d, %d, %d\n",
      strncmp("MICROTEK", &(result[8]), 8),
      strncmp("        ", &(result[8]), 8),
      result[62]);
  return SANE_STATUS_INVAL;
}



static SANE_Status 
attach_scanner(const char *devicename, Microtek_Device **devp)
{
  Microtek_Device *dev;
  int sfd;
  size_t size;
  unsigned char result[0x60];
  SANE_Status status;
  char *model_string;
  u_int8_t inquiry[] = { 0x12, 0, 0, 0, 0x60, 0 };

  DBG(15,"attach_scanner: %s\n", devicename);
  DBG(23, "attach_scanner: attaching %s...\n", devicename);
  /* check if device is already known... */
  for (dev = first_dev; dev; dev = dev->next) {
    if (strcmp(dev->sane.name, devicename) == 0) {
      if (devp) *devp = dev;
      return SANE_STATUS_GOOD;
    }
  }

  /* open scsi device... */
  DBG(3, "attach_scanner: opening %s\n", devicename);
  if (sanei_scsi_open(devicename, &sfd, sense_handler) != 0) {
    DBG(1, "attach_scanner: open failed\n");
    return SANE_STATUS_INVAL;
  }

  /* say hello... */
  DBG(3, "attach: sending INQUIRY\n");
  size = sizeof(result);
  status = sanei_scsi_cmd(sfd, inquiry, sizeof(inquiry), result, &size);
  if (status != SANE_STATUS_GOOD || size != 0x60) {
    DBG(1, "attach: inquiry failed (%s)\n", sane_strstatus (status));
    sanei_scsi_close (sfd);
    return status;
  }

  /* wait, and close. XXXXXXXXXXXX */
  /*  status = wait_ready(sfd);*/
  sanei_scsi_close(sfd);
  if (status != SANE_STATUS_GOOD)
    return status;

  if (id_microtek(result, &model_string) != SANE_STATUS_GOOD) {
      DBG(1, "attach: device doesn't look like a Microtek scanner ");
      return SANE_STATUS_INVAL;
  }
  
  dev=malloc(sizeof(*dev));
  if (!dev) return SANE_STATUS_NO_MEM;
  memset(dev, 0, sizeof(*dev));

  /* initialize dev structure */
  dev->sane.name   = strdup(devicename);
  dev->sane.vendor = "Microtek";
  dev->sane.model  = strdup(model_string);
  dev->sane.type   = "flatbed scanner";

  parse_inquiry(&(dev->info), result);
  dump_inquiry(&(dev->info), result);

  /* link into device list... */
  ++num_devices;
  dev->next = first_dev;
  first_dev = dev;
  if (devp) *devp = dev;

  DBG(23, "attach_scanner:  happy.\n");

  return SANE_STATUS_GOOD;
}



/***** sane_init() *****************************************/
SANE_Status
sane_init(SANE_Int *version_code, SANE_Auth_Callback authorize)
{
  char dev_name[PATH_MAX];
  size_t len;
  FILE *fp;

  DBG_INIT();

  if (version_code)
    *version_code = SANE_VERSION_CODE (V_MAJOR, V_MINOR, 0);

  fp = sanei_open_config (PATH_MICROTEK_CONFIG);
  if (!fp) {
    /* default to /dev/scanner instead of insisting on config file */
    attach_scanner("/dev/scanner", 0);
    return SANE_STATUS_GOOD;
  }
  
  while (fgets (dev_name, sizeof (dev_name), fp)) {
    if (dev_name[0] == '#') continue;	/* ignore line comments */
    len = strlen (dev_name);
    if (dev_name[len - 1] == '\n') dev_name[--len] = '\0';
    if (!len) continue;			/* ignore empty lines */
    attach_scanner(dev_name, 0);
    }
  fclose (fp);
  return SANE_STATUS_GOOD;
}
/*************************************************************/


/***** sane_exit() *******************************************/
void
sane_exit (void)
{
  Microtek_Device *next;

  DBG(23, "sane_exit...\n");
  /* close all leftover Scanners */
  /*(beware of how sane_close interacts with linked list) */
  while (first_handle != NULL)
    sane_close(first_handle);
  /* free up device list */
  while (first_dev != NULL) {
    next = first_dev->next;
    free((void *) first_dev->sane.name); /*XXXXXX*/
    free((void *) first_dev->sane.model);/*XXXXXX*/
    free(first_dev);
    first_dev = next;
  }
  DBG(23, "sane_exit:  MICROTEK says goodbye.\n");
}
/*************************************************************/

/***** sane_get_devices **************************************/
SANE_Status
sane_get_devices(const SANE_Device ***device_list, 
		 SANE_Bool local_only)
{
  static const SANE_Device **devlist = 0;
  Microtek_Device *dev;
  int i;

  DBG(10, "sane_get_devices");

  /* we keep an internal copy */
  if (devlist)
    free(devlist);  /* hmm, free it if we want a new one, I guess.  YYYYY*/

  devlist = malloc((num_devices + 1) * sizeof(devlist[0]));
  if (!devlist) return SANE_STATUS_NO_MEM;

  for (i=0, dev=first_dev; i < num_devices; dev = dev->next)
    devlist[i++] = &dev->sane;
  devlist[i++] = 0;

  *device_list = devlist;
  return SANE_STATUS_GOOD;
}
/*************************************************************/


/*** sane_open ***/

SANE_Status
sane_open(SANE_String_Const devicename,
	  SANE_Handle *handle)
{
  Microtek_Scanner *scanner;
  Microtek_Device *dev;
  SANE_Status status;

  DBG(10, "sane_open\n");

  /* find device... */
  if (devicename[0]) {
    for (dev = first_dev; dev; dev = dev->next) {
      if (strcmp(dev->sane.name, devicename) == 0)
	break;
    }
    if (!dev) {  /* not in list, try manually... */
      status = attach_scanner(devicename, &dev);
      if (status != SANE_STATUS_GOOD) return status;
    }
  } else {  /* no device specified, so use first */
    dev = first_dev;
  }
  if (!dev) return SANE_STATUS_INVAL;

  /* create a scanner... */
  scanner = malloc(sizeof(*scanner));
  if (!scanner) return SANE_STATUS_NO_MEM;

  memset(scanner, 0, sizeof(*scanner));

  /* initialize scanner dependent stuff */

  scanner->unit_type = 
    (dev->info.unit_type & MI_UNIT_PIXELS) ? MS_UNIT_PIXELS : MS_UNIT_18INCH;
  /*
  scanner->unit_type = 
    (dev->info.unit_type & MI_UNIT_8TH_INCH) ? MS_UNIT_18INCH : MS_UNIT_PIXELS;
    */
  scanner->res_type =
    (dev->info.res_step & MI_RESSTEP_1PER) ? MS_RES_1PER : MS_RES_5PER;
  scanner->midtone_support = 
    (dev->info.enhance_cap & MI_ENH_CAP_MIDTONE) ? SANE_TRUE : SANE_FALSE;
  scanner->paper_length = 
    (scanner->unit_type == MS_UNIT_PIXELS) ? 
    (SANE_Int)(SANE_UNFIX(dev->info.max_y) * dev->info.base_resolution) :
    (SANE_Int)(SANE_UNFIX(dev->info.max_y) * 8);
  
  scanner->bright_r = 0;
  scanner->bright_g = 0;
  scanner->bright_b = 0;
  scanner->allow_calibrate = SANE_TRUE;
  scanner->onepass = (dev->info.modes & MI_MODES_ONEPASS);
  scanner->useADF = SANE_FALSE;
  scanner->prescan = SANE_FALSE;
  scanner->transparency = SANE_FALSE;
  scanner->allowbacktrack = SANE_TRUE;  /* ??? XXXXXXX */
  scanner->reversecolors = SANE_FALSE;
  scanner->expandedresolution = SANE_FALSE;
  scanner->fastprescan = SANE_FALSE;
  scanner->filter = MS_FILT_CLEAR;
  scanner->bits_per_color = 8;

  /* init gamma tables */
  {
    int j;
    for (j=0; j<256; j++) {
      scanner->gamma_table[0][j] = j;
      scanner->gamma_table[1][j] = j;
      scanner->gamma_table[2][j] = j;
      scanner->gamma_table[3][j] = j;
    }
  }

  scanner->scanning = SANE_FALSE;
  scanner->sfd = -1;
  scanner->dev = dev;

  DBG(23, "sane_open:  gonna init opts:  ");
  /* init other stuff... XXXXXXXX */
  if ((status = init_options(scanner)) != SANE_STATUS_GOOD)
    return status;

  scanner->next = first_handle;
  first_handle = scanner;
  *handle = scanner;

  return SANE_STATUS_GOOD;
}




void
sane_close (SANE_Handle handle)
{
  Microtek_Scanner *ms = handle;

  DBG(10, "sane_close");
  /* free malloc'ed stuff (strdup counts too!) */
  free((void *) ms->sod[OPT_MODE].constraint.string_list);/*XXXXX*/
  free((void *) ms->sod[OPT_RESOLUTION].constraint.range);/*XXXXX*/
  free(ms->val[OPT_MODE].s);
  free(ms->val[OPT_HALFTONE_PATTERN].s);
  /* remove Scanner from linked list */
  if (first_handle == ms)
    first_handle = ms->next;
  else {
    Microtek_Scanner *ts = first_handle;
    while ((ts != NULL) && (ts->next != ms)) ts = ts->next;
    ts->next = ts->next->next; /* == ms->next */
  }
  /* finally, say goodbye to the Scanner */
  free(ms);
}




/***** get_option_descriptor *************************/
const SANE_Option_Descriptor *
sane_get_option_descriptor (SANE_Handle handle,
			    SANE_Int option)
{
  Microtek_Scanner *scanner = handle;

  DBG(23, "sane_get_option_descriptor (%d)...\n", option);
  if ((unsigned)option >= NUM_OPTIONS)
    return NULL;
  return &(scanner->sod[option]);
}

/*****************************************************/


/***** control_option ********************************/
SANE_Status 
sane_control_option (SANE_Handle handle,
		     SANE_Int option,
		     SANE_Action action,
		     void *value,
		     SANE_Int *info)
{
  Microtek_Scanner *scanner = handle;
  SANE_Option_Descriptor *sod;
  Microtek_Option_Value  *val;
  SANE_Status status;

  DBG(2, "control_option(handle=%p,opt=%d,act=%d,val=%p,info=%p)\n",
      handle, option, action, value, info);

  sod = scanner->sod;
  val = scanner->val;

  if (scanner->scanning) return SANE_STATUS_DEVICE_BUSY;
  if ((option >= NUM_OPTIONS) || (option < 0))
    return SANE_STATUS_INVAL;
  if (!SANE_OPTION_IS_ACTIVE(scanner->sod[option].cap))
    return SANE_STATUS_INVAL;

  if (info) *info = 0;

  /* choose by action */
  switch (action) {

  case SANE_ACTION_GET_VALUE:
    switch (option) {
      /* word options... */
    case OPT_RESOLUTION:
    case OPT_SPEED:
    case OPT_BACKTRACK:
    case OPT_NEGATIVE:
    case OPT_PREVIEW:
    case OPT_TL_X:
    case OPT_TL_Y:
    case OPT_BR_X:
    case OPT_BR_Y:
    case OPT_BRIGHTNESS:
    case OPT_CONTRAST:
    case OPT_HIGHLIGHT:
    case OPT_SHADOW:
    case OPT_MIDTONE:
    case OPT_CUSTOM_GAMMA:
    case OPT_GAMMA_BIND:
      *(SANE_Word *)value = val[option].w;
      return SANE_STATUS_GOOD;

      /* word-array options... */
      /*    case OPT_HALFTONE_PATTERN:*/
    case OPT_GAMMA_VECTOR:
    case OPT_GAMMA_VECTOR_R:
    case OPT_GAMMA_VECTOR_G:
    case OPT_GAMMA_VECTOR_B:
      memcpy(value, val[option].wa, sod[option].size);
      return SANE_STATUS_GOOD;
      
      /* string options... */
    case OPT_MODE:
    case OPT_HALFTONE_PATTERN:
      strcpy(value, val[option].s);
      return SANE_STATUS_GOOD;

      /* others.... */
    case 0:
      *(SANE_Word *) value = NUM_OPTIONS;
      return SANE_STATUS_GOOD;

    default:
      return SANE_STATUS_INVAL;
    }
    break;
    
    /* XXXXXXXXX  What about LSB/MSB byte order issues for portability? */
  case SANE_ACTION_SET_VALUE: {
    status = sanei_constrain_value(sod + option, value, info);
    if (status != SANE_STATUS_GOOD)
      return status;
    
    switch (option) {
      /* set word options... */
    case OPT_RESOLUTION:
      DBG(23, "sane_control_option:  r: %f\n", 
	      SANE_UNFIX(*(SANE_Word *)value));
    case OPT_TL_X:
    case OPT_TL_Y:
    case OPT_BR_X:
    case OPT_BR_Y:
      if (info)
	*info |= SANE_INFO_RELOAD_PARAMS;
    case OPT_SPEED:
    case OPT_PREVIEW:
    case OPT_BACKTRACK:
    case OPT_NEGATIVE:
    case OPT_BRIGHTNESS:
    case OPT_CONTRAST:
    case OPT_HIGHLIGHT:
    case OPT_SHADOW:
    case OPT_MIDTONE:
      val[option].w = *(SANE_Word *)value;
      return SANE_STATUS_GOOD;

    case OPT_CUSTOM_GAMMA:
    case OPT_GAMMA_BIND:
      val[option].w = *(SANE_Word *) value;
      if (info) *info |= SANE_INFO_RELOAD_OPTIONS;
      if (val[OPT_CUSTOM_GAMMA].w == SANE_FALSE) {
	sod[OPT_GAMMA_VECTOR].cap |= SANE_CAP_INACTIVE;
	sod[OPT_GAMMA_VECTOR_R].cap |= SANE_CAP_INACTIVE;
	sod[OPT_GAMMA_VECTOR_G].cap |= SANE_CAP_INACTIVE;
	sod[OPT_GAMMA_VECTOR_B].cap |= SANE_CAP_INACTIVE;
      } else {
	if (val[OPT_GAMMA_BIND].w == SANE_TRUE) {
	  sod[OPT_GAMMA_VECTOR].cap &= ~SANE_CAP_INACTIVE;
	  sod[OPT_GAMMA_VECTOR_R].cap |= SANE_CAP_INACTIVE;
	  sod[OPT_GAMMA_VECTOR_G].cap |= SANE_CAP_INACTIVE;
	  sod[OPT_GAMMA_VECTOR_B].cap |= SANE_CAP_INACTIVE;
	} else {
	  sod[OPT_GAMMA_VECTOR].cap |= SANE_CAP_INACTIVE;
	  sod[OPT_GAMMA_VECTOR_R].cap &= ~SANE_CAP_INACTIVE;
	  sod[OPT_GAMMA_VECTOR_G].cap &= ~SANE_CAP_INACTIVE;
	  sod[OPT_GAMMA_VECTOR_B].cap &= ~SANE_CAP_INACTIVE;
	}
      }
      return SANE_STATUS_GOOD;

    case OPT_MODE:
      if (val[option].s)
	free(val[option].s);
      val[option].s = strdup(value);
      if (info)
	*info |= SANE_INFO_RELOAD_OPTIONS | SANE_INFO_RELOAD_PARAMS;
      /* disable highlight/etc if not color or gray mode */
      if (strcmp(val[option].s, M_COLOR) &&
	  strcmp(val[option].s, M_GRAY)) {
	DBG(23, "sane_control_option:  hi dis\n");
	sod[OPT_HIGHLIGHT].cap |= SANE_CAP_INACTIVE;
	sod[OPT_SHADOW].cap |= SANE_CAP_INACTIVE;
	sod[OPT_MIDTONE].cap |= SANE_CAP_INACTIVE;
      } else {
	DBG(23, "sane_control_option:  hi en\n");
	sod[OPT_HIGHLIGHT].cap &= ~SANE_CAP_INACTIVE;
	sod[OPT_SHADOW].cap &= ~SANE_CAP_INACTIVE;
	sod[OPT_MIDTONE].cap &= ~SANE_CAP_INACTIVE;
      }
      if (strcmp(val[option].s, M_HALFTONE)) {
	sod[OPT_HALFTONE_PATTERN].cap |= SANE_CAP_INACTIVE;
      } else {
	sod[OPT_HALFTONE_PATTERN].cap &= ~SANE_CAP_INACTIVE;
      }
      if (strcmp(val[option].s, M_COLOR)) { /* not color */
	sod[OPT_GAMMA_BIND].cap |= SANE_CAP_INACTIVE;
        val[OPT_GAMMA_BIND].w = SANE_TRUE;
      } else {
	sod[OPT_GAMMA_BIND].cap &= ~SANE_CAP_INACTIVE;
      }
      return SANE_STATUS_GOOD;

    case OPT_HALFTONE_PATTERN:
      if (val[option].s)
	free(val[option].s);
      val[option].s = strdup(value);
      return SANE_STATUS_GOOD;

    case OPT_GAMMA_VECTOR:
    case OPT_GAMMA_VECTOR_R:
    case OPT_GAMMA_VECTOR_G:
    case OPT_GAMMA_VECTOR_B:
      memcpy(val[option].wa, value, sod[option].size);
      return SANE_STATUS_GOOD;

    default:
      return SANE_STATUS_INVAL;
    }
  }
  break;
  
  case SANE_ACTION_SET_AUTO:
    return SANE_STATUS_UNSUPPORTED;	/* We are DUMB */
  }
  return SANE_STATUS_GOOD;
}
/*****************************************************/



SANE_Status
sane_get_parameters (SANE_Handle handle,
		     SANE_Parameters *params)
{
  Microtek_Scanner *s = handle;

  DBG(23, "sane_get_parameters...\n");

  if (!s->scanning) {

    /* decipher scan mode */
    /*    if (s->val[OPT_PREVIEW].w) 
      s->mode = MS_MODE_GRAY;
    else*/ if (!(strcmp(s->val[OPT_MODE].s, M_LINEART)))
      s->mode = MS_MODE_LINEART;
    else if (!(strcmp(s->val[OPT_MODE].s, M_HALFTONE)))
      s->mode = MS_MODE_HALFTONE;
    else if (!(strcmp(s->val[OPT_MODE].s, M_GRAY)))
      s->mode = MS_MODE_GRAY;
    else if (!(strcmp(s->val[OPT_MODE].s, M_COLOR)))
      s->mode = MS_MODE_COLOR;

    s->onepasscolor = 
      (s->onepass) && (s->mode == MS_MODE_COLOR);

    if (s->res_type == MS_RES_1PER) {
      s->resolution = (SANE_Int)(SANE_UNFIX(s->val[OPT_RESOLUTION].w));
      s->resolution_code = 
	0xFF & ((s->resolution * 100) / s->dev->info.base_resolution);
      DBG(23, "sane_get_parameters:  res_code = %d (%2x)\n", 
	      s->resolution_code, s->resolution_code);
    } else {
      /* XXXXXXXXXXXXX */
    }
    /* What are the units/ranges on exposure and contrast? YYYY*/
    s->exposure = (s->val[OPT_BRIGHTNESS].w / 3) + 7;
    s->contrast = (s->val[OPT_CONTRAST].w / 7) + 7;
    s->velocity  = s->val[OPT_SPEED].w;
    s->shadow    = s->val[OPT_SHADOW].w;
    s->highlight = s->val[OPT_HIGHLIGHT].w;
    s->midtone   = s->val[OPT_MIDTONE].w;
    /* figure out halftone pattern selection... */
    if (s->mode == MS_MODE_HALFTONE) {
      int i = 0;
      while ((halftone_mode_list[i] != NULL) &&
	     (strcmp(halftone_mode_list[i], s->val[OPT_HALFTONE_PATTERN].s)))
	i++;
      s->pattern = ((i < s->dev->info.pattern_count) ? i : 0);
    } else
      s->pattern = 0;
      
    s->params.last_frame = SANE_TRUE;  /* ?? XXXXXXXX */
    
    {
      /* need to 'round' things properly!  XXXXXXXX */
      double widthmm, heightmm;
      double dots_per_mm = s->resolution / MM_PER_INCH;
      double units_per_mm = 
      	(s->unit_type == MS_UNIT_18INCH) ? 
	(8.0 / MM_PER_INCH) :                       /* 1/8 inches */
	(s->dev->info.base_resolution / MM_PER_INCH);   /* pixels     */
      
      DBG(23, "sane_get_parameters:  dots_per_mm:  %f\n", dots_per_mm);
      DBG(23, "sane_get_parameters:  units_per_mm:  %f\n", units_per_mm);

      /* calculate frame coordinates...
       *  scanner coords are in 'units' -- pixels or 1/8"
       *  option coords are MM
       */
      s->x1 = (SANE_Int)(SANE_UNFIX(s->val[OPT_TL_X].w) * units_per_mm + 0.5);
      s->y1 = (SANE_Int)(SANE_UNFIX(s->val[OPT_TL_Y].w) * units_per_mm + 0.5);
      s->x2 = (SANE_Int)(SANE_UNFIX(s->val[OPT_BR_X].w) * units_per_mm + 0.5);
      s->y2 = (SANE_Int)(SANE_UNFIX(s->val[OPT_BR_Y].w) * units_per_mm + 0.5);

      /* these are just an estimate...
       * real values come from scanner after sane_start.
       */
      widthmm = SANE_UNFIX(s->val[OPT_BR_X].w - s->val[OPT_TL_X].w);
      heightmm = SANE_UNFIX(s->val[OPT_BR_Y].w - s->val[OPT_TL_Y].w);
      s->params.pixels_per_line = (SANE_Int)(widthmm * dots_per_mm + 0.5);
      s->params.lines = (SANE_Int)(heightmm * dots_per_mm + 0.5);
    }

    switch (s->mode) {
    case MS_MODE_LINEART:
    case MS_MODE_HALFTONE:
      s->multibit = SANE_FALSE;
      s->params.format = SANE_FRAME_GRAY;
      s->params.bytes_per_line = (s->params.pixels_per_line + 7) / 8;
      s->params.depth = 1;
      break;
    case MS_MODE_GRAY:
      s->multibit = SANE_TRUE;
      s->params.format = SANE_FRAME_GRAY;
      s->params.bytes_per_line = s->params.pixels_per_line;
      s->params.depth = 8;
      break;
    case MS_MODE_COLOR:
      s->multibit = SANE_TRUE;
      /* it's one of the color modes... assume one-pass today. XXXXXX */
      s->params.format = SANE_FRAME_RGB;
      s->params.bytes_per_line = s->params.pixels_per_line * 3;
      s->params.depth = 8;
      break;
    }

  } else {
    /*return SANE_STATUS_INVAL;*/
  }

  if (params)
    *params = s->params;

  return SANE_STATUS_GOOD;
}



/********************************************************************/
/********************************************************************/
/*** Basic SCSI Commands ********************************************/
/********************************************************************/
/********************************************************************/


static SANE_Status
wait_ready(Microtek_Scanner *ms)
{
  u_int8_t comm[6] = { 0, 0, 0, 0, 0, 0 };
  SANE_Status status;
  int retry = 0;

  DBG(23, ".wait_ready %d\n", ms->sfd);
  while ((status = sanei_scsi_cmd(ms->sfd, comm, 6, 0, 0)) 
	 != SANE_STATUS_GOOD) {
    if (retry > 5) return SANE_STATUS_INVAL; /* XXXXXXXX */
    retry++;
    sleep(3);
  }
  return SANE_STATUS_GOOD;
}
  

static SANE_Status
scanning_frame(Microtek_Scanner *ms)
{
  u_int8_t *data, comm[15] = { 0x04, 0, 0, 0, 0x09, 0 };
  int x1, y1, x2, y2;

  if (ms->x1 > ms->x2) {
    x1 = ms->x2;
    x2 = ms->x1;
  } else {
    x1 = ms->x1;
    x2 = ms->x2;
  }
  if (ms->y1 > ms->y2) {
    y1 = ms->y2;
    y2 = ms->y1;
  } else {
    y1 = ms->y1;
    y2 = ms->y2;
  }

  /* E6 weirdness XXXXXXXXXX */
  if (ms->unit_type == MS_UNIT_18INCH) {
    x1 /= 2;
    x2 /= 2;
    y1 /= 2;
    y2 /= 2;
  }

  DBG(23, ".scanning_frame %d\n", ms->sfd);
  DBG(23, ".. %d,%d  %d,%d\n", x1, y1, x2, y2);
  data = comm + 6;
  data[0] = 
    ((ms->unit_type == MS_UNIT_PIXELS) ? 0x08 : 0 ) |
    ((ms->mode == MS_MODE_HALFTONE) ? 0x01 : 0 );
  data[1] = x1 & 0xFF;
  data[2] = (x1 >> 8) & 0xFF;
  data[3] = y1 & 0xFF;
  data[4] = (y1 >> 8) & 0xFF;
  data[5] = x2 & 0xFF;
  data[6] = (x2 >> 8) & 0xFF;
  data[7] = y2 & 0xFF;
  data[8] = (y2 >> 8) & 0xFF;
  /* FIX_FRAMESIZE ????  XXXXXXXXXXXX */
  /* also, check bounds on x,y vs. max x,y XXXXXX */
  {
    int i;  
    DBG(23, "SF:");
    for (i=0;i<6+0x09;i++) DBG(23, "%2x ", comm[i]);
    DBG(23, "\n");
  }
  return sanei_scsi_cmd(ms->sfd, comm, 6 + 0x09, 0, 0);
}



static SANE_Status
mode_select(Microtek_Scanner *ms)
{
  u_int8_t *data, comm[19] = { 0x15, 0, 0, 0, 0, 0 };

  DBG(23, ".mode_select %d\n", ms->sfd);
  data = comm + 6; 
  data[0] = 
    0x81 |
    ((ms->unit_type == MS_UNIT_18INCH) ? 0 : 0x08) |
    ((ms->res_type == MS_RES_5PER) ? 0 : 0x02);
  data[1] = ms->resolution_code;
  data[2] = ms->exposure;
  data[3] = ms->contrast;
  data[4] = ms->pattern;
  data[5] = ms->velocity;
  data[6] = ms->shadow;
  data[7] = ms->highlight;
DBG(23, "pap_len: %d\n", ms->paper_length);
  data[8] = ms->paper_length & 0xFF;
  data[9] = (ms->paper_length >> 8) & 0xFF;
  data[10] = ms->midtone;
  /* set command/data length */
  comm[4] = (ms->midtone_support) ? 0x0B : 0x0A;

  {
    int i;  
    DBG(23, "MSL:");
    for (i=0;i<6+comm[4];i++) DBG(23, "%2x ", comm[i]);
    DBG(23, "\n");
  }
  return sanei_scsi_cmd(ms->sfd, comm, 6 + comm[4], 0, 0); 
  /* check sense?? XXXXXX */
}


static SANE_Status
mode_select_1(Microtek_Scanner *ms)
{
  u_int8_t *data, comm[16] = { 0x16, 0, 0, 0, 0x0A, 0,
                               0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

  DBG(23, ".mode_select_1 %d\n", ms->sfd);
  data = comm + 6;
  data[1] = ms->bright_r;
  data[3] = ((ms->allow_calibrate) ? 0 : 0x02);

  {
    int i;  
    DBG(23, "MSL1:");
    for (i=0;i<6+0x0A;i++) DBG(23, "%2x ", comm[i]);
    DBG(23, "\n");
  }
  return sanei_scsi_cmd(ms->sfd, comm, 6 + 0x0A, 0, 0);
}


static SANE_Status
mode_sense(Microtek_Scanner *ms)
{
  u_int8_t data[20], comm[6] = { 0x1A, 0, 0, 0, 0, 0};
  size_t lenp;
  SANE_Status status;
  int i;

  DBG(23, ".mode_sense %d\n", ms->sfd);
  if (ms->onepass) {
    comm[4] = 0x13;
  } else if (ms->midtone_support) {
    comm[4] = 0x0B;
  } else {
    comm[4] = 0x0A;
  }
  lenp = comm[4];

  for (i=0; i<20; i++) data[i] = 0xAB;
  status = sanei_scsi_cmd(ms->sfd, comm, 6, data, &lenp);
  for (i=0; i<lenp; i++)
    DBG(23, "%2x ", data[i]);
  DBG(23, "\n");

  return status;
}
  
  
static SANE_Status
mode_sense_1(Microtek_Scanner *ms)
{
  u_int8_t *data, comm[36] = { 0x19, 0, 0, 0, 0x1E, 0 };

  DBG(23, ".mode_sense_1\n");
  data = comm + 6;
  memset(data, 0, 30);
  data[1] = ms->bright_r;
  data[2] = ms->bright_g;
  data[3] = ms->bright_b;

  {
    int i;  
    DBG(23, "MS1:");
    for (i=0;i<6+0x1E;i++) DBG(23, "%2x ", comm[i]);
    DBG(23, "\n");
  }
  return sanei_scsi_cmd(ms->sfd, comm, 6 + 0x1E, 0, 0);
}



static SANE_Status
accessory(Microtek_Scanner *ms)
{
  u_int8_t comm[6] = { 0x10, 0, 0, 0, 0, 0 };

  DBG(23, ".accessory\n");
  comm[4] = 
    ((ms->useADF) ? 0x41 : 0) |
    ((ms->prescan) ? 0x18 : 0) |
    ((ms->transparency) ? 0x24 : 0) |
    ((ms->allowbacktrack) ? 0x81 : 0);

  {
    int i;  
    DBG(23, "AC:");
    for (i=0;i<6;i++) DBG(23, "%2x ", comm[i]);
    DBG(23, "\n");
  }
  return sanei_scsi_cmd(ms->sfd, comm, 6, 0, 0);
}



static SANE_Status
start_scan(Microtek_Scanner *ms)
{
  u_int8_t comm[6] = { 0x1B, 0, 0, 0, 0, 0 };

  DBG(23, ".start_scan\n");
  comm[4] = 
    0x01 |
    ((ms->expandedresolution) ? 0x80 : 0) |
    ((ms->multibit) ? 0x40 : 0) |
    ((ms->onepasscolor) ? 0x20 : 0) |
    ((ms->reversecolors) ? 0x04 : 0) |
    ((ms->fastprescan) ? 0x02 : 0) |
    ((ms->filter == MS_FILT_RED) ? 0x08 : 0) |
    ((ms->filter == MS_FILT_GREEN) ? 0x10 : 0) |
    ((ms->filter == MS_FILT_BLUE) ? 0x18 : 0) ;

  {
    int i;  
    DBG(23, "SS:");
    for (i=0;i<6;i++) DBG(23, "%2x ", comm[i]);
    DBG(23, "\n");
  }
  return sanei_scsi_cmd(ms->sfd, comm, 6, 0, 0);
}


static SANE_Status
stop_scan(Microtek_Scanner *ms)
{
  u_int8_t comm[6] = { 0x1B, 0, 0, 0, 0, 0 };

  DBG(23, ".stop_scan\n");
  {
    int i;  
    DBG(23, "SPS:");
    for (i=0;i<6;i++) DBG(23, "%2x ", comm[i]);
    DBG(23, "\n");
  }
  return sanei_scsi_cmd(ms->sfd, comm, 6, 0, 0);
  /* do some wait crap?  XXXXXXXXXXXX */
}

  
static SANE_Status
get_scan_status(Microtek_Scanner *ms,
		int *busy,
		SANE_Int *bytes_per_line,
		SANE_Int *lines)
{
  u_int8_t data[6], comm[6] = { 0x0F, 0, 0, 0, 0x06, 0 };
  SANE_Status status;
  size_t lenp;
  int retry = 0;

  DBG(23, ".get_scan_status %d\n", ms->sfd);

  do {
    lenp = 6;
    /* do some retry stuff in here too XXXXXXXXXXXXX */
    status = sanei_scsi_cmd(ms->sfd, comm, 6, data, &lenp);
    *busy = data[0];
    *bytes_per_line = (data[1]) + (data[2] << 8);
    *lines = (data[3]) + (data[4] << 8) + (data[5] << 16);
    
    DBG(23, "gss(%lu): %d, %d, %d\n",
	(u_long) lenp, *busy, *bytes_per_line, *lines);
    DBG(23, "> %2x %2x %2x %2x %2x %2x\n",
	    data[0], data[1], data[2], data[3], data[4], data[5]);
    if (*busy != 0) {
      retry++;
      sleep(1);
    }
  } while ((*busy != 0) && (retry < 10));
    
  return status;
}


static SANE_Status
read_scan_data(Microtek_Scanner *ms,
	       int lines,
	       u_int8_t *buffer, 
	       size_t *bufsize)
{
  u_int8_t comm[6] = { 0x08, 0, 0, 0, 0, 0 };

  DBG(23, ".read_scan_data\n");
  comm[2] = (lines >> 16) & 0xFF;
  comm[3] = (lines >> 8) & 0xFF;
  comm[4] = (lines) & 0xFF;
  
  return sanei_scsi_cmd(ms->sfd, comm, 6, buffer, bufsize);
}
  
  
static SANE_Status
download_gamma(Microtek_Scanner *ms)
{
  u_int8_t *data, comm[266] = { 0x55, 0, 0x27, 0, 0,
				   0, 0,    0, 0, 0 };
  int i, pl;
  SANE_Status status;

  DBG(23, ".download_gamma\n");
  data = comm + 10;
  
  comm[7] = 1;
  comm[8] = 0;
  comm[9] = 0;

  if (ms->val[OPT_CUSTOM_GAMMA].w == SANE_TRUE) {
    if (ms->val[OPT_GAMMA_BIND].w == SANE_TRUE) {
      for (i=0; i<256; i++) 
	data[i] = (u_int8_t) ms->gamma_table[0][i];
      return sanei_scsi_cmd(ms->sfd, comm, 266, 0, 0);
    } else {
      pl = 1;
      do {
      for (i=0; i<256; i++) 
	data[i] = (u_int8_t) ms->gamma_table[pl][i];
      comm[9] = pl << 6;
      status = sanei_scsi_cmd(ms->sfd, comm, 266, 0, 0);
      pl++;
      } while ((pl < 4) && (status == SANE_STATUS_GOOD));
      return status;
    }
  } else {  /* no custom gamma tables */
    for (i=0; i<256; i++) 
      data[i] = (u_int8_t) i;
    return sanei_scsi_cmd(ms->sfd, comm, 266, 0, 0);
  }
}



/*********************************************************************/



SANE_Status
sane_start (SANE_Handle handle)
{
  Microtek_Scanner *s = handle;
  SANE_Status status;

  
  DBG(10, "sane_start...\n");

  status = sane_get_parameters(s, 0);
  if (status != SANE_STATUS_GOOD) return status;

  if (s->sfd < 0) {

    int busy;
    SANE_Int linewidth;

    status = sanei_scsi_open(s->dev->sane.name,
			     &(s->sfd),
			     sense_handler);
    if (status != SANE_STATUS_GOOD) {
      DBG(1, "open: open of %s failed: %s\n",
	  s->dev->sane.name, sane_strstatus (status));
      return status;
    }

    /*** allocate xferbuffer? XXXXXXX ***/
    status = wait_ready(s);
    if (status != SANE_STATUS_GOOD) {
      DBG(1, "open: wait_ready() failed: %s\n", sane_strstatus (status));
      /*      goto stop_scanner_and_return;*/
      DBG(23, "YIKESSSS\n");  /* XXXXXXXXXX */
    }

    if (s->dev->info.does_mode1) {
      status = mode_select_1(s);
      if (status != SANE_STATUS_GOOD) 
	DBG(23, "OY!");
      status = mode_sense_1(s);
      if (status != SANE_STATUS_GOOD) 
	DBG(23, "OY!");
    }

    if (s->dev->info.source_options & MI_SRC_FEED_BT) {
      status = accessory(s);   /* if SWslct ????  XXXXXXXXXXXXXXX */
      if (status != SANE_STATUS_GOOD) 
	DBG(23, "OY!");
    }

    status = mode_sense(s);
    if (status != SANE_STATUS_GOOD) 
      DBG(23, "OY!");

    status = scanning_frame(s);
    if (status != SANE_STATUS_GOOD) 
      DBG(23, "OY!");
    /*** dwnld gamma tables? XXXXXXXX ***/

    status = download_gamma(s);
    if (status != SANE_STATUS_GOOD) 
      DBG(23, "OY!");

    status = mode_select(s);
    if (status != SANE_STATUS_GOOD) 
      DBG(23, "OY!");
    status = mode_sense(s);
    if (status != SANE_STATUS_GOOD) 
      DBG(23, "OY!");
    status = start_scan(s);
    if (status != SANE_STATUS_GOOD) 
      DBG(23, "OY!");

    status = get_scan_status(s, &busy, &linewidth, &(s->lines));
    if (status != SANE_STATUS_GOOD) 
      DBG(23, "OY!");
    /*** check bpl? XXXXXXXXX ***/
    
    switch (s->mode) {
    case MS_MODE_LINEART:
    case MS_MODE_HALFTONE:
      s->pixel_bpl = linewidth;
      s->header_bpl = 0;
      s->ppl = linewidth * 8;
      s->planes = 1;
      break;
    case MS_MODE_GRAY:
      if (s->bits_per_color < 8) {
	s->pixel_bpl = linewidth;
	s->ppl = linewidth * (8 / s->bits_per_color);
      }	else {
	s->pixel_bpl = linewidth * ((s->bits_per_color + 7) / 8);
	s->ppl = linewidth;
      }
      s->header_bpl = 0;
      s->planes = 1;
      break;
    case MS_MODE_COLOR:
      switch (s->dev->info.color_sequence) {
      case MI_COLSEQ_PLANE:
	s->pixel_bpl = linewidth * ((s->bits_per_color + 7) / 8);
	s->ppl = linewidth;
	s->header_bpl = 0;
	s->planes = 1;
	break;
      case MI_COLSEQ_NONRGB:
	s->pixel_bpl = (linewidth - 2) * 3 * ((s->bits_per_color + 7) / 8);
	s->ppl = linewidth - 2;
	s->header_bpl = 2 * 3;
	s->planes = 3;
	break;
      default:
	s->pixel_bpl = linewidth * 3 * ((s->bits_per_color + 7) / 8);
	s->ppl = linewidth;
	s->header_bpl = 0;
	s->planes = 3;
	break;
      }
      break;
    default:
      /*** XXXXXXXXXXX ***/
    }

    s->params.lines = s->lines;
    s->params.pixels_per_line = s->ppl;
    s->params.bytes_per_line = s->pixel_bpl;

    s->buffer = (u_int8_t *) malloc(SCSI_BUFF_SIZE * sizeof(u_int8_t));
    if (s->buffer == NULL) return SANE_STATUS_NO_MEM;

    if ((s->mode == MS_MODE_COLOR) &&
	(s->onepass) &&
	(s->header_bpl != 0)) {
      s->pre_hold = (u_int8_t *) malloc(SCSI_BUFF_SIZE * sizeof(u_int8_t));
      if (s->pre_hold == NULL) return SANE_STATUS_NO_MEM;
      s->post_hold = (u_int8_t *) malloc(SCSI_BUFF_SIZE * sizeof(u_int8_t));
      if (s->post_hold == NULL)	return SANE_STATUS_NO_MEM;
      s->pre_lines = 0;
      s->post_lines = 0;
      s->extra_r_seg = 0;
      s->extra_g_seg = 0;
      s->extra_b_seg = 0;
    } else {
      s->pre_hold = NULL;
      s->post_hold = NULL;
    }

    s->scanning = SANE_TRUE;
    s->cancel = SANE_FALSE;

    DBG(23, "Scan Param:\n");
    DBG(23, "pix bpl: %d    hdr bpl: %d   ppl: %d\n",
	    s->pixel_bpl, s->header_bpl, s->ppl);
    DBG(23, "lines: %d   planes: %d\n",
	    s->lines, s->planes);

  } else {
    return SANE_STATUS_INVAL;
  }

  return SANE_STATUS_GOOD;
}


SANE_Status
sane_read (SANE_Handle handle, SANE_Byte *data,
	   SANE_Int max_length, SANE_Int *length)
{
  Microtek_Scanner *s = handle;
  SANE_Status status;
  int busy;
  SANE_Int linewidth;
  int nlines, remaining;
  size_t buffsize;
  int max_data_lines;

  DBG(23, "sane_read.....\n");

  if (!(s->scanning))
    return SANE_STATUS_INVAL;

  if ((s->lines <= 0) || (s->cancel)) {      /* we are done scanning here */
    status = stop_scan(s);
    if (status != SANE_STATUS_GOOD) 
      DBG(23, "OY!");
    s->scanning = SANE_FALSE;
    /* free the buffers we malloc'ed */
    free(s->buffer);
    s->buffer = NULL;
    free(s->pre_hold);
    s->pre_hold = NULL;
    free(s->post_hold);
    s->post_hold = NULL;
    /* close the SCSI device */
    sanei_scsi_close(s->sfd);
    s->sfd = -1;
    if (s->cancel)
      return SANE_STATUS_CANCELLED;
    else
      return SANE_STATUS_EOF;
  }

  status = get_scan_status(s, &busy, &linewidth, &remaining);
  if (status != SANE_STATUS_GOOD) 
    DBG(23, "OY!");
  
  DBG(23, "read l,r:  %d, %d, %d\n", busy, linewidth, remaining);

  max_data_lines = max_length / s->pixel_bpl; /* max line cap. of dest */

  /* perhaps nlines should also be a function of max_length XXXXXXX */
  if (remaining * (s->pixel_bpl + s->header_bpl) > SCSI_BUFF_SIZE)
    nlines = SCSI_BUFF_SIZE / (s->pixel_bpl + s->header_bpl);
  else
    nlines = remaining;
  if (nlines > max_data_lines)
    nlines = max_data_lines;
  DBG(23, "sane_read:  maxdl: %d, rem: %d, nlines: %d\n",
      max_data_lines, remaining, nlines);

  /* what about just using linewidth?  XXXXXXXXXXXXX */
  buffsize = nlines * (s->pixel_bpl + s->header_bpl);
  status = read_scan_data (s, nlines, s->buffer, &buffsize);
  if (status != SANE_STATUS_GOOD) 
    DBG(23, "OY!");
  
  s->lines -= nlines;

  DBG(23, "max_len: %d,  buffsize: %lu\n", max_length, (u_long) buffsize);

  /* what about max_length??  XXXXXXXXx */
  if ((s->mode == MS_MODE_LINEART) ||
      (s->mode == MS_MODE_HALFTONE) ||
      (s->mode == MS_MODE_GRAY) ||        /* or onepasscolor? XXXX */
      ((s->mode == MS_MODE_COLOR) && !(s->onepass))) { 
    memcpy(data, s->buffer, buffsize);
    *length = buffsize;
  } else { /* assume onepass COLOR ?? XXXXXXXXXx */

    if (s->header_bpl == 0) {
      SANE_Byte *r, *g, *b;
      SANE_Byte *d = data;
      int i, line;
      SANE_Byte *buffo = s->buffer;

      for (line=0; line<nlines; line++) {
	r = buffo;
	g = r + linewidth;
	b = g + linewidth;
      /* will data buff be big enough? XXXXXXXXx */
	for (i=0; i < s->ppl; i++) {
	  *(d++) = *(r++);
	  *(d++) = *(g++);
	  *(d++) = *(b++);
	}
	buffo += s->pixel_bpl + s->header_bpl;
      }
      *length = 3 * s->ppl * nlines;
    } else {  /* if that funky headered format */
      int seg, i;
      SANE_Byte *d;
      SANE_Byte *buffo = s->buffer;
      int r_seg, g_seg, b_seg;
      int max_dest_lines, lines_to_go;
      int complete, incomplete;
      int doffset = 0, sss = 0; /*XXXXX*/
      int pre_lcount, post_lcount;

#define MIN(a,b) (((a) < (b)) ? (a) : (b))      /* XXXXXXXXX */
#define MAX(a,b) (((a) > (b)) ? (a) : (b))      /* XXXXXXXXX */

      max_dest_lines = max_length / (3 * s->ppl); /* max line cap. of dest */
      lines_to_go = MIN(max_dest_lines, nlines);
      DBG(23, "sr:  max_length: %d  maxdl: %d  nlines: %d  ltg: %d\n",
	      max_length, max_dest_lines, nlines, lines_to_go);
      
      if (s->pre_lines > 0)
	memcpy(data, s->pre_hold, s->pre_lines * s->ppl * 3);
      if (s->post_lines > 0)
	memcpy(data + (s->pre_lines * s->ppl * 3), 
	       s->post_hold, s->post_lines * s->ppl * 3);
      
      /*      if (s->holding_lines != 0) {
	memcpy(data, s->holding, s->holding_lines * s->ppl * 3); */
      r_seg = s->extra_r_seg;
      g_seg = s->extra_g_seg;
      b_seg = s->extra_b_seg;
	/*      } else {
	r_seg = g_seg = b_seg = 0;
      }
      */
      for (seg=0; seg < nlines * 3; seg++) {
	buffo++;
	switch (*buffo) {
	case 'R': doffset = 0; sss = r_seg; r_seg++; break;
	case 'G': doffset = 1; sss = g_seg; g_seg++; break;
      	case 'B': doffset = 2; sss = b_seg; b_seg++; break;
	default:
	  DBG(23, "nonrgb: none!\n");
	}

	if (sss < /*lines_to_go*/ max_dest_lines) {
	  d = data + (sss * s->ppl * 3) + doffset;
	} else {
	  d = s->post_hold + doffset + 
	    ((sss - /*lines_to_go*/ max_dest_lines) * s->ppl * 3);
	  DBG(23, "extra:  seg-%d  do-%d  sss-%d   ltg-%d  mdl-%d\n",
	      seg, doffset, sss, lines_to_go,
	      max_dest_lines);
	}
	buffo++;
	for (i=0; i<s->ppl; i++) {
	  *d = *buffo;
	  buffo++;
	  d += 3;
	}
	
      }

      complete = MIN(r_seg, MIN(g_seg, b_seg));   /* lines completed */
      pre_lcount = lines_to_go - complete;  /* incomplete lines in data buf */
      incomplete = MAX(r_seg, MAX(g_seg, b_seg)) - complete;
      post_lcount =        /* incomplete lines in post_hold */
	incomplete - pre_lcount;
	/*	MAX(r_seg, MAX(g_seg, b_seg)) - lines_to_go;*/
      *length = 3 * s->ppl * complete;
      
      if (pre_lcount > 0) 
	memcpy(s->pre_hold, data + *length, pre_lcount * 3 * s->ppl);
      
      s->extra_r_seg = r_seg - complete;
      s->extra_g_seg = g_seg - complete;
      s->extra_b_seg = b_seg - complete;
      /*      s->holding_lines = incomplete; */
      s->pre_lines = pre_lcount;
      s->post_lines = post_lcount;

      DBG(23, "extra r: %d  g: %d  b: %d  prel: %d  postl: %d\n",
	      s->extra_r_seg, s->extra_g_seg, s->extra_b_seg,
	      s->pre_lines, s->post_lines);
    }

  }

  return SANE_STATUS_GOOD;
      /*  return SANE_STATUS_INVAL;*/
}







void
sane_cancel (SANE_Handle handle)
{
  Microtek_Scanner *ms = handle;
  DBG(23, "sane_cancel...");
  ms->cancel = SANE_TRUE;
}




SANE_Status
sane_set_io_mode (SANE_Handle handle, SANE_Bool non_blocking)
{
  DBG(23, "sane_set_io_mode...\n");
  return SANE_STATUS_UNSUPPORTED;
}


SANE_Status
sane_get_select_fd (SANE_Handle handle, SANE_Int * fd)
{
  DBG(23, "sane_get_select_fd...\n");
  return SANE_STATUS_UNSUPPORTED;
}
