/* epson.c - SANE library for Epson flatbed scanners.
   Copyright (C) 1997 Hypercore Software Design, Ltd.

   This file is part of the SANE package.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.

   As a special exception, the authors of SANE give permission for
   additional uses of the libraries contained in this release of SANE.

   The exception is that, if you link a SANE library with other files
   to produce an executable, this does not by itself cause the
   resulting executable to be covered by the GNU General Public
   License.  Your use of that executable is in no way restricted on
   account of linking the SANE library code into it.

   This exception does not, however, invalidate any other reasons why
   the executable file might be covered by the GNU General Public
   License.

   If you submit changes to SANE to the maintainers to be included in
   a subsequent release, you agree by submitting the changes that
   those changes may be distributed with this exception intact.

   If you write modifications of your own for SANE, it is your choice
   whether to permit this exception to apply to your modifications.
   If you do not wish that, delete this exception notice.  */

#ifdef HAVE_CONFIG_H
# include <sane/config.h>
#endif

#include <lalloca.h>

#include <stdio.h>
#ifdef STDC_HEADERS
# include <string.h>
# include <stdlib.h>
#endif

#ifndef FILENAME_MAX
# define FILENAME_MAX 1024
#endif

#include <sane/sane.h>
#include <sane/saneopts.h>
#include <sane/sanei_scsi.h>
#include "epson.h"

#define BACKEND_NAME	epson
#include <sane/sanei_backend.h>

#ifndef V_MAJOR
# define V_MAJOR 0
# ifndef V_MINOR
#  define V_MINOR 64
# endif
#endif /* not V_MAJOR */

#define TEST_UNIT_READY_COMMAND 0x00
#define READ_6_COMMAND 0x08
#define WRITE_6_COMMAND 0x0a
#define INQUIRY_COMMAND 0x12
#define TYPE_PROCESSOR 0x03

#define STX 0x02
#define ACK 0x06
#define NAK 0x15

static SANE_Status
send (int fd, const void *buf, size_t buf_size)
{
  unsigned char *cmd;
  int status;

  cmd = alloca (6 + buf_size);
  memset (cmd, 0, 6);
  cmd[0] = WRITE_6_COMMAND;
  cmd[2] = buf_size >> 16;
  cmd[3] = buf_size >> 8;
  cmd[4] = buf_size;
  memcpy (cmd + 6, buf, buf_size);
  status = sanei_scsi_cmd (fd, cmd, 6 + buf_size, NULL, NULL);
  return status;
}

static SANE_Status
receive (int fd, void *buf, size_t *buf_size)
{
  unsigned char cmd[6];
  SANE_Status status;

  memset (cmd, 0, 6);
  cmd[0] = READ_6_COMMAND;
  cmd[2] = *buf_size >> 16;
  cmd[3] = *buf_size >> 8;
  cmd[4] = *buf_size;
  status = sanei_scsi_cmd (fd, cmd, sizeof cmd, buf, buf_size);
  return status;
}

static SANE_Status
inquiry (int fd, int page_code, void *buf, size_t *buf_size)
{
  unsigned char cmd[6];
  int status;

  memset (cmd, 0, 6);
  cmd[0] = INQUIRY_COMMAND;
  cmd[2] = page_code;
  cmd[4] = *buf_size > 255 ? 255 : *buf_size;
  status = sanei_scsi_cmd (fd, cmd, sizeof cmd, buf, buf_size);
  return status;
}

static SANE_Status
expect_ack (int fd)
{
  unsigned char result[1];
  size_t len;
  SANE_Status status;

  len = sizeof result;
  status = receive (fd, result, &len);
  if (status != SANE_STATUS_GOOD)
    return status;

  if (result[0] != ACK)
    return SANE_STATUS_INVAL;

  return SANE_STATUS_GOOD;
}

static SANE_Status
set_mode (int fd, int mode)
{
  SANE_Status status;
  unsigned char params[1];

  send (fd, "\033C", 2);
  status = expect_ack (fd);
  if (status != SANE_STATUS_GOOD)
    return status;
  params[0] = mode;
  send (fd, params, 1);
  status = expect_ack (fd);
  return status;
}

static SANE_Status
set_resolution (int fd, int xres, int yres)
{
  SANE_Status status;
  unsigned char params[4];

  send (fd, "\033R", 2);
  status = expect_ack (fd);
  if (status != SANE_STATUS_GOOD)
    return status;
  params[0] = xres;
  params[1] = xres >> 8;
  params[2] = yres;
  params[3] = yres >> 8;
  send (fd, params, 4);
  status = expect_ack (fd);
  return status;
}

static SANE_Status
set_area (int fd, int x, int y, int width, int height)
{
  SANE_Status status;
  unsigned char params[8];

  send (fd, "\033A", 2);
  status = expect_ack (fd);
  if (status != SANE_STATUS_GOOD)
    return status;
  params[0] = x;
  params[1] = x >> 8;
  params[2] = y;
  params[3] = y >> 8;
  params[4] = width;
  params[5] = width >> 8;
  params[6] = height;
  params[7] = height >> 8;
  send (fd, params, 8);
  status = expect_ack (fd);
  return status;
}

static SANE_Status
set_depth (int fd, int depth)
{
  SANE_Status status;
  unsigned char params[1];

  send (fd, "\033D", 2);
  status = expect_ack (fd);
  if (status != SANE_STATUS_GOOD)
    return status;
  params[0] = depth;
  send (fd, params, 1);
  status = expect_ack (fd);
  return status;
}

static SANE_Status
set_halftone (int fd, int halftone)
{
  SANE_Status status;
  unsigned char params[1];

  send (fd, "\033B", 2);
  status = expect_ack (fd);
  if (status != SANE_STATUS_GOOD)
    return status;
  params[0] = halftone;
  send (fd, params, 1);
  status = expect_ack (fd);
  return status;
}

static SANE_Status
set_gamma (int fd, int gamma)
{
  SANE_Status status;
  unsigned char params[1];

  send (fd, "\033Z", 2);
  status = expect_ack (fd);
  if (status != SANE_STATUS_GOOD)
    return status;
  params[0] = gamma;
  send (fd, params, 1);
  status = expect_ack (fd);
  return status;
}

static SANE_Status
set_color (int fd, int color)
{
  SANE_Status status;
  unsigned char params[1];

  send (fd, "\033M", 2);
  status = expect_ack (fd);
  if (status != SANE_STATUS_GOOD)
    return status;
  params[0] = color;
  send (fd, params, 1);
  status = expect_ack (fd);
  return status;
}

static SANE_Status
set_speed (int fd, int flags)
{
  SANE_Status status;
  unsigned char params[1];

  send (fd, "\033g", 2);
  status = expect_ack (fd);
  if (status != SANE_STATUS_GOOD)
    return status;
  params[0] = flags;
  send (fd, params, 1);
  status = expect_ack (fd);
  return status;
}

static SANE_Status
set_lcount (int fd, int lcount)
{
  SANE_Status status;
  unsigned char params[1];

  send (fd, "\033d", 2);
  status = expect_ack (fd);
  if (status != SANE_STATUS_GOOD)
    return status;
  params[0] = lcount;
  send (fd, params, 1);
  status = expect_ack (fd);
  return status;
}

static SANE_Status
reset (int fd)
{
  SANE_Status status;

  send (fd, "\033@", 2);
  status = expect_ack (fd);
  return status;
}

static SANE_Status
identify (int fd, struct Epson_Device *dev)
{
  unsigned char result[4];
  size_t len;
  SANE_Status status;
  unsigned char *buf;

  status = send (fd, "\033I", 2);
  if (status != SANE_STATUS_GOOD)
    return status;
  len = 4;
  status = receive (fd, result, &len);
  if (status != SANE_STATUS_GOOD)
    return status;
  if (result[0] != STX)
    return SANE_STATUS_INVAL;

  len = result[3] << 8 | result[2];
  buf = alloca (len);
  receive (fd, buf, &len);
  if (buf[2] != 'R' || buf[len - 5] != 'A')
    return SANE_STATUS_INVAL;

  if (buf[0] != 'B')
    DBG (1, "Unknown type %c\n", buf[0]);
  dev->level = buf[1] - '0';
  DBG (2, "Command level is %d\n", dev->level);
  dev->dpi_range.min = buf[4] << 8 | buf[3];
  dev->dpi_range.max = buf[len - 6] << 8 | buf[len - 7];
  dev->dpi_range.quant = 0;
  dev->x_range.min = 0;
  dev->x_range.max = SANE_FIX ((buf[len - 3] << 8 | buf[len - 4]) * 25.4 / dev->dpi_range.max);
  dev->x_range.quant = 0;
  dev->y_range.min = 0;
  dev->y_range.max = SANE_FIX ((buf[len - 1] << 8 | buf[len - 2]) * 25.4 / dev->dpi_range.max);
  dev->y_range.quant = 0;
  return SANE_STATUS_GOOD;
}

static Epson_Device dummy_dev =
{
  {
    NULL,
    "Epson",
    NULL,
    "flatbed scanner"
  }
};

static SANE_Status
attach (const char *dev_name, Epson_Device **devp)
{
  SANE_Status status;
  int fd;
  unsigned char buf[36];
  size_t buf_size;
  char *str;

  DBG (3, "attach: opening %s\n", dev_name);
  status = sanei_scsi_open (dev_name, &fd, NULL);
  if (status != SANE_STATUS_GOOD)
    {
      DBG (1, "attach: open failed: %s\n", sane_strstatus (status));
      return status;
    }

  DBG (3, "attach: sending INQUIRY\n");
  buf_size = sizeof buf;
  status = inquiry (fd, 0, buf, &buf_size);
  if (status != SANE_STATUS_GOOD)
    {
      DBG (1, "attach: inquiry failed: %s\n", sane_strstatus (status));
      sanei_scsi_close (fd);
      return status;
    }

  if (buf[0] != TYPE_PROCESSOR
      || strncmp (buf + 8, "EPSON", 5) != 0
      || strncmp (buf + 16, "SCANNER ", 8) != 0)
    {
      DBG (1, "attach: device doesn't look like an Epson scanner\n");
      sanei_scsi_close (fd);
      return SANE_STATUS_INVAL;
    }

  status = identify (fd, &dummy_dev);
  if (status != SANE_STATUS_GOOD)
    {
      DBG (1, "attach: device doesn't look like an Epson scanner\n");
      sanei_scsi_close (fd);
      return SANE_STATUS_INVAL;
    }

  sanei_scsi_close (fd);

  str = malloc (strlen (dev_name) + 1);
  dummy_dev.sane.name = strcpy (str, dev_name);

  str = malloc (8 + 1);
  str[8] = '\0';
  dummy_dev.sane.model = (char *) memcpy (str, buf + 16 + 8, 8);

  return SANE_STATUS_GOOD;
}

SANE_Status
sane_init (SANE_Int *version_code, SANE_Auth_Callback authorize)
{
  char dev_name[FILENAME_MAX] = "/dev/scanner"; /* FIXME */

  DBG_INIT();
#if defined PACKAGE && defined VERSION
  DBG (2, "sane_init: " PACKAGE " " VERSION "\n");
#endif

  if (version_code != NULL)
    *version_code = SANE_VERSION_CODE (V_MAJOR, V_MINOR, 0);

  attach (dev_name, NULL);

  return SANE_STATUS_GOOD;
}

void
sane_exit (void)
{
  free ((char *) dummy_dev.sane.model);
  free ((char *) dummy_dev.sane.name);
}

SANE_Status
sane_get_devices (const SANE_Device ***device_list, SANE_Bool local_only)
{
  static const SANE_Device *devlist[2];
  int i;

  i = 0;
  if (dummy_dev.sane.name != NULL)
    devlist[i++] = &dummy_dev.sane;
  devlist[i] = NULL;

  *device_list = devlist;
  return SANE_STATUS_GOOD;
}

struct mode_param
{
  int color;
  int mode_flags;
  int dropout_mask;
  int depth;
};

static const struct mode_param mode_params[] =
{
  {0, 0x00, 0x30, 1},
  {0, 0x00, 0x30, 8},
  {1, 0x02, 0x00, 8},
};

static const SANE_String_Const mode_list[] =
{
  "Binary",
  "Gray",
  "Color",
  NULL
};

static int halftone_params[] =
{
  0x01,
  0x00,
  0x10,
  0x20,
  0x80,
  0x90,
  0xa0,
  0xb0
};

static const SANE_String_Const halftone_list[] =
{
  "None",
  "Halftone A",
  "Halftone B",
  "Halftone C",
  NULL
};

static const SANE_String_Const halftone_list_4[] =
{
  "None",
  "Halftone A",
  "Halftone B",
  "Halftone C",
  "Dither A",
  "Dither B",
  "Dither C",
  "Dither D",
  NULL
};

static int dropout_params[] =
{
  0x00,
  0x10,
  0x20,
  0x30,
};

static const SANE_String_Const dropout_list[] =
{
  "None",
  "Red",
  "Green",
  "Blue",
  NULL
};

static SANE_Status
init_options (Epson_Scanner *s)
{
  int i;

  for (i = 0; i < NUM_OPTIONS; ++i)
    {
      s->opt[i].size = sizeof (SANE_Word);
      s->opt[i].cap = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
    }

  s->opt[OPT_NUM_OPTS].title = SANE_TITLE_NUM_OPTIONS;
  s->opt[OPT_NUM_OPTS].desc = SANE_DESC_NUM_OPTIONS;
  s->opt[OPT_NUM_OPTS].cap = SANE_CAP_SOFT_DETECT;
  s->val[OPT_NUM_OPTS] = NUM_OPTIONS;

  /* "Scan Mode" group: */

  s->opt[OPT_MODE_GROUP].title = "Scan Mode";
  s->opt[OPT_MODE_GROUP].desc = "";
  s->opt[OPT_MODE_GROUP].type = SANE_TYPE_GROUP;
  s->opt[OPT_MODE_GROUP].cap = 0;

  /* scan mode */
  s->opt[OPT_MODE].name = SANE_NAME_SCAN_MODE;
  s->opt[OPT_MODE].title = SANE_TITLE_SCAN_MODE;
  s->opt[OPT_MODE].desc = SANE_DESC_SCAN_MODE;
  s->opt[OPT_MODE].type = SANE_TYPE_STRING;
  s->opt[OPT_MODE].size = 32;
  s->opt[OPT_MODE].constraint_type = SANE_CONSTRAINT_STRING_LIST;
  s->opt[OPT_MODE].constraint.string_list = mode_list;
  s->val[OPT_MODE] = 0;		/* Binary */

  /* halftone */
  s->opt[OPT_HALFTONE].name = SANE_NAME_HALFTONE;
  s->opt[OPT_HALFTONE].title = SANE_TITLE_HALFTONE;
  s->opt[OPT_HALFTONE].desc = "Selects the halftone.";
  s->opt[OPT_HALFTONE].type = SANE_TYPE_STRING;
  s->opt[OPT_HALFTONE].size = 32;
  s->opt[OPT_HALFTONE].constraint_type = SANE_CONSTRAINT_STRING_LIST;
  if (s->hw->level >= 4)
    s->opt[OPT_HALFTONE].constraint.string_list = halftone_list_4;
  else
    s->opt[OPT_HALFTONE].constraint.string_list = halftone_list;
  s->val[OPT_HALFTONE] = 1;	/* Halftone A */

  /* dropout */
  s->opt[OPT_DROPOUT].name = "dropout";
  s->opt[OPT_DROPOUT].title = "Dropout";
  s->opt[OPT_DROPOUT].desc = "Selects the dropout.";
  s->opt[OPT_DROPOUT].type = SANE_TYPE_STRING;
  s->opt[OPT_DROPOUT].size = 32;
  s->opt[OPT_DROPOUT].cap |= SANE_CAP_ADVANCED;
  s->opt[OPT_DROPOUT].constraint_type = SANE_CONSTRAINT_STRING_LIST;
  s->opt[OPT_DROPOUT].constraint.string_list = dropout_list;
  s->val[OPT_DROPOUT] = 0;	/* None */

  /* resolution */
  s->opt[OPT_RESOLUTION].name = SANE_NAME_SCAN_RESOLUTION;
  s->opt[OPT_RESOLUTION].title = SANE_TITLE_SCAN_RESOLUTION;
  s->opt[OPT_RESOLUTION].desc = SANE_DESC_SCAN_RESOLUTION;
  s->opt[OPT_RESOLUTION].type = SANE_TYPE_INT;
  s->opt[OPT_RESOLUTION].unit = SANE_UNIT_DPI;
  s->opt[OPT_RESOLUTION].constraint_type = SANE_CONSTRAINT_RANGE;
  s->opt[OPT_RESOLUTION].constraint.range = &s->hw->dpi_range;
  s->val[OPT_RESOLUTION] = s->hw->dpi_range.min;

  /* "Geometry" group: */

  s->opt[OPT_GEOMETRY_GROUP].title = "Geometry";
  s->opt[OPT_GEOMETRY_GROUP].desc = "";
  s->opt[OPT_GEOMETRY_GROUP].type = SANE_TYPE_GROUP;
  s->opt[OPT_GEOMETRY_GROUP].cap = SANE_CAP_ADVANCED;

  /* top-left x */
  s->opt[OPT_TL_X].name = SANE_NAME_SCAN_TL_X;
  s->opt[OPT_TL_X].title = SANE_TITLE_SCAN_TL_X;
  s->opt[OPT_TL_X].desc = SANE_DESC_SCAN_TL_X;
  s->opt[OPT_TL_X].type = SANE_TYPE_FIXED;
  s->opt[OPT_TL_X].unit = SANE_UNIT_MM;
  s->opt[OPT_TL_X].constraint_type = SANE_CONSTRAINT_RANGE;
  s->opt[OPT_TL_X].constraint.range = &s->hw->x_range;
  s->val[OPT_TL_X] = 0;

  /* top-left y */
  s->opt[OPT_TL_Y].name = SANE_NAME_SCAN_TL_Y;
  s->opt[OPT_TL_Y].title = SANE_TITLE_SCAN_TL_Y;
  s->opt[OPT_TL_Y].desc = SANE_DESC_SCAN_TL_Y;
  s->opt[OPT_TL_Y].type = SANE_TYPE_FIXED;
  s->opt[OPT_TL_Y].unit = SANE_UNIT_MM;
  s->opt[OPT_TL_Y].constraint_type = SANE_CONSTRAINT_RANGE;
  s->opt[OPT_TL_Y].constraint.range = &s->hw->y_range;
  s->val[OPT_TL_Y] = 0;

  /* bottom-right x */
  s->opt[OPT_BR_X].name = SANE_NAME_SCAN_BR_X;
  s->opt[OPT_BR_X].title = SANE_TITLE_SCAN_BR_X;
  s->opt[OPT_BR_X].desc = SANE_DESC_SCAN_BR_X;
  s->opt[OPT_BR_X].type = SANE_TYPE_FIXED;
  s->opt[OPT_BR_X].unit = SANE_UNIT_MM;
  s->opt[OPT_BR_X].constraint_type = SANE_CONSTRAINT_RANGE;
  s->opt[OPT_BR_X].constraint.range = &s->hw->x_range;
  s->val[OPT_BR_X] = s->hw->x_range.max;

  /* bottom-right y */
  s->opt[OPT_BR_Y].name = SANE_NAME_SCAN_BR_Y;
  s->opt[OPT_BR_Y].title = SANE_TITLE_SCAN_BR_Y;
  s->opt[OPT_BR_Y].desc = SANE_DESC_SCAN_BR_Y;
  s->opt[OPT_BR_Y].type = SANE_TYPE_FIXED;
  s->opt[OPT_BR_Y].unit = SANE_UNIT_MM;
  s->opt[OPT_BR_Y].constraint_type = SANE_CONSTRAINT_RANGE;
  s->opt[OPT_BR_Y].constraint.range = &s->hw->y_range;
  s->val[OPT_BR_Y] = s->hw->y_range.max;

  return SANE_STATUS_GOOD;
}

SANE_Status
sane_open (SANE_String_Const devicename, SANE_Handle *handle)
{
  Epson_Scanner *s;

  if (devicename[0] == '\0')
    devicename = dummy_dev.sane.name;

  if (strcmp (devicename, dummy_dev.sane.name) != 0)
    return SANE_STATUS_INVAL;

  s = calloc (1, sizeof (Epson_Scanner));
  if (s == NULL)
    return SANE_STATUS_NO_MEM;

  s->fd = -1;
  s->hw = &dummy_dev;

  init_options (s);

  *handle = (SANE_Handle) s;
  return SANE_STATUS_GOOD;
}

void
sane_close (SANE_Handle handle)
{
  Epson_Scanner *s = (Epson_Scanner *) handle;

  free (s->buf);

  if (s->fd != -1)
    sanei_scsi_close (s->fd);

  free (s);
}

const SANE_Option_Descriptor *
sane_get_option_descriptor (SANE_Handle handle, SANE_Int option)
{
  Epson_Scanner *s = (Epson_Scanner *) handle;

  if (option < 0 || option >= NUM_OPTIONS)
    return NULL;

  return s->opt + option;
}

static const SANE_String_Const *
search_string_list (const SANE_String_Const *list, SANE_String value)
{
  while (*list != NULL && strcmp (value, *list) != 0)
    ++list;

  if (*list == NULL)
    return NULL;

  return list;
}

SANE_Status
sane_control_option (SANE_Handle handle, SANE_Int option,
		     SANE_Action action, void *value,
		     SANE_Int *info)
{
  Epson_Scanner *s = (Epson_Scanner *) handle;
  SANE_Status status;
  const SANE_String_Const *optval;

  if (option < 0 || option >= NUM_OPTIONS)
    return SANE_STATUS_INVAL;

  if (info != NULL)
    *info = 0;

  switch (action)
    {
    case SANE_ACTION_GET_VALUE:
      switch (option)
	{
	case OPT_NUM_OPTS:
	case OPT_RESOLUTION:
	case OPT_TL_X:
	case OPT_TL_Y:
	case OPT_BR_X:
	case OPT_BR_Y:
	  *(SANE_Word *) value = s->val[option];
	  break;
	case OPT_MODE:
	case OPT_HALFTONE:
	case OPT_DROPOUT:
	  strcpy ((char *) value,
		  s->opt[option].constraint.string_list[s->val[option]]);
	  break;
	default:
	  return SANE_STATUS_INVAL;
	}
      break;
    case SANE_ACTION_SET_VALUE:
      status = sanei_constrain_value (s->opt + option, value, info);
      if (status != SANE_STATUS_GOOD)
	return status;

      optval = NULL;
      if (s->opt[option].constraint_type == SANE_CONSTRAINT_STRING_LIST)
	{
	  optval = search_string_list (s->opt[option].constraint.string_list,
				       (char *) value);
	  if (optval == NULL)
	    return SANE_STATUS_INVAL;
	}

      switch (option)
	{
	case OPT_RESOLUTION:
	case OPT_TL_X:
	case OPT_TL_Y:
	case OPT_BR_X:
	case OPT_BR_Y:
	  s->val[option] = *(SANE_Word *) value;
	  if (info != NULL)
	    *info |= SANE_INFO_RELOAD_PARAMS;
	  break;
	case OPT_MODE:
	  if (mode_params[optval - mode_list].depth != 1)
	    s->opt[OPT_HALFTONE].cap |= SANE_CAP_INACTIVE;
	  else
	    s->opt[OPT_HALFTONE].cap &= ~SANE_CAP_INACTIVE;
	  if (mode_params[optval - mode_list].color)
	    s->opt[OPT_DROPOUT].cap |= SANE_CAP_INACTIVE;
	  else
	    s->opt[OPT_DROPOUT].cap &= ~SANE_CAP_INACTIVE;
	  if (info != NULL)
	    *info |= SANE_INFO_RELOAD_OPTIONS | SANE_INFO_RELOAD_PARAMS;
	  /* fall through */
	case OPT_HALFTONE:
	case OPT_DROPOUT:
	  s->val[option] = optval - s->opt[option].constraint.string_list;
	  break;
	default:
	  return SANE_STATUS_INVAL;
	}
      break;
    default:
      return SANE_STATUS_INVAL;
    }

  return SANE_STATUS_GOOD;
}

SANE_Status
sane_get_parameters (SANE_Handle handle, SANE_Parameters *params)
{
  Epson_Scanner *s = (Epson_Scanner *) handle;
  int ndpi;

  memset (&s->params, 0, sizeof (SANE_Parameters));

  ndpi = s->val[OPT_RESOLUTION];

  s->params.pixels_per_line = SANE_UNFIX (s->val[OPT_BR_X] - s->val[OPT_TL_X]) / 25.4 * ndpi;
  s->params.lines = SANE_UNFIX (s->val[OPT_BR_Y] - s->val[OPT_TL_Y]) / 25.4 * ndpi;
  /* pixels_per_line seems to be 8 * n.  */
  s->params.pixels_per_line = s->params.pixels_per_line & ~7;

  s->params.last_frame = SANE_TRUE;
  s->params.depth = mode_params[s->val[OPT_MODE]].depth;
  if (mode_params[s->val[OPT_MODE]].color)
    {
      s->params.format = SANE_FRAME_RGB;
      s->params.bytes_per_line = 3 * s->params.pixels_per_line;
    }
  else
    {
      s->params.format = SANE_FRAME_GRAY;
      s->params.bytes_per_line = s->params.pixels_per_line * s->params.depth / 8;
    }

  if (params != NULL)
    *params = s->params;

  return SANE_STATUS_GOOD;
}

SANE_Status
sane_start (SANE_Handle handle)
{
  Epson_Scanner *s = (Epson_Scanner *) handle;
  SANE_Status status;
  const struct mode_param *mparam;
  int ndpi;
  int left, top;
  int lcount;

  status = sane_get_parameters (handle, NULL);
  if (status != SANE_STATUS_GOOD)
    return status;

  status = sanei_scsi_open (s->hw->sane.name, &s->fd, NULL);
  if (status != SANE_STATUS_GOOD)
    {
      DBG (1, "sane_start: %s open faild: %s\n", s->hw->sane.name,
	   sane_strstatus (status));
      return status;
    }

  reset (s->fd);

  mparam = mode_params + s->val[OPT_MODE];
  status = set_depth (s->fd, s->params.depth);
  if (status != SANE_STATUS_GOOD)
    {
      DBG (1, "sane_start: set_depth failed: %s\n", sane_strstatus (status));
      return status;
    }
  if (s->hw->level >= 5 && mparam->mode_flags == 0x02)
    status = set_mode (s->fd, 0x13);
  else
    status = set_mode (s->fd,
		       mparam->mode_flags
		       | (mparam->dropout_mask
			  & dropout_params[s->val[OPT_DROPOUT]]));
  if (status != SANE_STATUS_GOOD)
    {
      DBG (1, "sane_start: set_mode failed: %s\n", sane_strstatus (status));
      return status;
    }
  status = set_halftone (s->fd, halftone_params[s->val[OPT_HALFTONE]]);
  if (status != SANE_STATUS_GOOD)
    {
      DBG (1, "sane_start: set_halftone failed: %s\n", sane_strstatus (status));
      return status;
    }
  status = set_gamma (s->fd, s->params.depth == 1 ? 1 : 2);
  if (status != SANE_STATUS_GOOD)
    {
      DBG (1, "sane_start: set_gamma failed: %s\n", sane_strstatus (status));
      return status;
    }
  status = set_color (s->fd, 0x80);
  if (status != SANE_STATUS_GOOD)
    {
      DBG (1, "sane_start: set_color failed: %s\n", sane_strstatus (status));
      return status;
    }
  status = set_speed (s->fd, mode_params[s->val[OPT_MODE]].depth == 1 ? 1 : 0);
  if (status != SANE_STATUS_GOOD)
    {
      DBG (1, "sane_start: set_speed failed: %s\n", sane_strstatus (status));
      return status;
    }

  ndpi = s->val[OPT_RESOLUTION];
  status = set_resolution (s->fd, ndpi, ndpi);
  if (status != SANE_STATUS_GOOD)
    {
      DBG (1, "sane_start: set_resolution failed: %s\n", sane_strstatus (status));
      return status;
    }
  left = SANE_UNFIX (s->val[OPT_TL_X]) / 25.4 * ndpi + 0.5;
  top = SANE_UNFIX (s->val[OPT_TL_Y]) / 25.4 * ndpi + 0.5;
  status = set_area (s->fd, left, top, s->params.pixels_per_line,
		     s->params.lines);
  if (status != SANE_STATUS_GOOD)
    {
      DBG (1, "sane_start: set_area failed: %s\n", sane_strstatus (status));
      return status;
    }

  s->block = SANE_FALSE;
  lcount = 1;
  if (s->hw->level >= 5
      || (s->hw->level >= 4 && !mode_params[s->val[OPT_MODE]].color))
    {
      s->block = SANE_TRUE;
      lcount = sanei_scsi_max_request_size / s->params.bytes_per_line;
      if (lcount > 255)
	lcount = 255;
      if (lcount == 0)
	return SANE_STATUS_NO_MEM;
      status = set_lcount (s->fd, lcount);
      if (status != SANE_STATUS_GOOD)
	{
	  DBG (1, "sane_start: set_lcount failed: %s\n", sane_strstatus (status));
	  return status;
	}
    }

  status = send (s->fd, "\033G", 2);
  if (status != SANE_STATUS_GOOD)
    {
      DBG (1, "sane_start: start failed: %s\n", sane_strstatus (status));
      return status;
    }

  s->eof = SANE_FALSE;
  s->buf = realloc (s->buf, lcount * s->params.bytes_per_line);
  s->ptr = s->end = s->buf;
  s->canceling = SANE_FALSE;
  return SANE_STATUS_GOOD;
}

SANE_Status
sane_read (SANE_Handle handle, SANE_Byte *data,
	   SANE_Int max_length, SANE_Int *length)
{
  Epson_Scanner *s = (Epson_Scanner *) handle;

  if (s->ptr == s->end)
    {
      unsigned char result[6];
      size_t len;
      size_t buf_len;

      if (s->eof)
	{
	  free (s->buf);
	  s->buf = NULL;
	  sanei_scsi_close (s->fd);
	  s->fd = -1;
	  *length = 0;
	  return SANE_STATUS_EOF;
	}

      len = s->block ? 6 : 4;
      receive (s->fd, result, &len);
      buf_len = result[3] << 8 | result[2];
      if (s->block)
	buf_len *= (result[5] << 8 | result[4]);
      if (!s->block && s->params.format == SANE_FRAME_RGB)
	{
	  receive (s->fd, s->buf + s->params.pixels_per_line, &buf_len);
	  send (s->fd, "\006", 1);
	  len = 6;
	  receive (s->fd, result, &len);
	  buf_len = result[3] << 8 | result[2];

	  receive (s->fd, s->buf, &buf_len);
	  send (s->fd, "\006", 1);
	  len = 6;
	  receive (s->fd, result, &len);
	  buf_len = result[3] << 8 | result[2];

	  receive (s->fd, s->buf + 2 * s->params.pixels_per_line, &buf_len);
	}
      else
	receive (s->fd, s->buf, &buf_len);

      if (result[1] & 0x20)
	s->eof = SANE_TRUE;
      else
	{
	  if (s->canceling)
	    {
	      send (s->fd, "\030", 1);
	      expect_ack (s->fd);
	      free (s->buf);
	      s->buf = NULL;
	      sanei_scsi_close (s->fd);
	      s->fd = -1;
	      *length = 0;
	      return SANE_STATUS_CANCELLED;
	    }
	  else
	    send (s->fd, "\006", 1);
	}

      s->end = s->buf + buf_len;
      s->ptr = s->buf;
    }

  if (!s->block && s->params.format == SANE_FRAME_RGB)
    {
      max_length /= 3;
      if (max_length > s->end - s->ptr)
	max_length = s->end - s->ptr;
      *length = 3 * max_length;
      while (max_length-- != 0)
	{
	  *data++ = s->ptr[0];
	  *data++ = s->ptr[s->params.pixels_per_line];
	  *data++ = s->ptr[2 * s->params.pixels_per_line];
	  ++s->ptr;
	}
    }
  else
    {
      if (max_length > s->end - s->ptr)
	max_length = s->end - s->ptr;
      *length = max_length;
      if (s->params.depth == 1)
	{
	  while (max_length-- != 0)
	    *data++ = ~*s->ptr++;
	}
      else
	{
	  memcpy (data, s->ptr, max_length);
	  s->ptr += max_length;
	}
    }

  return SANE_STATUS_GOOD;
}

void
sane_cancel (SANE_Handle handle)
{
  Epson_Scanner *s = (Epson_Scanner *) handle;

  if (s->buf != NULL)
    s->canceling = SANE_TRUE;
}

SANE_Status
sane_set_io_mode (SANE_Handle handle, SANE_Bool non_blocking)
{
  return SANE_STATUS_UNSUPPORTED;
}

SANE_Status
sane_get_select_fd (SANE_Handle handle, SANE_Int *fd)
{
  return SANE_STATUS_UNSUPPORTED;
}
