/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rjones@orchestream.com>.	*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	lib/mutex.h
 * Summary:
 *	These functions provide mutual exclusion between signal handlers,
 *	and (in the future) threads when we update critical global data
 *	such as the object tree.
 * Other notes:
 *	You may need to fiddle around here to get round operating system
 *	dependencies, and also if you want to add support for threads.
 *	The overhead of providing mutual exclusion is so large that I've
 *	decided to turn it off by default. It requires two signal calls
 *	every time you enter the bounds checking library (at least two).
 *	There are some static variables in functions that will need to
 *	be made thread-local if you use multiple threads. Particularly
 *	look at string.c : __bounds_strtok ().
 * Author      	Date		Notes
 * RWMJ		15/3/95		Initial implementation.
 *----------------------------------------------------------------------*/

#ifndef _MUTEX_H_
#define _MUTEX_H_

#define MUTEX_FOR_SIGNAL_HANDLERS	0	/* Add support for signals. */
#define MUTEX_FOR_THREADS		0	/* Not yet implemented. */

#if MUTEX_FOR_SIGNAL_HANDLERS

#include <signal.h>

/* This is Sys V signal handling. BSD signal handling is different, so
 * someone needs to fix that.
 */
extern sigset_t __bounds_old_mask;		/* Previous signal mask. */
#endif /* MUTEX_FOR_SIGNAL_HANDLERS */

/* The following function is called whenever we enter a critical section of
 * code.
 */
static inline void
enter_critical_section (void)
{
#if MUTEX_FOR_SIGNAL_HANDLERS
  {
    /* This is Sys V signal handling. BSD signal handling is different, so
     * someone needs to fix that.
     */
    sigset_t new_mask;
    sigfillset (&new_mask);			/* Block all signals. */
    sigprocmask (SIG_BLOCK, &new_mask, &__bounds_old_mask);
  }
#endif /* MUTEX_FOR_SIGNAL_HANDLERS */
}

/* The following function is called whenever we leave a critical section.
 */
static inline void
leave_critical_section (void)
{
#if MUTEX_FOR_SIGNAL_HANDLERS
  /* This is Sys V signal handling. BSD signal handling is different, so
   * someone needs to fix that.
   */
  sigprocmask (SIG_SETMASK, &__bounds_old_mask, NULL);
#endif /* MUTEX_FOR_SIGNAL_HANDLERS */
}

#endif /* _MUTEX_H_ */
