/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rjones@orchestream.com>.	*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	lib/malloc/bounds.c
 * Summary:
 *	Bounds checking interface to the GNU C malloc library. The GNU
 *	malloc has been modified to provide the unchecked functions
 *	like '__bounds_malloc' directly. We provide the checked versions
 *	(eg. 'malloc') here.
 * Other notes:
 *	
 * Author      	Date		Notes
 * RWMJ		1/6/95		Initial implementation.
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>

#ifdef malloc
#undef malloc
#endif
#ifdef free
#undef free
#endif
#ifdef calloc
#undef calloc
#endif
#ifdef cfree
#undef cfree
#endif
#ifdef memalign
#undef memalign
#endif
#ifdef realloc
#undef realloc
#endif
#ifdef valloc
#undef valloc
#endif

#include "../bounds-lib.h"

#if defined(__BOUNDS_CHECKING_ON)
#error "This file must not be compiled with bounds checking enabled."
#endif

/*----------------------------------------------------------------------
 *	Free a memory area in reuse mode. We add the memory area to a
 *	list of memory areas where it will reside until the area ages
 *	sufficiently, and then it is actually reallocated.
 *----------------------------------------------------------------------*/

static inline void
free_reuse (void *pointer)
{
  static int initialized = 0, enter = 0, leave = 0, count = 0;
  static void **arealist;

  if (__bounds_age_limit == 0 || !__bounds_checking_on)
    {
      __bounds_free (pointer);
      return;
    }

  if (!initialized)
    {
      initialized = 1;
      arealist =
	(void **) __bounds_malloc (sizeof (void *) * __bounds_age_limit);
      if (arealist == NULL)
	__bounds_internal_error ("out of memory allocating memory ageing list",
				 __FILE__, __LINE__);
    }
  if (count == __bounds_age_limit)
    {
      __bounds_free (arealist [leave]);
      leave ++;
      if (leave >= __bounds_age_limit) leave = 0;
    }
  else
    count ++;
  arealist [enter] = pointer;
  enter ++;
  if (enter >= __bounds_age_limit) enter = 0;
}

/*----------------------------------------------------------------------
 *	Free a pointer in 'no-reuse-heap' mode, ie. don't free it at
 *	all. Nevertheless, we ought to combine such memory areas and
 *	see if we can hand them back to the operating system (with
 *	'munmap').
 *----------------------------------------------------------------------*/

static inline void
free_no_reuse (void *pointer)
{
  /* Not yet implemented. */
}

/*----------------------------------------------------------------------
 *	Wrapper functions for the major C library dynamic memory functions.
 *	These call the GNU library with the side effect of adding or
 *	deleting memory objects where necessary.
 *----------------------------------------------------------------------*/

void *
__bounds_check_malloc (char *filename, int line, size_t size)
{
  void *pointer;

  /* Although malloc (0) is a strange usage, we are required to support it.
   * We can return any pointer that cannot be dereferenced here, in
   * particular, some libraries return NULL. However, some software thinks
   * that means out of memory. We could also return ILLEGAL, but X11 has
   * a habit of passing that pointer back to free (). Instead we make
   * malloc (0) == malloc (1), and you'd better not dereference it, since
   * the objects library doesn't support zero-sized objects.
   */
  if (size == 0)
    size = 1;
  pointer = __bounds_malloc (size + 1);
  if (pointer != NULL)
    __bounds_add_heap_object (pointer, size, 1, "malloc", filename, line);
  return pointer;
}

void *
malloc(size_t size)
{
  return __bounds_check_malloc( NULL, 0, size);
}

void
free (void *pointer)
{
  if (pointer == NULL)
    {
      if (__bounds_warn_free_null)
	__bounds_warning (NULL, 0, "free", "attempted to free a NULL pointer");
      return;
    }
  if (pointer == ILLEGAL)
    {
      __bounds_error ("attempted to free an ILLEGAL pointer", NULL, 0,
		      ILLEGAL, NULL);
      return;
    }
  if (__bounds_is_heap_object (pointer))
    {
      /* This pointer is OK. Delete the associated memory object, then
       * decide how we're going to free this memory.
       */
      __bounds_delete_heap_object (pointer);
      if (__bounds_never_free_heap)
	free_no_reuse (pointer);
      else
	free_reuse (pointer);
    }
  else
    /* This pointer didn't come from 'malloc' so generate an error.
     */
    {
      __bounds_error ("invalid pointer passed to `free'",
		      NULL, 0, pointer, NULL);
      return;
    }
}

void*
__bounds_check_realloc (char *filename, int line,
		       void *pointer, size_t new_size)
{
  void *new_pointer;

  if (pointer != NULL)
    {
      if (pointer == ILLEGAL || !__bounds_is_heap_object (pointer))
	{
	  /* This pointer is non-NULL, but doesn't point to a heap object, so
	   * this is an error.
	   */
	  __bounds_error ("invalid pointer passed to `realloc'",
			  NULL, 0, pointer, NULL);
	  return NULL;
	}
      __bounds_delete_heap_object (pointer);
    }
  if (new_size == 0)
    new_size = 1;
  new_pointer = __bounds_realloc (pointer, new_size + 1);
  if (new_pointer != NULL)
    __bounds_add_heap_object (new_pointer, new_size, 1, "realloc", filename, line);
  return new_pointer;
}

void *
realloc(void *pointer, size_t new_size)
{
  return __bounds_check_realloc(NULL, 0, pointer, new_size);
}

void *
__bounds_check_memalign (char *filename, int line,
		        size_t alignment, size_t size)
{
  void *pointer;

  if (size == 0)
    size = 1;
  pointer = __bounds_memalign (alignment, size + 1);
  if (pointer != NULL)
    __bounds_add_heap_object (pointer, size, 1, "memalign", filename, line);
  return pointer;
}

void *
memalign(size_t alignment, size_t size)
{
  return __bounds_check_memalign(NULL, 0, alignment, size);
}

void *
__bounds_check_calloc (char *filename, int line,
		      size_t size1, size_t size2)
{
  size_t size = size1 * size2;
  void *pointer;

  if (size == 0)
    size = 1;
  pointer = __bounds_calloc (size + 1, 1);
  if (pointer != NULL)
    __bounds_add_heap_object (pointer, size, 1, "calloc", filename, line);
  return pointer;
}

void *
calloc(size_t size1, size_t size2)
{
  return __bounds_check_calloc(NULL, 0, size1, size2);
}

void
cfree (void *pointer)
{
  free (pointer);
}

void *
__bounds_check_valloc (char *filename, int line, size_t size)
{
  void *pointer;

  if (size == 0)
    size = 1;
  pointer = __bounds_valloc (size + 1);
  if (pointer != NULL)
    __bounds_add_heap_object (pointer, size, 1, "valloc", filename, line);
  return pointer;
}

void *
valloc(size_t size)
{
  return __bounds_check_valloc(NULL, 0, size);
}
