/*----------------------------------------------------------------------*
 * Bounds Checking for GCC.						*
 * Copyright (C) 1995 Richard W.M. Jones <rjones@orchestream.com>.	*
 *----------------------------------------------------------------------*
 * This program is free software; you can redistribute it and/or modify	*
 * it under the terms of the GNU General Public License as published by	*
 * the Free Software Foundation; either version 2 of the License, or	*
 * (at your option) any later version.					*
 *									*
 * This program is distributed in the hope that it will be useful,	*
 * but WITHOUT ANY WARRANTY; without even the implied warranty of	*
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the	*
 * GNU General Public License for more details.				*
 *									*
 * You should have received a copy of the GNU General Public License	*
 * along with this program; if not, write to the Free Software		*
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		*
 *----------------------------------------------------------------------*
 * File:
 *	lib/init.c
 * Summary:
 *	Code that initializes the library, and reads the bounds checking
 *	options from the environment variable.
 * Other notes:
 *	
 * Author      	Date		Notes
 * RWMJ		12/2/95		Initial implementation.
 *----------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>

#include "tm.h"		/* Get HAVE_ATEXIT */
#include "bounds-lib.h"

#if defined(__BOUNDS_CHECKING_ON)
#error "This file must not be compiled with bounds checking enabled."
#endif

/* The contents of the following environment variable determine how the
 * library works at run time.
 */
#define BOUNDS_CHECKING_VAR	"GCC_BOUNDS_OPTS"

#define PRINT_WELCOME_MSG	1	/* Print welcome message by default. */
#define IGNORE_BAD_FLAGS	0	/* Ignore badly formed env. variable */

static int print_welcome_msg = 1;
#if COLLECT_STATS
static int print_statistics = 1;
#endif

static int process_opts (char *);
static void usage (void);
#if COLLECT_STATS
static void print_stats (void);
#endif

/*----------------------------------------------------------------------
 * 	Initialize the library.
 * 	This function may be called multiple times while constructing
 * 	objects. (It is called once for each object file that was compiled
 *	with bounds checking). We set the '__bounds_checking_on' flag
 *	(which is in libgcc2.c) to 1, so we know if we have been
 *	called before. If this function is never called, then the
 *	'__bounds_checking_on' flag will remain at 0, which is what
 *	we want.
 *	Look for the location of environment variables (from 'environ').
 *----------------------------------------------------------------------*/

void
__bounds_initialize_library (void)
{
  char *v;
  extern char **environ;
  int i;

  if (!__bounds_checking_on) {
#if DEBUG_FEATURES
    if (__bounds_debug_print_calls)
      printf ("__bounds_initialize_library()\n");
#endif

    /* Set the __bounds_checking_on flag, so that we don't call this function
     * twice.
     */
    __bounds_checking_on = 1;

    /* If we are using a cache, initialize it.
     */
#if USE_CACHE
    __bounds_initialize_cache ();
#endif

    /* Look for the environment variable containing run-time options.
     */
    v = getenv (BOUNDS_CHECKING_VAR);
    if (v && !process_opts (v)) {
      printf ("bounds-checking: Badly formed variable '"
	       BOUNDS_CHECKING_VAR
	       "'.\n");
      usage ();
#if !IGNORE_BAD_FLAGS
      exit (1);
#endif
    }

#if PRINT_WELCOME_MSG
    /* Print a welcome message. This message can be suppressed by using
     * '-no-message' in the options variable, or by setting PRINT_WELCOME_MSG
     * to 0 at the top of this file.
     */
    if (print_welcome_msg) {
      printf ("Bounds Checking GCC v " VERSION_STRING " Copyright (C) 1995 Richard W.M. Jones\n"
	      "Bounds Checking comes with ABSOLUTELY NO WARRANTY. For details see file\n"
	      "`COPYING' that should have come with the source to this program.\n"
	      "Bounds Checking is free software, and you are welcome to redistribute it\n"
	      "under certain conditions. See the file `COPYING' for details.\n"
	      "For more information, set GCC_BOUNDS_OPTS to `-help'\n");
    }
#endif /* PRINT_WELCOME_MSG */

    /* Register the environment variables as static objects.
     * 31/5/95: Now use stack objects. Set 'no_padding' to 1.
     */
    for (i = 0; environ[i]; ++i) {
#if COLLECT_STATS
      ++__bounds_stats_environment;
#endif
      __bounds_internal_add_stack_object (environ[i],
					  __bounds_strlen (environ[i]) + 1,
					  1, "main", 0, "environ", 1);
    }
#if COLLECT_STATS
    ++__bounds_stats_environment;
#endif
    /* RWMJ: (21/9/95) Although the documentation suggests otherwise, SunOS
     * and Solaris need `i+2' here, because they appear to use 2 extra
     * NULL pointers to terminate environ, instead of 1. Whether this
     * will make a difference on other systems, I do not know.
     * RWMJ: (10/1/96) This breaks OS/2 and possibly others. Now only use
     * the `+2' on Suns. Thanks Eberhard Mattes.
     */
#if defined(sun)
#define ENV_SUN_HACK 2
#else
#define ENV_SUN_HACK 1
#endif
    __bounds_internal_add_stack_object (environ,
					(i + ENV_SUN_HACK) * sizeof (char *),
					sizeof (char *), "main", 0, "environ",
					1);
#undef ENV_SUN_HACK

    /* This could probebly be solved by changing the include files in
     * the gcc include directory. I put it here because I dont know how
     * to modify the fix-includes script's.
     */
#if defined(__svr4__) || defined(svr4)
#if defined(__STDC__)
    __bounds_note_constructed_object(__ctype,-514,1,NULL,0,"__ctype");
#else
    __bounds_note_constructed_object(_ctype,-514,1,NULL,0,"_ctype");
#endif
    /* HtB: iob struct has already a size on svr4 so whe do not have to 
     * include it here. It is included as an external reference.
     */
#elif defined(sun)
    /* RWMJ: This is reported not to work on standard Solaris 2.4. If
     * you get a problem, then simply comment out the next two lines. Can
     * someone tell me what the compiler symbol I need to look out for
     * is?
     */
/*
Reading specs from /opt/local/lib/gcc-lib/i386-unknown-solaris2.4/2.7.0/specs
gcc version 2.7.0
 /opt/local/lib/gcc-lib/i386-unknown-solaris2.4/2.7.0/cpp -lang-c -v -undef -D__GNUC__=2 -D__GNUC_MINOR__=7 -Di386 -Dunix -D__svr4__ -Dsun -D__i386__ -D__unix__ -D__svr4__ -D__sun__ -D__i386 -D__unix -D__sun -Asystem(unix) -Asystem(svr4) -Acpu(i386) -Amachine(i386) enigma.c /tmp/cca007B0.i
GNU CPP version 2.7.0 (i386 System V Release 4)
*/
 #if defined (__svr4__)
  #if defined(__STDC__)
    __bounds_note_constructed_object(__ctype,-514,1,NULL,0,"__ctype");
  #else
    __bounds_note_constructed_object(_ctype,-514,1,NULL,0,"_ctype");
  #endif
 #else
    __bounds_note_constructed_object(_ctype_,-257,1,NULL,0,"_ctype_");
    __bounds_note_constructed_object(_iob,-32*sizeof(struct _iobuf),1,
					NULL,0,"_iob");
 #endif
#endif

#if COLLECT_STATS
    /* If we are collecting statistics, print them out when the program
     * exits.
     */
/* Thanks to Michael Huehne <root@technik.3-s.de> for this: */
#if defined(__svr4__) && !defined(HAVE_ATEXIT)
#define HAVE_ATEXIT
#endif

#if defined(__GNU_LIBRARY__) || defined(HAVE_ATEXIT)
    atexit (print_stats);
#else
    on_exit (print_stats, NULL);
#endif
#endif
  }
}

/*----------------------------------------------------------------------
 *	The following functions and tables deal with processing arguments
 *	passed in the environment variable.
 *----------------------------------------------------------------------
 */

static void
usage (void)
{
  printf ("You may supply a list of the following arguments to a bounds-checked program\n"
	  "by listing them in the environment variable '" BOUNDS_CHECKING_VAR "' before\n"
	  "running the program. Separate the arguments by spaces.\n"
	  "General:\n"
#if PRINT_WELCOME_MSG
	  "  -no-message                 Don't print introductory message.\n"
#endif
#if COLLECT_STATS
	  "  -no-statistics              Don't print statistics.\n"
#endif
	  "  -?, -help                   Print this table of usage.\n"
	  "Control runtime behaviour:\n"
	  "  -array-index-check         *Check the index of all array references\n"
	  "  -no-array-index-check       Only check the pointer is within the array\n"
	  "  -never-fatal                Don't abort after a bounds error.\n"
	  "  -reuse-heap                *Re-use the heap.\n"
	  "  -reuse-age=<age>            Set the age limit before freeing (default: 0).\n"
	  "  -no-reuse-heap              Never really free old heap blocks.\n"
	  "  -warn-unchecked-statics     Warn if unchecked static objects are referenced.\n"
	  "  -no-warn-unchecked-statics *Switch off the above.\n"
	  "  -warn-unchecked-stack       Warn if unchecked stack objects are referenced.\n"
	  "  -no-warn-unchecked-stack   *Switch off the above.\n"
	  "  -warn-free-null            *Warn if free (0) is used.\n"
	  "  -no-warn-free-null          Switch off the above.\n"
	  "  -warn-misc-strings         *Warn for miscellaneous strings usage.\n"
	  "  -no-warn-misc-strings       Switch off the above.\n"
	  "  -warn-illegal               Warn when ILLEGAL pointers are created.\n"
	  "  -no-warn-illegal           *Switch off the above.\n"
	  "  -warn-unaligned            *Warn when pointers are used unaligned.\n"
	  "  -no-warn-unaligned          Switch off the above.\n"
	  "  -warn-all                   Turn on all warnings.\n"
#if DEBUG_FEATURES
	  "Debugging:\n"
	  "  -print-calls                Print calls to the bounds-checking library.\n"
	  "  -no-print-calls            *Don't print calls.\n"
	  "  -print-heap                 Print all heap data.\n"
	  "  -no-print-heap             *Don't print heap data.\n"
#endif
	  "Note: `*' means this is the default behaviour.\n"
);
}

static int
process_opts (char *opts)
{
  int len = __bounds_strlen (opts), done = 0;
  char buffer[len + 1], *in_opts = opts;

  while (!done) {
    /* Copy the next option into the private buffer (`buffer') and deal with
     * it there. Allow the user to escape spaces with `\ ' and to insert
     * tabs and newlines with `\t' and `\n' resp. A double quoted string
     * also shields spaces.
     */
    char *p = buffer;
    int in_quote = 0;

/*    printf ("reading %s\n", in_opts); */

    do {
      switch (*in_opts) {
      case ' ':				/* Last character in the argument. */
	if (!in_quote) *p = 0; else *p = ' '; break;
      case '\0':			/* Last character in the string. */
	if (in_quote) return 0;		/* Unterminated quote character. */
	*p = 0; done = 1;
	break;
      case '\\':			/* Escaped character. */
	in_opts ++;
	switch (*in_opts) {		/* Process \t and \n. */
	case 't':
	  *p = '\t'; break;
	case 'n':
	  *p = '\n'; break;
	case '\0':
	  return 0;			/* \ cannot be at end of the string */
	default:
	  *p = *in_opts;
	}
	break;
      case '"':				/* Start or end of "...". */
	in_quote = !in_quote; break;
      default:				/* Another character. */
	*p = *in_opts;
      }
      ++in_opts;
    } while (*p++);

    /* Now we have a single argument in the private buffer. See if we know
     * what it is. If we don't understand it, we return 0 immediately, which
     * causes the calling function to print usage. Arguments like '-no-
     * statistics' which are enabled & disabled by #defines in the library
     * are ignored silently if not applicable.
     */
    p = buffer;
    while (*p && (*p == ' ' || *p == '\t' || *p == '\n' || *p == '\r'))
				/* Skip leading whitespace. */
      p ++;
    if (*p != 0) {
/*      printf ("decoded as %s\n", p); */
      if (__bounds_strcmp (p, "-no-message") == 0)
	print_welcome_msg = 0;
      else if (__bounds_strcmp (p, "-no-statistics") == 0)
#if COLLECT_STATS
	print_statistics = 0
#endif
	  ;
      else if (__bounds_strcmp (p, "-?") == 0
	       || __bounds_strcmp (p, "-help") == 0) {
	usage ();
	exit (0);
      } else if (__bounds_strcmp (p, "-array-index-check") == 0)
	__bounds_array_index_check = 1;
      else if (__bounds_strcmp (p, "-no-array-index-check") == 0)
	__bounds_array_index_check = 0;
      else if (__bounds_strcmp (p, "-never-fatal") == 0)
	__bounds_never_fatal = 1;
      else if (__bounds_strcmp (p, "-reuse-heap") == 0)
	__bounds_never_free_heap = 0;
      else if (__bounds_strcmp (p, "-no-reuse-heap") == 0)
	__bounds_never_free_heap = 1;
      else if (__bounds_strncmp (p, "-reuse-age=", 11) == 0)
	__bounds_age_limit = atoi (p + 11);
      else if (__bounds_strcmp (p, "-print-calls") == 0)
#if DEBUG_FEATURES
	__bounds_debug_print_calls = 1
#endif
	  ;
      else if (__bounds_strcmp (p, "-no-print-calls") == 0)
#if DEBUG_FEATURES
	__bounds_debug_print_calls = 0
#endif
	  ;
      else if (__bounds_strcmp (p, "-print-heap") == 0)
#if DEBUG_FEATURES
	  __bounds_debug_print_heap = 1
#endif
	  ;
      else if (__bounds_strcmp (p, "-no-print-heap") == 0)
#if DEBUG_FEATURES
	  __bounds_debug_print_heap = 0
#endif
	  ;
      else if (__bounds_strcmp (p, "-warn-unchecked-statics") == 0)
	__bounds_warn_unchecked_statics = 1;
      else if (__bounds_strcmp (p, "-no-warn-unchecked-statics") == 0)
	__bounds_warn_unchecked_statics = 0;
      else if (__bounds_strcmp (p, "-warn-unchecked-stack") == 0)
	__bounds_warn_unchecked_stack = 1;
      else if (__bounds_strcmp (p, "-no-warn-unchecked-stack") == 0)
	__bounds_warn_unchecked_stack = 0;
      else if (__bounds_strcmp (p, "-warn-free-null") == 0)
	__bounds_warn_free_null = 1;
      else if (__bounds_strcmp (p, "-no-warn-free-null") == 0)
	__bounds_warn_free_null = 0;
      else if (__bounds_strcmp (p, "-warn-misc-strings") == 0)
	__bounds_warn_misc_strings = 1;
      else if (__bounds_strcmp (p, "-no-warn-misc-strings") == 0)
	__bounds_warn_misc_strings = 0;
      else if (__bounds_strcmp (p, "-warn-illegal") == 0)
	__bounds_warn_illegal = 1;
      else if (__bounds_strcmp (p, "-no-warn-illegal") == 0)
	__bounds_warn_illegal = 0;
      else if (__bounds_strcmp (p, "-warn-unaligned") == 0)
	__bounds_warn_unaligned = 1;
      else if (__bounds_strcmp (p, "-no-warn-unaligned") == 0)
	__bounds_warn_unaligned = 0;
      else if (__bounds_strcmp (p, "-warn-all") == 0)
	__bounds_warn_unchecked_statics =
	  __bounds_warn_unchecked_stack =
	  __bounds_warn_misc_strings =
	  __bounds_warn_free_null =
	  __bounds_warn_illegal =
	  __bounds_warn_unaligned = 1;
      else
	return 0;			/* Unrecognized option. */
    }
  } /* for (in_opts ...) */

  return 1;
}

#if COLLECT_STATS

/* Print call frequency statistics when the program exits.
 */
static void
print_stats (void)
{
 if (print_statistics) {
  printf ("Bounds library call frequency statistics:\n"
          "  Calls to push, pop, param function:        %d, %d, %d\n"
          "  Calls to add, delete stack:                %d, %d\n"
          "  Calls to add, delete heap:                 %d, %d\n"
	  "  Calls to check pointer +/- integer:        %d\n"
	  "  Calls to check array references:           %d\n"
	  "  Calls to check pointer differences:        %d\n"
	  "  Calls to check object references:          %d\n"
	  "  Calls to check component references:       %d\n"
	  "  Calls to check truth, falsity of pointers: %d, %d\n"
	  "  Calls to check <, >, <=, >= of pointers:   %d\n"
	  "  Calls to check ==, != of pointers:         %d\n"
	  "  Calls to check p++, ++p, p--, --p:         %d, %d, %d, %d\n"
	  "  References to unchecked static, stack:     %d, %d\n",
          __bounds_stats_push_function,
          __bounds_stats_pop_function,
          __bounds_stats_param_function,
          __bounds_stats_add_stack -
          __bounds_stats_environment,
          __bounds_stats_delete_stack,
          __bounds_stats_add_heap,
          __bounds_stats_delete_heap,
	  __bounds_stats_ptr_plus_int,
	  __bounds_stats_array_reference,
	  __bounds_stats_ptr_diff,
	  __bounds_stats_reference,
	  __bounds_stats_component_reference,
	  __bounds_stats_ptr_true,
	  __bounds_stats_ptr_false,
	  __bounds_stats_ptr_lt_ptr +
	  __bounds_stats_ptr_le_ptr +
	  __bounds_stats_ptr_gt_ptr +
	  __bounds_stats_ptr_ge_ptr,
	  __bounds_stats_ptr_ne_ptr +
	  __bounds_stats_ptr_eq_ptr,
	  __bounds_stats_ptr_postinc,
	  __bounds_stats_ptr_preinc,
	  __bounds_stats_ptr_postdec,
	  __bounds_stats_ptr_predec,
	  __bounds_stats_unchecked_static,
	  __bounds_stats_unchecked_stack);
  __bound_print_statistics();
 }
 if (__bounds_debug_print_heap)
  __bound_print_heap();
}

#endif /* COLLECT_STATS */
