;;; cp850-acc.el -- minor mode providing electric accent keys
;;; Copyright (C) 1993 Free Software Foundation, Inc.

;; Author: Johan Vromans <jv@mh.nl>,
;; modified by Bodo Huckestein <bodo@eu1.mpi-hd.mpg.de> for codepage 850
;; Version: 1.7 (modified)
;; Maintainer: FSF
;; Keywords: i18n

;; This file is part of GNU Emacs.

;; GNU Emacs is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:

;; Function `cp850-accents-mode' activates a minor mode
;; (`cp850-accents-minor-mode') in which typewriter "dead keys" are
;; emulated.  The purpose of this emulation is to provide a simple
;; means for inserting accented characters according to the codepage 850
;; character set.
;;
;; In `cp850-accents-minor-mode', pseudo accent characters are used to
;; introduce accented keys.  The pseudo-accent characterss are:
;;
;;   '  (minute)    -> grave accent
;;   `  (backtick)  -> acute accent
;;   "  (second)    -> diaeresis
;;   ^  (caret)     -> circumflex
;;   ~  (tilde)     -> tilde over the character
;;   /  (slash)     -> slash through the character.
;;                  Also:  /A is A-with-ring and /E is AE ligature.
;;
;; The action taken depends on the key that follows the pseudo accent.
;; In general: 
;;
;;   pseudo-accent + appropriate letter -> accented letter
;;   pseudo-accent + space -> pseudo-accent
;;   pseudo-accent + pseudo-accent -> accent (if available)
;;   pseudo-accent + other -> pseudo-accent + other
;;
;; If the pseudo-accent is followed by anything else than a 
;; self-insert-command, the dead-key code is terminated, the
;; pseudo-accent inserted 'as is' and the bell is rung to signal this.
;;
;; Function `cp850-accents-mode' can be used to enable the cp850 accents
;; minor mode, or disable it.

;;; Code:

(provide 'cp850-acc)

(defvar cp850-accents-list
  '(((?' ?A) ?\265)
    ((?' ?E) ?\220)
    ((?' ?I) ?\326)
    ((?' ?O) ?\340)
    ((?' ?U) ?\351)
    ((?' ?Y) ?\355)
    ((?' ?a) ?\240)
    ((?' ?e) ?\202)
    ((?' ?i) ?\241)
    ((?' ?o) ?\242)
    ((?' ?u) ?\243)
    ((?' ?y) ?\354)
    ((?' ?') ?\357)
    ((?' ? ) ?')
    ((?` ?A) ?\267)
    ((?` ?E) ?\324)
    ((?` ?I) ?\336)
    ((?` ?O) ?\343)
    ((?` ?U) ?\353)
    ((?` ?a) ?\205)
    ((?` ?e) ?\212)
    ((?` ?i) ?\215)
    ((?` ?o) ?\225)
    ((?` ?u) ?\227)
    ((?` ? ) ?`)
    ((?` ?`) ?`)		; no special code?
    ((?^ ?A) ?\266)
    ((?^ ?E) ?\322)
    ((?^ ?I) ?\327)
    ((?^ ?O) ?\342)
    ((?^ ?U) ?\352)
    ((?^ ?a) ?\203)
    ((?^ ?e) ?\210)
    ((?^ ?i) ?\214)
    ((?^ ?o) ?\223)
    ((?^ ?u) ?\226)
    ((?^ ? ) ?^)
    ((?^ ?^) ?^)		; no special code?
    ((?\" ?A) ?\216)
    ((?\" ?E) ?\323)
    ((?\" ?I) ?\330)
    ((?\" ?O) ?\231)
    ((?\" ?U) ?\232)
    ((?\" ?a) ?\204)
    ((?\" ?e) ?\211)
    ((?\" ?i) ?\213)
    ((?\" ?o) ?\224)
    ((?\" ?u) ?\201)
    ((?\" ?y) ?\230)
    ((?\" ?s) ?\341)
    ((?\" ? ) ?\")
    ((?\" ?\") ?\371)
    ((?\~ ?A) ?\307)
    ((?\~ ?C) ?\200)
    ((?\~ ?D) ?\321)
    ((?\~ ?N) ?\245)
    ((?\~ ?O) ?\345)
    ((?\~ ?a) ?\306)
    ((?\~ ?c) ?\207)
    ((?\~ ?d) ?\320)
    ((?\~ ?n) ?\244)
    ((?\~ ?o) ?\344)
    ((?\~ ?>) ?\257)
    ((?\~ ?<) ?\256)
    ((?\/ ?A) ?\217) ;; A-with-ring (Norwegian and Danish)
    ((?\/ ?E) ?\222) ;; AE-ligature (Norwegian and Danish)
    ((?\/ ?O) ?\235)
    ((?\/ ?a) ?\206) ;; a-with-ring (Norwegian and Danish)
    ((?\/ ?e) ?\221) ;; ae-ligature (Norwegian and Danish)
    ((?\/ ?o) ?\233)
    )
  "Association list for CP 850 accent combinations.")

(defvar cp850-accents-minor-mode nil
  "*Non-nil enables CP 850 Accents mode.
Setting this variable makes it local to the current buffer.
See `cp850-accents-mode'.")
(make-variable-buffer-local 'cp850-accents-minor-mode)

(defun cp850-accents-accent-key (prompt)
  "Modify the following character by adding an accent to it."
  ;; Pick up the accent character.
  (if cp850-accents-minor-mode
      (cp850-accents-compose prompt)
    (char-to-string last-input-char)))

(defun cp850-accents-compose-key (prompt)
  "Modify the following character by adding an accent to it."
  ;; Pick up the accent character.
  (let ((combined (cp850-accents-compose prompt)))
    (if unread-command-events
	(let ((unread unread-command-events))
	  (setq unread-command-events nil)
	  (error "Characters %s and %s cannot be composed"
		 (single-key-description (aref combined 0))
		 (single-key-description (car unread)))))
    combined))

(defun cp850-accents-compose (prompt)
  (let* ((first-char last-input-char)
	 ;; Wait for the second key and look up the combination.
	 (second-char (if (or prompt
			      (not (eq (key-binding "a")
				       'self-insert-command)))
			  (progn
			    (message "%s%c"
				     (or prompt "Compose with ")
				     first-char)
			    (read-event))
			(insert first-char)
			(prog1 (read-event)
			  (delete-region (1- (point)) (point)))))
	 (entry (assoc (list first-char second-char) cp850-accents-list)))
    (if entry
	;; Found it: delete the first character and insert the combination.
	(concat (list (nth 1 entry)))
      ;; Otherwise, advance and schedule the second key for execution.
      (setq unread-command-events (list second-char))
      (vector first-char))))

(or key-translation-map (setq key-translation-map (make-sparse-keymap)))
;; For sequences starting with an accent character,
;; use a function that tests cp850-accents-minor-mode.
(define-key key-translation-map "'"  'cp850-accents-accent-key)
(define-key key-translation-map "`"  'cp850-accents-accent-key)
(define-key key-translation-map "^"  'cp850-accents-accent-key)
(define-key key-translation-map "\"" 'cp850-accents-accent-key)
(define-key key-translation-map "~" 'cp850-accents-accent-key)
(define-key key-translation-map "/" 'cp850-accents-accent-key)

;; It is a matter of taste if you want the minor mode indicated
;; in the mode line...
;; If so, uncomment the next four lines.
;; (or (assq 'cp850-accents-minor-mode minor-mode-map-alist)
;;     (setq minor-mode-alist
;; 	  (append minor-mode-alist
;; 		  '((cp850-accents-minor-mode " CP850-Acc")))))

;;;###autoload
(defun cp850-accents-mode (&optional arg)
  "Toggle CP 850 Accents mode, in which accents modify the following letter.
This permits easy insertion of accented characters according to codepage 850.
When Cp850-accents mode is enabled, accent character keys
\(`, ', \", ^, / and ~) do not self-insert; instead, they modify the following
letter key so that it inserts an CP 850 accented letter.

Special combinations: ~c gives a c with cedilla,
~d gives a d with dash.
~< and ~> give guillemets.

With an argument, a positive argument enables CP850-accents mode, 
and a negative argument disables it."

  (interactive "P")

  (if (if arg
	  ;; Negative arg means switch it off.
	  (<= (prefix-numeric-value arg) 0)
	;; No arg means toggle.
	cp850-accents-minor-mode)
      (setq cp850-accents-minor-mode nil)

    ;; Enable electric accents.
    (setq cp850-accents-minor-mode t)))

;;; cp850-acc.el ends here
