/*
 *		UNIXIO.C
 * These functions negotiate with various flavors of the UNIX operating
 *   system for characters and write characters in a barely buffered fashion
 *   to the display.
 * Signals are also handled.
 * 
 * C Durland	Public Domain
 */

/* !!! NOTE !!!
 * Term IO
 *   Works on HP-UX, Sys V and ??HP OSF/1.
 *   Doesn't work on DEC 3100 OSF/1 (why?  it compiles OK).
 * BSD-style IO
 *   Works on BSD, DEC OSF/1, HP OSF/1.
 * I don't know what POSIX is supposed to use (I'll guess TermIO because Sun
 *   uses it).
 * To try and deal with this, I use the USE_TERMIO and USE_BSDIO macros.
 *   Ugly, but I don't know what else to do.
 */

#ifdef __STDC__

#ifdef __hpux			/* for ANSI C on HP-UX */
#define _HPUX_SOURCE
#endif  /* __hpux */

#ifdef _AIX			/* for ANSI C on IBM AIX */
#define _ALL_SOURCE
#endif  /* _AIX */

#ifdef __apollo			/* for ANSI C on Apollo BSD */
#define apollo	    /* Apollo Ansi C needs this to make <sys/ioctl.h> work */
#endif	/* __apollo */

#endif	/*  __STDC__ */

#include <stdio.h>
#include "me2.h"
#include "term.h"
#include "config.h"

	/* CTRL is part of the ED lib (ed.h, included in me2.h).  I don't
	 *   use it here.  It messes up ttychars.h (included by ioctl.h)
	 */
#undef CTRL

#if SYSV_OS || POSIX_OS
#define USE_TERMIO 1
#endif

#if BSD_OS
#define USE_BSDIO 1
#endif

#if USE_TERMIO

#include <termio.h>

struct termio ostate;		/* saved tty state */
struct termio nstate;		/* values for editor mode */

#endif


#if USE_BSDIO

#include <sgtty.h>		/* for stty/gtty functions */
#include <sys/ioctl.h>		/* to get at the typeahead */

struct sgttyb ostate;		/* saved tty state */
struct sgttyb nstate;		/* values for editor mode */
struct tchars otchars;		/* Saved terminal special character set */
struct tchars ntchars =		/* A lot of nothing */
	{ 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };
#endif

static int event_waiting();
static void set_signals();

/* ******************************************************************** */
/* ***************** Opening and Closing the Terminal ***************** */
/* ******************************************************************** */

/*
 * This function is called once to set up the terminal device streams.
 */
ttopen()
{

#if USE_TERMIO			/* use terminfo(7) */
  ioctl(0,TCGETA,&ostate);		/* save old state */
  ioctl(0,TCGETA,&nstate);		/* get base of new state */
/*  nstate.c_cflag |= CS8;		/* */
  nstate.c_lflag &= ~(ICANON | ISIG | ECHO | ECHOE | ECHOK | IEXTEN);
  nstate.c_iflag &=
	~(ICRNL | INLCR | IUCLC | IXON | /*IXOFF |*/ IXANY | BRKINT);
  nstate.c_cc[VMIN]  = 1;
  nstate.c_cc[VTIME] = 0;
  ioctl(0,TCSETA,&nstate);
  ioctl(0,TCXONC,1);			/* counter act any XOFF */
#endif

#if USE_BSDIO
  gtty(0,&ostate);			/* save old state */
  gtty(0,&nstate);			/* get base of new state */
  nstate.sg_flags |= RAW;
/*  nstate.sg_flags &= ~(ECHO | CRMOD);	/* no echo for now... */
nstate.sg_flags &= ~(ECHO | CRMOD | TANDEM);	/* no echo for now... */
  stty(0, &nstate);			/* set mode */
  ioctl(0, TIOCGETC, &otchars);		/* Save old characters */
  ioctl(0, TIOCSETC, &ntchars);		/* Place new character into K */
#endif

  set_signals();
}

/*
 * This function gets called just before we exit ME.
 */
ttclose()
{
#if USE_TERMIO
  ioctl(0,TCSETA,&ostate);
#endif

#if USE_BSDIO
  stty(0,&ostate);
  ioctl(0,TIOCSETC,&otchars);		/* Place old character into K */
#endif
}

/* ******************************************************************** */
/* *************************** Terminal I/O *************************** */
/* ******************************************************************** */

/*
 * Write a character to the display.
 * Use the rawest console output routine that handles backspace, \r and \n.
 */
t_putchar(c) unsigned char c; { fputc(c,stdout); }

/*
 * Flush terminal buffer. Does real work when the terminal output is
 *   buffered up.
 */
t_flush() { fflush(stdout); }

/*
 * Read a character from the terminal, performing no editing and doing no
 *   echo at all.
 * Map terminal dependent things (like softkeys and other things ME doesn't
 *   or can't know about) to ME keys.  Do *NOT* map control keys.
 * In general, try just to read a character and return it.
 * Note:
 *   On Unix systems, system calls (like read()) can be interrupted.  In
 *     this case, read() will return -1 and errno will be set to EINTR.
 */
#define TICK 1
EKeyCode t_getchar()
{
  extern int doing_nothing;	/* in main.c */
  extern int errno;

  unsigned char c;
  int n;
  long int tick;
  EKeyCode keycode;

	/* !!! PCKB and HPKB need to check for read() being interrupted */
#if PCKB		/* PC "style" keyboard: high bit set => special key */
  while (TRUE)
  {
    read(0,&c,1);
    if ((c & 0x80) == 0) return (EKeyCode)c;
    if (map_key(c & 0x7F,&keycode)) return keycode;
  }
#else
#if HPKB		/* HP style: escape sequences => special keys */
  extern EKeyCode que[];			/* in hpkmap.c */
  extern int que_ptr, que_siz;		/* in hpkmap.c */

  if (que_siz) { que_siz--; return que[que_ptr++]; }
  read(0,&c,1);
  if (c == TRIGGER && map_key(&keycode)) return keycode;
  return (EKeyCode)c;
#else			/* none of the above: generic: no special keys */

  tick = 0;
  while (TRUE)
  {
    while (doing_nothing && !event_waiting(TICK))	/* no key waiting */
      do_idle_hook(tick += TICK);

    n = read(0,&c,1);
    if (0 < n) return (EKeyCode)c;	/* the expected case */

	/* Nothing read, error or (most likely) a signal.
	 * Since it might be a signal that something needs to be done and we
	 *   aren't anything else right now, try to process any hooks.
	 * Notes:
	 *   On some OSs, read() is automatically restarted so we will never
	 *     get this case.
	 *   If a connection dies, read() fails to I need to check for that
	 *     case to avoid a infinite loop.  Actually, I shouldn't ever
	 *     get this case because SIGHUP or whatever will exit ME.
	 */
    process_hooks();
  }

#endif
#endif
}

    /* Ask the window/screen size how big it is.
     * Input:
     *   row:  Pointer to a int
     *   col:  Pointer to a int
     * Munges:
     *   row:  The number of rows on the window or screen is put here.
     *   col:  The number of columns on the window or screen is put here.
     * Returns:
     *   TRUE:  If I know how to do this and row,col updated.
     *   FALSE: Not supported or got bogus data.  row, col not changed.
     * Notes:
     *   According to the BSD tty(4) man page:  If, for some unknown (to me)
     *     reason, the terminal has bogus data, the winsize structure is
     *     zeroed.  On Apollo, this happens in special cases.  ws_xpixel and
     *     ws_ypixel always seem to be zero (probably something to do with
     *     my terminal windows).
     */
int get_size_from_window(row, col) int *row, *col;
{
#ifdef TIOCGWINSZ

  struct winsize term_stats;			/* <sys/ioctl.h> */

	/* Get the terminal size */
  if (-1 == ioctl(0, TIOCGWINSZ, &term_stats) || 0 == term_stats.ws_row)
	return FALSE;		/* something bad happened */

  *row = term_stats.ws_row;
  *col = term_stats.ws_col;

  return TRUE;

#else

  return FALSE;

#endif
}

/* ******************************************************************** */
/* ***************************** Waiting ****************************** */
/* ******************************************************************** */

/* Note:  need 1 or 2 routines to check to see if a key is in the input que.
 *   One routine is needed if it is reliable and fast or you need a fast
 *   routine (for HP escape sequence parsing) and one thats fast (for
 *   interrupting a display update).
 * Sync these up with config.h.
 */

#if BSD_OS || POSIX_OS || AIX_OS
#include <sys/time.h>

#else	/* SYSV_OS */

#include <time.h>
#endif

   /* Wait for a key to be pressed for no more than sec seconds.
    * Use 0 seconds to check to see if a key is in the input que.
    */
wait_for_key(sec)
{
  int x = 1;		/* stdin */
  struct timeval timeout;

  timeout.tv_sec = sec; timeout.tv_usec = 0;
  return select(1, &x, 0, 0, &timeout);
}


    /* Check to see if the process server has anything for us OR there is a
     *   key waiting.  If no process server, this is the same as
     *   wait_for_key(sec).
     */
extern int client_socket;		/* in process.c */
extern void process_talks();		/* in process.c */

#define MASK(fd)	(1 << (fd))
#define STDIN		0

static int event_waiting(sec)
{
  int select_mask;
  int biggest_fd, fd, n, process_mask;
  struct timeval timeout;

  process_mask = 0;
  if (client_socket != -1) process_mask = MASK(client_socket);
  select_mask = (MASK(STDIN) | process_mask);

  biggest_fd = imax(STDIN,client_socket) + 1;

  timeout.tv_sec = sec; timeout.tv_usec = 0;
  n = select(biggest_fd,&select_mask,0,0,&timeout);

  if (select_mask & process_mask)
  {
    call_me_sometime(PROCESS_HOOK,process_talks);
    process_hooks();
  }

  return (select_mask & MASK(STDIN));	/* TRUE if a key is waiting */
}


#if 0
/* key_waiting:
 * Check to see if any characters are already in the keyboard buffer.
 * This is reliable but slows things down if called much.
 * This HAS to work.  Sometimes used to parse (HP like) keyboard generated
 *   escape sequences.
 * Note: FIONREAD is no good on HPUX.
 * Not used anymore???
 */
key_waiting()
{
#if BSD_OS		/* since HPUX defines but doesn't use FIONREAD */
  long int x;
  return((ioctl(0,FIONREAD,&x) < 0) ? 0 : x);
#else
  int x = 1, y = 0;
  struct timeval timeout;

  timeout.tv_sec = 0; timeout.tv_usec = 90000;		/* .09 sec */
  return select(1,&x,&y,&y,&timeout);
#endif
}
#endif

/* ******************************************************************** */
/* ***************************** Signals ****************************** */
/* ******************************************************************** */

    /* Notes:
     *   On HP-UX, a signal() needs to be reset after it is caught.  This
     *   happens automatically on BSD.
     */

#include <signal.h>

#ifdef SIGWINCH

static void sig_do_the_resize();

    /* The screen has been resized.
     * WARNING:  Should NOT actually resize the display in a signal handler
     *   because ME could be in the middle of doing all kinds of things and
     *   this could really mess things up (I don't know for sure but I don't
     *   want to have to debug that case!) or the signal might arrive before
     *   the display system has been properly initialized.
     * Result:
     *   Tell the main loop to call us when it can.
     */
static void sig_resize(dummy)
{
  signal(SIGWINCH, SIG_IGN);
  call_me_sometime(RESIZE_HOOK,sig_do_the_resize);
}

static void sig_do_the_resize()
{
  int row, col;

  if (get_size_from_window(&row, &col))
  {
    display_resized(row,col);
    update();		/* force the screen to be redrawn */
  }
  signal(SIGWINCH, sig_resize);		/* reset the signal */
}

#endif

    /* An external process has told ME that it has some info.
     */
static void sig_expro(foo)
{
  extern void process_talks();		/* in process.c */

  signal(SIGUSR1, SIG_IGN);
  call_me_sometime(PROCESS_HOOK,process_talks);
  signal(SIGUSR1, sig_expro);		/* reset the signal */
}

  /* Somebody, somewhere, told me to bite the big one.  Well, I can take a
   *   hint.
   * If there are any modified buffers, too bad.  I suppose I could have a
   *   ME-quit-hook.
   */
static void sig_die(dummy)
{
  mlwrite("ME terminated via signal.\r\n");
  exit_ME(1);
}

  /* Put ME into the background.  This can be invoked via signal or called
   *   from ME.  This means I gotta be careful when I send myself a signal.
   */
void sig_pause(dummy)
{
  close_display(TRUE);	     /* restore terminal for normal user interface */

  kill(getpid(),SIGSTOP);	/* hello background.  Can't catch SIGSTOP. */

  open_display();		/* set up terminal for editor */
    /* ME might not be in the keywait loop when this is called (because of
     *   an external signal) and so might not redraw the screen until a key
     *   is pressed.
     */
  update();		/* force the screen to be redrawn */
}

  /* Setup the signals I'm going to catch:
   *   Signal	Action
   *   SIGTSTP	Put ME into the background.
   *   SIGTERM	Clean up and exit.
   *   SIGHUP	Clean up and exit.
   *   SIGINT	Clean up and exit.
   *   SIGUSR1  Unused.
   *   SIGUSR2  Unused.
   *   SIGWINCH Resize the screen.
   * Ignore these signals:
   *   SIGCONT  sig_pause() will take care of everything it would do.
   */
static void set_signals()
{
  signal(SIGTSTP,sig_pause);
  signal(SIGCONT,SIG_IGN);

  signal(SIGTERM,sig_die);
  signal(SIGHUP, sig_die);
  signal(SIGINT, sig_die);

#ifdef SIGWINCH
  signal(SIGWINCH, sig_resize);
#endif

  signal(SIGUSR1, sig_expro);
  signal(SIGUSR2, SIG_IGN);
}

/* ******************************************************************** */
/* ******************************* Misc ******************************* */
/* ******************************************************************** */

#include "driver.h"

static MouseInfo mouse_info;	/* !!!??? initialize? */

MouseInfo *get_mouse_info()
{
  return &mouse_info;
}
