/*
 *	TCAP.C: TermCap and TermInfo stuff
 *
 * C Durland	Public Domain
 */

/* Notes:
 *   t_nrow:  This is the number of rows on the display MINUS 1.  The minus
 *     1 is to leave a row for the mini buffer.  I don't know if this is a
 *     good idea (because everybody has to remember another little detail)
 *     but it has been that way forever and I don't think it would be worth
 *     the gain to change it.
 */

#include <stdio.h>
#include "me2.h"
#include "driver.h"

#define BLACK_FG	1	/* black characters on a white background */
#define WHITE_FG	0	/* white characters on a black background */

#if ATARI
#define NROW	25	/* Atari default number of row (if not in termcap) */
#else
#define NROW	24	/* default number of row (if not in termcap) */
#endif
#define NCOL	80	/* default number of columns (if not in termcap) */

#define BEL	0x07	/* ^G */

#define TCAPSLEN 315	/* long enough to hold my copy of the term info */

extern char *tgoto();
extern char *strcpy(), *strcat(), *strchr();

	/* escape sequence and terminfo pointers */
static char *CM, *CL, *CE, *CD, *SO,*SE, *TI,*TE;
char PC;		/* used by termcap library, initialize to '\0' */

int t_nrow = NROW - 1, t_ncol = 0, tcolor = WHITE_FG, mcolor = BLACK_FG;

extern void t_putchar();
static void get_term_info(), putpad();

	/* Prepare the terminal for io.  Save the old state. */
void t_open()
{
  if (t_ncol == 0)	/* only do this once */
  {
    que_version_info("display-driver", "termcap", (char *)NULL);
    get_term_info();
  }
  ttopen();
  if (TI) putpad(TI);
}

	/* Restore the terminal. */
void t_close()
{
  if (TE) putpad(TE);
  ttclose();
}

	/* move the cursor to (row,col) (zero relative). */
void t_move(row,col) register int row, col; { putpad(tgoto(CM,col,row)); }

	/* Clear from cursor to end of line. */
void t_eeol() { putpad(CE); }

	/* Clear from cursor to end of screen or just clear the screen. */
void t_eeop() { putpad(CD); }

	/* Honk the horn.  beeper has the volume (0-100%).
	 * If the terminal has visable bell set, sending it a BEL will cause
	 *   a flash and not a beep.
	 */
void t_beep()
{
  extern int beeper;

  if (beeper) { t_putchar(BEL); t_flush(); }
}

static int hpterm;	/* if strange HP type video attributes */

	/* 1: current_color may not be correct - force color.
	 * 2: switch color (if not already current_color).
	 * 3: same as 2 but at beginning of line.
	 */
void setcolor(color,x)
{
  static int current_color = -1;  /* initialize to something != [tm]color */

  if (SO == NULL) return;		/* inverse video not available */
  if (hpterm) { if (x != SWITCH_COLOR1) return; }
  else if (current_color == color && x != FORCE_COLOR) return;
  switch (current_color = color)
  {
    case 0: putpad(SE); break;	/* turn off inverse video */
    case 1: putpad(SO); break;	/* turn on inverse video */
  }
}

    /* "forground:background"
     *   ":"      => no change
     *   "color"  => only change foreground
     *   "color:" => only change foreground
     *   ":color" => only change background
     */
int do_color(which, color, set) char *color;
{
  int *c;

  switch(which)
  {
    case TEXT_COLOR:     c = &tcolor; break;	/* text color */
    case MODELINE_COLOR: c = &mcolor; break;	/* modeline color */
    case CURSOR_COLOR:				/* cursor color */
    case CURSOR_SHAPE:		/* Changing the cursor not supported */
      if (!set) strcpy(color, "unknown");
      return FALSE;
    default: return FALSE;			/* boo boo */
  }
  if (set)
  {
    char buf[100], *ptr;
    int x = *c, z;

    strcpy(buf, color); lowercase(buf);
    if (ptr = strchr(buf, ':')) *ptr = '\0';
    if (0 == strcmp(buf,"white"))   z = WHITE_FG;
    else
      if (0 == strcmp(buf,"black")) z = BLACK_FG;
      else
        return FALSE;		/* don't reconize color, no change */

    return (x != (*c = z));
  }
  else
    strcpy(color, (*c == WHITE_FG) ? "White:Black" : "Black:White");

  return FALSE;
}

/* ******************************************************************** */
/* ************* Below: support routines not needed elsewhere ********* */
/* ******************************************************************** */

static void putpad(str) char *str; { tputs(str, 1, t_putchar); }

    /* Dig terminal info out of the termcap entry.
     * Note:
     *   I call get_size_from_window() because on some systems tgetnum()
     *     doesn't seem to look at the LINES and COLUMNS environment
     *     variables.
     */
static void get_term_info()
{
  extern char *getenv(), *strcpy(), *strcat(), *tgetstr();

  static char tcapbuf[TCAPSLEN];
  char *t, *p, tcbuf[1024], *tv_stype, err_str[72];
  int x, y;

  if ((tv_stype = getenv("TERM")) == NULL)
  {
#if ATARI
    tv_stype = "atari";
#else
    puts("Environment variable TERM not defined!");
    exit(1);
#endif
  }
  if (tgetent(tcbuf,tv_stype) != 1)
  {
    puts(strcat(strcpy(err_str,"Unknown terminal type: "),tv_stype));
    exit(1);
  }

  p = tcapbuf;				/* pointer to local storage */

  if (t = tgetstr("pc",&p)) PC = *t;	/* pad_char: ???how used */

  CD = tgetstr("cd",&p);	/* clr_eos: clear to end of display */
  CE = tgetstr("ce",&p);	/* clr_eol: clear to end of line */

  CM = tgetstr("cm",&p);	/* cursor_address: cursor motion */
  TI = tgetstr("ti",&p);	/* enter_ca_mode:  init CM mode */
  TE = tgetstr("te",&p);	/* exit_ca_mode:   uninit CM mode */

  if (get_size_from_window(&x, &y))
  {
    t_nrow = x -1;
    t_ncol = y;
  }
  else		/* dig the info out of the termcap entry */
  {
				/* lines:   lines on screen */
    if ((x = tgetnum("li")) > 0) t_nrow = x -1;
				/* columns: screen columns */
    t_ncol = (x = tgetnum("co")) > 0 ? x : NCOL;
  }

  SO = tgetstr("so",&p);			   /* enter_standout_mode */
  if ((SE = tgetstr("se",&p)) == NULL) SO = NULL;  /* exit_standout_mode */
  hpterm = tgetflag("xs");   /* ceol_standout_glitch: HP type inverse video */

  if (CD == NULL || CM == NULL || CE == NULL)
  {
    puts("Incomplete termcap entry.");
    exit(1);
  }
  if (p >= &tcapbuf[TCAPSLEN])
  {
    puts("Terminal description too big!");
    exit(1);
  }
}
