/*
 *  PCIO.C: terminal io for IBM PC type things
 *  C Durland	 Public Domain
 *  John Burnell Added support for the grey keypad, C for some of the asm
 *		 routines.  3/92
 *  John Burnell Added some more support for the grey keypad.  5/92
 */

/*
 * The functions in this file negotiate with the operating system for
 *   characters, and write characters in a barely buffered fashion on the
 *   display.
 * Notes:
 *   If using FASTVIDEO, also link in pcfv.c.
 *   If not using FASTVIDEO, use ansi.c or something else.
 */

#include <stdio.h>
#include <dos.h>
#include <const.h>
#include "ed.h"
#include "term.h"
#include "config.h"
#include "driver.h"

static char read_kbd = 0x00, check_kbd = 0x01;		/* for DOS calls */
static int break_on, extended_keyboard = FALSE;

/*
 * This function is called to set up the terminal device streams.
 */
void ttopen()
{
  union REGS in, out;

		/* check to see if control checking is on */
  in.h.ah = 0x33; in.h.al = 0x00; int86(0x21,&in,&out);
  break_on = out.h.dl;

  bdos(0x33,0,1);		/* turn off control C checking */

  if (*(char *)0x400096L & 0x10)	/* we have an extended keyboard */
  {
    extended_keyboard = TRUE;
    read_kbd = 0x10, check_kbd = 0x11;
  }
}

/*
 * This function gets called just before we go back home to the command
 *   interpreter.
 */
void t_close()
{
  if (break_on) bdos(0x33,1,1);		/* turn on control C checking */
#if FASTVIDEO
  set_cursor_size(TRUE);
#endif
}

/*
 * Write a character to the display.
 * Use the rawest console output routine that handles backspace, \r and \n.
 * On MS-DOS terminal I/O is unbuffered, so we just write the byte out.
 */
void t_putchar(c) unsigned char c;
{
  bdos(6,c,0);
}

/*
 * Flush terminal buffer.  Does real work where the terminal output is
 *   buffered up.  A no-operation on systems where byte at a time terminal
 *   I/O is done.
 */
void t_flush() {}

/*
 * Read a character from the terminal, performing no editing and doing no
 *   echo at all.
 * Map terminal softkeys, etc to ME keys.  Do *NOT* map control keys.
 */
#define TICK 1
EKeyCode t_getchar()
{
  extern int doing_nothing;	/* in main.c */

  long int tick;
  EKeyCode keycode;
  union REGS in, out;

  tick = 0;
  while (TRUE)
  {
    while (doing_nothing && !wait_for_key(TICK))	/* no key waiting */
      do_idle_hook(tick += TICK);

    in.h.ah = read_kbd; int86(0x16,&in,&out);

	/* Distinguish the grey / * - + and <Enter> */
    if (out.h.ah >= 55 && out.h.al != 0)  /* do this in 2 goes to save time */
      if (out.h.ah == 55 || out.h.ah == 74 || out.h.ah == 78 ||
	  out.h.ah == 0xE0)
      {
	if (PCmap_key(TRUE, (0xE000 | out.h.al), &keycode)) return keycode;
	continue;
      }

	 /* Check for an extended key, pass AL to distinguish grey keypad */
    if (out.h.al == 0 || (out.h.al == 0xE0 && out.h.ah != 0))
    {
      if (PCmap_key(extended_keyboard, out.h.ah | (out.h.al << 8), &keycode))
	return keycode;
    }
    else
      return (EKeyCode)out.h.al;
  }
  /* NOTREACHED */
}


#define Z 0x40		/* 8086 zero flag */

/*
 * Check to see is a key is waiting.
 * Used to interrupt processing if input is pending or to detect if a soft
 *   key has been hit (the terminal sends multiple characters for each
 *   softkey).
 * Don't use function B because it checks for ^C.
 */
int keywaiting()
{
  union REGS r;

  r.h.ah = check_kbd; int86(0x16,&r,&r);

  return !(r.x.flags & Z);
}

    /* Return the number of ticks from a start time.
     * Ticks are .01 second
     * Input:
     *   init : TRUE  : Restart the clock
     *          FALSE : Return elapsed time since start
     * Returns:
     *   Number of ticks since last init.
     * Notes:
     *   With 32 bits and .01 second ticks, the range is 248 days.  However,
     *     this routine is only good for 24 hours because I only worry about
     *     one day rollover.
     *   d = 24h, h = 60m, m = 60s, s = 100t, t = 1
     */
static long int get_time(init)
{
  static int day;

  long int time = 0;
  union REGS r;

  r.h.ah = 0x2A; int86(0x21,&r,&r);		/* read day, month, year */
  if (init) day = r.h.al;			/* day of month */
  else if (r.h.al != day) time = 8640000L;	/* day changed */
  r.h.ah = 0x2C; int86(0x21,&r,&r);		/* read hh:mm:ss.tt */
  time +=				/* convert hh:mm:ss.tt to ticks */
	((r.h.ch*60L + r.h.cl)*60L + r.h.dh)*100L + r.h.dl;

  return time;
}

   /* Wait for a key to be pressed for no more than sec seconds.
    * Use 0 seconds to check to see if a key is in the input que.
    * Warning:  This rouine is only good for 32K seconds (about 9 hours).
    *   If sec is a long the limit is still 24 hours.
    */
wait_for_key(sec)
{
  long int times_up;

  if (sec == 0) return keywaiting();

  times_up = get_time(TRUE) + 100*sec;
  do
  {
    if (keywaiting()) return TRUE;
  } while (get_time(FALSE) < times_up);

  return FALSE;
}

#if FASTVIDEO		/* Also use pcfv.c */
  /* Things to note about FastVideo:
   *   putline() will do all of the writing to the text area (except for
   *     help messages).
   *   t_eeol() will only be called for the message line or for help
   *     messages.
   *   t_eeop() need only clear the message line.  It should use t_eeol() to
   *     do this.
   *   t_putchar() will only write to the message line, help messages or
   *     (puts "message").
   * So, this means that the message line is really seprate from the text
   *   areas and t_putchar() and t_eeol() should use the same color (others
   *   will use tcolor and mcolor).  Default for the message line would be
   *   tcolor.  Help and (puts ...)  messages will be written in message
   *   color (which is (I hope) reasonable).
   * PC note:  Due to the way the BIOS is done, it is easier to have
   *   t_putchar() and t_eeol() use the BIOS character attribute as the
   *   color.
   */

extern void t_eeol();

int t_nrow = 24, t_ncol = 0;

 /*
  * Called to set up the terminal.  Also called after a spawn.
  */
void t_open()
{
  ttopen();
  if (t_ncol == 0) fv_init();		/* only call fv_init() once */
  set_cursor_size(FALSE);
}

 /* Erase to end of page or screen.
  * For fast video, each line of the screen (except for the message line)
  *   will be completely filled by putline() so all this routine needs to
  *   clear is the message (last) line of the screen.
  */
void t_eeop() { movecursor(t_nrow,0,TRUE); t_eeol(); }

  /* Ring the bell.  beeper controls the volume:  0 (don't beep), n (some
   *   volume value).
   */
void t_beep()
{
  extern int beeper;

  if (beeper) t_putchar(BEL);
}

#endif /* FASTVIDEO */

/* ******************************************************************** */
/* ******************************* Misc ******************************* */
/* ******************************************************************** */

static MouseInfo mouse_info;	/* !!!??? initialize? */

MouseInfo *get_mouse_info()
{
  return &mouse_info;
}
