/*
 * Fast video support for IBM PC like displays.
 * Use with pcio.c and FASTVIDEO.
 * For faster display speed, set USE_ASM (if your compile supports inline
 *   asm).
 * Routines:
 *   fv_init()
 *   putline(row,buf,attr) int row,attr; char *buf;
 *   void t_move(row,col) int row, col;
 *   void t_eeol()
 * Notes:
 *   The only routine that will really speed things up is putline().  I
 *     don't think the other stuff is called enough to have much effect.
 *   I use the large [data] memory model so (char *)'s are (char far *)'s.
 * C Durland	Public Domain
 * J Burnell	C for some of the asm routines.  3/92
 * C Durland	Added asm stuff (I used Borland C++ 2.0).  6/93
 * C Durland	Added named color stuff and changing cursor size.  9/93
 */

#include <const.h>
#include "driver.h"

#define USE_ASM		1	/* 1 if compiler supports asm */

extern int t_ncol, t_nrow;	/* in pcio.c */

int tcolor, mcolor;		/* text and modeline colors */

static char *video_addr;
static int phys_cols;		/* Because t_ncol can change */


void putline(row, text, attr)	/* note: row is 0 relative */
  int row,attr; char *text;
{
  register char *screen = video_addr + 2*row*phys_cols;

#if !USE_ASM
  register int n = t_ncol;

  while (n--)
  {
    *screen++ = *text++;	/* character */
    *screen++ = attr;		/* attribute */
  }
#else
	/* Notes:
	 *   This routine speeds up the repaint by a noticable amount.
	 *     However, I'm not convinced its all that much.
	 */
  asm push DS

  asm mov  AH,byte ptr attr
  asm mov  CX,t_ncol

  asm lES  DI,screen
  asm lDS  SI,text

  asm cld
write_character:
  asm lodsb		/* AL = *text++ */
  asm stosw		/* *screen++ = character, *screen++ = attr */
  asm loop write_character

  asm pop DS
#endif	/* !USE_ASM */
}

#include <dos.h>

void t_move(row,col) int row, col;	/* move cursor to (row,col) */
{
  union REGS rg;

  rg.h.ah = 2;		/* set cursor position function code */
  rg.h.dl = col;
  rg.h.dh = row;
  rg.h.bh = 0;		/* set screen page number */
  int86(0x10, &rg, &rg);
}

static int
  BIOS_color,
  BIOS_cursor_start, BIOS_cursor_end,
  ME_cursor_start, ME_cursor_end;

void fv_init()
{
  union REGS x;

	/* Try and figure out addr of video */
  x.h.ah = 0xF; int86(0x10,&x,&x);
  if (x.h.al == 7) video_addr = (char *)0xB0000000L;	/* mono */
  else video_addr = (char *)0xB8000000L;		/* color */

	/* Figure out the BIOS color.  I'll use that to write the message
	 *   line.
	 */
  t_move(0,0);
  t_putchar(' ');	/* Let the BIOS write a character and attr */
  t_move(0,0);
	/* read character attr */
  x.h.ah = 0x8; x.h.bh = 0;
  int86(0x10,&x,&x);
  BIOS_color = x.h.ah;
  if (BIOS_color == 0) BIOS_color = 7;	/* make sure cursor displays */

	/* Save the cursor size */
  x.h.ah = 0x3; x.h.bh = 0;
  int86(0x10,&x,&x);
  BIOS_cursor_start = x.h.ch;
  BIOS_cursor_end   = x.h.cl;
  
	/* Figure out the number of columns (BIOS 40:4A == columns) */
  t_ncol = phys_cols = *((unsigned char *)0x0040004AL);

	/* Figure out the number of rows.
	 * For EGA type displays, the number of rows is in ram.  For CGA, it
	 *   is fixed.
	 */
  t_nrow = 24;		/* default */

  x.x.ax = 0x1200;
  x.x.bx = 0x0010;
  x.x.cx = 0xFFFF;
  int86(0x10,&x,&x);
  if (x.x.cx != 0xFFFF)		/* is at least EGA */
  {
    que_version_info("display-driver", "EGA", (char *)NULL);

    t_nrow = *((unsigned char *)0x00400084L);		/* 40:84 == rows-1 */
	/* Create a cursor that doesn't disappear on underlines "_" */
    ME_cursor_start = 5; ME_cursor_end = 7;
	/* text, modeline color defaults for color displays */
    do_color(TEXT_COLOR,    "green:black",TRUE);
    do_color(MODELINE_COLOR,"lightred:blue",TRUE);
  }
  else		/* Something like CGA */
  {
	/* Create a cursor that doesn't disappear on underlines "_" */
    ME_cursor_start = 5; ME_cursor_end = 5;
	/* text, modeline color defaults for mono or color (ugly on both) */
    do_color(TEXT_COLOR,    "lightgrey:black",	TRUE);
    do_color(MODELINE_COLOR,"brown:black",	TRUE);
  }
}

#define SPACE ' '

void t_eeol()		/* erase to end of line */
{
  register int n = t_ncol;
  int CurCol;		/* current column cursor */
  int CurRow;		/*	   row	*/
  union REGS rg;
  register char *ScreenAddr;		/* address to write to */

	/* find the current cursor position */
  rg.h.ah = 3;	
  rg.h.bh = 0;		/* current page */
  int86(0x10, &rg, &rg);
  CurCol = rg.h.dl;
  CurRow = rg.h.dh;

  ScreenAddr = video_addr + CurRow*phys_cols*2 + CurCol*2;

	/* Note:  You could probably save time by writing
	 *   (SPACE | (BIOS_color << 8)) to a short int pointer but I don't
	 *   think it will make any difference.
	 */
  for (n = phys_cols - CurCol; n--; )
  {
    *ScreenAddr++ = SPACE;		/* character */
    *ScreenAddr++ = BIOS_color;		/* attribute */
  }
}

/* ******************************************************************** */
/* ****************************** Cursor ****************************** */
/* ******************************************************************** */

extern char *strcpy(), *strcat(), *strchr();

void set_cursor_size(to_BOIS)
{
  union REGS x;

  if (to_BOIS)  { x.h.ch = BIOS_cursor_start; x.h.cl = BIOS_cursor_end; }
  else		{ x.h.ch = ME_cursor_start;   x.h.cl = ME_cursor_end; }
  x.h.ah = 0x1;
  int86(0x10,&x,&x);
}

static void do_cursor(cursor_shape, set) char *cursor_shape;
{
  if (set)
  {
    char *ptr, buf[100];

    strcpy(buf, cursor_shape);
    if (ptr = strchr(buf, ':')) *ptr++ = '\0';
    ME_cursor_start = (atoi(buf) & 0xFF);
    ME_cursor_end = ptr ? (atoi(ptr) & 0xFF) : 7;
    set_cursor_size(FALSE);

    return;
  }

  strcpy(cursor_shape, "unknown");
}

/* ******************************************************************** */
/* ****************************** Color ******************************* */
/* ******************************************************************** */

#define RGB_BITS(bits)		((bits) & 0xF)
#define FOREGROUND_BITS(bits)	((bits) & 0xF)
#define BACKGROUND_BITS(bits)	(((bits) >> 4) & 0x7)
#define MAKE_RGB(fg, bg)	( ((fg) & 0xF) | (((bg) & 7) << 4) )

static char *pc_color[] =
{			/* IRGB   I can be 1 for forground colors only */
  "black",		/* 0000 ==  0 */
  "blue",		/* 0001 ==  1 */
  "green",		/* 0010 ==  2 */
  "cyan",		/* 0011 ==  3 */
  "red",		/* 0100 ==  4 */
  "magenta",		/* 0101 ==  5 */
  "brown",		/* 0110 ==  6 */
  "lightgrey",		/* 0111 ==  7 */
  "darkgrey",		/* 1000 ==  8 */
  "lightblue",		/* 1001 ==  9 */
  "lightgreen",		/* 1010 == 10 */
  "lightcyan",		/* 1011 == 11 */
  "lightred",		/* 1100 == 12 */
  "magenta",		/* 1101 == 13 */
  "yellow",		/* 1110 == 14 */
  "white",		/* 1111 == 15 */
};

static int color_to_rgb(color_name, rgb, foreground)
  char *color_name; int *rgb;
{
  int i, max;

  lowercase(color_name);
  max = foreground ? NITEMS(pc_color) : (NITEMS(pc_color) + 1)/2;
  for (i = 0; i < max; i++)
    if (0 == strcmp(pc_color[i], color_name))
    {
      *rgb = i;
      return TRUE;
    }
  return FALSE;
}

static char *rgb_to_color(rgb)
{
  return pc_color[RGB_BITS(rgb)];
}

    /* "forground:background"
     *   ":"      => no change
     *   "color"  => only change foreground
     *   "color:" => only change foreground
     *   ":color" => only change background
     * Input:
     *   color : name of color.
     *   rgb : current color (incase name of new color is bogus)
     * Returns:
     *   New rgb bits.
     */
static int set_color(color, rgb) char *color;
{
  char *ptr, buf[100];
  int fg, bg;

  strcpy(buf, color);
  if (ptr = strchr(buf, ':')) *ptr++ = '\0';
  if (!color_to_rgb(buf, &fg, TRUE))		fg = FOREGROUND_BITS(rgb);
  if (!ptr || !color_to_rgb(ptr, &bg, FALSE))	bg = BACKGROUND_BITS(rgb);
  return MAKE_RGB(fg, bg);
}

static void get_color(buf,rgb) char *buf;
{
  strcpy(buf, rgb_to_color(FOREGROUND_BITS(rgb)));
  strcat(buf, ":");
  strcat(buf, rgb_to_color(BACKGROUND_BITS(rgb)));
}

int do_color(which, color, set) char *color;
{
  int *c;

  switch(which)
  {
    case TEXT_COLOR:     c = &tcolor; break;		/* text color */
    case MODELINE_COLOR: c = &mcolor; break;		/* modeline color */
    case CURSOR_COLOR:					/* cursor color */
      if (set) return FALSE;	/* can't change cursor color */
      c = &tcolor;		/* it is always the text color */
      break;
    case CURSOR_SHAPE: do_cursor(color, set); return TRUE;
    default: return FALSE;			/* boo boo */
  }
  if (set)
  {
    int x = *c;
    return (x != (*c = set_color(color, x)));
  }
  else get_color(color, *c);

  return FALSE;
}
