/* 
 * os.c : Misc routines that are OS dependent.
 * Routines:
 *   osinit():  Initialize OS dependent stuff.  Called once, very early.
 *   change_directory(dir):  Change the current directory to a new one.
 *     Returns TRUE on success.  If not implemented, return FALSE.  Does not
 *     munge dir.
 *   current_directory(canonize):  Returns a pointer to a string containing
 *     the current directory.  canonize is TRUE if being called for
 *     canonize().
 *   MMopen_code_file(name):  Open a Mutt machine code file.
 *   os_time() : Returns the current time and date.
 *   unique_file_name():  Create a name that can be used to open a file.
 *       Every name must be unique.  Espically important in multiprocess
 *       systems where more than one ME may be running at the same time.
 *     Note:  These file names are ment ONLY to be used when a temporary
 *       file is needed.  They are not ment to out live the ME process that
 *       creates them.  If they do, another ME will probably generate the
 *       same file name at some later date.
 *
 *  C Durland	Public Domain
 *  Atari stuff from Jwahar R. Bammi (bammi@cadence.com).
 */

#include <stdio.h>
#include "me2.h"
#include "config.h"

extern char *strcpy(), *getenv();

#define MUTT_ENV_VAR	"ME3"	/* !!!Name of this version of ME */

/* ******************************************************************** */
/* **************************** Unix Stuff **************************** */
/* ******************************************************************** */
#if UX_OS

#define PSEP ':'		/* search path separater */

   /* !!!Having a fixed size array is asking for trouble.  Especially on
    *   long file name systems.
    */
static char current_dir[NFILEN];
static char *mutt_search_path;

osinit()
{
  static char msp[NFILEN];	/* Filled with '\0'  !!!asking for trouble */

  char *ptr;

  que_version_info("OS", "Unix",
#if __hpux
	"HP-UX",
#endif
#if AIX_OS
	"AIX",
#endif
#if DOMAIN_OS
	"DOMAIN",
#endif
#if SYSV_OS
	"SYSV",
#endif
#if BSD_OS
	"BSD",
#endif
#if POSIX_OS
	"POSIX",
#endif

	(char *)NULL);

  canonize(".", current_dir, (char *)NULL);

  mutt_search_path = getenv(MUTT_ENV_VAR);
#if USE_DEFAULT_MUTT_SEARCH_PATH
  if (!mutt_search_path)
  {
    mutt_search_path = msp;

    if (ptr = getenv("HOME"))	/* ME=$HOME: */
    {
      strcat(msp, ptr);
      strcat(msp, ":");
    }
    strcat(msp, DEFAULT_MUTT_SEARCH_PATH);	/* ME=$HOME:/usr/local/me */
  }
#endif	/* USE_DEFAULT_MUTT_SEARCH_PATH */

  return TRUE;
}

char *current_directory(ignore)
{
  return current_dir;
}

    /* Notes:
     *   Need to arf() the directory so I can be sure that what I copy into
     *     current_dir is a real directory name.  I also need a canonized
     *     name for Mutt programs to use.  If I just chdir(directory) then
     *     arf() and arf screws up (as is happening on Domain/OS), its going
     *     to be a mess.
     */
change_directory(directory) char *directory;
{
  char buf[NFILEN];
  int s = FALSE;

  if (arf(directory, buf) && (s = (0 == chdir(buf)))) strcpy(current_dir, buf);
  return s;
}

    /* Create a unique file name with format:
     * </tmp/>ME<n>.<pid>
     *   n is 0...19999 and then rolls over.	(up to 5 characters)
     *   pid is 1...65535.			(up to 5 characters)
     * Notes:
     *   Unixes with short file name file systems have a max file name
     *     length of 14 characters (to the best of my knowledge) so this
     *     scheme should work on those systems.
     */
char *unique_file_name(name) char *name;
{
  extern char *l_to_a();

  static int n = 0;

  char *ptr;

  strcpy(name,"/tmp");
  if (ptr = getenv("TMP")) strcpy(name,ptr);

  strcat(name, "/ME");

  ptr = l_to_a((long int)n);
  strcat(name,ptr);

  strcat(name,".");

  ptr = l_to_a((long int)getpid());
  strcat(name,ptr);

  n = (n+1) % 20000;

  return name;
}

#endif	/* UX_OS */

/* ******************************************************************** */
/* *********************** MS-DOS Stuff ******************************* */
/* ******************************************************************** */
#if MSDOZ

#define PSEP ';'		/* search path separater */

static char current_dir[80];	/* current directory */
static char *mutt_search_path;

osinit()
{
  que_version_info("OS", "MS-DOS", "", (char *)NULL);

  mutt_search_path = getenv(MUTT_ENV_VAR);
#if USE_DEFAULT_MUTT_SEARCH_PATH
  if (!mutt_search_path) mutt_search_path = DEFAULT_MUTT_SEARCH_PATH;
#endif	/* USE_DEFAULT_MUTT_SEARCH_PATH */

  return TRUE;
}

    /* Notes:
     *   MS-DOS causes problems because the current directory can be changed
     *     without forking (ie on Unix, the only way for ME to change
     *     directories is for ME to do it - no way for another process to
     *     change it).  On DOS, "C-x!  cd A:"  can change the dir without ME
     *     knowing about it.  This is bad.  So, to get around this, I have
     *     to ask the OS every time the current directory needed.
     *     canonize() knows how to do this.
     *   canonize() always asks the OS (on DOS) so I don't need to do
     *     anything if this is being called for canonize().
     */
char *current_directory(called_from_canonize)
{
  if (!called_from_canonize) canonize(".", current_dir, (char *)NULL);

  return current_dir;
}

    /* 
     * Notes:
     *   chdir == bdosptr(0x3B,&buf[2],0) or bdosx(0x3B,&buf[n],0)
     *   setdisk == bdos(0xE, drive, 0)
     */
change_directory(directory) char *directory;
{
  char buf[NFILEN];
  int drive, current_drive;

  if (!arf(directory, buf)) return FALSE;	/* !!!??? */

  current_drive = getdisk();

  drive = buf[0] - 'A';

  setdisk(drive);
  if (drive != getdisk()) return FALSE;		/* couldn't change drive */

  if (chdir(&buf[2]))		/* 0 OK, 3 and -1 are errors, maybe others */
  {
    setdisk(current_drive);
    
    return FALSE;
  }

  return TRUE;
}

void blkmv(to,from,n) register char *to, *from; register unsigned int n;
	{ while (n--) *to++ = *from++; }

    /* Create a unique file name with format:
     * </tmp/>ME<n>.$$$
     *   n is 0...19999 and then rolls over.	(up to 5 characters)
     * Notes:
     *   MS-DOS doesn't multiprocess so only one ME at a time.  This means I
     *     can use a pretty simple scheme.
     */
char *unique_file_name(name) char *name;
{
  extern char *l_to_a();

  static int n = 0;

  char *ptr;

  *name = '\0';
  if (ptr = getenv("TMP")) strcpy(name,ptr);

  strcat(name, MUTT_ENV_VAR);

  ptr = l_to_a((long int)n);
  strcat(name,ptr);

  strcat(name,".$$$");

  n = (n+1) % 20000;

  return name;
}

    /* Return the current time and date.
     * If you can't do this, return zeros and FALSE.
     * Data formats:
     *   month: 1 (January) - 12 (December)
     *   day:   The day of the month (1 being the first day)
     *   year:  For example, 1993
     *   hh : hour : 0 - 23
     *   mm : minutes : 0 - 59
     *   ss : seconds : 0 - 59
     *   weekday:  0 (Sunday) - (6) Saturday
     * Notes:
     *   I could use the same routine as Unix but this is MUCH smaller.
     */

#include <dos.h>

int os_time(month,day,year, hh,mm,ss)
  int *month,*day,*year, *hh,*mm,*ss;
{
  union REGS r;

  r.h.ah = 0x2A; int86(0x21,&r,&r);		/* read day, month, year */
  *month = r.h.dh; *day = r.h.dl; *year = r.x.cx;
	/* r.h.al has the day of the week */
  r.h.ah = 0x2C; int86(0x21,&r,&r);		/* read hh:mm:ss.tt */
  *hh    = r.h.ch; *mm  = r.h.cl; *ss   = r.h.dh;
  return TRUE;
}

#endif	/* MSDOZ */


/* ******************************************************************** */
/* **************************** atariST Stuff ************************* */
/* ******************************************************************** */
#if ATARI

#define PSEP ';'		/* search path separater */

   /* !!!Having a fixed size array is asking for trouble.  Especially on
    *   long file name systems.
    */
static char current_dir[NFILEN];
static char *mutt_search_path;
long _stksize = 16384;

osinit()
{
  static char msp[NFILEN];	/* Filled with '\0'  !!!asking for trouble */

  char *ptr;

  que_version_info("OS", "Atari", "", (char *)NULL);

  canonize(".", current_dir, (char *)NULL);

  mutt_search_path = getenv(MUTT_ENV_VAR);
#if USE_DEFAULT_MUTT_SEARCH_PATH
  if (!mutt_search_path)
  {
    mutt_search_path = msp;

    if (ptr = getenv("HOME"))	/* ME=$HOME: */
    {
      strcat(msp, ptr);
      strcat(msp, ";");
    }
    strcat(msp, DEFAULT_MUTT_SEARCH_PATH);	/* ME=$HOME:/usr/local/me */
  }
#endif	/* USE_DEFAULT_MUTT_SEARCH_PATH */

  return TRUE;
}

char *current_directory(ignore)
{
  return current_dir;
}

    /* Notes:
     *   Same as Unix change_directory()
     */
change_directory(directory) char *directory;
{
  char buf[NFILEN];
  int s = FALSE;

  if (arf(directory, buf) && (s = (0 == chdir(buf)))) strcpy(current_dir, buf);
  return s;
}

    /* Create a unique file name
     */
char *unique_file_name(name) char *name;
{
  extern char *tmpnam();

  return tmpnam(name);
}

#endif	/* ATARI */

/* ******************************************************************** */
/* ************** Unix, MS-DOS and atariST Stuff ********************** */
/* ******************************************************************** */

#if UX_OS || MSDOZ || ATARI

   /* find_file:  Look around the file system for a file.
    * Input:
    *   name:  Name of the file to look for.  May have a leading path.
    *   search_path:  Where to look for name.  If NULL or "" then only the
    *     current directory is searched (if no leading path on name).
    *   callback:  Pointer to routine that checks to see if a file name is
    *     the one being looked for.  Returns TRUE if found, else FALSE.
    * Returns:
    *   Result of callback.
    * Path parsing alg (same as Unix (ksh)):
    *   If name has a slash in it, search_path is not parsed.
    *   Everything between sepraters is a path.
    *   If path starts with ":", the current directory is searched first.
    *   If path ends with ":", the current directory is searched last.
    */
int find_file(name, search_path, callback)
  char *name, *search_path;
  pfi callback;
{
  extern char *strchr();

  char buf[NFILEN +80], *ptr, *qtr;
  int n;

  if (strchr(name,U_SLASH)
#if MSDOZ || ATARI	/* MS-DOS "A:foo" */
      || strchr(name,M_SLASH) || name[1] == ':'
#endif
  )
  {
    return callback(name);
  }

  ptr = search_path;
  if (!ptr) ptr = mutt_search_path;

  while (TRUE)
  {
    qtr = ptr;
    while (*ptr != PSEP && *ptr) ptr++;
    n = ptr - qtr;
    strncpy(buf,qtr,n);
    if (n) buf[n++] = U_SLASH; 
    strcpy(buf + n, name);
    if (callback(buf)) return TRUE;
    if (!*ptr++) break;
  }

  return FALSE;
}

#endif	/* UX_OS || MSDOZ || ATARI */

/* ******************************************************************** */
/* ********************** Unix and atariST Stuff ********************** */
/* ******************************************************************** */

#if UX_OS || ATARI

    /* Return the current time and date.
     * If you can't do this, return zeros and FALSE.
     * Data formats:
     *   month: 1 (January) - 12 (December)
     *   day:   The day of the month (1 being the first day)
     *   year:  For example, 1993
     *   hh : hour : 0 - 23
     *   mm : minutes : 0 - 59
     *   ss : seconds : 0 - 59
     *   weekday:  0 (Sunday) - (6) Saturday
     */

#include <time.h>

int os_time(month,day,year, hh,mm,ss)
  int *month,*day,*year, *hh,*mm,*ss;
{
  int n;
  long t;
  struct tm *lt;

  time(&t); lt = localtime(&t);

  *month = lt->tm_mon+1;  *day = lt->tm_mday;  *year = lt->tm_year + 1900;
  *hh    = lt->tm_hour;	  *mm  = lt->tm_min;	*ss  = lt->tm_sec;
/*  *weekday = lt->tm_wday;	/*  */

  return TRUE;
}

#endif	/* UX_OS || ATARI */

/* ******************************************************************** */
/* ********** Generic Stuff That Doesn't Need To Be Modified ********** */
/* ******************************************************************** */

   /* MMopen_code_file(name) : open a Mutt code (.mco) file.
    * Input:
    *   name: Name of code file to open.  The extension is ".mco".  May have
    *     a leading path.
    * Returns:
    *   FILE * : Pointer to the opened code file.
    *   NULL : Couldn't find the file.
    * Path parsing is the same as Unix (ksh), see find_file().
    */
/* ??? arf the name so I can handle "~?" */
static FILE *mmfile;
static int mmopen(file_name) char *file_name;
{
  FILE *fptr;

  if (!(mmfile = fopen(file_name,FOPEN_BINARY))) return FALSE;
  return TRUE;
}
FILE *MMopen_code_file(name) char *name;
{
  find_file(name, mutt_search_path, mmopen);
  return mmfile;
}
