/*
 * MLINE.C:  Piddle with the modeline and message line
 */

/* Copyright 1990, 1991, 1992 Craig Durland
 *   Distributed under the terms of the GNU General Public License.
 *   Distributed "as is", without warranties of any kind, but comments,
 *     suggestions and bug reports are welcome.
 */

#ifdef __STDC__

#include <stdarg.h>
#define VA_START va_start

#else	/* __STDC__ */

#include <varargs.h>
#define VA_START(a,b) va_start(a)

#endif	/* __STDC__ */

#include <char.h>
#include "me2.h"
#include "config.h"

#include "mm.h"		/* for the modeline stuff */

extern char *strcpy(), *bag_to_string();

extern char result[];				/* in mm.c */
extern MMDatum RV, TV;				/* in mm.c */

/* ******************************************************************** */
/* **************************** Mode Line ***************************** */
/* ******************************************************************** */

/*
 * Format the mode line for window wp.
 * If modeline-hook exists, call that to format the modeline.  Take the last
 *   character of the result and repeat it to the end of the line.  This is
 *   done to make life easier for the hook and prevent some possible display
 *   problems.
 * Called by update() any time there is a window that needs its modeline
 *   updated (hopefully not very often).
 * Notes:
 *   Don't need to clear the modeline because I'm going to fill it.
 */

int ml_hook;		/* modeline-hook */

void modeline(wp) Window *wp;
{
  extern char *MMvtoa();			/* in mm.c */
  extern MMDatum RV;				/* in mm.c */

  register Buffer *bp = wp->wbuffer;	/* buffer displayed in window */
  register char *ptr, c;
  int n;

  if (!bhook(bp,ml_hook)) return;	/* try to run modeline-hook */

  n = wp->w_toprow + wp->w_ntrows;	/* Row wp's mode line is on */
  vtdmove(n); vtmline();		/* Seek to wp's mode line */

#if WFDEBUG		/* show the windows update flags */
  vtputc((wp->w_flag & WFMODE)  ? 'M' : '-');
  vtputc((wp->w_flag & WFHARD)  ? 'H' : '-');
  vtputc((wp->w_flag & WFEDIT)  ? 'E' : '-');
  vtputc((wp->w_flag & WFMOVE)  ? 'V' : '-');
  vtputc((wp->w_flag & WFFORCE) ? 'F' : '-');
#endif

  ptr = MMvtoa(&RV);
  vtputs(ptr);
  c = ptr[strlen(ptr)-1]; while (vtputc(c)) ;	/* repeat last char */

  return;
}

/* ******************************************************************** */
/* *************************** Message Line *************************** */
/* ******************************************************************** */

    /* The screen has t_nrow + 1 lines on it (numbered 0 ... t_nrow).  The
     *   last line on the screen is used for the message line.
     * mline_dirty is used by main() so it can clear the message line every
     *   now and then, in a reasonable manner).
     */

extern int MMask_pgm, t_nrow;

/*
 * Erase the message line.
 */
void mlerase()
{
  vtdmove(t_nrow); vtclear_line(); ml_update();
  mline_dirty = FALSE;
}

/* Write out a string.  Like mlwrite() but no format.  Control characters
 *   are NOT expanded.
 */
void mlputs(str) char *str;
{
  vtdmove(t_nrow); vtclear_line(); vtputcs(str); ml_update();
  mline_dirty = *str;		/* TRUE if msg line not blank */
}

/*
 * Write out an integer, in the specified radix [2,16].
 */
static void mlputi(i,r)
{
  register int q;
  static char hexdigits[] = "0123456789ABCDEF";

  if (i < 0) { i = -i; vtputc('-'); }
  q = i/r;
  if (q != 0) mlputi(q, r);
  vtputc(hexdigits[i%r]);
}

/*
 * Write a message into the message line.
 * A small class of printf like format items is handled.
 * Set the "message line" flag TRUE.
 */
/*VARARGS1*/
#ifdef __STDC__
void mlwrite(char *fmt, ...)
#else
void mlwrite(fmt,va_alist) char *fmt; va_dcl
#endif
{
  register char c;
  va_list ap;

  vtdmove(t_nrow); vtclear_line(); mline_dirty = TRUE;

  VA_START(ap, fmt);
  while ((c = *fmt++) != '\0')
  {
    if (c != '%') vtputc(c);
    else
      switch (c = *fmt++)
      {
        case 'd': mlputi(va_arg(ap,int),10);  break;
        case 'x': mlputi(va_arg(ap,int),16);  break;
        case 's': vtputcs(va_arg(ap,char *)); break;
        default:  vtputc(c);
      }
  }
  va_end(ap);
  ml_update();
}

/*
 * Ask a yes or no question in the message line.  Must be answered with
 *   something starting with n, N, y or Y.  A CR ain't good enough!
 * Notes:
 *   If this is called from program, only looks one arg - don't loop though
 *     the entire arg list looking for "y" or "n"!
 * Input:
 *   prompt:  What to ask the user.  " [y/n]? " is appended.
 * Returns:  TRUE, FALSE, or ABORT.
 */
mlyesno(prompt) char *prompt;
{
  char buf[128], answer[10];

  strcat(strcpy(buf,prompt)," [y/n]? ");
  while (TRUE)
  {
    if (mlreply(buf,answer,sizeof(answer),0) == ABORT) return ABORT;
    if (*answer == 'y' || *answer == 'Y') return TRUE;
    if (*answer == 'n' || *answer == 'N') return FALSE;
    if (MMask_pgm) return FALSE;
  }
  /* NOTREACHED */
}


    /* Direct writes to the screen.  For stuff like escape sequences or
     *   popup menus.
     */

	/* dump a string to the terminal */
void t_puts(string) register char *string;
  { while (*string) t_putchar(*string++); }

	/* dump a string (of max n chars) to the terminal */
void t_nputs(string, n) register char *string;
  { while (*string && n--) t_putchar(*string++); }


/* ******************************************************************** */
/* ********** mlreply():  Query the user on the message line ********** */
/* ******************************************************************** */

int cc_trigger = CC_TRIGGER;	/* the key that triggers command completion */

#define MAX_PRIME	150
static char prime_rib[MAX_PRIME + 1];	/* !!! should be dynamic */

    /* (prime-ask TRUE) returns the current prime.
     * (prime-ask) clears the prime.
     * (prime-ask text text ...) concats text and puts it into the prime.
     */
void Mprime_ask()
{
  if (!MMpull_nth_arg(&TV,0) || TV.type != BOOLEAN)
  {
    MMconcat();
    strncpy(prime_rib, result, MAX_PRIME);
    prime_rib[MAX_PRIME] = '\0';
  }
  RV.type = STRING; RV.val.str = prime_rib;
}

#if !LED

ml_init() { return TRUE; }	/* initialize the message line stuff */

/*
 * Write a prompt into the message line, then read back a response.  Keep
 *   track of the physical position of the cursor.  If we are in a keyboard
 *   macro throw the prompt away, and return the remembered response.  This
 *   lets macros run at full speed.  The reply is always terminated by a
 *   carriage return.  Handle erase, clear-line, quote and abort keys.
 * Command completion:  If you want it, set selector to the bits of the
 *   lists to complete.  0 => no completion.
 * Note:  It is NOT OK for prompt==buf since ^L and command completion both
 *   both reuse the prompt.  I don't like it either.
 * RETURNS: TRUE, FALSE or ABORT.
 */
mlreply(prompt,buf,nbuf,selector)
  char *prompt, *buf;
  int nbuf;		  /* max size of buf */
  unsigned int selector;
{
  unsigned char tmp[20];
  register EKeyCode c;
  register int cpos = 0, i;

  if (MMask_pgm)
  {
    if (!MMnext_arg(buf)) return stop_ME();	/* Zippie sez Yow! */
    return ((*buf == '\0') ? FALSE : TRUE);
  }
		/* keyboard macro? */
  if (get_McString(buf)) return ((*buf == '\0') ? FALSE : TRUE);

  selector &= CC_MASK;		/* turn off lists I can't complete */
  mlputs(prompt);

  nbuf--;	/* leave room for \0 */
  while (TRUE)
  {
    buf[cpos] = '\0';		/* save some code */
    c = t_getchar();
    if (selector && (c == cc_trigger || c == '?'))
    {
      if (c == '?') hbomb(buf,selector);
      else { complete(buf,selector); cpos = strlen(buf); }
      mlwrite("%s%s",prompt,buf);
      continue;
    }
    switch (c)
    {
      case 0x0A: case 0x0D:		/* Return or newline == end of line */
	if (!add_McString(buf)) return ABORT;
	t_putchar('\r'); t_flush();
	return ((*buf=='\0') ? FALSE : TRUE);
      case 0x07: return stop_ME();			/* Bell == Abort */
      case 0x7F: case 0x08:		/* Backspace & Rubout == erase */
	if (cpos != 0)
        {
	  t_puts("\b \b");
          if (iscntrl(buf[--cpos])) t_puts("\b \b");
          t_flush();
        }
        break;
      case 0x15: mlputs(prompt); cpos = 0; break;	/* C-u: clear-line */
      case 0xC:						/* ^L */
	screen_is_garbage = TRUE; update(); mlwrite("%s%s",prompt,buf); break;
      case 0x11:				/* C-q: quote next char */
	c = t_getchar();	/* get next char and fall though */
      default:
	if ((c & 0xFF00) == 0)	/* not a ME key */
	{
	  if (cpos<nbuf)
	  {
	    buf[cpos++] = c;
	    if (iscntrl(c)) { t_putchar('^'); c ^= 0x40; }
	    t_putchar(c); t_flush();
	  }
        }
	else		/* expand those modifiers */
	{
	  *tmp = '\0'; xpandkey(tmp,c);
	  for (i = 0; cpos<nbuf && tmp[i]; ) t_putchar(buf[cpos++] = tmp[i++]);
	  t_flush();
	}
    }
  }
  /* NOTREACHED */
}

#else	/* !LED */

#include <led.h>
#include <const.h>

extern char *Ltext;		/* in ed/led.c */

static int extended_callback();

t_clearline(n) { movecursor(t_nrow,0,TRUE); if (!n) t_eeol(); return n; }

#define COMPLETE 0
#define HELP	 1
#define REDRAW	 2
#define YANK	 3


	/* initialize the message line stuff */
ml_init()	/* do this after t_open() and alloc_display() */
{
  extern int Lflags;

  init_Led();
  Lmacskeys();

  Ldo_fcn(LF_BINDKEY,
      LF_CALLBACK,	REDRAW,	  CTRL|'L',
      LF_CALLBACK,	COMPLETE, cc_trigger,
      LF_CALLBACK,	HELP,	  '?',

      LF_EXTEND,	YANK,	  CTRL|'Y',

      LF_ABORT,		CTRL|'G',
      LF_UNDEF,		CTRL|'C',
      LF_UNDEF,		'\\',

      LF_STOP,LF_STOP);

  Lregister_callback(0, extended_callback);

  return TRUE;
}

static int
  selmask,		/* a way to pass selector to the callback */
  tigger = CC_TRIGGER;	/* So I can detect changes */

   /* If the cursor is moved out of the msg line, need to make sure to tell
    *   Led to move it back.
    */
static int callback(kc,fcn) EKeyCode kc; int fcn;
{
  int saveit;		/* Guard selmask against recursion. */
  char buf[200];

  switch(fcn)
  {
    case HELP:
      saveit = selmask; hbomb(Ltext,selmask); selmask = saveit;
      return Ldo_fcn(LF_CTYNC,LF_STOP);
    case COMPLETE:
      strcpy(buf,Ltext); complete(buf,selmask);
      return Ldo_fcn(LF_CLEAR_LINE,LF_INSERT_STRING,buf,LF_CTYNC,LF_STOP);
    case REDRAW:
      screen_is_garbage = TRUE; update();
      return Ldo_fcn(LF_REDRAW,LF_STOP);
  }
  return LF_NOOP;
}

static int extended_callback(kc,fcn, op) EKeyCode kc; int fcn, *op;
{
  char *ptr;

  switch(fcn)
  {
    default: return FALSE;		/* nothing I know about */
    case YANK:
      if (ptr = bag_to_string(cut_buffer))
	*op = Ldo_fcn(LF_INSERT_STRING,ptr,LF_CTYNC,LF_STOP);
  }
  return TRUE;
}

/*
 * Write a prompt into the message line, then read back a response.  Keep
 *   track of the physical position of the cursor.  If we are in a keyboard
 *   macro throw the prompt away, and return the remembered response.  This
 *   lets macros run at full speed.  The reply is always terminated by a
 *   carriage return.
 * Command completion:  If you want it, set selector to the bits of the
 *   lists to complete.  0 => no completion.
 * Note:  It is OK for prompt==buf.
 * RETURNS: TRUE, FALSE or ABORT.
 */
mlreply(prompt,buf,nbuf,selector)
  char *prompt, *buf;
  int nbuf;		  /* max size of buf */
  unsigned int selector;
{
  int s;

  if (MMask_pgm)
  {
    if (!MMnext_arg(buf)) return stop_ME();		/* Zippie sez Yow! */
    return ((*buf == '\0') ? FALSE : TRUE);
  }
		/* keyboard macro? */
  if (get_McString(buf)) return ((*buf == '\0') ? FALSE : TRUE);

  mline_dirty = TRUE;	/* assume message line is gonna have stuff in it */

/*  selector &= CC_MASK;		/* turn off lists I can't complete */
  selmask = selector;

	/* if the command completion trigger has changed, rebind */
  if (selector && cc_trigger != tigger)
  {
    Ldo_fcn(LF_BINDKEY,
      LF_UNDEF,tigger,			/* unbind old key */
      LF_CALLBACK,COMPLETE,cc_trigger,	/* bind to new key */
      LF_STOP,LF_STOP);
    tigger = cc_trigger;
  }

  s = Led(prompt,prime_rib,selector ? callback : (pfi)NULL);
  *prime_rib = '\0';
  switch (s)
  {
    case LF_DONE:		/* Return or newline == end of line */
      strncpy(buf,Ltext,nbuf); buf[nbuf-1] = '\0';
      if (!add_McString(buf)) return ABORT;
      t_putchar('\r'); t_flush();
      return ((*buf == '\0') ? FALSE : TRUE);
    case LF_ABORT: case LF_ERROR: return stop_ME();
  }
  /* NOTREACHED */
mlwrite("???????????????"); t_getchar();
}

#endif	/* !LED */
