/*
 * OS2IO.C: Terminal IO for OS/2
 * J Burnell	3/92    Public Domain
 * P McPhee	1/95	Use EMX's termio and video libraries, for DOS support
 */

/*
 * The functions in this file negotiate with the operating system for
 * characters, and write characters in a barely buffered fashion on the
 * display.
 */

#include <io.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include "ed.h"
#include "term.h"
#include <signal.h>
#include <termio.h>
#include <string.h>

/* for EMX */
#include <sys/video.h>		/* video library */

#include "me2.h"
#include "config.h"

void t_eeol ();

static struct termio oldtio, tio;
static struct {
  int start, end;
} oldcurs, curs;

/*
 * This function is called once to set up the terminal device streams.
 * It's also called after a spawn.
 */
void ttopen()
{
   /* ignore some signals */
   signal(SIGBREAK, SIG_IGN);
   signal(SIGINT, SIG_IGN);

  ioctl(0, TCSETA, &tio);
  v_ctype(curs.start, curs.end);
}

/*
 * This function gets called just before we go back home to the command
 *   interpreter.
 */
void t_close()
{
  /* close the keyboard */
  ioctl(0, TCSETA, &oldtio);

  /* restore the cursor scan lines */
  v_ctype(oldcurs.start, oldcurs.end);
}

int t_nrow = 24, t_ncol = 0;

/*
 * Write a character to the display.
 * Use the rawest console output routine that handles backspace, \r and \n.
 * On MS-DOS terminal I/O is unbuffered, so we just write the byte out.
 */
void t_putchar(c) unsigned char c;
{
  if (c == '\010')
    v_backsp(1);
  else if (c == '\r')
    v_gotoxy(0, t_nrow);
  else
    v_putc(c);
}

/*
 * Flush terminal buffer.  Does real work where the terminal output is
 *   buffered up.  A no-operation on systems where byte-at-a-time terminal
 *   I/O is done.
 */
void t_flush() {}

/*
 * Read a character from the terminal, performing no editing and doing no
 *   echo at all.
 * Map terminal softkeys, etc to ME keys.  Do *NOT* map control keys.
 */
EKeyCode t_getchar()
{
   EKeyCode keycode;
   unsigned char c;

   read(0, &c, 1);
   if (c == 0) {
     read(0, &c, 1);
     map_key(c, &keycode);
     return keycode;
     }
   else
     return (EKeyCode)c;
}

/*
 * Check to see is a key is waiting.
 * Used to interrupt processing if input is pending or to detect if a soft
 *   key has been hit (the terminal sends multiple characters for each
 *   softkey).
 * Don't use function B because it checks for ^C.
 */
int keywaiting()
{
  return wait_for_key(0);
}

   /* Wait for a key to be pressed for no more than sec seconds.
    * Use 0 seconds to check to see if a key is in the input que.
    * Warning:  This rouine is only good for 32K seconds (about 9 hours).
    *   If sec is a long the limit is still 24 hours.
    */
static struct _fd_set rdfds;
wait_for_key(int sec)
{
   struct timeval tv;

   tv.tv_sec = sec;
   tv.tv_usec = 0;

   return select(1, &rdfds, NULL, NULL, &tv) > 0;
}

  /* Things to note about FastVideo:
   *   putline() will do all of the writing to the text area (except for
   *     help messages).
   *   t_eeol() will only be called for the message line or for help
   *     messages.
   *   t_eeop() need only clear the message line.  It should use t_eeol() to
   *     do this.
   *   t_putchar() will only write to the message line, help messages or
   *     (puts "message").
   * So, this means that the message line is really seprate from the text
   *   areas and t_putchar() and t_eeol() should use the same color (others
   *   will use tcolor and mcolor).  Default for the message line would be
   *   tcolor.  Help and (puts ...)  messages will be written in message
   *   color (which is (I hope) reasonable).
   * PC note:  Due to the way the BIOS is done, it is easier to have
   *   t_putchar() and t_eeol() use the BIOS character attribute as the
   *   color.
   */

int tcolor = 7, mcolor = 96;    /* text, modeline color defaults */
int zcolor = 7;
static unsigned char *lbuf;

 /*
  * Called to set up the terminal.
  */
void t_open()
{
  int display_type;

  /* set up the keyboard */
  ioctl(0, TCGETA, &oldtio);
  tio = oldtio;
  tio.c_lflag &= ~(ICANON | ISIG | ECHO | ECHOE | ECHOK | IDEFAULT);
  tio.c_iflag &= ~(ICRNL | INLCR | IUCLC | IXON | /*IXOFF |*/ IXANY | BRKINT);
  tio.c_cc[VMIN] = 1;
  tio.c_cc[VTIME] = 0;	/* no timeout by default */

  /* for select() in wait_for_key() */
  FD_ZERO(&rdfds);
  FD_SET(0, &rdfds);

  /* initialise video library */
  v_init();

  v_getattr();

  display_type = v_hardware();
  curs.start = 0;
  curs.end = display_type == V_COLOR_8 ? 8 : 12;

  /* capture signals, set up keyboard, set cursor */
  ttopen();

  v_dimen(&t_ncol, &t_nrow);
  t_nrow--;	/* leave room for mini-buffer */
  lbuf = malloc(t_ncol*2);
}


void putline(row, buf, attr)    /* note: row is 0 relative */
  int row,attr; char *buf;
{
  register int i;
  for (i = 0; i < t_ncol; i++) {
    lbuf[i*2] = buf[i];
    lbuf[i*2+1] = attr;
  }
  v_putline(lbuf, 0, row, t_ncol);
}

 /* Erase to end of page or screen.
  * For fast video, each line of the screen (except for the message line)
  *   will be completely filled by putline() so all this routine needs to
  *   clear is the message (last) line of the screen.
  */
void t_eeop() { v_gotoxy(0, t_nrow); v_clreol(); }

  /* Ring the bell.  beeper controls the volume:  0 (don't beep), n (some
   *   volume value).
   */
void t_beep()
{
  extern int beeper;

  if (!beeper)
    return;
  putchar('\b');
}

void t_move(row,col) int row, col;  /* move cursor to (row,col) */
{
  v_gotoxy(col, row);
}

void t_eeol()       /* erase from cursor to end of line */
{
  v_clreol();
}

#include "driver.h"

#define RGB_BITS(bits)		((bits) & 0xF)
#define FOREGROUND_BITS(bits)	((bits) & 0xF)
#define BACKGROUND_BITS(bits)	(((bits) >> 4) & 0x7)
#define MAKE_RGB(fg, bg)	( ((fg) & 0xF) | (((bg) & 7) << 4) )

static char *pc_color[] =
{			/* IRGB   I can be 1 for forground colors only */
  "black",		/* 0000 ==  0 */
  "blue",		/* 0001 ==  1 */
  "green",		/* 0010 ==  2 */
  "cyan",		/* 0011 ==  3 */
  "red",		/* 0100 ==  4 */
  "magenta",		/* 0101 ==  5 */
  "brown",		/* 0110 ==  6 */
  "lightgrey",		/* 0111 ==  7 */
  "darkgrey",		/* 1000 ==  8 */
  "lightblue",		/* 1001 ==  9 */
  "lightgreen",		/* 1010 == 10 */
  "lightcyan",		/* 1011 == 11 */
  "lightred",		/* 1100 == 12 */
  "lightmagenta",	/* 1101 == 13 */
  "yellow",		/* 1110 == 14 */
  "white",		/* 1111 == 15 */
};

static int color_to_rgb(color_name, rgb, foreground)
  char *color_name; int *rgb;
{
  int i, max;

  strlwr(color_name);
  max = foreground ? NITEMS(pc_color) : (NITEMS(pc_color) + 1)/2;
  for (i = 0; i < max; i++)
    if (0 == strcmp(pc_color[i], color_name))
    {
      *rgb = i;
      return TRUE;
    }
  return FALSE;
}

static char *rgb_to_color(rgb)
{
  return pc_color[RGB_BITS(rgb)];
}

static void do_cursor(char * cursor_shape, int set);

    /* "forground:background"
     *   ":"      => no change
     *   "color"  => only change foreground
     *   "color:" => only change foreground
     *   ":color" => only change background
     * Input:
     *   color : name of color.
     *   rgb : current color (incase name of new color is bogus)
     * Returns:
     *   New rgb bits.
     */
static int set_color(color, rgb) char *color;
{
  char *ptr, buf[100];
  int fg, bg;

  strcpy(buf, color);
  if (ptr = strchr(buf, ':')) *ptr++ = '\0';
  if (!color_to_rgb(buf, &fg, TRUE))		fg = FOREGROUND_BITS(rgb);
  if (!ptr || !color_to_rgb(ptr, &bg, FALSE))	bg = BACKGROUND_BITS(rgb);
  return MAKE_RGB(fg, bg);
}

static void get_color(buf,rgb) char *buf;
{
  strcpy(buf, rgb_to_color(FOREGROUND_BITS(rgb)));
  strcat(buf, ":");
  strcat(buf, rgb_to_color(BACKGROUND_BITS(rgb)));
}

int do_color(which, color, set) char *color;
{
  int *c;

  switch(which)
  {
    case TEXT_COLOR:     c = &tcolor; break;		/* text color */
    case MODELINE_COLOR: c = &mcolor; break;		/* modeline color */
    case CURSOR_COLOR:					/* cursor color */
      if (set) return FALSE;	/* can't change cursor color */
      c = &tcolor;		/* it is always the text color */
      break;
    case CURSOR_SHAPE: do_cursor(color, set); return TRUE;
    default: return FALSE;			/* boo boo */
  }
  if (set)
  {
    int x = *c;
    return (x != (*c = set_color(color, x)));
  }
  else get_color(color, *c);

  return FALSE;
}

static void do_cursor(cursor_shape, set) char *cursor_shape;
{
  if (set)
  {
    char *ptr, buf[100];

    strcpy(buf, cursor_shape);
    if (ptr = strchr(buf, ':')) *ptr++ = '\0';
    curs.start = (atoi(buf) & 0xF);
    curs.end = ptr ? (atoi(ptr) & 0xF) : 15;

    v_ctype(curs.start, curs.end);
    return;
  }

  sprintf(cursor_shape, "%d:%d", curs.start, curs.end);
}

static MouseInfo mouse_info;	/* !!!??? initialize? */

MouseInfo *get_mouse_info()
{
  return &mouse_info;
}
