/* main.c
 * Mutt Editor 3: a screen text editor influenced by Emacs.
 * This program was based on MicroEmacs, orginally written by Dave G.
 *   Conroy and then completely modified/rewritten by Craig Durland.  I've
 *   kept most of the core editer structures and concepts because they were
 *   good work.  Many thanks to Dave for getting this project rolling.  I
 *   hope others will find this work as useful and fun to them as Dave's was
 *   to me.
 * Work on this version (converting ME2 to ME3) started around 8/92 with the
 *   first public release 6/93 with a few alpha's before that.
 */

/* Copyright 1990, 1991, 1992, 1993 Craig Durland
 *   Distributed under the terms of the GNU General Public License.
 *   Distributed "as is", without warranties of any kind, but comments,
 *     suggestions and bug reports are welcome.
 */

#define VNAME		"ME"
#define VMAJOR		"3"		/* Major version number */
#define VMINOR		"4"		/* Minor version number */
#define VPATCH		""		/* Patch level.  "" => no patches */
#define VREL		"3/93"		/* First release date */
#define VLATEST		"3/94"		/* Date of this release */

#define AUTHOR		"Craig Durland"
#define	VEMAIL		"craig@cv.hp.com"
#define VSMAIL		"3419 SW Knollbrook, Corvallis, OR 97333"

#ifdef __STDC__
static char what[] = "@(#)" VNAME " (Mutt Editor) "\
			VREL " v" VMAJOR "." VMINOR " " VLATEST;
#else
static char what[] = "@(#)ME (Mutt Editor)";	/*  */
#endif

#include <char.h>
#include "me2.h"
#include "mm.h"

extern void MMset_hooks(), process_hooks();

#ifdef __STDC__		/* !!!??? roll this into me2.h???? */
void que_version_info(char *, ...);
#else
void que_version_info();
#endif /* __STDC__ */

static int internal_Mutt();
static void bye_bye(), edinit(), send_version_info();

extern int t_nrow, t_ncol;	/* in the terminal dependent code */

static int time_to_go = 0;	/* !0 if should exit ME */

Bag *cut_buffer;		/* must be initialized very early! */
char **zargv;			/* a copy of argv for MUTT */
int
  beeper = 100,			/* Bell volume.  0 => bell turned off */
  doing_nothing = FALSE,	/* process control state variable */
  overstrike = FALSE,		/* insert mode is the default */
  idle_hook,
  kp_hook,			/* key pressed hook */
  ost_hook,			/* overstrike-toggle-hook */
  pro_hook,			/* external process hook */
  stop_hook,			/* (stop-ME-hook) */
  version_hook,			/*  */
  zargc;			/* a copy of argc for MUTT */
int32
  thisflag = 0,			/* Flags, this command */
  lastflag = 0;			/* Flags, last command */
EKeyCode key_pressed;		/* last key pressed in main loop */


    /* Here is where it all happens.  ME is initialized and then we sit in
     *   the MainLoop, waiting to be told to do something and doing it.
     * Notes:
     *   Since this is where the main user interaction takes place, I need
     *     to make sure that the buffers and windows are synchronized and
     *     the display is up to date so that the user sees what I want them
     *     to see.
     *   Every time anything is done, the display probably needs updating.
     *   Every time a Mutt program is run, the current buffer could get
     *	   outta sync with the current window.
     */
main(argc,argv) int argc; char *argv[];
{
  int n;
  register EKeyCode kc;

	/* Get everything ready to go */
  zargc = argc; zargv = argv;
  edinit(SCRATCH_BUFFER);

  que_version_info("version",
    VNAME, VMAJOR, VMINOR, VPATCH, VREL, VLATEST, (char *)NULL);
  que_version_info("author", AUTHOR, VEMAIL, VSMAIL, (char *)NULL);
  send_version_info();

  if (-1 != (n = MMpgm_lookup("start-ME-up"))) MMrun_pgm(n);

  lastflag = 0;			/* Fake last flags */
  while (!time_to_go)		/* The ME main loop */
  {
	/* Sync the current buffer with the current window in case something
	 *   changed and to prepare for upcomming changes.
	 * Don't need to change window flags because the current window is
	 *   up to date.
	 */
    if (curwp->wbuffer == curbp) sync_dot();
    else use_window(curwp);	/* sync current buffer with current window */

    update();	/* Fix up the screen so the user sees whats really there */

	/* Wait for somebody to type something.  Since we ain't doing
	 *   anything, set a flag (incase we get interrupted) saying its OK
	 *   to do stuff.  Since we are (maybe) going to be waiting, go a
	 *   head and process any queued interrupts.  Note that getkey()
	 *   might also call process_hooks().
	 */
    doing_nothing = TRUE;
    process_hooks();		/* and maybe run Mutt programs */
    kc = getkey();		/* Waiting for Godot to type */
    doing_nothing = FALSE;

	/* It possible that process_hooks() can run a Mutt program that
	 *   leaves the current buffer different than the current window.
	 *   Need to resync so if kc runs a Mutt program, the current buffer
	 *   will be synced.
	 * Probably should also update if hooks were run???
	 */
    if (curwp->wbuffer == curbp) sync_dot();
    else use_window(curwp);

    if (mline_dirty)		/* msg line has text on it */
    {
      mlerase();
      if (kc == ' ') continue;	/* ITS Emacs does this */
    }

    if (!add_McKey(kc)) continue;	/* Save macro key strokes */

    if (kp_hook != -1)			/* (key-pressed-hook <key pressed>) */
    {
      extern MMDatum TV;		/* in mm.c */

      MMStkFrame mark;

      MMopen_frame(&mark);
      TV.type = NUMBER; TV.val.num = kc; MMpush_arg(&TV);
      MMrun_pgm_with_args(kp_hook,&mark);
    }

    do_key(kc,FALSE,1);			/* and maybe run Mutt programs */

    if (thisflag != CFCHAR)		/* add sequence point */
	save_for_undo(BU_SP,(int32)0);
  } /* end main loop */
  bye_bye();
}

    /*
     * Initialize the editor.  The buffer name is passed in.
     */
static void edinit(bname) char *bname;
{
  extern int case_fold;		/* in search.c */

  MMset_hooks(); /* so no hooks are called before everything is initialized */

  fix_cmap(case_fold);		/* need to do this before Mutt code runs */

  	/* Set up display stuff */
  open_display(); display_resized(t_nrow +1, t_ncol);

  if (
	!osinit()			 || /* Display, terminal & others */
	!ml_init()			 || /* message line */
	!(cut_buffer = alloc_bag(TRUE))  || /* cut buffer */
	!init_keys()			 || /* global key table */
	!buf_init(bname)		 || /* buffer system */
	!win_init()			 || /* window system */
	!MMinitialize()			 || /* Mutt Machine */
	!internal_Mutt())		    /* internal Mutt code */
  {
    close_display(FALSE);		/* restore the display */
    exit(1);
  }
}

    /* Try and load the internal built in ME Mutt init file.  This contains
     *   the the ME frontend and does the command line processing.
     * Returns:
     *   TRUE:  Everything went as expected.
     *   FALSE: Couldn't load/init the internal Mutt code (out of mem or too
     *   	many pgms).  Bad news, abort.
     * Notes:
     *   Need to update before calling Mutt code because update will cause
     *     (modeline-hook) to be called.  If an update is done after the
     *     Mutt code is loaded but before the hooks initialize themselves,
     *     there will be problems.
     *   If the MAIN code aborts before the update is done, the update (in
     *     the main loop) will clear the screen and erase the error message.
     *     Real hard to debug.
     *   If I do update, the modeline (for scratch) will change from the
     *     default to the one in the Mutt code.  Ugly.
     */
static int internal_Mutt()
{
  int code_ran;

/*  update();	/*  */
  if (!MMload_internal_code(&code_ran))
  {
    mlwrite("Error loading internal Mutt code!  Can't continue!");
    return FALSE;
  }
  return TRUE;
}

    /* Hooks for Mutt programs.  When certain events occur in ME, an attempt
     *   to call a Mutt program will be made so that ME can be extended in
     *   useful ways.
     * Notes:
     *   Called from MM after every Mutt code block is loaded.
     *   Call this at start up to make sure all hooks are
     *     properly initialized to no hook.
     */
void MMset_hooks()	/* setup hooks used by ME */
{
  extern int
    bc_hook, bclr_hook, fb_hook,	/* in buffer.c */
    ml_hook,			/* in mline.c */
    kp_hook, pro_hook,		/* in main.c */
    od_hook, cd_hook;		/* in display.c */

  bclr_hook	= MMpgm_lookup("clear-buffer-hook");
  bc_hook	= MMpgm_lookup("create-buffer-hook");
  od_hook	= MMpgm_lookup("enter-ME-hook");
  fb_hook	= MMpgm_lookup("free-buffer-hook");
  idle_hook	= MMpgm_lookup("idle-hook");
  kp_hook	= MMpgm_lookup("key-pressed-hook");
  cd_hook	= MMpgm_lookup("leave-ME-hook");
  ml_hook	= MMpgm_lookup("modeline-hook");
  ost_hook	= MMpgm_lookup("overstrike-toggle-hook");
  pro_hook	= MMpgm_lookup("process-hook");
  stop_hook	= MMpgm_lookup("stop-ME-hook");
  version_hook	= MMpgm_lookup("version-hook");
}

    /*
     * This is the general "do a key" routine:
     *   If the is bound, run what it is bound to.
     *   If the key is not bound and it is the "printable" range, treat it
     *     as bound to "self-insert" and insert n of those into the current
     *     buffer at the dot.  Do word wrap if need to.
     *   Clear out "thisflag" and to move it to "lastflag", so that the next
     *     command can look at it.
     * Input:
     *   kc : Keycode to run
     *   f  : Flag to send to the command (TRUE if C-u was used).  Default
     *	  is FALSE.
     *   n  : The C-u count.  Default is 1.
     * Returns:
     *   Status of command: TRUE, FALSE or ABORT.
     *   FALSE : Couldn't do anything with the key.
     */
do_key(kc, f,n)
  EKeyCode kc;	/* command character */
  int f,	/* if argument (^U) used */
      n;	/* do c n times */
{
  int status;

  key_pressed = kc;
  thisflag = 0;

		/* is key bound to anything? */
  if (run_key(kc, f,n, &status)) { lastflag = thisflag; return status; }

	/* If self insert was typed, fill column is defined, argument is
	 * non-negative, are now past fill column and typing at end of line
	 * then perform word wrap.
         */
  if (0x20 <= kc && kc <= 0xFF)
  {
    register int wrap_col = curbp->wrap_col;

    if (n <= 0) { lastflag = 0; return !n; }	/* TRUE if n == 0 */

#if 1
	/* word wrap? */
    if (wrap_col > 0 && getccol() > wrap_col &&
	the_dot->offset == llength(the_dot->line))
    {
      wrapword();
      /* word wrap is more than a self-insert, don't set thisflag */
    }
    else 
#endif
      thisflag = CFCHAR;	/* self-insert character(s) */

    status = linsert(n,(unsigned char)kc,overstrike);
    lastflag = thisflag;

    return status;
  }

	/* Don't know what to do with the key */
  lastflag = 0;         /* Fake last flags since nothing happened */
  return FALSE;
}

void do_idle_hook(tick) long int tick;
{
  if (idle_hook != -1)
  {
    extern MMDatum TV;		/* in mm.c */

    MMStkFrame mark;

    MMopen_frame(&mark);
    TV.type = NUMBER; TV.val.num = tick; MMpush_arg(&TV);
    MMrun_pgm_with_args(idle_hook,&mark);
  }
}

/* ******************************************************************** */
/* ***************** Keyboard Macros ********************************** */
/* ******************************************************************** */

#define KOFF	  0	/* keyboard macro is off */
#define RECORDING 1	/* adding keys to macro */
#define PLAYBACK  2

typedef struct { int McLen; EKeyCode McKeys[NKBDM]; } KbdMacro;

static KbdMacro kbdm;	/* we know McLen is initialized to 0 */
static EKeyCode *kptr;
static int McLen, McState = KOFF;

add_McKey(key) EKeyCode key;
{
  if (McState != RECORDING) return TRUE;
  if (McLen++ > NKBDM-3) { stop_ME(); return FALSE; }
  *kptr++ = key;
  return TRUE;
}

add_McString(str) char *str;
{
  int s;

  if (McState != RECORDING) return TRUE;
  while ((s = add_McKey((EKeyCode)*str)) && *str++!='\0') ;
  return s;
}

	/* FALSE => not in a macro or no keys left
	 * TRUE => got keys
	 */
get_McKey(key) EKeyCode *key;
{
  if (McState != PLAYBACK || McLen-- <= 0) return FALSE;
  *key = *kptr++;
  return TRUE;
}

get_McString(str) char *str;
{
  EKeyCode kc;
  register int s;

  if (McState != PLAYBACK) return FALSE;
  while ((s = get_McKey(&kc)) && (*str++ = kc)) ;
  return s;
}

get_McKc(kc,getakey) EKeyCode *kc;
{
  if (get_McKey(kc)) return TRUE;	/* keyboard macro running */
  *kc = getakey ? getkey() : t_getchar();
  return add_McKey(*kc);
}

/*
 * Begin a keyboard macro.  Error if already in a macro.
 */
static int start_macro()
{
  if (McState != KOFF) return FALSE;

  McState = RECORDING;
  kbdm.McLen = 0;	/* wipe out old macro in case of (abort) */
  kptr = kbdm.McKeys; McLen = 0;

  return TRUE;
}

/*
 * End keyboard macro.
 */
static int end_macro()
{
  if (McState != RECORDING) return FALSE;

  kbdm.McLen = McLen -1;	/* length of macro (not including "C-x )") */
  McState = KOFF;

  return TRUE;
}

/*
 * Run a macro.
 * The argument is the number of times to loop.  Quit as soon as a command
 *   gets an error.  Return TRUE if all ok, else FALSE.
 * !!!???This should set thisflag and lastflag!
 */
static int run_macro(n)
{
  EKeyCode key;
  register int s = TRUE;

  if (McState != KOFF) return FALSE;
  McState = PLAYBACK;
  while (s == TRUE && 0 < n--)
  {
    McLen = kbdm.McLen; kptr = kbdm.McKeys;
    while (get_McKey(&key) && (s = do_key(key,FALSE,1)) == TRUE) ;
  } 
  McState = KOFF;

  return s;
}


key_macro(op, n) int op, n;
{
  switch (op)
  {
    case 10: return start_macro();	/* Macro: Start */
    case 11: return end_macro();	/* Macro: End */
    case 12: return run_macro(n);	/* Macro: Replay */
    case 13: return McState;		/* Macro: Return state */
  }
  return FALSE;
}

/* ******************************************************************** */
/* *********************************  ********************************* */
/* ******************************************************************** */

static void stop_it(x)
{
  if (stop_hook != -1)		/* (stop-ME-hook <x>) */
  {
    extern MMDatum TV;		/* in mm.c */

    MMStkFrame mark;

    MMopen_frame(&mark);
    TV.type = NUMBER; TV.val.num = x; MMpush_arg(&TV);
    MMrun_pgm_with_args(stop_hook,&mark);
  }
}

    /* Call this to actually exit ME.  This gets things cleaned up and
     *   allows prorgrams to do any house keeping they might need to do.
     * Notes:
     *   When this is called, there should be no programs running.  This is
     *     to keep from setjmp()ing (in MM).
     */
static void bye_bye()
{
  stop_it(time_to_go);
  close_display(TRUE);
  exit(0);
}

    /* Call this to get ready to exit ME or exit ME if ready to.  This stops
     *   any running programs and tells the main loop to exit.
     * Input:
     *   n : Normally 0, 1 if abnormally terminated (such as by a signal).
     * Notes:
     *   I can't just call stop_it() and exit() because if exit_ME() is
     *     called and a program is running and (stop-ME-hook) tanks (or
     *     calls stop_ME()), the setjmp() in MM will cause a return to
     *     whoever called the running program (ie not return here).  This is
     *     a problem when called from something like a signal handler
     *     because ME won't exit in that case.
     */
void exit_ME(n)
{
  time_to_go = (1 + n);
  if (MMpgm_running()) MMabort_pgm(0);	/* setjmp()s back to the main loop */
  bye_bye();		/* else no program running */
}

/*
 * Stop everything and return to the main loop.
 * Turn off any keyboard macro, etc., that is in progress.
 * Called as a routine to do general stopping of stuff.
 * Notes:
 *   If called from a Mutt program, this routine exits via setjmp().  It
 *     will still return to the main loop.
 *   Gotta be careful of potential infinite loops - an stop-hook calling
 *     stop_ME()!  The loop isn't very infinite because the MM stacks
 *     overflow way before the C stack does but all the same ...
 *   Need to reset McState after calling the hook so that a hook can use the
 *     macro state.  Of course, that means that macros can be buggered if
 *     the hook crashes ...
 */
stop_ME()
{
  stop_it(0);
  McState = KOFF;
  if (MMpgm_running()) MMabort_pgm(0);	/* setjmp()s back to the main loop */
  return ABORT;
}

/* ******************************************************************** */
/* *************************** Version Info *************************** */
/* ******************************************************************** */

    /* Send information to Mutt programs for interested parties.
     * This also allows routines like drivers to send OS specific info
     *   (which is good because it isolates that info to the driver).
     * Problem:  Need to send the info to a Mutt program but can't do that
     *   until the Mutt programs have been loaded.  Can't load the Mutt
     *   programs until ME is initialized (because a program could really
     *   mess things up by accessing uninitialized internals).  So, need to
     *   send the info after the drivers have been called.  To do that, the
     *   info is queued, then sent out later.
     */

#ifdef __STDC__

#include <stdarg.h>
#define VA_START va_start

#else	/* __STDC__ */

#include <varargs.h>
#define VA_START(a,b) va_start(a)

#endif

static char **vinfos = NULL;
static int num_infos = 0;

    /* Notes:
     *   The infos CAN'T be automatic variables (because I just save
     *     pointers to them.  Use constants ("foo") or globals.
     */
/*VARARGS1*/
#ifdef __STDC__
void que_version_info(char *info, ...)
#else
void que_version_info(info,va_alist) char *info; va_dcl
#endif
{
  char *ptr;
  va_list ap;

  if (!vinfos && !(vinfos = (char **)malloc(200 * sizeof(char *)))) return;

  VA_START(ap, info);
  ptr = info;
  while (ptr)
  {
    vinfos[num_infos++] = ptr;
    ptr = va_arg(ap,char *);
  }
  vinfos[num_infos++] = NULL;
  va_end(ap);
}

static void send_version_info()
{
  extern MMDatum TV;		/* in mm.c */

  char *ptr;
  int i;
  MMStkFrame mark;

  if (!vinfos) return;		/* malloc() didn't */
  if (version_hook != -1)	/* hook exists */
  {
    i = 0;
    while (i < num_infos)
    {
      MMopen_frame(&mark);
      while (TRUE)
      {
	TV.type = STRING; TV.val.str = vinfos[i++]; MMpush_arg(&TV);
	if (!vinfos[i]) { i++; break; }
      }
      MMrun_pgm_with_args(version_hook,&mark);
    }
  }
  free((char *)vinfos);		/* don't need these anymore */
}

#if 0
/*VARARGS1*/
#ifdef __STDC__
void send_version_info(char *info, ...)
#else
void send_version_info(info,va_alist) char *info; va_dcl
#endif
{
  extern MMDatum TV;		/* in mm.c */

  char *ptr;
  MMStkFrame mark;
  va_list ap;

  if (version_hook == -1) return;		/* no hook */

  MMopen_frame(&mark);
  VA_START(ap, info);
  ptr = info;
  while (ptr)
  {
    TV.type = STRING; TV.val.str = ptr; MMpush_arg(&TV);
    ptr = va_arg(ap,char *);
  }
  va_end(ap);

  MMrun_pgm_with_args(version_hook,&mark);
}
#endif

/* ******************************************************************** */
/* ************************ Internal ME Hooks ************************* */
/* ******************************************************************** */

/* Internal hooks are so things like signal handlers can do things at "safe"
 * times.  Some times you can't things do something immediately because ME
 * might be in the middle of a bunch of other stuff.
 */

/* This is a "only-one" queue.  You can call the queuer as many times as you
 *   like but the service routine is only called once (and then the process
 *   repeats - queue once or more and get called once).  This for the cases
 *   where a person repeated resizes a window - we only want to service that
 *   request once if we can get away with it.
 */

/* Right now, this stuff is bog simple because I don't have much to do.  If
 *   I expand this stuff, here is a list of things to take care of:
 *   - Real queues so that n routines can be handled.
 *   - Make it so I can queue hooks while processing hooks.  Gotta be
 *     careful about the case where a hooks is being marked as processed as
 *     another hook (maybe the same one) is being queued.
 */

static pfv fcns_to_call[NUM_HOOKS];	/* all inited to NULL */

    /* Called by signal handler.
     */
/* !!!Gotta get a interrupt that breaks back to the main loop (eg
 * stop_ME()).   ?? what about breaking out of write(), etc??? */
void call_me_sometime(hook, routine) int hook; pfv routine;
{
  fcns_to_call[hook] = routine;
  process_hooks();		/* Maybe we can take care of it right now */
}

    /* Called when somebody thinks its a good time to process queued
     *   hooks/interrupts but NEVER from a signal handler.  Checking is done
     *   to see if this is a "safe" time to process a set of hooks.
     * This routine needs to be reentrant!  Protect against
     *   call_me_sometime() being called while I'm messing with
     *   fcns_to_call[].  Can't really, but I can try.
     */
void process_hooks()
{
  int n;
  pfv fcn;

  if (doing_nothing)
  {
    doing_nothing = FALSE;		/* protect against recursion */
    for (n = 0; n < NUM_HOOKS; n++)
    {
      fcn = fcns_to_call[n];
      if (fcn)
      {
      	(*fcn)();
        fcns_to_call[n] = NULL;		/* ??? before or after???? */
      }
    }
    doing_nothing = TRUE;
  }
}
